;*****************************************************************************
;** This is the JEMM386 ASM source code.
;** JEMM386 is a branch of FreeDOS Emm386, which in turn used the source of
;** an EMM made public in c't (a german IT magazine) in 08/90, page 214,  
;** written by Harald Albrecht.
;**
;** original author and copyright
;**   (c) 1990       c't/Harald Albrecht
;**   (c) 2001-2004  tom ehlert
;**
;**	Licensed under the Artistic License version
;**	please see LICENSE.TXT for details
;**
;*****************************************************************************
;**
;** c't/Harald Albrecht: created the v86 monitor part and EMS functions
;** 1990 
;** 
;** Thomas Gloeckler, Rechenzentrum FH Ulm: DMA enhancements + fixes
;** 1990 
;**
;** junk@drivesnapshot.de: put into current shape as a potential EMM386 
;** 2001                   (EMM386-VCPI-DPMI-...)
;**
;** tom ehlert: english comments, ... 
;** 2001-2004   may have cut/copy/pasted ... things around
;**
;** Imre Leber: All German strings finally translated to English.
;** 2003        Merged code from Eric Auer.
;**
;** michael devore: implemented support for VCPI, VDS (partially), dynamic
;** 2004-2006       XMS memory allocation, EMS v4.0 (partially)
;** - Modified for >64M and VCPI support
;** - Updated for EMS 4.0, some extended API returns unsupported, this will
;**   change as requested or needed by other software requirements
;** - Documented EMS 4.0 only supports up to 32M (0x800 pages), but some EMS
;**   drivers allow more than 32M.  The EMS 4.0 spec could be extended to 1G-32K
;**   (map value of 0xFFFF means unmap) without breaking the spec API,
;**   but unfortunately breaking unsophisticated programs which get confused.
;**   I have arbitrarily decided to allow up to 32M of EMS allocations,
;**   leaving the high bit of page allocation alone since specific drivers
;**   may make use of high bit values other than 0xFFFF for their own purposes,
;**   or may only check high bit for unmap -- as FreeDOS EMM386 does.
;**	- Minor English corrections where useful for comprehension
;**
;**   Michael Devore's changes are not copyrighted and are released
;**   to the public domain.
;**   This does not affect copyright on the rest of the code.

;--   the following modifications are done for JEMM386:
;--   generally: 
;--    + comments were deleted if they were definitely wrong or outdated.
;--    + All modifications are public domain.
;--    + The source is now best viewed with Tab Size 4!
;--
;--   1. bugfix: changes done in proc pm_entry: switch to host stack
;--      and back.
;--   2. bugfix: in proc VCPI_GetInterface: do not return a fixed value
;--      in DI for the page table 0 offset, instead calculate this value
;--      from the V86-monitor code descriptor. Return with an error
;--      if page table 0 is "full". (obsolete).
;--   3. moved V86 monitor stack into shared extended memory.
;--      temp stack removed. saves 1 kB of conv. memory.
;--   4. moved GDT into shared extended memory, thrown away LDT and 
;--      temp. TSS.
;--   5. IO-Bitmap and IDT created on the fly. Reduces size of binary by
;--      about 10 kB.
;--   6. rearranged segments to simplify binary structure.
;--      RSEG        resident code and data (=GROUP RESGRP)
;--      V86SEG      protected-mode code (+data) moved to extended memory
;--      INITSEG     code for monitor initialization
;--      _TEXT,_DATA real-mode code+data shared with the C part
;--   7. illegal opcode error handler rewritten which further reduces
;--      conv. memory requirements.
;--   8. most of the VDS code moved to extended memory, runs in protected
;--      mode now. The real-mode part (about 40h bytes) will not stay 
;--      resident if option NOVDS is given (this has been made obsolete
;--      by later development, the real-mode part is now just a few bytes).
;--   9. display more infos for exceptions occuring in ring 0.
;--      page faults and GPFs will display the proper protected mode CS:EIP.
;--      external exceptions (caused by intruders) may be detected and 
;--      will prompt the user to reboot.
;--  10. code rearranged. now paging is enabled much earlier and
;--      the monitor code will always be located at linear address 110000h
;--      regardless where the physical address is. large parts of the
;--      data moved to linear address 0FFExxxxx, to release PTEs in
;--      page table 0. This also makes 2. obsolete.
;--  11. the full physical memory is no longer mapped in the monitor's 
;--      address space. The Int 15h emulation has been adjusted and will
;--      now always use scratch PTEs for the copy operation. This reduces 
;--      extended memory usage a lot. On 80486+ cpus the INVLPG opcode
;--      will be used to flush TLB entries.
;--  12. option NOALTBOOT deleted. It is obsolete now since IRQ 1 is never
;--      hooked. Instead a HLT breakpoint has been added at FFFF:0000.
;--  13. option MEMCHECK deleted. It is obsolete now, the Int 15h emulation
;--      will *always* build the PTEs on the fly. NOCHECK still checks if
;--      the memory range is inside physical ram. Else a page fault is
;--      generated.
;--  14. command line switch S=XXXX-YYYY added to allow Jemm386 to include
;--      upper memory being activated by UMBPCI as UMBs. This allows UMBPCI
;--      to be called before Himem and then removed entirely from DOS memory.
;--  15. if a DOS app hooks IVT vector 67h, the interrupt is now routed to
;--      v86-mode (and catched at last by a breakpoint, which will transfer
;--      control to the V86 monitor). Previously hooking IVT vector 67h in
;--      real-mode was useless, because this interrupt was handled in 
;--      protected-mode and the hook proc was never called. This is mainly 
;--      to increase compatibility with MS Emm386.
;-------- v5.00pre
;--  16. implemented the EMMXXXX0 device, and allow some minimal
;--      communication with it. Makes several apps happy, among them MS-DOS.
;--  17. DMA buffering code redesigned. Int 13h and 40h are no longer
;--      hooked, instead it is checked in int 15h, ax=9100h and ax=9101h 
;--      if the DMA buffer content should be transfered back (this was
;--      reverted later, Int 13h and 40h are hooked again).
;--  17a.All variables handling DMA moved to V86SEG segment. Furthermore all
;--      real-mode ints are hooked *after* the monitor has been installed.
;--  18. DMA bug fixed: the buffer is 64k max, but phys. addr. wasn't checked 
;--      if it crosses a 64 kB boundary. Now the check is done and also size
;--      is checked if it exceeds maximum size. (no error msg is displayed,
;--      though). The DMA buffer size may be below 64 kB now (fixed with 26.).
;--      Bugs still open: A truly safe buffer would beginn on a 128 kB border
;--      and be 128 kB in size. It also must be located entirely in the first
;--      16 MB (should be no problem usually).
;--  19. DMA: Single Mask and Master Mask ports also trapped now. It is
;--      checked if the channel is active, and nothing is done unless it is.
;--  20. DMA bugfix: the DMA buffer is not always used when it has to be
;--      used. The CONTINUOUS? proc was not strict enough.
;--  21. bugfix accessing XMS handle array: unused handles got the FREE flag
;--      set (bit 0). Now unused handles got the "INPOOL" flag set (bit 2).
;--      This makes JEMM386 work better with MS Himem (at least).
;--  22. bugfix: mapped page at F0000 (to catch RESET) wasn't write protected.
;--      The fix requires to catch page faults caused by write accesses and
;--      "emulate" them.
;--  23. display FS and GS in invalid opcode handler and wait for ESC key
;--      before trying to abort the current program with int 21h, ah=4Ch.
;--  24. A20 disable/enable emulation activated
;--  25. VME extension supported. Software interrupt redirection bitmap 
;--      added to the TSS. Detection of CPUID opcode support. If VME
;--      extensions are present (Pentium+), the VME bit in CR4 is set.
;--  26. DMA buffer is now 64 kB in size again and begins on a physical
;--      64 kB border. This is achieved by rearranging PTEs. 
;--  27. the EMMXXXX0 device interface may be used to query the current EMM 
;--      status.
;-------- v5.00
;--  28. bugfix: the VDS disable/enable DMA translation cannot be dummies.
;--      They must decrease/increase a counter and if zero the automatic
;--      translation for the channel must be disabled.
;--  29. debug display of "unsupported" VDS functions removed. These functions
;--      will only succeed if a valid buffer ID has been supplied, which 
;--      currently is never the case. Now just the proper error code is
;--      returned.
;-------- v5.01
;--  30. support for variable DMA buffer size added.
;--  31. VDS functions 07/08 (req/rel buffer) and 09/0A (copy in/out of 
;--      DMA buffer) implemented. VDS functions 03/04 (lock/unlock region)
;--      now really implemented (they will use functions 07/08 if needed). 
;--  32. option ALTBOOT implemented. Uses Int 15h, ah=4Fh as keyboard hook.
;--  33. bugfix: install an XMS handler even if no UMBs are supplied, since
;--      the XMS handler is also used for A20 handling.
;--  34. bugfix: NoVCPI no longer requires NoEMS to work. Why was this ever
;--      implemented?
;--  35. bugfix: INT 67h, AH=4Eh, Subfunc 2 didn't set state.
;--  36. INT 67h, AH=4Fh implemented
;--  37. bugfix: INT 67h, AH=43h left EMM in an inconsistent state if it 
;--      ran out of free pages during the allocation.
;-------- v5.10
;--  38. bugfix: ALTBOOT not set and a write into FF00 segment caused a
;--      page fault which was not handled correctly, causing crash.
;--  39. EMS/VCPI memory management improved, no space wasted.
;--  40. 32 MB limit for EMS may be overcome with MIN= option.
;--  41. if pool sharing cannot be activated and MIN= parameter is not
;--      set in command line, a reasonable assumption is made about amount 
;--      of EMS/VCPI memory to allocate.
;--  42. debug messages during protected-mode initialization implemented.
;--  43. JEMM386.ASM can be assembled with MASM 6.1
;--  44. protected-mode segments V86SEG and INITSEG made 32-bit.
;--  45. save/restore CR2 on page faults at page FF (ALTBOOT).
;--      (reverted in v5.21 since it was useless).
;--  46. bugfix: if pool sharing is on, A20 disabled and Himem has placed
;--      its handle table into the HMA ...
;--  47. A20 emulation now (also) done on the "port" level. The XMS hook
;--      still is useful if XMS host has decided not to disable A20 ever
;--      (because it has found A20 on on startup).
;--  48. all assembler code in JEMM386C.C moved to JEMM386.ASM to make
;--      JEMM386C.C compileable with other compilers (Open Watcom, MS VC).
;--      Compiler specific helper functions for DWORD div/mod/mul/shl/shr
;--      added.
;--  49. bugfix: some EMS functions (int 67h, ah=57h) destroyed hiword(eax).
;--  50. int 67h, ax=5900 "implemented".
;--  51. bugfix: in MAP_PAGE, the logical page to map may be invalid
;--      (because the page was mapped into the page frame, but its handle
;--      has long been released). Then do not try to get pool information
;--      for this page, this may map "random" pages into the page frame.
;--      Instead, unmap the page, which will result in linear=phys.
;-------- v5.20
;--  52. check CX parameter is not > 8000h in int 15h, ah=87h emulation.
;--  53. LOAD command line parameter support added.
;-------- v5.21
;--  54. bugfix: EMS realloc may have changed logical page order of a handle.
;-------- v5.22
;--  55. close files before going resident as TSR.
;--  56. implemented VDS 8105 (scatter lock) with bit 6 of DL set 
;--      (returning PTEs).  
;--  57. EMX option added.
;-------- v5.23
;--  58. bugfix: int 67h, ax=5001h, didn't work, but may have reported
;--      success.
;--  59. PGE option added.
;-------- v5.24
;--  60. bugfix: if Ctrl-Alt-Del is detected the real-mode int 15h chain 
;--      is called first before rebooting (allows SmartDrv and others to
;--      cleanup).
;-------- v5.25
;--  61. bugfix: VDS Request/Release Buffer called VDS Copy In/Out of buffer,
;--      but the latter has an additional parameter (offset in buffer in 
;--      BX:CX), which has to be set to 0 first.
;--  62. bugfix: VDS Request Buffer was unable to alloc the last kb of the
;--      DMA buffer.
;--  63. bugfix: VDS Release Buffer was unable to release a buffer if size
;--      was < 400h and > 0.
;-------- v5.26
;--  64. bugfix: int 67h, ah=5Ah with BX=0 didn't return a valid handle in DX
;--  65. bugfix: int 67h, ah=51h destroyed content of AL register
;--  66. bugfix: int 67h, ah=51h may have called function TryFreeToXMS with
;--      DF set, which might have caused data corruption.
;-------- v5.27
;--  67. bugfix: int 4Bh, ax=8105h (scatter lock): if bits 0-11 of offset in
;--      EDDS were <> 000h, the first page was returned as a separate region.
;--  68. bugfix: int 4Bh, ax=8105h (scatter lock): if this function failed
;--      with AL=9, it should have set field NumUsed in EDDS with a value 
;--      required to describe the full region. This wasn't always the case.
;--  69. bugfix: Int 67h, ah=53h and 54h accessed (nonexistant) name table
;--      entries. Since it is possible to alloc a handle even with NOEMS,
;--      the name table must always exist. 
;--  70. bugfix: int 67h, ah=57h (move to/from expanded memory) didn't work
;--      without a page frame.
;-------- v5.28
;--  71. int 67h, ah=58h now works with NOEMS.
;--  72. error code of int 67h, ah=41h, 47h, 48h if no page frame is defined
;--      changed to 80h from 91h, because 91h seems to be defined for EMS
;--      v4.0 only.
;--  73. NOINVLPG option added.
;-------- v5.29
;--  74. int 67h, ah=55h/56h implemented.
;--  75. bugfix: int 67h, ah=57h didn't always flush the correct TLB entries.
;--  76. bugfix: int 67h, ax=4F01h didn't work.
;-------- v5.30
;--  77. bugfix int 67h, ah=57h: test for overlapping EMS regions didn't
;--      work.
;--  78. bugfix int 67h, ah=57h: conventional memory regions weren't tested 
;--      if they exceed the 1 MB boundary, in which case error A2h should be
;--      returned.
;--  79. bugfix: int 67h, ah=57h didn't return status 92h if an overlapp
;--      occured (92h is a hint that part of the source has been overwritten).
;--  80. bugfix: if an exception occured in the v86-monitor, the register
;--      contents were displayed wrongly in v5.30.
;--  81. exceptions occured in protected-mode are now displayed the same way
;--      as the v86-exceptions, that is, through int 29h in v86-mode. The 
;--      previous solution (writing directly into the video screen buffer  
;--      from protected-mode) didn't work if screen was in graphics mode.
;--  82. handler for v86-int06 (invalid opcode exception) moved to protected
;--      mode, reduces monitor's conventional memory usage to 192 bytes.
;-------- v5.31
;--  83. removed an optimisation in MAP_PAGE, which caused problems with    
;--      PKZIP/PKUNZIP.
;-------- v5.32
;--  84. changes in REBOOT (some old machines still didn't reboot with
;--      Ctrl-Alt-Del).
;--  85. monitor stack, GDT and IDT added to V86SEG. Thus the extra
;--      monitor stack segment is no longer needed, and SS can be used
;--      to access monitor data. Small catch: file size increases by about
;--      2,5 kB because stack, GDT and IDT are now a static part of the
;--      binary.
;--  86. V86SEG + INITSEG grouped into one 32bit segment (_TEXT32),
;--      and RSEG + _TEXT grouped into one 16bit code segment (_TEXT16).
;--      This finally reduces number of physical segments to 3:
;--      _TEXT16: 16 bit code (real-mode)
;--      _TEXT32: 32 bit code+data (protected-mode)
;--       DGROUP: 16 bit data (real-mode)
;--  87. bugfix: EMS might have been disabled due to a query with a
;--      wrong segment register assume.
;-------- v5.33
;--  88. bugfix: ALTBOOT was erroneously always enabled since int 15h
;--      is always hooked in protected-mode.
;--      
;*****************************************************************************
		TITLE   JEMM386 - Virtual 8086-Monitor for 80386 PCs
		NAME    JEMM386
;
; (c) 1990 c't/Harald Albrecht
; (c) 2001 tom.ehlert@ginko.de
;
; to be assembled with MASM 6.1 or TASM 4.1/5.0
;
; LIM-EMS-Driver for 386 PCs (EMS Version 4.0)
;
; (c) 1989 Harald Albrecht
;
		.486P
        .SEQ

	include jemm386.inc	
	include debug.inc

;--- some parts of the EMM binary are written in C
;--- to prevent the linker from including CRT modules,
;--- some compiler helper functions are (re)implemented in ASM

ifndef ?WCC			; Open Watcom WCC
?WCC equ 0
endif
ifndef ?MSC			; MS Visual C++ 1.52 compiler
?MSC equ 0
endif
ifndef ?DMC			; Digital Mars C++
?DMC equ 0
endif

?WT		equ 0		; set WT bit in PTE on int 15h, ah=87 emulation
?NEARC	equ 0		; set to 1 to support TCC 3

ife ?MASM
	LOCALS
endif
;
;-- public data items

    public _ALTBOOT			; ALTBOOT switch
    public _DMABUFFSIZE		; DMA buffer size in kB (D=nnn option)
	public _EMM_MEMORY_END	; physical end of memory for EMM
	public _FRAME			; EMS page frame segment (FRAME=xxxx)
	public _MAXMEM16K		; max VCPI in 16 kB (MAX=nnn * 16)
	public _MAXEMSPAGES		; EMS pages maximal available
	public _MONITOR_ADDR	; physical start address monitor
if ?A20PORTS or ?A20XMS    
    public _NoA20			; A20/NOA20 switch
endif    
	public _NoEMS			; NOEMS switch
    public _NoFrame			; FRAME=NONE switch
	public _NoPool			; NODYN switch
	public _NoVCPI			; VCPI/NOVCPI switch
	public _NoInvlPg		; NOINVLPG switch
if ?VDS    
    public _NoVDS			; VDS/NOVDS switch
endif    
if ?VME    
    public _NoVME			; VME/NOVME switch
endif    
if ?PGE
    public _NoPGE			; PGE/NOPGE switch
endif
	public _EMSPAGESAVAIL	; EMS pages available
	public _bV86Flags		; "minor" flags: SB, NOCHECK, ...
	public _TOTAL_MEMORY	; highest physical address in machine
	public _XMS_CONTROL_HANDLE

;--- public procs

	public _AddIfContiguousWithDosMem	;currently not used
	public _EmmStatus		; display EMM status
	public _EmmUpdate		; update Jemm386 status
	public _InstallXMSHandler	; install Jemm XMS handler proc
	public _ISPROTECTEDMODE	; proc to determine if cpu is in v86 mode
	public _PUT_CONSOLE		; write to console
	public _TestForSystemRAM	;test if a page is RAM in upper memory
    public _fmemset
    public _fmemcpy
	PUBLIC	_ltob
    

;--- segment definitions

RSEG SEGMENT PARA USE16 'CODE'	;resident code + data
RSEG ENDS

_TEXT	segment	word public 'CODE' use16
		extrn	_printf:near
		extrn _startup_driver:near
		extrn _startup_exe:near
		extrn _finishing_touches:near
_TEXT	ends

V86SEG  SEGMENT PARA USE32 public 'CODE'	;protected mode code + data
V86SEG 	ENDS

INITSEG	SEGMENT PARA USE32 public 'CODE'	;protected mode nonres code
INITSEG	ENDS

BEGDATA segment PARA public 'DATA' use16	;make sure DGROUP is para aligned
BEGDATA ends

_DATA	segment word public 'DATA' use16
_DATA	ends
if ?MSC
CONST	segment word public 'CONST' use16
CONST	ends
endif

_BSS	segment word public 'BSS'  use16
_BSS	ends

_STACK	segment STACK  'STACK' use16
_STACK	ends

_TEXT16 GROUP RSEG, _TEXT
_TEXT32	GROUP V86SEG, INITSEG

if ?MSC
DGROUP	group	BEGDATA,_DATA,CONST,_BSS,_STACK
else
DGROUP	group	BEGDATA,_DATA,_BSS,_STACK
endif

;--- monitor data items (protected mode only)

V86SEG  SEGMENT

		dd V86_TOS/4 dup ("KATS")	;monitor stack

; GDT - Global Descriptor Table

; Macro for describing descriptors

?GDTOFS = 8
TSS_LEN equ 68h+2000h+8+20h

SELECTOR MACRO BEGIN_,LIMIT_,ACCESS_,GRANULARITY_, NAME_
ifnb <NAME_>
NAME_	equ ?GDTOFS
endif
		DW      LIMIT_                  ;; Size of Segment (15..0)
		DW      BEGIN_                  ;; Start of Segment (15..0)
		DB      0                       ;; Beginning (23..16)
		DB      ACCESS_
		DB      GRANULARITY_            ;; u.a. Size (19..16)
		DB      0                       ;; Beginning (31..24)
?GDTOFS = ?GDTOFS + 8        
		ENDM

GDT     LABEL BYTE
  DQ 0                                      ; NULL-Entry
  SELECTOR 0,TSS_LEN-1,89H,0,V86_TSS_SEL    ; TSS V86
  SELECTOR 0,-1,9AH,40h,V86_CODE_SEL        ; V86SEG Code (32 Bit)
  SELECTOR 0,-1,92H,40h,V86_DATA_SEL        ; V86SEG Data (32 Bit)
  SELECTOR 0,-1,9AH,0CFh,FLAT_CODE_SEL      ; flat 4 GB Code (32 Bit)
  SELECTOR 0,-1,92H,0CFh,FLAT_DATA_SEL      ; flat 4 GB Data (32 Bit)
  SELECTOR 0,<RSEG_END-1>,92H,0,RSEG_SEL    ; resident data (16 Bit)
  SELECTOR 0,-1,9AH,0,REAL_CODE_SEL         ; std 64k Code Descriptor
  SELECTOR 0,-1,92H,0,REAL_DATA_SEL         ; std 64k Data Descriptor

	PURGE   SELECTOR

GDT_SIZE	equ 200h

; IDT - Interrupt Descriptor Table

	ORG V86_TOS+GDT_SIZE

GATEOFS = 0

IDTGATE MACRO 
if ?MASM
	dw LOWWORD(offset INT_TABLE) + GATEOFS
else
	dd offset INT_TABLE + GATEOFS
    org $-2
endif
    dw V86_CODE_SEL
    dw 0EE00h		;P=1, DPL=3, DT=0, type=E (386 interrupt gate)
    dw 0
GATEOFS = GATEOFS + 4
	ENDM

	REPT 100H
		IDTGATE    
	ENDM

	PURGE IDTGATE

;--- global monitor data

;PAGEDIR DD 0		; linear address page directory
_CR3    DD 0        ; CR3 for monitor address context

IDT_PTR LABEL FWORD	; size+base IDT
        DW 7FFH
        DD 0      ; Will be set later
        
GDT_PTR LABEL FWORD	; size+base GDT
        DW GDT_SIZE-1
        DD 0
        
        align 4
        
;dwTSS		dd	0	;linear address TSS
dwFeatures	dd  0	;features from CPUID 

MPIC_BASE 		dw 0008h
SPIC_BASE 		dw 0070h
_TOTAL_MEMORY	dd 0                    ; r/o				
_MAXMEM16K		DD MAXMEM16K_DEFAULT	; maximum amount of VCPI memory in 16K


;--- EMS variables

EMSSTATUSTABLE  DD  0   ; array (item size 8) of EMM Handles
EMSNAMETABLE	DD  0	; linear address storage for EMS handle names
EMSPAGETABLE    DD  0   ; array of pages assigned to EMS handles
FRAMEANCHOR     DD  0	; linear address where PTEs for EMS-Frame are stored
FIRSTPAGE       DD  0   ; start EMS pages (physical address!)
_MAXEMSPAGES    DD  MAX_EMS_PAGES_ALLOWED	; total EMS pages
_EMSPAGESAVAIL  DD  0   ; free EMS pages
PHYSPAGETABLE   DW  4 DUP (-1)  ; EMS pages currently mapped in EMS frame
_FRAME          DW  0E000H	; EMS frame segment address

				align 4

;--- address of pointers to EMS page allocations within allocation blocks
;--- each item consists of 2 words

EMSPageAllocationStart DD	0
EMSPageAllocationEnd   DD	0

;-- address of EMS/VCPI allocation block start
;-- each item consists of 64 bytes

PoolAllocationTable     DD	0
PoolAllocationEnd       DD	0

LastBlockAllocator      DD 0	; pointer to last block allocated from
LastBlockFreed          DD 0	; pointer to last block freed from
XMSBlockSize            DD 0	; current size of XMS block to allocate for sharing, in K
XMSPoolBlockCount       DD 0	; count of XMS handles allocated for pool blocks (not counting initial UMB block)

if ?DMA
		align 4

DMABuffStart    DD  0   ; Beginning of the DMA-Buffer (linear address)
DMABuffSize     DD  0   ; max size of the DMA-Buffer in bytes
DMABuffStartPhys DD 0   ; Beginning of the DMA-Buffer (physical address)
TargetAdr       DD  0   ; Original adress for DMA
TargetLen       DD  0   ; Utilized part of the Buffer

DMAREQ struc        
ChanMode	db ?	;saved "mode" value (register 0Bh or D6h)
bFlags		db ?	;flags, see below
BlockLen 	dw ?	;saved "block length" value
BaseAdr 	dw ?	;saved "block address" value
PageReg 	db ?	;saved "page register" value
cDisable 	db ?	;is automatic translation disabled? (VDS)
DMAREQ ends

;bFlags values

DMAF_ENABLED 	equ 1	;1=channel is enabled/unmasked

DmaChn			DMAREQ 8 dup (<0,0,0,0,0,0>)
DMABuffFree     DD ?DMABUFFMAX/32 dup (0)   ; bits for DMA buffer handling
				db 0	; space for 2 additional bits is needed!
                db 0	; alignment byte
GblFlags        DW 0	; global flags

endif
				align 4

bEmsPhysPages	DB  4	; 4 if page frame set, else 0

;-- byte vars (variable)

if ?A20PORTS
bLast64			DB  0
endif
if ?A20XMS
wA20			dw 1	; local + global XMS A20 count
endif
if ?A20PORTS or ?A20XMS
_NoA20  		DB  0
bA20Stat		DB  1	; default is A20 enabled
endif

;-- byte vars (rather constant)

_NoEMS			DB	0	; flags no EMS services
_NoFrame		DB	0	; flags no page frame
_NoVCPI			DB	0	; flags no VCPI services
_NoPool			DB	0	; flags pooling with XMS memory
_NoInvlPg       DB  -1	; <> 0 -> dont use INVLPG
_bV86Flags  	DB	0	; other flags 

V86F_SB			equ 1	; soundblaster driver compat
V86F_NOCHECK	equ 2	; flag NOCHECK option
if ?EMX
V86F_EMX		equ 4	; EMX compat
endif

; Flag to enable/disable special
; handling of INT67/44 (MAP_PAGE)
; during init phase, abused to map UMB everywhere
_INIT_DONE		db  0	; reset after initialization
bIs486          db  0	; cpu is 486 or better (INVLPG opcode available)
if ?PGE
bPageMask	db  0	;mask for PTEs in 0-110000h
endif

		align 4

RunEmuInstr:
EmuInstr	DB	90h,90h,90h	; run self-modifying code here
		ret
        
;--- place rarely modified data here
;--- so data and code is separated by at least 64 "constant" bytes

if ?ROMRO
dwSavedRomPTE dd 0	;saved PTE for FF000 page faults
dqSavedInt01  dq 0	;saved INT 01
endif

pmUMBsegments	dd 8 dup (0)
XMS_Handle_Table	XMS_HANDLE_TABLE_STRUC	<0,0,0,0>

		align 16        

V86SEG 	ENDS


RSEG SEGMENT

    ASSUME  CS:_TEXT16
	ASSUME  DS:NOTHING,ES:NOTHING,FS:NOTHING,GS:NOTHING

;******************************************************************************
; device driver header

device_header:
		dd  -1			    ; last driver in list
		dw  0c000h		    ; driver flags :
							; 8000 - character device
							; 4000 - supports IOCTL - like EMM386
PStratOfs dw  offset strategy	    ; pointer to strategy routine
pIntOfs dw  offset _driver_entry    ; pointer to interrupt handler

device_name label byte
		db  'EMMXXXX0'	    ; device driver name


request_ptr dd  0			    ; pointer to request header

;--- start of real-mode resident data area

if ?DMA
bRFlags 	DB 0
endif

;--- end of the data part

;******************************************************************************
; strategy routine. is called by DOS to initialize the driver once.
; only thing to be done here is to store the address of the device driver
; request block.
; In:   ES:BX - address of request header
; Out:  nothing

strategy_new:
	push es
    push bx
    pop dword ptr cs:[request_ptr]		; this is 80386 code!
	retf

interrupt:
BPDRV:
	@BPOPC
	retf

BP67:                       ; In case software (among them TURBO Pascal
	@BPOPC		            ; Programs) gets the idea, by
	IRET                    ; calling CS:IP from the Interrupt-table
                            ; of the EMM, to in fact call the EMM

BPBACK:						; bp to return to v86 monitor from a v86 far proc
	@BPOPC

v86int29:
	int 29h
    retf

;*********************************************************************
; invalid opcode exception
; print some diagnostic
; end program
;
; if int21/4c returns (during device init or more regularily
; for command.com), there is nothing left do to. show 'I am alive'
; and wait for '3FingerSalut'
;*********************************************************************

NEW06 proc FAR

if ?MASM
BP06::
else
BP06:
endif
	@BPOPC

NEW06 endp

Int06_WaitKey:
	mov ah,0
    int 16h
    cmp al,1Bh
    jnz Int06_WaitKey
    mov al,13
    int 29h
    mov al,10
    int 29h
	mov ax,04C7Fh
	int 21h
BP06_FATAL:
    sti
	@BPOPC
    jmp BP06_FATAL

RSEG ENDS


RSEG segment

	align 4
    
if ?DMA

NEW13 PROC FAR
	mov cs:[bRFlags],2	; tell the monitor that a new DMA op has started
	pushf
    db 09Ah
OLDINT13 dd 0    
NEW13 ENDP

int1340common:
	jc  iret_with_new_CY
    btr word ptr cs:[bRFlags],0
	jnc iret_with_new_CY
    clc
BP1340:
	@BPOPC
	jmp iret_with_new_CY
    
NEW40 PROC FAR
	mov cs:[bRFlags],2	; tell the monitor that a new DMA op has started
	pushf
    db 09Ah
OLDINT40 dd 0    
	jmp int1340common
NEW40 ENDP

endif

;--- reboot machine breakpoint. the address of the break is 
;--- set by int 15 protected-mode handler if ctrl-alt-del has
;--- been detected. It is important that the real-mode int 15
;--- chain is then called first before reboot actually is done.
;--- this allows hookers (like SmartDrv) to do their cleanup stuff.
;--- when finally the real-mode IRET is reached, it will "return"
;--- to this breakpoint here.

REBOOT_RM:
	@BPOPC

;******************************************************************************
; INT15 handler:
;    everything forwarded to old handler, except int15/87 = move_memory
;******************************************************************************

	align 4

NEW15 PROC FAR
	CMP     AH,87H               	; is it a blockmove ?
	JZ      SHORT @@do_int1587 	 	;
    db 0EAh
OLDINT15 dd 0    

; suggested by eric auer:
;	the int 15h, ah=87h is routed to int 67h, registers unchanged.
;   works because int 67h, ah=87h is not used by EMS.
;
;INT 15 - SYSTEM - COPY EXTENDED MEMORY
;
;        AH = 87h
;        CX = number of words to copy (max 8000h)
;        ES:SI -> GDT (4 descriptors)
;Return: CF set on error
;        CF clear if successful
;        AH = status

@@do_int1587:
	int 067h
if 0    
	cmp ah,1
    cmc
endif

;--- fall through
    
NEW15 ENDP

; only update carry flag status, keep rest of original flags

iret_with_new_CY:
updated_iret:
	push bp
    pushf
	mov	bp,sp
	and	WORD PTR [bp+2*2+4],not 1   ; bp is ss: relative, strip old CY
    popf
	adc WORD PTR [bp+2*2+4],0
	pop	bp
iret_nop:
	iret
    
RSEG ends

if ?VDS

RSEG segment

;--- VDS code real-mode

	align 4

NEW4B:
	cmp ah,081h
	je  vds_handler_rm

int4b_old:
	db 0eah
OLDINT4B dd 0
	
vds_handler_rm:
BPVDS:			;break into protected mode
	@BPOPC
    jmp iret_with_new_CY
    

RSEG ENDS

endif

;--- UMB code

RSEG segment

;*********************************************************
; XMS hook - required for UMBs and A20 emulation
;*********************************************************

XMShandler proc

	jmp short @@XMSHandler	; standard XMS link chain
	nop					    ; with 3 required NOPs
	nop
	nop
@@XMSHandler:
if ?A20XMS
	cmp	ah,3
    jb  @@not_for_us 
    cmp ah,6
	jbe	A20rm
endif    
if ?MASM
XMSUMB::
else
XMSUMB:
endif
	cmp ah,10h				; 10h=alloc, 11h=free, 12=realloc
	jb  @@not_for_us
	cmp	ah,12h
	jbe @@IsUMBFunction
						    ; else let the others return
						    ; 'not implemented'

@@not_for_us:
	db 0eah					; jmp far  UMBoldhandler
XMSoldhandler dd 0

@@IsUMBfunction:
if ?MASM
BPUMB::
else
BPUMB:
endif
	@BPOPC
    retf

XMShandler endp

;-- for A20 disable and enable emulation it is useful to trap
;-- the XMS functions as well, even if the A20 ports (92, 60, 64)
;-- are trapped. That's because if certain conditions are true
;-- the XMS host will never access the ports and leave A20 unchanged.

if ?A20XMS

A20rm proc
	mov al,ah
	mov	ah,3fh		; special EMM function (QEMM does it, so can we)
	int	67h
if ?MASM    
BEHIND_A20::
else
BEHIND_A20:    
endif
	retf
A20rm endp

endif

RSEG ends

if 0

RSEG	segment

;--- print a string in real-mode (for debugging)

RPRINTSTR PROC
	push bp
    mov bp,sp
    XCHG BX,[bp+2]
    PUSHF
	PUSH AX
@@NEXTCHAR:
    MOV AL,CS:[BX]
    INC BX
    CMP AL,0
    JZ  @@DONE
    push bx
    cmp al,10
	jnz @@isnotlf
    mov al,13
    xor bx,bx
    mov ah,0Eh
    int 10h
    mov al,10
@@isnotlf:    
    xor bx,bx
    mov ah,0Eh
    int 10h
    pop bx
    JMP @@NEXTCHAR
@@DONE:
	POP AX
	POPF
	XCHG BX,[bp+2]
    pop bp
    RET
RPRINTSTR ENDP

rwordout proc near
		push	ax
		mov 	al,ah
		call	rbyteout
		pop 	ax
rwordout endp
rbyteout proc near
		pushf
		push	ax
        mov		ah,al
		shr 	al,4
        call    rnibout
        mov		al,ah
        call    rnibout
        pop     ax
        popf
        ret
rnibout:        
		and 	al,0Fh
		cmp 	al,10
		sbb 	al,69H
		das
        push bx
        push ax
        xor bx,bx
        mov ah,0Eh
        int 10h
        pop ax
        pop bx
        ret
rbyteout endp

RSEG ends

endif

;--- this must be the last RSEG item

RSEG	segment
		align 16
RSEG_END EQU     $		; End of resident part
RSEG	ends

_TEXT segment

;--- the original strategy proc, must be 8086 compatible

strategy:
	mov word ptr cs:[request_ptr+0],bx
	mov word ptr cs:[request_ptr+2],es
	retf

;**********************************************
; driver init part
; this code is only necessary on init and    
; will not go resident. It must however be
; in the same physical segment as the
; resident part.
;**********************************************

_driver_entry proc far

	push ds
    push es
    push di
	lds di, cs:[request_ptr]          ; load address of request header
	mov word ptr ds:[di+3],8103h
    pop di
    call TestCPU
    jc @@no386
	push fs
	push gs
	pushad
	lds di, cs:[request_ptr]        ; load address of request header
	cmp byte ptr ds:[di+2],0 		; command == 0 : do we have to initialize?
	jne @@exit
	call go_driver					; go_driver is still ASM!
	mov word ptr ds:[di+3],0100h	; STATUS_OK
	mov ds:[di+0eh+0],ax  			; if ax == 0, driver won't be installed
	mov ds:[di+0eh+2],cs			; set end address
@@exit:
	popad
	pop gs
	pop fs
@@no386:    
    pop es
	pop ds
	ret
    
_driver_entry ENDP

_TEXT ends

;--- the _DATA segment is meant for all items
;--- which are to be accessed by ASM and C during initialization

_DATA	segment

		extrn _startup_verbose:BYTE		; more (debug) output in Init Phase
		extrn _XMShandleTable:dword
        extrn _XMSdriverAddress:dword
        extrn _UMBsegments:near
        extrn _emmfunction:byte
if ?LOADSUPP
        extrn _bLoad:BYTE
endif


_MONITOR_ADDR   DD 0    ; r/o beginning of the Monitor-Code
_EMM_MEMORY_END dd 0    ; r/o end of monitor memory block
_XMS_CONTROL_HANDLE	DW	0	; initial XMS handle allocated for EMM386 control/fixed allocations
_DMABUFFSIZE    dw ?DMABUFFDEFAULT
if 0;?A20PORTS
wBiosA20		DW 1+2	;default: trap both kbdc + ps2 ports
endif
_ALTBOOT		DB 0
if ?VME
_NOVME  		DB 0
endif
if ?VDS
_NoVDS			DB 0
endif
if ?PGE
_NOPGE  		DB 1
endif

dBye     db " JEMM386 initialized",13,10,'$'
dAlreadyInstalled DB 'JEMM386: EMM already installed',CR,LF,07,'$'
dFailed  DB ' something failed - driver aborted',CR,LF,07,'$'
dError1	 db "JEMM386 not installed.",13,10,'$'

sig1        DB 'EMMXXXX0',0
sig2        DB 'EMMQXXX0',0	; ID if NOEMS specified
szDispVer   db "JEMM386 v%u.%02u installed.",LF,0
szEMSStatus db "EMS is %s",0
szEMSMemory db ", %u of max. %u pages allocated",0
szVCPIOff   db "VCPI is Off.",LF,0
szVCPIOn    db "VCPI is On, %lu of max. %lu pages allocated.",LF,0
if ?VME
szVMEStatus db "VME is %s.",LF,0
endif
szA20Status db "A20 emulation is %s.",LF,0
if ?PGE
szPGEStatus db "PGE is %s.",LF,0
endif
szOn        db "On",0
szOff       db "Off",0
szFrameNone db ", no Page Frame",0
szDotLF		db ".", 10, 0
szFrameYes  db ", Page Frame at %04X",0
szDMABuffer db "DMA buffer at %08lX, size %u kB.",LF,0
szUMB       db "UMB supplied at %04X-%04X.",LF,0
szUMBErr1   db "UMB handler already installed, not installing another one",LF,0

_DATA	ends

;--- protected mode initialization code

INITSEG  SEGMENT

	ASSUME CS:_TEXT32
    assume FS:V86SEG

setstatustable proc    
    cmp		[EMSSTATUSTABLE],0
    jnz     @@done
	MOV     [EMSSTATUSTABLE],EDI; Here starts the status table of the
	MOV     AX,EMSH_SYSTEM      ; handles. Handle 0 is reserved
	STOS    WORD PTR [EDI]      ; for the System, all others are free
	BIG_NOP
	MOV     AX,EMSH_FREE
	MOV     ECX, MAX_HANDLES*4  ; Entries still to fill for 255
	REP     STOS WORD PTR [EDI]	; Handles (REP STOSW, addr. size =
	BIG_NOP
@@done:    
    ret
setstatustable endp

;--- set int 15h to be handled by monitor
;--- in: edx -> software int redirection bitmap in TSS
;--- in: ecx = int15 protected-mode entry

ActivateI15 proc    
	mov		eax, 15h
    bts		[edx], eax
    jc		@@vectorset
    mov		eax, dword ptr [IDT_PTR+2]
    mov		word ptr [eax+15h*8+0], cx
;   shr		ecx,16
;   mov		word ptr [eax+15h*8+6], cx
@@vectorset:    
 	ret
ActivateI15 endp    

; protected mode initialization.
; Starts in PROTECTED-MODE now, in a 32-bit segment!

;--- input expected:
;--- CPU in protected-mode, interrupts disabled
;--- _MONITOR_ADDR: start of EMB for the monitor
;--- _TOTAL_MEMORY: total size of physical memory
;--- _NoEMS, _NoVCPI, _NoCheck, _NoPool: cmdline options
;--- _XMShandleTable: returned by int 2f, ax=4309h
;--- _MAXMEM16K: MAX=xxxx cmdline option given, or 1E00 (120 MB)
;--- _EMM_MEMORY_END: end of EMB for monitor
;--- _XMS_CONTROL_HANDLE: XMS handle for monitor extended memory block
;--- _FRAME: FRAME=xxxx parameter

;--- shared memory (110000 - 11xxxx):

;--- <= 3.0 kB    wasted space because of rounding to page border
;---    0.5 kB    monitor stack
;---    0.5 kB    GDT  [GDT_PTR]
;---    2.0 kB    IDT  [IDT_PTR]
;---   13.0 kB    monitor data+code [V86_CODE_SEL]
;--- <= 3.0 kB    wasted space because of rounding to page border
;---              (if > 2 kB it is filled with STATUSTABLE)

;--- linear addresses in sys space (0FFC00000 - 0FFCxxxxx):

;--- PAGEDIR      (4 kB + 4 kB + 4 kB)
;--- DMABuffStart (0-?DMABUFFMAX kB)
;--- V86_TSS      (90h byte + 8 kB IO-Bitmap)
;--- para align
;--- EMSSTATUSTABLE (2 kB) (8*256)
;--- EMSNAMETABLE (2 kB [or just 8 byte if NOEMS])
;--- EMSPageAllocationStart (EMS pages * 4 bytes) 
;--- EMSPAGETABLE (EMS pages * 1 bytes)
;--- 64 byte align
;--- PoolAllocationTable (128*64 + x)
;--- page align  <= 3kB wasted space
;--- FIRSTPAGE   start of "free" memory in this very first XMS block

;--- segment registers:
;--- CS: _TEXT32 (previously was INITSEG)
;--- DS: FLAT  
;--- ES: FLAT  
;--- FS: V86SEG data
;--- GS: DGROUP data

GO_PROTECTED PROC FAR

	CLD
	XOR     EAX,EAX    		    ; initialise LDT
	LLDT    AX

	MOV     AL,FLAT_DATA_SEL	; Addressing everything
	MOV     DS,EAX
    assume	DS:NOTHING
	MOV     ES,EAX
    assume	ES:NOTHING
	push	V86_DATA_SEL
    pop		FS
	assume	FS:V86SEG
    push	REAL_DATA_SEL
    pop     GS
	assume	GS:DGROUP

if ?INITDBG
	@DbgOuts <"Welcome in PM",10>,1
    @WaitKey 1,0
endif    
    
;-- SS still contains real-mode value!

    MOV     EDI,[_MONITOR_ADDR] ; Start of Monitor-Code

	ADD     EDI,4095        ; Round to the next page border
	AND     DI,NOT 4095		; may loose up to 3 kB	

;-- calc size of the items which must be shared:
;-- GDT, IDT, stack and code.

    mov		eax, offset V86_LEN	;size of code
    add		eax, 1000h-1
    and		ax, not 1000h-1	;round to next page (may waste another 3 kB!)

    mov     ebx, edi		;EBX = physical address of shared items
    add		edi, eax
    mov		ebp, eax		;save size of code, GDT + IDT

    MOV     CR3, EDI		;set CR3, but paging still disabled

;-- clear pagedir, pagetab 0 + scratch table

	push	edi
    mov		ecx, 3000h/4
	xor		eax, eax
    rep		stos dword ptr [edi]
    BIG_NOP
    pop		edi

;-- init page dir (2 PDEs)

	lea		edx, [edi+1000h]
	MOV     [EDI+000h],EDX
	OR      DWORD PTR [EDI+000h],111B
    add		edx, 1000h
    mov		[EDI+?SYSPDE], EDX
	OR      DWORD PTR [EDI+?SYSPDE],111B
    
    add		edi, 1000h	;edi -> mapped page table 0
	mov		eax, edx	;eax -> mapped system page table (usually FFC)

;-- init page table 0 address space 0-110000h

    mov		edx, 0+7
if ?PGE
	cmp		_NoPGE,0
    jnz		@@nopge
	test	byte ptr dwFeatures+1,20h	;PGE supported?
    jz		@@nopge
    push	eax
    @mov_eax_cr4
    or al,80h	
    @mov_cr4_eax
    pop		eax
    or 		dh,1		;set G bit (page global)
@@nopge:    
    mov		bPageMask,dh
endif    
    mov		cx,110h		;hiword ecx is cleared
@@FILL_PAGETAB0:    
	MOV     [EDI],EDX
	ADD     EDX,1000h
	ADD     EDI,4
	loop    @@FILL_PAGETAB0
    
if 0
;-- give the video region A000-BFFF some special attributes
	push edi
    sub  edi, (110h*4 - 0A0h*4)
	mov		cl,20h
@@FILL_VPAGE:
	or	dword ptr [edi],8h	;set "WT"
    add edi,4
    loop @@FILL_VPAGE
	pop edi	
endif

;-- init page table 0 address space 110000h-? (shared space)
    
	mov		ecx, ebp	;size of shared space above 110000h (page aligned)
    shr		ecx, 12		;is just a handful of pages
    mov		edx, ebx	;get physical address of this space
    or		dl, 111b
if ?PGE
;--- set first page (which is shared) global
	or		dh, bPageMask		
endif
    
;	or		dl, 101b	;set R/O
@@FILL_PAGETAB0X:    
	MOV     [EDI],EDX
	ADD     EDX,1000h
if 1    
    and		dl, not 2	;all pages except the first r/o
endif    
if ?PGE
	and		dh, 0F0h
endif
	ADD     EDI,4
	loop    @@FILL_PAGETAB0X

;-- for VCPI intruders, set remaining PTEs in page tab 0
;-- if they still aren't satisfied, they may be just stupid.

	mov     esi, edx
if 1	;SBEINIT.COM *is* stupid, needs linear=physical mapping
	movzx	edx, di
    and		dh, 0Fh
    shl		edx, 10		;transform offset in page table -> linear address
    or		dl,7
endif
    mov		cx, 400h	;hiword ecx is clear
@@FILL_PAGETAB0XX:   
	MOV     [EDI],EDX
	ADD     EDX,1000h
	ADD     EDI,4
    test	di,0FFFh
	loopnz  @@FILL_PAGETAB0XX
	mov     edx, esi   
    
;-- init system page table with the rest we need

	lea		edi, [eax+?SYSPTE]

;-- set PTEs for high space

;-- what is the maximum which is needed?
;-- 3             for page tables
;-- dmasize/4     for dma buffer 
;-- 15            max wasted because of dma buffer alignment
;-- 3             for tss
;-- 1             for ems handles + names
;-- maxems*5/4096 for ems management
;-- maxmem16k * 16 (kb needed) ->
;--           / 1536 (blocks needed) ->
;             * 64 (bytes needed)
;-- -> maxmem16k * 1024 / 1536 -> bytes
;-- -> maxmem16k * 2 / 3 -> bytes
;-- -> maxmem16k / 1536 -> kb
;-- -> maxmem16k / 6144 -> pages

if 0
	mov     ecx, [_EMM_MEMORY_END]
    sub     ecx, [_MONITOR_ADDR]
    sub		ecx, ebp			;subtract the low space
    shr		ecx, 12				;size -> PTEs
    inc		ecx
    movzx	ecx, cx
else
	push	edx
	mov 	eax, [_MAXMEM16K]
    cdq
    mov		ecx, 6144
    div		ecx
    pop		edx
    add		eax, 1+3+15+3 +2       ;add 2 for rounding issues
    mov		ecx, [_MAXEMSPAGES]
    lea		ecx, [ecx+ecx*4]
    shr		ecx, 12
    add		eax, ecx
    movzx	ecx, [_DMABUFFSIZE]
    shr		ecx, 2
    add		ecx, eax
endif

@@FILL_PAGETABF:
	MOV     [EDI],EDX
	ADD     EDX,1000h
	ADD     EDI,4
;    test    DI, 0FFFh			;make sure not to write beyond the page table
	loopnz  @@FILL_PAGETABF

;	@DbgOuts <"page tables initialized",10>

;--- page dir, page tab 000 and sys page tab are now initialized

;--- switch the context now!

	MOV     EAX,CR0
	OR      EAX,80000000H       ; enable Paging
	MOV     CR0,EAX

;--- begin to set linear addresses

if ?INITDBG
	@DbgOuts <"Paging is enabled",10>,1
    @WaitKey 1,0
endif    

;-- copy the V86SEG segment to extended memory

	mov		edi, 110000h
	lea		ebx, [edi+V86_TOS]	; store linear address of GDT in ebx
	MOV     SI,V86SEG			; get position of V86SEG segment
	MOVZX   ESI,SI				; in conventional memory
	SHL     ESI,4
	MOV     ECX, offset V86_LEN ; Length of the code (is dword aligned)
    shr		ecx, 2
    rep		movs dword ptr [edi], [esi]
    BIG_NOP

if ?INITDBG
	@DbgOuts <"V86SEG moved in ext mem",10>,1
    @WaitKey 1,0
endif    

;-- set the V86SEG descriptor bases

    mov		eax, 110000h
	MOV     WORD PTR DS:[EBX+V86_CODE_SEL+2],AX
	MOV     WORD PTR DS:[EBX+V86_DATA_SEL+2],AX
	SHR     EAX,16
	MOV     BYTE PTR DS:[EBX+V86_CODE_SEL+4],AL
	MOV     BYTE PTR DS:[EBX+V86_DATA_SEL+4],AL
	MOV     BYTE PTR DS:[EBX+V86_CODE_SEL+7],AH
	MOV     BYTE PTR DS:[EBX+V86_DATA_SEL+7],AH

;--- reload GDTR

	movzx	esp,sp
    push	ebx
    push	word ptr GDT_SIZE-1
    lgdt	fword ptr [esp]
    add		esp,4+2

;-- reload FS cache

	push	fs
    pop		fs
    
;-- set SS:ESP

    push	fs
    pop 	ss
    mov		esp,V86_TOS-8*4	;set the stack below top, so we can catch
    						;exceptions

if ?INITDBG
	@DbgOuts <"GDTR + SS:ESP set",10>,1
    @WaitKey 1,0
endif    

	MOV		EAX,CR3
    MOV		[_CR3],eax
;    mov		word ptr [GDT_PTR+0], GDT_SIZE-1
	mov		dword ptr [GDT_PTR+2], EBX
    lea		eax,[ebx+GDT_SIZE]
	mov		DWORD PTR [IDT_PTR+2], EAX

	LIDT    FWORD PTR [IDT_PTR]  ; load IDTR

if ?INITDBG
	@DbgOuts <"IDTR set",10>,1
    @WaitKey 1,0
endif    

;--- test if 2 kB of the wasted space could be used for EMS

	mov  cx, di
    and  cx, 0FFFh
    jz   @@nothingtoregain
    cmp  cx, 800h
	ja   @@nothingtoregain
    call setstatustable
@@nothingtoregain:

;--- the memory in page tab 0 is initialized

;--- until now are consumed:
;--- 3 pages for pagedir+pagetab
;--- 4-5 pages for monitor stack, GDT, IDT, data, code

if ?INITDBG
	@DbgOuts <"GDT, IDT, stack and code in ext mem initialised",10>,1
    @WaitKey 1,0
endif    

    mov     eax, ?SYSLINEAR
;   MOV     [PAGEDIR],eax
	lea		edi, [eax + 3000h]  ; skip page dir + 2 page tables

if ?DMA

	MOV     [DMABuffStart], EDI
    movzx	eax, [_DMABUFFSIZE]	;buffer size in kB
    shl		eax, 10
    mov		ecx, eax
    shr		ecx, 10
    inc		ecx		;add start and end bit
    inc		ecx
    xor		edx, edx
@@nextdmabuffbit:    
    bts		[DMABuffFree], edx
    inc		edx
    loop	@@nextdmabuffbit

;-- get the physical address of current EDI in ECX

    mov     ecx, edi
	sub		ecx, ?SYSLINEAR
    add		ecx, ebp
    add		ecx, [_MONITOR_ADDR]
    cmp		ecx, 1000000h-20000h
    jc		@@dmaaddrok
    xor		eax, eax
@@dmaaddrok:
    mov		[DMABuffSize], eax
	and		eax, eax
    jz		@@buffergood
    
	lea		edx, [ecx+eax-1];edx -> last byte of buffer
    mov		eax, edx
    shr		eax, 16
    mov		esi, ecx
    shr		esi, 16
    cmp		ax, si	;does buffer cross a 64 kB boundary?
    jz		@@buffergood

if ?INITDBG
	@DbgOuts <"linear address DMA buffer=">,1
    @DbgOutd ?DMALINEAR,1
    @DbgOuts <10>,1
endif    
    
    mov		[DMABuffStart],?DMALINEAR

;-- align it to the next 64 kB boundary
    
    inc		edx
    xor		dx, dx
    sub		edx, ecx	; some pages left till next physical 64 kb border
	add		ecx, edx

	pushad				; rearrange PTEs
    shr		edx, 12		; bytes -> pages
    mov		eax, edi
    sub		eax, ?SYSLINEAR
    shr		eax, 12
    @GETPTEPTR edi, eax*4+2000h+?SYSPTE	;edi -> curr. PTEs for DMA buffer
    lea		esi, [edi+edx*4]	;esi -> PTEs to be used for DMA buffer
    mov     ebx, esi
    mov		ecx, [DMABuffSize]	;will be page aligned
    jecxz	@@NoDMABuff
    shr		ecx, 12
;--- now move the PTEs to the place for the
;--- DMA buffer and clear them here!
    @GETPTEPTR edi,	2000h+?DMAPTE, 1
@@nextpage1:
	lods    dword ptr [esi]
    and     eax, eax 			;shouldnt happen
    jz      @@ptedone
    stos    dword ptr [edi]
    mov     dword ptr [esi-4],0 
    loop	@@nextpage1

;-- PTEs for DMA buff are moved out, there might be a rest of PTEs 
;-- which has to be moved now "upwards", num PTEs in EDX.
;-- as a result, there is a hole in address space of size "DMABuffSize",
;-- but this is required for linear to phys translation!

	mov		ecx, edx
    jecxz	@@ptedone
    mov		edx, 4
    mov		edi, esi    		;this is the destination
    mov		esi, ebx
@@nextpage2:    
    sub		esi, edx
    sub		edi, edx
    xor		eax, eax
    xchg	eax, [esi]
    mov		[edi], eax
    loop	@@nextpage2
@@ptedone:
	mov		eax, cr3            ;flush TLB
    mov		cr3, eax
if ?INITDBG
	mov		edi, ?DMALINEAR
    mov		ecx, [DMABuffSize]
    shr		ecx, 2
    mov		eax, "BAMD"
    rep		stos dword ptr [edi]
endif
@@NoDMABuff:
	popad
@@buffergood:
    add		edi, [DMABuffSize]
    MOV		[DMABuffStartPhys], ecx

endif

;--- now create the TSS (will begin on a page boundary, since DMA buffer
;--- size is rounded to 4 kB).

;	mov		[dwTss], edi
    mov     eax, edi
	MOV     WORD PTR DS:[EBX+V86_TSS_SEL+2],AX
	SHR     EAX,16
	MOV     BYTE PTR DS:[EBX+V86_TSS_SEL+4],AL
	MOV     BYTE PTR DS:[EBX+V86_TSS_SEL+7],AH

;--- init TSS and the software interrupt redirection bitmap (256 bits)
;--- it is known by Pentium+ cpus only, but doesn't hurt for previous cpus

	mov		edx, edi    
    mov		ecx, 68h/4 + 32/4
    xor		eax, eax
    rep		stos dword ptr [edi]	;clear TSS
    BIG_NOP
    mov		dword ptr [edx+4],V86_TOS
    mov		dword ptr [edx+8],V86_DATA_SEL
if 1    
	mov		eax, [_CR3]   		; save value for CR3 in TSS (not needed)
	mov     [edx+1Ch], eax
endif
    mov		byte ptr [edx+66h],68h+20h ;let 32 bytes space below IO Bitmap

	lea		edx, [edx+68h]

;--- int 67h must be handled by the monitor in any case

	xor		eax, eax
	mov		al, 67h
    bts		[edx], eax
if ?ALTBOOT
	cmp		_AltBoot,0
    jz		@@noaltboot
    mov		ecx, offset INT15_ENTRY
    call	ActivateI15
@@noaltboot:    
endif
if ?A20PORTS
    mov		ecx, offset INT15_ENTRY_EX
    call	ActivateI15
endif
if ?VME
	cmp		_NoVME, 0
    jnz		@@novme
	test	byte ptr dwFeatures, 2	;VME supported?
    jz		@@novme
    @mov_eax_cr4
    or		al,1                ;enable VME
	@mov_cr4_eax
@@novme:
endif

;-- init the io permission bitmap

	mov	esi,SEG IOBM
	shl	esi,4
	mov	eax,OFFSET IOBM
	add	esi,eax

	mov edx, edi
	mov	ecx,IOBM_COPY_LEN
    rep movs byte ptr [edi], [esi]
	BIG_NOP
    mov cx,2000h - IOBM_COPY_LEN
    mov al,0
    REP STOS BYTE PTR [EDI]
	BIG_NOP
    
    mov al, 0FFh
    stos BYTE PTR [EDI]		;the IO bitmap must be terminated by a FF byte

;-- finally load TR

	mov	    ax, V86_TSS_SEL
    ltr     ax
    
if ?A20PORTS
 if 0
 	xor		eax, eax
	test	byte ptr wBiosA20,1	;keyboard controller can switch A20?
    jnz		@@kbdA20
    mov		al, 60h
	btr		[edx], eax
    mov		al, 64h
	btr		[edx], eax
@@kbdA20:
	test	byte ptr wBiosA20,2	;port 92h can switch A20?
    jnz		@@ps2A20
    mov		al, 92h
	btr		[edx], eax
@@ps2A20:
 endif
endif

;--- here CR3, GDT, IDT, TR and LDT all have their final values

	ADD     EDI,15          ; round to para
	AND     EDI,NOT 15

if ?INITDBG
	@DbgOuts <"Jemm386 initialised, edi=">,1
    @DbgOutd edi,1
    @DbgOuts <10>,1
endif    

;---  is XMS pooling on? then direct access to XMS handle table required?

	cmp		_NoPool,0
	jne		@@noxmsarray
	mov	    ecx, [_XMShandleTable]
    movzx	eax, cx
    shr		ecx, 12
    and		cl, 0F0h
    add     ecx, eax
; transfer XMS table info to fixed memory location, assume two dwords
	mov		eax,[ecx+0]
	mov	DWORD PTR [XMS_Handle_Table],eax
	movzx edx,word ptr [ecx+4]
	movzx eax,word ptr [ecx+6]
	shl	eax,4
	add	eax,edx
    cmp eax, 100000h	;is handle array in HMA?
    jb @@nohmaadjust
    @GETPTEPTR edx, 2000h+?HMAPTE, 1	;make a copy of the HMA PTEs
	@DbgOuts <"copy of HMA PTEs at ">,?INITDBG
	@DbgOutd edx, ?INITDBG
	@DbgOuts <10>, ?INITDBG
    mov  ecx,10h
    push eax
    mov eax, 100000h + 7
@@loophmacopy:
	mov  [edx], eax
    add  edx, 4
    add  eax, 1000h
    loop @@loophmacopy
    pop eax
    mov edx, ?HMALINEAR
    lea eax, [eax + edx - 100000h]
@@nohmaadjust:
	mov	[XMS_Handle_Table.xht_array],eax
@@noxmsarray:

;--- EMS/VCPI init

	cmp		[_NoEMS],0
    jz		@@emsactive
    mov		[_NoFrame], 1
    push	ds
    push	RSEG_SEL
    pop		ds
    assume	ds:RSEG
    mov		[device_name+3],'Q'
    pop		ds
    assume	ds:nothing
@@emsactive:    

; alloc 2K status table (8 bytes * 256 handles)
;  each entry is 4 words, first word is either
;  -1 == system handle
;  -2 == free handle
;  -3 == occupied handle
; != the above values, first page number of saved mapping
; 2-4 words are 2nd-4th page number of saved mapping

	call	setstatustable
    
;--- the EMS name table requires at least 1 "SYSTEM" entry

	mov	[EMSNAMETABLE],EDI
	mov	eax,'TSYS'		; store "SYSTEM" as first handle name
	mov	[edi+0],eax
	mov	eax,'ME'
	mov	[edi+4],eax
	add	edi,8

if 0
	cmp	[_NoEMS],0
	jnz @@nonamesetup
endif

; allocate room for handle names (8*256 = 2K) and zero them

	mov	ecx,8*MAX_HANDLES/4
	xor	eax,eax
	rep	stos DWORD PTR [edi]
	BIG_NOP
@@nonamesetup:

	@DbgOuts <"EMS status table=">,?INITDBG
	@DbgOutd EMSSTATUSTABLE,?INITDBG
	@DbgOuts <", name table=">,?INITDBG
	@DbgOutd EMSNAMETABLE,?INITDBG
    @DbgOuts <10>,?INITDBG

; allocate and -1 store space for EMS page allocation pointer/descriptors
;  dword per page, points into EMS/VCPI block which controls page allocation
; allocate 4 bytes per 16K XMS total, 128K-4 max (no more than 32K-1 pages controlled)
;  page descriptor high word == 64-byte EMS/VCPI allocation block count from start, RELATIVE 1!
;  page descriptor low word ==  half-byte offset of allocation page (2 * 48)
;   within allocation block, not including system bytes

	mov ecx,[_MAXEMSPAGES]
	mov	edx,ecx		; save as page count
	mov	[EMSPageAllocationStart],edi
	mov	eax,-1
	rep	stos DWORD PTR [edi]
	BIG_NOP
	mov	[EMSPageAllocationEnd],edi

	@DbgOuts <"EMS page alloc start/end=">,?INITDBG
	@DbgOutd [EMSPageAllocationStart],?INITDBG
	@DbgOuts <"/">,?INITDBG
	@DbgOutd edi,?INITDBG
    @DbgOuts <10>,?INITDBG

; alloc page count-sized table flagging owner handle, if any, of EMS page

	MOV     [EMSPAGETABLE],EDI

	mov	ecx,edx				; page count

	MOV     AL,FREEPAGE_ID		    ; all currently free
	REP     STOS BYTE PTR [EDI]	    ; REP STOSB (addr. size = 32)
	BIG_NOP

; allocate and zero ((XMS total / 1.5M) + 128) * 64 bytes for pool
;  allocation table entries
; 1.5M is pool allocation maximum memory control, 128 is max number XMS handles,
;  64 is pool block size

	add	edi,63
	and	edi,NOT 63		; 64-byte align tables for proper location by count
	mov	[PoolAllocationTable],edi

	mov	eax,[_MAXMEM16K]
    cdq
	mov	ecx,1536/16
	div	ecx
    inc eax				; round up, not down
    cmp [_NoPool],0
    jnz @@isnopool
	movzx ecx,[XMS_Handle_Table.xht_numhandle]	;was 127 previously
    add eax, ecx
@@isnopool:    
	shl	eax,6-2			; 64 bytes/16 dwords each
	mov	ecx,eax

	xor	eax,eax
	rep	stos DWORD PTR [edi]
	BIG_NOP

	mov	[PoolAllocationEnd],edi

if ?INITDBG
	@DbgOuts <"pool start/end=">,1
	@DbgOutd PoolAllocationTable,1
	@DbgOuts <"/">,1
	@DbgOutd PoolAllocationEnd,1
	@DbgOuts <10>,1
endif    

;--- convert EDI back into a physical address
;--- use the page directory for the conversion

    mov		eax, edi
    sub		eax, ?SYSLINEAR
    shr		eax, 12
    @GETPTE eax, 2000h+eax*4+?SYSPTE
    and     ah, 0F0h
    mov		al, 0
    
	and		edi, 0FFFh
    
;--- now check if current phys address is < DMA buffer    
;--- if yes, all pages below must be skipped and are wasted
;--- since the EMS/VCPI memory managment needs a contiguous block
;--- of physical memory as input.

    mov		ecx, [DMABuffStartPhys]
    cmp		eax, ecx
    jnc		@@abovedma
if ?INITDBG
	@DbgOuts <"must waste space, phys end of monitor=">,1
	@DbgOutd eax,1
	@DbgOuts <" is below DMA buff=">,1
	@DbgOutd ecx,1
    @DbgOuts <10>,1
endif
    add		ecx, [DMABuffSize]		;get physical end of the DMA buff
    mov		eax, ecx
    xor		edi, edi
@@abovedma:    
    add		edi, eax
	mov eax, [_EMM_MEMORY_END]

	cmp eax, EDI
	jc	@@killvcpi		;we run out of memory!!!

if ?INITDBG
	@DbgOuts <"end of monitor data, physical=">,1
	@DbgOutd edi,1
	@DbgOuts <", end of XMS memory block=">,1
	@DbgOutd eax,1
	@DbgOuts <10>,1
endif    

; force 4K alignment for EMS/VCPI fixed pages and UMB's

	ADD     EDI,4095                ; Round to the next page border
	AND     DI,NOT 4095
	MOV     [FIRSTPAGE],EDI	    	; Mark the beginning of first EMS-page
    mov		ebp, edi				; save it in ebp
    
    sub		eax, edi
    jnc		@@isvalid
    xor		eax, eax
@@isvalid:    
	shr	eax,14			; compute 16K pages of memory available
    
	cmp	[_NoPool], 0
	je	@@setblocks

if ?INITDBG
	@DbgOuts <"pool sharing is off, mem=">,1
	@DbgOutd eax,1
	@DbgOuts <10>,1
endif    

;-- pool sharing is off, rest of block is used for EMS/VCPI
;-- make sure current values of MAXMEM16 and MAXEMSPAGES
;-- are not too high

	cmp eax,[_MAXMEM16K]
    jnc @@isnodecrease
	mov	[_MAXMEM16K], eax
@@isnodecrease:
	mov ecx,[_MAXEMSPAGES]	;value is 7FFF max
    cmp eax, ecx
    jnc @@isnodecrease2
	mov	[_MAXEMSPAGES],eax
@@isnodecrease2:   
	mov eax, [_MAXMEM16K]

; if pool sharing, first pool allocation block points to remainder of initial
;  XMS allocated memory so that UMB code still has linear==physical memory

; setup control blocks (pool allocation blocks which are fixed) for the
;  fixed EMS/VCPI allocations
; start and end adjustments leave as initialized zero value since the memory
;  is originally 4K aligned, via edi alignment above

@@setblocks:
	mov	ecx,eax			; count of available 16K blocks
	mov	esi, [PoolAllocationTable]
	movzx edx, [_XMS_CONTROL_HANDLE]

	@assumem esi, LPPOOL_SYSTEM_INFO_STRUC

@@fixblkloop:
	mov	[esi].psi_descptr,edx
	mov	eax,ecx
	cmp	eax,POOLBLOCK_ALLOCATION_SPACE * 2	; 92 == 1536 / 16
	jb	@@fix2
	mov	eax,POOLBLOCK_ALLOCATION_SPACE * 2
@@fix2:
    
	sub	ecx,eax

if 0;?INITDBG
	@DbgOuts <"pool desc init, block=">,1
	@DbgOutd esi,1
	@DbgOuts <" 16k pg=">,1
	@DbgOutd eax,1
	@DbgOuts <" remaining=">,1
	@DbgOutd ecx,1
	@DbgOuts <10>,1
endif    
    mov bl,al
	mov	[esi].psi_16kmax,al
	mov	[esi].psi_16kfree,al
	shl	eax,2
	mov	[esi].psi_4kfree,ax
    test bl,1		;was it uneven?
    jz  @@iseven
    inc bl						;add one page since pool must have an even
	mov	[esi].psi_16kmax,bl		;number for max
    movzx ebx, bl
    shr ebx, 1
    mov byte ptr [esi+ebx+POOLBLOCK_SYSTEM_SPACE],0F0h	;last page is "used"
@@iseven:
	mov	ebx,edi
	shr	ebx,10			; convert byte address to K
	mov	[esi].psi_addressK,ebx
	shl	eax,12			; convert 4K count to bytes
	add	edi,eax			; okay to increment edi free memory ptr, it's not used further

; never expand these blocks, the descriptor holds a real XMS handle rather than
;  an XMS pseudo-handle/pointer taken from the XMS handle array
	or	[esi].psi_flags,POOLBLOCK_FLAG_DONTEXPAND

	and ecx, ecx
	jz	@@fixblkdone  	; no more memory to distribute to allocation blocks
    
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@fixblkloop
    jmp @@fixblkdone

@@killvcpi:

; ran out of memory, shouldn't happen, avoid disaster by shutting off vcpi,
;  pool-sharing, and leaving max/available memory at 0

if ?INITDBG
	@DbgOuts <"out of memory condition on init!, EMM_MEMORY_END=">,1
    @DbgOutd _EMM_MEMORY_END,1
	@DbgOuts <" EDI=">,1
    @DbgOutd edi,1
	@DbgOuts <10>,1
endif    

	mov	[_NoVCPI], 1	
	mov	[_NoEMS], 1	
	mov	[_NoPool], 1

@@fixblkdone:

	@assumem esi, nothing

if ?INITDBG
	@DbgOuts <"EMS/VCPI memory handling initialised, MAXMEM16K=">,1
	@DbgOutd [_MAXMEM16K],1
    @DbgOuts <10,"jump to v86-mode",10>,1
endif    

;-- calc the position of the PTEs for EMS page frame in page table 0
;-- Saves the need for doing later returning calculations

compute_anchor:
	MOVZX   EAX,[_FRAME]    ; Segmentaddress of the Frame
	SHR     EAX,8           ; transform in a page no. (E000 -> E0)
	cmp 	[_NoFrame], 0
    jz		@@emswithframe
    mov		[bEmsPhysPages], 0
    mov		al,0A0h         ; for safety reasons if no frame is given
@@emswithframe:    
    @GETPTEPTR eax, EAX*4+1000h
	MOV     [FRAMEANCHOR],EAX
    
;-- clear all dirty + accessed flags in page table 0

	add		esi, 1000h
	mov		ecx, 1000h/4
    mov		al, not (20h or 40h)
@@nextpte:
    and		[esi],al
    add		esi, 4
    loop	@@nextpte
    xor		esi, esi	;clear hiwords of esi, edi
    xor		edi, edi

	mov		eax, offset V86_LEN - 1
	MOV     word ptr [GDT+V86_CODE_SEL+0],AX
	MOV     word ptr [GDT+V86_DATA_SEL+0],AX

    
; Now finally all is prepared, so jump to v86 mode with an IRETD
; create Stackframe for the switch to 8086-mode

    mov		edx, SEG _KEEP
	mov		ecx, DGROUP
    mov		ebx, offset DGROUP:exe_stacktop - 20h
    mov		eax, OFFSET _KEEP
    
	PUSH    0            ; GS 
	PUSH    0            ; FS
	PUSH    edx          ; DS (=_TEXT16)
	PUSH    ecx          ; ES (=DGROUP)
	PUSH    ecx          ; SS (=DGROUP)
	PUSH	ebx          ; ESP
	PUSH    00023000H    ; EFL (VM=1, IOPL=3, IF=0)
	PUSH    edx			 ; CS
	PUSH    eax          ; EIP
    
	CLTS       ; TS-Flag Clear (Task Switch) absolutely
               ; thus the next x87-command without INT 7 is being executed

	IRETD      ; this command switches to v86 mode
    
    assume  DS:NOTHING
    assume  FS:NOTHING
    assume  GS:NOTHING
                                     
GO_PROTECTED ENDP

	align 16

INITSEG ENDS

;
; The actual Monitor for the virtual 8086-Modus. This 
; is moved to Extended Memory
;

V86SEG SEGMENT

    assume  ES:NOTHING,FS:NOTHING,GS:NOTHING
    
	assume CS:_TEXT32
	assume SS:V86SEG

	align 16
    
if ?CODEIN2PAGE
	ORG 1000h	;locate code in 2. page (so data and code are separated)
endif

if ?VCPI

;--- the following 3 procs must be located in the first page
;--- of the V86SEG segment ! If this is no longer possible,
;--- the VCPI shared address space has to be increased.

; V86-entry only
; AX=DE0C: VCPI switch from V86 mode to protected mode
;  entry esi -> entry linear address of data structure
;  entry esp -> ecx, edi, esi
;  exit GDTR, IDTR, LDTR, TR loaded for client
;    transfer control to client
;    modify only eax, esi, DS, ES, FS, GS
;

VCPI_V86toPM	PROC	NEAR

	@assumem esi,<LPSWSTR>

	mov	eax,[esi].SW_CR3	; set client's context *first*
	mov	cr3,eax

	mov	eax,[esi].SW_GDTOFFS    ; set up client's GDT
	lgdt	fword ptr [eax]
	mov	eax,[esi].SW_IDTOFFS    ; set up client's IDT
	lidt	fword ptr [eax]
	lldt	[esi].SW_LDTR		; set up client's LDT
	mov	eax,[esi].SW_GDTOFFS
	movzx ecx,[esi].SW_TR
	mov	eax,[eax+2]             ; EAX == linear base address of client's GDT
	and	BYTE PTR [eax+ecx+5],NOT 2	; clear task busy bit in TSS descriptor
	ltr	[esi].SW_TR				; set up client's TSS
	mov ecx,[esp+0]
;    add esp, 12
	jmp	FWORD PTR [esi].SW_EIP	; jump to client's entry point

	@assumem esi, nothing

VCPI_V86toPM	ENDP

	align 4
    
; protected mode-entry only
; AX=DE0C: VCPI switch from protected mode to V86 mode
;  inp: SS:ESP set for IRETD to V86, PM far call return address to discard
;       ds -> linear address space from 0de01h call
;  exit CR3, GDTR, IDTR, LDTR, TR loaded for server,
;    SS:ESP and all segment registers loaded with values on stack
;    transfer control to client
;    modify only eax
;
VCPI_PMtoV86	PROC	NEAR
	cli					; client should have disabled interrupts, but not all do
if ?SAFEMODE
	mov eax,ss
    lar eax,eax
    test eax, 400000h
    jnz @@is32bit
    movzx esp,sp
@@is32bit:    
endif

;-- for speed reasons the following lines are a bit mixed.
;-- but what is done is simple:
;-- 1. switch to host context (SS:ESO will stay valid, stack is in 1. MB)
;-- 2. load IDTR, GDTR, LDTR, TR for VCPI host
;-- 3. clear task busy bit in host's TSS descriptor
;-- 4. clear task switch flag in CR0
;-- 5. IRETD will switch to v86 mode

	add	esp,8			; eat far call return address

	mov	eax,cs:[_CR3]

	mov	DWORD PTR [esp+8],23002h	; set EFLAGS: VM, IOPL=3, IF=0

	mov	cr3,eax

	lidt	fword ptr cs:[IDT_PTR]
    
	mov	eax,dword ptr cs:[GDT_PTR+2]

	lgdt	fword ptr cs:[GDT_PTR]		;-- set up host GDT

	and	BYTE PTR [eax+5+V86_TSS_SEL],NOT 2

	clts

	xor	eax,eax
	lldt ax
    
	mov	al,V86_TSS_SEL
	ltr	ax
    
	iretd

VCPI_PMtoV86	ENDP


;-- put the VCPI PM entry at the very beginning. This entry
;-- must be in the shared address space, and just the first page
;-- is shared!
;-- entry for EMM routines called from protected mode

	align 4

VCPI_PM_ENTRY	PROC

	cmp	ax,0de0ch		; see if switch from protected mode to V86 mode
	je	VCPI_PMtoV86	; yes, give it special handling
	pushfd
	push	ds			; have to save segments for p-mode entry
	push	es
	push	fs

    push	esi
    push	edi
	push	ecx

	mov	ecx,cs
	add	ecx,8			; V86_DATA_SEL
	mov	fs,ecx
    mov esi,ecx
	add	ecx,8			; FLAT_DATA_SEL
	mov	ds,ecx
	mov	es,ecx

;--- segment registers should *all* be set *before* switching contexts!
;--- why? because GDT of client might be located in unreachable space.
;--- get client SS:ESP + CR3, switch to host stack and context

    mov		edi, ss
	mov		ecx, esp
	cli					; don't allow interruptions
    mov		ss, esi
	mov		esp, V86_TOS
	push	edi			;save client SS:ESP
    push	ecx
	mov		esi, [_CR3]
	mov		edi, cr3
	mov		cr3, esi
    push	edi			;save client CR3

if ?VCPIDBG
	sub		esp,8
    sgdt	[esp]
    lgdt    fword ptr [GDT_PTR]
    mov     cx, V86_DATA_SEL
	mov		fs, ecx
    mov     cx, FLAT_DATA_SEL
	mov		ds, ecx
    mov		es, ecx
endif

	cld

if ?VCPIDBG
	@DbgOuts <"VCPI call in PM, ax=">,1
    @DbgOutw ax,1
    @DbgOuts <10>,1
endif

	cmp	ah,0deh
	jne	vcpi_INV_CALL	; only allow VCPI calls from protected mode interface

; other than de0ch, don't allow anything other than de03h,de04h,de05h from PM
	cmp	al,3
	jb	vcpi_INV_CALL
	cmp	al,5
	ja	vcpi_INV_CALL

	movzx	ecx,al
	call 	cs:[VCPI_CALL_TABLE+ECX*4]

PM_ret:
if ?VCPIDBG
	lgdt fword ptr [esp]
    add esp,8
endif
	pop	ecx
if 1    
	mov	cr3,ecx
    lss esp,[esp]
else    
    pop esi
    pop edi
	mov	cr3,ecx			; restore client context *first*
    mov ss, edi
    mov esp, esi		; then restore client SS:ESP
endif    
	pop	ecx
    pop edi
    pop esi
	pop	fs
	pop	es
	pop	ds
	popfd
	retf
vcpi_INV_CALL:
	mov	ah,8fh			; use bad subfunction code, per VCPI spec
    jmp PM_ret
VCPI_PM_ENTRY	ENDP

endif    

	align 4
    
EMS_CALL_TABLE label dword
	Dd EMS_GET_STATUS			; 40h
	Dd EMS_GET_PAGE_FRAME_ADDRESS		;41h
	Dd EMS_GET_UNALLOCATED_PAGE_COUNT	;42h
	Dd EMS_ALLOCATE_PAGES		; 43h
	Dd EMS_MAP_HANDLE_PAGE		; 44h
	Dd EMS_DEALLOCATE_PAGES		; 45h
	Dd EMS_GET_VERSION			; 46h
	Dd EMS_SAVE_PAGES			; 47h
	Dd EMS_RESTORE_PAGES		; 48h
	Dd EMS_NOT_IMPL				; 49h (get io port addresses)
	Dd EMS_NOT_IMPL				; 4ah (get translation array)
	Dd EMS_GET_OPEN_HANDLES_COUNT		; 4bh
	Dd EMS_GET_NR_OF_ALLOCATED_PAGES	; 4ch
	Dd EMS_GET_ALLOCATED_PAGES	; 4dh
	Dd EMS_GET_SET_PAGE_MAP		; 4eh	
	Dd EMS4_GET_SET_PARTIAL_PAGE_MAP	; 4fh (4: get/set partial page map)
	Dd ems4_map_multi			; 50h
	Dd ems4_realloc				; 51h
	Dd ems4_attribute			; 52h
	Dd ems4_handle_names		; 53h
	Dd ems4_get_handle_info		; 54h
	Dd ems4_alter_map_jump  	; 55h (4: alter page map and jump)
	Dd ems4_alter_map_call  	; 56h (4: alter page map and call)
	Dd ems4_memory_region		; 57h
	Dd ems4_get_mappable_info	; 58h
	Dd ems4_get_config			; 59h
	Dd ems4_allocate_pages		; 5ah
	Dd EMS_NOT_IMPL				; 5bh (4: alternate map register set)
	Dd EMS_NOT_IMPL				; 5ch (4: prepare EMS for warm boot)
	Dd EMS_NOT_IMPL				; 5dh (4: enable/disable OS functions)

if ?VCPI

VCPI_CALL_TABLE label dword
	Dd VCPI_Presence	; 0
	Dd VCPI_GetInterface
	Dd VCPI_GetMax		; 2
	Dd VCPI_GetFree
	Dd VCPI_Allocate4K	; 4
	Dd VCPI_Free4K
	Dd VCPI_GetAddress	; 6
	Dd VCPI_GetCR0
	Dd VCPI_ReadDR		; 8
	Dd VCPI_WriteDR
	Dd VCPI_GetMappings	; 0ah
	Dd VCPI_SetMappings
;	Dd VCPI_V86toPM		; 0ch
endif

if ?VDS

VDS_CALL_TABLE label dword
	Dd vds_reserved		; 00
	Dd vds_reserved		; 01
    Dd vds_version		; 02 get version
    Dd vds_lock			; 03 lock region, ES:DI -> DDS  
    Dd vds_unlock		; 04 unlock region, ES:DI -> DDS
    Dd vds_scatterlock	; 05 scatter lock, ES:DI -> EDDS
    Dd vds_scatterunlock; 06 scatter unlock, ES:DI -> EDDS
    Dd vds_reqbuff		; 07 request DMA buffer, ES:DI -> DDS
    Dd vds_relbuff		; 08 release DMA buffer, ES:DI -> DDS
    Dd vds_copyinbuff	; 09 copy into DMA buffer, ES:DI -> DDS
    Dd vds_copyoutbuff	; 0A copy out of DMA buffer, ES:DI -> DDS
    Dd vds_disabletrans	; 0B disable DMA translation
    Dd vds_enabletrans	; 0C enable DMA translation

endif
    
;--- stack frame V86 monitor

V86FRAME struc
_ECX	dd ?	;+0
_EBX	dd ?	;+4
_EAX	dd ?	;+8
_Ret	dd ?	;+12 return address to skip
_IP		dw ?	;+16
		dw ?
_CS		dw ?    ;+20
		dw ?
union
_EFL	dd ?
struc
_FL 	dw ?	;+24
		dw ?
ends
ends
union        
_ESP	dd ?
struc
_SP		dw ?	;+28
		dw ?
ends        
ends        
_SS		dw ?	;+32
		dw ?
_ES		dw ?	;+36
		dw ?
_DS		dw ?	;+40
		dw ?
_FS		dw ?	;+44
		dw ?
_GS		dw ?	;+48
		dw ?
V86FRAME ends        

;
; Within this routine the interrupts and exceptions handled as simply the
; interrupt call into the virtual 8086-Modus is reflected. Exceptions form
; only the error INTs 08h to 0Fh, which must be examined for it, if they are
; generated from hardware.
;
; on entry, the CPU has set DS,ES,FS and GS to NULL!

	align 4
    
V86_MONITOR PROC NEAR
	sub		esp,2	;account for that we are called with a 16-bit opcode!
	PUSH    EAX
	PUSH    EBX
	PUSH    ECX

;;	movzx   esp,sp	;we are on a 32bit stack now!
                    ;In fact we don't know for sure what size the stack is.
                    ;There might exist other code which intruded in the  
                    ;monitors address context and modified GDT/IDT entries

					; ECX is a offset into interrupt table
					; or intno*4
	MOVZX   ECX,word ptr [ESP].V86FRAME._Ret+2
	SUB     ECX,OFFSET INT_TABLE+4	; return address as unique identifier
;;    AND     cl,0FCh	            ; for the corresponding IDT entry!
;;								; (ECX/4 = interrupt number)

; Three cases are considered:
; - an INT executed in V86 mode     SP == V86_TOS - 34h
; - an exception in V86 mode        SP == V86_TOS - 38h
; - an EXC/IRQ (HLT) in v86 monitor SP <  V86_TOS - 38h

	CMP     ESP,V86_TOS-38H     ; What did actually happen?!
	JZ      @@V86_EXCEPTION     ; an exception in V86 mode
	JB      @@IRQOREXC          ; Monitor is being interrupted (IRQ or EXC)

	MOV     BX,FLAT_DATA_SEL	; DS has full access
	MOV     DS,EBX				; to all Bytes.

	MOVZX   EBX, [ESP].V86FRAME._SS	; linear address of 86er-Stacks
	MOVZX   EAX, [ESP].V86FRAME._SP ; use ONLY SP, HIWORD(ESP) no usable
	SHL     EBX,4
    SUB		AX, 6                   ; Create space for IRET frame
	ADD 	EBX,EAX
	MOV     [ESP].V86FRAME._SP,AX

; copy Interrupt frame down

	MOV     EAX,dword ptr [ESP].V86FRAME._CS
    SHL		EAX, 16
	MOV     AX,[ESP].V86FRAME._IP
	MOV     [EBX+0],EAX
	MOV     EAX,[ESP].V86FRAME._EFL
	MOV     [EBX+4],AX

; always clear interrupt flag, irrespective of whether hardware IRQ
;; reflecting an IRQ to whichever handler the V86 task "IDT" offers)...

; Clear IF and TF as it is done in real-mode

	AND     BYTE PTR [ESP].V86FRAME._FL+1, NOT (1 or 2)

; route call to vector in real-mode IVT

	MOV     EAX,[ECX]
	MOV     [ESP].V86FRAME._IP,AX
    SHR     EAX, 16
	MOV     [ESP].V86FRAME._CS,AX

	POP     ECX				; Clean everything properly
	POP     EBX
	POP     EAX

	add     ESP,4   		; Correct old call-address

	IRETD	                ; Return to virtual 86-Modus
                            ; (only) via IRETD !
;
; Remove EFLAG, CS, IP, that are saved by the Interrupt again
; * This is called when an exception is generated by the monitor itself,
; * which we recognize by how much context is on stack (v86
;

;-- IRQ or exception in ring 0
;-- an exception may be with or without error code
;-- an IRQ will have offset "BEHINDHLT" as EIP and V86_CODE_SEL as CS

	align 4

@@IRQOREXC:

if ?V86DBG
	@DbgOuts <"V86 IRQ/EXC, cx=">,1
    @DbgOutw cx,1
	@DbgOuts <" CS=">,1
    mov ax, word ptr [esp+3*4+4+4]
    @DbgOutw ax,1
    @DbgOuts <10>,1
endif
	cmp		dword ptr [esp+3*4+4],offset BEHINDHLT
    JNZ		ring0_exc
	cmp		word ptr [esp+3*4+4+4],V86_CODE_SEL
    JNZ		ring0_exc

;-- ESP before halt -> IRETD frame [eip, cs, efl, esp, ss, es, ds, fs, gs)
;-- now 7 dword are additionally pushed: ECX, EBX, EAX, ret, EIP, CS, EFL

if ?HLTDBG
	@DbgOuts <"HLT triggered, int*4=">,1
	@DbgOutw cx,1
	@DbgOuts <10>,1
endif
	MOV     ECX, [ESP+3*4]       ; put interrupt offset in correct position
    MOV     [ESP+6*4],ECX
    mov		ecx, [ESP]           ; restore ECX
    add		esp, 6*4+2           ; adjust stack  
    jmp		V86_MONITOR          ; and restart
	align 4

;--- print a string. Since ring3 code cannot be "called", the
;--- ring3 stack has to be used to save the state.
;--- 4 words are saved on v86-SS:SP:
;--- [SP+0]: current SI
;--- [SP+2]: current AX
;--- [SP+4]: LOWWORD(retaddr)
;--- [SP+6]: HIGHWODR(retaddr)
;--- DS=FLAT

printstr proc    
	movzx	ebx, word ptr [esp+8].V86FRGP.gSP
	movzx	ecx, word ptr [esp+8].V86FRGP.gSS
    sub		bx, 8
    shl		ecx, 4
    mov		word ptr [esp+8].V86FRGP.gSP, bx
    add		ebx, ecx

    mov		[ebx+0], si
    mov		[ebx+2], ax
    pop		dword ptr [ebx+4]

	sub		ebx, size v86_call
    pop		eax
    mov		si,ax
    mov		[ebx].v86_call.dwRetAddr, offset @@nextchar
@@nextchar:
	lods	byte ptr ss:[si]
    and		al,al
    jz		@@done
    mov		byte ptr [ESP].V86FRGP.gEAX,al
    sub		[esp].V86FRGP.gSP, size v86_call
    
    mov  	cx, offset RSEG:v86int29
    mov     ax, RSEG
    mov		word ptr [ebx].v86_call.dwBP+0, offset BPBACK	;store breakpoint
    mov		word ptr [ebx].v86_call.dwBP+2, ax
    xchg	cx, word ptr [esp].V86FRGP.gIP
    xchg	ax, word ptr [esp].V86FRGP.gCS
    and		dword ptr [esp].V86FRGP.gFL+1, not 1		;reset TF
    mov		word ptr [ebx].v86_call.dwOldCSIP+0, cx
    mov		word ptr [ebx].v86_call.dwOldCSIP+2, ax
    
	pop		ecx
    pop		ebx
    pop		eax
	add     ESP,2*4 		; skip call return + error code
    IRETD
@@done:
    add  	[ESP].V86FRGP.gSP, 8
	mov		si,[ebx+size v86_call+0]
	mov		ax,[ebx+size v86_call+2]
    mov		word ptr [esp].V86FRGP.gEAX, ax
	jmp		dword ptr [ebx + size v86_call + 4]

printstr endp

EXCFR struc
	PUSHADS <>
	V86FRGP <>
EXCFR ends

;--- handle exc 06 in v86-mode

do_exc_ex proc
	call	SimIRET
    mov		[esp].V86FRGP.gRet,6
    pop		ecx
    pop		ebx
    pop		eax
    jmp		errorcode_pushed
do_exc_ex endp

;--- handle exceptions in ring0 protected-mode

ring0_exc:

;--- handle exceptions (protected-mode and v86-mode)
;--- display current register set
;--- if it is caused by external protected-mode code, display REBOOT option
;--- else jump to v86-mode and try to abort current PSP

do_exc proc
    mov		eax,ss					; the stack size is not known
	lar		eax, eax
    test	eax,400000h
    jnz		@@is32
	movzx	esp,sp					; make sure ESP is valid
@@is32:    
    shr		ecx,2
    mov		[esp].V86FRGP.gRet, ecx
    pop		ecx
    pop		ebx
    pop		eax

	cmp		dword ptr [esp],8
    jnc		errorcode_pushed
    push	dword ptr [esp]
;    mov		dword ptr [esp+4],0
if ?MASM
errorcode_pushed::
else
errorcode_pushed:
endif
    push	eax
    push	ebx
    push	ecx
    pushad
    mov		ebp,esp

	push	FLAT_DATA_SEL
    pop		ds
	push    V86_DATA_SEL
	pop     ES
    cld

    test	byte ptr [ebp].EXCFR.gFL+2,2	;V86 mode?
    jnz		@@isv86
    lar		eax, dword ptr [ebp].EXCFR.gCS
    and 	ah,60h
    jnz 	@@isring3
    mov		ebx, ss
    lea		ecx,[ebp+32+3*4+4+4*4]
    jmp		@@isring0
@@isv86:    
	movzx   esi,[ebp].EXCFR.gCS
    shl		esi,4
    add		esi,[ebp].EXCFR.gIP
    mov		cl,8
    mov		edi,offset exc_csip
@@nextitem:
    lods	byte ptr [esi]
    call	byteout
    inc		edi
    dec		cl
    jnz		@@nextitem
    push 	offset exc_v86segregs			;render V86 segment registers
    call 	renderitems
@@isring3:    
	mov		ebx,dword ptr [ebp].EXCFR.gSS
    MOV     ecx,dword ptr [ebp].EXCFR.gSP
@@isring0:    
    push	ebx 					; ebp-4 == SS
    push	ecx						; ebp-8 == ESP
    mov		eax, cr0
    push	eax						; ebp-12
    mov		eax, cr2
    push	eax						; ebp-16
    
    push	offset exc_format
    call	renderitems

    mov		esp,ebp

    mov		eax,ss
    cmp     ax,V86_DATA_SEL
    jnz     @@external_exc
    cmp		byte ptr [ebp].EXCFR.gRet, 0Ch
    jz      @@external_exc
    cmp		[_INIT_DONE],0
    jnz 	@@printexcstr
@@external_exc:
	mov		es:[exc_reboot],10
@@printexcstr:
    popad
	pop		ecx
    pop		ebx
    pop		eax
    push	es
	pop		ss
	mov		esp, V86_TOS - 38h + 3*4
    push	eax
    push	ebx
    push	ecx

	push	offset exc_str
	call	printstr			;this will call v86-mode
    cmp		ss:[exc_reboot],10	;dont use ES anymore
	jz		@@waitkey
    test	byte ptr [esp].V86FRGP.gFL+2,2	;V86 mode?
    jz		@@nov86
	push	offset exc_str2
	call	printstr
@@nov86:    
    mov 	word ptr [esp].V86FRGP.gIP, offset Int06_WaitKey
    mov 	word ptr [esp].V86FRGP.gCS, RSEG
	pop		ecx
    pop		ebx
    pop		eax
	add     ESP,2*4 		; skip call return + error code
    IRETD
    
@@waitkey:    
	in		al,64h
    test	al,1
    jz		@@waitkey
    in		al,60h
    cmp		al,1Ch		;RETURN make code?
    jnz		@@waitkey
    xor 	ecx,ecx
    push	ecx
    push	ecx
    lidt	fword ptr [esp]	;load IDT with limit 0
    int		3				;this will cause a triple fault and reboot
do_exc endp

v86_exc06_fatal proc

	push 	offset exc_str3
    call 	printstr
	pop		ecx
    pop		ebx
    pop		eax
	add     ESP,2*4 		; skip call return + error code
    IRETD
    
v86_exc06_fatal endp

;--- exception in V86 mode
    
@@V86_EXCEPTION:

if 0; ?V86DBG  ;this happens quite often, so mostly not good to activate
	@DbgOuts <"exception in v86-mode, cx=">,1
    @DbgOutw cx,1
    @DbgOuts <", CS:IP=">,1
    mov ax, [esp].V86FRGP.gCS
    @DbgOutw ax,1
    @DbgOuts <":">,1
    mov eax, [esp].V86FRGP.gIP
    @DbgOutw ax,1
    @DbgOuts <10>,1
endif
	MOV     AX,FLAT_DATA_SEL
	MOV     DS,EAX
    assume	DS:NOTHING

	CMP     ECX,0DH*4            ; general protection exception?
	JNZ     @@V86_TEST_EXC	     ; no, check further

	MOVZX   EAX,[ESP].V86FRGP.gCS  ; examine whether a HLT-command
	SHL     EAX,4                  ; has caused the Privilege fault
	ADD     EAX,[ESP].V86FRGP.gIP  ; EAX = linear CS:IP

;
; examination of the error releasing instruction
;
; check which command triggered the GPF:

; - might be an I/O command. I/O only causes GPF for masked ports.
;   Currently the DMA, KBC and P92 ports are trapped. These ports
;   have no 32 registers, so only 8/16bit IN/OUT is trapped. String 
;   IN/OUT is not trapped either, but makes no sense in these cases.
; - HLT opcode, which then might be:
;   + a "Breakpoint" if CS is RSEG. Do special handling if the offset
;     is "known" as well.
;   + other HLTs. Is handled by doing HLT for the user in the monitor.
; - other privileged opcode. Some are emulated (mov CRx, reg ...),  
;   some are not and then are just translated to an int 6, illegal opcode,
;   which is then reflected to the V86 task!).
;
	MOV     BL,[EAX]                ; check opcode
	cmp		BL, ?BPOPC
    jz		@@Is_BP					; breakpoint?
if ?BPOPC ne 0F4h
	CMP     BL,0F4H                 ; HLT-
	JZ      @@Is_Hlt                ; command ?
endif

if ?SB
; see if SoundBlaster INT 3 forced to 1ah error code GPF

	CMP	[ESP].V86FRGP.gErrc,1ah
	jne	@@notsb
	test [_bV86Flags],V86F_SB
	je	@@notsb					; SB option not turned on
	pop	ecx
	pop	ebx
	pop	eax
	add	esp,4+4 				; discard excess GPF error code
	inc	[ESP].IRETDV86.vIP      ; increment IP past INT 3 instruction
if ?MASM    
    push word ptr LOWWORD(OFFSET INT_TABLE)+3*4+4	; set configuration as if INT 3 occurred
else
	sub esp,2
	push eax
    mov eax, OFFSET INT_TABLE+3*4+4
    mov [esp+4], ax
    pop eax
endif
	jmp	V86_MONITOR
@@notsb:
endif

if ?EMUDBG
	@DbgOuts <"Opcode ">,1
    @DbgOutb BL,1
    mov bh, [eax+1]
    @DbgOuts <" ">,1
    @DbgOutb BH,1
    mov bh, [eax+2]
    @DbgOuts <" ">,1
    @DbgOutb BH,1
    @DbgOuts <" caused GPF at ">,1
    @DbgOutd eax,1
    @DbgOuts <10>,1
endif

	cmp		bl,0fh                  ; check if potentially mov <reg>,cr#
	je		@@ExtendedOp

	CMP     BL,0E4H                 ; IN/OUT ??
	JB      @@V86_EXC06
	CMP     BL,0E7H
	JBE     @@DoIO_Im
	CMP     BL,0ECH
	JB      @@V86_EXC06
	CMP     BL,0EFH
	JBE     @@DoIO_DX

	; no handling for 32bit IN/OUT or other potential GPF causes

@@NIX:
	JMP     @@V86_EXC06				; else complain!

@@ExtendedOp:
	mov bh,[eax+1]
	cmp	bh,9
	je	@@wbinvd
	cmp	bh,8
	je	@@invd
	cmp	bh,30h
	je	@@wrmsr
	cmp	bh,31h
	je	@@rdtsc
	cmp	bh,32h
	je	@@rdmsr

	cmp	bh,20h
	jb	@@NIX		; not an opcode we emulate
	cmp	bh,23h
	ja	@@NIX

; opcodes 0F 20 xx to 0F 23 xx emulated via self-modifying code

;	push V86_DATA_SEL
;	pop	fs
;   assume FS:V86SEG
	mov	al,[eax+2]	; get third byte of opcode
	mov	WORD PTR ss:[EmuInstr+0],bx
	mov	BYTE PTR ss:[EmuInstr+2],al
	pop	ecx
	pop	ebx
	pop	eax
    call RunEmuInstr
	add	WORD PTR [esp+8],3	; jump over emulated instruction
	jmp	@@Extret

;   assume FS:nothing

@@invd:
	invd				; 0f 08 is invd opcode
	jmp	@@invdshare
@@wbinvd:
	wbinvd				; 0f 09 is wbinvd opcode
@@invdshare:
	pop	ecx				; restore registers
	pop	ebx
	pop	eax
	jmp	@@twoeat

@@wrmsr:
	pop ecx				; restore ecx
    pop ebx
    pop eax
	@wrmsr
	jmp	@@twoeat

; early pentiums and such will throw an exception on rdtsc instruction in V86
;  regardless of CR4 setting, later CPU versions won't
@@rdtsc:
	pop	ecx				; restore registers
	pop	ebx
    pop eax
	@rdtsc
	jmp	@@twoeat
@@rdmsr:
	pop ecx
    pop ebx
    pop eax
	@rdmsr				; 0f 32 is rdmsr opcode

@@twoeat:
	add	WORD PTR [esp+8],2	; jump over instruction

@@Extret:
	add	esp,4+4			; eat return address and error code
	iretd
	
    align 4

;--- DS=FLAT

@@Is_BP:
	movzx   ECX, word ptr [ESP].V86FRGP.gCS
	MOV     EAX,[ESP].V86FRGP.gIP
	CMP     cx, RSEG				; CS segment known?
	JNZ     @@No_BP

	mov		cl, NUMBP
	mov     ebx, offset bptab
nextbp:
    cmp		ax, cs:[ebx+0]
    jz		bpfound
    add		ebx,4
    dec		cl
    jnz		nextbp
    mov		cx,RSEG
@@No_BP:    
    shl		ecx, 4
    add		eax, ecx				; to catch "resets" on NOALTBOOT
    cmp		eax, 0FFFF0h
    jz		REBOOT_PM
if ?BPOPC ne 0F4h
	jmp		@@V86_EXC06
endif


;--- HLT opcode handling

@@Is_Hlt:

if ?HLTDBG
	@DbgOuts <"True HLT occured at CS:IP=">,1
    mov ax, [esp].V86FRGP.gCS
    @DbgOutw ax,1
    @DbgOuts <":">,1
    mov eax, [esp].V86FRGP.gIP
    @DbgOutw ax,1
    @DbgOuts <10>,1
endif    

	INC     [ESP].V86FRGP.gIP   ; Jump over the HLT instruction
	POP     ECX                 ; restore Register
	POP     EBX
	POP     EAX
	ADD     ESP,4+4             ; throw away errorcode & returnaddress.
	STI                         ; give Interrupts free and then wait,
	HLT                         ; wait, wait........
BEHINDHLT:
	iretd		; but will it ever hit this?

bpfound:
;	movzx	eax, word ptr cs:[ebx+2]
;	jmp		eax
	jmp		word ptr cs:[ebx+2]
    
DEFBP macro x, y
	dw offset RSEG:x
    dw LOWWORD(offset y)
	endm

	align	4
    
bptab label dword
if ?VDS
	DEFBP   BPVDS, vds_handler_pm	; break in VDS handler (often used)
endif
if ?DMA
	DEFBP   BP1340, TRANSFER_BUFF   ; break in INT13/40 handler
endif
    DEFBP	BP67, V86_EMM_ENTRY		; break in EMM handler (int 67h caught by another app)
    DEFBP	BPBACK, V86_BACK		; break to return to V86 after calling v86-code
    DEFBP	BPDRV, EMMXXXX0_ENTRY	; communication with EMMXXXX0 device
	DEFBP   BPUMB, umb_handler_pm	; break in XMS UMB handler
	DEFBP   REBOOT_RM, REBOOT_PM
    DEFBP	BP06, do_exc_ex			; break in invalid opcode handler
    DEFBP	BP06_FATAL, V86_EXC06_FATAL
NUMBP equ ($ - bptab) / 4

	PURGE DEFBP

if ?BPOPC ne 0F4h
	align 4
if ?MASM    
INT06_ENTRY::
else
INT06_ENTRY:
endif
	push	0
    push	0
    push	eax
    push	ebx
    push	ecx
    mov		ax,FLAT_DATA_SEL
    movzx	ecx, [esp].V86FRGP.gCS
    mov		ds, eax
    shl		ecx, 4
    add		ecx, [esp].V86FRGP.gIP
    
    cmp		byte ptr [ecx],?BPOPC
    jz		@@Is_BP
    jmp		@@V86_EXC06
endif

;--- if int 67h vector is hooked in real-mode, the hooker code is
;--- called and will finally met a breakpoint which will get us here.

V86_EMM_ENTRY:
	call	SimIRET
	POP     ECX                 ; restore Register
	POP     EBX
	POP     EAX
	ADD     ESP,4+4             ; throw away errorcode & returnaddress.
    JMP		EMM_ENTRY_EX

;--- this breakpoint allows v86 code to be called by the monitor
;--- the caller has to push 3 dword values on the v86-stack:

v86_call struc
dwBP        dd ?  ; far16 address of breakpoint
dwOldCSIP   dd ?  ; previous CS:IP of v86
dwRetAddr   dd ?  ; near32 address of monitor address to jump to
v86_call ends

V86_BACK:
    MOVZX	ebx, [ESP].V86FRGP.gSP
    MOVZX	ecx, [ESP].V86FRGP.gSS
    add  	[ESP].V86FRGP.gSP,size v86_call - 4
    SHL		ecx, 4
    add		ebx, ecx
    sub		ebx, 4
    mov		eax, [ebx].v86_call.dwOldCSIP
    mov		word ptr [ESP].V86FRGP.gIP,ax
    shr		eax,16
    mov		[ESP].V86FRGP.gCS,ax
    jmp     [ebx].v86_call.dwRetAddr

;--- called by EMMXXXX0 device driver after EMM installation
;--- DS=FLAT
    
EMMXXXX0_ENTRY:
	INC     [ESP].V86FRGP.gIP   ; Jump over the HLT instruction
	push	RSEG_SEL
    pop 	FS
    assume  FS:RSEG
    mov		eax, FS:[request_ptr]
    MOVZX	ecx, ax
    shr		eax, 12
    and		al, 0F0h
    add		ecx, eax
if ?EMMXXXX0
    mov		al, ds:[ecx+2]
if ?EMXDBG
	@DbgOuts <"EMMXXXX0 request ">,1
    @DbgOutb al,1
    @DbgOuts <10>,1
endif
    cmp     al, 3			;IOCTL input?
    jz 		@@ioctl_read
    cmp     al, 8			;write?
    jz		@@resp_writeerr
    cmp     al, 9			;write+verify?
    jz		@@resp_writeerr
    cmp     al, 10  		;write status
    jz		@@resp_ok
    cmp     al, 11  		;write+flush?
    jz		@@resp_writeerr
    cmp     al, 12  		;IOCTL output?
    jz		@@ioctl_write
    cmp     al, 13			;open device?
    jz		@@resp_ok
    cmp     al, 14			;close device?
    jz		@@resp_ok
    mov		ax, 8103h
    jmp		@@device_done
@@ioctl_read:
	pushad
	call    ioctl_read
    popad
	jc		@@resp_readerr
    jmp		@@resp_ok
@@ioctl_write:
	pushad
	call    ioctl_write
    popad
	jc		@@resp_writeerr
    jmp		@@resp_ok
@@resp_readerr:    
    mov		ax, 810Bh
    jmp		@@device_done
@@resp_writeerr:    
    mov		ax, 810Ah
    jmp		@@device_done
@@resp_ok:
	mov		ax, 100h
@@device_done:
else
	mov		ax, 8103h
endif
    mov		word ptr ds:[ecx+3], ax
	POP     ECX                 ; restore Register
	POP     EBX
	POP     EAX
	ADD     ESP,4+4             ; throw away errorcode & returnaddress.
	iretd
	assume FS:NOTHING	

;--- exception (not 0Dh) in V86 mode
;--- DS=FLAT

@@V86_TEST_EXC:
if ?ROMRO
	cmp		ECX, 0Eh*4
    jnz		@@V86_EXC06
    mov		eax, CR2
    shr		eax, 12
    cmp		eax, 0FFh      ;it is the FF000 page (which is r/o!)
    jnz 	@@V86_EXC06
    @DbgOuts <"Write access to FF000",10>, ?V86DBG
    @GETPTEPTR ebx, 1000h+0FFh*4, 1
    mov     ecx, ds:[ebx]	;get PTE for FF000
    mov		dword ptr ds:[ebx], 0FF000h + 111B
;    push	V86_DATA_SEL
;    pop		fs
;    assume	fs:V86SEG
    mov		ss:[dwSavedRomPTE], ecx
    call	invlpgFF
if ?MASM    
    mov		eax, (V86_CODE_SEL shl 16) + LOWWORD(offset resettf)
else
    mov		eax, (V86_CODE_SEL shl 16) + offset resettf
endif
    mov		ecx, 0EE00h
    mov		ebx, dword ptr [IDT_PTR+2]
    xchg	eax, [ebx+1*8+0]
    xchg	ecx, [ebx+1*8+4]
    mov		dword ptr ss:[dqSavedInt01+0], eax
    mov		dword ptr ss:[dqSavedInt01+4], ecx
	POP     ECX                 ; restore Register
	POP     EBX
	POP     EAX
	ADD     ESP,4+4
    or		byte ptr [esp+2*4+1], 1	;set TF
	iretd
;    assume fs:nothing
resettf:
	pushad
    push	FLAT_DATA_SEL
    pop		ds
    @DbgOuts <"After write access to FF000",10>, ?V86DBG
    mov		ecx, [dwSavedRomPTE]
    @GETPTEPTR eax, 1000h+0FFh*4, 1
    mov		ds:[eax],ecx
	xor		eax, eax
    mov		cr2, eax
    mov		eax, dword ptr [dqSavedInt01+0]
    mov		ecx, dword ptr [dqSavedInt01+4]
    mov		ebx, dword ptr [IDT_PTR+2]
    mov		[ebx+1*8+0], eax
    mov		[ebx+1*8+4], ecx
    call	invlpgFF
    popad
    and		byte ptr [esp+2*4+1],not 1	;clear TF
    iretd
invlpgFF:
if ?INVLPG
	cmp		[_NoInvlPg],0
    jnz		@@noinvlpg
    invlpg	ds:[0FF000h]
    ret
@@noinvlpg:
endif
    mov		eax, cr3
    mov		cr3, eax
    ret
endif    

;*************************************************************
; this driver isn't abortable (it holds UMBs,...)
;
; so we generate an INT 6 in v86-mode. this notifies any hookers
; about the problem. If noone has hooked v86-int 06, we finally
; end at the monitor again, display a register dump and will try
; to terminate the current PSP.
;*************************************************************

if ?MASM
V86_EXC06::
else
V86_EXC06:
endif
@@V86_EXC06:

; simulate invalid opcode interrupt in v86 mode

	push	ebp
    lea		ebp, [esp+4].V86FRGP.gIP
    push	6
    call	SimIntV86
    pop		ebp


	POP     ECX             ; Clean everything again
	POP     EBX
	POP     EAX

	add     ESP,4+4         ; remove error word + old calling address

	IRETD                   ; return to virtual 86-Mode

;***************************************

; an IN/OUT-command now HAS to be emulated and the data has to be checked
; esp = ecx, ebx, eax, ret, errc, eip, cs, ...

IOESPFR struc
iEdi dd ?	;+0
iEsi dd ?	;+4
iEdx dd ?	;+8
iEcx dd ?	;+12
iEbx dd ?	;+16
iEax dd ?	;+20
iRes dd ?
iErr dd ?
iIp  dw ?
     dw ?
iCs  dw ?
     dw ?
IOESPFR ends

@@DoIO_DX:
	REPCMD  PUSH,<EDX,ESI,EDI>	    ; save Register  .
	INC     [ESP].IOESPFR.iIp		; Jump over instruction
	JMP     @@WithDX
@@DoIO_Im:
	REPCMD  PUSH,<EDX,ESI,EDI>
	MOVZX   EDX,BYTE PTR [EAX+1]    ; get I/O port in DX
	ADD     [ESP].IOESPFR.iIp,2     ; jump over instruction
@@WithDX:
	TEST    BL,00000010B        ; is it an IN-command ?
	JNZ     @@Im_Out            ; Noe ... 'n oller OUT.
	TEST    BL,00000001B        ; Width Word or Byte ?
	JNZ     @@Im_Word
	IN      AL,DX               ; Read the date from Port
if ?A20PORTS
	cmp		DL,60h
    jz		ISA2060IN
	cmp		DL,92h
    jz		ISA2092IN
if ?MASM    
A20INCONT::    
else
A20INCONT:
endif
endif
	MOV     byte ptr [ESP].IOESPFR.iEax,AL
    jmp		@@Bye
@@Im_Word:
	IN      AX,DX							; Same action like above but
	MOV     word ptr [ESP].IOESPFR.iEax,AX	; now with/for 16 Bit
@@Bye:
	REPCMD  POP,<EDI,ESI,EDX,ECX,EBX,EAX>
	ADD     ESP,4+4
	IRETD

;--- immediate and DX OUT

@@Im_Out:
	mov		cx,V86_DATA_SEL
    mov		ds,ecx
    assume	ds:V86SEG
if 1    
	mov		cx,RSEG_SEL
    mov		fs,ecx
    assume	fs:RSEG
endif
	MOV     EAX,[ESP].IOESPFR.iEax
    push	offset @@Bye
	TEST    BL,00000001B        ; if the Hi-Byte still can be send
	JZ      @@Do_IO
    push	edx
    push	eax
    call	@@Do_IO
    pop		eax
	pop		edx
	INC     EDX
	MOV     AL,AH

;--- output AL to DX

@@Do_IO:
if ?A20PORTS
	CMP     DL,60H
	JZ      ISA2060
	CMP     DL,64H
	JZ      ISA2064
	CMP     DL,92H
	JZ      ISA2092OUT
endif
if ?DMA
	CMP     DL,10H				; Has a Page-register
	JB      LIKE_DMA			; been adressed, or one
	CMP     DL,80H				; Has a Page-register
	JB      @@nodma
	CMP     DL,90H				; of both DMA-
	JB      LIKE_PAGE			; Controllers ?
	CMP     DL,0C0H
	JB      @@nodma
	CMP     DL,0E0H
	JB      LIKE_DMA
@@nodma:    
endif
	OUT		DX, AL
	RET

if ?DMA

PageLookUp      DB      0,2,3,1,0,0,0,0,0,6,7,5,0,0,0,0
PageXLat        DB      87H,83H,81H,82H,0,8BH,89H,8AH

;
; Access to a pageregister
;
; In: DX : Port-address; AL : Value
;
LIKE_PAGE PROC NEAR

if ?DMADBG
	@DbgOuts <"DMA page reg OUT, dx=">
    @DbgOutw dx
    @DbgOuts <" AL=">
    @DbgOutb al
    @DbgOuts <10>
endif    
    
	OUT     DX,AL                   ; release data
	MOVZX   EBX,DL                  ; Look up the number that corresponds to the port
	MOVZX   EDI,BYTE PTR PageLookUp[EBX-80H]
	MOV     [DmaChn+EDI*8].PageReg,AL         ; Buffer Bits 24-16
	JMP     READY?

LIKE_PAGE ENDP
;
; Supervise certain registers of the DMA (direct memory access) components
; DX = port
; AL = value
;
LIKE_DMA PROC NEAR

if ?DMADBG
	@DbgOuts <"DMA OUT, dx=">,1
    @DbgOutw dx,1
    @DbgOuts <" AL=">,1
    @DbgOutb al,1
    @DbgOuts <" 08=">,1
    mov ah,al
    in al,8
    @DbgOutb al,1
	mov al,ah    
    @DbgOuts <10>,1
    @waitkey 1, 0;?DMADBG
endif    

	CMP     DL,8                    ; Program addresses and/or length of the blocks
	JB      @@BlockSet8
	CMP     DL,10H
	JB      @@Control8
	CMP     DL,0C0h+10h             ; dto. 2nd DMA-Controller ?
	JB      @@BlockSet16
	CMP     DL,0C0h+0AH*2           ; single mask?
	JZ      @@SMask16
	CMP     DL,0C0h+0BH*2           ; mode set?
	JZ      @@Mode16
	CMP     DL,0C0h+0Ch*2
	JZ      @@Clear16
if ?MMASK    
	CMP     DL,0C0h+0Dh*2
	JZ      @@MMask16On
	CMP     DL,0C0h+0Eh*2
	JZ      @@MMask16Off
	CMP     DL,0C0h+0Fh*2           ; master mask?
	JZ      @@MMask16
endif    
    @DbgOuts <"port was unhandled!?",10>, ?DMADBG
    out		dx, al
	JMP     @@Bye                   ; If everything fails ...
@@Control8:
	CMP     DL,0AH                  ; single mask?
	JZ      @@SMask8
	CMP     DL,0BH                  ; "Mode Register" responded ?
	JZ      @@Mode8                 ; DMA-Controller #1 ?
	CMP     DL,0CH
	JZ      @@Clear8                ; clear flip-flop?
if ?MMASK    
	CMP     DL,0DH
	JZ      @@MMask8On
	CMP     DL,0EH
	JZ      @@MMask8Off
	CMP     DL,0FH
	JZ      @@MMask8                ; master mask?
endif    
    @DbgOuts <"port was unhandled!?",10>, ?DMADBG
    out		dx, al
	JMP     @@Bye                   ; If everything fails ...
	
;
; Access to Startadress and/or Blocklength of a DMA-Channel
;

@@BlockSet16:
	MOV     EDI,EDX	                ; Compute the DmaChannel-number from the
	OUT     DX,AL                   ; Pass on only once ...
	SHR     EDI,2                   ; C0 -> 0, C4 -> 1
	MOV     BX,HiLoFlag2
	AND     EDI,011B                ; I/O-Adress (likewise already transformed)
	ADD     EDI,4
	TEST    DL,02H                  ; meant block length  ?
	JNZ     @@Length
    JMP		@@Adr
@@BlockSet8:
	MOV     EDI,EDX                 ; Compute the DmaChannel-number from the port
	OUT     DX,AL                   ; Pass on only once ...
	SHR     EDI,1
	MOV     BX,HiLoFlag1            ; ... currently Controller #1
	AND     EDI,011B                ; I/O-Adress
	TEST    DL,01H                  ; meant block length  ?
	JNZ     @@Length
@@Adr:
	xor		ecx,ecx
	BTC     [GblFlags],BX           ; toggle the Hi/Lo-Flag. Is
    setc	cl
	MOV     BYTE PTR [DmaChn+ECX+EDI*8].DMAREQ.BaseAdr,AL
	JC      READY?
    ret

@@Length:
	xor		ecx, ecx
	BTC     [GblFlags],BX               ; toggle Hi/Lo-Flag
    setc	cl
	MOV     BYTE PTR [DmaChn+ECX+EDI*8].BlockLen,AL
	jc      READY?
    ret

;
; Monitor "Mode Register" for the Transferdirection DMA <--> I/O
;
@@Mode16:
	MOV     EDI,EAX
	AND     EDI,011B	        ; Likewise out-mask the number of the DMA
                               	; (direct memory access) channel.
	ADD     EDI,4				; It's about a 16bit channel
	JMP     @@Mode
@@Mode8:
	MOV     EDI,EAX             ; Mask away the number of the DMA-Channel
	AND     EDI,011B
@@Mode:
	out		dx, al
	mov		[DmaChn+EDI*8].ChanMode,al

if ?MASM
READY?::
else
READY?:
endif
	test	bRFlags,2			;new Int 13h/40h DMA op?
    jz		@@nonewreq
    and		bRFlags,not 2
    and 	[DmaChn+EDI*8].bFlags,not DMAF_ENABLED ;wait until channel is enabled
@@nonewreq:    
	test	[DmaChn+EDI*8].bFlags,DMAF_ENABLED
	jz		@@Bye
	CALL    CHK_CHANNEL	; Then check value
@@Bye:
	RET

;
; Clear the Hi/Lo flip-flop of the appropriate DMA (direct memory access) CONTROLLER
;
@@Clear8:
	out		dx, al
	BTR     [GblFlags],HiLoFlag1		; It was like that already
    ret
@@Clear16:
	out		dx, al
	BTR     [GblFlags],HiLoFlag2		; Number 2's turn..
    ret
    
;--- Single mask port (000A, 00D4)
;--- bits 0-1 select channel
;--- bit 2=1 -> enable channel mask (=channel inactive)

@@SMask16:
    mov		edi, eax
	AND     EDI,0011B
    add		edi,4
    jmp		@@SMask
@@SMask8:
    mov		edi, eax
	AND     EDI,0011B
@@SMask:
	and		[DmaChn+EDI*8].bFlags,not DMAF_ENABLED
    test	al,4
    jnz		@@isdisable
    or 		[DmaChn+EDI*8].bFlags,DMAF_ENABLED
	and		bRFlags,not 2
    push	edx
    push	eax
    call	CHK_CHANNEL
    pop		eax
    pop		edx
@@isdisable:
	out		dx, al
    ret

if ?MMASK
;--- Master mask port (000F, 00DE)
;--- bit 0 -> 1=channel 0 inactive, 0=channel 0 active
;--- bit 1 -> ... chn 1
;--- bit 2 -> ... chn 2
;--- bit 3 -> ... chn 3

;--- this port cannot be read reliably !!!!!

@@MMask16On:			;mask all 4 channels
	mov		al,0Fh
    jmp		@@MMask16
@@MMask16Off:			;unmask all 4 channels
	mov		al,0
@@MMask16:
	mov		ah, al
    shl		ah, 4
    mov		edi, 4
    jmp		@@MMask
@@MMask8On:				;mask all 4 channels
	mov		al,0Fh
    jmp		@@MMask8
@@MMask8Off:			;unmask all 4 channels
	mov		al,0
@@MMask8:
	mov		ah, al
    xor		edi, edi
@@MMask:
	mov		cl,4
    xchg	al,ah	;value to write to port now in AH, in AL transformed mask
@@nextcnl:
;	test	[DmaChn+EDI*8].bFlags,DMAF_ENABLED
;	jnz 	@@skipchannel    	;channel is already active
	and		[DmaChn+EDI*8].bFlags,not DMAF_ENABLED
    bt  	eax, edi
    jc 		@@skipchannel    	;channel will become/stay inactive
    or 		[DmaChn+EDI*8].bFlags,DMAF_ENABLED
    push	eax
    push	ecx
    push	edx
    push	edi
    call	CHK_CHANNEL
    pop		edi
    pop		edx
    pop		ecx
    pop		eax
@@skipchannel:
	inc		edi
    dec		cl
    jnz 	@@nextcnl
    mov		al, ah
	out		dx, al
    ret
endif

LIKE_DMA ENDP

;
; Check a memory-area, whether it lies in continuous physical memory
; in pages of 4K each.
; However, this is not enough. the physical region must not cross a
; 64 kB (or 128 kB for 16bit) region 
;
; In:  ESI: linear Startadress (bits 0-23 only)
;      ECX: Length of the range (?DMABUFFMAX is max in kB)
;      EDI: channel
;      DS=V86
; Out: CY-Flag: reset, ESI = physical adress
;               set, not constantly
;

CONTINUOUS? PROC NEAR
	PUSH    ES
	PUSH    ECX
	PUSH    FLAT_DATA_SEL
	POP     ES
	cmp 	ESI, 400000h-20000h     ; don't touch PTEs > 3E0000h
    jnc		@@Ok2

	cmp		ecx,1                   ; make sure ecx is at least 1
	adc     ecx,0

	PUSH    ESI
	lea     ECX,[ecx+esi-1]         ; ecx -> last byte 

	SHR     ESI,12                  ; linear start address -> PTE
	SHR     ECX,12                  ; linear end   address -> PTE
    sub		ecx, esi
    
	@GETPTEPTR ESI, ESI*4+1000h     ; ESI -> PTE
	MOV     EAX,ES:[ESI]            ; get PTE

if ?DMADBG
	@DbgOuts <"continuous?: PTE=">
    @DbgOutd eax
    @DbgOuts <" ecx=">
    @DbgOutd ecx
    @DbgOuts <10>
endif

	AND     AX, 0F000h              ; mask out bits 0-11 of PTE

;--- ESI -> PTE, EAX=1. PTE

    test	eax, 0FF000000h         ; if physical address is > 16 MB		
    jnz		@@Fail2                 ; a buffer is needed in any case

	JECXZ   @@Ok
    
	PUSH    EAX                     ; save Bits 31-12
@@Loop:
	ADD     EAX,1000h               ; Go one page and
	ADD     ESI,4                   ; one entry further
	MOV     EBX,ES:[ESI]
	AND     BX, 0F000h
	CMP     EAX,EBX                 ; no memory in between?
	JNZ     @@Fail                  ; ...Below!
	loop    @@Loop
    mov		eax, [esp]
    shr		eax, 16
    shr		ebx, 16
    cmp		edi,4
    jb		@@is8bit
    and		al,not 1
    and		bl,not 1
@@is8bit:    
	cmp		bx,ax
    JNZ     @@Fail                  ; a 64/128 kB Border has been crossed!!!
	POP     EAX
@@Ok:
    pop     ESI
	AND     ESI,0FFFh               ; from the bits 31-12 of the page
	OR      ESI,EAX                 ; and the offset 11-0
@@Ok2:
    pop     ECX
    pop     ES                      ; The physical address results
	CLC                             ; ...its ok.
	RET
@@Fail: 
	POP     EAX
@@Fail2: 
    POP     ESI
    POP     ECX
    POP     ES
	STC	                            ; ...not possible  !
	RET
CONTINUOUS? ENDP

;
; A DMA-Channel is completely with data about beginning and length
; supplied, so that a check can (and has to) take place.
;
; In: EDI: Channelnumber 0..7
; modifies ECX, ESI, ECX, DX
;
CHK_CHANNEL PROC NEAR

	@DbgOuts <"CHK_CHANNEL enter",10>, ?DMADBG
    @waitkey 1, 0;?DMADBG
;    int		3

	cmp		[DmaChn+EDI*8].cDisable,0	;translation disabled by VDS?
    jnz		@@Bye
	test    [DmaChn+EDI*8].ChanMode,1100B  ; If a verify is wanted
	JZ      @@Bye 			               ; nothing is to be done
    
	MOVZX   ECX,[DmaChn+EDI*8].BlockLen	; get block length
	MOVZX   ESI,[DmaChn+EDI*8].PageReg	; The base-adress always lies on a
	INC     ECX
	CMP     EDI,4                   ; In case we're dealing with a
	JB      @@Only8                 ; 16bit DMA-channel , words are transferred.
	ADD     ECX,ECX
	SHL     ESI,15                     ; Word-border and Bit 0 of the
	MOV     SI,[DmaChn+EDI*8].BaseAdr  ; pageregister is being ignored
	SHL     ESI,1
	JMP     @@Chk
@@Only8:
	SHL     ESI,16                     ; Adress-calculation is a bit
	MOV     SI,[DmaChn+EDI*8].BaseAdr  ; more easy...
@@Chk:

;--- now ESI holds the start address (linear!)

	BTR     [GblFlags],NeedBuffer   ; Initialise.

;--- Is the block occupying contiguous physical pages?
;--- Or does it cross a 64kB/128 kB boundary?
;--- after the call ESI will hold the physical address (if NC)

    push	esi
	CALL    CONTINUOUS?
    pop		ebx
	JNC     @@Set                   

	test	[DmaChn+EDI*8].ChanMode,10h	;auto-init? Then a buffer is useless
    jnz		@@Set

	@DbgOuts <"block not contiguous or crosses 64k border, DMA buffer needed!",10>, ?DMADBG
    @waitkey 1, 0;?DMADBG
    
	MOV     [TargetAdr],EBX             ; save linear address of block
	MOV     ESI,[DMABuffStartPhys]		; get DMA Buffer physical address
    BTS		[GblFlags], NeedBuffer
    cmp		ecx, [DMABuffSize]
    jbe		@@blocksizeok
    mov		ecx, [DMABuffSize]          ; error: DMA buffer overflow    
@@blocksizeok:
	MOV     [TargetLen],ECX
@@Set:
if ?DMADBG
	@DbgOuts <"target lin. start address=">,1
    @DbgOutd ebx,1
    @DbgOuts <" length=">,1
    @DbgOutd ecx,1
    @DbgOuts <" phys=">,1
    @DbgOutd esi,1
    @DbgOuts <" chn=">,1
    @DbgOutd edi,1
    cmp esi, ebx
    jz @@nothingtodo
    @DbgOuts <"*">,1
@@nothingtodo:    
    @DbgOuts <10>,1
    @waitkey 1, 0
endif    
    
	cmp		esi, ebx				; has address changed
    jz		@@Bye
    
	CMP     EDI,4                   ; 8-bit or 16-bit channel ?
	JB      @@Set8

	LEA     EDX,[EDI*4+0C0h]        ; get I/O-address from DMA channel
	BTR     [GblFlags],HiLoFlag2    ; DMA #2, HiLo-FlipFlop-Flag
	OUT     [0D8H],AL               ; clear Hi/Lo-FlipFlop
	JMP     $+2
	SHR     ESI,1
	MOV     EAX,ESI                 ; reprogram base address
	OUT     DX,AL
	MOV     AL,AH
	JMP     $+2
	OUT     DX,AL
	SHR     ESI,15
	JMP     @@Cont
@@Set8:
	LEA     EDX,[EDI*2]             ; get I/O-address from DMA channel
	BTR     [GblFlags],HiLoFlag1    ; DMA #1, HiLo-FlipFlop-Flag
	OUT     [0CH],AL                ; clear Hi/Lo-FlipFlop
	JMP     $+2
	MOV     EAX,ESI                 ; reprogram base address
	OUT     DX,AL                   ;
	MOV     AL,AH
	JMP     $+2
	OUT     DX,AL
	SHR     ESI,16                  ; get bits 16-23 into 0-7
@@Cont:
	MOVZX   DX,PageXLat[EDI]        ; get I/O-Adress of page-register
	MOV     EAX,ESI                 ; set the page register
	OUT     DX,AL
    
; check, if the data has to be copied into the buffer

	BT      [GblFlags], NeedBuffer  ; DMA buffer used?
	JNC     @@Bye
	MOV     AL,[DmaChn+EDI*8].ChanMode   ; copy data into buffer required?
    and		al,1100b
    cmp		al,1000b
	JNZ     @@IsRead

	@DbgOuts <"CHK_CHANNEL, copy into DMA buffer",10>, ?DMADBG
    
	MOV     ESI,[TargetAdr]
	MOV     EDI,[DMABuffStart]
	MOV     ECX,[TargetLen]
	REPCMD  PUSH,<DS,ES>
	MOV     AX,FLAT_DATA_SEL
	MOV     DS,EAX
	MOV     ES,EAX
	CLD
	MOV     EAX,ECX
	AND     ECX,3
	REP     MOVS BYTE PTR [ESI],BYTE PTR [EDI]
	BIG_NOP
	MOV     ECX,EAX
	SHR     ECX,2
	REP     MOVS DWORD PTR [ESI], DWORD PTR [EDI]
	BIG_NOP
	REPCMD  POP,<ES,DS>
    jmp     @@Bye
@@IsRead:
    or		[bRFlags],1
@@Bye:
	RET
CHK_CHANNEL ENDP

	assume fs:NOTHING
    assume ds:NOTHING

; Copy the DMA-buffercontents after termination of the DISK-I/O to the
; wanted target/location
; * This is triggered by an HLT at a magical location in our own int 13
; * and int 40 handlers. First the original int 13 / 40 is done, and
; * then things are copied to / from where the mapped memory -really- is.
; * Should only trigger if a matching DMA is pending. Tricky!
;
TRANSFER_BUFF PROC NEAR

	INC     [ESP].V86FRGP.gIP   ; Jump over the breakpoint
	REPCMD  PUSH,<ESI,EDI>
    
	MOV     ESI,[DMABuffStart]  ; Get the basic data of the block which
	MOV     EDI,[TargetAdr]     ; can be shifted. Read currently
                                ; simulated DMA source / dest. / length
	MOV     ECX,[TargetLen]

if ?DMADBG                                
	@DbgOuts <"TRANSFER_BUFF: dst=">,1
    @DbgOutd edi,1
	@DbgOuts <" src=">,1
    @DbgOutd esi,1
	@DbgOuts <" siz=">,1
    @DbgOutd ecx,1
    @DbgOuts <10>,1
endif

	MOV     AX,FLAT_DATA_SEL
	MOV     DS,EAX
    assume	DS:NOTHING
	MOV     ES,EAX
	CLD
	MOV     EAX,ECX
	AND     ECX,3
	REP     MOVS BYTE PTR [ESI],BYTE PTR [EDI]
	BIG_NOP
	MOV     ECX,EAX
	SHR     ECX,2
	REP     MOVS DWORD PTR [ESI], DWORD PTR [EDI]
	BIG_NOP
	REPCMD  POP,<EDI,ESI,ECX,EBX,EAX>
	ADD     ESP,4+4             ; throw away error code + return-address.
	IRETD                       ; back to virtual 8086-Mode.

TRANSFER_BUFF ENDP

    assume ds:NOTHING

endif	;?DMA

;--- simulate an V86 Int
;--- INP: SS:EBP -> IRETDV86
;--- INP: [ESP+4] == INT #
;--- modifies EAX, EBX, ECX

SimIntV86 proc    

	PUSH    FLAT_DATA_SEL
	POP     DS
	assume	DS:NOTHING
	MOVZX   EBX,[EBP].IRETDV86.vSS	; get linear address of v86 SS:SP
	SHL     EBX,4
    movzx   eax,[EBP].IRETDV86.vSP
	ADD     EBX, EAX
	SUB     [EBP].IRETDV86.vSP,6	; room for IP,CS,FLAGS

	MOV     AX,[EBP].IRETDV86.vCS	; get v86 CS:IP into EAX
    shl		EAX, 16
	MOV     AX,word ptr [EBP].IRETDV86.vIP
	MOV     CX,[EBP].IRETDV86.vFL	; + v86 Flags
	MOV     [EBX-6],EAX
	MOV     [EBX-2],CX				; and store them onto v86 stack

	MOV		EBX,[ESP+4]
	MOV     EAX,dword ptr ds:[EBX*4]		; now set new v86 CS:IP to value
	MOV     word ptr [EBP].IRETDV86.vIP,AX	; found at [0000:6*4]
	SHR     EAX,16
	MOV     [EBP].IRETDV86.vCS,AX
	AND     BYTE PTR [EBP].IRETDV86.vFL+1,NOT (1+2); clear TF+IF
    RET		4
SimIntV86 endp

;--- simulate an IRET in v86-mode
;--- INP: SS:ESP+4 -> V86FRGP

SimIRET proc
    MOVZX	eax, [ESP+4].V86FRGP.gSP
    MOVZX	ecx, [ESP+4].V86FRGP.gSS
    SHL		ecx, 4
    add		eax, ecx
    MOV		ebx, [eax+0]
    mov		word ptr [ESP+4].V86FRGP.gIP,bx
    shr		ebx,16
    mov		[ESP+4].V86FRGP.gCS,bx
    MOV		bx, [eax+4]
    mov		[ESP+4].V86FRGP.gFL,bx
	ADD     [ESP+4].V86FRGP.gSP,6
    ret
SimIRET endp


if ?A20PORTS

	assume DS:V86SEG

ISA2060 proc
	cmp bLast64,0D1h	;last value written to 64 was "write output port"?
    jnz @@notoutp
    mov bLast64,0
	@DbgOuts <"write to port 60h kbc output port, al=">,?A20DBG
	@DbgOutb al,?A20DBG
	@DbgOuts <10>,?A20DBG
    push eax
    shr al,1
    and al,1
    mov bA20Stat,al
    call EmuA20
    pop eax
    or  al,2
@@notoutp:    
	out dx, al
	ret
ISA2060 endp
ISA2064:
    mov [bLast64],al	;last value written to port 64h
if ?A20DBG    
	cmp al,0D1h
    jnz @@nokbcout
	@DbgOuts <"write to port 64h, al=">,?A20DBG
	@DbgOutb al,?A20DBG
	@DbgOuts <10>,?A20DBG
@@nokbcout:    
endif    
	out dx, al
	ret
ISA2092OUT:
	@DbgOuts <"write to port 92h, al=">,?A20DBG
	@DbgOutb al,?A20DBG
	@DbgOuts <10>,?A20DBG
    push eax
    shr al,1
    and al,1
    mov bA20Stat,al
    call EmuA20
    pop eax
	or  al, 2		;dont allow disable
	out dx, al
	ret

	assume ds:nothing
    assume fs:nothing

ISA2060IN:
	cmp [bLast64],0D0h	;last value written to 64 was "read output port"?
    jnz A20INCONT
;    mov cx, V86_DATA_SEL
;    mov fs, ecx
;    assume fs:V86SEG
    mov ss:[bLast64], 0
;    assume fs:nothing
ISA2092IN:
	and al, not 2
	mov cl, [bA20Stat]
    shl cl, 1
    or  al, cl
	jmp A20INCONT
    
endif

if ?EMMXXXX0

;--- read ioctl EMMXXXX0 device
;--- all registers may be modified!
;--- DS=FLAT

ioctl_read proc 
    mov  	eax, [ecx+14];buffer
    movzx	ebx, ax
    shr		eax, 12
    and		al, 0F0h
    add		ebx, eax

	mov		dx, [ecx+18]    ;buffer size
    mov		al, [EBX+0]
    cmp     al, 0		;get "API"
    jz		@@func00
if 0    
	cmp		al, 1    	;GEMMIS not supported
    jz		@@func01
endif    
    cmp     al, 2		;get version
    jz		@@func02
    cmp     al, 6		;get system vars
    jz		@@func06
    cmp     al, 7		;get UMBs
    jz		@@func07
    jmp		@@error
@@func00:    
    cmp		dx,6	;bytes to read
    jb      @@error
    mov     word ptr [ebx+0], 0028h
    mov     dword ptr [ebx+2], 0	;API entry
    jmp		@@ok
@@func02:    
    cmp		dx,2	;bytes to read
    jb      @@error
    mov     word ptr [ebx+0], ?VERSIONHIGH + ?VERSIONLOW * 256
    jmp		@@ok
@@func06:    

    cmp		dx,16	;bytes to read
    jb      @@error
    mov		al, [_NoEMS]
    mov     [ebx].EMX06.e_NoEMS, al
    xor		eax, eax
    cmp		[_NoFrame],0
    jnz		@@nopf
    mov		ax, [_FRAME]
@@nopf:    
    mov     [ebx].EMX06.e_Frame, ax
    mov		al, [_NoVCPI]
    mov     [ebx].EMX06.e_NoVCPI, al
    cmp		dx,24					;to get VCPI memory info, we need 24 byte
    jb		@@novcpimem
    mov		eax, [_MAXMEM16K]	;VCPI in 16 kB 
    shl		eax, 2
    mov     [ebx].EMX06.e_VCPITotal, eax
    call	GetCurrent4KPageCount	;return total in edx, free in eax
    sub		edx, eax
    mov     [ebx].EMX06.e_VCPIUsed, edx
@@novcpimem:    
if ?DMA    
    mov		eax, [DMABuffStartPhys]
else
	xor		eax, eax
endif    
    mov     [ebx].EMX06.e_DMABuff, eax
if ?DMA    
    mov		eax, [DMABuffSize]
    shr		eax, 10
else
	xor		eax, eax
endif    
    mov     [ebx].EMX06.e_DMASize, ax
if ?VME    
    mov		al,1
    test	byte ptr [dwFeatures],2
    jz		@@novme
    @mov_eax_cr4
    and     al,1
    xor		al,1
@@novme:
	mov		[ebx].EMX06.e_NoVME,al
endif    
if ?PGE    
    mov		al,1
    test	byte ptr [dwFeatures+1],20h
    jz		@@nopge
    @mov_eax_cr4
    shr		al,7
    xor		al,1
@@nopge:
	mov		[ebx].EMX06.e_NoPGE,al
endif    
if ?A20PORTS or ?A20XMS
	mov		al,[_NoA20]
    mov		[ebx].EMX06.e_NoA20,al
endif
@@ok:
	clc
    ret
@@error:
	stc
	ret
@@func07:    
    cmp		dx,8*4	;bytes to read
    jb      @@error
    mov		edi, ebx
    mov		esi, offset pmUMBsegments
    push	ds
    pop		es
    push	ds
    push	V86_DATA_SEL
    pop		ds
    mov		ecx,8
    rep		movs dword ptr [edi], [esi]
    pop		ds
    clc
	ret
    
ioctl_read endp    

;--- all registers may be modified!

ioctl_write proc 

    mov  	eax, [ecx+14];buffer
    movzx	esi, ax
    shr		eax, 12
    and		al, 0F0h
    add		esi, eax

    lods	byte ptr [esi]
    cmp     al, 15
    jz		@@func15
	stc
	ret
@@func15:
	mov		dh, [ecx+18]    ;buffer size
    
;	mov		cx, V86_DATA_SEL
;    mov		fs, ecx
;    assume	fs: V86SEG
    lods	byte ptr [esi]
if ?VME
	cmp		al,-1
    jz		@@novme
	test	byte ptr [dwFeatures], 2		;VME supported?
    jz		@@novme
	@mov_ecx_cr4
    and		al,1
    xor		al,1
    and		cl,not 1
    or		cl,al
    @mov_cr4_ecx
@@novme:
endif
    lods	byte ptr [esi]
if ?A20PORTS or ?A20XMS
	cmp		al,-1
    jz		@@noa20
    and		al, al
    jz		@@a20emuon
    push	eax
    mov		al,1
    call	EmuA20		;enable A20 gate
    pop		eax
@@a20emuon:
	mov		[_NoA20], al
@@noa20:
endif    
    lods	byte ptr [esi]
    cmp		al,-1
    jz		@@novcpi
    mov		[_NoVCPI],al
@@novcpi:
    lods	byte ptr [esi]
if ?PGE
	cmp		dh,5
    jb		@@nopge
    cmp		al,-1
    jz		@@nopge
	test	byte ptr [dwFeatures+1], 20h   ;PGE supported?
    jz		@@nopge
    pushad
    and		al,1
    xor		al,1
    mov		[bPageMask],al
    @GETPTEPTR edi, 1000h, 1	;start of pagetab 0
    mov		ecx,110h+1
@@FILL_PAGETAB0:
	mov		edx, [edi]
    and		dh,not 1		;mask out G
    or		dh,al
	MOV     [EDI],EDX
	ADD     EDI,4
	loop    @@FILL_PAGETAB0
	@mov_ecx_cr4
    shl		al,7
    and		cl,not 80h
    or		cl,al
    @mov_cr4_ecx
    popad
@@nopge:
endif
    clc
    ret
    assume fs:nothing
ioctl_write endp    

endif

;--- EAX, EBX, ECX are saved on stack

?NUMUMBS	equ 8

UMBBLK struc
wSegm	dw ?
wSize	dw ?	;high bit used as flag
UMBBLK ends

umb_handler_pm proc

	mov ax, V86_DATA_SEL
	mov ds, eax
    assume	ds:V86SEG
    
    inc [esp].V86FRGP.gIP		;skip breakpoint
	mov eax, [ESP].V86FRGP.gEAX
    
    push esi
	mov	ecx, ?NUMUMBS	;set CL to max umbs, clear CH!
    
	cmp	ah,11h			;free UMB, DX=segment address to release
	je	UMBfree_pm
	cmp	ah,12h			;realloc UMB, DX=segment to resize, BX=new size
	je	UMBrealloc_pm

;--- UMBalloc

UMBalloc_pm:		;DX=size of block in paragraphs

if ?UMBDBG
	@DbgOuts <"UMBalloc enter, DX=">,1
    @DbgOutw dx,1
    @DbgOuts <10>,1
endif

					; find first available memory block
	mov	esi, offset pmUMBsegments
	xor	ebx,ebx		; holds largest too-small block size

@@UMBloop:
	cmp	[esi].UMBBLK.wSegm,0	; see if valid UMB
	je	@@UMBnext			; no
	test BYTE PTR [esi].UMBBLK.wSize+1,80h	; see if UMB already allocated (high bit size set)
	jne	@@UMBnext			;  yes
	cmp	dx,[esi].UMBBLK.wSize; dx = requested block size (high bit of UMB size known reset)
	jbe	@@UMBfound			; enough memory available in UMB
	mov	ch,1				; flag UMB was found, although too small
	cmp	bx,[esi].UMBBLK.wSize
	ja	@@UMBnext
	mov	bx,[esi].UMBBLK.wSize; update largest too-small block size

@@UMBnext:
	add	esi,size UMBBLK
    dec cl
	jnz @@UMBloop
	or	ch,ch
	jne	@@umb_too_small
	mov	bl,0B1h		; error "no UMB's are available"
	xor	dx,dx
	xor	ax,ax		; flag failure
	jmp	umb_exit
@@umb_too_small:
	mov	dx,bx		; return largest UMB in DX
	mov	bl,0B0h		; error "only smaller UMB available"
	xor	ax,ax		; flag failure
    jmp umb_exit

@@UMBfound:

; see if actual UMB size exceeds request size by >=2K
	mov	ax,80h				; 128 paras == 2K
	add	ax,dx
	cmp	ax,[esi].UMBBLK.wSize
	ja	@@good_umb			; can't split it, just use it

;  2K or over would be unused, see if we can split the block

	mov	cl, ?NUMUMBS
	mov	ebx, offset pmUMBsegments

@@splitloop:
	cmp	WORD PTR [ebx].UMBBLK.wSegm,0
	jne	@@splitnext

; split the block
	mov	ax,dx
	add	ax,7fh
	and	ax,0ff80h			; round up allocation to next 2K in paras
	mov	cx,[esi].UMBBLK.wSegm
	add	cx,ax
	mov	[ebx].UMBBLK.wSegm,cx	; new block has segment offset of old block+allocation
	mov	cx,[esi].UMBBLK.wSize	; get original UMB block size, in paras
	sub	cx,ax				; subtract allocation
	mov	[ebx].UMBBLK.wSize,cx	; update new block with old block size minus allocation
	mov	[esi].UMBBLK.wSize,ax	; update original UMB block size to allocation

	jmp	@@good_umb

@@splitnext:
	add	ebx,size UMBBLK
	dec	cl
	jne	@@splitloop

@@good_umb:
	mov	dx,[esi].UMBBLK.wSize	; actual block size to dx
	or	BYTE PTR [esi].UMBBLK.wSize+1,80h	; flag UMB allocated
	mov	bx,[esi].UMBBLK.wSegm	; get UMB address in bx
	mov word ptr [ESP+4].V86FRGP.gEBX,bx
	mov ax,1
    
umb_exit:    
if ?UMBDBG
	@DbgOuts <"UMB exit, ax=">,1
    @DbgOutw ax,1
    @DbgOuts <", bx=">,1
    @DbgOutw bx,1
    @DbgOuts <", dx=">,1
    @DbgOutw dx,1
    @DbgOuts <10>,1
endif    
	pop	esi
    pop ecx
    and ax,ax
    jnz @@umbexit_noerror
    mov byte ptr [esp], bl
@@umbexit_noerror:
    pop ebx
    add esp,4+4+4	;skip eax, returnaddr, errorcode
    iretd

;--- UMBFree (do not modify hiword(eax)!)

UMBfree_pm:
	@DbgOuts <"UMBfree enter",10>, ?UMBDBG
	mov	esi,offset pmUMBsegments
	xor	ax,ax			; flag failure
@@freeloop:
	cmp	[esi].UMBBLK.wSegm,dx	; see if matches existing UMB allocation
	je	@@free_found
	add	esi,size UMBBLK
	loop @@freeloop
@@free_error:    
	mov	bl,0b2h			; invalid UMB segment number error code
	jmp umb_exit

@@free_found:
	test byte ptr [esi].UMBBLK.wSize+1,80h
    jnz @@free_error
	and	BYTE PTR [esi].UMBBLK.wSize+1,7fh	; flag UMB not allocated
	inc	eax			; flag success
	jmp umb_exit

;--- UMBrealloc (do not modify hiword(eax)!)

UMBrealloc_pm:		;DX=segment, BX=new size of block in paragraphs

	@DbgOuts <"UMBrealloc enter",10>, ?UMBDBG
	mov	esi,offset pmUMBsegments
	xor	ax,ax			; flag failure
@@realloop:
	cmp	[esi].UMBBLK.wSegm,dx	; see if matches existing UMB allocation
	je	@@real_found
	add	esi, size UMBBLK
	loop @@realloop
	mov	bl,0b2h			; invalid UMB segment number error code
	jmp umb_exit

@@real_found:
	test byte ptr [esi].UMBBLK.wSize+1,80h	;is it an allocated block?
    jnz @@umbreal_error1
	mov cx, [esi].UMBBLK.wSize
	and ch, 7Fh
    cmp bx, cx
    ja @@umbreal_error2
	inc	eax			; flag success
	jmp umb_exit
@@umbreal_error1:	;block is not allocated
	mov bl,0B2h
	jmp umb_exit
@@umbreal_error2:	;block is too small
	mov dx,cx
	mov bl,0B0h
	jmp umb_exit
    
	assume DS:NOTHING

UMB_handler_pm endp


if ?VDS

;--- flags in DX used by various VDS functions

VDSF_COPY		equ 02h
VDSF_NOBUFFER	equ 04h
VDSF_64KALIGN   equ 10h
VDSF_128KALIGN  equ 20h
    

; entry ax = 4k page number 
; exit edx = physical address

GetPhysAddr	PROC	NEAR
    movzx edx, ax
	@GETPTE edx, edx*4+1000h
    and dx, 0F000h
	ret
GetPhysAddr	ENDP

;--- test if a region is contiguous and crosses 64/128 kB borders
;--- ax = start page, bx = end page, dx=flags
;--- for first 4 MB only
;--- returns initial physical address in EAX
;--- in case of error: size which would be ok in edx

VDS_retcode equ <byte ptr [ebp+12]>	;save VDS returncode on stack

vds_contiguous proc

	push ecx
    push esi
    push edi
    movzx eax, ax

	@GETPTEPTR esi, 1000h, 1	;get start of page tab 0

    mov edi, [esi+eax*4]
    and di, 0F000h
    push edi
@@nextitem:    
	cmp ax, bx
    jz @@iscontiguous
    inc eax
    add edi, 1000h
	mov ecx, [esi+eax*4]
    and cx,0F000h
    cmp edi, ecx
    je @@nextitem
    pop eax
	mov	VDS_retcode,1
    mov edx, edi
    sub edx, eax
@@failed:
    pop edi
    pop esi
    pop ecx
    stc
    ret
@@failed2:			;failed cause 64 kb border crossing
	mov edx, esi
    xor dx, dx
    sub edx, eax
	mov	VDS_retcode,2
    jmp @@failed
@@failed3:			;failed cause 128 kb border crossing
	mov edx, esi
    and edx, 0FFFE0000h
    sub edx, eax
	mov	VDS_retcode,2
    jmp @@failed
@@iscontiguous:
	pop edi			;get start physical address
    mov eax, edi
    mov esi, ecx	;save ecx in case of border cross errors
    shr edi, 16
	shr ecx, 16
    test dl, VDSF_64KALIGN	;check for 64 kb border cross?
    jz @@no64check
    cmp di,cx
    jnz @@failed2
@@no64check:
    test dl,VDSF_128KALIGN	;check for 128 kb?
    jz @@no128check
    shr di,1
    shr cx,1
    cmp di,cx
    jnz @@failed3
@@no128check:
    pop edi
    pop esi
    pop ecx
    ret
    align 4
    
vds_contiguous endp

;-- simplified version of vds_contiguous
;-- assumes linear == physical (for regions above 400000h)
;-- eax = linear start address
;-- ebx = size of region
;-- ret C -> boundary error (eax==size ok)
;-- ret NC -> ok, eax == physical address [== linear address]

vds_contiguous2 proc

;-- see if boundary alignment check

	test	dl,VDSF_64KALIGN or VDSF_128KALIGN	; boundary check?
	jz	@@ok
	mov	ecx,eax
	lea	ebx,[eax+ebx-1]
	shr	ebx,16		; convert start/end to alignment 64K frames
	shr	ecx,16
	test dl,VDSF_64KALIGN
	jne	@@aligncheck	; yes, if 64K works, then 128K will too
	shr	ecx,1			;-- 128K alignment check
	shr	ebx,1
@@aligncheck:
	cmp	bx,cx
	je	@@ok
	inc	ecx			; ecx == starting alignment frame+1
	shl	ecx,16		; convert to next alignment frame address
	test dl,VDSF_64KALIGN
	jne	@@check2
	shl	ecx,1
@@check2:
	sub	ecx,eax		; get bytes to next alignment frame address from start
	mov	VDS_retcode,2	; region crossed alignment boundary error code
	mov	eax,ecx		; eax == size ok
    stc
@@ok:
	ret
    align 4
vds_contiguous2 endp

	align 4

;--- bit vector which vds func needs ES:DI translation
;------ CBA9876543210
bDDS dd 0011111111000b

;--- DDS, used by 03-04, 07-08, 09-0A

DDS struc
dwSize	dd ?	;+0  size of region
dwOfs	dd ?	;+4  offset virtual start address
wSeg	dw ?	;+8  segment/selector virtual start address (or 0000)
wID		dw ?	;+10 buffer ID
dwPhys	dd ?	;+12 physical address
DDS ends

;--- EDDS, used by 05-06

EDDS struc
dwSize	dd ?	;+0
dwOfs	dd ?	;+4
wSeg	dw ?	;+8
wRes	dw ?	;+10
wNumAvail	dw ?	;+12
wNumUsed	dw ?	;+14
EDDS ends

;--- EDDS suffix for regions

EDDSRG struc
dwPhysAddr	dd ?	;+16
dwSizeRg	dd ?	;+20
EDDSRG ends

;--- EDDS suffix for PTEs

EDDSPT struc
dwPTE       dd ?	;+16
EDDSPT ends

;--- EDDS, used by 05-06, for PTEs

;--- handle VDS in protected mode
;--- DS=flat, others=NULL
;--- ESP->V86FRGP (includes EAX, EBX, ECX)
;--- other std registers not modified
;--- segment registers in V86FRGP

;    assume FS:V86SEG

vds_handler_pm proc

;	mov ax, V86_DATA_SEL
;   mov fs, eax
    push ebp
    lea ebp, [esp+4]
    push edi
	movzx eax, byte ptr [EBP].V86FRGP.gEAX

if 0;?VDSDBG
	@DbgOuts <"VDS entry, al=">,1
    @DbgOutb al,1
    @DbgOuts <10>,1
endif
	mov	VDS_retcode,0

	cmp al,0Ch
    jbe @@vds_funcok
	mov al,00
@@vds_funcok:
    inc [ebp].V86FRGP.gIP                   ; skip breakpoint
    bt bDDS, eax
    jnc @@nodds
;--- make DDS/EDDS accessible
	movzx	edi,di
	movzx	ecx,[ebp].V86FRGP.gES 
    shl		ecx, 4
    add		edi, ecx
@@nodds:
	mov		ecx,ds
    mov		es,ecx
    call 	[VDS_CALL_TABLE+eax*4]
    pop edi
	pop ebp
    pop ecx
    pop ebx
    pop eax
    add esp,4+4	;skip error code + 16-bit ret addr
    IRETD
vds_handler_pm endp

vds_reserved proc
	mov	word ptr [ebp].V86FRGP.gIP, offset RSEG:int4b_old
    ret
vds_reserved endp

vds_scatterunlock proc	;dummy, just fall through tds_ok
vds_scatterunlock endp

vds_ok proc
	and byte ptr [ebp].V86FRGP.gFL, not 1   ; flag no-op success
    ret
vds_ok endp

vds_unsup proc
	mov al, 0Fh		; error "function not supported"
vds_unsup endp		; fall through

vds_fail proc
	mov byte ptr [ebp].V86FRGP.gEAX,al		; set code in AL
	or  byte ptr [ebp].V86FRGP.gFL, 1       ; set Carry
    stc
    ret
vds_fail endp
    
;--- int 4b, ax=8102h

vds_version proc
	mov al,10h	;"reserved bit set in dx"
	and dx,dx
    jnz vds_fail
	mov word ptr [ebp].V86FRGP.gEAX, 100h   ; major/minor spec version
	mov word ptr [ebp].V86FRGP.gEBX, 1      ; product number
	mov	word ptr [ebp].V86FRGP.gECX, 1      ; product revision
    mov di, word ptr [DMABuffSize+0]
    mov word ptr [ebp-2*4],di				; EDI is saved on stack!
    mov si, word ptr [DMABuffSize+2]
	xor	dx,dx		; flags
	jmp	vds_ok
vds_version endp

;--- int 4b, ax=8107h, request DMA buffer
;--- DX = flags, bit 1: copy data into buffer
;--- ES:DI -> DDS

vds_reqbuff proc

    mov [edi].DDS.wID, 0
	mov ecx, [edi].DDS.dwSize
if ?VDSDBG
	@DbgOuts <"VDS request buff, flgs=">,1
    @DbgOutw dx,1
	@DbgOuts <" siz=">,1
	@DbgOutd ecx,1
	@DbgOuts <10>,1
endif
    and ecx, ecx
    jz vds_ok
	mov	al,5		; error "region too large for buffer"
    add ecx, 400h-1	; align to kB
    and cx, 0FC00h
    cmp ecx, [DMABuffSize]
    ja vds_fail
    
;--- scan for a free region in buffer

	mov eax, [DMABuffSize]
    shr eax, 10			; transform in KB
    inc eax				; the bit vector is 1-based!
    shr ecx, 10
    inc ecx				; test for 1 bit more!
@@rescan:    
    mov ebx, ecx
@@nextbit:
	dec eax
    js @@fail
    bt [DMABuffFree], eax
    jnc @@rescan
	dec ebx
    jnz @@nextbit
    
    dec ecx		; now use the real size
    inc eax		; skip the last bit found

;--- a free region large enough found

if ?VDSDBG
	@DbgOuts <"VDS free region found at ">,1
    @DbgOutw ax,1
	@DbgOuts <10>,1
endif

	mov ebx, eax
    mov [edi].DDS.wID, ax
@@marknextbit:
	btr ss:[DMABuffFree], ebx
    inc ebx
    loop @@marknextbit
    dec eax
    shl eax, 10		;convert to byte
    add eax, [DMABuffStartPhys]
	mov [edi].DDS.dwPhys, eax
if ?VDSDBG
	@DbgOuts <"VDS req buff, phys=">,1
    @DbgOutd eax,1
	@DbgOuts <10>,1
endif
    test dl,VDSF_COPY		; copy into buffer?
	jz vds_ok
    xor ecx, ecx
    jmp vds_copyinbuffEx
@@fail:    
if ?VDSDBG
	@DbgOuts <"VDS req buff aborted, eax=">,1
    @DbgOutd eax,1
	@DbgOuts <", ebx=">,1
    @DbgOutd ebx,1
	@DbgOuts <10>,1
endif
    mov al, 6				; error "buffer currently in use"
    jmp vds_fail
    
vds_reqbuff endp

;--- int 4b, ax=8104h, unlock region
;--- DX=Flags, bit 1: copy data out of buffer
;--- ES:DI -> DDS
;--- "unlock region" is the same as "release buffer"

vds_unlock proc

if 0;?VDSDBG
	@DbgOuts <"VDS unlock flgs=">,1
    @DbgOutw dx,1
    @DbgOuts <" addr=">,1
	@DbgOutw [edi].DDS.wSeg,1
	@DbgOuts <":">,1
	@DbgOutd [edi].DDS.dwOfs,1
	@DbgOuts <" siz=">,1
	@DbgOutd [edi].DDS.dwSize,1
	@DbgOuts <" id=">,1
	@DbgOutw [edi].DDS.wID,1
	@DbgOuts <" phys=">,1
	@DbgOutd [edi].DDS.dwPhys,1
	@DbgOuts <10>,1
endif
    
vds_unlock endp	;fall through


;--- int 4b, ax=8108h, release DMA buffer
;--- DX = flags, bit 1: copy data out of buffer
;--- ES:DI -> DDS

vds_relbuff proc

    movzx ebx, [edi].DDS.wID
if ?VDSDBG
	@DbgOuts <"VDS release buff, flgs=">,1
    @DbgOutw dx,1
	@DbgOuts <" DDS=[siz=">,1
	@DbgOutd [edi].DDS.dwSize,1
	@DbgOuts <" ID=">,1
	@DbgOutw bx,1
	@DbgOuts <" addr=">,1
	@DbgOutd [edi].DDS.dwPhys,1
	@DbgOuts <"]",10>,1
endif
if 0;?VDSDBG
	@DbgOuts <"DMA buffer bits:">,1
    @DbgOutd DMABuffFree+0,1
	@DbgOuts <"-">,1
    @DbgOutd DMABuffFree+4,1
	@DbgOuts <"-">,1
    @DbgOutd DMABuffFree+8,1
	@DbgOuts <"-">,1
    @DbgOutd DMABuffFree+12,1
	@DbgOuts <"-">,1
    @DbgOutb <byte ptr DMABuffFree+16>,1
	@DbgOuts <10>,1
endif
    and ebx, ebx
    jz vds_ok

	mov al,0Ah	;"invalid buffer id"
    cmp bx, ?DMABUFFMAX
    ja vds_fail

;--- the bit at position -1 must be "1"
;--- and bit 0 must be "0"
;--- the region ends with a "1" bit

	movzx ebx, bx
    dec ebx
    bt [DMABuffFree], ebx
    jnc vds_fail
    inc ebx
    bt [DMABuffFree], ebx
    jc vds_fail

    test dl, VDSF_COPY
    jz @@nextbit
    push ebx
    xor  ecx, ecx
    call vds_copyoutbuffEx
	pop ebx    
    jc @@error

@@nextbit:
    bts ss:[DMABuffFree], ebx
    inc ebx
    jnc @@nextbit
    jmp vds_ok
@@error:
	ret

vds_relbuff endp

;-- test if a buffer is valid and as large as required
;-- used by copy into/out of DMA buffer
;-- ECX == offset in buffer
;-- returns with EBX == linear address of src/dst in DMA-buffer

vds_testbuff proc    

    movzx ebx, [edi].DDS.wID
if ?VDSDBG
	@DbgOuts <"VDS testbuff id=">,1
    @DbgOutw bx,1
	@DbgOuts <10>,1
endif    
	mov	al,0Ah					;"invalid buffer ID"
    and ebx, ebx
	jz	@@fail
    cmp bx, ?DMABUFFMAX
    ja  @@fail
    dec ebx
    bt [DMABuffFree], ebx	;bit at -1 *must* be set
    jnc @@fail
    inc ebx
    bt [DMABuffFree], ebx	;bit at 0 bit *must* be clear
    jc @@fail
    mov eax, ecx			;eax == offset in DMA buffer
    
    mov ecx, [edi].DDS.dwSize
    and ecx, ecx
    jz @@ok

	lea ecx, [ecx+eax+3FFh]
    shr ecx, 10

	push eax
	lea eax, [ebx+ecx]
    cmp eax, ?DMABUFFMAX
    pop eax
    ja @@fail3
    
    push ebx
@@nextbit:
	bt [DMABuffFree], ebx
    jc @@fail2
    inc ebx
    dec ecx
    jnz @@nextbit
	pop ebx    
	
    dec ebx
    shl ebx, 10
    add ebx, [DMABuffStart]
    add ebx, eax
@@ok:    
if ?VDSDBG
	@DbgOuts <"VDS testbuff ok, buff start (linear)=">,1
    @DbgOutd ebx,1
    @DbgOuts <10>,1
endif    
	clc
    ret
@@fail2:
if ?VDSDBG
	@DbgOuts <"VDS testbuff @@fail2, ebx=">, 1
    @DbgOutd ebx, 1
	@DbgOuts <" ecx=">, 1
    @DbgOutd ecx, 1
    @DbgOuts <10>, 1
endif    
	pop ebx
@@fail3:
	mov al,0Bh	;"copy out of buffer range"
@@fail:
	@DbgOuts <"VDS testbuff failed, al=">,?VDSDBG
	@DbgOutb al,?VDSDBG
	@DbgOuts <10>,?VDSDBG
	stc
    ret
	
vds_testbuff endp

;--- int 4b, ax=8109h, copy into DMA buffer
;--- DX = 0000
;--- BX:CX = offset in buffer
;--- ES:DI -> DDS

vds_copyinbuff proc

    mov cx, word ptr [ebp].V86FRGP.gEBX
    shl ecx, 16
    mov cx, word ptr [ebp].V86FRGP.gECX	;ecx == offset in DMA buffer
if ?MASM    
vds_copyinbuffEx::
else
vds_copyinbuffEx:
endif
	call vds_testbuff
    jc vds_fail
    mov ecx, [edi].DDS.dwSize
    movzx eax, [edi].DDS.wSeg
    shl eax, 4
    add eax, [edi].DDS.dwOfs
if ?VDSDBG
	@DbgOuts <"VDS copyinbuff src=">,1
	@DbgOutd eax,1
	@DbgOuts <" dst=">,1
	@DbgOutd ebx,1
	@DbgOuts <" siz=">,1
	@DbgOutd ecx,1
	@DbgOuts <10>,1
endif    
    pushad
    mov esi, eax
    mov edi, ebx
    mov dl,cl
    shr ecx, 2
    cld
    rep movs dword ptr [edi], [edi]
    mov cl,dl
    and cl, 3
    rep movs byte ptr [edi], [edi]
    BIG_NOP
    popad
	jmp	vds_ok
    
vds_copyinbuff endp

;--- int 4b, ax=8109h, copy out of DMA buffer
;--- DX = 0000
;--- BX:CX = offset in buffer
;--- ES:DI -> DDS

vds_copyoutbuff proc

    mov cx, word ptr [ebp].V86FRGP.gEBX
    shl ecx, 16
    mov cx, word ptr [ebp].V86FRGP.gECX	;ecx == offset in DMA buffer
if ?MASM    
vds_copyoutbuffEx::    
else
vds_copyoutbuffEx:
endif
	call vds_testbuff
    jc vds_fail
    mov ecx, [edi].DDS.dwSize
    movzx eax, [edi].DDS.wSeg
    shl eax, 4
    add eax, [edi].DDS.dwOfs
    pushad
    mov esi, ebx
    mov edi, eax
    mov dl,cl
    shr ecx, 2
    rep movs dword ptr [edi], [edi]
    mov cl,dl
    and cl, 3
    rep movs byte ptr [edi], [edi]
    BIG_NOP
    popad
	jmp	vds_ok

vds_copyoutbuff endp

;--- int 4b, ax=8103h, lock region
;--- ES:DI -> DDS
;--- DX=Flags
;--- 0:reserved
;--- 1:data should be copied into buffer (if necessary) [requires 2 cleared]
;--- 2:buffer disable (buffer should not be allocated if noncontiguous or crosses 64/128 kB)
;--- 3:dont attempt automatic remap
;--- 4:region must not cross 64 kb border
;--- 5:region must not cross 128 kb border

;--- there are several cases:
;--- 1. region is contiguous (and does not cross borders)
;---    return with carry clear, buffer ID == 0
;--- 2. region is not contiguous and/or does cross borders
;---    2.1 buffer disable flag set
;---        return with carry set and code 1,2,3 
;---    2.2 buffer disable flag cleared
;---       2.2.1 buffer available and large enough
;---             alloc buffer
;---             if copy required copy data into buffer
;---             return with carry clear and buffer ID <> 0
;---       2.2.2 buffer too small
;---             return with carry set and code 5
;---       2.2.3 buffer not available
;---             return with carry set and code 6
;---
;---     field "Physical Address" may be filled by "Lock Region"

vds_lock proc

	push	edx

if ?VDSDBG
	@DbgOuts <"VDS lock flgs=">,?VDSDBG
    @DbgOutw dx,?VDSDBG
    @DbgOuts <" addr=">,?VDSDBG
	@DbgOutw [edi].DDS.wSeg,?VDSDBG
	@DbgOuts <":">,?VDSDBG
	@DbgOutd [edi].DDS.dwOfs,?VDSDBG
	@DbgOuts <" siz=">,?VDSDBG
	@DbgOutd [edi].DDS.dwSize,?VDSDBG
	@DbgOuts <10>,?VDSDBG
endif

	xor eax, eax
	mov	ebx, [edi].DDS.dwSize	; region size
	cmp	ebx, eax
if 0
	je	@@locksuccess		; zero byte-sized region always works
else
	setz al 		;size 0 is always ok, but the physical address must be set
    add ebx, eax	;as well. So handle size 0 as if it is size 1
endif
	mov ax, [edi].DDS.wSeg
	shl	eax,4
	add	eax, [edi].DDS.dwOfs
;    jc --> overflow error
	mov	ecx,eax		; ecx == start linear address
    lea eax, [eax+ebx-1]
    cmp eax, 400000h; region in first 4 MB ?
	jb	@@below4MB
	mov	eax,ecx		; restore linear address to eax

;-- assume linear == physical
;-- call a simplified version of vds_contiguous

	call vds_contiguous2
    jnc @@locksuccess
    jmp @@lenfail

@@below4MB:
	mov ebx, eax	; ebx == final linear address
	shr	ebx,12		; convert to 4K frame
	mov eax, ecx
	shr	eax,12		; convert to 4K frame
	call vds_contiguous
	jc @@notcontiguous
    and ch, 0Fh
    or ax, cx
    jmp @@locksuccess

; physical memory is noncontiguous, error code is 1 or 2
; return maximum length which would be ok

@@notcontiguous:

	mov eax, edx
	and	ecx, 0fffh
	sub eax, ecx
    mov dx, [esp]		;restore flags
    test dl, VDSF_NOBUFFER ;buffering disabled?
    jnz @@nobuffering
    push eax
    call vds_reqbuff
    pop ecx
	jnc @@lockok
    mov VDS_retcode,al
    mov eax, ecx
@@nobuffering:    

if ?VDSDBG
	@DbgOuts <"VDS lock failed, ret size=">,1
    @DbgOutd eax,1
	@DbgOuts <" addr=">,1
    @DbgOutd edx,1
	@DbgOuts <" rc=">,1
    @DbgOutb VDS_retcode,1
	@DbgOuts <10>,1
endif

@@lenfail:
	mov	[edi].DDS.dwSize,eax	; update maximum contiguous length
	mov	[edi].DDS.wID,0       ; zero buffer id?
	pop edx
	mov	al,VDS_retcode
	jmp	vds_fail

@@locksuccess:
if ?VDSDBG
	@DbgOuts <"VDS lock ok, ret size=">,1
    @DbgOutd [edi].DDS.dwSize,1
	@DbgOuts <" phys=">,1
    @DbgOutd eax,1
	@DbgOuts <10>,1
endif
	mov	[edi].DDS.dwPhys,eax ; physical address
	mov	[edi].DDS.wID,0      ; zero buffer id

@@lockok:
	pop	edx
    jmp vds_ok

vds_lock endp

;--- int 4b, ax=8105h
;--- inp: ES:DI -> EDDS
;---      DX=flags
;--- out: NC
;---      wNumUsed: no of entries used to describe region
;---      regions/PTEs behind EDDS are set
;--- error: C set
;---        AL=error code
;---        dwSize: length that can be locked with current entries
;---        wNumUsed: number of entries required to fully describe region!

vds_scatterlock proc

	push	edx

if ?VDSDBG
	@DbgOuts <"VDS scatlock flgs=">,1
    @DbgOutw dx,1
    @DbgOuts <" addr=">,1
	@DbgOutw [edi].EDDS.wSeg,1
	@DbgOuts <":">,1
	@DbgOutd [edi].EDDS.dwOfs,1
	@DbgOuts <" siz=">,1
	@DbgOutd [edi].EDDS.dwSize,1
	@DbgOuts <" avl=">,1
	@DbgOutw [edi].EDDS.wNumAvail,1
	@DbgOuts <10>,1
endif

	mov	ebx,[edi].EDDS.dwSize
	xor	ecx,ecx		; cx holds entries used
	mov	eax,ecx		; zero eax for later calcs
	mov	[edi].EDDS.wNumUsed,cx	; EDDS number entries used
	cmp	ebx,ecx
	je	@@lockok	; zero sized region always successful

	mov	ax,[edi].EDDS.wSeg
	shl	eax,4
	add	eax,[edi].EDDS.dwOfs

	test dl,40h		; PTEs flagged?
	jne	@@getptes

	mov	edx,eax			; edx == start linear address
	shr	eax,12			; convert start to 4K frame
	cmp	eax,256
	jb	@@checklock		; inside of UMB remapping range (<1M)

; outside of UMB remapping range, assuming linear == physical
	mov	[edi].EDDS.wNumUsed,1   ; one region used/needed
	cmp	cx,[edi].EDDS.wNumAvail
	jnc	@@notenoughregions
	mov	[edi+size EDDS].EDDSRG.dwPhysAddr,edx	; region physical address
	mov	[edi+size EDDS].EDDSRG.dwSizeRg,ebx		; region size
@@lockok:
	pop edx
	jmp	vds_ok
@@notenoughregions:
    pop edx
	mov	al,9		;error "NumAvail too small"
    jmp vds_fail

;--- return regions, 1. MB

@@checklock:
	push	esi

    push	edx			; save start linear address
    
	lea	ebx,[ebx+edx-1]	; ebx == final linear address
	shr	ebx,12			; convert to 4K frame
	call GetPhysAddr	; expects page in AX!
	mov	esi,edx			; current physical address of 4K page
	cmp	cx,[edi].EDDS.wNumAvail
	jnc	@@bumpused
	mov	[edi+size EDDS].EDDSRG.dwPhysAddr,edx
	mov	[edi+size EDDS].EDDSRG.dwSizeRg,0
	jmp	@@bumpused

@@entryloop:
	cmp	cx,[edi].EDDS.wNumAvail	; entries available count
	jnc	@@bumpused
	mov	[edi+ecx*8+size EDDS].EDDSRG.dwSizeRg,1000h	; init region size
	mov [edi+ecx*8+size EDDS].EDDSRG.dwPhysAddr, esi

@@bumpused:
	inc	[edi].EDDS.wNumUsed	; bump count of used/needed entries

@@nextframe:
	inc	eax				; next linear page frame
	cmp	eax,ebx
	ja	@@scatdone		; no more regions to map
	cmp	ah,0  			; page below 100h?
	jz	@@next2			; not at end of first 1M
	cmp	ax,256
	ja	@@scatdone		; finishing off final region entry
	cmp	esi,0ff000h		; end of 1M, see if final 4K block was identity mapped
	je	@@scatdone		; yes

; start new region for final
	inc	ecx
	mov	esi, 100000h
	jmp	@@entryloop

@@next2:
	add	esi, 1000h
	call GetPhysAddr	;expects page in AX!
	cmp	edx, esi
	je	@@samereg
	inc	ecx				; have to move to next region/entry
	mov	esi,edx			; update current physical address
	jmp	@@entryloop

@@samereg:
	cmp	cx,[edi].EDDS.wNumAvail	; entries available count
	jnc	@@nextframe
	add	[edi+ecx*8+size EDDS].EDDSRG.dwSizeRg,1000h
	jmp	@@nextframe

; calculate final region byte size or maximum allowed
@@scatdone:
	pop edx
	cmp	[edi].EDDS.wNumAvail,0	; entries available count
    jz  @@noregions
	and	edx,0fffh
	add	[edi+size EDDS].EDDSRG.dwPhysAddr, edx
	mov	ebx,1000h
	sub	ebx,edx	
	add	[edi+size EDDS].EDDSRG.dwSizeRg,ebx
@@noregions:
	xor	ebx,ebx
	mov	edx,ebx
	movzx ecx,[edi].EDDS.wNumUsed	; number regions used (cannot be 0)
    mov al,0
	cmp cx, [edi].EDDS.wNumAvail
	jbe	@@finalloop
	mov	cx,[edi].EDDS.wNumAvail	; only count up to minimum of available/used
    mov al,9
if ?VDSDBG
	and ecx, ecx
    jz @@scatfail
else
    jecxz @@scatfail 
endif    
@@finalloop:
	mov	esi,edx			; keep previous, last known valid value
	add	edx,[edi+ebx*8+size EDDS].EDDSRG.dwSizeRg
	inc	ebx
	loop @@finalloop
	cmp	al,0
	jne	@@scatfail		; not all regions represented, update EDDS.dwSize

	mov	edx,[edi].EDDS.dwSize	; update final region byte count
	sub	edx,esi
	dec	ebx
    mov [edi+ebx*8+size EDDS].EDDSRG.dwSizeRg, edx
if ?VDSDBG
	@DbgOuts <"VDS scatlock exit, rc=0, used=">,1
	@DbgOutw [edi].EDDS.wNumUsed, 1
    @DbgOuts <", addr=">,1
	@DbgOutd [edi+size EDDS].EDDSRG.dwPhysAddr, 1
    @DbgOuts <", siz=">,1
	@DbgOutd [edi+size EDDS].EDDSRG.dwSizeRg, 1
	@DbgOuts <10>,1
endif
	pop	esi
	pop edx
	jmp vds_ok
@@scatfail:
	mov	[edi].EDDS.dwSize,edx
if ?VDSDBG
	@DbgOuts <"VDS scatlock exit, rc=">,1
	@DbgOutb al,1
    @DbgOuts <", siz=">,1
	@DbgOutd [edi].EDDS.dwSize, 1
	@DbgOuts <" used=">,1
	@DbgOutw [edi].EDDS.wNumUsed, 1
	@DbgOuts <10>,1
endif
	pop	esi
	pop edx
	jmp	vds_fail
    
;--- return PTEs

@@getptes:
    push eax
    
	shr	eax,12			; convert linear start to PTE index
    push esi
    @GETPTEPTR esi, 1000h, 1
@@loop:
	xor edx, edx
    cmp eax, 400h		; don't cross page table 0 border!
    jnc @@noPTE
    mov edx, [esi+eax*4]
    and dx, 0F001h
@@noPTE:    
	cmp cx, [edi].EDDS.wNumAvail
    jnc @@noPTEupdate
    mov [edi+ecx*4+size EDDS].EDDSPT.dwPTE, edx
@@noPTEupdate:
	inc ecx
    inc eax
    sub ebx, 1000h
    ja @@loop
    mov [edi].EDDS.wNumUsed,cx
    pop esi
    pop eax
    
    pop edx
    and ah,0Fh
	cmp cx, [edi].EDDS.wNumAvail
    ja @@notenoughregions2
	mov word ptr [ebp].V86FRGP.gEBX, ax
    jmp vds_ok
@@notenoughregions2:
	movzx ecx, [edi].EDDS.wNumAvail
    shl ecx, 12
    movzx eax,ax
    sub ecx, eax
	mov	[edi].EDDS.dwSize, ecx
	mov	al,9		;error "NumAvail too small"
    jmp vds_fail

vds_scatterlock endp

;--- disable automatic translation for a DMA channel
;--- BX=DMA channel number
;--- DX=flags (all reserved and must be 0)

vds_disabletrans proc

	mov ebx, [EBP].V86FRGP.gEBX
if ?VDSDBG
	@DbgOuts <"VDS enable translation for channel ">,1
    @DbgOutw bx,1
    @DbgOuts <10>,1
endif
    cmp bx, 8
    mov al, 0Ch		;error "invalid channel"
    jnc vds_fail
    mov al, 10h		;error "reserved flags set in DX"
    and dx, dx
    jnz vds_fail
if ?DMA    
    mov al, 0Dh		;error "disable count overflow"
    movzx ebx, bx
    cmp [DMAChn+ebx*8].cDisable,255
    jz vds_fail
    inc [DMAChn+ebx*8].cDisable
endif
    jmp vds_ok
    
vds_disabletrans endp

;--- enable automatic translation for a DMA channel
;--- BX=DMA channel number
;--- DX=flags (all reserved and must be 0)
    
vds_enabletrans proc

	mov ebx, [EBP].V86FRGP.gEBX
if ?VDSDBG
	@DbgOuts <"VDS disable translation for channel ">,1
    @DbgOutw bx,1
    @DbgOuts <10>,1
endif
    cmp bx, 8
    mov al, 0Ch		;error "invalid channel"
    jnc vds_fail
    mov al, 10h		;error "reserved flags set in DX"
    and dx, dx
    jnz vds_fail
if ?DMA    
    mov al, 0Eh		;error "disable count underflow"
    movzx ebx, bx
    cmp [DMAChn+ebx*8].cDisable,0
    jz vds_fail
    dec [DMAChn+ebx*8].cDisable
endif    
    jmp vds_ok

vds_enabletrans endp

	assume FS:NOTHING

VDS_retcode equ <>	;undefine stack variable

endif ;?VDS

if 1;?RING0EXC

;--- helper routines

dwordout proc near		;--- display DWORD in eax into EDI
		push	eax
        shr		eax,16
		call	wordout
		pop 	eax
dwordout endp        
wordout proc near		;--- display WORD in ax into EDI
		push	eax
		mov 	al,ah
		call	byteout
		pop 	eax
wordout endp
byteout proc near		;--- display BYTE in al into EDI
		push	eax
		shr 	al,4
        call    nibout
        pop		eax
nibout:        			;--- display NIBBLE in al[0..3] into EDI
		and 	al,0Fh
		cmp 	al,10
		sbb 	al,69H
		das
        stosb
        ret
byteout endp

;--- ES=V86SEG
;--- [ESP+4]: item descriptor

renderitems proc
	pop		eax
    pop		esi
    push	eax
@@nextitem:    
    lods	dword ptr es:[esi]
    cmp		al,-1
    jz		@@done_exc
    mov		ebx, eax
    shr		eax,16
    mov		edi, eax
    movsx	eax, bh
	MOV     eax, ss:[ebp+eax]
    push	offset @@nextitem
    cmp		bl,2
    jz		byteout
    cmp		bl,4
    jz		wordout
    jmp 	dwordout
@@done_exc:
	retn
renderitems endp


exc_str label byte
    DB 13,10,"JEMM386: exception "
exc_no db 2 dup (' ')
    db " occured at CS:EIP="
exc_cs db 4 dup (' ')
	db ':'
exc_eip db 8 dup (' ')
	db 13,10
	db "SS:ESP="
exc_ss db 4 dup (' ')
	db ':'
exc_esp db 8 dup (' ')
	db " EBP="
exc_ebp db 8 dup (' ')
    db " EFL="
exc_efl db 8 dup (' ')
    DB " CR0="
exc_cr0 db 8 dup (' ')
    DB " CR2="
exc_cr2 db 8 dup (' ')
	db 13,10
    db "EAX="
exc_eax db 8 dup (' ')
	db " EBX="
exc_ebx db 8 dup (' ')
	db " ECX="
exc_ecx db 8 dup (' ')
	db " EDX="
exc_edx db 8 dup (' ')
    db " ESI="
exc_esi db 8 dup (' ')
	db " EDI="
exc_edi db 8 dup (' ')
	db 13,10
exc_reboot db 0
    DB "Press RETURN to Reboot"
    db 0

exc_str2 label byte
	db "DS="
exc_ds db 4 dup (' ')
	db " ES="
exc_es db 4 dup (' ')
	db " FS="
exc_fs db 4 dup (' ')
	db " GS="
exc_gs db 4 dup (' ')
    db ' [CS:IP]='
exc_csip db 8*3 dup (' ')
    db CR,LF,'Press ESC to abort program ', 0
	db 0

exc_str3 db 13,'JEMM386: unable to continue. Please reboot '
	db 0

DEFWORD macro x
if ?MASM
	dw LOWWORD(x)
else
	dd x
    org $ - 2
endif
	endm

exc_format label byte
	db 2, EXCFR.gRet
	DEFWORD <offset exc_no>
    db 4, EXCFR.gCS
    DEFWORD <offset exc_cs>
    db 8, EXCFR.gIP
    DEFWORD <offset exc_eip>
    db 4, -4
    DEFWORD <offset exc_ss>
    db 8, -8
    DEFWORD <offset exc_esp>
    db 8, PUSHADS.rEBP
    DEFWORD <offset exc_ebp>
    db 8, EXCFR.gFL
    DEFWORD <offset exc_efl>
    db 8, -12
    DEFWORD <offset exc_cr0>
    db 8, -16
    DEFWORD <offset exc_cr2>
    db 8, PUSHADS.rEAX
    DEFWORD <offset exc_eax>
    db 8, PUSHADS.rEBX
    DEFWORD <offset exc_ebx>
    db 8, PUSHADS.rECX
    DEFWORD <offset exc_ecx>
    db 8, PUSHADS.rEDX
    DEFWORD <offset exc_edx>
    db 8, PUSHADS.rESI
    DEFWORD <offset exc_esi>
    db 8, PUSHADS.rEDI
    DEFWORD <offset exc_edi>
    db -1

exc_v86segregs label byte
    db 4, EXCFR.gDS
    DEFWORD <offset exc_ds>
    db 4, EXCFR.gES
    DEFWORD <offset exc_es>
    db 4, EXCFR.gFS
    DEFWORD <offset exc_fs>
    db 4, EXCFR.gGS
    DEFWORD <offset exc_gs>
    db -1

endif


if ?DBGOUT

;--- display DWORD in eax

VDWORDOUT proc
		push	eax
        shr		eax,16
		call	VWORDOUT
		pop 	eax
VDWORDOUT endp        
VWORDOUT proc
		push	eax
		mov 	al,ah
		call	vbyteout
		pop 	eax
VWORDOUT endp
VBYTEOUT proc
		pushfd
		push	eax
        mov		ah,al
		shr 	al,4
        call    vnibout
        mov		al,ah
        call    vnibout
        pop     eax
        popfd
        ret
VBYTEOUT endp
VNIBOUT proc        
		and 	al,0Fh
		cmp 	al,10
		sbb 	al,69H
		das
		jmp     VPUTCHR
VNIBOUT endp        

VPUTCHR PROC
	push ds
	PUSHAD
    push	FLAT_DATA_SEL
    pop		ds
if ?USEMONO    
	mov		edi,0B0000h
    mov		ebx,7
else
    MOV		EDI,0B8000h
	CMP		BYTE ptr DS:[463h],0B4h
	JNZ     @@IS_COLOR
    XOR     DI,DI
@@IS_COLOR:
    movzx   EBX, WORD PTR DS:[44Eh]
    ADD     EDI, EBX
    MOVZX   EBX, BYTE PTR DS:[462h]
endif    
	mov		esi, edi
    MOVZX	ECX, BYTE PTR DS:[EBX*2+450h+1]	;ROW
if ?USEMONO
    MOV     EAX, 80
else
    MOVZX   EAX, WORD PTR DS:[44Ah]
endif    
    MUL     ECX
    MOVZX   EDX, BYTE PTR DS:[EBX*2+450h]	;COL
    ADD     EAX, EDX
    MOV     DH,CL
    LEA     EDI, [EDI+EAX*2]
    MOV     AL, [ESP+1Ch]
    CMP     AL, 10
    JZ      @@NEWLINE
    MOV     [EDI], AL
    MOV     byte ptr [EDI+1], 07
    INC     DL
if ?USEMONO
	cmp		dl,80
else
    CMP     DL, BYTE PTR DS:[44Ah]
endif    
    JB      @@OLDLINE
@@NEWLINE:    
    MOV     DL, 00
    INC     DH
if ?USEMONO
    CMP     DH, 24
else
    CMP     DH, BYTE PTR DS:[484h]
endif    
    JBE     @@OLDLINE
    DEC     DH
    CALL    @@SCROLL_SCREEN
@@OLDLINE:    
    MOV     DS:[EBX*2+450h],DX
    POPAD
    pop ds
    RET

;--- scroll screen up 1 line
;--- esi -> start screen

@@SCROLL_SCREEN:
    push es
    push ds
    pop	es
    CLD
	mov   edi,esi
if ?USEMONO
	mov   eax,80
else
	movzx eax,word ptr ds:[44Ah]
endif
	push  eax
    lea   esi, [esi+2*eax]
if ?USEMONO
	mov   CL, 24
else
    MOV   CL, DS:[484h]
endif
	mul   cl
	mov   ecx,eax
	rep   MOVS WORD PTR [EDI], WORD PTR [ESI]
	pop   ecx
	mov   ax,0720h
	rep   stos WORD PTR [EDI]
    pop es
	retn

VPUTCHR ENDP


VPRINTSTR PROC
    XCHG EBX,[ESP]
	PUSH EAX
@@NEXTCHAR:
    MOV AL,CS:[EBX]
    INC EBX
    CMP AL,0
    JZ  @@DONE
    call VPUTCHR
    JMP @@NEXTCHAR
@@DONE:
	POP EAX
	XCHG EBX,[ESP]
    RET
VPRINTSTR endp

endif

if ?VDSDBG

;--- support for 386SWAT: check if int 3 vector still points to
;--- the monitor code segment. If no, assume 386SWAT has intruded

DebugBreak proc
        push  eax
        sub   esp,8
        sidt  [esp]
        mov   eax,[esp+2]
        cmp   word ptr [eax+3*8+4],V86_CODE_SEL
        lea   esp,[esp+8]
        pop   eax
        jnz   swatfound
        ret
swatfound:
		pushfd
        or  byte ptr [esp+1],1	;set TF
        popfd
		ret        
DebugBreak endp

endif

V86_MONITOR     ENDP

if ?ALTBOOT or ?A20PORTS

	align 4

;; Special Ctrl-Alt-Del handler (should be off by default and only
;; be enabled by the ALTBOOT option!). catches Ctrl-Alt-Del 
;; and makes sure that mapping / protected mode
;; is not blocking the proper reboot process.
;
INT15_ENTRY PROC NEAR
if ?ALTBOOT
	cmp		ax,4F53h
    jz 		@@isdel
endif
if ?MASM
INT15_ENTRY_EX::
else
INT15_ENTRY_EX:
endif
if ?A20PORTS
	cmp		ah,24h
    jz		@@isa20
endif    
@@v86mon:
if ?MASM
    push	word ptr LOWWORD(offset INT_TABLE) + 15h*4+4
else
	sub		esp,2
    push	eax
    mov 	eax, offset INT_TABLE + 15h*4+4
    mov		[esp+4],ax
    pop		eax
endif    
	JMP     V86_MONITOR

if ?A20PORTS

;--- catch int 15h, ax=2400h and ax=2401h

@@isa20:
    cmp		al,2
    jnb		@@v86mon
	@DbgOuts <"Int 15h, ax=">,?A20DBG	
	@DbgOutw ax,?A20DBG
	@DbgOuts <" called",10>,?A20DBG	
 if 0
	call	EmuA20
    mov		ah,0
    and		byte ptr [esp].IRETDV86.vFL, not 1
 else
    mov		ah,86h
    or 		byte ptr [esp].IRETDV86.vFL, 1
 endif
    iretd
endif

if ?ALTBOOT    
@@isdel:
	PUSH    EAX
	MOV     AX,FLAT_DATA_SEL
	MOV     DS,EAX
 if ?MASM
    assume	DS:DGROUP	;avoids 32bit offsets with MASM
 else
    assume	DS:NOTHING
 endif    
	MOV     AL,DS:[@KB_FLAG]      ; Have the keys CTRL & ALT
	AND     AL,1100B              ; been pressed ?
	CMP     AL,1100B              ; If not,  continue working
	POP     EAX
	JNZ     @@v86mon
    push	eax
    push	ecx
    MOVZX	eax, [ESP+8].IRETDV86.vSP
    MOVZX	ecx, [ESP+8].IRETDV86.vSS
    SHL		ecx, 4
    add		eax, ecx
    mov		word ptr [eax+0],offset REBOOT_RM
    mov		word ptr [eax+2],RSEG
	POP     ECX                 ; restore Register
	POP     EAX
    JMP		@@v86mon
endif

INT15_ENTRY ENDP

    assume DS:NOTHING

endif

;--- reboot
;--- inp: DS=FLAT
;--- this proc is reached AFTER the v86 int 15h chain has been called

REBOOT_PM proc

if ?FASTBOOT

;--- how to implement FASTBOOT?
;--- the original vectors may be stored at F000:FEE3
;--- but that is far from sufficient.
;--- * what about vectors modified by adapter ROM bios (graphics?)
;--- * what to do with the BIOS memory area at 0040h?

	mov		dl,0
    mov		esi, 0F0000h+0FEE3h
	cmp		word ptr ds:[esi+9*2], 0E987h	;are original vectors stored here?
    jnz		@@nofast
    push	ds
    pop		es
    xor		eax, eax
    mov		edi, eax
    mov		ecx, 100h
    rep		stos dword ptr [edi]
    mov		edi, eax
    mov		cl, 20h
    mov		eax, 0F0000000h
@@rebloop:    
    lods	word ptr [esi]
    stos	dword ptr [edi]
    loop	@@rebloop
    mov		cl,8
    add		edi,50h*4
@@rebloop2:    
    lods	word ptr [esi]
    stos	dword ptr [edi]
    loop	@@rebloop2
    mov		dl,1
@@nofast:
endif

    assume DS:DGROUP 	;make sure offsets are 16 bits
    
	MOV     WORD PTR DS:[@RESET_FLAG],1234H	; 1234h=warm boot

if 1
    MOV		AL,8Fh	;disable NMI, set shutdown byte
    out		70h,al
    MOV		AL,0	;software-reset
    out		71h,al
endif

ife ?USETRIPLEFAULT
	mov		edi,7C00h
    
    mov		dword ptr [edi+0],00FC1220Fh	;0F 22 C1 mov cr0, ecx
    mov		dword ptr [edi+4],0D08ED822h	;0F 22 D8 mov cr3, eax
    											;8E D0    mov ss, ax
    mov		 byte ptr [edi+8],0EAh         ;jmp Ffff:0000
    mov		dword ptr [edi+9],0FFFF0000h
  if ?FASTBOOT
    cmp		dl,1
    jnz		@@nofast2
    mov		word ptr ds:[edi+8], 19CDh
@@nofast2:
  endif
	xor		edx, edx
	push	edx
	push	word ptr 3FFh
	LIDT	FWORD ptr [esp]     ; reset the IDT to real-mode

  if 1
	cmp		edx,cs:[dwFeatures]
    jz		@@nocr4
    @mov_cr4_edx
@@nocr4:
  endif

;	mov		word ptr [GDT+REAL_DATA_SEL+2],dx
;	mov		byte ptr [GDT+REAL_DATA_SEL+4],dl

	MOV     AX,REAL_DATA_SEL    ; before returning to real-mode set the
	MOV     DS,EAX              ; segment register caches
	MOV     ES,EAX
	MOV     FS,EAX
	MOV     GS,EAX
	MOV     ECX,CR0             ; prepare to reset CR0 PE and PG bits
	AND     ECX,7FFFFFFEH
	MOV     SS,EAX              ; set SS:SP
	MOV     ESP,EDI             ; set SP here to save some conv. bytes
	XOR		EAX, EAX
	db 066h                     ; jmp far16 0000:7C00h
	db 0eah
	Dw      7C00h
    dw		REAL_CODE_SEL
else
	xor		edx,edx				; cause a triple fault to reboot
    push	edx
    push	edx
    lidt	fword ptr [esp]
    int		3
endif

    assume	ds:nothing

REBOOT_PM endp

;
; here starts the IDT of the virtual 8086 Monitor.
; in protected mode the IDT looks differently than the IVT in real-mode

INTENTRY_O MACRO SEQNR       ;; Unfortunaly the symbol name INTxx is only
INT_&SEQNR:                  ;; correctly produced in this way.
		ENDM

;--- a 32-bit call doesn't fit in 4 bytes and OTOH it should be easy to get
;--- the int number from the address. So a trick is used here: a 16-bit
;--- call is generated, which pushes the return address as a word. Such a
;--- call still fits in 4 bytes.

INTENTRY MACRO VON,BIS
	LOCAL   ENTRY
ENTRY   =       VON
		REPT    BIS-VON+1
;;		INTENTRY_O %(ENTRY)
		db 66h, 0E8h
        dw V86_MONITOR - ($ + 2)
;;		CALL    V86_MONITOR ;; use the return address as reference
		align 4
ENTRY   =       ENTRY+1
		ENDM
		ENDM

INTENTRYNR MACRO INTRNR,ADDRESS_     ;; Purposefully jump to a routine
		INTENTRY_O %(INTRNR)
        db 66h
        db 0E9h
		dw ADDRESS_ - ($ + 2)
		align 4
		ENDM

		align 8
        
INT_TABLE label dword
if ?BPOPC ne 0F4h
		INTENTRY 0,5
		INTENTRYNR 6,<INT06_ENTRY>
		INTENTRY 7,066h
else
		INTENTRY 0,066h
endif        
		INTENTRYNR 67h,<EMM_ENTRY>; EMM-Driver
		INTENTRY 68h,0FFh

		PURGE   INTENTRY,INTENTRY_O,INTENTRYNR

;
; Here starts the Expanded Memory Manager (EMM) Version 4.0
;

EMM_INDIRECT:
		pop		ecx
        pop		edi
        pop		esi
if ?MASM        
	   	push	word ptr 67h*4+4+LOWWORD(OFFSET INT_TABLE)
else
		sub		esp,2
        push	eax
        mov		eax, 67h*4+4+OFFSET INT_TABLE
        mov		[esp+4],ax
        pop		eax
endif        
		jmp		V86_MONITOR
EMM_ENTRY_EX:
		PUSH	ESI
        PUSH	EDI
        PUSH	ECX
        MOV		CX,FLAT_DATA_SEL
        MOV		DS,ECX
        JMP		EMM_ENTRY_2

		align 4

EMM_ENTRY PROC NEAR

		PUSH    ESI               ; In any case save the endangered
		PUSH    EDI               ; registers
		PUSH    ECX

		MOV     CX,FLAT_DATA_SEL   ; address everything
		MOV     DS,ECX

; hack: int67/87 function = simulated int15/87

		cmp     ah,087h
		je		int6787
if ?A20XMS
		cmp		ah,3fh		; see if special handling function
		je		emm_special3fx
endif        

if ?MASM
		assume ds:DGROUP	;make masm assume a 16-bit segment
endif

        cmp		word ptr ds:[67h*4+2],RSEG	;IVT vector 67h modified?
        jnz		EMM_INDIRECT
if ?MASM        
EMM_ENTRY_2::
else
EMM_ENTRY_2:        
endif
		CLD
        assume	DS:NOTHING
		MOV     ES,ECX
		MOV     CX,V86_DATA_SEL
		MOV     FS,ECX
		ASSUME  FS:V86SEG

if ?VCPI
		cmp	ah,0deh			; see if VCPI function
		jne	not_vcpi_api
		cmp al,0Ch
        jz VCPI_V86toPM
		movzx ecx,al

  if ?VCPIDBG
		@DbgOuts <"VCPI rm, ax=">,1
        @DbgOutw ax,1
        @DbgOuts <" edx=">,1
        @DbgOutd edx,1
        @DbgOuts <" ... ">,1
  endif

		cmp	[_NoVCPI],0	; check if VCPI turned off
		jne	emm_INV_CALL	; yes, return invalid code, flags VCPI not present for 0de00h
		cmp	al,0ch
		ja	emm_INV_CALL	; invalid VCPI call
		call [VCPI_CALL_TABLE+ECX*4]
  if ?VCPIDBG
		@DbgOuts <"VCPI exit, ax=">,1
        @DbgOutw ax,1
        @DbgOuts <", edx=">,1
        @DbgOutd edx,1
        @DbgOuts <10>,1
  endif
		jmp	BYEEMSX
not_vcpi_api:
endif

if ?EMSDBG
@dircall macro func
local isnotfunc
		cmp ah,func
        jnz isnotfunc
        movzx ecx,ah
        call [EMS_CALL_TABLE+ecx*4-40h*4]
        jmp BYEEMSX
isnotfunc:
		endm
;		@dircall 4Fh	;to exclude AH=4Fh from debug displays
;		@dircall 50h
		@dircall 57h
		@DbgOuts <"EMM entry, ax=">,1
        @DbgOutw ax,1
        @DbgOuts <" dx=">,1
        @DbgOutw dx,1
        @DbgOuts <" bx=">,1
        @DbgOutw bx,1
        @DbgOuts <" ... ">,1
endif

		MOVZX   ECX,AH		    ; Set the adress of the wanted Routine.
		SUB     CL,40H		        ; If outside of the permitted range,
		JB      SHORT emm_INV_CALL  ; then report
		CMP     CL,1DH              ; failure/error
		JA      SHORT emm_INV_CALL
		CALL    [EMS_CALL_TABLE+ECX*4]

if ?MASM
BYEEMS::
else
BYEEMS:
endif
if ?EMSDBG
		@DbgOuts <"EMM exit, ax=">,1
        @DbgOutw ax,1
		@DbgOuts <", dx=">,1
        @DbgOutw dx,1
        @DbgOuts <10>,1
endif
BYEEMSX:
		POP     ECX
		POP     EDI
		POP     ESI                  ; restore Register again
		IRETD                        ; 32-Bit-Return !!!
        
if ?MASM
emm_INV_CALL::
else
emm_INV_CALL:
endif
		MOV     AH,84H               ; "Invalid Functioncode in AH"
		JMP     BYEEMS
        
        align	4

int6787:
		mov		es,ecx
		mov		ecx,[esp]	; restore original [e]cx value
		CALL    SIMULATE_INT1587
if 1        
		jc		@@int6787err
        and		dword ptr [esp-4].EMMFRAME.eEfl+1,not 1
		jmp     BYEEMSX
@@int6787err:   
        or		dword ptr [esp-4].EMMFRAME.eEfl+1,1
endif        
		jmp     BYEEMSX

        align	4
        
EMM_ENTRY ENDP

;--- 49, 4A, 55, 56, 5b, 5c, 5d aren't implemented and will call EMS_NOT_IMPL
;--- (so they can be "active" even with NOEMS/NOFRAME)
;--- 41, 47, 48: page frame needed
;---------                 5               4
;---------  FEDCBA9876543210FEDCBA9876543210
;oEmsVec dd 00111111111111111111111111111111b
;oPfVec  dd 00111111111111111111111111111111b

if ?A20XMS

; special function 673Fh, if signatures match
; al contains original AH value
; 3 = global enable
; 4 = global disable
; 5 = local enable
; 6 = local disable

if 1
emm_special3fx:    
;	MOV     CX,V86_DATA_SEL
;	MOV     FS,ECX
endif    

emm_special3f proc

;	cmp	word ptr [esp-4].EMMFRAME.eEcx, 4652h
;	jne	emm_INV_CALL	; must match signature
;	cmp	dx,4545h
;	jne	emm_INV_CALL

;-- must be called from known CS:IP
	cmp	word ptr [esp-4].EMMFRAME.eCS, RSEG
	jne	emm_INV_CALL
	cmp	word ptr [esp-4].EMMFRAME.eEIP, offset BEHIND_A20
	jne	emm_INV_CALL
    
    @DbgOuts <"XMS A20 emulation, ah=">,?A20DBG
    @DbgOutb al,?A20DBG
    @DbgOuts <", curr cnt=">,?A20DBG
    @DbgOutw [wA20],?A20DBG
    @DbgOuts <", curr state=">,?A20DBG
    @DbgOutb [bA20Stat],?A20DBG
    @DbgOuts <10>,?A20DBG
    
    mov cx, word ptr [wA20]
	cmp al,4
    jb @@glen
    jz @@gldi
	cmp al,6
    jb @@loen
    jmp @@lodi
    
@@glen:
	or ch,1
    jmp testa20
@@gldi:
	and ch,not 1
    jcxz testa20
    jmp @@stillenabled
@@loen:
	inc cl
    jz  @@localerr
    jmp testa20
@@lodi:    
	sub cl,1
    jc  @@localerr2
    and cx, cx
    jnz @@stillenabled
testa20:
    and cx, cx
    setnz al
	cmp al, [bA20Stat]
    jz @@notchanged
    mov [bA20Stat],al
    call EmuA20
@@notchanged:
	mov ax,1
    mov bl,0
	mov [wA20],cx
	jmp	BYEEMS
@@localerr2:
if 1			;if this is to be changed, check EDR-DOS first!
	inc cl
    dec ch
    jz  testa20
endif    
@@localerr:
if 1
	xor ax,ax
    mov bl,82h
else
	mov ax,1
endif
    jmp BYEEMS
@@stillenabled:    
	mov [wA20],cx
	xor ax,ax
    mov bl,94h
    jmp BYEEMS

emm_special3f endp

endif

if ?A20PORTS or ?A20XMS

;--- set PTEs for HMA to emulate enable/disable A20
;--- in: AL bit 0 == 1 : enable, else disable

EmuA20 proc    
    cmp [_NoA20], 0
    jnz @@exit
	pushad
    push ds
    mov CX, FLAT_DATA_SEL
    mov ds, ecx
    and  al,1
    shl  al,4		;00h or 10h
	xor	ecx,ecx
	@GETPTEPTR edi, 1000h+256*4+2, 1
@@spec_loop:
	mov [edi+ecx*4],al
	inc	ecx
	cmp	cl,16
	jb	@@spec_loop
if ?INVLPG
	cmp [_NoInvlPg],0
    jnz @@noinvlpg
	mov edx, 100000h
@@nextpte:
    invlpg ds:[edx]
    add dh, 10h
    jnz @@nextpte
    pop ds
    popad
    ret
@@noinvlpg:
endif
; flush TLB to update page maps
	mov	eax,CR3
	mov	CR3,eax
    pop ds
    popad
@@exit:    
    ret

EmuA20 endp

endif

;--- inp: eax = linear address
;--- cl = size in PTEs
;--- ebx = pagedir
;--- edx = page table

if ?WT
?PA	equ 1+2+8	;PRESENT + R/W + WRITE THROUGH
else
?PA	equ 1+2		;PRESENT + R/W
endif

setPTEs proc
    and ax, 0F000h
    or al,?PA
if ?INVLPG    
	cmp [_NoInvlPg],0
    jz @@setPTEs486
endif    
@@nextPTE1:   
    mov [edx], eax
    add eax, 1000h
    add edx,4
    dec cl
	jnz  @@nextPTE1
    ret
if ?INVLPG    
@@setPTEs486:
@@nextPTE2:   
    mov [edx], eax
    invlpg ds:[ebx]
    add edx,4
    add eax, 1000h
    add ebx, 1000h
    dec cl
	jnz  @@nextPTE2
    ret
endif    
setPTEs endp

;************************************************************
; simulate INT15/87
;
;INT 15 - SYSTEM - COPY EXTENDED MEMORY (by RBIL)
;
;        AH = 87h
;        CX = number of words to copy (max 8000h)
;        ES:SI -> global descriptor table
;Return: CF set on error
;        CF clear if successful
;        AH = status 
;
;Values for extended-memory copy status:
; 00h    source copied into destination
; 01h    parity error
; 02h    interrupt error
; 03h    address line 20 gating failed
; 80h    invalid command (PC,PCjr)
; 86h    unsupported function (XT,PS30)
;
;Format of global descriptor table:
;Offset  Size    Description
; 00h 16 BYTEs   zeros (used by BIOS)
; 10h    WORD    source segment length in bytes (2*CX-1 or greater)
; 12h  3 BYTEs   24-bit linear source address, low byte first
; 15h    BYTE    source segment access rights (93h)
; 16h    BYTE    more rights
; 17h    BYTE    8 bit  linear source adress, high
; 18h    WORD    destination segment length in bytes (2*CX-1 or greater)
; 1Ah  3 BYTEs   24-bit linear destination address, low byte first
; 1Dh    BYTE    destination segment access rights (93h)
; 1eh    byte    more rights
; 1fh    BYTE    8 bit  linear source adress, high
;************************************************************

I15MOVE struc
	dq ?
    dq ?
wSrcLim 	dw ?
wSrcA0015	dw ?
bSrcA1623	db ?
wSrcAcc		dw ?
bSrcA2431	db ?
wDstLim 	dw ?
wDstA0015	dw ?
bDstA1623	db ?
wDstAcc		dw ?
bDstA2431	db ?
I15MOVE ends

if 1
i15_error2:
	mov ah,02h
    stc
	ret
endif    

;--- registers ESI, EDI, ECX are saved on host stack!
;--- DS+ES=FLAT
;--- segment registers don't need to be saved!

SIMULATE_INT1587 proc near


;-- MS Emm386 returns with error AH=2 if CX > 8000h!
if 1
	cmp cx, 8000h	
    ja i15_error2
endif

	movzx	ecx,cx				; verify size

;	jecxz	@@ok				; we are done
	or	ecx,ecx
	je	@@ok			; done

	MOVZX   edi,WORD PTR [esp].EMMFRAME.eES	; make edi = linear address of command
	SHL     edi,4
	MOVZX   esi,si
	add     edi,esi

	mov esi, ecx		; save ecx
	lea ecx,[ecx*2-1]   ; verify, source and destination
						; descriptors are ok
						; we don't care about segment access rights
	cmp	cx, [edi].I15MOVE.wSrcLim	; 16-bit overflow not an issue (0->ffff)
	ja	@@invalid_command

	cmp	cx, [edi].I15MOVE.wDstLim
	ja	@@invalid_command

	mov	ecx,esi			; restore ecx

	push edx			; must be protected

						; load the source/destination
						; adresses


	mov dh,[edi].I15MOVE.bSrcA2431	; get linear source address
	mov dl,[edi].I15MOVE.bSrcA1623
	shl edx,16
	mov dx,[edi].I15MOVE.wSrcA0015

	mov esi,edx

	mov dh,[edi].I15MOVE.bDstA2431	; get linear destination address
	mov dl,[edi].I15MOVE.bDstA1623
	shl edx,16
	mov dx,[edi].I15MOVE.wDstA0015

	mov edi,edx

if ?I15DBG
	@DbgOuts <"Int 15h, ah=87h, src=">,1
    @DbgOutd esi,1
	@DbgOuts <", dst=">,1
    @DbgOutd edi,1
	@DbgOuts <", siz=">,1
    @DbgOutw cx,1
	@DbgOuts <10>,1
endif

;-- NOCHECK -> moves for addresses not backuped with RAM/ROM will fail
;-- (cause an exception)

	test [_bV86Flags], V86F_NOCHECK
	je	@@memcheck

	lea edx, [esi+ecx*2]
	cmp	edx, [_TOTAL_MEMORY]
	jae	@@fail
	lea edx, [edi+ecx*2]
	cmp	edx, [_TOTAL_MEMORY]
	jae	@@fail

;-- Now 1 scratch page table is used and there is no more
;-- linear == physical mapping

@@memcheck:
	push eax
    push ebx
    push ecx
					;get no of PTEs involved. This depends on the
                    ;bases of the descriptors. test cases:

	add cx,800h-1	;round up size to page boundary
    shr ecx,11		;words -> PTEs (8000h -> 10h)
    inc ecx			;low12(base) == FFFh, cx=8000h -> 10h+1 PTEs
    mov ch, cl

    cmp esi, 110000h+0000h	;the 111xxx page contains GDT/IDT
    jc  @@src_is_shared
	@GETPTEPTR edx, 2000h+?SCRATCHPTE, 1
    mov eax, esi
    and esi, 0FFFh
	or  esi, ?SCRATCHLINEAR
    mov ebx, esi
    call setPTEs
    mov cl, ch
@@src_is_shared:
    cmp edi, 110000h+0000h
    jc  @@dst_is_shared
    @GETPTEPTR edx, 2000h+?SCRATCHPTE+11h*4, 1
    mov eax, edi
    and edi, 0FFFh
	or  edi, ?SCRATCHLINEAR + 11000h
    mov ebx, edi
    call setPTEs
@@dst_is_shared:
if ?INVLPG
	cmp [_NoInvlPg],0
    jz @@flushdone
endif    
	mov eax, cr3
    mov cr3, eax
@@flushdone:
    pop ecx
    pop ebx
    pop eax

@@accessok:

	shr ecx,1
	REP MOVS DWORD PTR [ESI],DWORD PTR [EDI]
	BIG_NOP
	adc ecx,ecx
	REP MOVS WORD PTR [ESI],WORD PTR [EDI]
	BIG_NOP

	pop edx

@@ok:
	mov   AH,0  ; everything OK and finished
    clc
	RET

@@invalid_command:
	mov ah,80h
    stc
	ret
    
@@fail:
	pop edx
	mov al,ds:[0FFFFFFFCh]
	ret

SIMULATE_INT1587 endp

	align 4

;
; 6740: AH = 40h: return the actual state of the EMM-driver.
;
EMS_GET_STATUS PROC NEAR
	MOV     AH,00                    ; Allways everything OK.
	RET
EMS_GET_STATUS ENDP
;
; 6741: AH = 41h: request the segment address of the EMS-window
;
EMS_GET_PAGE_FRAME_ADDRESS PROC NEAR
	mov		ah,80h
	cmp		[_NoFrame],0
    jnz		@@BYE
	MOV     AH,00                 ; No error occurred
	MOV     BX,[_FRAME]           ; Segment address of EMS-Window/Frame
@@BYE:    
	RET
EMS_GET_PAGE_FRAME_ADDRESS ENDP

;
; 6742: AH = 42h: Request number of ( also maximum) available EMS-pages
;

EMS_GET_UNALLOCATED_PAGE_COUNT PROC NEAR

	call	DynamicEMSPageCount	;update PAGESAVAIL

	MOV     BX,word ptr [_EMSPAGESAVAIL] ; free EMS pages
	MOV     DX,word ptr [_MAXEMSPAGES]   ; total EMS pages

; follow MS-DOS EMM386 7.x lead and don't throttle pages on NOEMS
;	cmp	cs:[_NoEMS],0
;	je	unalloc_ret
;	or	bx,bx
;	je	unalloc_ret
;	mov	bx,1			; only show maximum of 1 EMS page if NOEMS set

unalloc_ret:
	MOV     AH,00
	RET
EMS_GET_UNALLOCATED_PAGE_COUNT ENDP
;
; 6743: AH = 43h: Reserve Memoryspace of the EMS-add-on-card (ahemm..extinct...)
; in  BX = EMS pages to reserve
; out AH=00, DX = handle

EMS_ALLOCATE_PAGES PROC NEAR
	MOV     AH,89H			; "Request, to reserve null pages"
	AND     BX,BX
	JZ      SHORT @@BYE

if ?MASM
allocate_pages_plus_zero::	;this entry allows to alloc zero pages!
else
allocate_pages_plus_zero:
endif
	MOV     AH,87H                  ; "Not enough pages available"
	CMP     BX,word ptr [_MAXEMSPAGES]
	JA      SHORT @@BYE

	call	DynamicEMSPageCount		;update PAGESAVAIL
	MOV     AH,88H                  ; "Not enough pages available anymore"
	CMP     BX,word ptr [_EMSPAGESAVAIL]
	JA      SHORT @@BYE

	MOV     ESI,[EMSSTATUSTABLE]	; Now search for a free Handle in the table
	MOV     CX,MAX_HANDLES		;
@@SEARCH:
	CMP     WORD PTR [ESI],EMSH_FREE; Is there one free ... ?
	JZ      SHORT @@FOUND
	ADD     ESI,8
    dec		cx
	jnz     @@SEARCH
	MOV     AH,85H                  ; "No more free Handles"
@@BYE:
	RET

@@FOUND:
	MOV     WORD PTR [ESI],EMSH_USED ; mark Handle as occupied

	MOV     DX,MAX_HANDLES	    ; Set in DX now the actual Handle-
	SUB     DX,CX				; number

; zero page allocations allowed, so test and bypass code if found

	or		bx,bx
	je		@@allocate_exit
    
	push	ebx
	MOV     EDI,[EMSPAGETABLE]	; mark the pages in the
	MOV     ECX,[_MAXEMSPAGES]	; page-usage-table as used
    PUSH	EAX
	MOV     AL,FREEPAGE_ID
@@SEARCH_PAGES:
	REPNZ   SCAS BYTE PTR [EDI]	; After searching for a free page
	jnz	@@nofind			; free page could not be located
	call GetEMSPage			; allocate the page
	jc	@@nofind
    DEC		[_EMSPAGESAVAIL]
	MOV     [EDI-1],DL		; Assign the handle
	DEC     BX			; Continue until all desired pages are occupied
	JNZ     @@SEARCH_PAGES
    POP		EAX
	POP     EBX
@@allocate_exit:
	MOV     AH,00
	RET
@@nofind:
if ?POOLDBG
	@DbgOuts <"EMS_ALLOCATE_PAGES: @@nofind reached, BX=">,1
	@DbgOutw bx,1
    @DbgOuts <10>,1
endif
	call EMS_DEALLOCATE_PAGES	;free the handle in DX	
	pop eax
	pop	ebx
	MOV AH,88H                  ; "Not enough pages available anymore"
	ret
    
EMS_ALLOCATE_PAGES ENDP

;--- get absolute page
;--- inp BX = logical page
;--- inp DX = handle
;--- out BX = absolute page if NC
;--- modifes ECX, EDI

get_abs_page proc    
	INC     EBX             	; fade out
	XCHG    EAX,EDX				; get the handle to search in AL
	MOV     ECX,[_MAXEMSPAGES]	; the range to search
	MOV     EDI,[EMSPAGETABLE]	; Search the allocation table by the pages
@@LOOP:
	REPNZ   SCAS BYTE PTR [EDI]	; occupied by the handle
	JNZ     @@NIX
	DEC     BX
	JNZ     @@LOOP
	MOV     BX,word ptr [_MAXEMSPAGES]
	SUB     BX,CX               ; abs pagenumber -> BX
	DEC     BX
	XCHG    EAX, EDX
    ret
@@NIX:
	XCHG	EAX, EDX
	MOV     AH,8AH              ; "logical page out of reserved area"
	stc
    ret
get_abs_page endp    

; 6744
; AH = 44h: mirror logical page in the EMS-window
; AL = physical page (0-3)
; DX = handle
; BX = logical page #
;
EMS_MAP_HANDLE_PAGE PROC NEAR
	CMP     AL,[bEmsPhysPages]		; not" Only pages 0..3
	JAE     SHORT @@PAGE_TO_LARGE   ; are allowed!

	CALL    TEST_HANDLE
	PUSH    EBX             ; save BX  (since it is changed)
	AND     BH,BH           ; bx < 0 means unmap this phys. page
	JS      SHORT @@MAP
    call	get_abs_page
    jc		@@NIX
@@MAP:
	CALL    MAP_PAGE
	MOV     AH,00
@@NIX:
	POP     EBX
	RET
@@ERR8B:
	MOV     AH,8BH              ; "Indicated physical page does not exist"
    RET
@@endinit:
	mov	    [_INIT_DONE],1	; finish initialization
    movzx	ecx, word ptr [esp].EMMFRAME.eDS
    shl		ecx, 4
    movzx	esi, word ptr [esp].EMMFRAME.eESI
    add		esi, ecx
    mov		edi, offset pmUMBsegments
    push	ss
    pop		es
    mov		ecx,8
    rep		movs dword ptr [edi], [esi]
	mov	    ah,00
    RET

;*******************************************************************************
; here comes the funny part (by Tom)
;		during initialization phase, calling EMM_MAP_PAGE (0x44)
;		with pysical page (AL) > 3 is possible.
;		meaning:
;
;		AL = highest 8 bits of logical adress, AL=E4 --> address E4000
;
;		initialization phase is terminated with AL=9F
;*******************************************************************************

; since this is used to map UMB's and that first block of memory is set
;  linear==physical even with pool sharing blocks, we shouldn't
;  need to modify any code in here to make it work with pool sharing

@@PAGE_TO_LARGE:
	cmp     [_INIT_DONE],0		; still in init phase ?
	jne     @@ERR8B
	cmp	    al,09fh	            ; AL=9f finishes init phase
	je	    @@endinit

; the fun part - map in page BX at address AL
; BX is interpreted as a 4k page here.
; That is, bits 2-15 are the true logical page
; and bits 0-1 are the page offset in this page

	AND     BH,BH         ; In case the log. page number
	JS      @@ERR8B       ; negative is, no search

; code stolen above
; find the memory for handle/page

	CALL    TEST_HANDLE
	PUSH    EBX

; convert 4K-based logical page to 16K-based
	shr	    bx,2

	call	get_abs_page
    jc		@@NIX
    
;****
;**** stolen from MAP_PAGE (hiword(eax) is destroyed here!)
;****

	MOVZX   EAX,AL                  ; use the page # to get ptr to PTE
	@GETPTEPTR esi, [eax*4+1000h]   ; in ESI   

	MOVZX   EDI,BX                  ; Now calculate, from the pagenumber
	SHL     EDI,12+2                ; (absolute) the situation of the EMS-page
	ADD     EDI,[FIRSTPAGE]         ; in Extended Memory .
	ADD     DI,111B                 ; Statusbits: R/W=1,U/S=1,P=1

;	MOV     CX,4                    ; 1 EMS page 4 (virtual) pages

	POP EBX
	movzx ecx,bl
	and	cl,3		; get 4K multiplier offset into 16K page
	shl	ecx,12		; convert multiplier offset to true 4K offset
	add	edi,ecx

; if segment frame value is 0ffh, then we know we're shadowing the ROM into RAM
; to catch jumps to FFFF:0, so copy ROM image of block to RAM

	cmp	al,0ffh
	jne	noshadow

;-- map the new page in linear address space at linear scratch pos
;-- and copy the content of page AL into in

	push	esi
	push	edi
	@GETPTEPTR ECX, 2000h+?SCRATCHPTE, 1
    mov		ds:[ecx], edi
    mov		edi, cr3            ;flush TLB
    mov		cr3, edi
    mov		edi, ?SCRATCHLINEAR
	movzx	esi,al
	shl	esi,12		; convert segment frame to absolute address
	mov	ecx,1000h/4	; map 4K block in dwords
    rep movs dword ptr [edi], [esi] 
	BIG_NOP
    
	mov	BYTE PTR [edi-10h],?BPOPC	; set breakpoint at FFFF:0000
	pop	edi
	pop	esi
    and	di, not 2           ;since it shadows ROM, make PTE r/o

noshadow:
	MOV     [ESI],EDI       ; set the PTE
    
if ?EMSDBG
	@DbgOuts <"page ">,1
	@DbgOutb al,1
	@DbgOuts <" got PTE ">,1
	@DbgOutd edi,1
	@DbgOuts <", mapped at ">,1
	@DbgOutd esi,1
	@DbgOuts <", PAGESAVAIL=">,1
	@DbgOutd [_EMSPAGESAVAIL],1
	@DbgOuts <10>,1
endif

	MOV     AH,00
if ?INVLPG
	cmp		[_NoInvlPg],0
    jnz		@@noinvlpg
    movzx	esi,al
	shl	esi,12
    invlpg	ds:[esi]
	RET
@@noinvlpg:
endif
	MOV		ESI, CR3
	MOV     CR3, ESI        ; and flush TLB
	RET

;*******************************************************************************
; end of fun :-)
;*******************************************************************************

EMS_MAP_HANDLE_PAGE ENDP

;
; 6745: AH = 45h: Release reserved memoryspace again
;       DX = handle to release
; any pages of this handle mapped in page frame remain mapped!
; this is same behaviour as MS Emm386

EMS_DEALLOCATE_PAGES PROC NEAR
	CALL    TEST_HANDLE
	MOV     AH,86H                  ; "A saved state still
	CMP     WORD PTR [ESI],EMSH_USED; exists" ?
	JNZ     SHORT @@BYE
	MOV     ECX,[_MAXEMSPAGES]		; All pages in the allocation table
	XCHG    EAX,EDX                 ; have to be marked again as free
	MOV     EDI,[EMSPAGETABLE]
@@LOOP:
	REPNZ   SCAS BYTE PTR [EDI]     ; after searching pages with Handlenumber,
	JNZ     SHORT @@OK              ; done?

	call	ReleaseEMSPage			; preserves registers

	MOV     BYTE PTR [EDI-1],FREEPAGE_ID; Mark page as free
	INC     [_EMSPAGESAVAIL]
	JMP     SHORT @@LOOP
@@OK:
	MOV     WORD PTR [ESI],EMSH_FREE ; release Handle

; zero handle name on free
	mov	edi, [EMSNAMETABLE]
    xor esi, esi
	movzx	ecx, al	; handle (index)
	mov	DWORD PTR [edi+ecx*8+0],esi
	mov	DWORD PTR [edi+ecx*8+4],esi

	XCHG    EAX,EDX
	MOV     AH,00                   ; Function executed duly
@@BYE:
	RET
EMS_DEALLOCATE_PAGES ENDP
;
; 6746: AH = 46h: determine Version-number of EMM
;
EMS_GET_VERSION PROC NEAR
;	MOV     AX,0032H                ; Currently version 3.2
	MOV     AX,0040H                ; Currently version 4.0
	RET
EMS_GET_VERSION ENDP

;--- save the frame window mapping state

SAVE_4_TO_ESI proc
	mov		cl,4
SAVE_4_TO_ESI endp	;fall through

;--- save CL (0-4) pages to ESI
;--- modifies CL

SAVE_TO_ESI PROC
	PUSH    EAX
    PUSH	EDX
    XOR		EDX, EDX
    INC		CL
@@NEXTPAGE:
	DEC		CL
    jz		@@done
	MOV     AX, [PHYSPAGETABLE+EDX*2]
	MOV     [ESI+EDX*2],AX
    INC		DL
    JMP		@@NEXTPAGE
@@done:    
    POP		EDX
	POP     EAX
    RET
SAVE_TO_ESI ENDP

;--- restore the frame window mapping state

RESTORE_4_FROM_ESI PROC
	mov		cl, 4
RESTORE_4_FROM_ESI ENDP		;fall through

;-- restore CL (0 - 4) pages for frame
;-- esi -> ems handle (8 bytes, where 4 EMS pages stored are stored)

RESTORE_FROM_ESI PROC

	PUSH    EAX	            ; save everything! Data- and System-
	PUSH    EBX	            ; registers first!
    xor		eax, eax
    INC		CL
@@NEXTPAGE:    
	DEC		CL
    jz		@@done
	MOV     BX,[ESI+EAX*2]
    push	Ecx
    push	Eax
	PUSH    ESI
	CALL    MAP_PAGE		;expects phys. page in AL, log. page in BX
	POP     ESI
    pop		Eax
    pop		Ecx
	INC     EAX
    jmp 	@@NEXTPAGE
@@done:
	POP     EBX
	POP     EAX
	MOV     AH,00           ; report OK (because of functions $4E01/$4E02)
    RET
    
RESTORE_FROM_ESI ENDP


;
; 6747: AH = 47h: Save number of the fit-in page in EMS-window
; DX = handle
; it might be that there are pages mapped into the EMS page frame
; which don't belong to any handles!
;
EMS_SAVE_PAGES PROC NEAR
	mov		ah,80h
	cmp		[_NoFrame],0
    jnz		@@BYE
	CALL    TEST_HANDLE
	MOV     AH,8DH                 ; "State for Handle already saved"
	CMP     WORD PTR [ESI],EMSH_USED
	JNZ     @@BYE
    call	SAVE_4_TO_ESI
	MOV     AH,00                    ; report 'everything OK'
@@BYE:
	RET
EMS_SAVE_PAGES ENDP
;
; 6748: AH = 48h: Restore saved state of the EMS-window
; DX = handle
;
EMS_RESTORE_PAGES PROC NEAR
	mov		ah,80h
	cmp		[_NoFrame],0
    jnz		@@BYE
	CALL    TEST_HANDLE
	MOV     AH,8EH                    ; "A saved stated does not exist"
	CMP     WORD PTR [ESI],EMSH_USED
	JZ      SHORT @@BYE
	CALL    RESTORE_4_FROM_ESI
	MOV     WORD PTR [ESI],EMSH_USED  ; Nothing stored for Handle
	MOV     AH,00                     ; report 'everything OK'
@@BYE:
	RET

EMS_RESTORE_PAGES ENDP

;
; report the failure so that we can maybe support it in the future
;
EMS_NOT_IMPL PROC NEAR

IF	?UNIMPL_EMS_DBG
	mov		edi,offset unimpl_ax
	call	WORDOUT
    mov		esi,offset unimpl_func
	call	PRINTSTR
ENDIF
	MOV     AH,84H                    ; "Invalid function code"
	RET

if ?UNIMPL_EMS_DBG    
unimpl_func db "Unimplemented EMS function called, ax="
unimpl_ax   db "    "
			db 10
            db 0
endif

EMS_NOT_IMPL ENDP
;
; 674B: AH = 4Bh: return Number of open Handles in BX
;
EMS_GET_OPEN_HANDLES_COUNT PROC NEAR
	MOV     ESI,[EMSSTATUSTABLE] ; Search Handle-status-table for
	MOV     ECX,MAX_HANDLES      ; assigned/given handles
	XOR     BX,BX
@@LOOP:
	CMP     WORD PTR [ESI],EMSH_FREE ; Free ?
	JZ      SHORT @@CLOSED
	INC     EBX
@@CLOSED:
	ADD     ESI,8                    ; Next entry.
	loop    @@LOOP
	MOV     AH,00
	RET
EMS_GET_OPEN_HANDLES_COUNT ENDP
;
; 674C: AH = 4Ch: Determine number of reserved pages for a Handle
;  inp: handle in DX.
;  out: pages in BX.
;
EMS_GET_NR_OF_ALLOCATED_PAGES PROC NEAR
	CALL    TEST_HANDLE
	xchg    EAX,EDX                  ; get handle in AL, save EAX in EDX
	MOV     ECX,[_MAXEMSPAGES]
	MOV     EDI,[EMSPAGETABLE]
	XOR     BX,BX
    mov		esi,edi 			;dont remove, esi should return last pos!
@@LOOP:
	REPNZ   SCAS BYTE PTR [EDI]
	JNZ     SHORT @@OK		; No more found, so done
    mov		esi,edi			; update last find pos
	INC     EBX				; one page more
	JMP     @@LOOP
@@OK:
	xchg	EAX,EDX         ; restore EAX and EDX
	MOV     AH,00           ; Ok.
	RET
EMS_GET_NR_OF_ALLOCATED_PAGES ENDP
;
; 674D: AH = 4Dh: determine Number of reserved pages for all Handles
;   ES:DI -> array of 2 WORD entries (handle, pages)
; out: AH=00 success, BX=number of handles stored in array
;
EMS_GET_ALLOCATED_PAGES PROC NEAR
	MOVZX   ESI,WORD PTR [esp].EMMFRAME.eES  ; ES from the Level-0-Stack
	SHL     ESI,4	                    ; ES:DI ^ from the storage area
	MOVZX   EDI,DI
	ADD     ESI,EDI
	PUSH    EAX
	PUSH    EDX
	MOV     EDI,[EMSSTATUSTABLE]
	XOR     EAX,EAX                      ; actual/current Handle-Number
	XOR     BX,BX                        ; sofar no Handle open
@@NEXT_HANDLE:
	CMP     WORD PTR [EDI],EMSH_FREE     ; Assign handle at all ? If
	JZ      SHORT @@NEXT                 ; not, then jump over
	INC     EBX                          ; One more Handle is open...
	MOV     [ESI+0],AX                   ; Place handle
	PUSH    EDI
	MOV     EDI,[EMSPAGETABLE]           ; count reserved pages
	XOR     EDX,EDX                      ; EDX is counter
	MOV     ECX,[_MAXEMSPAGES]
@@LOOP:
	REPNZ   SCAS BYTE PTR [EDI]          ; Search through Page-Allocation-Table
	JNZ     SHORT @@NOPE                 ; No more found
	INC     EDX
	JMP     @@LOOP
@@NOPE:
	MOV     [ESI+2],DX                   ; Set the number of determined pages
    add		ESI, 4
	POP     EDI
@@NEXT:
	ADD     EDI,8                        ; next Handle, please !
	INC     EAX
	CMP     AL,MAX_HANDLES               ; All Handles processed ?
	JB      SHORT @@NEXT_HANDLE
	POP     EDX
	POP     EAX
	MOV     AH,00                        ; Everything ok.
	RET
EMS_GET_ALLOCATED_PAGES ENDP

;
; 674E: AH = 4Eh: Get & Set Map
; AL = 0,1,2,3
; AL = 0: ES:DI -> array to get info
; AL = 1: DS:SI -> array to set info
; AL = 2: DS:SI -> array to set info, ES:DI -> array to get info
; AL = 3: AL returns size of array (bytes)

CHKSUM  MACRO   REG
	MOV     AX,[REG+2]           ; Calculate checksum
	ADD     AX,[REG+4]
	ADD     AX,[REG+6]
	ADD     AX,[REG+8]
	ENDM

EMS_GET_SET_PAGE_MAP PROC NEAR
	CMP     AL,3                    ; Subfunction 0 to 3 ?
	JA      bad_subfunc
	JZ      SHORT @@SUBF_3          ; Size of field
	CMP     AL,1
	JZ      SHORT @@SUBF_1          ; Set Page Map
; AL = 2: Get & Set Page Map
; AL = 0: Get Page Map - save Hardwarestatus
@@SUBF_0:
	MOVZX   ECX, WORD PTR [esp].EMMFRAME.eES; ES:DI ^ convert from statusfield in
	SHL     ECX, 4                  ; usual REAL-Mode-Format
	MOVZX   EDI, DI
	ADD     EDI, ECX
    PUSH	ESI
	PUSH    EAX	                    ; save Subfunctioncode
	LEA		esi, [edi+2]            ; Currently skip checsum
    mov		cl, [bEmsPhysPages]
	CALL    SAVE_TO_ESI
	DEC     ESI
    DEC		ESI
	CHKSUM  ESI                     ; Calculate checksum and ...
	MOV     [ESI],AX                ; ... store
	POP     EAX                     ; restore and examen subfunctioncode
    POP     ESI
	CMP     AL,2                    ; if subfuction 2
	JZ      SHORT @@SUBF_1          ; is wanted , since then also
	MOV     AH,00                   ; Subf. 1 still needs be done.
	RET                             ; Everything OK.

; Subf. 1: Set Page Map - restore Hardwarestatus
@@SUBF_1:
	MOVZX   ECX,WORD PTR [esp].EMMFRAME.eDS	; DS:SI ^ convert from statusfield in
	SHL     ECX,4                    ; usual REAL-Mode-Format
	MOVZX   ESI,SI
	ADD     ESI,ECX
	CHKSUM  ESI                      ; Calculate checksum and check it
	CMP     [ESI],AX
	JNZ     SHORT @@CHKERR
	ADD     ESI,2                    ; Jump over checksum
    mov		cl, [bEmsPhysPages]
	JMP     RESTORE_FROM_ESI

; Checksum is incorrect !
@@CHKERR:
	MOV     AH,0A3H                  ; data is destroyed !
	RET

; Subf. 3: Size of the field
@@SUBF_3:
	MOV     AL, [bEmsPhysPages]
    add		al,al					; need 2 bytes for a page
    add		al,2                    ; + 2 Bytes for checksum
	MOV     AH,00                   ; ChkSum
	RET	                            ; That was it then.

EMS_GET_SET_PAGE_MAP ENDP

bad_subfunc:
	MOV     AH,8FH                   ; Invalid subfunctioncode !
	RET

;--- check if segment in AX is a mapped page
;--- if yes, return NC and convert segment to physical page in AL

IsMappedSegm proc
    sub ax, [_FRAME]
    jc @@notvalid
    test ax,0F3FFh		;valid ax is 0, 400, 800, C00h
    jnz @@notvalid
    shr ax,10			;al = 0, 1, 2, 3
    cmp al, [bEmsPhysPages]
    jae @@notvalid
    clc
    ret
@@notvalid:
    stc
    ret
IsMappedSegm endp

;
; 674F: AH = 4Fh: Get & Set partial Map
; AL = 0,1,2 
; AL = 0 (get map), DS:SI -> map to get, ES:DI -> status to receive
; AL = 1 (set map), DS:SI -> map to restore
; AL = 2 (get size), items in BX, returns size of map in AL
; the list for sub function 0 DS:SI points to has following structure:
; WORD   : items in list
; WORD[] ; segment! addresses for which to get the map info 

?DBG4F equ 0

EMS4_GET_SET_PARTIAL_PAGE_MAP PROC

if ?EMM4F
	cmp al,2
    ja bad_subfunc
    jz @@getsize
	movzx ecx,WORD PTR [esp].EMMFRAME.eDS
	shl	ecx,4
    movzx esi, si
	add	esi,ecx
    
    cmp al,1
    jz @@setmap

	movzx ecx,WORD PTR [esp].EMMFRAME.eES
	shl	ecx,4
    movzx edi, di
	add	edi,ecx

    push eax
    lods word ptr [esi]
    movzx eax,ax
    movzx ecx, [bEmsPhysPages]
    cmp eax,ecx
    ja @@failA3
	mov ecx, eax
    stos word ptr [edi]
    jecxz @@done00
@@nextsegm:
	lods word ptr [esi]
    call IsMappedSegm
    jc @@fail8B
    stos byte ptr [edi]
    movzx eax, al
    mov ax, [PHYSPAGETABLE+eax*2]
    stos word ptr [edi]
    loop @@nextsegm
@@done00:
    pop eax
	mov ah,00
	ret
    
@@fail8B:
	pop eax
    mov ah,8Bh
    ret
@@failA3:
	pop eax
@@failA3_1:
    mov ah,0A3h	;segment count exceeds mappable pages
    ret

@@setmap:
    push eax
    lods word ptr [esi]
    movzx eax,ax
    movzx ecx, [bEmsPhysPages]
    cmp eax,ecx
    ja @@failA3
	mov ecx, eax
    jecxz @@done01
@@nextsegm1:
	mov al, [esi+0]
    cmp al, [bEmsPhysPages]
    jae @@fail8B
    pushad
    mov bx, [esi+1]
    call MAP_PAGE
    popad
    add esi, 3
	loop @@nextsegm1
@@done01:
    pop eax
	mov ah,0
	ret

@@getsize:
	cmp bx, 4
    ja @@failA3_1
    mov al, bl
if 0    
    add al, al
    add al, bl		;3 bytes for each entry required
    add al, 2		;+2 bytes for size
else
	inc al			;4 extra bytes for size + checksum?
	shl al,2		;4 bytes (makes Kyrandia 3 work [better]!?)
endif
    mov ah, 00
	ret
else
	jmp EMS_NOT_IMPL
endif

EMS4_GET_SET_PARTIAL_PAGE_MAP ENDP

; Paste a logical page anywhere in address space
; ESI == ptr to PTE
; EAX == linear address
; BX = absolute logical page
; modifies eax, esi, edi, ecx

MAP_PAGE_EX proc    
	AND     BH,BH               ; log. page < 0, then fade out
	JS      @@UNMAP
	MOVZX   EDI,BX              ; Calculate now the (absolute) pagenumber

	push esi
	shl	edi,2					; convert absolute page to dword entry offset
	add	edi, [EMSPageAllocationStart]	; edi -> EMS page descriptor entry
	movzx esi,[edi].EMSPD.wPD		; pool allocation block count relative 1
	movzx ecx,[edi].EMSPD.wNibOfs	; 16K offset from pool block base
if 1
	cmp si,-1	;bad pointer?
    jz @@unmap2	;then unmap this page
endif
	dec	si
	shl	esi,6					; convert to 64-byte block offset
	add	esi, [PoolAllocationTable]	; esi -> pool allocation block for page

	@assumem esi,<LPPOOL_SYSTEM_INFO_STRUC>

	shl	ecx,14				; 16K to bytes
	mov	edi,[esi].psi_addressK
	shl	edi,10				; K address to bytes
	add	edi,ecx				; edi -> page memory
    
	pop	esi					;restore PTE ptr
    
    @assumem esi,nothing

@@SET:
	ADD     EDI,111B	    ; Statusbits: R/W=1,U/S=1,P=1
	MOV     CL,4			; 1 EMS page are 4 (virtual) pages
@@LOOP:
	MOV     [ESI],EDI       ; Register the new physical Address of
	ADD     ESI,4           ; window
	ADD     EDI,4096        ; Process next 4K-page
    DEC     CL
	JNZ     @@LOOP
if ?INVLPG    
    cmp		[_NoInvlPg],0
    jnz		@@noinvlpg
    mov		cl, 4			; EMS page == 4 physical pages
@@nextpage:    
    INVLPG  ds:[eax]
    add		eax, 1000h
    dec		cl
    jnz     @@nextpage
    RET
@@noinvlpg:    
endif
	MOV     EAX,CR3         ; flush TLB
	MOV     CR3,EAX
	RET
@@UNMAP2:
	pop		esi
@@UNMAP:
	mov		edi, eax
    
	@DbgOuts <"Unmap EMS page at ">, ?EMSDBG
    @DbgOutd edi, ?EMSDBG
    @DbgOuts <", page table ptr=">, ?EMSDBG
    @DbgOutd esi, ?EMSDBG
    @DbgOuts <10>, ?EMSDBG
	
	JMP     @@SET
    
MAP_PAGE_EX endp
    
; Paste a logical page into EMS-frame.
; AL = physical page 0 - 3, BX = logical page (absolute) or -1
; modifies ESI, EDI, ECX
; the logical page in BX might belong to no handle!
;
MAP_PAGE PROC NEAR
    PUSH	EAX
    movzx	eax,al
	MOV     ESI,EAX
    movzx	ecx, [_FRAME]
    shl		eax,12+2			;0->0, 1->4000, 2->8000, 3->C000
    shl		ecx, 4
    add		eax,ecx
if 0
;--- this optimisation should be safe, but it turned out it is not.
;--- some versions of PKZIP/PKUNZIP don't like it.
	cmp		bx,[PHYSPAGETABLE+ESI*2]
    jz		@@nochange
endif
	MOV     [PHYSPAGETABLE+ESI*2],BX
	SHL     ESI,4               ; get pointer to PTEs into ESI
	ADD     ESI,[FRAMEANCHOR]
    call	MAP_PAGE_EX
if ?EMSDBG
	jmp     @@change
@@nochange:
	@DbgOuts <"MAP_PAGE: mapping of ">,1
    @DbgOutw bx,1
	@DbgOuts <" to page ">,1
    @DbgOutb al,1
	@DbgOuts <" optimised",10>,1
@@change:    
else    
@@nochange:    
endif
    pop		eax
    ret
MAP_PAGE ENDP

	@assumem esi,<nothing>

; 6750:
; AH = 50h: EMS 4.0 map multiple pages
; DX = handle
;  DS:SI -> mapping array
;  CX = items in array (ecx is destroyed, reload it from stack)
;  structure of mapping array:
;  WORD logical page (or -1 to unmap page)
;  WORD physical page (AL=0) or segment address (AL=1)

ems4_map_multi PROC NEAR

	cmp	al,1
	ja	bad_subfunc
    
	movzx ecx, word ptr [esp].EMMFRAME.eEcx	; load EMM entry CX value
	movzx edi, word ptr [esp].EMMFRAME.eDs	; DS from the Level-0-Stack
	shl	edi,4
	movzx esi,si
	add	edi,esi		; edi -> map address buffer

if ?MASM
ems4_map_multi_edi::
else
ems4_map_multi_edi:
endif

; perform handle check here so that stack return address isn't blown

	call	TEST_HANDLE
    mov esi, edi	

	push ebx
    push eax
    jecxz @@success
@@multi_loop:
	mov	bx,[esi+0]
	mov	ax,[esi+2]
	add	esi,4
	cmp byte ptr [esp+0],1	;subfunction 1?
	jne	@@mappage
    call IsMappedSegm	;convert segment in ax to a page no
    mov ah,8Bh
    jc  @@multi_out
@@mappage:
	push ecx
	push esi
	call EMS_MAP_HANDLE_PAGE
	pop	esi
	pop	ecx
	test ah,ah
	jne	@@multi_out	; error occurred
	loop @@multi_loop
@@success:    
	MOV	ah,00	; no error return
@@multi_out:
	mov [esp+1],ah
	pop eax
	pop	ebx
	ret
    
ems4_map_multi ENDP

; 6751:
; AH = 51h: EMS 4.0 reallocate pages for handle
; DX = handle
; BX = new page count for handle
; out: BX=pages assigned to handle
;
ems4_realloc PROC NEAR
	call	TEST_HANDLE
	call	DynamicEMSPageCount	;update PAGESAVAIL

	push ebx		; save new pages

	call	EMS_GET_NR_OF_ALLOCATED_PAGES
    
	mov	edi,esi
    mov esi,ebx		; current pages for this handle
	mov	ecx,ebx
	pop	ebx			; restore new page request amount
	add	ecx,[_EMSPAGESAVAIL]	; get current pages for handle + available
	cmp	bx,cx
	ja	@@toomany
    
	push ebx
	push eax
	cmp	bx, si		; check new page count against original
	jb	shrinkalloc
	je	realloc_success	; no change needed

	MOV ECX,[_MAXEMSPAGES]
    add ecx,[EMSPAGETABLE]
	sub ecx,edi

; growing the allocation of pages

	sub	bx,si		; get count of new pages needed
growpages:
	mov	al,FREEPAGE_ID
	repnz	scas BYTE PTR [edi]

	jnz	@@nofind			; couldn't find new page
	call GetEMSPage			; allocate the page (preserves all registers)
	jc	@@nofind			; allocation successful?
	mov	[edi-1],dl
	dec	[_EMSPAGESAVAIL]
	dec	bx
	jnz	growpages
	jmp	realloc_success
@@nofind:
    pop eax
	pop	ebx			; todo: release the pages already allocated
@@toomany:
	mov	bx,si		; return original pages owned by handle
	mov	ah,88h
	ret

; trim off the top pages, this avoids modifying the page order!

shrinkalloc:
	xchg esi,ebx
	sub	bx,si		; get count of pages to reduce
    mov ecx, edi
    sub ecx, [EMSPAGETABLE]
	dec	edi			
	mov	al,dl
shrinkpages:
	std				; scan backwards
	repnz	scas BYTE PTR [edi]
	cld

	add	edi,2		; ReleaseEMSPage expects edi-1 ptr, not edi+1
	call ReleaseEMSPage	;preserves registers (but expects DF to be clear!)
	sub	edi,2

	mov	BYTE PTR [edi+1],FREEPAGE_ID
	inc	[_EMSPAGESAVAIL]
	dec	bx
	jnz	shrinkpages

realloc_success:
    pop eax
	pop	ebx
	mov	ah,00		; no error occurred
	ret

ems4_realloc ENDP

; 6752
; AH = 52h: EMS 4.0 attribute related
; AL = 0/1/2, DX=handle, BL=0/1
;
ems4_attribute PROC NEAR
	cmp	al,2
	jb	@@get_set_attribute
	ja	bad_subfunc	; this is an invalid subfunction
@@is_volatile:
	xor	ax,ax	; al == 0, volatile attribute only, ah == successful call
	ret
@@get_set_attribute:
	call TEST_HANDLE	; only valid handles please
	or al,al			; 0 is get, 1 is set
	jz @@is_volatile	; only volatile here (none survive warm reboot)
	or bl,bl			; 0 is "make volatile" (true anyway)
	jnz @@cannot_make_nonvolatile
	mov ah,00   		; be happy
	ret
@@cannot_make_nonvolatile:
	mov ah,91h		; feature not supported
	ret
ems4_attribute ENDP

; 6753:
; AH = 53h: EMS 4.0 get/set handle name
; AL = 0: get handle name in ES:DI
; AL = 1: set handle name in DS:SI
; DX = handle
;
ems4_handle_names PROC NEAR
	cmp	al,1
	ja	bad_subfunc	; this is an invalid subfunction
    jz  @@ems4_setname

	call	TEST_HANDLE
	movzx	esi,WORD PTR [esp].EMMFRAME.eES	; ES from the Level-0-Stack
	shl	esi,4
	movzx	edi,di
	add	edi,esi		; edi -> handle name buffer address (dest)
	mov	esi, [EMSNAMETABLE]
	movzx	ecx,dx	; handle (index)
    lea esi, [esi+ecx*8]
    jmp @@ems4_getsetname

@@ems4_setname:
	mov edi, esi	;save SI
	call	TEST_HANDLE
	movzx	esi,WORD PTR [esp].EMMFRAME.eDS	; DS from the Level-0-Stack
	shl	esi,4
	movzx edi, di	;this is original SI
	add	esi,edi		; esi -> handle name (source)
	mov	edi, [EMSNAMETABLE]
	movzx	ecx,dx	; handle (index)
    lea edi, [edi+ecx*8]
@@ems4_getsetname:    
	mov	ecx,[esi+0]	; transfer handle name to es:di
	mov	[edi+0],ecx
	mov	ecx,[esi+4]
	mov	[edi+4],ecx
	mov	ah,00	; no error return
	ret

ems4_handle_names ENDP

; 6754:
; AH = 54h: EMS 4.0 get various handle info
;
; AL = 0: get handle directory into ES:DI
; AL = 1: search handle by name in DS:SI, return handle in DX
; AL = 2: get total handles in BX

ems4_get_handle_info PROC NEAR
	cmp	al,1
	jb	getallhand
	je	@@find_handle_by_name
	cmp	al,2
	ja	bad_subfunc	; this is an invalid subfunction

	mov	bx,MAX_HANDLES
	mov	ah,00   ; zero ah, no error return
	ret

; write handle directory to caller buffer
; return in AL number of open handles
getallhand:
	movzx	esi,WORD PTR [esp].EMMFRAME.eES	; ES from the Level-0-Stack
	shl	esi,4
	movzx edi,di
	add	esi,edi
	mov	edi, [EMSSTATUSTABLE]
	mov	al,0		; AL will be count of open handles
    xor ecx, ecx
@@scan_handles:
	cmp	WORD PTR [edi], EMSH_FREE
	jz	@@free_handle
	inc	al		; count that open handle

	push eax
	mov	[esi+0],cx
	mov	eax,edi
	sub	eax, [EMSSTATUSTABLE]	; convert to table offset
	add	eax, [EMSNAMETABLE]		; offsets are identical
	push DWORD PTR [eax+0]		; copy handle name
	pop	DWORD PTR [esi+2]
	push DWORD PTR [eax+4]
	pop	DWORD PTR [esi+6]
	add	esi,10
	pop	eax
    
@@free_handle:
	add	edi,8
    inc ecx
    cmp cl, MAX_HANDLES
	jb	@@scan_handles
	mov	ah,00
	ret

@@find_handle_by_name:
	movzx	edi,WORD PTR [esp].EMMFRAME.eDS	; DS from the Level-0-Stack
	shl	edi,4
	movzx	esi,si
	add	edi,esi
    
    push ebx
    push eax
	mov	eax,[edi+0]		; fetch to-be-searched name
	mov	ebx,[edi+4]		; (8 byte binary string)
if 0 ;it is valid to search a handle with no name!
	or	eax,eax
	jnz	@@valid_search_term
	or	ebx,ebx
	jz	@@invalid_search_term
@@valid_search_term:
endif
	xor	ecx,ecx
	mov	edi, [EMSNAMETABLE]
	mov	esi, [EMSSTATUSTABLE]
@@scan_for_name:
	cmp	WORD PTR [esi], EMSH_FREE	; closed, auto-fail
	jz	@@another_handle
	cmp	[edi+0],eax
	jnz	@@another_handle
	cmp	[edi+4],ebx			; Note that open handles do not have
	jz	@@found_handle		; to have a name.
@@another_handle:
	add	esi,8
    add edi,8
	inc	ecx
	cmp	cl,MAX_HANDLES
	jb	@@scan_for_name
    pop eax
    pop ebx
	mov	ah,0a0h			; "no handle could be found"
	ret
@@found_handle:
    pop eax
	pop ebx
    mov dx,cx
	mov	ah,00
	ret
if 0
@@invalid_search_term:	; The error code descr. is misleading:
    pop eax
	pop ebx
	mov	ah,0a1h			; "handle found had no name"
 	ret
endif

ems4_get_handle_info ENDP

log_phys_map struc
wLogPage       DW ?
wPhysPage      DW ?	;physical pages if AL=0, segment if AL=1
log_phys_map ends

map_and_jump struc
dwJmpAddr    DD ? ; far16 jump address
bSize        DB ? ; items in log_phys_map member
pLogPhysMap  DD ? ; pointer to log_phys_map structure
map_and_jump ends

if ?EMM5556
mapmulti proc
	shl	edi,4
	movzx esi,si
	add	edi,esi		; edi -> map address buffer
    push edi
    movzx ecx, word ptr [edi].map_and_jump.pLogPhysMap+0
    movzx esi, word ptr [edi].map_and_jump.pLogPhysMap+2
    shl esi, 4
    add esi, ecx
    movzx ecx, [edi].map_and_jump.bSize
    mov edi, esi
    call ems4_map_multi_edi
    pop edi
    ret
mapmulti endp
endif

; 6755:
; AH = 55h: EMS 4.0 alter page map and jump
; AL = 0/1
; DX = handle
; DS:SI -> map_and_jump structure

ems4_alter_map_jump proc
if ?EMM5556
	movzx	edi, word ptr [esp].EMMFRAME.eDs	; DS from the Level-0-Stack
	call mapmulti
    and ah,ah
    jnz @@bye
    mov ecx,[edi].map_and_jump.dwJmpAddr
    mov word ptr [esp].EMMFRAME.eEip, cx
    shr ecx, 16
    mov word ptr [esp].EMMFRAME.eCs, cx
@@bye:
	ret
else
	jmp EMS_NOT_IMPL
endif
ems4_alter_map_jump endp

; 6756:
; AH = 56h: EMS 4.0 alter page map and call
; AL = 0/1/2
; DX = handle
; if AL=0/1: [in] DS:SI -> map_and_jump structure ()
; if AL=2: [out] BX = additional stack space required

ems4_alter_map_call proc
if ?EMM5556
	cmp al,2
    jz  @@getstackspace
	movzx	edi, word ptr [esp].EMMFRAME.eDs	; DS from the Level-0-Stack

    push dword ptr [PHYSPAGETABLE+4]
    push dword ptr [PHYSPAGETABLE+0]
	
    call mapmulti
    and ah,ah
    jnz @@bye

;--- now store a v86_call structure onto the v86 stack
;--- + 8 extra bytes for the old frame state

	movzx	esi, word ptr [esp+8].EMMFRAME.eEsp
	movzx	ecx, word ptr [esp+8].EMMFRAME.eSs
    sub		si, size v86_call + 2*4
    shl		ecx, 4
    mov		word ptr [esp+8].EMMFRAME.eEsp, si
    add		esi, ecx

    pop		dword ptr [esi+size v86_call+0]	;save the old frame state
    pop		dword ptr [esi+size v86_call+4]
    
    mov ecx,[edi].map_and_jump.dwJmpAddr	;set new v86 CS:IP and
    xchg cx, word ptr [esp].EMMFRAME.eEip
    mov word ptr [esi].v86_call.dwOldCSIP+0, cx	;save old CS:IP in v86_call
    shr ecx, 16
    xchg cx, word ptr [esp].EMMFRAME.eCs
    mov word ptr [esi].v86_call.dwOldCSIP+2, cx
    
    mov		word ptr [esi].v86_call.dwBP+0, offset BPBACK	;store breakpoint
    mov		word ptr [esi].v86_call.dwBP+2, SEG BPBACK		;on top of stack
    mov		[esi].v86_call.dwRetAddr, offset restore_page_frame
    
    ret
@@bye:
	add		esp,2*4
	ret
@@getstackspace:
	mov bx,size v86_call + 2*4
	ret

;--- this is called when the v86 code executes a RETF
;--- ebx -> v86 stack (the first DWORD of v86_call is skipped already!)
    
restore_page_frame:	
    add  	[ESP].V86FRGP.gSP,8		;adjust v86 stack for the 8 extra bytes
    add		ebx, size v86_call		;skip v86_call structure
    pushad
    push	V86_DATA_SEL
    pop		fs
    mov		esi,ebx
    mov		cl, [bEmsPhysPages]
    call	restore_from_esi
    mov		[ESP+32+8+1],ah
    popad
	POP     ECX             ; Clean everything again
	POP     EBX
	POP     EAX
	add     ESP,4+4         ; remove error word + old calling address
	IRETD
else
	jmp EMS_NOT_IMPL
endif
ems4_alter_map_call endp

; 6757:
; AH = 57h: EMS 4.0 move/exchange memory region
; AL = 0: move memory region
; AL = 1: exchange memory region
; DS:SI -> EMM57

;-- this function should work even if no EMS page frame is defined!
;-- EMS regions may overlapp!

EMM57 struc
e57_dwSize	DD ?	; +0  size of region
e57_bSrcTyp DB ?	; +4  src memory type
e57_wSrcHdl DW ?	; +5  src handle
e57_wSrcOfs DW ?	; +7  src ofs
e57_wSrcSeg DW ?	; +9  src segm./log. page
e57_bDstTyp DB ?	; +11 dst memory type
e57_wDstHdl	DW ?	; +12 dst handle
e57_wDstOfs	DW ?	; +14 dst ofs
e57_wDstSeg	DW ?	; +16 dst segm./log. page
EMM57 ends

;--- memory type:
;--- 00: conv. memory
;--- 01: expanded memory
;--- handle:
;--- == NULL: conv. memory
;--- <> NULL: EMS handle

Saved_Eax	equ <ebp+4>
RegionDest 	equ <ebp-4>

ems4_memory_region PROC NEAR

	cmp	al,1
	ja	bad_subfunc	; this is an invalid subfunction
	movzx edi,WORD PTR [esp].EMMFRAME.eDS	; DS from the Level-0-Stack
	movzx esi,si
	shl	edi,4
	add	edi,esi		; edi -> EMS region buffer address

	push	ebx
	push	edx
    push	eax
    push	ebp
    mov		ebp, esp
    sub		esp, 1*4
	push	offset @@exit	;push error return address for test_handle

if ?EMSDBG
	@DbgOuts <"EMM 57h: siz=">,1
    @DbgOutd [edi].EMM57.e57_dwSize,1
	@DbgOuts <", src=">,1
    @DbgOutb [edi].EMM57.e57_bSrcTyp,1
    @DbgOuts <"/">,1
    @DbgOutw [edi].EMM57.e57_wSrcHdl,1
    @DbgOuts <"/">,1
    @DbgOutw [edi].EMM57.e57_wSrcSeg,1
    @DbgOuts <"/">,1
    @DbgOutw [edi].EMM57.e57_wSrcOfs,1
	@DbgOuts <", dst=">,1
    @DbgOutb [edi].EMM57.e57_bDstTyp,1
    @DbgOuts <"/">,1
    @DbgOutw [edi].EMM57.e57_wDstHdl,1
    @DbgOuts <"/">,1
    @DbgOutw [edi].EMM57.e57_wDstSeg,1
    @DbgOuts <"/">,1
    @DbgOutw [edi].EMM57.e57_wDstOfs,1
endif

	mov	ecx,[edi].EMM57.e57_dwSize
	test ecx,ecx
	je	@@ok  		; always succeeds if no bytes are moved
	mov	ah,96h		; preload error code, region length exceeds 1M
	cmp	ecx,65536*16
	ja	@@exit

; process region destination information

	movzx	ecx,[edi].EMM57.e57_wDstOfs
	movzx	ebx,[edi].EMM57.e57_wDstSeg
	cmp		[edi].EMM57.e57_bDstTyp,0	;0 = conv, 1 = expanded
	je	@@calc_dest_conv

; destination is EMS

	mov dx, [edi].EMM57.e57_wDstHdl
    call test_handle
    
	mov	ah,95h		; preload error code, specified offset is outside logical page
	test ch,0C0h   	; ecx must be < 4000h
	jnz	@@exit

    mov		eax, ?SCRATCHLINEAR+104000h
    lea		esi, [ecx+eax]
    mov		[RegionDest],esi
	add		ecx,[edi].EMM57.e57_dwSize
    
; try to map in all needed pages (max space is 1 MB + 16 kB)

    @GETPTEPTR esi, 2000h+?SCRATCHPTE+104h*4, 1
    call	mappages
	jmp 	@@calc_dest_done

@@calc_dest_conv:
	shl	ebx,4		; convert seg to memory offset
	add	ebx,ecx
	mov	[RegionDest],ebx
    mov ah,0A2h		; conv memory region must not exceed 1 MB boundary
	add	ebx, [edi].EMM57.e57_dwSize
    cmp ebx, 100000h
    ja  @@exit
@@calc_dest_done:    

; process region source information

	movzx	ecx,[edi].EMM57.e57_wSrcOfs
	movzx	ebx,[edi].EMM57.e57_wSrcSeg
	cmp		[edi].EMM57.e57_bSrcTyp,0	; 0 = conv, 1 = expanded
	je	@@calc_src_conv

; source is EMS

    mov dx, [edi].EMM57.e57_wSrcHdl
    call test_handle
    
	mov	ah,95h		; preload error code to specified offset is outside logical page
	test ch,0C0h   	; ecx must be < 4000h
	jnz	@@exit
    
    mov		eax,?SCRATCHLINEAR
	add		ecx,[edi].EMM57.e57_dwSize
    
; try to map in all needed pages (max size is 1 MB)

    @GETPTEPTR esi, 2000h+?SCRATCHPTE, 1
    call	mappages
	mov		ecx, [edi].EMM57.e57_dwSize
    movzx   esi, [edi].EMM57.e57_wSrcOfs
    add		esi, ?SCRATCHLINEAR

; if src and dest are EMS test if they overlapp
    
	cmp		[edi].EMM57.e57_bDstTyp,0	; is dst also expanded memory?
    jz  	@@calc_src_done
	cmp		dx, [edi].EMM57.e57_wDstHdl	; are handles equal?
    jnz  	@@calc_src_done

	movzx eax, [edi].EMM57.e57_wDstSeg
	movzx edx, [edi].EMM57.e57_wSrcSeg
    shl eax, 14
    shl edx, 14
    or  ax, [edi].EMM57.e57_wDstOfs
    or  dx, [edi].EMM57.e57_wSrcOfs

;--- the problem case is:
;--- source < destination AND source + size > destination

	cmp edx, eax		; source < destination?
    jnc @@calc_src_done	; no, use std copy
	add edx, ecx		; edx == source + size
    cmp eax, edx		; destination >= source + size?
    jnc @@calc_src_done	; yes, use std copy
    jmp @@overlapped

@@calc_src_conv:
	shl	ebx,4			; convert seg to memory offset
	add	ebx,ecx
	mov	ecx, [edi].EMM57.e57_dwSize
	mov	esi,ebx
    mov ah,0A2h			; conv memory region must not exceed 1 MB boundary
    add ebx, ecx
    cmp ebx, 100000h
    ja  @@exit
    
@@calc_src_done:
	mov	edi, [RegionDest]
	cmp	byte ptr [Saved_Eax],0
	jne	@@xchg
	MOV     EAX,ECX
	SHR     ECX,2
	REP     MOVS DWORD PTR [ESI], DWORD PTR [EDI]
	BIG_NOP
	MOV     ECX,EAX
	AND     ECX,3
	REP     MOVS BYTE PTR [ESI],BYTE PTR [EDI]
	BIG_NOP
@@ok:
	mov	ah,00   ; zero ah, no error return

; exit with code in AH

@@exit:
if ?EMSDBG
    @DbgOuts <"=">,1
    @DbgOutb ah,1
    @DbgOuts <10>,1
endif
	mov [Saved_Eax+1], ah	;set AH in saved EAX on stack
	mov esp, ebp
    pop ebp
    pop eax
	pop	edx
	pop	ebx
	ret

@@overlapped:
	mov	edi, [RegionDest]
	cmp	byte ptr [Saved_Eax],0
	jne	@@xchg
	std
    lea		esi,[esi+ecx-1]
    lea		edi,[edi+ecx-1]
    mov		eax,ecx
    and		ecx,3
	REP     MOVS BYTE PTR [ESI],BYTE PTR [EDI]
	MOV     ECX,EAX
    sub		esi,3
    sub		edi,3
	SHR     ECX,2
	REP     MOVS DWORD PTR [ESI], DWORD PTR [EDI]
    cld
    mov		ah,92h	;status "overlapp occured"
    jmp		@@exit
@@xchg:
	mov	ah,cl
	and	cl,3
	je	@@fullx
@@xb:
	mov	al,[edi]
    movsb
	mov	[esi-1],al
    dec cl
	jnz	@@xb
@@fullx:
	mov	cl,ah
	shr	ecx,2
    jz @@ok
@@xdw:
	mov	eax,[edi]
    movsd
	mov	[esi-4],eax
    dec ecx
	jnz @@xdw
    jmp	@@ok
    
mappages:
    add		ecx, eax		;let ecx point to end of region
@@nextpagetomap:
	pushad
    call	get_abs_page	;get abs page of DX:BX in BX
    jc      @@exit
	call	map_page_ex		;requires EAX,BX,ESI to be set
    popad
    inc		ebx				;next EMS page
    add		eax, 4000h		;proceed with linear address
    add		esi, 4*4		;proceed with PTE pointer
    cmp		eax,ecx
    jb  	@@nextpagetomap
    retn
    
Saved_Eax	equ <>
RegionDest 	equ <>

ems4_memory_region ENDP

; 6758:
; AH = 58h: EMS 4.0 get addresses of mappable pages, number of mappable pages
; AL = 0: ES:DI -> buffer, returns segments in CX
; structure of buffer item: WORD segment, WORD physical page
; AL = 1: returns segments in CX

ems4_get_mappable_info PROC NEAR
	cmp	al,1
	ja	bad_subfunc	; this is an invalid subfunction

	movzx ecx, [bEmsPhysPages]
	mov	WORD PTR [esp].EMMFRAME.eEcx,cx	; return mappable pages in CX

	cmp al,1
	je	nummap
	jecxz nummap
    
	movzx	esi,WORD PTR [esp].EMMFRAME.eES	; ES from the Level-0-Stack
	shl	esi,4
	movzx	edi,di
	add	edi,esi	; edi -> handle name buffer address
	mov	si,[_FRAME]
	xor	ecx,ecx
@@mapinfo_loop:
	mov	[edi+0],si	; base address
	mov	[edi+2],cx	; physical page
	add	si,400h		; next base address (16384/16 paragraphs) 
	add	edi,4		; next dword entry
    inc ecx			; next physical page
    cmp cl,[bEmsPhysPages]
	jnz	@@mapinfo_loop
nummap:
	mov	ah,00	; no error return
	ret

ems4_get_mappable_info ENDP

; 6759:
; AH = 59h: EMS 4.0 get hardware config/get number of raw pages
; AL = 1: return raw pages in DX and BX
; AL = 0: get hardware config in ES:DI

EMM59 struc
e59_pgsize	dw ?	;raw page size in paragraphs
e59_altsets dw ?	;number of alternate register sets
e59_sizcont dw ?	;size of mapping context save area in bytes
e59_dmasets dw ?	;dma register sets
e59_dmaflgs dw ?	;dma flags
EMM59 ends

ems4_get_config PROC NEAR
	cmp	al,1	; only subfunctions 0+1 supported
	ja	bad_subfunc
	je 	EMS_GET_UNALLOCATED_PAGE_COUNT
    
if 0
	mov	al,ds:[7FFF0000h]	;generate an exception 0Eh
endif

	movzx	esi, WORD PTR [esp].EMMFRAME.eES
	shl		esi, 4
	movzx	edi, di
    add		edi, esi
    mov		[edi].EMM59.e59_pgsize, 1024
    mov		[edi].EMM59.e59_altsets, 0
    mov		[edi].EMM59.e59_sizcont, 4*2
    mov		[edi].EMM59.e59_dmasets, 0
    mov		[edi].EMM59.e59_dmaflgs, 0
    mov		ah,0
	ret

ems4_get_config ENDP

; 675A:
; AH = 5ah: EMS 4.0 allocate handle and standard/raw pages
; in  AL = 0/1
; in  BX = pages to allocate (may be 0)
; out DX = handle
;
ems4_allocate_pages PROC NEAR

	cmp	al,1	; subfunction must be 0 or 1, we don't care if either
	ja	bad_subfunc
	jmp	allocate_pages_plus_zero

ems4_allocate_pages ENDP

if ?VCPI
;
; AX=DE00: VCPI presence detection
;  return BH = 1 (major version), BL = 0 (minor version)
;
VCPI_Presence	PROC	NEAR
	mov	bx,100h
	mov	ah,00
	ret
VCPI_Presence	ENDP

;
; AX=DE01: VCPI get protected mode interface
;  inp: es:di -> client zero page table (to fill)
;       ds:si -> three descriptor table entries in client's GDT (to fill)
;  out: [es:di] page table filled
;       di: first uninitialized page table entry (advanced by 4K)
;      ebx: offset to server's protect mode code segment
;
;--- dont forget: ECX, ESI, EDI are saved on the stack

VCPI_GetInterface	PROC	NEAR
	movzx	ebx,WORD PTR [esp].EMMFRAME.eDS	; DS from the Level-0-Stack
	shl	ebx,4
	movzx	esi,si
	add	esi,ebx			; esi -> client GDT entries
	mov	ebx, dword ptr [GDT_PTR+2]

	push	eax
	mov	eax, ds:[ebx + V86_CODE_SEL + 0]
	mov	ecx, ds:[ebx + V86_CODE_SEL + 4]
	mov	[esi+0],eax
	mov	[esi+4],ecx
	mov	eax, ds:[ebx + V86_DATA_SEL + 0]
	mov	ecx, ds:[ebx + V86_DATA_SEL + 4]
	mov	[esi+08],eax
	mov	[esi+12],ecx
	mov	eax, ds:[ebx + FLAT_DATA_SEL + 0]
	mov	ecx, ds:[ebx + FLAT_DATA_SEL + 4]
	mov	[esi+16],eax
	mov	[esi+20],ecx
	pop	eax

	movzx	esi,WORD PTR [esp].EMMFRAME.eES	; ES from the Level-0-Stack
	shl	esi,4
	movzx	edi,di
	add	edi,esi				; edi -> client zero page table
	@GETPTEPTR	esi,1000h,1	; esi -> page table for first 1M

;--- Jemm386 must ensure that 
;--- the VCPI_PM_ENTRY label will be in shared memory. Since this label is
;--- now at the very beginning of V86 segment, 1 page should suffice

if ?CODEIN2PAGE
	mov     ecx, 440h+8 ;this is offset in page table for 112000h
else
	mov     ecx, 440h+4 ;this is offset in page table for 111000h
endif    
	add		word ptr [esp].EMMFRAME.eEdi,cx	; modify edi
    shr		ecx, 2

	push	eax
vgiloop:
	lods dword ptr [esi]
	and	ah,0F1h		; clear bits 9-11
    stos dword ptr [edi]
	loop vgiloop
;;  and byte ptr es:[edi-4], not 2	;reset r/w for 110xxx page.
	pop	eax

	mov	ebx,OFFSET VCPI_PM_ENTRY
	mov	ah,00
	ret

VCPI_GetInterface	ENDP

; AX=DE02: VCPI get maximum physical memory address
;  return edx == physical address of highest 4K page available
;
VCPI_GetMax	PROC	NEAR
	mov	edx,[_TOTAL_MEMORY]
if ?EMX
	test _bV86Flags, V86F_EMX
    jz @@noemx
	mov	edx,[_MAXMEM16K]	;mem in 16K
    shl edx, 12+2			;convert to bytes
    add edx, [FIRSTPAGE]
@@noemx:    
endif    
	dec	edx
	and	dx,NOT 0fffh

	mov	ah,00			; flag success
	ret
VCPI_GetMax	ENDP

;
; AX=DE03: VCPI get number of free pages
;  return edx == number of free pages
;
VCPI_GetFree	PROC	NEAR
	push	eax
	call	GetFreeXMS4KPageCount
	mov	ecx,edx		; ecx == XMS free 4K pages

	call	GetCurrent4KPageCount	; current free in eax, current total in edx
	add	ecx,eax		; ecx == all free pages
	sub	edx,eax		; edx == current allocated (total - free)
	jc	@@nofree			; this shouldn't happen, but return no free it if does
	xchg	edx,ecx		; free pages count in edx, allocated count in ecx

; total free must be <= MAXMEM16K * 4 - current allocated
;  otherwise set total free = MAXMEM16K * 4 - current allocated
	mov	eax,[_MAXMEM16K]
	shl	eax,2			; convert to maximum 4K blocks
	sub	eax,ecx
	jc	@@nofree
	cmp	eax,edx
	jae	@@done
	mov	edx,eax

@@done:
	pop	eax
	mov	ah,00			; flag success
	ret

@@nofree:
	xor	edx,edx
	jmp	@@done

VCPI_GetFree	ENDP

;
; AX=DE04: VCPI allocate a 4K page
;  return edx == physical address of 4K page allocated
;
VCPI_Allocate4K	PROC	NEAR
	push	eax		; save high word of eax
	push	edx
	call	GetCurrent4KPageCount	; current free in eax, current total in edx

; fail if current total - current free (allocated) >= MAXMEM16K * 4
	sub	edx,eax		; edx == current allocated (total - free)
	jc	@@fail			; shouldn't happen, but fail if it does
	mov	eax,[_MAXMEM16K]
	shl	eax,2			; convert to maximum 4K blocks
	cmp	eax,edx
	jbe	@@fail

	call	Allocate4KPageFromPoolBlock
	or	edx,edx
	jne	@@success

	call	ExpandAnyPoolBlock
	or	edx,edx			; see if any pool block has 4K free
	je	@@tryxms

; pool block was expanded, so an allocation should succeed
	call	Allocate4KPageFromPoolBlock
	or	edx,edx
	jne	@@success

; shouldn't reach this point since pool expansion complete,
;  fall through to trying XMS

@@tryxms:
	call	AllocateXMSForPool
	jc	@@fail
	call	Allocate4KPageFromPoolBlock	; this should always succeed
	or	edx,edx
	je	@@bad2			; failed due to internal fault, so fail allocation

@@success:
	pop eax				; throw away saved EDX on stack
	pop	eax
	mov	ah,00
	ret

@@bad2:
@@fail:
;;	xor	edx,edx			; dont modify EDX on failure!
	pop edx
    pop eax
	mov	ah,88h
    ret

VCPI_Allocate4K	ENDP

;
; AX=DE05: VCPI free a 4K page
;  entry edx == physical address of 4K page to free
;
VCPI_Free4K	PROC	NEAR
	call Free4KPage
	jc	@@bad
	mov	ah,00			; flag success
	ret

@@bad:
	mov	ah,8ah
	ret

VCPI_Free4K	ENDP

;
; AX=DE06: VCPI get physical address of 4K page in first megabyte
;  entry cx = page number (cx destroyed, use stack copy)
;  return edx == physical address of 4K page
;
VCPI_GetAddress	PROC	NEAR
	movzx	ecx, word ptr [esp].EMMFRAME.eEcx
	cmp	cx,256
	jae	vga_bad			; page outside of first megabyte

	@GETPTE	edx, ecx*4+1000h
	and	dx,0f000h		; mask to page frame address
	mov	ah,00			; flag success
	ret

vga_bad:
;;	xor	edx,edx			; do not modify EDX in case of failure!
	mov	ah,8bh
	ret

VCPI_GetAddress	ENDP

;
; AX=DE07: VCPI read CR0
;  return EBX == CR0
;
VCPI_GetCR0	PROC	NEAR
	mov	ebx,cr0
	mov	ah,00			; flag success
	ret
VCPI_GetCR0	ENDP
 
;
; AX=DE08: VCPI read debug registers
;  call with ES:DI buffer pointer. Returns with buffer filled.
;  (8 dwords, dr0 first, dr4/5 undefined)
;
VCPI_ReadDR	PROC	NEAR
	movzx	esi,WORD PTR [esp].EMMFRAME.eES	; ES from the Level-0-Stack
	shl	esi,4
	movzx	edi,di
	add	esi,edi
	mov	edi,dr0
	mov	[esi],edi
	mov	edi,dr1
	mov	[esi+4],edi
	mov	edi,dr2
	mov	[esi+8],edi
	mov	edi,dr3
	mov	[esi+12],edi
;	mov	edi,dr4
;	mov	[esi+16],edi
;	mov	edi,dr5
;	mov	[esi+20],edi
	mov	edi,dr6
	mov	[esi+24],edi
	mov	[esi+16],edi
	mov	edi,dr7
	mov	[esi+28],edi
	mov	[esi+20],edi
	mov	ah,00			; flag success
	ret
VCPI_ReadDR	ENDP

;
; AX=DE09: VCPI write debug registers
;  call with ES:DI buffer pointer. Updates debug registers.
;  (8 dwords, dr0 first, dr4/5 ignored)
;
VCPI_WriteDR	PROC	NEAR
	movzx	esi,WORD PTR [esp].EMMFRAME.eES	; ES from the Level-0-Stack
	shl	esi,4
	movzx	edi,di
	add	esi,edi
	mov	edi,[esi]
	mov	dr0,edi
	mov	edi,[esi+4]
	mov	dr1,edi
	mov	edi,[esi+8]
	mov	dr2,edi
	mov	edi,[esi+12]
	mov	dr3,edi
;	mov	edi,[esi+16]
;	mov	dr4,edi
;	mov	edi,[esi+20]
;	mov	dr5,edi
	mov	edi,[esi+24]
	mov	dr6,edi
	mov	edi,[esi+28]
	mov	dr7,edi
	mov	ah,00			; flag success
	ret
VCPI_WriteDR	ENDP

;
; AX=DE0A: VCPI get 8259A interrupt vector mappings
;  return bx == 1st vector mapping for master 8259A (IRQ0-IRQ7)
;    cx == 1st vector mapping for slave 8259A (IRQ8-IRQ15)
;
VCPI_GetMappings	PROC	NEAR
	mov	bx, [MPIC_BASE]
	mov	cx, [SPIC_BASE]
	mov	word ptr [esp].EMMFRAME.eEcx,cx	; have to save cx value to stack storage
	mov	ah,00			; flag success
	ret
VCPI_GetMappings	ENDP

; AX=DE0B: VCPI set 8259A interrupt vector mappings
;  entry bx == 1st vector mapping for master 8259A (IRQ0-IRQ7)
;    cx == 1st vector mapping for slave 8259A (IRQ8-IRQ15)
;-- this is meant just as info, the client has to program the PIC itself

VCPI_SetMappings	PROC	NEAR

    mov ecx, [esp].EMMFRAME.eEcx
    mov [MPIC_BASE],bx
    mov [SPIC_BASE],cx
	mov	ah,00			; flag success
	ret
    assume DS:NOTHING

VCPI_SetMappings	ENDP


endif	;?VCPI

;
; Check a given Handle for validness.
; In case the Handle is invalid the returnaddress is thrown away
; and afterwards through RET returned to dispatcher.
; Else ESI will point to the handle in EMSSTATUSTABLE array

TEST_HANDLE PROC NEAR
	CMP     DX,MAX_HANDLES            ; Out of area ?
	JAE     SHORT @@BYE
	MOVZX   ESI,DL                    ; Form the pointer from the Handle-Status-
	SHL     ESI,3                     ; Table and ...
	ADD     ESI, [EMSSTATUSTABLE]
	CMP     WORD PTR [ESI], EMSH_FREE ; ... examine, if the Handle has
	JZ      SHORT @@BYE               ; been marked as free
	RET
@@BYE:
	ADD     ESP,4                     ; Throw away call(ing)-address
	MOV     AH,83H                    ; "Handed over Handle is unknown"
	RET
TEST_HANDLE ENDP


; EMS/VCPI memory management routines

; dynamically compute EMS pages, store to _EMSPAGESAVAIL
;  if fixed allocation, check only current EMS store
; destroy no registers

DynamicEMSPageCount	PROC

	pushad

	xor	ecx,ecx			; init count of free 16K XMS pool blocks
	cmp	[_NoPool],0		; check if pool sharing
	jne	@@pastxms

	call	GetFreeXMS4KPageCount
	shr	edx,2			; convert 4K count to 16K pages
	mov	ecx,edx

@@pastxms:
; get used EMS pages in edx, available in eax, VCPI/EMS used in edi
	call GetCurrentEMSPageCount
	cmp	[_NoPool],0	; check if pool sharing
	jne	@@ret
	add	eax,ecx			; eax == potential pages available

; total free must be <= MAXMEM16K - current VCPI/EMS allocated/used
;  otherwise set total free = MAXMEM16K - allocated

	mov	ecx, [_MAXMEM16K]
	sub	ecx,edi			; amount that can still be allocated (total - used)
	jc	@@nofree
	cmp	eax,ecx
	jbe	@@pagemax		; available less than amount allowed for allocation
	mov	eax,ecx

; limit possible EMS pages to maximum allowed
@@pagemax:
	cmp	eax, [_MAXEMSPAGES]
	jb	@@noadj1
	mov	eax, [_MAXEMSPAGES]
@@noadj1:
	cmp	eax,edx			; ensure no overflow from improper current value
	ja	@@noadj2
	mov	eax,edx
@@noadj2:
	sub	eax,edx			; max - used == max available

@@ret:
if ?POOLDBG
	@DbgOutS <"DynamicEMSPageCount: PAGESAVAIL=">,1
    @DbgOutw ax,1
    @DbgOuts <10>,1
endif
	mov	[_EMSPAGESAVAIL],eax
	popad
	ret

@@nofree:
	xor	eax,eax
	jmp	@@ret

DynamicEMSPageCount	ENDP


; allocate an EMS page
; upon entry [edi-1] -> EMS page table entry for page
; return carry if fail, due to internal failure
; destroy no registers

GetEMSPage	PROC
	push	eax
	push	ebx
	push	edx

	lea	ebx,[edi-1]
	sub	ebx, [EMSPAGETABLE]		; absolute offset in table
	shl	ebx,2					; dword/entry
	add	ebx, [EMSPageAllocationStart]	; ebx -> EMS page descriptor entry
	cmp	ebx, [EMSPageAllocationEnd]
	jae	@@bad2			; out of range

	call	Allocate16KPageFromPoolBlock
	or	edx,edx
	jne	@@success

	call	ExpandAnyPoolBlock
	or	edx,edx			; see if any pool block has 16K free
	je	@@tryxms

; pool block was expanded, so an allocation should succeed
	call	Allocate16KPageFromPoolBlock
	or	edx,edx
	jne	@@success

; shouldn't reach this point since pool expansion complete,
;  fall through to trying XMS
@@bad1:
	@DbgOuts <"GetEMSPage, bad1 reached",10>,?POOLDBG

@@tryxms:
	call	AllocateXMSForPool
	jc	@@fail
	call	Allocate16KPageFromPoolBlock	; this should always succeed
	or	edx,edx
	je	@@bad2			; failed due to internal fault, so fail allocation

@@success:
	clc					; flag no errors

@@allocret:
	pop	edx
	pop	ebx
	pop	eax
@@ret:
	ret

@@bad2:
	@DbgOuts <"GetEMSPage, bad2 reached",10>,?POOLDBG
@@fail:
	@DbgOuts <"GetEMSPage, fail reached",10>,?POOLDBG
	stc
	jmp	@@allocret

GetEMSPage	ENDP


; release EMS page, [edi-1] -> EMS page table entry for page
; destroy no registers

ReleaseEMSPage	PROC
	push	edx
	mov	edx,edi
	dec	edx
	sub	edx, [EMSPAGETABLE]				; absolute offset in table
	shl	edx,2							; dword/entry
	add	edx, [EMSPageAllocationStart]	; edx -> EMS page descriptor entry
	cmp	edx, [EMSPageAllocationEnd]
	jae	@@ret							; out of range

	call	Free16KPage

@@ret:
	pop	edx
	ret
ReleaseEMSPage	ENDP


; find count of available XMS 4K-aligned 4K pages in 32K chunks
;  return count in edx
;  destroy no other registers

GetFreeXMS4KPageCount	PROC
	push	esi
	push	eax
	push	ebx
	push	ecx
	xor	edx,edx
	cmp	[_NoPool], 0	; XMS memory pool?
	jne	@@countdone
	movzx ecx, [XMS_Handle_Table.xht_numhandle]
	jecxz @@countdone
	mov	esi, [XMS_Handle_Table.xht_array]
	or	esi,esi
	je	@@countdone

	@assumem esi, <LPXMS_ARRAY_STRUC>

@@hanloop:
	test [esi].xas_flag,XMS_FREE
	jz	@@next
	xor	eax,eax
	cmp	eax,[esi].xas_addressK
	je	@@next		; can't count blank or zero-sized handle
	cmp	eax,[esi].xas_sizeK
	je	@@next

	mov	eax,[esi].xas_addressK
	mov	ebx,eax
	add	ebx,3		; round up
	add	eax,[esi].xas_sizeK
	and	al,0fch		; align to 4K boundary
	and	bl,0fch
	sub	eax,ebx		; compute size of block after alignments
	jbe	@@next
	and	al,NOT 1fh	; mask to 32K
	shr	eax,2		; convert 1K to 4K
	add	edx,eax		; update total count

@@next:
	movzx	eax, [XMS_Handle_Table.xht_sizeof]
	add	esi,eax	; move to next handle descriptor
	dec	ecx
	jne	@@hanloop

	@assumem esi, nothing

@@countdone:
	pop	ecx
	pop	ebx
	pop	eax
	pop	esi
	ret
GetFreeXMS4KPageCount	ENDP


; get total and free 4K page count in pool
; return total in edx, free in eax
; destroy no other registers

GetCurrent4KPageCount	PROC
	push	esi
	push	ecx
	mov	esi, [PoolAllocationTable]

	xor	eax,eax
	mov	edx,eax
    mov ecx,eax

	@assumem esi,<LPPOOL_SYSTEM_INFO_STRUC>

@@findblkloop:
	cmp	[esi].psi_addressK,0
	je	@@nextblock		; unused/deallocated block
	movzx ecx,[esi].psi_16kmax
	lea	edx,[edx+ecx*4]	; convert to 4K count, known 16-bit quantity
	mov cx,[esi].psi_4kfree
	add	eax,ecx

@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop

	pop	ecx
	pop	esi
	ret

	@assumem esi, nothing
    
GetCurrent4KPageCount	ENDP


; get used 16K (EMS) page count in pool
; out: EDX = used EMS page count
; out: EAX = available EMS pages
; out: EDI = VCPI/EMS used pages
; destroy no other registers

GetCurrentEMSPageCount	PROC
	push	esi
	push	ecx

	mov ecx, [_MAXEMSPAGES]
	and ecx, ecx
	jz @@noems
	mov	edi, [EMSPageAllocationStart]
    or eax,-1
    xor edx, edx
@@countloop:
	repz scas dword ptr [edi]
	jz @@done1    
	inc	edx
    and ecx, ecx
    jnz @@countloop
@@done1:    
	xor edi, edi
    xor eax, eax
	mov	esi, [PoolAllocationTable]

	@assumem esi,LPPOOL_SYSTEM_INFO_STRUC

@@findblkloop:

	cmp	[esi].psi_addressK,0
	je	@@nextfind

	movzx	ecx,[esi].psi_16kmax
	sub	cl,[esi].psi_16kfree
	add	edi,ecx					; update total pages used
	mov	cl,[esi].psi_16kfree	; high words known zero
	add	eax,ecx

@@nextfind:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
@@ret:
if ?POOLDBG
	@DbgOuts <"GetCurrEMSPageCount: free=">,1
	@DbgOutd eax,1
	@DbgOuts <", used=">,1
	@DbgOutd edi,1
	@DbgOuts <", used EMS=">,1
	@DbgOutd edx,1
	@DbgOuts <" pool =">,1
	@DbgOutd [PoolAllocationTable],1
	@DbgOuts <"-">,1
	@DbgOutd [PoolAllocationEnd],1
	@DbgOuts <")",10>,1
endif
	pop	ecx
	pop	esi
	ret
@@noems:
	xor	eax,eax
	mov	edi,eax
	mov	edx,eax
	jmp @@ret

	@assumem esi, nothing

GetCurrentEMSPageCount	ENDP


; locate any adjacent free XMS block to current EMS/VCPI pool allocation block
; if found, try consuming 32K of it for pool allocation block
; the adjacent XMS block must be at the end of current pool block,
;  since the pool block base cannot be changed once set
; pool block base+block size == owner XMS handle base+handle size (end match end)
; ends must match since you can't span noncontiguous sub-blocks of an owner XMS
;  block with a single EMS/VCPI pool allocation block
; INP: EDX -> current EMS/VCPI block
; OUT: NC if success, C if fail
; other registers preserved

ExpandCurrentPoolBlock	PROC

	pushad

	@assumem edx,LPPOOL_SYSTEM_INFO_STRUC

if ?POOLDBG
	@DbgOuts <"ExpandCurrentPoolBlock: edx=">,1
    @DbgOutd edx,1
    @DbgOuts <", addrK=">
    @DbgOutd [edx].psi_addressK
    @DbgOuts <", pArray=">
    @DbgOutd [edx].psi_descptr
	@DbgOuts <10>,1
endif

	cmp	[_NoPool],0	; no XMS handle table info available
	jne	@@locfail
	cmp	[edx].psi_16kmax,2*POOLBLOCK_ALLOCATION_SPACE
	jae	@@locfail				; current block is full
	test [edx].psi_flags,POOLBLOCK_FLAG_DONTEXPAND
	jne	@@locfail				; can't expand this block

	mov	edi,[edx].psi_descptr
    
	@assumem edi, LPXMS_ARRAY_STRUC
    
	mov	ebx,[edi].xas_addressK
	add	ebx,[edi].xas_sizeK	; ebx -> end of current pool block owner XMS

; see if owner XMS for EMS/VCPI allocation block end matches
;  end of pool allocation block

	movzx ecx,[edx].psi_startadj
	mov	eax,[edx].psi_addressK
	sub	eax,ecx					; true XMS start when pool allocation block created
	movzx ecx,[edx].psi_16kmax
	shl	ecx,4						; convert to K
	add	eax,ecx
	movzx ecx,[edx].psi_endadj
	add	eax,ecx					; true XMS end when block created
	cmp	eax,ebx
	jne	@@locfail				; owner XMS end no longer matches initial pool block owner XMS end

	movzx ecx, [XMS_Handle_Table.xht_numhandle]
	or	ecx,ecx
	je	@@locfail
	mov	esi, [XMS_Handle_Table.xht_array]
	test	esi,esi
	je	@@locfail

	@assumem esi, LPXMS_ARRAY_STRUC

; esi -> test XMS block

	movzx	eax, [XMS_Handle_Table.xht_sizeof]
@@hanloop:
	cmp	ebx,[esi].xas_addressK		; see if test block immediately follows current block	
	je	@@found
	add	esi,eax		; move to next handle descriptor
	dec	ecx
	jne	@@hanloop
@@locfail:
	popad
	stc				; flag failure
	ret

@@found:
	test [esi].xas_flag,XMS_FREE	; if block is not free, abort scan
	je	@@locfail
	movzx	eax,[edx].psi_endadj
	add	eax,[esi].xas_sizeK
	cmp	eax,32		; free block plus unused end overlap must be >=32K
	jb	@@locfail

; transfer 32K of following block to current block - unused end K in current
	mov	eax,32
	movzx	ecx,[edx].psi_endadj
	sub	eax,ecx					; adjust amount to change preceding block
	add	[esi].xas_addressK,eax	; move changed block address ahead
	sub	[esi].xas_sizeK,eax		; and adjust size
	mov	edi,[edx].psi_descptr
	add	[edi].xas_sizeK,eax		; increase EMS/VCPI associated XMS block size
	mov	[edx].psi_endadj,0		; no end overlap

	add	[edx].psi_16kmax,2		; adjust allocation tracking bytes
	add	[edx].psi_16kfree,2
	add	[edx].psi_4kfree,2*4
	movzx	eax,[edx].psi_16kmax
	shr	eax,1					; byte offset in allocation space (32K/byte)
	dec	eax						; relative 0
	mov	BYTE PTR [edx+eax+POOLBLOCK_SYSTEM_SPACE],0	; zero tracking allocation byte

; see if changed contiguous XMS block size went to <32K,
;  if so, transfer any remainder to pool block and zero XMS block

	mov	eax,[esi].xas_sizeK
	cmp	eax,31
	ja	@@loc2
	mov	[edx].psi_endadj,al

	xor	eax,eax
	mov	[esi].xas_addressK,eax
	mov	[esi].xas_sizeK,eax
	mov	[esi].xas_lockcount,al
	mov	[esi].xas_flag,XMS_INPOOL 	; flag: free handle!

@@loc2:
	mov	[LastBlockFreed],edx	; expanding block size is same as freeing space
	popad
	clc					; flag success
	ret

	@assumem edi, nothing
	@assumem esi, nothing
	@assumem edx, nothing

ExpandCurrentPoolBlock	ENDP


; expand any available allocation pool block by 32K, if possible
;  return edx -> expanded allocation pool block, zero if none
; destroy no other registers

ExpandAnyPoolBlock	PROC
	push	esi
	cmp	[_NoPool],0	; no XMS handle table info available
	jne	@@fail

	mov	esi, [PoolAllocationTable]
    
	@assumem esi, LPPOOL_SYSTEM_INFO_STRUC

@@findblkloop:
	cmp	[esi].psi_addressK,0	; unused/deallocated block
	je	@@nextblock
	cmp	[esi].psi_16kmax,2*POOLBLOCK_ALLOCATION_SPACE
	jae	@@nextblock				; current block is full
	mov	edx,esi
	call ExpandCurrentPoolBlock
	jnc @@done
@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop

@@fail:
	xor	edx,edx			; failure

@@done:
	pop	esi
	ret

	@assumem esi, nothing

ExpandAnyPoolBlock	ENDP


; find and allocate free 4K (VCPI) block in pool blocks
; return edx == physical address, zero if none found
; destroy ecx,esi,edi
Allocate4KPageFromPoolBlock	PROC
	push	eax

	@assumem esi, LPPOOL_SYSTEM_INFO_STRUC

@@begin:
	xor	edx,edx

; first try last block allocated, to avoid searching full blocks if possible
	mov	esi, [LastBlockAllocator]
	or	esi,esi
	je	@@nolastalloc
	cmp	[esi].psi_addressK,edx
	je	@@nolastalloc
	cmp	[esi].psi_4kfree,dx
	jne	@@searchbytes

; try last freed chunk
@@nolastalloc:
	mov	esi, [LastBlockFreed]
	or	esi,esi
	je	@@nolastfreed
	cmp	[esi].psi_addressK,edx
	je	@@nolastfreed
	cmp	[esi].psi_4kfree,dx
	jne	@@searchbytes

@@nolastfreed:
	mov	esi, [PoolAllocationTable]

@@findblkloop:
	cmp	[esi].psi_addressK,0	; unused/deallocated block
	je	@@nextblock
	cmp	[esi].psi_4kfree,0
	je	@@nextblock

@@searchbytes:
	movzx	ecx,[esi].psi_16kmax
	shr	ecx,1			; count of allocation bytes in block
	xor	edi,edi

@@findbyteloop:
	mov	al,[esi+edi+POOLBLOCK_SYSTEM_SPACE]
	xor	al,-1			; unallocated 4K areas show as bit set
	jne	@@freebyte	; at least one unallocated area
	inc	edi
	loop @@findbyteloop

; nothing free, although block indicated there was
	@CheckBlockIntegrity
	jmp	@@begin

@@freebyte:
	mov	ecx,edi
	shl	ecx,15		; each byte covers 32K
	mov	edx,[esi].psi_addressK
	shl	edx,10		; convert from K to bytes
	add	edx,ecx		; compute base address of block addressed by byte

	mov	ah,1

; al holds bitcodes of allocations, set if available due to xor
@@bitloop:
	shr	al,1
	jc	@@found
	add	edx,4096
	shl	ah,1
	jmp	@@bitloop

@@found:
	or	[esi+edi+POOLBLOCK_SYSTEM_SPACE],ah	; flag that 4K area has been allocated
	dec	[esi].psi_4kfree
	mov	al,[esi+edi+POOLBLOCK_SYSTEM_SPACE]
	cmp	ah,0fh		; see if allocated from low nybble or high
	ja	@@highnyb

; low nybble
	and	al,0fh
	jmp	@@nybshared

@@highnyb:
	and	al,0f0h

; see if first allocation in that nybble
;  if so, then reduce free 16K region count since it was partially consumed
@@nybshared:
	mov	[LastBlockAllocator],esi	; update last block allocated from

	xor	al,ah			; turn off bit we turned on
	jne	@@finddone	; not the first bit allocated in 16K region (nybble)
	dec	[esi].psi_16kfree
	jnc	@@finddone

	@CheckBlockIntegrity
	jmp	@@finddone

@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop

@@finddone:
	pop	eax
	ret
	@assumem esi, nothing

Allocate4KPageFromPoolBlock	ENDP


; find and allocate free 16K (EMS) block in pool blocks
; upon entry ebx -> EMS page descriptor entry
; return edx == physical address, zero if none found
;  set page descriptor high word == 64-byte EMS/VCPI allocation block count from start, RELATIVE 1!
;  set page descriptor low word ==  half-byte offset of allocation page (2 * 48)
;   within allocation block, not including system bytes
; destroy no other registers

Allocate16KPageFromPoolBlock	PROC
	push	eax
	push	ecx
	push	esi
	push	edi

	@assumem esi, LPPOOL_SYSTEM_INFO_STRUC

@@begin:
	xor	edx,edx

; first try last block allocated, to avoid searching full blocks if possible
	mov	esi, [LastBlockAllocator]
	or	esi,esi
	je	@@nolastalloc
	cmp	[esi].psi_addressK,edx
	je	@@nolastalloc
	cmp	[esi].psi_16kfree,dl
	jne	@@searchbytes

; try last freed chunk
@@nolastalloc:
	mov	esi, [LastBlockFreed]
	or	esi,esi
	je	@@nolastfreed
	cmp	[esi].psi_addressK,edx
	je	@@nolastfreed
	cmp	[esi].psi_16kfree,dl
	jne	@@searchbytes

@@nolastfreed:
	mov	esi, [PoolAllocationTable]

@@findblkloop:
	cmp	[esi].psi_addressK,0	; unused/deallocated block
	je	@@nextblock
	cmp	[esi].psi_16kfree,0
	je	@@nextblock

@@searchbytes:
	movzx	ecx,[esi].psi_16kmax
	shr	ecx,1			; count of allocation bytes in block
	xor	edi,edi

@@findbyteloop:
	mov	al,[esi+edi+POOLBLOCK_SYSTEM_SPACE]
	xor	al,-1			; unallocated 4K areas show as bit set
	mov	ah,al
	and	al,0fh
	cmp	al,0fh
	je	@@lowfree		; low nybble unallocated, free 16K area

	and	ah,0f0h
	cmp	ah,0f0h
	je	@@highfree		; high nybble unallocated, free 16K area

; no free 16K area
	inc	edi
	loop @@findbyteloop
	@CheckBlockIntegrity
	jmp	@@begin

@@lowfree:
	or	BYTE PTR [esi+edi+POOLBLOCK_SYSTEM_SPACE],0fh
	mov	cl,0
	jmp	@@freeshared	; edx == 0

@@highfree:
	or	BYTE PTR [esi+edi+POOLBLOCK_SYSTEM_SPACE],0f0h
	mov	cl,1
	mov	edx,16		; nybble offset is four 4K pages, 16K

@@freeshared:
	mov	eax,edi
	shl	eax,15		; each byte covers 32K
	add	edx,[esi].psi_addressK	; add in base value
	shl	edx,10		; convert K to bytes
	add	edx,eax		; edx == 16k page memory address
	dec	[esi].psi_16kfree
	sub	[esi].psi_4kfree,4
	jnc	@@valid2
	@CheckBlockIntegrity	; force valid value

; update ebx pointer
@@valid2:
	mov	eax,esi
	sub	eax, [PoolAllocationTable]
	shr	eax,6		; convert to 64-byte offset (block count)
	inc	eax			; make count relative 1
	mov	[ebx].EMSPD.wPD,ax
	movzx eax,cl	; get even/odd offset
	shl	edi,1		; half-byte offset within block, not counting system info bytes
	add	eax,edi		; ax holds half-byte offset within block
	mov	[ebx].EMSPD.wNibOfs,ax

	mov	[LastBlockAllocator],esi	; update last block allocated from

@@alloc16ret:
	pop	edi
	pop	esi
	pop	ecx
	pop	eax
	ret

@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
	jmp	@@alloc16ret

	@assumem esi, nothing

Allocate16KPageFromPoolBlock	ENDP


; find an unused allocation block
; return edx -> allocation block, or zero if none and no space available
; no other registers modified
GetUnusedAllocationBlock	PROC
	mov	edx, [PoolAllocationTable]

	@assumem edx, LPPOOL_SYSTEM_INFO_STRUC

@@findblkloop:
	cmp	[edx].psi_addressK,0	; unused/deallocated block
	je	@@getret

@@nextblock:
	add	edx,POOLBLOCK_TOTAL_SPACE
	cmp	edx, [PoolAllocationEnd]
	jb	@@findblkloop

	xor	edx,edx		; failed

@@getret:
	ret
	@assumem edx, nothing

GetUnusedAllocationBlock	ENDP

; prepare allocation block for use
; upon entry:
;  edx -> pool allocation block
;  ecx == raw size in K before alignment
;  edi == raw address in K before alignment
;  esi == owner XMS handle psi_descptr value, do NOT use XMS handle values for
;   size and address since this call may be part of a multi-pool block span
; destroys no registers

PrepAllocationBlock	PROC

	pushad
    
	@assumem edx, LPPOOL_SYSTEM_INFO_STRUC
    
	mov	[edx].psi_descptr,esi

	mov	eax,edi			; raw address
	mov	bl,al
	and	bl,3
	mov	bh,4
	sub	bh,bl
	and	bh,3
	mov	[edx].psi_startadj,bh
	add	eax,3			; round up address
	and	al,NOT 3
	mov	[edx].psi_addressK,eax

; block size = (raw size - start adjustment) rounded down to 32K boundary
;  since each allocation byte covers 32K
	mov	eax,ecx			; raw size
	movzx	ebx,bh		; start adustment
	sub	eax,ebx
	and	al,NOT 31
	shr	eax,4			; 16K count, known 16-bit value going to 8-bit

	cmp	ax,POOLBLOCK_ALLOCATION_SPACE*2
	jbe	@@setmax
	mov	ax,POOLBLOCK_ALLOCATION_SPACE*2

@@setmax:
	mov	[edx].psi_16kmax,al
	mov	[edx].psi_16kfree,al
	shl	eax,2			; 8-bit value potentially going to 16-bit
	mov	[edx].psi_4kfree,ax

; compute end adjustment, raw size - (block size + start adjustment)
	movzx	ebx,[edx].psi_16kmax
	shl	ebx,4			; convert to K
	movzx	eax,[edx].psi_startadj
	add	eax,ebx
	sub	ecx,eax			; ecx == raw size
	mov	[edx].psi_endadj,cl

; zero allocation entries
	xor	eax,eax
	lea	edi,[edx+POOLBLOCK_SYSTEM_SPACE]
	mov	ecx,POOLBLOCK_ALLOCATION_SPACE/4
    rep stos dword ptr [edi]
    BIG_NOP

	popad
	ret
	@assumem edx, nothing

PrepAllocationBlock	ENDP


; populate empty pool allocation blocks with XMS owner info
; upon entry esi -> owner XMS (pseudo-)handle, ecx == size, edi == address
;  NOTE: ecx and edi may not match owner XMS size/address
; return carry set if insufficient number of empty blocks to cover XMS handle range
;  reset otherwise
; destroy eax,ecx,edx
;  update edi
GetBlocksForXMSHandle	PROC
	push	ebx
	mov	ebx,ecx

@@allocloop:
	call	GetUnusedAllocationBlock
	or	edx,edx
	je	@@exhausted			; no more blocks, remainder of XMS is effectively discarded

	mov	eax,edi				; compute size of candidate block/offset to new
	and	al,3
	mov	ah,4
	sub	ah,al
	and	ah,3
	movzx eax,ah
	add	eax,1536			; 1.5M (in K) plus alignment adjustment size
	cmp	eax,ebx
	jbe	@@sizeok
	mov	eax,ebx

@@sizeok:
	mov	ecx,eax
	call	PrepAllocationBlock	; uses esi entry condition
	sub	ebx,eax			; update size left to allocate
	je	@@allocsuccess
	cmp	ebx,32			; see if should remainder what's left
	jb	@@remainder
	add	edi,eax			; update pool allocation block address
	jmp	@@allocloop

@@exhausted:
	stc
	jmp	@@getret

	@assumem edx,LPPOOL_SYSTEM_INFO_STRUC

@@remainder:
	mov	[edx].psi_endadj,bl

@@allocsuccess:
	clc

@@getret:
	pop	ebx
	ret

	@assumem edx,nothing

GetBlocksForXMSHandle	ENDP


; walk XMS blocks, find largest XMS block which is sized 1.5M or smaller >=32K
;  after 4K alignment and allocate it for new EMS/VCPI pool allocation block
; if all XMS blocks >1.5M, then pick smallest and try to put remainder
;  into a free handle.
; If no free handle, allocate sufficient new EMS/VCPI pool blocks to cover full range.
; If not enough free EMS/VCPI blocks available, then remaining allocation is lost
;  until handle is freed.  This could only happen under very bad XMS fragmentation,
;  if at all.
; return carry clear if success, set if fail
; destroy no other registers

AllocateXMSForPool	PROC

	pushad

	cmp	[_NoPool],0	; check if pool sharing
	jne	@@allocfail

	mov	esi, [XMS_Handle_Table.xht_array]
	movzx ecx, [XMS_Handle_Table.xht_numhandle]

if ?POOLDBG
	@DbgOuts <"AllocateXMSForPool: XMS array=">,1
    @DbgOutd esi,1
    @DbgOuts <", handles=">,1
    @DbgOutw cx,1
	@DbgOuts <10>,1
endif

	or	esi,esi
	je	@@allocfail
	or	ecx,ecx
	je	@@allocfail

	@assumem esi, LPXMS_ARRAY_STRUC
    
	call	GetUnusedAllocationBlock	; test only, don't keep pointer
	jc	@@allocfail			; unable to make initial pool block allocation
	xor	edx,edx			; edx -> largest block <= 1.5M or smallest if none <=1.5M

@@hanloop:
	test [esi].xas_flag,XMS_FREE
	je	@@next				; yes, don't check
	cmp	[esi].xas_addressK,0
	je	@@next				; can't check blank or zero-sized handle
	mov	eax,[esi].xas_sizeK
	or	eax,eax
	je	@@next
	or	edx,edx
	je	@@newcandidate		; auto-match if first xms block available

; adjust for alignment loss (1-3K) in test
	mov	bl,BYTE PTR [esi].xas_addressK
	and	bl,3
	mov	bh,4
	sub	bh,bl
	and	bh,3
	movzx	ebx,bh
	sub	eax,ebx

	@assumem edx, LPXMS_ARRAY_STRUC

; adjust for alignment lost in candidate
	cmp	[edx].xas_sizeK,32	; ensure candidate size isn't so small that adjustment will underflow
	jb	@@next			; doesn't even meet minimum requirements
	mov	bl,BYTE PTR [edx].xas_addressK
	and	bl,3
	mov	bh,4
	sub	bh,bl
	and	bh,3
	movzx	ebx,bh
	neg	ebx
	add	ebx,[edx].xas_sizeK

; eax holds test value size, ebx holds current candidate, both alignment adjusted
	cmp	eax,ebx
	je	@@next
	ja	@@larger

; test XMS block smaller than candidate block
	cmp	ebx,1536			; in K
	jbe	@@next			; current candidate closer to match size
	cmp	eax,32
	jb	@@next				; test too small
	jmp	@@newcandidate

; test XMS block larger than candidate block
@@larger:
	cmp	ebx,1536
	jae	@@next			; current candidate closer to match size
	cmp	eax,1536
	ja	@@next				; test too large

@@newcandidate:
	cmp	[esi].xas_sizeK,32
	jb	@@next			; candidate doesn't even meet unadjusted requirements
	mov	edx,esi			; new best candidate

@@next:
	movzx	eax, [XMS_Handle_Table.xht_sizeof]
	add	esi,eax			; move to next handle descriptor
	dec	ecx
	jne	@@hanloop
	or	edx,edx
	je	@@allocfail

; candidate must be at least 32K, after 4K alignment adjustment
	mov	bl,BYTE PTR [edx].xas_addressK
	and	bl,3
	mov	bh,4
	sub	bh,bl
	and	bh,3
	movzx	ebx,bh
	neg	ebx
	add	ebx,[edx].xas_sizeK
	cmp	ebx,32
	jb	@@allocfail				; candidate too small

if ?POOLDBG
	@DbgOuts <"AllocateXMSForPool: used XMS block handle=">,1
    @DbgOutd edx,1
    @DbgOuts <" addr=">,1
    @DbgOutd [edx].xas_addressK,1
    @DbgOuts <" siz=">,1
    @DbgOutd [edx].xas_sizeK,1
	@DbgOuts <10>,1
endif

	mov	[edx].xas_flag,XMS_USED	; flag candidate as used so it doesn't show up as free
	mov	[edx].xas_lockcount,1	; and show as locked

	mov	[XMSBlockSize],1536	; default allocation maximum size
	mov	eax, [XMSPoolBlockCount]
	cmp	eax,1
	jbe	@@trailadj			; use standard 1.5M size for first two blocks
	dec	eax					; should never overflow before we hit 4G total allocated

@@noadj:
	and	al,0fh				; but ensure that overflow doesn't happen anyway
	mov	cl,al				; shift the block size higher by factor of two
	shl	[XMSBlockSize],cl

; default is same as a MAX setting
;	cmp	[_MAXMEM16K],MAXMEM16K_DEFAULT
;	je	@@trailadj			; no MAX setting

; MAX setting, don't overallocate XMS we can't use
	push	edx
	call	GetCurrent4KPageCount	; current free in eax, current total in edx
	sub	edx,eax		; edx == current allocated (total - free)
	jc	@@adjusted		; shouldn't happen, continue without adjustment if it does

; if XMSBlockSize >= MAXMEM16K * 4 - allocated, then reduce XMSBlockSize
	mov	eax,[_MAXMEM16K]
	shl	eax,2			; convert to maximum 4K blocks
	sub	eax,edx
	jc	@@adjusted		; shouldn't happen, continue without adjustment
	shl	eax,2			; convert to 1K blocks

@@checksize:
	cmp	eax, [XMSBlockSize]
	jae	@@adjusted
	cmp	[XMSBlockSize],1536	; see if XMSBlockSize is at minimum default
	jbe	@@adjusted		; yes, can't reduce it any further
	shr	[XMSBlockSize],1	; reduce block size by one shift and try again
	jmp	@@checksize

@@adjusted:
	pop	edx

; allow up to 31K trailing bytes
@@trailadj:
	mov	eax, [XMSBlockSize]
	add	eax,31				; adjust for possible trail
	cmp	ebx,eax
	jbe	@@setblock			; no need to split XMS handle allocation

; search for a free XMS handle

	mov	  edi, [XMS_Handle_Table.xht_array]
	movzx ecx, [XMS_Handle_Table.xht_numhandle]
	movzx eax, [XMS_Handle_Table.xht_sizeof]
	@assumem edi, LPXMS_ARRAY_STRUC
@@freeloop:
	test [edi].xas_flag,XMS_INPOOL
    jnz @@gotfree
	cmp	[edi].xas_flag,XMS_USED	; some Himems dont set XMS_INPOOL, so
	je	@@nextfree				; check FREE items if address/size is NULL
	cmp	[edi].xas_addressK,0
	je	@@gotfree
	cmp	[edi].xas_sizeK,0
	je	@@gotfree
@@nextfree:
	add	edi,eax			; move to next handle descriptor
	loop @@freeloop

; no free handle found, try to allocate multiple blocks, discarding excess
	mov	ecx,[edx].xas_sizeK
	mov	edi,[edx].xas_addressK
	mov	esi,edx			; esi -> owner XMS pseudo-handle
	call	GetBlocksForXMSHandle	; ignore return status, one block minimum allocated
	jmp	@@allocsuccess

@@gotfree:
	mov	cl,BYTE PTR [edx].xas_addressK	; compute size of candidate block/offset to new
	and	cl,3
	mov	ch,4
	sub	ch,cl
	and	ch,3
	movzx	ecx,ch

;	add	ecx,1536				; 1.5M (in K) plus alignment adjustment size
	add	ecx, [XMSBlockSize]	; maximum size (exceeded) plus alignment adjustment size

; edx -> candidate block being allocated, edi -> new block receiving remainder
; update candidate XMS block size
	mov	eax,[edx].xas_sizeK		; keep original size for updating new block
	mov	[edx].xas_sizeK,ecx

; update new XMS block info
	sub	eax,ecx					; new block size == old block original size - old block new size
	mov	[edi].xas_sizeK,eax
	mov	[edi].xas_flag,XMS_FREE	; explicitly flag free
	mov	[edi].xas_lockcount,0
	mov	eax,[edx].xas_addressK
	add	eax,ecx
	mov	[edi].xas_addressK,eax	; new block start == old block start + old block new size

if ?POOLDBG
	@DbgOuts <"AllocateXMSForPool: free XMS block handle=">,1
    @DbgOutd edi,1
    @DbgOuts <" addr=">,1
    @DbgOutd [edi].xas_addressK,1
    @DbgOuts <" siz=">,1
    @DbgOutd [edi].xas_sizeK,1
	@DbgOuts <10>,1
endif

; edx -> owner XMS handle for new pool allocation block(s)
; may be multiple blocks due to XMSBlockCount shifter
@@setblock:
	mov	esi,edx
;	call	GetUnusedAllocationBlock	; assume success, since entry test worked

;; update pool allocation block system info
;; edx -> pool allocation block, esi -> owner XMS handle
	mov	ecx,[esi].xas_sizeK
	mov	edi,[esi].xas_addressK
;	call	PrepAllocationBlock

	call	GetBlocksForXMSHandle

@@allocsuccess:
	inc	[XMSPoolBlockCount]
    popad
	clc
	ret

@@allocfail:
	popad
	stc
	ret
    @assumem edx,nothing
    @assumem esi,nothing
    @assumem edi,nothing

AllocateXMSForPool	ENDP


; upon entry edx -> 4K page absolute address to free
; return carry clear on success, set on fail
; destroy ecx,esi,edi

Free4KPage	PROC
	push	eax
	push	ebx
	push	edx

	mov	ebx,edx
	shr	ebx,10			; convert bytes to K
	and	bl,NOT 3		; ensure 4K alignment

	@assumem esi, LPPOOL_SYSTEM_INFO_STRUC

	mov	esi, [LastBlockFreed]
	or	esi,esi
	je	@@notlastfreed
	mov	eax,[esi].psi_addressK
	or	eax,eax
	je	@@notlastfreed	; unused/deallocated block

; ebx == start of 4K page in K after alignment adjustment
	cmp	ebx,eax
	jb	@@notlastfreed	; pool block starts after page
	movzx	ecx,[esi].psi_16kmax
	shl	ecx,4			; convert 16K to 1K
	add	ecx,eax			; ecx == end of range holding 4K pages
	cmp	ebx,ecx
	jb	@@rightblock	; after start, before end

@@notlastfreed:
	mov	esi, [LastBlockAllocator]
	or	esi,esi
	je	@@notlastalloc
	mov	eax,[esi].psi_addressK
	or	eax,eax
	je	@@notlastalloc	; unused/deallocated block

	cmp	ebx,eax
	jb	@@notlastalloc	; pool block starts after page
	movzx	ecx,[esi].psi_16kmax
	shl	ecx,4			; convert 16K to 1K
	add	ecx,eax			; ecx == end of range holding 4K pages
	cmp	ebx,ecx
	jb	@@rightblock	; after start, before end

@@notlastalloc:
	mov	esi, [PoolAllocationTable]

@@findblkloop:
	mov	eax,[esi].psi_addressK
	or	eax,eax
	je	@@nextblock		; unused/deallocated block

	cmp	ebx,eax
	jb	@@nextblock	; pool block starts after page
	movzx	ecx,[esi].psi_16kmax
	shl	ecx,4			; convert 16K to 1K
	add	ecx,eax			; ecx == end of range holding 4K pages
	cmp	ebx,ecx
	jb	@@rightblock	; page to free within pool block

@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop

@@fail:
	@CheckBlockIntegrity
	stc
	jmp	@@ret

; this is the proper pool allocation block
@@rightblock:
	sub	ebx,eax		; 4K offset from block base in K
	mov	eax,ebx
	shr	eax,2			; K to 4K page
	mov	cl,al			; keep bit offset
	shr	eax,3			; 4K page to 32K byte offset
	and	cl,7
	mov	bl,1
	rcl	bl,cl			; get mask bit

	test	[esi+eax+POOLBLOCK_SYSTEM_SPACE],bl	; see if bit set (was allocated)
	je	@@fail			; no

	not	bl				; turn on all bits except current allocation's
	and	[esi+eax+POOLBLOCK_SYSTEM_SPACE],bl

	inc	[esi].psi_4kfree

; check if this frees up a 16K chunk
	mov	al,[esi+eax+POOLBLOCK_SYSTEM_SPACE]
	and	bl,0f0h
	cmp	bl,0f0h		; see if high nybble was freed
	je	@@nothigh		; all bits set, freed portion was low

	test	al,0f0h		; see if all high bits of nybble cleared
	jne	@@success	; no
	jmp	@@inc16

@@nothigh:
	test	al,0fh		; see if all low bits of nybble cleared
	jne	@@success		; no

@@inc16:
	inc	[esi].psi_16kfree

@@success:
	mov	[LastBlockFreed],esi
	call	TryFreeToXMS	; free empty pool allocation block to XMS if appropriate
	clc

@@ret:
	pop	edx
	pop	ebx
	pop	eax
	ret
    @assumem esi,nothing

Free4KPage	ENDP


; upon entry edx -> EMS page descriptor pointer
;  upon return set page descriptor pointer to -1 (unused)
; destroy no registers

Free16KPage	PROC
	push	esi
	push	eax
	push	ecx

	movzx	esi,[edx].EMSPD.wPD	; [e]si == 64-byte block count, relative 1
	cmp	si,-1
	je	@@fail			; bad pointer
	dec	esi				; make relative 0
	shl	esi,6			; convert 64-byte count to byte offset
	add	esi, [PoolAllocationTable]	; esi -> pool allocation block

	movzx ecx,byte ptr [edx].EMSPD.wNibOfs	; half-byte offset
	shr	ecx,1			; byte offset
	mov	al,[esi+ecx+POOLBLOCK_SYSTEM_SPACE]
	test BYTE PTR [edx].EMSPD.wNibOfs,1	; see if odd/even nybble
	jne	@@oddbyte

	and	al,0fh
	and	BYTE PTR [esi+ecx+POOLBLOCK_SYSTEM_SPACE],0f0h	; reset all expected bits
	cmp	al,0fh
	jne	@@fail		; not all expected bits were set
	jmp	@@success

@@oddbyte:
	and	al,0f0h
	and	BYTE PTR [esi+ecx+POOLBLOCK_SYSTEM_SPACE],0fh	; reset all expected bits
	cmp	al,0f0h
	jne	@@fail

	@assumem esi, LPPOOL_SYSTEM_INFO_STRUC

@@success:
	inc	[esi].psi_16kfree
	add	[esi].psi_4kfree,4
	mov	[LastBlockFreed],esi
	call	TryFreeToXMS	; free empty pool allocation block to XMS if appropriate
	clc

@@ret:
	mov	DWORD PTR [edx].EMSPD.wNibOfs,-1
	pop	ecx
	pop	eax
	pop	esi
	ret

@@fail:
	@CheckBlockIntegrity
	stc
	jmp	@@ret

    @assumem esi,nothing

Free16KPage	ENDP


; upon entry esi -> pool allocation block to check if freeable to XMS
; perform the free if possible
; destroys eax,ecx,esi

TryFreeToXMS	PROC
	push	edx
	push	edi

	cmp	[_NoPool],0	; check if pool sharing
	jne	@@checkdone		; no

	@assumem esi, LPPOOL_SYSTEM_INFO_STRUC
    
	test	[esi].psi_flags,POOLBLOCK_FLAG_DONTEXPAND
	jne	@@checkdone		; can't free this block

@@proccheck:
	mov	al,[esi].psi_16kfree
	cmp	al,[esi].psi_16kmax
	ja	@@bad			; free more than max, try to fix
	jne	@@checkdone		; free is less than maximum, used

	movzx eax,[esi].psi_4kfree
	shr	eax,2
	or	ah,ah
	jne	@@bad
	cmp	al,[esi].psi_16kmax
	ja	@@bad
	jne	@@checkdone		; free less than max

; walk all pool blocks, see if all blocks for XMS handle are empty or nonexistent
;  if so, then mark XMS handle as free
	mov	esi,[esi].psi_descptr
	mov	edx,[PoolAllocationTable]
    
	@assumem edx, LPPOOL_SYSTEM_INFO_STRUC
	@assumem esi, LPXMS_ARRAY_STRUC

@@checkblkloop:
	cmp	[edx].psi_addressK,0
	je	@@checknext			; unused block

	cmp	esi,[edx].psi_descptr
	jne	@@checknext

	test	[edx].psi_flags,POOLBLOCK_FLAG_DONTEXPAND
	jne	@@checkdone		; can't free this block

; see if block empty
	movzx ecx,[edx].psi_16kmax
	cmp	cl,[edx].psi_16kfree
	jne	@@checkdone

	shl	ecx,2				; convert to 4K max
	cmp	cx,[edx].psi_4kfree
	jne	@@checkdone

@@checknext:
	add	edx, POOLBLOCK_TOTAL_SPACE
	cmp	edx, [PoolAllocationEnd]
	jb	@@checkblkloop

; checked all blocks as empty, go through them again and mark unused

	mov	edx, [PoolAllocationTable]

@@freeblkloop:
	cmp	[edx].psi_addressK,0
	je	@@freenext			; unused block

	cmp	esi,[edx].psi_descptr
	jne	@@freenext

; zero the block
	mov	ecx,POOLBLOCK_TOTAL_SPACE/4
	xor	eax,eax
	mov	edi,edx
    rep stos dword ptr [edi]
    BIG_NOP

@@freenext:
	add	edx,POOLBLOCK_TOTAL_SPACE
	cmp	edx, [PoolAllocationEnd]
	jb	@@freeblkloop

	mov	[esi].xas_lockcount,0
	mov	[esi].xas_flag,XMS_FREE		; XMS block is free
	dec	[XMSPoolBlockCount]

	call	DefragXMS

@@checkdone:
	pop	edi
	pop	edx
	ret

@@bad:
	@CheckBlockIntegrity
;;	jmp	@@proccheck		;this would loop "forever"
	jmp	@@checkdone

	@assumem esi, nothing
	@assumem edx, nothing

TryFreeToXMS	ENDP

; try to defrag XMS blocks
; scan through the XMS handle array
; and try to merge FREE blocks
; destroys eax,ecx,edx,esi,edi

DefragXMS	PROC
	push	ebx
	mov	esi, [XMS_Handle_Table.xht_array]
	movzx ecx, [XMS_Handle_Table.xht_numhandle]

	@assumem esi, LPXMS_ARRAY_STRUC

	or	esi,esi
	je	@@nodefrag
	or	ecx,ecx
	je	@@nodefrag
    
	movzx	eax, [XMS_Handle_Table.xht_sizeof]
@@defragloop:
	cmp	[esi].xas_flag,XMS_FREE	; see if free
	jne	@@defragnext			; anything else is to ignore
	cmp	[esi].xas_addressK,0
	je	@@defragnext			; can't check blank handle

@@rescan:
	mov	ebx,[esi].xas_addressK
	add	ebx,[esi].xas_sizeK		; ebx -> end of test block

;--- now check array again if a successor exist which is also FREE

	mov	edi, [XMS_Handle_Table.xht_array]
	movzx edx, [XMS_Handle_Table.xht_numhandle]

	@assumem edi, LPXMS_ARRAY_STRUC

@@checkloop:
	cmp	[edi].xas_flag,XMS_FREE ; check if a FREE block exists 
	jne @@checknext
	cmp	ebx,[edi].xas_addressK	; which starts at EBX (end of test block)
	jne	@@checknext
	mov	edx,[edi].xas_sizeK
	add	[esi].xas_sizeK,edx	; update merger block size
	mov	[esi].xas_lockcount,0	; ensure not showing as locked, shouldn't be

	mov	[edi].xas_flag,XMS_INPOOL	; flag handle as free
	mov	[edi].xas_lockcount,0
	mov	[edi].xas_addressK,0
	mov	[edi].xas_sizeK,0
	jmp	@@rescan		;there might come just another free block to merge
@@checknext:
	add	edi,eax			; move to next handle descriptor
	dec	edx
	jne	@@checkloop

@@defragnext:
	add	esi,eax			; move to next handle descriptor
	loop @@defragloop
@@nodefrag:
	pop	ebx
	ret

	@assumem esi, nothing
	@assumem edi, nothing

DefragXMS	ENDP

; hook left for debugging, no current actions taken
;; upon entry esi -> pool allocation block to perform integrity check upon
;;  update with valid information if allocation counts mismatch
;; destroy no registers

if ?POOLDBG
CheckBlockIntegrity	PROC
	ret
CheckBlockIntegrity	ENDP
endif

_DATA segment
;--- IO permission bitmap init values
;--- this is stored in V86 segment,since the DMA port watch
;--- is now dynamically switched on and off

if ?MASM
IOBM label byte
else
IOBM:
endif

; I/O-control of the DMA-port
; * trap ports 0..7, A, B, C, (D, E, F)
; * trap ports 81,82,83, 87, 89,8A,8B
; * trap ports c0..cf
; * trap port d4 d6 d8 (da, dc, de)

if ?DMA or ?A20PORTS
;----- 76543210--FEDCBA98
 if ?DMA
  if ?MMASK
	DB 11111111B,11111100B  ; DMA-Controller #1 (00-0F)
  else
	DB 11111111B,00011100B  ; DMA-Controller #1 (00-0F)
  endif
 else
	DW 0
 endif
	DW 0,0,0,0,0			; ports 10-5F
 if ?A20PORTS    
	DW 0000000000010001b	; ports 60-6F
 else
	DW 0					; ports 60-6F
 endif   
	DW 0					; ports 70-7F
 if ?DMA
;----- 76543210--FEDCBA98
	DB 10001110B,00001110B  ; page register (80-8F)
 else
 	DW 0
 endif
 if ?A20PORTS
	DW 0000000000000100b	; ports 90-9F
 else
	DW 0					; ports 90-9F
 endif   
	DW 0,0                  ; ports A0-BF
 if ?DMA
	DB 11111111B,11111111B	; DMA-Controller #2 (C0-CF)
;----- 76543210--FEDCBA98
  if ?MMASK
    DB 01010000B,01010101B  ; DMA-Controller #2 (D8-DF)
  else
    DB 01010000B,00000001B  ; DMA-Controller #2 (D8-DF)
  endif
 else
    DW 0,0
 endif
endif

IOBM_COPY_LEN equ $ - IOBM

_DATA ends

	align 16

if ?MASM
V86_LEN        label byte
else
V86_LEN        label byte 
endif

V86SEG ENDS

;
; installation part of the virtual Monitor, that later is given back again
; to the system.
;

_TEXT SEGMENT

	assume CS:_TEXT16
    
    assume FS:nothing
    assume GS:nothing

;
; check, if this program runs after all on a 386-Computer (o.ae.)
; (... you never know)
;
IS386 PROC NEAR
	PUSHF
	mov     AX,7000h
	PUSH    AX
	POPF                        ; on a 80386 in real-mode, bits 15..12
	PUSHF                       ; should be 7, on a 8086 they are F,
	POP     AX                  ; on a 80286 they are 0
	POPF
    and		ah,0F0h
	cmp     AH,70H
    stc
	JNZ     @@NO_386
    clc
@@NO_386:
	RET
IS386 ENDP

;--- test if CPU is 386, display error if not
;--- returns C and modifies DS in case of error
;--- other registers preserved

TestCPU proc near
	push ax
	call IS386
    and ax, ax
    pop ax
    jnc @@is386
    push ax
    push dx
    push cs
    pop ds
    mov ah,9
    mov dx,offset _TEXT16:dErrCpu
    int 21h
    pop dx
    pop ax
    stc
@@is386:    
    ret
TestCPU endp

dErrCpu  db "Error: JEMM386 requires at least a 80386 to run",13,10,'$'

;--- check cpu type, return features if CPUID is supported

IS486 proc
	pushfd						; save EFlags
    xor	edx,edx
	push 24h					; set AC+ID bits in eflags
    push 0						; clear IF
	popfd
	pushfd
	pop ax
	pop ax						; get HiWord(EFlags) into AX
	popfd						; restore EFlags
	test al,04					; AC bit set? then it is a 486+
    mov ah,0
	je @@no_486
    inc ah
    test al,20h                 ; CPUID supported?
    jz @@nocpuid
    xor	eax,eax
    inc	eax				  		; get register 1
	@cpuid
    mov ah,1
@@nocpuid:
@@no_486:
	mov al,ah
	ret
IS486 endp

if ?LOADSUPP

precheck proc
	MOV     AX,3567H         ; get INT 67h
	INT     21H
	MOV     AX,ES            ; EMM already installed ?
	OR      AX,BX
	JZ      @@ok
	MOV     DI,10
	MOV     SI,OFFSET sig1
	CLD
	MOV     CX,8
	REPZ    CMPSB
	je @@ERROR      	; matched 1st ID string
	mov	di, 10			; didn't match, check 2nd ID (NOEMS version)
	mov	si, OFFSET sig2
	mov	cx, 8
	repz	cmpsb
	je @@ERROR   		; did match 2nd ID string
@@ok:
	clc
	ret
@@error:
	stc
    ret
precheck endp    

postcheck proc
	mov ah,52h
    int 21h
    add bx,22h
	push ds
    mov ax, _TEXT16
    mov ds, ax
    mov eax,es:[bx]
    mov ds:[0], eax
	mov word ptr es:[bx+0],0
	mov es:[bx+2],ds
    pop ds
	ret
postcheck endp

endif

;*********************************************
; startpoint when executing as EXE
;*********************************************

GO_EXE proc

	mov     ax, DGROUP
	mov     ds,ax
    assume	ds:DGROUP
	mov     ss,ax
	mov     sp, offset DGROUP:exe_stacktop
    call	TestCPU
    jc		@@exit
if ?LOADSUPP
	push	es
    call	precheck
    pop		es
    pushf
endif
	push  es		    ; startup_exe(commandline);
	push  80h
	call  _startup_exe
    pop cx
    pop cx

if ?LOADSUPP
	popf
    adc		al,al
    jnz		@@exit
    cmp		_bLoad,1
    jnz		@@exit
    push cs
    push offset @@return_in_v86
	push ds
	push di
	mov word ptr cs:[driverstk+0],sp
	mov word ptr cs:[driverstk+2],ss
    jmp  Install_It
@@return_in_v86:
    push ax
    call postcheck
    mov  ah,51h
    int  21h
    mov  es,bx
    mov  es,es:[002Ch]
    mov  ah,49h
    int  21h
    mov  bx,0
@@nextfile:    
    mov	 ah,3Eh
    int  21h
    inc  bx
    cmp  bx,5
    jb   @@nextfile 
    pop  dx
    shr  dx, 4
    add  dx, 10h
    mov	 ax,3100h
    int  21h
endif

@@exit:    
	mov     ah,04ch         ; that was all
	int 	21h
GO_EXE endp

;struc	request_hdr
;  req_size	db	?		; number of bytes stored
;  unit_id	db	?		; unit ID code
;  cmd		db	?		; command code
;  status	dw	?		; status word
;  rsvd		db	8 dup (?)	; reserved
;ends	request_hdr

;struc	init_strc
;  init_hdr	db	size request_hdr dup (?)
;  units		db	?	; number of supported units
;  end_addr	dd	?		; end address of resident part
;  cmd_line	dd	?		; address of command line
;ends	init_strc

driverstk dd 0
driverret dw 0

go_driver PROC

	mov cs:[driverret],0
	push ds					; save DS:DI
	push di
	mov word ptr cs:[driverstk+0],sp
	mov word ptr cs:[driverstk+2],ss

	les     dx, ds:[di+18]	; fetch driver commandline

	mov     ax, DGROUP
	mov     ds, ax
    assume	DS:DGROUP
	mov     ss, ax
	mov     sp, offset DGROUP:exe_stacktop
	push	es               ; startup_driver(char far *cmdLine)
    push	dx
    
    call	precheck
    jc		@@error1
	call _startup_driver
	add sp,4
				;some bugs ??
	or ax,ax
	jnz fail_driver

if ?MASM    
Install_It::
else
Install_It:
endif

if 0; ?A20PORTS			;this info is unreliable, dont use
    mov		ax,2403h	;get A20 gate support
    int		15h
    cmp		ah,00		;ignore carry flag, it is not reliable
    jnz		@@noi1524
    mov		wBIOSA20,bx	
@@noi1524:    
endif
if 0
	mov		ah,5		;enable A20 local
    call 	dword ptr ds:[_XMSdriverAddress]
    and		ax, ax
    jz		@@ERROR2
endif    
;
; Adjust GDT descriptors for jump to protected-mode
;

	MOV     AX, V86SEG
    MOV		DS, AX
    assume	DS:V86SEG
	MOVZX   EAX,AX
	SHL     EAX,4
    mov		edx, eax
    add		edx, offset V86SEG:GDT
	MOV     WORD PTR [GDT+V86_CODE_SEL+2],AX
	MOV     WORD PTR [GDT+V86_DATA_SEL+2],AX
	SHR     EAX,16
	MOV     BYTE PTR [GDT+V86_CODE_SEL+4],AL
	MOV     BYTE PTR [GDT+V86_DATA_SEL+4],AL
	mov     DWORD PTR [GDT_PTR+2],EDX; set linear address of GDT
    
    MOV		AX, DGROUP
	MOVZX   EAX,AX
	SHL     EAX,4
	MOV     DX,_TEXT16
	MOVZX   EDX,DX
	SHL     EDX,4
    
	MOV     WORD PTR [GDT+RSEG_SEL+2],DX
;	MOV     WORD PTR [GDT+REAL_CODE_SEL+2],DX
	MOV     WORD PTR [GDT+REAL_DATA_SEL+2],AX
	SHR     EDX,16
	SHR     EAX,16
	MOV     BYTE PTR [GDT+RSEG_SEL+4],DL
;	MOV     BYTE PTR [GDT+REAL_CODE_SEL+4],DL
	MOV     BYTE PTR [GDT+REAL_DATA_SEL+4],AL

	call    IS486
    mov     [bIs486], al
    mov		[dwFeatures], edx
    and		al,al
    jz		@@is486
    cmp		[_NoInvlPg],-1
    jnz		@@is486
    mov		[_NoInvlPg],0
@@is486:

	CLI
	LGDT    FWORD PTR [GDT_PTR]	; Initialise GDTR
    pushf                       ; clear the NT flag!
    movzx	esp,sp              ; MS-DOS tends to set this flag in real-mode
    and		byte ptr [esp+1],not 40h  ;the next IRET in protected-mode
    popf                        ; will then usually reboot!
	MOV     EAX,CR0			    ; Set the Protected-Mode-Enable-Bit in
	OR      AL,1			    ; CR0
	MOV     CR0,EAX			    ; In Protected-Mode !
    mov		eax, offset _TEXT32:GO_PROTECTED
    push	V86_CODE_SEL
    push	ax
    retf

INTMOD struc
bInt	db ?
wNew	dw ?
wOld	dw ?
INTMOD ends

intvecs label INTMOD
	INTMOD <15h,offset NEW15, offset OLDINT15>
if ?DMA    
	INTMOD <13h,offset NEW13, offset OLDINT13>
	INTMOD <40h,offset NEW40, offset OLDINT40>
endif
	INTMOD <67h,offset BP67, -1>
	INTMOD <06h,offset NEW06, -1>
    db -1

;
; In case the switch was succesfull, the remaining commands are
; already executed in virtual 8086-mode

if ?MASM
_KEEP::
else
_KEEP:
endif

;--- the virtual monitor exits with
;--- SS = DGROUP, SP = stacktop-20h
;--- DS = _TEXT16
;--- EBP = physical address of start of EMS/VCPI memory

	assume	ss:DGROUP
    assume	DS:_TEXT16

	mov		si,offset _TEXT16:intvecs
@@nextint:
    mov		al,[si].INTMOD.bInt
    cmp		al,-1
    jz		@@intmoddone
    mov		di,[si].INTMOD.wOld
    cmp		di,-1
    jz		@@nooldsave
    mov		ah,35h
    int		21h
    mov		[di+0],bx
    mov		[di+2],es
@@nooldsave:
	mov		ah,25h
    mov		dx,[si].INTMOD.wNew
    int		21h
    add		si,size INTMOD
    jmp		@@nextint
@@intmoddone:

if ?VDS
    cmp		_NoVDS, 0
    jnz		@@novds
    mov		ax,354Bh
    int		21h
    mov		ax,es
    or		ax,bx
    jnz		@@int4bnotzero
    push	ds
    pop		es
    mov		bx,offset RSEG:iret_nop	;make sure we dont jump to 0000:0000
@@int4bnotzero:
    mov		WORD PTR [OLDINT4B+0],bx
    mov		WORD PTR [OLDINT4B+2],es
    mov		dx,offset NEW4B
    mov		ax,254Bh
    int		21h
    push	40h
    pop		es
    assume	es:BEGDATA	;tell MASM that ES has a 16-bit segment
    or		byte ptr es:[07Bh],20h
@@novds:
endif

;-- set new interrupt routine offset in the driver header, so any further
;-- access to device EMMXXXX0 is handled by the monitor

    mov 	[pIntOfs],offset BPDRV
    mov 	[pStratOfs],offset strategy_new

    sti

	push	ss
    pop		ds
	assume	DS:DGROUP
    push	ebp 				; FIRSTPAGE value
	call  	_finishing_touches	; do some postprocessing work

	push	ax					; ax == 1 if UMBs are to be installed
    call	_InstallXMSHandler


; local disable A20 again (since we enabled it in TheRealMain)

	push	6					; disable A20
	call	_xmscall

	mov cs:[driverret], OFFSET RSEG:RSEG_END	;set end offset resident part
    
    cmp _startup_verbose,0
    jz  @@nomsg
    mov dx, offset DGROUP:dBye
driver_exit:
    mov ah, 9
    int 21h
@@nomsg:    
	lss sp, cs:[driverstk]
    pop di
    pop ds
	mov ax,cs:[driverret]
	ret

@@ERROR1:
	mov 	dx, OFFSET DGROUP:dAlreadyInstalled
	jmp 	driver_exit
fail_driver:
	MOV     DX, OFFSET DGROUP:dFailed
	jmp driver_exit

	assume es:nothing
    assume ss:nothing

go_driver ENDP

	assume ds:DGROUP

_ISPROTECTEDMODE proc near
	SMSW    AX
	AND    AX,0001H             ; PE-Bit (Protect Enable) set ?
	ret
_ISPROTECTEDMODE ENDP

;--- this will eventually make segment A000-AFFF (and B000-B7FF)
;--- part of conventional DOS memory if option RAM=A000-AFFF is given

_AddIfContiguousWithDosMem proc 
	push bp
    mov bp,sp
    push si
    push es
if 0    
    mov ah, 52h
    int 21h
	mov es, es:[bx-2]
@@nextblock:
    mov al, es:[0]
    cmp al,'Z'
    jz @@endfound
    cmp al,'M'
    jnz @@error
    mov ax, es:[3]
    mov cx, es
    add ax, cx
    inc ax
    mov es, ax
    jmp @@nextblock
@@endfound:
	mov ax, es:[3]
    inc ax
    mov cx, es	;save last Z block in cx
    add ax, cx
    mov es, ax
    cmp word ptr es:[1],8
    jnz @@error
    cmp word ptr es:[8],"CS"
    jnz @@error
    inc ax
    mov si, [bp+4]
    cmp ax, si
    jnz @@error

	mov si, [bp+6]
    
;--- ok, found and contiguous
    
@@error:    
endif
    xor ax, ax
    pop es
    pop si
    pop bp
    ret
_AddIfContiguousWithDosMem endp    

_fmemcpy proc
	push bp
    mov bp,sp
    push ds
    push si
    push di
    les di,[bp+4]
    lds si,[bp+8]
    mov cx,[bp+12]
    rep movsb
    pop di
    pop si
    pop ds
    pop bp
    ret
_fmemcpy endp

_fmemset proc
	push bp
    mov bp,sp
	push di
    les di,[bp+4]
    mov ax,[bp+8]
    mov cx,[bp+10]
    rep stosb
    pop di
    pop bp
	ret
_fmemset endp

;--- convert long to string
;--- ltob(long n, char * s, int base);


_ltob	PROC

n	equ <bp+4>
s	equ <bp+8>
base equ <bp+10>

	push	bp
    mov		bp,sp
	push	edi
	push	si
;	n = 4
;	s = 8
;	base = 10
;	u = -4
;	register si = p
	mov ch,0
	movzx edi,WORD PTR [base]
	mov	eax,[n]
	cmp	di,-10
	jne	@@ispositive
    and eax,eax
    jns @@ispositive
	mov	di,10
    neg eax
    mov ch,'-'
@@ispositive:
	mov	bx,[s]
	lea	si,[bx+10]
	mov	BYTE PTR [si],0
	dec	si
@@nextdigit:
	xor edx, edx
    div edi
	add dl,'0'
    cmp dl,'9'
    jbe @@isdigit
    add dl,7+20h
@@isdigit:    
	mov	[si],dl
	dec	si
	cmp	eax,0
	jne	@@nextdigit
    cmp ch,0
	je	@@nosign
	mov	[si],ch
	dec	si
@@nosign:
	inc si
	mov	ax,si
	pop	si
	pop	edi
	pop bp
	ret	

n	equ <>
s	equ <>
base equ <>

_ltob	ENDP


;--- TestForSystemRAM(void *, int, int *);

_TestForSystemRAM proc
	push bp
    mov bp,sp
    push 0
    push si
    push di
    xor di, di
    mov si, [bp+4]
    mov dx, [bp+6]
    add si, dx
    mov cx, 100h
    sub cx, dx
    jbe @@done
@@nextpage:
	lodsb
    cmp al,'U'
    jz @@testitem
    cmp al,'I'			;'I' is also tested, but not modified
    jnz @@skipitem		;so a warning can be displayed 
@@testitem:    
    mov ax, dx
    shl ax, 8
    mov es, ax
    cli
    mov ax, es:[0]
    mov bx, ax
    xor ax, 055AAh
    mov es:[0], ax
    cmp ax, es:[0]
    jnz @@noram
    xor ax, 0AA55h
    mov es:[0], ax
    cmp ax, es:[0]
    jnz @@noram
    
    cmp byte ptr [si-1], 'U'
    jnz @@nochange
	mov byte ptr [si-1], 'R'    
@@nochange:
	and di,di
    jnz  @@notstart
    mov di, es
@@notstart:    
    add word ptr [bp-2], 100h	;in paragraphs
    jmp @@shared
    
@@noram:
    and di, di
    jz @@shared	;skip test now, found a region
    mov cx,1
@@shared:
	cmp bx, es:[0]
    jz  @@unchanged
	mov es:[0],bx
@@unchanged:    
    sti
@@skipitem:
	inc dx
	loop @@nextpage
@@done:    
    mov bx, [bp+8]
    mov cx, [bp-2]
    mov [bx],cx
    mov ax, di
    pop di
    pop si
    mov sp,bp
    pop bp
	ret 
_TestForSystemRAM endp

IsJemmInstalled proc
	mov bx, -1
	mov dx, offset sig1
    mov ax, 3D00h
    int 21h
    jnc @@found
    mov dx, offset sig2
    mov ax, 3D00h
    int 21h
    jc @@nojemm
@@found:
    mov bx,ax
    xor ax,ax
    push ax
    push ax
    push ax
    mov cx,6
    mov dx, sp
    mov ax,4402h	;read ioctl
    int 21h
    pop ax
    pop cx
    pop cx
    jc  @@nojemm
    cmp ax,0028h	;this is JEMM386!
    jnz @@nojemm
    mov ax,bx
    clc
	ret
@@nojemm:
	cmp bx,-1
    jz  @@noclose
    mov ah,3Eh
    int 21h
@@noclose:    
	stc
    ret
IsJemmInstalled endp


	assume DS:DGROUP

buff equ <bp-34>

_EmmStatus proc
    push di
    push si
	push bp
    mov bp,sp
    push -1
    sub sp,32
	call IsJemmInstalled
    jc  @@nojemm
    mov [bp-2],ax
    mov bx, ax
    lea dx, [buff]	;get version
	mov byte ptr [buff],2
    mov cx,2
    mov ax,4402h	;read ioctl
    int 21h
    jc  @@nojemm
    movzx ax, byte ptr [buff+0]
    movzx cx, byte ptr [buff+1]
    push cx
    push ax
    push offset szDispVer
    call _printf
    add sp,3*2

;--- get EMS, VCPI, UMB, VME, A20 infos

    lea dx, [buff]
	mov byte ptr [buff],6
    mov cx,24
	mov bx, [bp-2]
    mov ax,4402h	;read ioctl
    int 21h
    jc  @@close

    mov di, ax

    mov dx, offset szOff
    lea si, [buff]
    cmp [si].EMX06.e_NoEMS, 0
	jnz @@emsflag
    mov dx, offset szOn
@@emsflag:    
    push dx
    push offset szEMSStatus
    call _printf 
    pop cx
    pop cx

	cmp [si].EMX06.e_NoEMS,0
    jnz @@nodispframe			;dont display FRAME status if no EMS

    mov ah,42h
    int 67h
    mov ax, dx
    sub ax, bx
	push dx
    push ax
    push offset szEMSMemory
    call _printf
    add sp,3*2

	mov ax, offset szFrameNone
    mov cx, [si].EMX06.e_Frame
	jcxz @@noframe
	mov ax, offset szFrameYes
@@noframe:
    push cx
    push ax
    call _printf
    pop cx
    pop cx

@@nodispframe:
	push offset szDotLF
    call _printf
    pop cx

    cmp [si].EMX06.e_NoVCPI, 0            	;_NoVCPI flag
	jnz @@vcpioff
    push [si].EMX06.e_VCPITotal
    push [si].EMX06.e_VCPIUsed
    push offset szVCPIOn
    call _printf
    add sp,2+4+4
    jmp @@vcpidone
    
@@vcpioff:
	push offset szVCPIOff
    call _printf
    pop cx
@@vcpidone:

	mov ax, 64		;default DMA buffer size
    cmp di, 16		;could the DMA buffer size be read?
    jb  @@nodmasize
    mov ax, [si].EMX06.e_DMASize
@@nodmasize:    
    mov ecx, [si].EMX06.e_DMABuff
    push ax
    push ecx
    push offset szDMABuffer
    call _printf
    add sp,2+4+2

if ?A20PORTS or ?A20XMS
	mov ax, offset szOff
    cmp [si].EMX06.e_NoA20, 0
	jnz @@a20flag
	mov ax, offset szOn
@@a20flag:
    push ax
    push offset szA20Status
    call _printf
    pop cx
    pop cx
endif

if ?VME
	mov ax, offset szOff
    cmp [si].EMX06.e_NoVME, 0
	jnz @@vmeflag
	mov ax, offset szOn
@@vmeflag:
    push ax
    push offset szVMEStatus
    call _printf
    pop cx
    pop cx
endif

if ?PGE
	mov ax, offset szOff
    cmp [si].EMX06.e_NoPGE, 0
	jnz @@pgeflag
	mov ax, offset szOn
@@pgeflag:
    push ax
    push offset szPGEStatus
    call _printf
    pop cx
    pop cx
endif

    lea dx, [buff]
	mov byte ptr [buff],7
    mov cx,8*4
	mov bx, [bp-2]
    mov ax,4402h	;read ioctl
    int 21h
    jc  @@close

	lea si,[buff]
    mov cx, 8
@@nextumb:
    mov ax, [si+0]
    and ax, ax
	jz @@umbdone    
    push cx
    mov dx, [si+2]
    and dh, 7Fh ;reset highest flag
    add dx, ax
    dec dx
    push dx
    push ax
    push offset szUMB
    call _printf
    add sp, 3*2
    pop cx
    add si, 4
    loop @@nextumb
@@umbdone:
@@noumbs:

@@close:
	mov bx, [bp-2]
	cmp bx, -1
    jz  @@exit
    mov ah, 3Eh
    int 21h
@@exit:
	mov	sp,bp
    pop bp
	pop si
	pop di
    ret
@@nojemm:
	MOV     DX,OFFSET dError1
	MOV     AH,9
	INT     21H
	jmp		@@close
_EmmStatus endp

buff equ <bp-16>

_EmmUpdate proc
    push di
    push bp
	mov bp, sp
    sub sp,16
    xor di,di
    call IsJemmInstalled
    jnc @@jemmok
	MOV     DX,OFFSET dError1
	MOV     AH,9
	INT     21H
    jmp		@@exit
@@jemmok:    
    mov bx, ax
    mov byte ptr [buff+0],15
    mov ax,V86SEG
    mov es,ax
    assume es:V86SEG
?IOCTLBUFFSIZ = 4
if ?VME    
    mov al, _NoVME
    mov [buff+1],al
endif    
if ?A20PORTS or ?A20XMS
    mov al, [_NoA20]
    mov [buff+2],al
endif    
    mov al, [_NoVCPI]
    mov [buff+3],al
if ?PGE
    mov al, [_NoPGE]
    mov [buff+4],al
?IOCTLBUFFSIZ = ?IOCTLBUFFSIZ + 1
endif    
    assume es:nothing

    mov cx,?IOCTLBUFFSIZ
    lea dx,[buff]
    mov ax,4403h	;write ioctl
    int 21h
    jc @@noioctlwrite
    inc di
@@noioctlwrite:    
    mov ah,3Eh
    int 21h
@@exit:
	mov ax,di
    mov sp,bp
    pop bp
    pop di
    ret
_EmmUpdate endp

_InstallXMSHandler proc
	push bp
    mov bp,sp
ife ?A20XMS      		;if there is no XMS A20 trapping
    mov ax, [bp+4]		;XMS hook is needed *only* for UMBs.
    and ax, ax			;then dont install if no UMBs are supplied
    jz @@umbdone
endif    
    mov ax,4300h
    int 2Fh
    cmp al,80h
    jnz @@noxms
    mov ax,4310h
    int 2Fh
    push es
    push bx
    mov dx, -1
    mov ah, 10h
    call dword ptr [bp-4]
    and ax, ax
    jnz @@umbalreadythere
    les bx,[bp-4]
@@nexttest:
    cmp byte ptr es:[bx],0EBh
    jz  @@endofchain
    les bx,es:[bx+1]
    jmp @@nexttest
@@endofchain:
    mov byte ptr es:[bx+0],0EAh
    mov word ptr es:[bx+1],offset XMSHandler
    mov ax, seg XMSHandler
    mov es:[bx+3], ax
    add bx,5
    push ds
    mov ax, RSEG
    mov ds, ax
    assume DS:RSEG
    mov word ptr [XMSoldhandler+0],bx
    mov word ptr [XMSoldhandler+2],es
if ?A20XMS
    mov ax, [bp+4]
    and ax, ax
    jnz @@xmswithumb
    mov byte ptr ds:[XMSUMB], 0EAh
    mov word ptr ds:[XMSUMB+1], bx
    mov word ptr ds:[XMSUMB+3], es
@@xmswithumb:
endif
    pop ds
    assume DS:DGROUP
    jmp @@umbdone
@@umbalreadythere:
	push offset DGROUP:szUMBErr1
    call _printf
    pop cx
@@noxms:
@@umbdone:

	mov si, offset DGROUP:_UMBsegments
if 1    
	mov cx,8
    push si
@@nextumb:    
    mov ax, [si+0]
    and ax, ax
    jz @@umbcleared
    mov es, ax
    xor di, di
    push cx
    movzx ecx, word ptr [si+2]
    shl ecx, 2			;para -> dword
    xor eax, eax
    rep stosd
    pop cx
    add si, 4
    loop @@nextumb
@@umbcleared:
	pop si
endif    
	mov ax, 449Fh		;request end of initialization phase
    xor dx, dx
    int 67h
    
	mov sp,bp
    pop bp
	ret
_InstallXMSHandler endp

_PUT_CONSOLE proc

	pop cx
    pop dx
    push dx
    push cx
    cmp dl,10
    jnz @@isnotlf
    push dx
    mov dl,13
    mov ah,2
    int 21h
    pop dx
@@isnotlf:    
    mov ah,2
	int 21h
    ret
    
_PUT_CONSOLE endp

;--- C compiler helper procs. Avoids to need the C runtime libs

divprocs proc

ife (?WCC + ?MSC + ?DMC)
?TCC equ 1
else
?TCC equ 0
endif

if ?TCC

	public	LDIV@			; the procs which were in LIBM.LIB
    public	LUDIV@			; are now implemented here
	public	LMOD@
    public	LUMOD@
	public	LXLSH@
	public	LXRSH@
	public	LXURSH@
	public	LXMUL@
if ?NEARC    
    public	N_LUDIV@
	public	N_LXLSH@
	public	N_LXRSH@
	public	N_LXURSH@
endif    

;--- [bp+6] mod|div [bp+10]
;--- clears stack!


if ?MASM
LMOD@::
LUMOD@::
else
LMOD@:
LUMOD@:
endif
		mov bl,1
        jmp @@common
if ?MASM
LDIV@::
LUDIV@::
else
LDIV@:
LUDIV@:
endif
		mov bl,0
@@common:        
    	push BP
		mov	BP,SP
        mov eax, [bp+6]
        mov ecx, [bp+10]
        cdq
        div ecx
        test bl,1
        jz @@isdiv
        mov eax, edx
@@isdiv:        
        push eax
        pop ax
        pop dx
        pop bp
        retf 8
if ?NEARC        
if ?MASM
N_LDIV@::
N_LUDIV@::
else
N_LDIV@:
N_LUDIV@:
endif
		mov bl,0
    	push BP
		mov	BP,SP
        mov eax, [bp+6]
        mov ecx, [bp+10]
        cdq
        div ecx
        test bl,1
        jz @@n_isdiv
        mov eax, edx
@@n_isdiv:        
        push eax
        pop ax
        pop dx
        pop bp
        ret 8
endif

;--- shl DX:AX CL bits

if ?MASM
LXLSH@::
else
LXLSH@:
endif
		push dx
        push ax
        pop eax
        shl eax, cl
        push eax
        pop ax
        pop dx
        retf

if ?NEARC
if ?MASM
N_LXLSH@::
else
N_LXLSH@:
endif
		push dx
        push ax
        pop eax
        shl eax, cl
        push eax
        pop ax
        pop dx
        ret
endif

;--- shr DX:AX CL bits

if ?MASM
LXURSH@::
LXRSH@::
else
LXURSH@:
LXRSH@:
endif
		push dx
        push ax
        pop eax
        shr eax, cl
        push eax
        pop ax
        pop dx
		retf
if ?NEARC
if ?MASM
N_LXURSH@::
N_LXRSH@::
else
N_LXURSH@:
N_LXRSH@:
endif
		push dx
        push ax
        pop eax
        shr eax, cl
        push eax
        pop ax
        pop dx
		ret
endif

;--- mul DX:AX with CX:BX, result in DX:AX

if ?MASM
LXMUL@::
else
LXMUL@:
endif
		push dx
		push ax
        pop eax
        push cx
        push bx
        pop ecx
        mul ecx
        push eax
        pop ax
        pop dx
		retf
endif

if ?WCC        
        public __U4D	;used by WCC (dx:ax / cx:bx = dx:ax, remainder in cx:bx)
        public __I4M	;used by WCC (dx:ax * cx:bx = dx:ax)
if ?MASM
__U4D::
else
__U4D:
endif
		push dx
		push ax
        pop eax
        push cx
        push bx
        pop ecx
        cdq
        div ecx
        push edx	;remainder into CX:BX
        pop bx
        pop cx
        push eax
        pop ax
        pop dx
		ret
if ?MASM
__I4M::
else
__I4M:
endif
		push dx
		push ax
        pop eax
        push cx
        push bx
        pop ecx
        cdq
        mul ecx
        push eax
        pop ax
        pop dx
		ret
endif

if ?MSC

;--- some publics for MS C 1.5

		public __aNulrem
		public __aNuldiv
		public __aNulshr
		public __aNlmul
		public __aNlshl

if ?MASM
__aNulrem::
else
__aNulrem:
endif
		pop dx
        pop eax
        pop ecx
        push dx
        cdq
        div ecx
        push edx
        pop ax
        pop dx
		ret
if ?MASM        
__aNuldiv::
else
__aNuldiv:
endif
		pop dx
        pop eax
        pop ecx
        push dx
        cdq
        div ecx
        push eax
        pop ax
        pop dx
		ret
if ?MASM        
__aNlmul::
else
__aNlmul:
endif
		pop dx
        pop eax
        pop ecx
        push dx
        mul ecx
        push eax
        pop ax
        pop dx
		ret
if ?MASM
__aNulshr::
else
__aNulshr:
endif
		push dx
        push ax
        pop eax
        shr eax, cl
        push eax
		pop ax
        pop dx
		ret
if ?MASM        
__aNlshl::
else
__aNlshl:
endif
		push dx
        push ax
        pop eax
        shl eax, cl
        push eax
		pop ax
        pop dx
		ret

endif

if ?DMC

;--- some publics for DMC

		public __ULREM@
		public __ULDIV@
		public __ULSHR@
		public __aNlmul
		public __aNlshl

if ?MASM
__ULREM@::
else
__ULREM@:
endif
		pop dx
        pop eax
        pop ecx
        push dx
        cdq
        div ecx
        push edx
        pop ax
        pop dx
		ret
if ?MASM        
__ULDIV@::
else
__ULDIV@:
endif
		pop dx
        pop eax
        pop ecx
        push dx
        cdq
        div ecx
        push eax
        pop ax
        pop dx
		ret
if ?MASM        
__aNlmul::
else
__aNlmul:
endif
		pop dx
        pop eax
        pop ecx
        push dx
        mul ecx
        push eax
        pop ax
        pop dx
		ret
if ?MASM
__ULSHR@::
else
__ULSHR@:
endif
		push dx
        push ax
        pop eax
        shr eax, cl
        push eax
		pop ax
        pop dx
		ret
if ?MASM        
__aNlshl::
else
__aNlshl:
endif
		push dx
        push ax
        pop eax
        shl eax, cl
        push eax
		pop ax
        pop dx
		ret

endif

divprocs endp

STREG16 struc
st_ax	dw ?
st_bx	dw ?
st_cx	dw ?
st_dx	dw ?
STREG16 ends

		public _emmcall
        extrn _emmreg16:near
        
_emmcall proc
		push bp
        mov bp,sp
        push si
        mov si,offset DGROUP:_emmreg16
        mov ax,[si].STREG16.st_ax
        mov ah, [bp+4]
        mov _emmfunction, ah
        mov bx,[si].STREG16.st_bx
        mov cx,[si].STREG16.st_cx
        mov dx,[si].STREG16.st_dx
        int 67h
        mov [si].STREG16.st_ax,ax
        mov [si].STREG16.st_bx,bx
        mov [si].STREG16.st_cx,cx
        mov [si].STREG16.st_dx,dx
        movzx ax,ah
        pop si
        pop bp
		ret
_emmcall endp

		public _xmscall
        extrn _reg16:near
        
_xmscall proc
		push bp
        mov bp,sp
        push si
        mov si,offset DGROUP:_reg16
        mov bx,[si].STREG16.st_bx
        mov dx,[si].STREG16.st_dx
        mov ah, [bp+4]
        call dword ptr ds:[_XMSdriverAddress]
        mov [si].STREG16.st_ax,ax
        mov [si].STREG16.st_bx,bx
        mov [si].STREG16.st_dx,dx
        pop si
        pop bp
		ret
_xmscall endp

STREG32 struc
st_eax	dd ?
st_ebx	dd ?
st_ecx	dd ?
st_edx	dd ?
STREG32 ends

		public _xmscall32
        extrn _reg32:near

_xmscall32 proc
		push bp
        mov bp,sp
        push si
        mov si,offset DGROUP:_reg32
        mov ebx,[si].STREG32.st_ebx
        mov edx,[si].STREG32.st_edx
        mov ah, [bp+4]
        call dword ptr ds:[_XMSdriverAddress]
        mov [si].STREG32.st_eax,eax
        mov [si].STREG32.st_ebx,ebx
        mov [si].STREG32.st_ecx,ecx
        mov [si].STREG32.st_edx,edx
        pop si
        pop bp
		ret
_xmscall32 endp

		public _xmsinit

_xmsinit proc
	    mov ax, 4300h
		int 2fh
		cmp al, 80h
		jne @@not_detected
		mov ax, 4310h
		int 2fh
		mov word ptr _XMSdriverAddress+0, bx
		mov word ptr _XMSdriverAddress+2, es

		mov ax, 4309h		;  XMS get xms handle table
		int 2fh
		cmp al,43h
		jne @@no_table
		mov word ptr _XMShandleTable+0, bx
		mov word ptr _XMShandleTable+2, es
@@no_table:
		mov ax,1
		ret
@@not_detected:
		xor ax,ax
        ret
_xmsinit endp

		public _VMwareDetect

_VMwareDetect proc
		
		mov eax, 564D5868h	;/* mov eax,564d5856h */
		mov ecx, 0000000ah
		mov ebx,ecx
		mov dx, 5658h
		in eax,dx
		cmp ebx, 564D5868h
		jne @@failed
        mov ax,1
        ret
@@failed:
		xor ax, ax
        ret
_VMwareDetect endp

		public _small_code_	;required by Open Watcom WCC
        
_small_code_ label byte

	assume DS:nothing

_TEXT ENDS


_STACK	segment

	my_stack db 1024 dup(?)       ; stack for the C - things

exe_stacktop label byte

_STACK	ends

		END     GO_EXE
