/* edlib.c -- block manipulation routines for edline

  AUTHOR: Gregory Pietsch <GKP1@flash.net>

  DESCRIPTION:

  This file contains block manipulation routines for edline, an 
  edlin-style line editor.

  COPYRIGHT NOTICE AND DISCLAIMER:

  Copyright (C) 2003 Gregory Pietsch

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.

*/

/* includes */

#include "config.h"
#include <ctype.h>
#include <stdio.h>
#if defined(__STDC__) || defined(STDC_HEADERS) || defined(HAVE_STDLIB_H)
#include <stdlib.h>
#endif
#if defined(__STDC__) || defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include "dynstr.h"
#include "xmalloc.h"

/* typedefs */

/* static variables */

DYNARRAYSTRING_T *buffer = 0;

/* functions */

#ifndef __STDC__
#ifndef HAVE_STRCHR
#ifdef HAVE_INDEX
#define strchr index
#else
/* no strchr(), so roll our own */
#ifndef OPTIMIZED_FOR_SIZE
#include <limits.h>
 /* Nonzero if X is not aligned on an "unsigned long" boundary.  */
#ifdef ALIGN
#define UNALIGNED(X) ((unsigned long)X&(sizeof(unsigned long)-1))
#else
#define UNALIGNED(X) 0
#endif
 /* Null character detection.  */
#if ULONG_MAX == 0xFFFFFFFFUL
#define DETECTNULL(X) (((X)-0x01010101UL)&~(X)&0x80808080UL)
#elif ULONG_MAX == 0xFFFFFFFFFFFFFFFFUL
#define DETECTNULL(X) (((X)-0x0101010101010101UL)&~(X)&0x8080808080808080UL)
#else
#error unsigned long is not 32 or 64 bits wide
#endif
#if UCHAR_MAX != 0xFF
#error char is not 8 bits wide
#endif
 /* Detect whether the character used to fill mask is in X */
#define DETECTCHAR(X,MASK) (DETECTNULL(X^MASK))
#endif                          /* OPTIMIZED_FOR_SIZE */
static char *(strchr) (const char *s, int c) {
    char pc = (char) c;
#ifndef OPTIMIZED_FOR_SIZE
    unsigned long *ps, mask = 0;
    size_t i;

    /* If s is unaligned, punt into the byte search loop.  This should be
       rare.  */
    if (!UNALIGNED(s)) {
        ps = (unsigned long *) s;
        for (i = 0; i < sizeof(unsigned long); i++)
            mask =
                (mask << CHAR_BIT) +
                ((unsigned char) pc & ~(~0 << CHAR_BIT));
        /* Move ps a block at a time. */
        while (!DETECTNULL(*ps) && !DETECTCHAR(*ps, mask))
            ps++;
        /* Pick up any residual with a byte searcher.  */
        s = (char *) ps;
    }
#endif
    /* The normal byte-search loop.  */
    while (*s && *s != pc)
        s++;
    return *s == pc ? (char *) s : 0;
}

#ifndef OPTIMIZED_FOR_SIZE
#undef UNALIGNED
#undef DETECTNULL
#undef DETECTCHAR
#endif                          /* OPTIMIZED_FOR_SIZE */
#endif
#endif
#endif

/* commands */

/* transfer_file - merges the contents of a file on disk with a file in memory
 */
void transfer_file(unsigned long line, char *filename)
{
    DYNSTRING_T *s = DScreate();
    FILE *f;
    int c;
    unsigned long init_line = line;

    if ((f = fopen(filename, "r"))) {
        while ((c = getc(f)) != EOF) {
            if (c == '\n') {
                /* add s to buffer and reset s */
                DASinsert(buffer, line++, &s, 1, 1);
                DSresize(s, 0, '\0');
            } else
                DSappendchar(s, c, 1);
        }
        fclose(f);
    }
    printf("%s: %lu line(s) read\n", filename, line - init_line);
    DSdestroy(s);
}

/* write X number of lines to a file */
void write_file(unsigned long lines, char *filename)
{
    FILE *f;
    unsigned long i;

    if ((f = fopen(filename, "w"))) {
        i = DASlength(buffer);
        if (lines >= i)
            lines = i;
        for (i = 0; i < lines; i++) {
            fputs((const char *) DScstr(DASgetat(buffer, i)), f);
            fputc('\n', f);
        } fclose(f);
        printf("%s: %lu lines written\n", filename, i);
    }
}

/* copy a block of lines elsewhere in the buffer */
void copy_block(unsigned long line1, unsigned long line2,
                unsigned long line3, size_t count)
{
    size_t i, j;
    DYNSTRING_T **s;
    size_t numlines;

    if (line1 < line3 && line3 <= line2)
        puts("Entry error");
    else {
        numlines = line2 - line1 + 1;
        s = XMALLOC(DYNSTRING_T *, numlines);
        for (i = line1, j = 0; i <= line2; i++, j++)
            s[j] = DASgetat(buffer, i);
        for (i = 0; i < count; i++)
            DASinsert(buffer, line3, s, numlines, 1);
        XFREE(s);
    }
}

/* delete a block from line1 to line2 */
void delete_block(unsigned long line1, unsigned long line2)
{
    unsigned long i;

    if (line1 > line2)
        puts("Entry error");
    else {
        for (i = line1; i <= line2; i++)
            DSdestroy(DASgetat(buffer, i));
        DASremove(buffer, line1, line2 - line1 + 1);
    }
}

/* move the block from line1 to line2 to immediately before line3 */
void move_block(unsigned long line1, unsigned long line2,
                unsigned long line3)
{
    size_t i, j;
    DYNSTRING_T **s;
    size_t numlines;

    if (line1 < line3 && line3 <= line2)
        puts("Entry error");
    else {
        numlines = line2 - line1 + 1;
        s = XMALLOC(DYNSTRING_T *, numlines);
        for (i = line1, j = 0; i <= line2; i++, j++)
            s[j] = DASgetat(buffer, i);
        if (line3 >= line2) {
            DASinsert(buffer, line3, s, numlines, 1);
            DASremove(buffer, line1, numlines);
        } else {
            DASremove(buffer, line1, numlines);
            DASinsert(buffer, line3, s, numlines, 1);
        }
        XFREE(s);
    }
}

/* read a line from stdin */
char *read_line(char *prompt)
{
    static DYNSTRING_T *ds = 0;
    int c;

    if (ds == 0)
        ds = DScreate();
    DSresize(ds, 0, 0);
    fputs(prompt, stdout);
    fflush(stdout);
    while ((c = getchar()) != EOF && c != '\n')
        DSappendchar(ds, c, 1);
    return DScstr(ds);
}

/* display a block of text */
void display_block(unsigned long first_line, unsigned long last_line,
                   unsigned long current_line, size_t page_size)
{
    unsigned long i;
    size_t lines_written;

    for (i = first_line, lines_written = 0;
         i <= last_line && i < DASlength(buffer); i++) {
        printf("%lu:%c%s\n", i + 1, i == current_line ? '*' : ' ',
               DScstr(DASgetat(buffer, i)));
        lines_written++;
        if (lines_written == page_size && i != last_line) {
            read_line("Press <Enter> to continue");
            lines_written = 0;
        }
    }
}

/* translate_string - translate a string with escapes into regular string */
static DYNSTRING_T *translate_string(char *s, int tc)
{
    static DYNSTRING_T *r = 0;
    static char *escs = "abeftv\\\'\".";
    static char *xlat = "\a\b\033\f\t\v\\\'\".";
    static char *xdigs = "0123456789abcdef";
    char *p;
    int x;

    if (r == 0)
        r = DScreate();
    DSresize(r, 0, 0);
    while (*s && *s != tc) {
        if (*s == '\\') {
            /* found an escape */
            s++;
            if ((p = strchr(escs, *s)) != 0)
                DSappendchar(r, xlat[p - escs], 1);
            else if (*s == 'x') {
                /* handle hex digits */
                s++;
                x = 0;
                while (isxdigit((unsigned char) *s)) {
                    p = strchr(xlat, tolower((unsigned char) *s));
                    x <<= 4;
                    x |= (p - xlat);
                    s++;
                } s--;
                DSappendchar(r, (x & 255), 1);
            } else if (*s >= '0' && *s <= '7') {
                /* handle octal digits */
                x = 0;
                while (isdigit((unsigned char) *s)) {
                    x <<= 3;
                    x |= (*s - '0');
                    s++;
                }
                s--;
                DSappendchar(r, (x & 255), 1);
            } else if (*s == '^') {
                /* control character */
                s++;
                x = toupper((unsigned char) *s) ^ 64;
                DSappendchar(r, x, 1);
            }
        } else
            DSappendchar(r, *s, 1);
        s++;
    }
    return r;
}

/* modify_line - modify a line in the buffer */
void modify_line(unsigned long line)
{
    char *new_line;
    DYNSTRING_T *xline;

    display_block(line, line, line, 1);
    printf("%lu: ", line + 1);
    new_line = read_line("");
    xline = translate_string(new_line, 0);
    DASputat(buffer, (size_t) (line), xline);
}

/* insert_block - go into insert mode */
unsigned long insert_block(unsigned long line)
{
    char *new_line;
    DYNSTRING_T *xline;

    while (strcmp((new_line = read_line(" : ")), ".") != 0) {
        xline = translate_string(new_line, 0);
        DASinsert(buffer, line++, &xline, 1, 1);
    }
    return line + 1 < DASlength(buffer) ? line + 1 : DASlength(buffer);
}

/* search_buffer - search a buffer for a string */
unsigned long search_buffer(unsigned long current_line,
                            unsigned long line1, unsigned long line2,
                            int verify, char *s)
{
    unsigned long line;
    DYNSTRING_T *ds;
    int q = 0;
    char *yn;

    while (isspace((unsigned char) *s))
        s++;
    if (*s == '\'' || *s == '\"')
        q = *s++;
    ds = translate_string(s, q);
    if (DSlength(ds) != 0)
        for (line = line1; line <= line2; line++) {
            if (DSfind(DASgetat(buffer, (size_t) line), DScstr(ds), 0,
                       DSlength(ds)) != NPOS) {
                display_block(line, line, line, 1);
                if (verify) {
                    yn = read_line("O.K.? ");
                    if (*yn == 0 || *yn == 'y' || *yn == 'Y')
                        return line + 1;
                } else
                    return line + 1;
            }
        }
    puts("Not found");
    return current_line;
}

/* search_buffer - search a buffer for a string */
unsigned long replace_buffer(unsigned long current_line,
                             unsigned long line1, unsigned long line2,
                             int verify, char *s)
{
    unsigned long line;
    DYNSTRING_T *ds, *ds1, *dc;
    int q = 0;
    char *yn;
    size_t origpos;

    while (isspace((unsigned char) *s))
        s++;
    if (*s == '\'' || *s == '\"')
        q = *s++;
    else
        q = ',';
    ds = DScopy(translate_string(s, q));
    /* pick off second string */
    while (*s != q && *s)
        s += (*s == '\\' ? 2 : 1);
    while (*s != ',')
        s++;
    s++;
    while (isspace((unsigned char) *s))
        s++;
    if (*s == '\'' || *s == '\"')
        q = *s++;
    else
        q = 0;
    ds1 = translate_string(s, q);
    if (DSlength(ds) != 0 && strcmp(DScstr(ds), DScstr(ds1)) != 0)
        for (line = line1; line <= line2; line++) {
            origpos = 0;
            while ((origpos = DSfind(DASgetat(buffer, (size_t) line),
                                     DScstr(ds), origpos, DSlength(ds)))
                   != NPOS) {
                dc = DScopy(DASgetat(buffer, line));
                DSreplace(dc, origpos, DSlength(ds), DScstr(ds1),
                          DSlength(ds1));
                printf("%lu: %s\n", line + 1, DScstr(dc));
                if (verify)
                    yn = read_line("O.K.? ");
                if (!verify || (*yn == 0 || *yn == 'y' || *yn == 'Y')) {
                    current_line = line + 1;
                    origpos += DSlength(ds1);
                    DASputat(buffer, line, dc);
                } else
                    origpos++;
                DSdestroy(dc);
            }
        }
    DSdestroy(ds);
    return current_line;
}

/* get the last line in the buffer */
unsigned long get_last_line(void)
{
    return DASlength(buffer);
}
/* initialize the buffer */ void create_buffer(void)
{
    buffer = DAScreate();
}
/* destroy the buffer */ void destroy_buffer(void)
{
    unsigned long last = get_last_line(), i;

    for (i = 0; i < last; i++)
        DSdestroy(DASgetat(buffer, i));
    DASdestroy(buffer);
    buffer = 0;
}

/* END OF FILE */
