/* dynstr.c -- dynamic strings & arrays of strings

  AUTHOR: Gregory Pietsch <GKP1@flash.net>

  DESCRIPTION:

  This file is a dynamic string library for edline, an edlin-style line editor.

  COPYRIGHT NOTICE AND DISCLAIMER:

  Copyright (C) 2003 Gregory Pietsch

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.

*/

/* includes */

#include "config.h"
#include <stdio.h>
#if defined(__STDC__) || defined(STDC_HEADERS) || defined(HAVE_STDLIB_H)
#include <stdlib.h>
#endif
#if defined(__STDC__) || defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include "dynstr.h"
#include "xmalloc.h"
#include "error.h"

/* defines */
#define DS_MINSIZE 31

/* DYNSTRING_T functions */

#if !defined(__STDC__) && !defined(STDC_HEADERS)
#ifndef HAVE_MEMCHR
/* no memchr(), so roll our own */
static void *(memchr)(const void *s, int c, size_t n)
{
	const unsigned char uc = c;
	const unsigned char *su;

	for (su = s; 0 < n; ++su, --n)
		if (*su == uc)
			return (void *)su;
	return 0;
}
#endif
#ifndef HAVE_MEMCMP
#define memcmp bcmp
#endif
#ifndef HAVE_MEMCPY
#define memcpy(x,y,z) bcopy(y,x,z)
#endif
#ifndef HAVE_MEMMOVE
#define memmove(x,y,z) bcopy(y,x,z)
#endif
#ifndef HAVE_MEMSET
/* no memset(), so roll our own */
static void *(memset)(void *s, int c, size_t n)
{
	const unsigned char uc = c;
	unsigned char *su;

	for (su = s; 0 < n; ++su, --n)
		*su = uc;
	return s;
}
#endif
#ifndef HAVE_BCMP
/* no bcmp(), so roll our own */
static int (bcmp)(const void *s1, const void *s2, size_t n)
{
	const unsigned char *su1, *su2;

	for (su1 = s1, su2 = s2; 0 < n; ++su1, ++su2, --n)
		if (*su1 != *su2)
			return ((*su1 < *su2) ? -1 : 1);
	return 0;
}
#endif
#ifndef HAVE_BCOPY
/* no bcopy(), so roll our own */
static void *(bcopy)(const void *s2, void *s1, size_t n)
{
	char *sc1 = s1;
	const char *sc2 = s2;

	if (sc2 < sc1 && sc1 < sc2 + n)
		for (sc1 += n, sc2 += n; 0 < n; --n)
			*--sc1 = *--sc2;
	else
		for ( ; 0 < n; --n)
			*sc1++ = *sc2++;
	return s1;
}
#endif
#endif

/* DStidy - destroy any allocated string storage */
static void DStidy(DYNSTRING_T *this, int constructed)
{
	if (constructed && this->ptr)
		free(this->ptr);
	this->ptr = 0;
	this->len = 0;
	this->res = 0;
}

/* DSgrow - make the DS grow */
static int DSgrow(DYNSTRING_T *this, size_t n, int trim)
{
	size_t osize = this->ptr == 0 ? 0 : this->res;
    size_t size;
	char *s;

	if (n == 0) {
		if (trim && DS_MINSIZE < osize)
			DStidy(this, 1);
		else if (this->ptr)
			this->ptr[this->len = 0] = '\0';
		return 0;
	} else if (n == osize || n < osize && !trim)
		return 1;
	else {
		size = this->ptr == 0 && n < this->res ? this->res : n;
		if ((size |= DS_MINSIZE) == NPOS)
			--size;
        if ((s = realloc(this->ptr, size + 1)) == 0
			&& (s = realloc(this->ptr, (size = n) + 1)) == 0) {
			error(ERR_OUT_OF_MEMORY); 
			abort();
		}
		this->ptr = s;
		this->res = size;
		return 1;
	}
}

/* DScreate - create a new DYNSTRING_T */
DYNSTRING_T *DScreate(void)
{
	DYNSTRING_T *ds = XMALLOC(DYNSTRING_T, 1);

	DStidy(ds, 0);
	return ds;
}

/* DSdestroy - return a DYNSTRING_T to the void */
void DSdestroy(DYNSTRING_T *ds)
{
	if (ds) {
		DStidy(ds, 1);
		XFREE(ds);
	}
}

/* DScopy - return a copy of a DYNSTRING_T */
DYNSTRING_T *DScopy(DYNSTRING_T *orig)
{
	DYNSTRING_T *ds = DScreate();

	if (orig) {
		if (orig->ptr) {
			/* make a copy of the string */
			ds->ptr = XMALLOC(char, orig->res);
			strcpy(ds->ptr, orig->ptr);
		} else
			ds->ptr = 0;
		ds->len = orig->len;
		ds->res = orig->res;
	}
	return ds;
}

/* append char c to the end of ds nr times */
DYNSTRING_T *DSappendchar(DYNSTRING_T *ds, int c, size_t nr)
{
    size_t n;
	
	if (NPOS - ds->len <= nr)
		error(ERR_LENGTH); 
    if (0 < nr && DSgrow(ds, n = ds->len + nr, 0)) {
		/* append to make nonempty string */
		memset(ds->ptr + ds->len, c, nr);
		ds->ptr[ds->len = n] = '\0';
	}
	return ds;
}

/* remove a substring */
DYNSTRING_T *DSremove(DYNSTRING_T *ds, size_t p, size_t nr)
{
	size_t n;

	if (ds->len < p)
		error(ERR_STRING_POSITION); /* invalid string position error */
	if (ds->len - p < nr)
		nr = ds->len - p;
	if (0 < nr) {
		/* remove the substring */
		memmove(ds->ptr + p, ds->ptr + p + nr, ds->len - p - nr);
		n = ds->len - nr;
		if (DSgrow(ds, n, 0))
			ds->ptr[ds->len = n] = '\0';
	}
	return ds;
}

/* resize a string */
void DSresize(DYNSTRING_T *ds, size_t n, int c)
{
	n <= ds->len ? DSremove(ds, n, NPOS) : DSappendchar(ds, c, n - ds->len);
}

/* find a substring */
size_t DSfind(DYNSTRING_T *ds, const char *s, size_t p, size_t n)
{
	size_t nmax;
	const char *t, *u;

	if (n == 0 || n == NPOS && (n = strlen(s)) == 0)
		return 0;
	if (p < ds->len && n <= (nmax = ds->len - p)) {
		/* find non-null substring in string */
		for (nmax -= n - 1, u = ds->ptr + p;
			 (t = (const char *)memchr(u, *s, nmax)) != 0;
			 nmax -= t - u + 1, u = t + 1)
			if (memcmp(t, s, n) == 0)
				return (size_t)(t - ds->ptr);
	}
	return NPOS;
}

/* replace a substring with a string */
DYNSTRING_T *DSreplace(DYNSTRING_T *ds, size_t p, size_t n, const char *s,
					   size_t ns)
{
	size_t nm, nn;
	
	if (ds->len < p)
		return ds;
	if (ns == NPOS)
		ns = strlen(s);
	if (NPOS - ns <= ds->len - n)
		error(ERR_STRING_POSITION);
	nm = ds->len - n - p;
	if (ns < n)
		memmove(ds->ptr + p + ns, ds->ptr + p + n, nm);
	if ((0 < ns || 0 < n) && DSgrow(ds, nn = ds->len + ns - n, 0)) {
		/* replace to make nonempty string */
		if (n < ns)
			memmove(ds->ptr + p + ns, ds->ptr + p + n, nm);
		memcpy(ds->ptr + p, s, ns);
		ds->ptr[ds->len = nn] = '\0';
	}
	return ds;
}

/* get a string's char pointer */
char *DScstr(DYNSTRING_T *ds)
{
	return ds->ptr ? ds->ptr : "";
}

/* get a string's length */
size_t DSlength(DYNSTRING_T *ds)
{
	return ds->ptr ? ds->len : 0;
}

/* DYNARRAYSTRING_T functions */

/* DAStidy - destroy any allocated array storage.
   Note that it DOES NOT destroy the strings in the array.
 */
static void DAStidy(DYNARRAYSTRING_T *this, int constructed)
{
	if (constructed && this->ptr) 
		free(this->ptr);
	this->ptr = 0;
	this->len = 0;
	this->res = 0;
}

/* DASgrow - make the DAS grow */
static void DASgrow(DYNARRAYSTRING_T *this, size_t n, DYNSTRING_T *s, int trim)
{
	size_t osize = this->ptr == 0 ? 0 : this->res;
    size_t size, i, m;
	DYNSTRING_T **np;

	if (n == 0) {
		if (trim)
			DAStidy(this, 1);
	} else if (n == osize || n < osize && !trim)
		;
	else if (n == NPOS)
		error(ERR_LENGTH);
	else {
		size = this->ptr == 0 && n < this->res ? this->res : n;
		np = XMALLOC(DYNSTRING_T*, size);
		m = n < this->len ? n : this->len;
		for (i = 0; i < m; i++)
			np[i] = this->ptr[i];
		if (s)
			for ( ; i < this->res; i++)
				np[i] = DScopy(s);
		DAStidy(this, 1);
		this->ptr = np;
		this->res = size;
	}
	this->len = n;
}

/* DAScreate - create a new DYNARRAYSTRING_T */
DYNARRAYSTRING_T *DAScreate(void)
{
	DYNARRAYSTRING_T *ds = XMALLOC(DYNARRAYSTRING_T, 1);

	DAStidy(ds, 0);
	return ds;
}

/* DASdestroy - return a DYNARRAYSTRING_T to the void */
void DASdestroy(DYNARRAYSTRING_T *ds)
{
	if (ds) {
		DAStidy(ds, 1);
		XFREE(ds);
	}
}

/* insert copies of strings into a DAS */
DYNARRAYSTRING_T *DASinsert(DYNARRAYSTRING_T *ds, size_t p, DYNSTRING_T **s, 
							size_t n, size_t d)
{
	size_t i;

	if (ds->len < p)
		error(ERR_STRING_POSITION);
	if (NPOS - ds->len <= n)
		error(ERR_ARRAY_TOO_BIG); 
	if (0 < n) {
		i = ds->len - p;
		for (DASgrow(ds, n + ds->len, 0, 0); 0 < i; ) {
			--i;
			ds->ptr[p + n + i] = ds->ptr[p + i];
		}
		for (i = 0; i < n; i++, s += d)
			ds->ptr[p + i] = DScopy(*s);
	}
	return ds;
}

/* remove a bunch of lines */
DYNARRAYSTRING_T *DASremove(DYNARRAYSTRING_T *ds, size_t p, size_t n)
{
	size_t m, i;

	if (ds->len < p)
		error(ERR_INVALID_POSITION);
	if (ds->len - p < n)
		n = ds->len - p;
	if (0 < n) {
		m = ds->len - p - n;
		for (i = 0; i < m; i++)
			ds->ptr[p + i] = ds->ptr[p + i + n];
		DASgrow(ds, ds->len - n, 0, 0);
	}
	return ds;
}

/* get the string at position i */
DYNSTRING_T *DASgetat(DYNARRAYSTRING_T *ds, size_t i)
{
	if (ds->len <= i)
		error(ERR_INVALID_POSITION);
	return ds->ptr[i];
}

/* put a string at position i */
void DASputat(DYNARRAYSTRING_T *ds, size_t i, DYNSTRING_T *x)
{
	if (ds->len <= i)
		error(ERR_INVALID_POSITION);
	DSdestroy(ds->ptr[i]);
	ds->ptr[i] = DScopy(x);
}


/* get however many lines are in the DAS */
size_t DASlength(DYNARRAYSTRING_T *ds)
{
	return ds->len;
}

/* END OF FILE */
