/*
   DISKCOPY.EXE, floppy diskette duplicator similar to MS-DOS Diskcopy.
   Copyright (C) 1998, Matthew Stanford.
   Copyright (C) 1999, 2000, Imre Leber.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have recieved a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


   If you have any questions, comments, suggestions, or fixes please
   email me at:  ilebr@vub.ac.be

   module: parser.c - diskcopy.ini syntactic analyser.

*/

#include <stdio.h>
#include <stdlib.h>

#include "parser.h"
#include "scanner.h"
#include "drive.h"
#include "diskcopy.h"
#include "datgen.h"

/*
   Grammar: not accurate!

	INIFILE  -> (TknSPACE|TknRETURN)* TknMEMORYHEADER TknSPACE* TknRETURN MEMORIES
		 -> (TknSPACE|TknRETURN)* TknOPTIONHEADER TknSPACE* TknRETURN OPTIONS

	MEMORIES -> MEMORIES TknRETURN MEMORIES
		 -> TknSPACE* TknEMS  TknSPACE* TknASSIGN TknSPACE* BOOLEAN     TknSPACE* 
		 -> TknSPACE* TknXMS  TknSPACE* TknASSIGN TknSPACE* BOOLEAN     TknSPACE* 
		 -> TknSPACE* TknDISK TknSPACE* TknASSIGN TknSPACE* BOOLEAN     TknSPACE* 
		 -> 

	OPTIONS  -> OPTIONS TknRETURN OPTIONS
		 -> TknSPACE* TknAUDIBLE     TknSPACE* TknASSIGN TknSPACE* BOOLEAN TknSPACE*
		 -> TknSPACE* TknVERIFY      TknSPACE* TknASSIGN TknSPACE* BOOLEAN TknSPACE* 
		 -> TknSPACE* TknINFORMATIVE TknSPACE* TknASSIGN TknSPACE* BOOLEAN TknSPACE* 
		 -> TknSPACE* TknOVERWRITE   TknSPACE* TknASSIGN TknSPACE* TIME    TknSPACE*
		 -> TknSPACE* TknAUTOEXIT    TknSPACE* TknASSIGN TknSPACE* BOOLEAN TknSPACE* 
		 -> TknSPACE* TknMODE        TknSPACE* TknASSIGN TknSPACE* MODE    TknSPACE*
		 ->

	BOOLEAN  -> TknYES
		 -> TknNO

	TIME     -> TknALWAYS
		 -> TknNEVER


	MODE     -> TknRECOVERY
		 -> TknNORMAL

*/

static void SkipWhiteSpace(int losely);
static void ParseINIFILE(void);
static void ShowParserError(void);
static void InitParserStruct(void);

static void ActOnXMS(TOKEN rvalue);
static void ActOnEMS(TOKEN rvalue);
static void ActOnDISK(TOKEN rvalue);
static void ActOnAUDIBLE(TOKEN rvalue);
static void ActOnVERIFY(TOKEN rvalue);
static void ActOnINFORMATIVE(TOKEN rvalue);
static void ActOnAUTOEXIT(TOKEN rvalue);
static void ActOnASKDISK(TOKEN rvalue);
static void ActOnOVERWRITE(TOKEN rvalue);
static void ActOnMODE(TOKEN rvalue);
static void ActOnUSEDATFILE(TOKEN rvalue);
static void ParseBlock(struct LValueArray* lvalues, int amount);

static struct IniParserStruct INIParserData;

struct LValueArray MemoryLvals[] = {{TknXMS,  ActOnXMS},
				    {TknEMS,  ActOnEMS},
				    {TknDISK, ActOnDISK}};

struct LValueArray OptionLvals[] = {{TknAUDIBLE,     ActOnAUDIBLE},
				    {TknVERIFY,      ActOnVERIFY},
				    {TknINFORMATIVE, ActOnINFORMATIVE},
				    {TknAUTOEXIT,    ActOnAUTOEXIT},
				    {TknASKDISK,     ActOnASKDISK},
				    {TknOVERWRITE,   ActOnOVERWRITE},
				    {TknMODE,        ActOnMODE}};

struct LValueArray GenerateLVals[] = {{TknUSEDATFILE, ActOnUSEDATFILE}};

#define AMOFHEADERS 3
struct HeaderArray Headers[] = {{3, TknMEMORYHEADER,   MemoryLvals},
				{7, TknOPTIONHEADER,   OptionLvals},
				{1, TknGENERATEHEADER, GenerateLVals}};

/*
** Public interface to the parser.
*/

int ParseIniFile(char* filename)
{
    int result;

    InitParserStruct();

    if (ExploreDATFile(&INIParserData)) return PARSERSUCCESS;

    if ((result = ReadScannerFile(filename)) <= 0) return result;

    ParseINIFILE();

    if (INIParserData.MakeDAT) MakeDATFile(&INIParserData);

    /* Release scanner memory. */
    CloseScanner();
    return PARSERSUCCESS;
}

/*
** Return a pointer to the parsed results.
*/ 
struct IniParserStruct* GetParsedData()
{
    return &INIParserData;
}

/* 
** Skip white space. 
*/

static void SkipWhiteSpace(int losely)
{
       int token;

       for (;;)
       {
	   token = PeekToken();
	   if ((token == TknSPACE) || (losely && (token == TknRETURN)))
	      GetNextToken();
	   else
	      break;
       }

}

/*
** Parse an .ini file.
*/

static void ParseINIFILE()
{
       int token, i, found;

       for (;;)
       {
	   SkipWhiteSpace(TRUE);

	   token = GetNextToken();
	   SkipWhiteSpace(FALSE);

	   if (token == TknDONE) break;
	   if (GetNextToken() != TknRETURN) ShowParserError();

	   found = FALSE;
	   for (i = 0; i < AMOFHEADERS; i++)
	       if (token == Headers[i].header)
	       {
		  found = TRUE;
		  ParseBlock(Headers[i].lvalues, Headers[i].amount);
	       }

	   if (!found) ShowParserError();
       }
}

/*
** Parse an individual block.
*/

static void ParseBlock(struct LValueArray* lvalues, int amount)
{
       int token, i, found;

       for (;;)
       {
	   SkipWhiteSpace(TRUE);
	   token = PeekToken();

	   found = FALSE;
	   for (i = 0; i < amount; i++)
	   {
	       if (lvalues[i].token == token)
	       {
		  found = TRUE;
		  break;
	       }
	   }
	   if (!found) return;

	   GetNextToken();
	   SkipWhiteSpace(FALSE);
	   
	   token = GetNextToken();
	   if (token == TknNONE) ShowParserError();
	   if ((token == TknRETURN) || (token == TknDONE)) continue;

	   if (token == TknASSIGN)
	   {
	      SkipWhiteSpace(FALSE);
	      token = PeekToken();

	      if ((token != TknRETURN) && (token != TknDONE))
		 lvalues[i].func(GetNextToken());
	   }
	   else
	      lvalues[i].func(token);

	   SkipWhiteSpace(FALSE);
	   if (((token = GetNextToken()) != TknRETURN) && (token != TknDONE))
	      ShowParserError();
       }
}

/*
** Show a syntactic error message.
*/
static void ShowParserError()
{
     printf("Syntax error on line %d in configuration file.\n", GetScannerLine());
     
     CloseScanner();
     exit(INITERROR);
}

/*
** Show a semantic error message.
*/
static void ShowParserErrorMsg(char* msg)
{
     printf("Semantic error in configuration file (%d): %s!\n", 
	    GetScannerLine(), msg);  

     CloseScanner();
     exit(INITERROR);
}

/*
** Initialise parser result struct.
*/ 

static void InitParserStruct()
{
   INIParserData.UseEMS  = YES;
   INIParserData.UseXMS  = YES;
   INIParserData.UseSWAP = YES;

   INIParserData.audible     = NO;
   INIParserData.verify      = NO;
   INIParserData.informative = NO;
   INIParserData.overwrite   = NEVER;
   INIParserData.autoexit    = NO;
   INIParserData.askdisk     = YES;
   INIParserData.mode        = NORMAL;
   INIParserData.MakeDAT     = NO;
}

/*
** Act on <lvalue> = YES|NO
*/

static void PrivateActOnYesNo(int* field, TOKEN rvalue)
{
   switch (rvalue)
   {
     case TknYES:
	  *field = TRUE;
	  break;

     case TknNO:
	  *field = FALSE;
	  break;

     case TknASSIGN:
	  ShowParserErrorMsg("do not enter more than one '='");
	  break;

     case TknRETURN:
	  break;

     default:
	  ShowParserErrorMsg("please enter YES or NO");
   }
}

/*
** Act on XMS = YES|NO
*/

static void ActOnXMS(TOKEN rvalue)
{
   PrivateActOnYesNo(&INIParserData.UseXMS, rvalue);
}

/*
** Act on EMS = YES|NO
*/

static void ActOnEMS(TOKEN rvalue)
{
   PrivateActOnYesNo(&INIParserData.UseEMS, rvalue);
}

/*
** Act on DISK = YES|NO
*/

static void ActOnDISK(TOKEN rvalue)
{
   PrivateActOnYesNo(&INIParserData.UseSWAP, rvalue);
}

/*
** Act on AUDIBLE = YES|NO
*/

static void ActOnAUDIBLE(TOKEN rvalue)
{
   PrivateActOnYesNo(&INIParserData.audible, rvalue);
}

/*
** Act on VERIFY = YES|NO
*/

static void ActOnVERIFY(TOKEN rvalue)
{
   PrivateActOnYesNo(&INIParserData.verify, rvalue);
}

/*
** Act on INFORMATIVE = YES|NO
*/

static void ActOnINFORMATIVE(TOKEN rvalue)
{
   PrivateActOnYesNo(&INIParserData.informative, rvalue);
}

/*
** Act on AUTOEXIT = YES|NO
*/

static void ActOnAUTOEXIT(TOKEN rvalue)
{
   PrivateActOnYesNo(&INIParserData.autoexit, rvalue);
}

/*
** Act on ASKDISK = YES|NO
*/

static void ActOnASKDISK(TOKEN rvalue)
{
   PrivateActOnYesNo(&INIParserData.askdisk, rvalue);
}

/*
** Act on OVERWRITE = ALWAYS|NEVER
*/

static void ActOnOVERWRITE(TOKEN rvalue)
{
   switch (rvalue)
   {
     case TknNEVER:
	  INIParserData.overwrite = NEVER;
	  break;

     case TknALWAYS:
	  INIParserData.overwrite = ALWAYS;
	  break;
     
     case TknASSIGN:
	  ShowParserErrorMsg("do not enter more than one '='");
	  break;

     case TknRETURN:
	  break;

     default:
	  ShowParserErrorMsg("please enter NEVER or ALWAYS");
   }
}

/*
** Act on MODE = RECOVERY|NORMAL
*/

static void ActOnMODE(TOKEN rvalue)
{
   switch (rvalue)
   {
     case TknRECOVERY:
	  INIParserData.mode = RECOVERY;
	  break;

     case TknNORMAL:
	  INIParserData.mode = NORMAL;
	  break;
     
     case TknASSIGN:
	  ShowParserErrorMsg("do not enter more than one '='");
	  break;

     case TknRETURN:
	  break;

     default:
	  ShowParserErrorMsg("please enter RECOVERY or NORMAL");
   }
}

/*
** Act on USEDATFILE = YES|NO
*/
static void ActOnUSEDATFILE(TOKEN rvalue)
{
   PrivateActOnYesNo(&INIParserData.MakeDAT, rvalue);
}


