; Cute Mouse Driver - a tiny mouse driver
; Copyright (c) 1997-2000 Nagy Daniel <nagyd@almos.vein.hu>
;
; This program is free software; you can redistribute it and/or
; modify it under the terms of the GNU General Public License
; as published by the Free Software Foundation; either version 2
; of the License, or (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU Library General Public
; License along with this program; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
;
;
; ChangeLog:
;
; 1.71 - Huge optimizations, many bugfixes from
;        Arkady V. Belousov <ark@mos.ru)
;
; 1.6  - Positioning bugs fixed in videomode 13h
;      - Added protocol analyzer (debugging purposes only)
;
; 1.5  - Fixed memory allocation bugs
;      - Bugfixes and optimizations
;
; 1.4  - Added self-load high capability
;      - Horizontal and vertical speed can be different
;        (good for some games...)
;      - Added support for the 1Ah and 1Bh INT 33h calls
;      - Fixed some small bugs and a BIG bug
;
; 1.3  - Fixed some small bugs and added 'ctmousep.com' for PS/2 mice
;
; 1.2a - Fixed a bug in the Driver Disable function (1fh)
;
; 1.2  - Added command line option to force IRQ number of COM port
;
; 1.1  - Added comand line options to force a specific mode
;      - Rewritten Mouse Systems initalization routine. Now more Genius mice
;        with 3 buttons will work
;
; 1.0  - First public release


;PS2=1

WARN

J		equ	jmp short

movSeg		macro	dest,src
		push	src
		pop	dest
	endm

saveFAR		macro	addr,segm,offs
		mov	word ptr addr[0],offs
		mov	word ptr addr[2],segm
	endm

IFz		macro	var,addr
		cmp	var,0
		jz	addr
	endm

IFnz		macro	var,addr
		cmp	var,0
		jnz	addr
	endm

PS2serv		macro	serv,errlabel
		mov	ax,serv
		int	15h
	IFNB <errlabel>
		jc	errlabel
		or	ah,ah
		jne	errlabel
	ENDIF
	endm

PrintS		macro	addr
	IFNB <addr>
	 IFDIFI <addr>,<dx>
		mov	dx,offset addr
	 ENDIF
	ENDIF
		mov	ah,9
		int	21h
	endm

driverversion	equ	626h		; Microsoft driver version

.model tiny				; it is a COM file
.code
		org	100h
start:		jmp	real_start

IDstring	db	'CuteMouse 1.71',0
IDstringlen = $ - IDstring
		even
RILversion	label
msversion	db	driverversion / 100h,driverversion MOD 100h

;----- FAR pointers storage -----

oldint33	dd	0		; old INT 33h handler address
oldIRQaddr	dd	0		; old IRQ handler address

;----- driver state begins here -----

POINT		struc
  X		dw	0
  Y		dw	0
POINT		ends

disabled?	db	1		; 1=driver disabled?
videounlock	db	1		; 0=screen manipulation is in progress

mousetype	db	0		; Mouse type
IRQintnum	db	68h+12,25h	; INT number of selected IRQ
IFNDEF PS2
IO_number	dw	?		; IO port number
PICstate	db	?		; indicates which bit to clear in PIC
logitech?	db	0		; 0 if no, 1 if yes
IObyte4		db	?		; extra byte of MouseMan protocol
ENDIF
IObyte1		db	?		; buttons state in serial protocol
IOdone		db	?		; indicates processed mouse bytes
		even
increment	POINT	?
userprocunlock	db	1		; 0=user defined proc is in progress

mem_type	db	0		; 1 = DOS 5.0+ UMB
					; 2 = XMS UMB
					; 3 = conventional memory

;----- application state begins here -----

		even
StartSaveArea = $

mresolution	dw	0303h		; mouse resolution
autores?	db	0		; indicates whether auto/man resolution
lefthand?	db	0		; 0 - right, 1 - left
cursorhidden	db	?		; 0 - cursor visible, else hidden

		even
StartInitArea = $

mickey		POINT	?
remain		POINT	?
		even
LenInitArea1 = $ - StartInitArea	; cleared by setpos_04

rangemin	POINT	?		; horizontal/vertical range min
		even
LenInitArea2 = $ - StartInitArea	; cleared by setupvideo

cursortype	db	?		; software/hardware
regionchk?	db	?		; indicates if have to check region
upleft		POINT	?		; upper left for updating
lowright	POINT	?		; lower right for updating
hotspot		POINT	?		; hot spot in cursor bitmap
callmask	dw	?		; user call mask
buttstatus	dw	?		; buttons status
BUTTLASTSTATE	STRUC
  counter	dw	?
  lastrow	dw	?
  lastcol	dw	?
	ENDS
buttpress	BUTTLASTSTATE	4 DUP()
buttrelease	BUTTLASTSTATE	4 DUP()
nolightpen?	db	?		; 0 = emulate light pen
		even
LenInitArea3 = $ - StartInitArea	; cleared by softreset_21

rangemax	POINT	?		; horizontal/vertical range max
pos		POINT	?
granpos		POINT	?
mickey8		POINT	?		; mickeys per 8 pixel
doublespeed	dw	?		; double speed threshold (mickeys/sec)
startscan	dw	?		; screen mask/cursor start scanline
endscan		dw	?		; cursor mask/cursor end scanline
screenmask	db	3*16 dup (?)	; user defined screen mask
cursormask	db	3*16 dup (?)	; user defined cursor mask
Xcheck		db	?
		even
LenSaveArea = $ - StartSaveArea

		even
shape		dw	03FFFh, 01FFFh, 00FFFh, 007FFh
		dw	003FFh, 001FFh, 000FFh, 0007Fh
		dw	0003Fh, 0001Fh, 001FFh, 000FFh
		dw	030FFh, 0F87Fh, 0F87Fh, 0FCFFh

		dw	00000h, 04000h, 06000h, 07000h
		dw	07800h, 07C00h, 07E00h, 07F00h
		dw	07F80h, 07C00h, 06C00h, 04600h
		dw	00600h, 00300h, 00300h, 00000h

;----- video state begins here -----

cursorcoord	POINT	?
cursorptr	dd	0		; video memory pointer
nextrow		dw	?		; add this to reach next row
cursordrawn	db	0		; 1=indicate to restore screen data

SHIFTHIVMODE	equ	0Dh-8

videomode	db	?		; video mode number
shift320	db	?		; 1=320x* video mode
planar?		db	?		; 1=multiplanar video modes

		even
maxcoord	POINT	?		; max coordinate
granu		POINT	{X=0,Y=0}
spritewidth	dw	0		; sprite row length in bytes
nextxor		dw	?
ptr2buffer	dd	?		; pointer to screen data
buffer		db	3*16 dup (?)	; screen data buffer

vdata1		dw	10h,1, 10h,3, 10h,4, 10h,5, 10h,8, 08h,2
VDATA1cnt	equ	($-vdata1)/4

vdata2		dw	10h,1, 10h,4, 10h,0105h, 10h,0FF08h, 08h,0F02h
VDATA2cnt	equ	($-vdata2)/4

;----- registers values for RIL -----
;!!! WARNING: registers order and RGROUPDEF contents must be fixed !!!
		even
StartVRegsArea = $

regs_SEQC	db	5 dup (?)
reg_MISC	db	?
regs_CRTC	db	25 dup (?)
regs_ATC	db	21 dup (?)
regs_GRC	db	9 dup (?)
reg_FC		db	?
reg_GPOS1	db	?
reg_GPOS2	db	?
		even
LenVRegsArea = $ - StartVRegsArea

		even
StartDefVRegsArea = $

def_SEQC	db	5 dup (?)
def_MISC	db	?
def_CRTC	db	25 dup (?)
def_ATC		db	21 dup (?)
def_GRC		db	9 dup (?)
def_FC		db	?
def_GPOS1	db	?
def_GPOS2	db	?
		even
ERRIF ($-StartDefVRegsArea ne LenVRegsArea) "VRegs area contents corrupted!"

RGROUPDEF	struc
  @port		dw	?
  @regs		dw	?
  @def		dw	?
  regscnt	db	1
  rmodify?	db	0
RGROUPDEF	ends

@videoregs	label
			; CRTC
	RGROUPDEF {@port=3D4h,@regs=regs_CRTC,@def=def_CRTC,regscnt=25}
			; Sequencer
	RGROUPDEF {@port=3C4h,@regs=regs_SEQC,@def=def_SEQC,regscnt=5}
			; Graphics controller
	RGROUPDEF {@port=3CEh,@regs=regs_GRC, @def=def_GRC, regscnt=9}
			; VGA attrib controller
	RGROUPDEF {@port=3C0h,@regs=regs_ATC, @def=def_ATC, regscnt=20}
			; VGA misc output and input
	RGROUPDEF {@port=3C2h,@regs=reg_MISC, @def=def_MISC}
			; Feature Control
	RGROUPDEF {@port=3DAh,@regs=reg_FC,   @def=def_FC}
			; Graphics 1 Position
	RGROUPDEF {@port=3CCh,@regs=reg_GPOS1,@def=def_GPOS1}
			; Graphics 2 Position
	RGROUPDEF {@port=3CAh,@regs=reg_GPOS2,@def=def_GPOS2}
singlemodify?	db	?


;
;				IRQ handler
;

		even
IRQhandler	proc	far
		cld
		push	ax bx cx dx si di bp ds es
		movSeg	ds,cs
IFNDEF PS2
		mov	al,20h
		out	20h,al			; {20h} end of interrupt

		mov	dx,[IO_number]
		add	dx,5
		in	al,dx			; {3FDh} check for overrun
		sub	dx,5
		mov	ah,al
		in	al,dx			; flush receive buffer

		test	ah,2
		jz	@@nooverrun		; jump if no overrun occured
		mov	[IOdone],0		; zero counter
		j	@@exitIRQ

@@nooverrun:	test	ah,1
		jz	@@exitIRQ		; jump if data not ready
		db	0E8h			; CALL NEAR to mouseproc
mouseproc	dw	?

@@exitIRQ:	jmp	@rethandler
IRQhandler	endp

;
;		Process mouse bytes the Microsoft/Logitech way
;

MSLGproc	proc
		mov	cl,[IOdone]
		mov	ch,0
		test	al,40h			; synchro check
		jz	@@MSLGbyte2		; jump if non first byte

		cmp	cx,3			; first byte after 3 bytes?
		jne	@@MSLGbyte1
		mov	[IObyte4],cl		; middle button released
@@MSLGbyte1:	mov	[IOdone],1		; request next 2/3 bytes
		mov	[IObyte1],al
@@MSLGprocret:	ret

@@MSLGbyte2:	jcxz	@@MSLGprocret		; skip nonfirst byte at start
		inc	[IOdone]		; request next byte
		loop	@@MSLGbyte3
		mov	byte ptr [increment.X],al ; store X increment LO
		ret

@@MSLGbyte3:	loop	@@LGbyte4
		mov	cl,al
		mov	ah,[IObyte1]

		mov	al,0
		shr	ax,2			; bits 1/0 - X increment HI
		or	al,byte ptr [increment.X]
		mov	bh,ah
		cbw
		xchg	bx,ax			; X increment

		mov	al,0
		shr	ax,2			; bits 3/2 - Y increment HI
		or	al,cl
		mov	ch,ah			; bits 5/4 - L/R buttons
		cbw
		xchg	cx,ax			; Y increment

		mov	al,[IObyte4]
		and	ah,3			; clear service info
		and	al,4			; extract middle button
		or	al,ah			; AL = Buttons status
		mov	[IObyte4],al
		j	mouseupdate

@@LGbyte4:	mov	[IOdone],0		; request next 3/4 bytes
		mov	cl,3
		cmp	[mousetype],cl		; microsoft mouse must
		jne	@@MSLGprocret		; produce only 3 bytes
		shr	al,cl
		xor	al,[IObyte4]
		and	al,4
		jz	@@MSLGprocret		; return if state not changed
		xor	al,[IObyte4]
		mov	[IObyte4],al
		xor	cx,cx
		xor	bx,bx
		j	mouseupdate
MSLGproc	endp

;
;		Process mouse bytes the Mouse Systems way
;

MSMproc		proc
		cbw
		mov	cl,[IOdone]
		mov	ch,0
		jcxz	@@MSMbyte1
		inc	[IOdone]		; request next byte
		dec	cx
		jz	@@MSMbyte24
		dec	cx
		jz	@@MSMbyte3
		loop	@@MSMbyte5

@@MSMbyte24:	add	[increment.X],ax
		ret

@@MSMbyte1:	mov	[IObyte1],al		; save buttons state
		and	al,not 7
		cmp	al,80h			; sync check
		jne	@@MSMnosync
		inc	[IOdone]		; request next byte
@@MSMnosync:	mov	[increment.X],0
		ret

@@MSMbyte3:	mov	[increment.Y],ax
		ret

@@MSMbyte5:	mov	[IOdone],0
		add	ax,[increment.Y]
		mov	cx,ax
		neg	cx
		mov	bx,[increment.X]
		mov	al,[IObyte1]		; bits 2/1/0 - L/M/R buttons
		not	al			;  (0 if pressed)
		and	al,7
		test	al,6			; check the L and M state
		jpe	@@MSMsame		; skip if the same
		xor	al,6			; swap L and M buttons
@@MSMsame:	;j	mouseupdate
MSMproc		endp

ELSE
		mov	al,20h
		out	0A0h,al			; {0A0h} end of interrupt
		out	20h,al			; {20h} end of interrupt

		in	al,60h
		mov	cl,[IOdone]
		mov	ch,0
		jcxz	@@PS2byte1

@@PS2byte2:	loop	@@PS2byte3
		mov	byte ptr [increment.X],al ; 2. byte received
		j	@@exitIRQnext

@@PS2byte1:	mov	bl,0C8h			; try to synchronize
		and	bl,al
		cmp	bl,8
		jne	@@exitIRQ		; the transmission went out of order
		mov	[IObyte1],al		; 1. Byte received
@@exitIRQnext:	inc	[IOdone]		; request next byte
		j	@@exitIRQ

@@PS2byte3:	mov     [IOdone],0
		mov	ah,0
		mov	cx,ax			; increment.Y
		mov	al,byte ptr [increment.X]
		mov	bx,ax			; increment.X
		mov	al,[IObyte1]		; buttons status
		jcxz	@@skipY
		test	al,20h
		jnz	@@noyneg
		dec	ch
@@noyneg:	neg	cl
@@skipY:	test	al,10h
		jz	@@noxneg
		dec	bh
@@noxneg:	and	al,3
		call	mouseupdate

@@exitIRQ:	jmp	@rethandler
IRQhandler	endp
ENDIF

;
;			Update mouse status
;
;
; In:	AL					(new buttons state)
;	BX					(X increment)
;	CX					(Y increment)
; Out:	none
; Use:	mickey8, granu, lefthand?, callmask
; Modf: AX, BX, CX, DX, mickey, remain, pos, granpos, buttstatus,
;	userprocunlock
; Call: resolute, cutrangex, cutrangey, updatebutton, showcursor
;
mouseupdate	proc
		push	ax

;---- calculate X movement in pixels ----
		mov	ax,bx
		or	ax,ax			; is X movement 0?
		jz	@@xmov0			;  jump if yes

		xor	bx,bx
		call	resolute
		add	[mickey.X],ax

		shl	ax,3
		add	ax,[remain.X]
		cwd
		idiv	[mickey8.X]
		mov	[remain.X],dx

		add	ax,[pos.X]
		call	cutrangex
		mov	[pos.X],ax
		xor	dx,dx
		div	[granu.X]
		mul	[granu.X]
		mov	dx,ax
		xchg	ax,[granpos.X]		; the new column is ready
		sub	ax,dx			; X screen shift
@@xmov0:
;---- calculate Y movement in pixels ----
		jcxz	@@ymov0			; jump if Y movement is 0
		xchg	ax,cx

		mov	bx,1
		call	resolute
		add	[mickey.Y],ax

		shl	ax,3
		add	ax,[remain.Y]
		cwd
		idiv	[mickey8.Y]
		mov	[remain.Y],dx

		add	ax,[pos.Y]
		call	cutrangey
		mov	[pos.Y],ax
		xor	dx,dx
		div	[granu.Y]
		mul	[granu.Y]
		mov	dx,ax
		xchg	ax,[granpos.Y]		; the new row is ready
		sub	ax,dx			; Y screen shift
@@ymov0:
;------------
		or	cx,ax			; if both are 0
		neg	cx			; then no carry flag
		sbb	cx,cx
		neg	cx			; and result 0, else 1

		pop	ax
IFNDEF PS2
		IFnz	[lefthand?],@@updatebut
ELSE
		IFz	[lefthand?],@@updatebut
ENDIF
		test	al,3			; check the L and R buttons
		jpe	@@updatebut		; skip if the same
		xor	al,3			; swap

@@updatebut:	mov	ah,al			; AH=buttons state
		mov	ch,al
		xchg	al,byte ptr [buttstatus]
		xor	ch,al			; CH=buttons change state
		mov	al,cl			; AL=unrolled change state

		mov	cl,2			; indicate that 1 is pressed
		mov	bx,offset buttpress
		call	updatebutton
		mov	cl,8			; indicate that 2 is pressed
		mov	bx,offset buttpress+SIZE BUTTLASTSTATE
		call	updatebutton
IFNDEF PS2
		mov	cl,20h			; indicate that 3 is pressed
		mov	bx,offset buttpress+(SIZE BUTTLASTSTATE)*2
		call	updatebutton
ENDIF
		dec	[userprocunlock]
		jnz	@@updatedone		; exit if user proc running

;------------ call User Defined Handler
		mov	bx,[buttstatus]
		mov	cx,[granpos.X]
		mov	dx,[granpos.Y]
		mov	si,[mickey.X]
		mov	di,[mickey.Y]
		sti
;*		push	ax
		and	al,byte ptr [callmask]	; is there a user call mask?
		jz	@@updateshow		; exit if not
		db	9Ah			; CALL FAR to userproc
userproc	dd	0
		movSeg	ds,cs
;------------
@@updateshow:
;*		pop	ax
;*		shr	ax,1			; was there movement?
;*		jnc	@@updatedone		;  exit if no
		call	showcursor

@@updatedone:	cli
		inc	[userprocunlock]
		ret
mouseupdate	endp

;
;		Update one button status regs to new values
;
;
; In:	AL					(unrolled button change state)
;	AH					(current button state)
;	CL					(press bit mask)
;	CH					(button change state)
;	BX					(offset to item of buttpress)
; Out:	AX, CH
; Use:	granpos
; Modf: BX, CL, buttpress, buttrelease
; Call: none
;
updatebutton	proc
		shr	ah,1
		jc	@@updset		; jump if pressed
		shl	cl,1			; indicate that is released
		add	bx,offset buttrelease-offset buttpress
@@updset:	shr	ch,1
		jnc	@@updret		; jump if button unchanged
		or	al,cl
		inc	[bx.counter]
		push	ax
		mov	ax,[granpos.Y]
		mov	[bx.lastrow],ax
		mov	ax,[granpos.X]
		mov	[bx.lastcol],ax
		pop	ax
@@updret:	ret
updatebutton	endp

;
;			Use selected resolution
;
;
; In:	AX					(X/Y increment)
;	BX					(0/1 - indicates X/Y)
; Out:	AX					(scaled increment)
; Use:	autores?, mresolution
; Modf: BX, DX
; Call: none
;
resolute	proc
		mov	bl,byte ptr mresolution[bx]
		IFz	[autores?],@@resolute
		sar	ax,2
		push	ax
		and	al,0Fh			;*** ??? *** signed value
		cmp	al,10
		jle	@@le
		mov	al,10
@@le:		mov	bl,al
		pop	ax

@@resolute:	cmp	bl,1			; is resolution 1 or 0?
		jbe	@@resret		;  exit if yes
		cmp	ax,3			; was movement smaller than +3?
		jb	@@resret		;  exit if yes
		cmp	ax,-3			; was movement larger than -3?
		ja	@@resret		;  exit if yes
		cmp	ax,8			; was movement between +3 and +8?
		jb	@@ressmall		;  jump if yes
		cmp	ax,-8			; was movement between -3 and -8?
		ja	@@ressmall		;  jump if yes
		mov	bh,0
		imul	bx			; else multiply it with resolution
		ret
@@ressmall:	shl	ax,1			; small movement -> multiply by 2
@@resret:	ret
resolute	endp


; END OF IRQ HANDLERS 


;
;				INT 10 handler
;

		even
RILfuncs	dw	offset RIL_F0		; RIL functions
		dw	offset RIL_F1
		dw	offset RIL_F2
		dw	offset RIL_F3
		dw	offset RIL_F4
		dw	offset RIL_F5
		dw	offset RIL_F6
		dw	offset RIL_F7

int10handler	proc
		cld
		or	ah,ah			; set video mode?
		jz	@@setmodreq
		cmp	ah,4			; light pen func?
		je	@@lightpen
		cmp	ah,0F0h			; RIL func requested?
		jb	@@jmpold10
		cmp	ah,0F7h
		jbe	@@RIL
		cmp	ah,0FAh
		je	@@RIL_FA
@@jmpold10:	db	0EAh			; JMP FAR to old Int10 handler
oldint10	dd	?

;============= RIL
@@RIL:		push	ax cx dx ds es di si
		movSeg	ds,cs
		mov	al,ah
		and	ax,0Fh
		shl	ax,1
		mov	si,ax
		call	RILfuncs[si]
		pop	si di es ds dx cx ax
		iret

@@RIL_FA:	movSeg	es,ds			; INTERROGATE DRIVER
		mov	bx,offset RILversion
		iret

;============ emulate lightpen
@@lightpen:	IFnz	cs:[nolightpen?],@@jmpold10
;+*		sti
;+*		mov	ax,cs:[granpos.Y]
;+*		mov	ch,al
;+*		cmp	cs:[videomode],0Fh-SHIFTHIVMODE
;+*		jb	@@lpen2
;+*		mov	cx,ax
;+*@@lpen2:	div	cs:[pixboxheight]
;+*		mov	dh,al
;+*		mov	ax,cs:[granpos.X]
;+*		mov	bx,ax
;+*		cwd
;+*		div	cs:[granu.X]
;+*		xchg	bx,ax
;+*		div	cs:[pixboxwidth]
;+*		mov	dl,al
;+*		mov	ah,1
		iret

;============ set video mode
@@setmodreq:	push	ax
		mov	cs:[cursorhidden],0
		mov	ax,2
		int	33h			; mouse driver, hide cursor
		pop	ax
		pushf
		call	cs:[oldint10]
		push	ax bx cx dx si di bp ds es
		call	setupvideo
		jmp	@rethandler
int10handler	endp

;
;			RIL functions
;

RIL_F0		proc				; READ ONE REGISTER
		mov	si,dx
		mov	si,@videoregs[si].@regs
		cmp	dx,20h
		jge	@@R0ret
		add	si,bx
@@R0ret:	mov	bl,[si]
		ret
RIL_F0		endp

;

RIL_F1		proc				; WRITE ONE REGISTER
		mov	ax,bx
		mov	si,dx
		cmp	dl,20h
		mov	dx,@videoregs[si].@port
		mov	@videoregs[si].rmodify?,dl
		mov	si,@videoregs[si].@regs
		jb	@@R0group		; jump if group of registers

		mov	[singlemodify?],dl
		mov	[si],al
		out	dx,al
		ret

@@R0group:	mov	bh,0
		mov	[bx+si],ah
		mov	bl,ah
		;j	RIL_write
RIL_F1		endp

;
;
; In:	DX					(IO port)
;	AL					(reg number)
;	AH					(value to write)
; Out:	none
; Use:	none
; Modf: none
; Call: none
;
RIL_write	proc
		cmp	dl,0C0h
		je	@@RwATC			; jump if ATTR controller
		out	dx,ax
		ret
@@RwATC:	push	ax dx
		mov	dl,byte ptr @videoregs[(SIZE RGROUPDEF)*5].@port
		in	al,dx			; {3DAh} force address mode
		pop	dx ax
		out	dx,al			; {3C0h} select ATC register
		xchg	al,ah
		out	dx,al			; {3C0h} modify ATC register
		xchg	al,ah
		ret
RIL_write	endp

;

RIL_F2		proc				; READ REGISTER RANGE
		sti
		mov	di,bx
		mov	si,dx
		mov	si,@videoregs[si].@regs
		xor	ax,ax
		xchg	al,ch
		add	si,ax
RILmemcopy:	shr	cx,1
		rep	movsw
		adc	cx,cx
		rep	movsb
		ret
RIL_F2		endp

;

RIL_F3		proc				; WRITE REGISTER RANGE
		sti
		mov	si,bx
		mov	di,dx
		mov	dx,@videoregs[di].@port
		mov	@videoregs[di].rmodify?,dl
		mov	di,@videoregs[di].@regs
		xor	ax,ax
		xchg	al,ch
		add	di,ax
@@R3loop:	mov	ah,al
		lods	byte ptr es:[si]
		mov	[di],al
		inc	di
		xchg	al,ah
		call	RIL_write
		inc	ax			; OPTIMIZE: AX instead AL
		loop	@@R3loop
		ret
RIL_F3		endp

;

RIL_F4		proc				; READ REGISTER SET
		sti
		mov	di,bx
@@R4loop:	mov	si,es:[di]
		inc	di
		inc	di
		mov	si,@videoregs[si].@regs
		mov	al,es:[di]
		inc	di
		cbw
		add	si,ax
		movsb
		loop	@@R4loop
		ret
RIL_F4		endp

;

RIL_F5		proc				; WRITE REGISTER SET
		sti
		mov	si,bx
@@R5loop:	lods	word ptr es:[si]
		mov	di,ax
		lods	word ptr es:[si]
		mov	dx,@videoregs[di].@port
		mov	@videoregs[di].rmodify?,dl
		mov	di,@videoregs[di].@regs
		push	ax
		cbw
		add	di,ax
		pop	ax
		mov	[di],ah
		call	RIL_write
		loop	@@R5loop
		ret
RIL_F5		endp

;

RIL_F6		proc				; REVERT REGISTERS TO DEFAULT
		sti
		push	bx
		movSeg	es,ds
		mov	bx,offset @videoregs
		xor	cx,cx

@@R6loop:	cmp	[bx].rmodify?,ch
		je	@@R6next

		mov	[bx].rmodify?,ch
		mov	cl,[bx].regscnt
		mov	si,[bx].@def
		mov	di,[bx].@regs
		mov	dx,[bx].@port
		mov	al,0

@@R6rloop:	mov	ah,al
		lodsb
		stosb
		xchg	al,ah
		cmp	dx,03CEh		; {3CEh} graphics controller
		jne	@@R6noGRC
		cmp	al,6
		je	@@R6rnext		; skip register 6 of GRC
@@R6noGRC:	call 	RIL_write
@@R6rnext:	inc	ax			; OPTIMIZE: AX instead AL
		loop	@@R6rloop

@@R6next:	add	bx,SIZE RGROUPDEF
		cmp	bx,offset @videoregs+(SIZE RGROUPDEF)*4
		jb	@@R6loop

		cmp	[singlemodify?],ch
		je	@@R6ret
		mov	[singlemodify?],ch
		mov	dx,@videoregs[(SIZE RGROUPDEF)*5].@port
		mov	al,[def_FC]
		mov	[reg_FC],al
		out	dx,al			; {3DAh}
		mov	dx,3C2h
		mov	al,[def_MISC]
		mov	[reg_MISC],al
		out	dx,al			; {3C2h} EGA misl out reg
		mov	dx,3CCh
		mov	al,[def_GPOS1]
		mov	[reg_GPOS1],al
		out	dx,al			; {3CCh} EGA graphics 1 pos
		mov	dx,3CAh
		mov	al,[def_GPOS2]
		mov	[reg_GPOS2],al
		out	dx,al			; {3CAh} EGA graphics 2 pos

@@R6ret:	pop	bx
		ret
RIL_F6		endp

;

RIL_F7		proc				; DEFINE REGISTERS DEFAULTS
		sti
		mov	si,bx
		mov	di,dx
		mov	cl,@videoregs[di].regscnt
		mov	@videoregs[di].rmodify?,cl
		mov	[singlemodify?],cl
		mov	ch,0
		mov	di,@videoregs[di].@def
		push	es ds
		pop	es ds
		jmp	RILmemcopy
RIL_F7		endp


; END OF INT 10 HANDLER 


IFNDEF PS2
;
;			Enable serial interrupt in PIC
;
;
; In:	none
; Out:	none
; Use:	IO_number, mousetype, logitech?, PICstate
; Modf: AX, DX
; Call: none
;
enableIRQ	proc
;---------- set communication parameters (speed, parity, etc.)
		mov	dx,[IO_number]
		add	dx,3
		mov	al,80h
		out	dx,al			; {3FBh} set DLAB on

		sub	dx,3
		mov	ax,96
		out	dx,ax			; {3F8h},{3F9h} 1200 baud

		add	dx,3
		db	0B0h			; MOV AL,byte
COMLCR		db	?
		out	dx,al			; {3FBh} DLAB off, no parity,
						;  stop=1, length=7/8
		inc	dx
		mov	al,0Bh
		out	dx,al			; {3FCh} reset hardware
						;  (DTR/RTS/OUT2 on)
		sub	dx,3
		mov	al,1
		out	dx,al			; {3F9h} DR int enable

		add	dx,4			; {3FDh} read LSR thus
		in	al,dx			;  clearing error bits
		sub	dx,5
		in	al,dx			; {3F8h} flush receive buffer
;----------
		in	al,21h			; {21h} int IMR
		mov	ah,[PICstate]
		not	ah
		and	al,ah			; enable serial interrupts
@outPIC:	out	21h,al
		ret
enableIRQ	endp

;
;			Disable serial interrupt of PIC
;
;
; In:	none
; Out:	none
; Use:	PICstate
; Modf: AL
; Call: none
;
disableIRQ	proc
		in	al,21h			; get PIC mask in al
		or	al,[PICstate]		; disable serial interrupts
		j	@outPIC
disableIRQ	endp

ELSE
;
;				Enable PS/2
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: AX, BX, ES
; Call: none
;
enableIRQ	proc
		mov	bh,1
@PS2C200:	PS2serv	0C200h			; set mouse on/off
		ret
enableIRQ	endp

;
;				Disable PS/2
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: AX, BX, ES
; Call: none
;
disableIRQ	proc
		mov	bh,0
		j	@PS2C200
disableIRQ	endp
ENDIF

;
;			Set up user defined graphics cursor
;
;
; In:	DS:SI					(pointer to bitmaps)
; Out:	none
; Use:	none
; Modf: BX, SI, DI, ES, screenmask, cursormask, Xcheck
; Call: none
;
setusershape	proc
		movSeg	es,cs
		mov	bl,0FFh
		mov	di,offset screenmask	; copy screen mask
		call	@@copybitmap

		mov	bl,0
		mov	cs:[Xcheck],bl
		;mov	di,offset cursormask	; copy cursor mask

@@copybitmap:	mov	cx,16
@@copymask:	lodsw
		xchg	al,ah
		stosw
		mov	al,bl
		stosb
		loop	@@copymask
		ret
setusershape	endp

;
;			Cut coordinate inside range
;

cutrangex	proc
		cmp	ax,[rangemin.X]
		jg	@@cutxhi
		mov	ax,[rangemin.X]
@@cutxhi:	cmp	ax,[rangemax.X]
		jl	@@cutxret
		mov	ax,[rangemax.X]
@@cutxret:	ret
cutrangex	endp

cutrangey	proc
		cmp	ax,[rangemin.Y]
		jg	@@cutyhi
		mov	ax,[rangemin.Y]
@@cutyhi:	cmp	ax,[rangemax.Y]
		jl	@@cutyret
		mov	ax,[rangemax.Y]
@@cutyret:	ret
cutrangey	endp

;
;			Draw mouse cursor
;

showcursor	proc
		dec	[videounlock]		; is drawing in progress?
		jnz	@showdone		;  exit if yes
		cmp	[cursorhidden],0	; is cursor hidden?
		jne	@showdone		;  exit if yes

		mov	ax,[granu.Y]
		dec	ax			; text mode?
		jnz	drawtextcursor		; jump if yes
		cmp	[videomode],13h-SHIFTHIVMODE
		ja	@showdone
		call	drawgraphcurs		; else draw graphics cursor

@showdone:	inc	[videounlock]		; drawing stopped
@showret:	ret
showcursor	endp

;

restorescreen	proc
		dec	[videounlock]		; is drawing in progress?
		jnz	@showdone		;  exit if yes
		call	restoregraph
		call	restorevregs
		j	@showdone
restorescreen	endp

;
;			Draw text mode cursor
;

drawtextcursor	proc
		IFz	[cursortype],@@softcurs	; jump if software cursor

;------------ draw hardware text mode cursor
		call	gettxtoffset
		shr	di,1

		mov	dx,@videoregs[0].@port	; CRTC port
		mov	ax,di
		mov	ah,al
		mov	al,0Fh
		out	dx,ax			; cursor position lo

		mov	ax,di
		mov	al,0Eh
		out	dx,ax			; cursor position hi
		j	@showdone

;------------ draw software text mode cursor
@@softcurs:	call	restoreoldscr
;*		mov	ax,[granpos.Y]
;*		mov	di,[granpos.X]		; get coordinates
;+*		mov	cx,ax
;+*		mov	dx,di
;+*		add	cx,[granu.Y]
;+*		add	dx,[granu.X]
;+*		call	checkregion		; out of update region?
;+*		jc	@showdone

		call	gettxtoffset
		cli
		call	waitretrace
		mov	ax,es:[di]		; store char under cursor
		mov	word ptr [buffer],ax
		and	ax,[startscan]
		xor	ax,[endscan]
		mov	es:[di],ax		; draw to new position
		sti
		inc	[cursordrawn]		; we have to restore later
		j	@showdone
drawtextcursor	endp

;
;			Draw graphics cursor
;

drawgraphcurs	proc
		mov	ax,[granpos.Y]
		sub	ax,[hotspot.Y]		; Y calculated
		mov	di,[granpos.X]
		sub	di,[hotspot.X]
		mov	cl,byte ptr granu.X[0]
		dec	cx			; OPTIMIZE: CX instead CL
		sar	di,cl
		xor	cl,[shift320]
		shl	di,cl			; X calculated

;+*		mov	cx,ax
;+*		add	cx,16
;+*		sub	ax,8
;+*		and	di,not 7
;+*		mov	dx,di
;+*		add	dx,18h
;+*		call	checkregion		; out of update region?
;+*		jc	@restorescr

;*		cmp	ax,[cursorcoord.Y]
;*		jne	@@drawgrcurs
;*		cmp	di,[cursorcoord.X]
;*		jne	@@drawgrcurs
;*		IFnz	[cursordrawn],@showret	; jump if no move &
						;  cursor drawn
@@drawgrcurs:	push	ax di
		call	restoregraph
		pop	di ax
		mov	[cursorcoord.X],di
		mov	[cursorcoord.Y],ax
		call	getgroffset
		mov	al,0
		call	copysprite

		cmp	[videomode],13h-SHIFTHIVMODE
		je	@@getoffset
		call	scrollmask
@@getoffset:	les	di,[cursorptr]
		mov	cx,[nextrow]

		IFz	[planar?],@@make
		mov	dx,3CEh
		mov	ax,5
		out	dx,ax			; {3CEh},{3CFh} set write mode

;----------
@@make:		mov	ax,[cursorcoord.Y]
		xor	bx,bx			; mask offset: 3*16 bytes

@@makeloop:	cmp	ax,[maxcoord.Y]
		jae	@@makenext1		; jump if Y > max || Y < 0

		push	ax bx cx di
		mov	si,[cursorcoord.X]
		cmp	[videomode],13h-SHIFTHIVMODE
		je	make13
		call	makeno13
@makenext:	pop	di cx bx ax

@@makenext1:	inc	ax
		add	di,cx
		xor	cx,[nextxor]
		add	bx,3
		cmp	bx,3*16
		jb	@@makeloop
;----------
		inc	[cursordrawn]
		;j	restorevregs
drawgraphcurs	endp

;
;		Restore graphics card video registers
;

restorevregs	proc
		IFz	[planar?],@nogrparam
		movSeg	es,cs
		mov	bx,offset vdata1
		mov	cx,VDATA1cnt
		j	@writevideo
restorevregs	endp

;
;		Save & update graphics card video registers
;

updatevregs	proc
		IFz	[planar?],@nogrparam
		movSeg	es,cs

		mov	bx,offset vdata1
		mov	cx,VDATA1cnt
		mov	ah,0F4h			; read register set
		int	10h

		mov	bx,offset vdata2
		mov	cx,VDATA2cnt
@writevideo:	mov	ah,0F5h			; write register set
		int	10h
@nogrparam:	ret
updatevregs	endp

;
;		Transform the cursor mask to screen content
;
;
; In:	SI					(X coordinate)
;	BX					(mask offset)
;	ES:DI					(video memory pointer)
; Out:	none
; Use:	maxcoord.X, screenmask, cursormask
; Modf: AX, BX, CX, DX, SI, DI
; Call: none
;
make13		proc
		mov	dx,word ptr screenmask[bx]
		mov	bx,word ptr cursormask[bx]
		mov	cx,16			; 2*8 bytes per row

@@loopx13:	cmp	si,[maxcoord.X]
		jae	@@nextx13		; jump if X > max || X < 0
		mov	al,0
		test	dl,80h
		jz	@@chkcmask
		mov	al,es:[di]
@@chkcmask:	test	bl,80h
		jz	@@putbyte13
		xor	al,0Fh
@@putbyte13:	mov	es:[di],al
@@nextx13:	inc	si
		rol	dx,1
		rol	bx,1
		inc	di
		loop	@@loopx13
		j	@makenext
make13		endp

;
;		Display cursor in other than mode 13h modes
;
;
; In:	SI					(X coordinate)
;	BX					(mask offset)
;	ES:DI					(video memory pointer)
;	DX = 3CEh				(when planar?)
; Out:	none
; Use:	maxcoord.X, screenmask, cursormask
; Modf: AX, BX, SI, DI
; Call: none
;
makeno13	proc
		call	@@makeno13
		call	@@makeno13

@@makeno13:	cmp	si,[maxcoord.X]
		jae	@@nextxno13		; jump if X > max || X < 0

		IFnz	[planar?],@@planar
		mov	al,es:[di]
		and	al,screenmask[bx]
		xor	al,cursormask[bx]
		mov	es:[di],al
		j	@@nextxno13

@@planar:	mov	ax,0803h		; data ANDed with latched data
		out	dx,ax
		mov	al,screenmask[bx]
		xchg	es:[di],al
		mov	ax,1803h		; data XORed with latched data
		out	dx,ax
		mov	al,cursormask[bx]
		xchg	es:[di],al

@@nextxno13:	add	si,8
		inc	bx
		inc	di
		ret
makeno13	endp

;
;		Wait for video retrace in text modes
;
;
; In:	ES					(video segment)
; Out:	none
; Use:	@videoregs[(SIZE RGROUPDEF)*5].@port
; Modf: AX, CX, DX
; Call: none
;
waitretrace	proc
		mov	ax,es
		cmp	ax,0B800h
		jne	@@waitret

		xor	cx,cx
		mov	dx,@videoregs[(SIZE RGROUPDEF)*5].@port
						; {3DAh}/{3BAh}
@@waitstop:	in	al,dx			; wait until retrace stop
		test	al,1
		loopnz	@@waitstop

		xor	cx,cx
@@waitstart:	in	al,dx			; wait until retrace start
		test	al,9
		loopz	@@waitstart
@@waitret:	ret
waitretrace	endp

;
;			Restore old screen content
;

restoregraph	proc
		call	updatevregs
		;j	restoreoldscr
restoregraph	endp

restoreoldscr	proc
		xor	cx,cx
		xchg	cl,[cursordrawn]	; clear indicator
		jcxz	@@restret		; jump if no cursor drawn

		les	di,[cursorptr]
		mov	cx,[nextrow]
		mov	ax,granu.Y[0]
		cmp	al,1			; text mode?
		je	copysprite		; jump if no

		IFnz	[cursortype],@@restret	; exit if hardware cursor
		cli
		call	waitretrace
		mov	ax,word ptr [buffer]
		mov	es:[di],ax		; restore old text char attrib
		sti
@@restret:	ret
restoreoldscr	endp

;
;		Copy screen sprite back and forth
;
;
; In:	AL					(0-screen src./1-screen dst.)
;	ES:DI					(video memory pointer)
;	CX					(offset to next row)
; Out:	none
; Use:	ptr2buffer, cursorcoord, maxcoord, nextxor, spritewidth
; Modf: AX, BX, CX, DX, SI, DI, ES
; Call: none
;
copysprite	proc
		push	ds
		mov	bx,[cursorcoord.Y]
		mov	ah,16
		lds	si,[ptr2buffer]
		or	al,al
		jnz	@@copyloop
		xchg	si,di
		push	es ds
		pop	es ds

@@copyloop:	cmp	bx,cs:[maxcoord.Y]
		jae	@@copynext1		; jump if Y > max || Y < 0

		push	cx si di
		mov	dx,cs:[cursorcoord.X]
		mov	cx,cs:[spritewidth]
@@copyloopx:	cmp	dx,cs:[maxcoord.X]
		jae	@@copynextx		; jump if X > max || X < 0
		movsb
		dec	si
		dec	di
@@copynextx:	inc	dx
		inc	si
		inc	di
		loop	@@copyloopx
		pop	di si cx

@@copynext1:	inc	bx
		or	al,al
		jnz	@@copynext2
		xchg	si,di
@@copynext2:	add	di,cx
		xor	cx,cs:[nextxor]
		add	si,cs:[spritewidth]
		or	al,al
		jnz	@@copynext3
		xchg	si,di
@@copynext3:	dec	ah
		jnz	@@copyloop
		pop	ds
		ret
copysprite	endp

;

scrollmask	proc
		mov	al,byte ptr [cursorcoord.X]
		and	al,7
		mov	bl,[Xcheck]
		mov	[Xcheck],al
		mov	si,offset cursormask
		sub	al,bl
		jz	@@noscroll
		jl	@@scrolleft
		call	@@rscroll
		mov	si,offset screenmask

@@rscroll:	mov	bl,al

@scrn:		mov	di,si
		mov	cx,30h

@scrloop:	rcr	byte ptr [di],1
		inc	di
		loop	@scrloop

		jnc	@scrc
		or	byte ptr [si],80h
		j	@scrnc
@scrc:		and	byte ptr [si],not 80h
@scrnc:		dec	bl
		jnz	@scrn
@@noscroll:	ret
;------------
@@scrolleft:	neg	al
		call	@@lscroll
		mov	si,offset screenmask

@@lscroll:	mov	bl,al
		add	si,30h

@scln:		mov	di,si
		mov	cx,30h

@sclloop:	dec	di
		rcl	byte ptr [di],1
		loop	@sclloop

		mov	di,si
		jnc	@sclc
		or	byte ptr [di-1],1
		j	@sclnc
@sclc:		and	byte ptr [di-1],not 1
@sclnc:		dec	bl
		jnz	@scln
		ret
scrollmask	endp

;
;		Return graphic mode video memory offset
;
;
; In:	AX					(Y coordinate in pixels)
;	DI					(X coordinate in pixels)
; Out:	ES:DI					(video memory pointer)
;	CX					(offset to next row)
; Use:	videomode
; Modf: AX, BX, DX, nextrow, nextxor
; Call: @getoffsret
;
getgroffset	proc
;
;4/5 (320x200x4)   byte offset = (y%2)*2000h + (y/2)*80 + (x*2)/8,
;			low bit (of pair) offset = 6 - (x*2)%8
;  6 (640x200x2)   byte offset = (y%2)*2000h + (y/2)*80 + x/8,
;			bit offset = 7 - (x % 8)
;0Dh (320x200x16)  byte offset = y*40 + x/8, bit offset = 7 - (x % 8)
;0Eh (640x200x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;0Fh (640x350x4)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;10h (640x350x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;11h (640x480x2)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;12h (640x480x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;13h (320x200x256) byte offset = y*320 + x
;HGC (720x348x2)   byte offset = (y%4)*2000h + (y/4)*90 + x/8
;			bit offset = 7 - (x % 8)
;
		mov	bx,320			; length of scanline
		cmp	[videomode],13h-SHIFTHIVMODE
		je	@@interlace
		mov	cl,3
		sar	di,cl			; DI=X/8
		mov	bx,80
		mov	cl,[shift320]
		shr	bx,cl

@@interlace:	mov	cx,bx			; [nextrow]
		xor	dx,dx			; [nextxor]
		cmp	[videomode],6		; is videomode 4-6?
		ja	@@ymul			; jump if not
		mov	cx,2000h
		mov	dx,2000h xor -(2000h-80)
		sar	ax,1			; AX=Y/2
		jnc	@@ymul
		add	di,cx			; DI=X/8+(Y%2)*2000h
		xor	cx,dx

@@ymul:		mov	[nextrow],cx
		mov	[nextxor],dx
		mul	bx
		add	di,ax
		j	@getoffsret
getgroffset	endp

;
;		Return text mode video memory offset
;
;
; In:	none
; Out:	ES:DI					(video memory pointer)
; Use:	0:44Ah, 0:44Eh, granpos, granu
; Modf: AX, DX, cursorptr
; Call: getpageoffset
;
gettxtoffset	proc
;
;0/1 (40x25) byte offset = (y*40 + x)*2 = (y*0:44Ah + x)*2
;2/3 (80x25) byte offset = (y*80 + x)*2 = (y*0:44Ah + x)*2
;  7 (80x25) byte offset = (y*80 + x)*2 = (y*0:44Ah + x)*2
;
		mov	ax,[granpos.Y]
		xor	dx,dx
		mov	es,dx
		div	[granu.Y]		; calculate Y in characters
		mul	word ptr es:[44Ah]	; screen width
		mov	di,ax
		mov	ax,[granpos.X]
		;xor	dx,dx
		div	[granu.X]		; calculate X in characters
		add	di,ax
		shl	di,1

@getoffsret:	xor	ax,ax
		mov	es,ax
		add	di,es:[44Eh]		; add page offset
		mov	es,word ptr cursorptr[2]
		mov	word ptr cursorptr[0],di
		ret
gettxtoffset	endp

;
;			Check update region
;

;+*checkregion	proc
;+*		IFz	[regionchk?],@@regexit	; have to check region?
;+*						; jump if not
;+*		dec	cx
;+*		dec	dx
;+*		cmp	ax,[upleft.Y]
;+*		jg	@@regexit
;+*		cmp	di,[upleft.X]
;+*		jg	@@regexit
;+*		cmp	cx,[lowright.Y]
;+*		jl	@@regexit
;+*		cmp	dx,[lowright.X]
;+*		jl	@@regexit
;+*		stc
;+*		ret
;+*@@regexit:	clc
;+*		ret
;+*checkregion	endp

;
;		Setup video regs values for current video mode
;
;
; In:	none
; Out:	none
; Use:	0:449h, 0:463h, 0:487h, 0:488h, 0:4A8h; buffer
; Modf: *
; Call: @setcoords
;
setupvideo	proc
		xor	ax,ax
		mov	ds,ax
		mov	ax,ds:[463h]		; CRTC base I/O port address
		mov	cs:@videoregs[0].@port,ax ; 3D4h/3B4h
		add	ax,6
		mov	cs:@videoregs[(SIZE RGROUPDEF)*5].@port,ax
		mov	al,ds:[449h]		; current video mode
		and	al,not 80h
		push	ax

		cmp	al,11h			; VGA videomodes?
		jb	@@checkv1
		add	al,9
		j	@@getshift

@@checkv1:	cmp	al,0Fh			; 0F-10 videomodes?
		jb	@@checkv2
		test	byte ptr ds:[487h],60h	; check RAM size on adapter
		jz	@@getshift
		add	al,2
		j	@@getshift

@@checkv2:	cmp	al,4			; not color text modes?
		jae	@@getshift
		mov	ah,ds:[488h]		; get display combination
		and	ah,1Fh
		cmp	ah,3			; MDA+EGA/ECD?
		je	@@lines350
		cmp	ah,9			; EGA/ECD+MDA?
		jne	@@getshift
@@lines350:	add	al,13h

@@getshift:	les	si,dword ptr ds:[4A8h]
		les	si,dword ptr es:[si]
		mov	ah,0
		mov	cl,6
		shl	ax,cl
		add	al,5			; OPTIMIZE: AL instead AX
		add	si,ax			; SI += AL*64+5

		movSeg	ds,es
		movSeg	es,cs
		mov	di,offset StartDefVRegsArea
		push	di
		mov	al,3
		stosb
		mov	cx,50/2
		rep	movsw			; copy default registers value
		inc	di
		movsw
		movsw
		movsw
		movsw
		movsb

		movSeg	ds,cs
		pop	si
		mov	di,offset StartVRegsArea
		mov	cx,LenVRegsArea/2
		rep	movsw			; copy current registers value

		mov	al,0
		mov	@videoregs[(SIZE RGROUPDEF)*0].rmodify?,al
		mov	@videoregs[(SIZE RGROUPDEF)*1].rmodify?,al
		mov	@videoregs[(SIZE RGROUPDEF)*2].rmodify?,al
		mov	@videoregs[(SIZE RGROUPDEF)*3].rmodify?,al
		mov	[singlemodify?],al
		mov	[reg_GPOS1],al
		mov	[def_GPOS1],al
		inc	ax			; OPTIMIZE: AX instead AL
		mov	[reg_GPOS2],al
		mov	[def_GPOS2],al

;------------ set parameters for current video mode
; mode	 seg   screen  cell  planar
;  0	B800h  640x200 16x8	-
;  1	B800h  640x200 16x8	-
;  2	B800h  640x200	8x8	-
;  3	B800h  640x200	8x8	-
;  4	B800h  640x200	2x1	0
;  5	B800h  640x200	2x1	0
;  6	B800h  640x200	1x1	0
;  7	B000h  640x200	8x8	-
; 0Dh	A000h  320x200	2x1	1
; 0Eh	A000h  640x200	1x1	1
; 0Fh	A000h  640x350	1x1	1
; 10h	A000h  640x350	1x1	1
; 11h	A000h  640x480	1x1	1
; 12h	A000h  640x480	1x1	1
; 13h	A000h  320x200	2x1	0
;
		pop	ax			; current video mode
; mode 0-3
		mov	ah,0B8h			; B800h: 0-6
		mov	[maxcoord.X],640
		mov	di,200			; x200: 0-7, 0Dh-0Eh, 13h
		mov	dx,0810h		; 16x8: 0-1
		mov	cx,300h			; sprite: 3 bytes/row; 640x
		mov	[planar?],cl
		mov	si,cs
		mov	bx,offset buffer
		cmp	al,2
		jb	@@setvmode
		mov	dl,8			; 8x8: 2-3, 7
		cmp	al,4
		jb	@@setvmode
; mode 7
		mov	ah,0B0h			; B000h: 7
		cmp	al,7
		je	@@setvmode
; mode 4-6
		mov	ah,0B8h
		mov	dx,0101h		; 1x1: 6, 0Eh-12h
		cmp	al,6
		je	@@setvmode
		jb	@@set2x1
; mode 13h
; for modes 0Dh-13h we'll be storing the cursor shape and the hided
; screen contents at first free byte in vidmem to save mem
		sub	al,SHIFTHIVMODE
		mov	ah,0A0h			; A000h: 0Dh-
		mov	si,0A000h
		mov	bx,0FA00h
		cmp	al,13h-SHIFTHIVMODE
		jb	@@mode0812
		ja	@@nonstandard
		mov	ch,16
@@set320:	inc	cx			; OPTIMIZE: CX instead CL
@@set2x1:	inc	dx			; OPTIMIZE: DX instead DL
		j	@@setvmode
; mode 8-0Dh
@@mode0812:	cmp	al,0Dh-SHIFTHIVMODE
		jb	@@nonstandard
		mov	bx,3E82h		; 16002: 0Dh-0Eh
		mov	[planar?],bl		; set for modes 0Dh-12h
		je	@@set320
; mode 0Eh-12h
		cmp	al,0Fh-SHIFTHIVMODE
		jb	@@setvmode
		mov	di,350			; x350: 0Fh-10h
		add	bh,40h			; 32386: 0Fh-10h
		cmp	al,11h-SHIFTHIVMODE
		jb	@@setvmode
		mov	di,480			; x480: 11h-12h
		add	bh,20h			; 40578: 11h-12h
		j	@@setvmode
;
@@nonstandard:	;mov	dx,0101h
		;mov	cl,0
;+++++ for nonstandard textual modes set:
;+++++ [maxcoord.X] := 8*[0:44Ah], di := 8*([0:484h]+1), dx := 0808h
		mov	al,14h-SHIFTHIVMODE
;
@@setvmode:	mov	[videomode],al
		mov	byte ptr cursorptr[3],ah
		shr	[maxcoord.X],cl
		mov	[maxcoord.Y],di
		mov	byte ptr granu.X[0],dl
		mov	byte ptr granu.Y[0],dh
		mov	[shift320],cl
		mov	byte ptr spritewidth[0],ch
		saveFAR	ptr2buffer,si,bx

;------------ set parameters for ranges and cursor position
@@setuprange:	mov	cx,LenInitArea2/2	; clear area 2
		j	@setcoords
setupvideo	endp


;
;			INT 33 handler services
;

_ARG_ES_	equ	word ptr [bp]
_ARG_DS_	equ	word ptr [bp+2]
_ARG_BP_	equ	word ptr [bp+4]
_ARG_DI_	equ	word ptr [bp+6]
_ARG_SI_	equ	word ptr [bp+8]
_ARG_DX_	equ	word ptr [bp+10]
_ARG_CX_	equ	word ptr [bp+12]
_ARG_BX_	equ	word ptr [bp+14]
_ARG_AX_	equ	word ptr [bp+16]

;
; 00 - Reset driver
;
;
; In:	none
; Out:	[AX] = 0/FFFFh				(not installed/installed)
;	[BX] = 0/2/3/FFFFh			(number of buttons)
; Use:	none
; Modf: none
; Call: enabledriver_20, softreset_21
;
resetdriver_00	proc
		call	enabledriver_20
		;j	softreset_21
resetdriver_00	endp

;
; 21 - Software reset
;
;
; In:	none
; Out:	[AX] = 21h/FFFFh			(not installed/installed)
;	[BX] = 0/2/3/FFFFh			(number of buttons)
; Use:	mousetype, shape, maxcoord, shift320, granu
; Modf: *
; Call: restorescreen, setusershape
;
softreset_21	proc
IFNDEF PS2
		mov	al,[mousetype]
ELSE
		mov	al,2
ENDIF
		cbw
		mov	[_ARG_BX_],ax
		mov	[_ARG_AX_],0FFFFh

		mov	[cursorhidden],1
		call	restorescreen
		mov	[startscan],77FFh
		mov	[endscan],7700h
		mov	[mickey8.X],8
		mov	[mickey8.Y],16
		mov	[doublespeed],64
		mov	si,offset shape
		call	setusershape
		mov	cx,LenInitArea3/2	; clear area 3

@setcoords:	mov	ax,[maxcoord.Y]		; set lower range
		mov	bx,ax
		dec	ax
		mov	[rangemax.Y],ax
		mov	ax,[maxcoord.X]		; set right range
		dec	ax
		push	cx
		mov	cl,[shift320]
		shl	ax,cl
		pop	cx
		mov	[rangemax.X],ax
		shr	bx,1
		shr	ax,1
		inc	ax

@setpos:	cli
		mov	[pos.X],ax
		xor	dx,dx
		div	[granu.X]
		mul	[granu.X]
		mov	[granpos.X],ax

		mov	ax,bx
		mov	[pos.Y],ax
		;xor	dx,dx
		div	[granu.Y]
		mul	[granu.Y]
		mov	[granpos.Y],ax

		movSeg	es,cs
		mov	di,offset StartInitArea
		xor	ax,ax
		rep	stosw
		ret
softreset_21	endp

;
; 1F - Disable mouse driver
;
;
; In:	none
; Out:	[AX] = 1Fh/FFFFh			(success/unsuccess)
;	[ES:BX]					(old int33 handler)
; Use:	oldint33, IRQintnum, oldIRQaddr, oldint10
; Modf: AX, DX, ES, disabled?, cursorhidden
; Call: Int21/25, restorescreen, disableIRQ
;
disabledrv_1F	proc
		les	ax,[oldint33]
		mov	[_ARG_BX_],ax
		mov	[_ARG_ES_],es

		mov	al,1
		cmp	[disabled?],al
		je	@ret
		mov	[disabled?],al
		mov	[cursorhidden],al
		call	restorescreen
		call	disableIRQ

;------------ restore old IRQ handler
		mov	ax,word ptr [IRQintnum]
		push	ds
		lds	dx,[oldIRQaddr]
		int	21h
		pop	ds

;------------ restore old INT 10 handler
		mov	ax,2510h
		push	ds
		lds	dx,[oldint10]
		int	21h
		pop	ds
;------------ 
@ret:		ret
disabledrv_1F	endp

;
; 20 - Enable mouse driver
;
;
; In:	none
; Out:	[AX] = 20h/FFFFh			(success/unsuccess)
; Use:	IRQintnum
; Modf: AX, DX, disabled?, IOdone
; Call: Int21/25, setupvideo, enableIRQ
;
enabledriver_20	proc
		cli
		mov	al,0
		mov	[IOdone],al
		cmp	[disabled?],al
		je	@@enabled
		mov	[disabled?],al

;------------ set new INT 10 handler
		mov	ax,2510h
		mov	dx,offset int10handler
		int	21h

;------------ set new IRQ handler
		mov	ax,word ptr [IRQintnum]
		mov	dx,offset IRQhandler
		int	21h
;------------
@@enabled:	call	setupvideo
		jmp	enableIRQ
enabledriver_20	endp

;
; 01 - Show mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: AL, regionchk?, cursorhidden
; Call: showcursor
;
showcursor_01	proc
		mov	al,0
		mov	[regionchk?],al
		cmp	[cursorhidden],al
		je	@ret			; jump if already shown
		dec	[cursorhidden]
		jmp	showcursor
showcursor_01	endp

;
; 02 - Hide mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: cursorhidden
; Call: restorescreen
;
hidecursor_02	proc
		inc	[cursorhidden]
		jnz	@@restore
		dec	[cursorhidden]
@@restore:	jmp	restorescreen
hidecursor_02	endp

;
; 03 - Return position and button status
;
;
; In:	none
; Out:	[BX]					(buttons status)
;	[CX]					(column)
;	[DX]					(row)
; Use:	buttstatus, granpos
; Modf: AX
; Call: none
;
status_03	proc
		mov	ax,[buttstatus]
		mov	[_ARG_BX_],ax
		mov	ax,[granpos.X]
		mov	[_ARG_CX_],ax
		mov	ax,[granpos.Y]
		mov	[_ARG_DX_],ax
		ret
status_03	endp

;
; 05 - Return button press data
;
;
; In:	BX					(button number)
; Out:	[AX]					(buttons states)
;	[BX]					(press times)
;	[CX]					(last press column)
;	[DX]					(last press row)
; Use:	buttpress
; Modf: AX
; Call: @retbuttstat
;
butpresdata_05	proc
		mov	ax,offset buttpress
		j	@retbuttstat
butpresdata_05	endp

;
; 06 - Return button release data
;
;
; In:	BX					(button number)
; Out:	[AX]					(buttons states)
;	[BX]					(release times)
;	[CX]					(last release column)
;	[DX]					(last release row)
; Use:	butrelease, buttstatus
; Modf: AX, BX
; Call: none
;
buttreldata_06	proc
		mov	ax,offset buttrelease
@retbuttstat:	and	bx,7
ERRIF (6 ne SIZE BUTTLASTSTATE) "BUTTLASTSTATE structure size changed!"
		shl	bx,1
		add	ax,bx
		shl	bx,1
		add	bx,ax			; =AX+BX*SIZE BUTTLASTSTATE
		xor	ax,ax
		xchg	[bx.counter],ax
		mov	[_ARG_BX_],ax
		mov	ax,[bx.lastrow]
		mov	[_ARG_DX_],ax
		mov	ax,[bx.lastcol]
		mov	[_ARG_CX_],ax
		mov	ax,[buttstatus]
		mov	[_ARG_AX_],ax
		ret
buttreldata_06	endp

;
; 07 - Define horizontal cursor range
;
;
; In:	CX					(min column)
;	DX					(max column)
; Out:	none
; Use:	pos
; Modf: CX, DX, rangemin.X, rangemax.X
; Call: setpos_04
;
hrange_07	proc
		cmp	dx,cx
		jb	@@swminmaxx
		xchg	dx,cx
@@swminmaxx:	mov	[rangemin.X],dx
		mov	[rangemax.X],cx
@resetpos:	mov	dx,[pos.Y]
		mov	cx,[pos.X]
		;j	setpos_04
hrange_07	endp

;
; 04 - Position mouse cursor
;
;
; In:	CX					(column)
;	DX					(row)
; Out:	none
; Use:	rangemin, rangemax
; Modf: AX, BX
; Call: cutrangex, cutrangey, @setpos, @show
;
setpos_04	proc
		mov	ax,dx
		call	cutrangey
		mov	bx,ax
		mov	ax,cx
		call	cutrangex
		mov	cx,LenInitArea1/2	; clear area 1
		call	@setpos
		j	@show
setpos_04	endp

;
; 08 - Define vertical cursor range
;
;
; In:	CX					(min row)
;	DX					(max row)
; Out:	none
; Use:	none
; Modf: CX, DX, rangemin.Y, rangemax.Y
; Call: @resetpos
;
vrange_08	proc
		cmp	dx,cx
		jb	@@swminmaxy
		xchg	dx,cx
@@swminmaxy:	mov	[rangemin.Y],dx
		mov	[rangemax.Y],cx
		j	@resetpos
vrange_08	endp

;
; 09 - Define graphics cursor
;
;
; In:	BX					(hot spot column)
;	CX					(hot spot row)
;	ES:DX					(pointer to bitmaps)
; Out:	none
; Use:	none
; Modf: SI, hotspot
; Call: restorescreen, setusershape, showcursor
;
graphcursor_09	proc
		mov	[hotspot.X],bx
		mov	[hotspot.Y],cx
		movSeg	ds,es
		mov	si,dx
		call	setusershape
		movSeg	ds,cs
		call	restorescreen
@show:		jmp	showcursor
graphcursor_09	endp

;
; 0A - Define text cursor
;
;
; In:	CX					(screen mask/start scanline)
;	DX					(cursor mask/end scanline)
;	BX					(0=SW/1=HW text cursor)
; Out:	none
; Use:	none
; Modf: AX, CX, DX, cursortype, startscan, endscan
; Call: Int10/1, @show
;
textcursor_0A	proc
		push	bx cx dx
		call	restorescreen
		pop	cx dx ax
		mov	[cursortype],al
		mov	[startscan],dx
		mov	[endscan],cx
		or	al,al			; software cursor?
		jz	@show

		mov	ch,dl
		mov	ah,1
		int	10h			; set cursor shape & size
@ret2:		ret
textcursor_0A	endp

;
; 0B - Read motion counters
;
;
; In:	none
; Out:	[CX]			(number of mickeys mouse moved
;	[DX]			 horizontally/vertically since last call)
; Use:	none
; Modf: AX, mickey
; Call: none
;
readmcounter_0B	proc
		xor	ax,ax
		xchg	[mickey.X],ax
		mov	[_ARG_CX_],ax
		xor	ax,ax
		xchg	[mickey.Y],ax
		mov	[_ARG_DX_],ax
		ret
readmcounter_0B	endp

;
; 0D - Light pen emulation On
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: nolightpen?
; Call: lightpenoff_0E
;
lightpenon_0D	proc
		mov	[nolightpen?],1
		;j	lightpenoff_0E
lightpenon_0D	endp

;
; 0E - Light pen emulation Off
;
;
; In:	none
; Out:	none
; Use:	none
; Modf: nolightpen?
; Call: none
;
lightpenoff_0E	proc
		dec	[nolightpen?]
		ret
lightpenoff_0E	endp

;
; 0F - Define Mickey/Pixel ratio
;
;
; In:	CX					(number of mickeys per 8 pix
;	DX					 horizontally/vertically)
; Out:	none
; Use:	none
; Modf: mickey8
; Call: none
;
micperpixel_0F	proc
		mov	[mickey8.X],cx
		mov	[mickey8.Y],dx
		ret
micperpixel_0F	endp

;
; 10 - Define screen region for updating
;
;
; In:	CX, DX					(X/Y of upper left corner)
;	SI, DI					(X/Y of lower right corner)
; Out:	none
; Use:	none
; Modf: CX, DX, SI, DI, upleft, lowright, regionchk?
; Call: @show
;
defregion_10	proc
		cmp	si,cx
		jb	@@nosw1
		xchg	si,cx
@@nosw1:	cmp	di,dx
		jb	@@nosw2
		xchg	di,dx
@@nosw2:	mov	[upleft.X],cx
		mov	[lowright.X],si
		mov	[upleft.Y],dx
		mov	[lowright.Y],di
		mov	[regionchk?],1
		j	@show
defregion_10	endp

;
; 14 - Exchange interrupt subroutines
;
;
; In:	CX					(new call mask)
;	ES:DX					(new FAR routine)
; Out:	[CX]					(old call mask)
;	[ES:DX]					(old FAR routine)
; Use:	callmask, userproc
; Modf: AX
; Call: intpar_0C
;
exchangeint_14	proc
		mov	ax,[callmask]
		mov	[_ARG_CX_],ax
		mov	ax,word ptr userproc[0]
		mov	[_ARG_DX_],ax
		mov	ax,word ptr userproc[2]
		mov	[_ARG_ES_],ax
		;j	intpar_0C
exchangeint_14	endp

;
; 0C - Define interrupt subroutine parameters
;
;
; In:	CX					(call mask)
;	ES:DX					(FAR routine)
; Out:	none
; Use:	none
; Modf: userproc, callmask
; Call: none
;
intpar_0C	proc
		saveFAR [userproc],es,dx
		mov	[callmask],cx
		ret
intpar_0C	endp

;
; 15 - Return driver storage requirements
;
;
; In:	none
; Out:	[BX]					(buffer size)
; Use:	none
; Modf: none
; Call: none
;
storagereq_15	proc
		mov	[_ARG_BX_],LenSaveArea
		ret
storagereq_15	endp

;
; 16 - Save driver state
;
;
; In:	BX					(buffer size)
;	ES:DX					(buffer)
; Out:	none
; Use:	StartSaveArea
; Modf: CX, SI, DI
; Call: none
;
savestate_16	proc
		mov	di,dx
		mov	si,offset StartSaveArea
@saveareamove:	cmp	bx,LenSaveArea
		jb	@ret2
		mov	cx,LenSaveArea/2
		rep	movsw
		ret
savestate_16	endp

;
; 17 - Restore driver state
;
;
; In:	BX					(buffer size)
;	ES:DX					(saved state buffer)
; Out:	none
; Use:	none
; Modf: SI, DI, DS, ES, StartSaveArea
; Call: @saveareamove
;
restorestate_17	proc
		push	ds es
		pop	ds es
		mov	si,dx
		mov	di,offset StartSaveArea
		j	@saveareamove
restorestate_17	endp

;
; 1A - Set mouse sensitivity
;
;
; In:	BX				(number of mickeys per 8 pix
;	CX				 horizontally/vertically)
;	DX				(threshold speed in mickeys/second)
; Out:	none
; Use:	none
; Modf: mickey8
; Call: doublespeed_13
;
setsens_1A	proc
		mov	[mickey8.X],bx
		mov	[mickey8.Y],cx
		;j	doublespeed_13
setsens_1A	endp

;
; 13 - Define double speed threshold
;
;
; In:	DX				(threshold speed in mickeys/second)
; Out:	none
; Use:	none
; Modf: /AX/, /BX/, /DX/, doublespeed
; Call: none
;
doublespeed_13	proc
;+*		or	dx,dx
;+*		jz	@@savespeed
;+*		mov	dx,64
;+*@@savespeed:
;*		mov	ax,dx
;*		xor	dx,dx
;*		mov	bx,30			;*** ??? ***
;*		div	bx
;*		mov	[doublespeed],ax
		mov	[doublespeed],dx
		ret
doublespeed_13	endp

;
; 1B - Return mouse sensitivity
;
;
; In:	none
; Out:	[BX]				(number of mickeys per 8 pix
;	[CX]				 horizontally/vertically)
;	[DX]				(threshold speed in mickeys/second)
; Use:	mickey8, doublespeed
; Modf: AX, DX
; Call: none
;
getsens_1B	proc
		mov	ax,[mickey8.X]
		mov	[_ARG_BX_],ax
		mov	ax,[mickey8.Y]
		mov	[_ARG_CX_],ax
;*		mov	ax,30			;*** ??? ***
;*		mul	[doublespeed]
		mov	ax,[doublespeed]
		mov	[_ARG_DX_],ax
		ret
getsens_1B	endp

;
; 1E - Return display page number
;
;
; In:	none
; Out:	[BX]					(display page number)
; Use:	0:462h
; Modf: AX, ES
; Call: none
;
getpage_1E	proc
		xor	ax,ax
		mov	es,ax
		mov	al,es:[462h]
		mov	[_ARG_BX_],ax
		ret
getpage_1E	endp

;
; 24 - Get software version, mouse type and IRQ
;
;
; In:	none
; Out:	[AX] = 24h/FFFFh			(error/installed)
;	[BX]					(version)
;	[CH] = 1=bus/serial/InPort/PS2/HP	(mouse type)
;	[CL] = 0=PS/2				(interrupt #)
; Use:	driverversion, IRQintnum
; Modf: AX
; Call: none
;
getversion_24	proc
		mov	[_ARG_BX_],driverversion ; version number
IFNDEF PS2
		mov	ax,word ptr [IRQintnum]	; OPTIMIZE: AX instead AL
		sub	ax,2308h		; AH=2=serial mouse
		mov	[_ARG_CX_],ax
ELSE
		mov	[_ARG_CX_],400h		; PS/2 mouse
ENDIF
		ret
getversion_24	endp

;
; 26 - Get maximum virtual coordinates
;
;
; In:	none
; Out:	[BX]					(mouse disabled flag)
;	[CX]					(maximum virtual X)
;	[DX]					(maximum virtual Y)
; Use:	disabled?, maxcoord
; Modf: AX
; Call: none
;
getmaxvirt_26	proc
		mov	al,[disabled?]
		cbw
		mov	[_ARG_BX_],ax
		mov	ax,[maxcoord.X]
		dec	ax
		mov	[_ARG_CX_],ax
		mov	ax,[maxcoord.Y]
		dec	ax
		mov	[_ARG_DX_],ax

;
; 11, 18, 19, 1C, 22, 23, 25 - Null function for not implemented calls
;

nullfunc	proc
		ret
nullfunc	endp

getmaxvirt_26	endp

;
;				INT 33 handler
;

funcsoffsets	dw	offset resetdriver_00
		dw	offset showcursor_01
		dw	offset hidecursor_02
		dw	offset status_03
		dw	offset setpos_04
		dw	offset butpresdata_05
		dw	offset buttreldata_06
		dw	offset hrange_07
		dw	offset vrange_08
		dw	offset graphcursor_09
		dw	offset textcursor_0A
		dw	offset readmcounter_0B
		dw	offset intpar_0C
		dw	offset lightpenon_0D
		dw	offset lightpenoff_0E
		dw	offset micperpixel_0F
		dw	offset defregion_10
		dw	offset nullfunc		;11 - genius driver only
		dw	offset nullfunc		;12 - large graphics cursor
		dw	offset doublespeed_13
		dw	offset exchangeint_14
		dw	offset storagereq_15
		dw	offset savestate_16
		dw	offset restorestate_17
		dw	offset nullfunc		;18 - set alternate handler
		dw	offset nullfunc		;19 - return alternate address
		dw	offset setsens_1A
		dw	offset getsens_1B
		dw	offset nullfunc		;1C - InPort mouse only
		dw	offset nullfunc		;1D - define display page #
		dw	offset getpage_1E
		dw	offset disabledrv_1F
		dw	offset enabledriver_20
		dw	offset softreset_21
		dw	offset nullfunc		;22 - set language
		dw	offset nullfunc		;23 - get language
		dw	offset getversion_24
		dw	offset nullfunc		;25 - get general information
		dw	offset getmaxvirt_26

handler33	proc	far
		cmp	ax,26h			; is it implemented?
		ja	@@checkother		; jump if not

		cld
		push	ax bx cx dx si di bp ds es
		mov	bp,sp
		movSeg	ds,cs
		shl	ax,1
		mov	si,ax
		call	funcsoffsets[si]	; call by calculated offset
@rethandler:	pop	es ds bp di si dx cx bx ax
@@iret:		iret

@@checkother:	cmp	ax,4dh			; copyright string query?
		je	@@verstr
		cmp	ax,6dh			; version string query?
		jne	@@iret

;
; 6D - Return pointer to version string
;
;
; In:	none
; Out:	[ES:DI]					(version string)
; Use:	msversion
; Modf:	none
; Call: none
;
@@msver:	mov	di,offset msversion
		j	@@retstr

;
; 4D - Return pointer to copyright string
;
;
; In:	none
; Out:	[ES:DI]					(copyright string)
; Use:	IDstring
; Modf: none
; Call: none
;
@@verstr:	mov	di,offset IDstring
@@retstr:	movSeg	es,cs
		iret
handler33	endp

; END OF INT 33 SERVICES 


;==========================================================================
;------------------------ Below is not resident --------------------------


NonTSRpart	label

Copyright	db	'CuteMouse v1.71/000630',0dh,0ah,'$'
IFNDEF PS2
S_info		db	'Installed on: $'
S_atCOM		db	'COM '
com_port	db	0,' for $'
S_drvMS		db	'Microsoft mouse$'
S_drvMouse	db	'Mouse Systems mouse$'
S_drvLogi	db	'Logitech MouseMan$'
E_ps2not	db	'Error: PS/2 not supported',0dh,0ah,'$'
ELSE
S_info		db	'Installed on: $'
S_drvPS2	db	'PS/2 port$'
ENDIF

S_released	db	'Mouse driver has been released from memory',0dh,0ah,'$'
S_notinst	db	'CuteMouse driver is not installed!',0dh,0ah,'$'
S_already	db	'Mouse driver already installed',0dh,0ah,'$'
S_resolution	db	0dh,0ah,'Resolution: $'
S_auto		db	'Auto',0dh,0ah,'$'
S_times		db	'3/3 times horizontally/vertically',0dh,0ah,'$'
E_option	db	0dh,0ah,'Error: Invalid parameter',0dh,0ah
		db	'Enter /? on command line for help',0dh,0ah,'$'
E_find		db	'Error: device not found',0dh,0ah,'$'

Syntax		db	'Options:',0dh,0ah
IFNDEF PS2
		db	'  /n[m]	- force COM port to n and IRQ line to m (n=1-4, m=2-7;',0dh,0ah
		db	'	  default is COM1, IRQ4 for COM1/3 and IRQ3 for COM2/4)',0dh,0ah
		db	'  /M	- Force Microsoft mode (2 buttons)',0dh,0ah
		db	'  /S	- Force Mouse Systems mode (3 buttons)',0dh,0ah
		db	'  /T	- Force Logitech MouseMan mode (3 buttons)',0dh,0ah
ENDIF
		db	'  /R[0]	- Auto hardware resolution',0dh,0ah
		db	'  /Rnm	- resolution horizontally/vertically (n,m=1-9; default is R33)',0dh,0ah
		db	'  /L	- Left hand mode (default is right hand mode)',0dh,0ah
		db	'  /U	- Release driver',0dh,0ah
		db	'  /?	- Show this help',0dh,0ah,'$'

ctstr		db	'CTMOUSE '
paragraphs	dw	0
dest_seg	dw	0
XMSentry	dd	0

; Real Start 

real_start:	cld
		PrintS	Copyright		; 'Cute Mouse Driver'
		call	commandline		; examine command line
IFNDEF PS2
		IFnz	[com_port],@@COMforced
		mov	al,'1'
		call	setCOMport
		IFnz	[mousetype],@@COMforced
		int	11h			; check for PS/2
		test	al,4			; pointing device installed?
		jz	@@COMforced
		mov	bh,3
		PS2serv 0C205h,@@COMforced	; initalize mouse
		mov	dx,offset E_ps2not	; 'Error: PS/2 not supported'
		jmp	EXITERRMSG

@@COMforced:	call	COMexist?		; check COM port
		jc	@@devnotfound		; jump if not available

;------------ determine mouse type
		IFnz	[mousetype],@@lastcheck	; jump if mode forced
		mov	[mousetype],2		; Set Microsoft mouse type
		call	checktype
		jnc	@@typefound
		inc	[mousetype]		; Set Mouse systems mode
@@lastcheck:	call	checktype
		jnc	@@typefound
;------------
ELSE
		call	checkPS2		; check PS2
		IFnz	[mousetype],@@typefound
ENDIF
@@devnotfound:	mov	dx,offset E_find	; 'Error: device not found'
		jmp	EXITERRMSG

@@typefound:	call	getCuteSeg		; check if driver installed
		je	changeparam		; change params and exit

;------------ save old INT 33h handler
		mov	ax,3533h
		int	21h
		mov	ax,es
		saveFAR [oldint33],ax,bx
		or	ax,bx
		jz	@@nodrv
		mov	ax,1Fh
		int	33h			; disable driver
@@nodrv:
;------------ detect VGA card
		mov	ax,1A00h
		int	10h			; get display type in bx
		cmp	al,1Ah
		jne	@@noVGA
		cmp	bl,7			; monochrome VGA?
		je	@@setATCVGA
		cmp	bl,8			; color VGA?
		jne	@@noVGA
@@setATCVGA:	inc	@videoregs[(SIZE RGROUPDEF)*3].regscnt
@@noVGA:
;------------ save old video handler
		mov	ax,3510h
		int	21h
		saveFAR	[oldint10],es,bx
;------------
		call	prepareTSR		; new memory segment in ES
		call	setupIRQ
		movSeg	ds,es
		mov	dx,offset handler33
		mov	ax,2533h
		int	21h			; setup new 33 handler
		xor	ax,ax
		int	33h			; Reset driver

		movSeg	ds,cs
		call	printinfo		; print COM, mode & resolution
		cmp	[mem_type],3		; if conventional, TSR
		je	@@tsrexit
		jmp	EXIT0

;------------ TSR exit if installed in conventional memory
@@tsrexit:	mov	es,ds:[2Ch]		; release environment
		mov	ah,49h
		int	21h

		mov	dx,offset NonTSRpart+15
		mov	cl,4
		shr	dx,cl
		mov	ax,3100h
		int	21h			; TSR exit

;
;		If already installed, change params and exit
;
;
; In:	ES					(installed driver segment)
; Out:	none
; Use:	lefthand?, mresolution, autores?, mousetype, logitech?,
;	IRQintnum, IO_number, PICstate
; Modf: AX, DX
; Call: Int33/1F, Int33/20, closeoldIRQ, setupIRQ, printinfo
;
changeparam	proc
		mov	al,[lefthand?]		; set up new params
		mov	es:[lefthand?],al
		mov	ax,[mresolution]
		mov	es:[mresolution],ax
		mov	al,[autores?]
		mov	es:[autores?],al
		mov	al,[mousetype]
		mov	es:[mousetype],al
IFNDEF PS2
		mov	al,[logitech?]
		mov	es:[logitech?],al

;------------ change IRQ handler to new IRQ number
		mov	ax,[IO_number]
		xchg	es:[IO_number],ax
		call	closeoldIRQ
		mov	al,[PICstate]
		mov	es:[PICstate],al
		mov	al,[IRQintnum]
		mov	es:[IRQintnum],al
		call	setupIRQ
;------------
ENDIF
		mov	ax,20h
		int	33h			; enable driver
		call	printinfo		; print new params
		mov	dx,offset S_already	; 'Mouse driver already'
		jmp	EXIT0MSG
changeparam	endp

;
;			Save old IRQ handler
;

setupIRQ	proc
		push	es
		mov	al,[IRQintnum]
		mov	ah,35h
		int	21h
		mov	ax,es
		pop	es
		saveFAR es:[oldIRQaddr],ax,bx
IFNDEF PS2
		mov	[IObyte4],0
		mov	bx,offset MSLGproc-offset mouseproc-2
		mov	al,[mousetype]
		cmp	al,2			; Microsoft mouse?
		je	@@setmousep
		dec	ax			; OPTIMIZE: AX instead AL
		IFnz	[logitech?],@@setmousep	; Logitech mouse?
		inc	ax			; OPTIMIZE: AX instead AL
		mov	bx,offset MSMproc-offset mouseproc-2
@@setmousep:	mov	es:[mouseproc],bx
		mov	es:[COMLCR],al
ENDIF
		ret
setupIRQ	endp

IFNDEF PS2
;
;			Check if COM port available
;
;
; In:	none
; Out:	Carry flag				(COM port not exist)
; Use:	IO_number
; Modf: AX, DX
; Call: disableIRQ
;
COMexist?	proc
		cli
		call	disableIRQ
		mov	dx,[IO_number]
		or	dx,dx
		stc
		jz	@@checkret
		add	dx,3
		mov	al,0
		out	dx,al			; {3FBh} reset comm params
		dec	dx
		dec	dx
		in	al,dx			; {3F9h} get int enable reg
		and	al,0F0h
		mov	ah,al			; store reserved bits
		add	dx,3
		in	al,dx			; {3FCh} get modem ctrl reg
		and	al,0E0h			; get reserved bits
		or	al,ah			; AL must be 0 if port exists
		neg	al			; nonzero makes carry flag
@@checkret:	sti
		ret
COMexist?	endp

;
;				Disable old IRQ
;
;
; In:	AX					(old COM port)
; Out:	none
; Use:	none
; Modf: AL, DX
; Call: none
;
closeoldIRQ	proc
		mov	dx,ax
		add	dx,3
		mov	al,0
		out	dx,al			; {3FBh} set DLAB off
		dec	dx
		dec	dx
		mov	al,0
		out	dx,al			; {3F9h} all interrupts off
		add	dx,3
		in	al,dx			; {3FCh} modem ctrl
		and	al,0F3h
		out	dx,al			; disable auxilary outputs
		inc	dx
		in	al,dx			; {3FDh} clear error bits
		sub	dx,5
		in	al,dx			; {3F8h} flush receive buffer
		ret
closeoldIRQ	endp

;
;			Reset mouse hardware
;
;
; In:	SI					(COM port number)
; Out:	none
; Use:	mousetype
; Modf: AL, CX, DX
; Call: none
;
mouseHWreset	proc
		mov	dx,si
		add	dx,3
		mov	al,80h
		out	dx,al			; {3FBh} set DLAB on

		mov	dx,si
		mov	ax,96
		out	dx,ax			; {3F8h},{3F9h} 1200 baud

		add	dx,3
		mov	al,[mousetype]
		out	dx,al			; {3FBh} DLAB off, no parity,
						; stop=1, length=7/8
		dec	dx
		dec	dx
		mov	al,0
		out	dx,al			; {3F9h} all interrupts off

		add	dx,3
		inc	ax			; OPTIMIZE: instead mov AL,1
		out	dx,al			; {3FCh} DTR on, RTS off
		xor	cx,cx
		loop	$

		inc	dx
		in	al,dx			; {3FDh} clear error bits
		ret
mouseHWreset	endp

;
;		Check if requested mouse type available
;

checktype	proc
		mov	si,[IO_number]
		call	mouseHWreset
		cmp	[mousetype],2		; microsoft mouse?
		jne	@@notMSmode
		call	detmicrosoft
		jnc	@@setmode
retC:		stc
		ret

@@notMSmode:	mov	dx,si
		add	dx,4
		mov	al,8			; RTS/DTR off, OUT2 on
		IFz	[logitech?],@@write8	;*** ??? ***
		mov	al,8+3			; RTS/DTR/OUT2 on
@@write8:	out	dx,al			; {3FCh}

@@setmode:	mov	dx,si
		inc	dx
		mov	al,1
		out	dx,al			; {3F9h} DR int enable
		clc
		ret
checktype	endp

;
;			Detect if Microsoft Mouse present
;
;
; In:	SI					(COM port number)
; Out:	none
; Use:	none
; Modf: AX, BX, CX, DX, logitech?
; Call: none
;
detmicrosoft	proc
		mov	dx,si
		add	dx,4
		mov	al,0Bh
		out	dx,al			; {3FCh} DTR/RTS/OUT2 on
		mov	dx,si
		in	al,dx			; {3F8h} flush receive buffer

		mov	bx,4			; error if there is no 'M'
						;  within 4 received bytes
@@getM:		xor	cx,cx
		add	dx,5
@@chkDTR:	in	al,dx			; {3FDh} get line stat reg
		test	al,1			; check if data ready
		jnz	@@DTRok			; jump if yes
		loop	@@chkDTR		; if not, try again
		j	@@DTRnotok		; if no data received, exit

@@DTRok:	mov	dx,si
		in	al,dx			; {3F8h} get that bastard byte
		cmp	al,'M'
		jne	@@notM
		inc	bh
@@notM:		cmp	al,'3'
		jne	@@not3
		inc	[logitech?]
@@not3:		dec	bl
		jnz	@@getM

@@DTRnotok:	or	bh,bh			; 'M' received?
		jz	retC			; jump if not
		IFnz	[logitech?],retC	; jump if '3' received
		clc
		ret
detmicrosoft	endp

;
;				Set Mouse Port
;
;
; In:	AL					(COM port letter)
; Out:	none
; Use:	none
; Modf: AL, com_port, IO_number
; Call: setIRQ
;
setCOMport	proc
		mov	[com_port],al
		sub	al,'1'

		push	bx ds
		xor	bx,bx
		mov	ds,bx
		mov	bl,al
		shl	bx,1
		mov	bx,400h[bx]
		pop	ds
		mov	[IO_number],bx
		pop	bx

		shr	al,1
		mov	al,4			; IRQ4 for COM1/3
		jnc	setIRQ
		dec	ax			; OPTIMIZE: AX instead AL
						; IRQ3 for COM2/4
		;j	setIRQ
setCOMport	endp

;
;				Set IRQ number
;
;
; In:	AL					(IRQ number)
; Out:	none
; Use:	none
; Modf: AL, CL, PICstate, IRQintnum
; Call: none
;
setIRQ		proc
		mov	cl,al
		add	al,8
		mov	[IRQintnum],al
		mov	al,1
		shl	al,cl
		mov	[PICstate],al		; PIC interrupt enabler
		ret
setIRQ		endp

ELSE
;
;			Dummy IRQ handler for PS/2
;

dummyIRQ	proc	far
		retf
dummyIRQ	endp

;
;				Check for PS/2
;
;
; In:	none
; Out:	Carry flag
; Use:	none
; Modf: AX, BX, CX
; Call: none
;
checkPS2	proc
		mov	bh,3
		PS2serv 0C205h,@@checkPSret	; initialize mouse, bh=datasize

		mov	cx,2
		movSeg	es,cs
@@try2:		mov	bx,offset dummyIRQ
		PS2serv	0C207h			; mouse, es:bx=ptr to handler
		jnc	@@PSokyet
		cmp	ah,4
		jne	@@checkPSret
		loop	@@try2
		ret

@@PSokyet:	mov	bh,1
		PS2serv 0C200h,@@checkPSret	; set mouse on
		mov	[mousetype],4
@@checkPSret:	ret
checkPS2	endp
ENDIF

;
;		Get CuteMouse driver segment, if installed
;
;
; In:	none
; Out:	Zero flag				(zero if installed)
;	ES					(driver segment)
; Use:	none
; Modf: AX, ES, BX, SI, DI, CX
; Call: Int21/3533
;
getCuteSeg	proc
		mov	ax,3533h
		int	21h
		cmp	bx,offset handler33
		jne	@ret3
		mov	si,offset IDstring
		mov	di,si
		mov	cx,IDstringlen
		rep	cmpsb
		jne	@ret3
		push	es
		mov	ax,1Fh
		int	33h			; disable driver
		pop	es
@ret3:		ret
getCuteSeg	endp

;
;			Examine Command Line
;

commandline	proc
		mov	si,80h
		lodsb
		mov	ah,0
		mov	di,ax
		add	di,si

newopt:		cmp	si,di
		jae	@ret3
		lodsb
		cmp	al,' '
		jbe	newopt
		cmp	al,'/'
		jne	@badoption
		lodsb
		cmp	al,'?'
		je	_printhelp		; '/?' -> print help and exit
IFNDEF PS2
		cmp	al,'1'
		jb	@@nocomport
		cmp	al,'4'
		ja	@@nocomport
		call	setCOMport		; '/n' -> set COM port
		mov	al,[si]
		cmp	al,'2'
		jb	newopt
		cmp	al,'7'
		ja	newopt
		inc	si
		sub	al,'0'
		call	setIRQ			; '/nm' -> set IRQ line
		j	newopt
ENDIF
@@nocomport:	and	al,not 20h
		cmp	al,'L'
		jne	@@notleft
		inc	[lefthand?]		; '/L' -> set lefthand mode
		j	newopt
@@notleft:	cmp	al,'R'
		je	_resolution		; '/R' -> set resolution
		cmp	al,'U'
		je	_reldriver		; '/U' -> release drv and exit
IFNDEF PS2
		mov	[mousetype],2
		mov	[logitech?],0
		cmp	al,'M'			; force microsoft?
		je	newopt
		inc	[mousetype]
		cmp	al,'S'			; force mouse systems?
		je	newopt
		inc	[logitech?]
		cmp	al,'T'			; force logitech?
		je	newopt
ENDIF
@badoption:	mov	dx,offset E_option	; 'Invalid parameter'
EXITERRMSG:	PrintS
		mov	ax,4CFFh
		int	21h			; exit with errorlevel
commandline	endp

;
;			Print Help and Quit
;

_printhelp	proc
		mov	dx,offset Syntax	; 'Cute Mouse Driver'
EXIT0MSG:	PrintS
EXIT0:		mov	ax,4C00h
		int	21h			; terminate, al=return code
_printhelp	endp

;
;				Set Resolution
;

_resolution	proc
		mov	[autores?],0
		lodsb
		cmp	al,'1'
		jb	@@autores
		cmp	al,'9'
		ja	@@autores
		mov	S_times[0],al		; 'x/x times'
		mov	ah,al

		lodsb				; second argument
		cmp	al,'1'
		jb	@@no2ndres
		cmp	al,'9'
		jbe	@@store2nd
@@no2ndres:	mov	al,ah
		dec	si			; fixup for command line ptr
@@store2nd:	mov	S_times[2],al
		xchg	al,ah
		sub	ax,'11'
		j	@@setres

@@autores:	cmp	al,'0'
		je	@@autores0
		dec	si			; fixup if AL not equal '0'
@@autores0:	inc	[autores?]
		xor	ax,ax

@@setres:	mov	word ptr [mresolution],ax
		jmp	newopt
_resolution	endp

;
;			Release driver and quit
;

_reldriver	proc
		call	getCuteSeg
		mov	dx,offset S_notinst	; 'CuteMouse driver is not...'
		jne	EXITERRMSG

		mov	ax,2533h
		push	ds
		lds	dx,es:[oldint33]
		int	21h			; restore old int33 handler
		mov	ax,es
		dec	ax
		mov	ds,ax
		mov	ah,62h
		int	21h
		mov	ds:[1],bx		; modify MCB
		pop	ds
		call	FreeMem
		mov	ax,word ptr es:oldint33[0]
		or	ax,word ptr es:oldint33[2]
		jz	@@relexit
		mov	ax,20h
		int	33h			; enable driver
@@relexit:	mov	dx,offset S_released	; 'Mouse driver has been...'
		j	EXIT0MSG
_reldriver	endp

;
;			Print mode and resolution
;
;
; In:	none
; Out:	none
; Use:	mousetype, logitech?, autores?
; Modf: AX, DX
; Call: none
;
printinfo	proc
		PrintS	S_info			; 'Installed on:'
IFNDEF PS2
		PrintS	S_atCOM
		mov	dx,offset S_drvMS	; 'Microsoft mouse'
		cmp	[mousetype],3
		jne	@@printmode
		mov	dx,offset S_drvMouse	; 'Mouse Systems mouse'
		IFz	[logitech?],@@printmode
		mov	dx,offset S_drvLogi	; 'Logitech MouseMan'
ELSE
		mov	dx,offset S_drvPS2	; 'PS/2 port'
ENDIF
@@printmode:	PrintS
		PrintS	S_resolution		; 'Resolution:'
		mov	dx,offset S_auto	; 'Auto'
		IFnz	[autores?],@@prres
		mov	dx,offset S_times	; '3/3 times'
@@prres:	PrintS
		ret
printinfo	endp


;-------------------
; DOS 5.0+ UMB's
SaveMemStrat	dw	0
SaveUMBLink	db	0

;
prepareTSR	proc
		mov	ax,offset NonTSRpart	; get number of bytes
		call	AllocMem		;  we need memory for
		cmp	[mem_type],3
		je	convent

		mov	bx,ax
		dec	bx
		mov	es,bx
		mov	word ptr es:[1],ax	; modify MCB
		mov	si,offset ctstr
		mov	di,8
		movsw				; copy process name
		movsw
		movsw
		movsw

		mov	bx,ax
		add	bx,word ptr es:[3]
		mov	es,ax
		mov	dx,ax
		mov	ah,26h
		int	21h			; create PSP
		mov	es:[2],bx		; fix upper segment number

;------------ relocate resident portion
		mov	si,100h			; relocate the resident
		mov	di,si			;  portion
		mov	cx,offset NonTSRpart-100h
		rep	movsb
;------------
		ret

convent:	mov	es,ax
		ret
prepareTSR	endp

;
; Get "XMS" handler address (2 words)
;
; In:	none
; Out:	Zero flag
; Use:	none
; Modf: AX, BX, ES, XMSentry
; Call: none
;
getXMSaddr	proc
		xor	bx,bx
		mov	es,bx
		mov	ax,4310h		; XMS: Get Driver Address
		int	2Fh
		saveFAR [XMSentry],es,bx
		mov	ax,es
		or	ax,bx			; ZF indicates error: JZ error
		ret
getXMSaddr	endp

;
; Get Allocation Srategy
;
; In:	none
; Out:	Carry flag
; Use:	none
; Modf: AX, SaveMemStrat, SaveUMBLink
; Call: none
;
GetAllocStrat	proc
		mov	ax,5800h
		int	21h			; get DOS memory alloc strategy
		jc	@@getallocret
		mov	[SaveMemStrat],ax
		mov	ax,5802h
		int	21h			; get UMB link state
		mov	[SaveUMBLink],al
@@getallocret:	ret
GetAllocStrat	endp

;
; Restore allocation strategy
;
; In:	none
; Out:	none
; Use:	SaveMemStrat, SaveUMBLink
; Modf: AX, BX
; Call: none
;
ResAllocStrat	proc
		mov	bx,[SaveMemStrat]
		mov	ax,5801h
		int	21h			; set DOS memory alloc strategy
		mov	bl,[SaveUMBLink]
		mov	bh,0
		mov	ax,5803h
		int	21h			; set UMB link state
		ret
ResAllocStrat	endp

;
; function AllocMem
;
; In:	AX					(memory required)
; Out:	AX					(segment or 0 if error)
;	dest_seg
;	mem_type
;
AllocMem	proc				; call with AX = n of bytes
		push	ds es
		mov	[mem_type],0
		mov	[dest_seg],0
		add	ax,15
		mov	cl,4
		shr	ax,cl
		mov	[paragraphs],ax

; Check if UMB is DOS5+ type

		mov	ah,30h
		int	21h
		cmp	al,5			; DOS >= 5.0, supports UMBs
		jb	@@noDOS5UMBs

		call	GetAllocStrat
		jc	@@noDOS5UMBs

		xor	bx,bx			; low mem, first fit
		mov	ax,5801h
		int	21h			; set DOS memory alloc strategy
		jc	@@noDOS5UMBs		; reports >= 5.0 no support

		mov	bx,1			; add UMB to MCB chain
		mov	ax,5803h
		int	21h			; set UMB link state
		jc	@@noDOS5UMBs

; try to set a good strategy to allocate DOS supported UMBs

		mov	bx,41h			; hi mem, best fit
		mov	ax,5801h
		int	21h			; set alloc strategy
		jnc	@@linkUMB		; jump if success

; try a worse one then

		mov	bx,81h			; hi mem then low mem, best fit
		mov	ax,5801h
		int	21h			; set alloc strategy
		jc	@@dev5			; jump if error

@@linkUMB:	mov	bx,1
		mov	ax,5803h
		int	21h			; add UMBs to link state
		mov	bx,[paragraphs]
		mov	ah,48h
		int	21h			; allocate UMB memory
		jc	@@dev5
		cmp	ax,0A000h		; check if allocated mem is
		ja	@@Dumb_OK		; beyond 640k. Jump if yes

		mov	es,ax			; if below, then free it
		mov	ah,49h
		int	21h
		stc				; indicate it
		j	@@dev5

@@Dumb_OK:	clc
@@dev5:		pushf
		push	ax
		call	ResAllocStrat		; restore allocation strategy
		pop	ax
		popf
		jc	@@noDOS5UMBs		; jump if UMB allocating not successful
		mov	[dest_seg],ax		; else set proper variables and exit
		mov	[mem_type],1		; 1 = DOS 5.0+ UMB
		j	@@fin

; try XMS driver to allocate UMB

@@noDOS5UMBs:	mov	ax,4300h		; XMS: Installation Check
		int	2Fh
		cmp	al,80h
		jne	@@noXMS_UMB
		call	getXMSaddr
		jz	@@noXMS_UMB
		mov	ah,10h			; XMS: Request UMB
		mov	dx,[paragraphs]
		call	[XMSentry]
		cmp	ax,1
		jne	@@noXMS_UMB
		cmp	dx,[paragraphs]
		jb	@@noAlcanza
		mov	[dest_seg],bx
		mov	[mem_type],2		; 2 = XMS UMB
		j	@@fin

@@noAlcanza:	mov	dx,bx
		mov	ah,11h			; XMS: Release UMB
		call	[XMSentry]

; use conventional memory

@@noXMS_UMB:	mov	ax,cs
		mov	[dest_seg],ax		; AX=segment
		mov	[mem_type],3		; 3 = conventional memory

@@fin:		pop	es ds
		mov	ax,[dest_seg]
		ret
AllocMem	endp

;
;
; In:	ES					(segment to free)
;
FreeMem		proc
		push	ds es
		mov	al,es:[mem_type]
		cmp	al,2
		je	@@xms
		cmp	al,3
		je	@@finConv

@@dos5:		call	GetAllocStrat
		jc	@@finfree		; error?
		xor	bx,bx
		mov	ax,5803h
		int	21h			; unlink UMBs
		mov	ah,49h
		int	21h			; free allocated memory
		call	ResAllocStrat
		j	@@finfree

@@xms:		call	getXMSaddr
		jz	@@finfree		; error?
		mov	ah,11h			; XMS: Release UMB
		mov	dx,es
		call	[XMSentry]
		j	@@finfree

@@finConv:	mov	ah,49h
		int	21h			; free allocated memory

@@finfree:	pop	es ds
		ret
FreeMem		endp

;
		end	start
