/*
 *  ALIAS.C - alias administration module
 *
 *
 *
 *  Comments:
 *
 *  02/02/1996 (Oliver Mueller)
 *    started.
 *
 *  02/03/1996 (Oliver Mueller)
 *    Added sorting algorithm and case sensitive substitution by using
 *    partstrupr().
 *
 * 27 Jul 1998  John P. Price
 * - added config.h include
 * - added ifdef's to disable aliases
 *
 * 1998/10/27 ska
 * - changed: rmtmpfile() closes file itself
 *
 * 01-Dec-1998  John P. Price
 * - changed to convert to lower case instead of upper
 *
 * 1998/12/05 ska
 * - bugfix: if useFlag warps its type range, all the use flags are
 *  clear, but in an infinite loop
 *
 * 16-Dec-1998  John P. Price
 * - Changed expander so that if the first non-whitespace character is a
 *   dot then not to do alias expansion.  This was "noalias" but I prefer
 *   the shorter version.
 *
 * 1999/01/24 ska
 * bugfix: aliasdel(): after the pointer is free()'ed, the loop accessed
 *      the memory again (reported by Eric Kohl <ekohl@abo.rhein-zeitung.de>)
 *
 * 28-Jan-1999  John P. Price
 * - Changed noalias character from a dot to an asterisk (*).
 *
 * 1/Feb/1999 Rob Linwood
 * - added code to allow alias commands to contain extra whitespace
 *
 * 24-Mar-1999  John P. Price
 * - fixed bug in aliasexpand
 */

#include "config.h"

#ifdef FEATURE_ALIASES

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>

#include "command.h"
#include "tempfile.h"
#include "alias.h"

typedef struct TAlias
{
  char *name,
   *subst;
  unsigned short used;
  struct TAlias *next;
}
TAlias;

static unsigned short useFlag = 0;
static TAlias *first = 0,
 *last = 0;

/* module internal functions */
void partstrlower(char *str)    /* strlwr only for first word in string */
{
  char *c = str;
  while (*c && !isspace(*c))
  {
    *c = tolower(*c);
    c++;
  }
}

/*
 * Name: ltrim() - left trims a string by removing leading spaces
 * Input: str - a pointer to a string
 * Output: returns a trimmed copy of str, or NULL if the string is all
 *         spaces
 * Date: 1/Feb/1999
 */
char *ltrim(char *str)
{
  char c;

  while ((c = *(char *)str++) == ' ')
  {
    if (!c)
      return NULL;
  }
  return (char *)str - 1;
}

/*
 * Name: rtrim() - right trims a string by removing trailing spaces
 * Input: str - a pointer to a string
 * Output: str will have all spaces removed from the right.
 * Date: 1/Feb/1999
 */
void rtrim(char *str)
{
  int i;

  for (i = strlen(str) - 1; i >= 0; i--)
  {
    if (str[i] != ' ')
    {
      return;
    }
    else
    {
      str[i] = 0;
    }
  }
}

void aliasprint(void)
{
  TAlias *ptr = first;
  while (ptr)
  {
    printf("%s=%s\n", ptr->name, ptr->subst);
    ptr = ptr->next;
  }
}

void aliasdel(char *name)
{
  TAlias *ptr = first,
   *prev = 0;

  while (ptr)
  {
    if (!strcmp(ptr->name, name))
    {
      if (prev)
        prev->next = ptr->next;
      else
        first = ptr->next;
      free(ptr->name);
      free(ptr->subst);
      free(ptr);
      return;
    }
    prev = ptr;
    ptr = ptr->next;
  }
}

int aliasadd(char *name, char *subst)
{
  TAlias *ptr = first,
   *prev,
   *entry;
  char *s;

  while (ptr)
  {
    if (!strcmp(ptr->name, name))
    {
      s = (char *)malloc(strlen(subst) + 1);
      if (!s)
      {
        errno = ENOMEM;
        return -1;
      }
      free(ptr->subst);
      ptr->subst = s;
      strcpy(ptr->subst, subst);
      return 0;
    }
    ptr = ptr->next;
  }

  ptr = (TAlias *) malloc(sizeof(TAlias));
  if (!ptr)
  {
    errno = ENOMEM;
    return -1;
  }
  ptr->next = 0;

  ptr->name = (char *)malloc(strlen(name) + 1);
  if (!ptr->name)
  {
    free(ptr);
    errno = ENOMEM;
    return -1;
  }
  strcpy(ptr->name, name);

  ptr->subst = (char *)malloc(strlen(subst) + 1);
  if (!ptr->subst)
  {
    free(ptr->name);
    free(ptr);
    errno = ENOMEM;
    return -1;
  }
  strcpy(ptr->subst, subst);
  partstrlower(ptr->subst);     /* it's necessary for recursive substitution */

  ptr->used = 0;

 /* Alias table must be sorted!
  * Here a little example:
  *   command line = "ls -c"
  * If the entries are
  *   ls=dir
  *   ls -c=ls /w
  * command line will be expanded to "dir -c" which is not correct.
  * If the entries are sortet as
  *   ls -c=ls /w
  *   ls=dir
  * it will be expanded to "dir /w" which is a valid DOS command.
  */
  entry = first;
  prev = 0;
  while (entry)
  {
    if (strcmp(ptr->name, entry->name) > 0)
    {
      if (prev)
      {
        prev->next = ptr;
        ptr->next = entry;
      }
      else
      {
        ptr->next = entry;
        first = ptr;
      }
      return 0;
    }
    prev = entry;
    entry = entry->next;
  }

  /* The new entry is the smallest (or the first) and must be
   * added to the end of the list.
   */
  if (!first)
    first = ptr;
  else
    last->next = ptr;
  last = ptr;

  return 0;
}

/* specified routines */
void aliasexpand(char *cmd, int maxlen)
{
  unsigned n = 0;
  unsigned m;
  unsigned i;
  unsigned len;
  short d = 1;
  TAlias *ptr = first;

  if (++useFlag == 0)           /* (int) overflow */
  {                             /* The useFlag specifies, if the particular ALIAS
                                   is used by the _current_ expand. To avoid to
                                   clear the flag each time when to expand a string
                                   a different flag value is used each time. */
    while (ptr)
    {                           /* reset all values to be sure we hit no old one */
      ptr->used = 0;
      ptr = ptr->next;
    }
    ptr = first;
    useFlag = 1;
  }

  /* skipping white spaces */
  while (isspace(cmd[n]))
    n++;

  partstrlower(&cmd[n]);

  if (cmd[n] == '*')
  {
    memmove(cmd, &cmd[n + 1], strlen(&cmd[n + 1]) + 1);
    return;
  }

  /* substitution loop */
  while (d)
  {
    d = 0;
    while (ptr)
    {
      len = strlen(ptr->name);
      if (!strncmp(&cmd[n], ptr->name, len) && (isspace(cmd[n + len]) || cmd[n + len] == '\0') && ptr->used != useFlag)
      {
        m = strlen(ptr->subst);
        if (strlen(cmd) - len + m - n > maxlen)
        {
          error_command_too_long();
          cmd[0] = '\0';        /* the parser won't cause any problems with an empty line */
          return;  /*JPP 19990324*/
        }
        else
        {
          memmove(&cmd[m], &cmd[n + len], strlen(&cmd[n + len]) + 1);
          for (i = 0; i < m; i++)
            cmd[i] = ptr->subst[i];
          ptr->used = useFlag;
          n = 0;                /* whitespaces are removed! */
          d = 1;
        }
      }
      ptr = ptr->next;
    }
  }
}

#pragma argsused
int cmd_alias(char *rest)
{
  char *ptr;
  int n = 0;

  if (rest[0] == '\0')
  {
    aliasprint();
    return 0;
  }

  if ((ptr = strchr(rest, '=')) == 0)
  {
    errno = EINVAL;
    return -1;
  }

  /* Split rest into name and substitute */
  *ptr++ = '\0';

  partstrlower(rest);

  if (ptr[0] == '\0')
    aliasdel(rest);
  else
  {
    rtrim(rest);
    n = aliasadd(rest, ltrim(ptr));
  }
  return n;
}

#endif
