/*
 * COMMAND.C - command-line interface.
 *
 * Comments:
 *
 * 17 Jun 1994 (Tim Norman)
 *   started.
 *
 * 08 Aug 1995 (Matt Rains)
 *   I have cleaned up the source code. changes now bring this source
 *   into guidelines for recommended programming practice.
 *
 *   A added the the standard FreeDOS GNU licence test to the
 *   initialize() function.
 *
 *   Started to replease puts() with printf(). this will help
 *   standardize output. please follow my lead.
 *
 *   I have added some constants to help making changes easier.
 *
 * 15 Dec 1995 (Tim Norman)
 *   major rewrite of the code to make it more efficient and add
 *   redirection support (finally!)
 *
 * 6 Jan 1996 (Tim Norman)
 *   finished adding redirection support!  Changed to use our own exec
 *   code (MUCH thanks to Svante Frey!!)
 *
 * 29 Jan 1996 (Tim Norman)
 *   added support for CHDIR, RMDIR, MKDIR, and ERASE, as per suggestion
 *   of Steffan Kaiser
 *
 *   changed "file not found" error message to "bad command or filename"
 *   thanks to Dustin Norman for noticing that confusing message!
 *
 *   changed the format to call internal commands (again) so that if they
 *   want to split their commands, they can do it themselves (none of the
 *   internal functions so far need that much power, anyway)
 *
 *
 * 27 Aug 1996 (Tim Norman)
 *   added in support for Oliver Mueller's ALIAS command
 *
 * 14 Jun 1997 (Steffan Kaiser)
 *   added ctrl-break handling and error level
 *
 * 16 Jun 1998 (Rob Lake)
 *   Runs command.com if /P is specified in command line.  Command.com
 *   also stays permanent.  If /C is in the command line, starts the
 *   program next in the line.
 *
 * 21 Jun 1998 (Rob Lake)
 *   Fixed up /C so that arguments for the program
 *
 * 08-Jul-1998 (John P. Price)
 *   - Now sets COMSPEC environment variable
 *   - misc clean up and optimization
 *   - added date and time commands
 *   - changed to using spawnl instead of exec.  exec does not copy the
 *     environment to the child process!
 *
 * 14 Jul 1998 (Hans B Pufal)
 *   Reorganised source to be more efficient and to more closely follow
 *   MS-DOS conventions. (eg %..% environment variable replacement
 *   works form command line as well as batch file.
 *
 *   New organisation also properly support nested batch files.
 *
 *   New command table structure is half way towards providing a
 *   system in which COMMAND will find out what internal commands
 *   are loaded
 *
 * 24 Jul 1998 (Hans B Pufal) [HBP_003]
 *   Fixed return value when called with /C option
 *
 * 27 Jul 1998  John P. Price
 * - added config.h include
 *
 * 28 Jul 1998  John P. Price
 * - added showcmds function to show commands and options available
 *
 * 07-Aug-1998 (John P Price <linux-guru@gcfl.net>)
 * - Fixed carrage return output to better match MSDOS with echo on or off.
 *   (marked with "JPP 19980708")
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <dos.h>
#include <process.h>
#include <time.h>
#include <errno.h>
#include <dir.h>
#include <fcntl.h>
#include <io.h>
#include <sys\stat.h>

#include "command.h"
#include "batch.h"

int exitflag = 0;               /* indicates EXIT was typed */
int canexit = 1;                /* indicates if this shell is exitable */
int ctrlBreak = 0;              /* Ctrl-Break or Ctrl-C hit */
int errorlevel = 0;             /* Errorlevel of last launched external program */

extern struct CMD
  cmds[];                       /* The internal command table */

void fatal_error(char *s)
{
  /*
   *  fatal error handler.
   *
   */

  printf("fatal_error() : %s\n", s);

  exit(100);
}

static char is_delim(char c)
{
  /*
   *  is character a delimeter when used on first word?
   *
   */

  return (c == '/' || c == '=' || c == 0 || isspace(c));
}

static void execute(char *first, char *rest)
{
  /*
   * This command (in first) was not found in the command table
   *
   *
   * first - first word on command line
   * rest  - rest of command line
   *
   */

  char *fullname;

  /* check for a drive change */
  if ((strcmp(first + 1, ":") == 0) && isalpha(*first))
  {
    setdisk(toupper(*first) - 'A');

    if (getdisk() != toupper(*first) - 'A')
      puts(INVALIDDRIVE);

    return;
  }

  /* get the PATH environment variable and parse it */
  /* search the PATH environment variable for the binary */
  fullname = find_which(first);

  if (!fullname)
  {
    error_bad_command();
    return;
  }

  /* check if this is a .BAT file */

  if (stricmp(strrchr(fullname, '.'), ".bat") == 0)
  {
    dprintf(("[BATCH: %s %s]\n", fullname, rest));
    batch(fullname, first, rest);
  }

  else
    /* exec the program */
  {
    int result;

    dprintf(("[EXEC: %s %s]\n", fullname, rest));
    result = spawnl(P_WAIT, fullname, fullname, rest, NULL);

    if (result == -1)
      perror("executing spawnl function");
    else
      errorlevel = result;
  }
}

static void docommand(char *line)
{
  /*
   * look through the internal commands and determine whether or not this
   * command is one of them.  If it is, call the command.  If not, call
   * execute to run it as an external program.
   *
   * line - the command line of the program to run
   *
   */

  char
    com[128],                   /* the first word in the command                */
   *cp = com,
   *cstart,
   *rest = line;                /* pointer to the rest of the command line      */

  int
    cl;

  struct CMD
   *cmdptr;

  while (isspace(*rest))        /* Skip over initial white space */
    rest++;

  cstart = rest;

  if (*rest)                    /* Anything to do ? */
  {
    while (!is_delim(*rest))    /* Copy over 1st word as lower case */
      *cp++ = tolower(*rest++);

    *cp = '\0';                 /* Terminate first word */

    while (isspace(*rest))      /* Skip over whitespace to rest of line */
      rest++;

    for (cmdptr = cmds;; cmdptr++)  /* Scan internal command table */
    {
      if (cmdptr->name == NULL) /* If end of table execute ext cmd */
      {
        execute(com, rest);
        break;
      }

      if (strcmp(com, cmdptr->name) == 0)
      {
        cmdptr->func(com, rest);
        break;
      }

      /* The following code handles the case of commands like CD which
       * are recognised even when the command name and parameter are
       * not space separated.
       *
       * e.g dir..
       * cd\freda
       */

      cl = strlen(cmdptr->name);  /* Get length of command name */

      if ((cmdptr->flags & CMD_SPECIAL) &&
          (strncmp(cmdptr->name, com, cl) == 0) &&
          (strchr("\\.", *(com + cl))))
      {
        // OK its one of the specials...

        com[cl] = '\0';         /* Terminate first word properly */

        cmdptr->func(com, cstart + cl); /* Call with new rest */
        break;
      }
    }
  }
}

/*
 * process the command line and execute the appropriate functions
 * full input/output redirection and piping are supported
 *
 */
void parsecommandline(char *s)
{
  char in[128] = "";
  char out[128] = "";
  char tempdir[128] = ".\\";
  char fname[2][128] = {"", ""};
  char *t = NULL;

  int
    of_attrib = O_CREAT | O_TRUNC | O_TEXT | O_WRONLY,
    num,
    oldinfd,
    oldoutfd;

  dprintf(("[parsecommandline ('%s')]\n", s));

  /* first thing we do is alias expansion */

#ifdef FEATURE_ALIASES
  aliasexpand(s, 127);
#endif

  /* find the temp directory to store temporary files */

  if (NULL != (t = getenv("TEMP")))
  {
    strcpy(tempdir, t);

    if (tempdir[strlen(tempdir) - 1] != '\\')
      strcat(tempdir, "\\");
  }

  /* get the redirections from the command line */

  num = get_redirection(s, in, out, &of_attrib);

  /* more efficient, but do we really need to do this? */

  for (t = in; isspace(*t); t++)
    ;
  strcpy(in, t);

  for (t = out; isspace(*t); t++)
    ;
  strcpy(out, t);

  /* Set up the initial conditions ... */

  oldinfd = oldoutfd = -1;

  if (in[0] || (num > 1))       /* Need to preserve stdin */
    oldinfd = dup(0);

  if (in[0])                    /* redirect input from this file name */
  {
    close(0);
    if (0 != open(in, O_TEXT | O_RDONLY, S_IREAD))
    {
      printf("Can't redirect input from file %s\n", in);
      return;
    }
  }

  if (out[0] || (num > 1))      /* Need to preserve stdout */
    oldoutfd = dup(1);

  /* Now do all but the last pipe command */

  *fname[1] = '\0';

  while (num-- > 1)
  {
    close(1);                   /* Close currrent output file */
    open(tmpnam(fname[0]), O_CREAT | O_TRUNC | O_TEXT | O_WRONLY,
         S_IREAD | S_IWRITE);

    docommand(s);

    if (*fname[1])
      remove(fname[1]);

    close(0);
    strcpy(fname[1], fname[0]);
    open(fname[1], O_TEXT | O_RDONLY, S_IREAD);

    s = s + strlen(s) + 1;
  }

  /* Now set up the end conditions... */

  if (out[0])                   /* Final output to here */
  {
    close(1);
    if (1 != open(out, of_attrib, S_IREAD | S_IWRITE))
    {
      printf("Can't redirect to file %s\n", out);
      return;
    }

    if (of_attrib & O_APPEND)
      lseek(1, 0, SEEK_END);

  }
  else if (oldoutfd != -1)      /* Restore original stdout */
  {
    close(1);
    dup2(oldoutfd, 1);
    close(oldoutfd);
    oldoutfd = -1;
  }

  docommand(s);                   /* process final command */

  if (*fname[1])
    remove(fname[1]);

  if (oldinfd != -1)            /* Restore original STDIN */
  {
    close(0);
    dup2(oldinfd, 0);
    close(oldinfd);
  }

  if (oldoutfd != -1)           /* Restore original STDOUT */
  {
    close(1);
    dup2(oldoutfd, 1);
    close(oldoutfd);
  }
}

/*
 * do the prompt/input/process loop
 *
 */

static int process_input(int xflag)
{
  char
    commandline[512],
    readline[129],
   *evar,
   *tp,
   *ip,
   *cp;

  /* JPP 19980807 - changed name so not to conflict with echo global */
  int echothisline;

  do
  {
    if (NULL == (ip = readbatchline(&echothisline)))  /* if no batch input then... */
    {
      if (xflag)
        return 0;

      readcommand(readline, 128);
      ip = readline;
      echothisline = 0;
    }

    cp = commandline;
    while (*ip)
    {
      if (*ip == '%')
      {
        switch (*++ip)
        {
          case '%':
            *cp++ = *ip++;
            break;

          case '0':
          case '1':
          case '2':
          case '3':
          case '4':
          case '5':
          case '6':
          case '7':
          case '8':
          case '9':
            if (NULL != (tp = find_arg(*ip - '0')))
            {
              cp = stpcpy(cp, tp);
              ip++;
            }
            else
              *cp++ = '%';

            break;

          case '?':
            cp += sprintf(cp, "%u", errorlevel);
            ip++;
            break;

          default:
            if ((tp = strchr(ip, '%')) != NULL)
            {
              *tp = '\0';

              if (((evar = getenv(ip)) != NULL) ||
                  ((evar = getenv(strupr(ip))) != NULL))
                cp = stpcpy(cp, evar);

              ip = tp + 1;
            }
            break;
        }

        continue;
      }

      if (iscntrl(*ip))
        *ip = ' ';

      *cp++ = *ip++;
    }

    *cp = '\0';

    while ((--cp >= commandline) && isspace(*cp)) /* strip trailing spaces */
      ;

    *(cp + 1) = '\0';

    /* JPP 19980807 */
    if (echothisline)                 /* Echo batch file line */
    {
      printprompt();
      puts(commandline);
    }

    if (*commandline)
    {
      parsecommandline(commandline);
      if (echo) putchar('\n');    /* JPP 19980807 */
    }
  }
  while (!canexit || !exitflag);

  return 0;
}

/*
 * control-break handler.
 */
int c_brk(void)
{
  ctrlBreak = 1;                /* indicate the break condition */
  return 1;                     /* continue execution */
}

/*
 * show commands and options that are available.
 *
 */
static void showcmds(void)
{
  struct CMD *cmdptr;
  unsigned char y;

  printf("\nInternal commands available:\n");
  y = 0;
  cmdptr = cmds;
  while (cmdptr->name)
  {
    if (++y == 8)
    {
      puts(cmdptr->name);
      y = 0;
    }
    else
     printf("%-10s", cmdptr->name);

    cmdptr++;
  }
  if (y != 0) putchar('\n');
  printf("\nFeatures available: ");
#ifdef FEATURE_ALIASES
  printf("[aliases] ");
#endif
#ifdef FEATURE_HISTORY
  printf("[history] ");
#endif
#ifdef FEATURE_FILENAME_COMPLETION
  printf("[filename completion] ");
#endif
  putchar('\n');
  putchar('\n');
}


/*
 * set up global initializations and process parameters
 *
 * argc - number of parameters to command.com
 * argv - command-line parameters
 *
 */
static void initialize(int argc, char *argv[])
{
  char temp[256];

  /* Added by Rob Lake 06/16/98.  This enables the command.com
   * to run the autoexec.bat at startup */

#ifdef DEBUG
  int x;

  dprintf(("[command args:\n"));
  for (x = 0; x < argc; x++)
  {
    dprintf(("%d. %s\n", x, argv[x]));
  }
  dprintf(("]\n"));
#endif

  if (argc >= 2)
  {
    if (strcmpi(argv[1], "/p") == 0)
    {
      if (!exist("\\autoexec.bat"))
      {
/* figure out if we're a permanent shell... and make it do this */
//        OwnerPSP = _psp;

#ifdef INCLUDE_CMD_DATE
        cmd_date("", "");
#endif
#ifdef INCLUDE_CMD_TIME
        cmd_time("", "");
#endif
      }
      else
        parsecommandline("\\autoexec.bat");
      canexit = 0;
    }
    else if (strcmpi(argv[1], "/c") == 0)
    {
      /* This just runs a program and exits, RL: 06/16,21/98 */
      char commandline[128];
      int i;
      i = 2;
      strcpy(commandline, argv[2]);
      while (argv[++i])
      {
        strcat(commandline, " ");
        strcat(commandline, argv[i]);
      }

      parsecommandline(commandline);

 /* HBP_003 { Fix return value when /C used */
      exit(process_input(1));
    }
  }

  short_version();
  showcmds();

  /* set up environment space and such */
  if (!EnvSeg)                  /* fix this to later make its own environment */
  {
    /* JPP 07/08/1998 - use puts instead of printf */
    puts(NOENVERR);
    exit(1);
  }

  /* JPP 07/08/1998 - Set COMSPEC environment variable */
  sprintf(temp, "COMSPEC=%s", argv[0]);
  set_env(temp);
}

int main(int argc, char *argv[])
{
  /*
   * * main function
   */

  initialize(argc, argv);       /* check switches on command-line */

  return process_input(0);      /* call prompt routine */
}
