/*
 * MISC.C -- Misc. Functions
 *
 * 07/12/98 (Rob Lake)
 *  started
 *
 * 07/13/98 (Rob Lake)
 *  moved functions in here
 *
 * 27-Jul-1998 (John P Price <linux-guru@gcfl.net>)
 * - added config.h include
 *
 * 09-Aug-1998 (Rob Lake <rlake@cs.mun.ca>)
 * - changed split function
 * - added freep function
 *
 * 10-Aug-1998 ska
 * - fix someone removed the ^Break catcher --> readded, this time using
 *   ANSI-like (DOS emulation) of signal handling functions
 *
 * 09-Sep-1998 ska
 * - fix/chg: ^Break handler now handles external program properly
 *
 * 1998/12/04 ska
 * - chg: vcgetchar() moved here
 * - add: vcgetcstr() gets visual user input and validates it
 *
 * 1999/01/24 ska
 * bugfix: split(): the loop skipping over leading whitespaces crashed
 *	at end of string ('\0') (reported by Eric Kohl <ekohl@abo.rhein-zeitung.de>)
 *
 * 1999/04/23 ska
 * add: match(), matchtok(), match_() test if a word
 *	begins a line, if so, skip over it and following spaces
 * add: comFile(): returns the absolute filename of COMMAND.COM
 * add: comPathFile(): returns the name of a file located in the
 *	directory as COMMAND.COM
 * add: ltrim() / rtrim()
 * chg: ltrim() never returns NULL
 */

#include "config.h"

#include <assert.h>
#include <conio.h>
#include <ctype.h>
#include <dir.h>
#include <direct.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "command.h"
#include "batch.h"
#include <dfn.h>

/*
 * exist -- Checks if a file exists
 *
 */
int exist(const char *fn)
{
  assert(fn);
  return (access(fn, 0) == 0);
}

/*
 * get a character out-of-band and honor Ctrl-Break characters
 */
int cgetchar(void)
{
  int c;

  if ((c = getch()) == 0)
    c = getch() << 8;

  if (c == 3)
    ctrlBreak = 1;

  return c;
}

/* Visible cgetchar() that does not advance the cursor */
int vcgetchar(void)
{
  int ch;

  ch = cgetchar();
  putchar(isprint(ch) ? ch : ' ');
  putchar('\b');
  fflush(stdout);

  return ch;
}

/*
 *  get a visual character that must match one of the supplied
 *  ones
 */
int vcgetcstr(const char *const legalCh)
{
  int ch;

  assert(legalCh);

  fflush(stdout);               /* Make sure the pending output arrives the
                                   screen as we bypass the stdio interface */
  while ((ch = toupper(vcgetchar())) == 0
         || !strchr(legalCh, ch))
  {
    if (cbreak)
    {
      ch = '\3';                /* This is ^C for ^Break */
      break;
    }
    beep();                     /* hit erroreous character */
  }
  putchar('\n');                /* advance to next line */
  return ch;
}

/*
 * Check if Ctrl-Break was pressed during the last calls
 */
int chkCBreak(int mode)
{
  static int leaveAll = 0;      /* leave all batch files */
  int c;

  switch (mode)
  {
    case BREAK_ENDOFBATCHFILES:
      leaveAll = 0;
      return 0;

    case 0:
      if (!bc)
        goto justCheck;

    case BREAK_BATCHFILE:
      if (leaveAll)
        return 1;
      if (!ctrlBreak)
        return 0;

      /* we need to be sure the string arrives on the screen! */
      do
        cprintf("\r\nCtrl-Break pressed.\r\nCancel batch file '%s'? (Yes/No/All) ", bc && bc->bfnam ? bc->bfnam : "");
      while (!strchr("YNA\3", c = toupper(cgetchar())) || !c);

      cputs("\r\n");

      if (c == 'N')
        return ctrlBreak = 0;   /* ignore */

      leaveAll = c == 'A' || c == '\3'; /* leave all batch files */

      break;

    justCheck:
    case BREAK_FORCMD:         /* FOR commands are part of batch processing */
      if (leaveAll)
        return 1;
      /* fall through */

    case BREAK_INPUT:
      if (!ctrlBreak)
        return 0;
      break;
  }

  ctrlBreak = 0;                /* state processed */
  return 1;
}

/*
 * split - splits a line up into separate arguments, deliminators
 *      are spaces and /'s
 */
char **split(char *s, int *args)
{
  char **arg,
  **p,
   *start,
   *q;
  unsigned ac,
    ch;
  int len;

	assert(s);
	assert(args);

  arg = malloc(sizeof(char *));
  if (!arg)
    return NULL;
  *arg = NULL;

  ac = 0;
  while ((ch = *s) != '\0')
  {
    while (isspace(ch = *s) || iscntrl(ch))     /* skip leading spaces */
      ++s;
    if(isspace(ch) || iscntrl(ch))   /* skip leading spaces */
    {
      ++s;
      continue;
    }
    start = s;
    if (ch == '/')              /* the first character can be '/' */
      ++s;
    /* skip to next word delimiter or start of next option */
    while (isprint(ch = *s) && !isspace(ch) && ch != '/')
      ++s;
    if (s != start)             /* a word was found */
    {
      /* add new entry for new argument */
      arg = realloc(p = arg, (ac + 2) * sizeof(char *));
      if (!arg)
      {
        freep(p);
        return NULL;
      }
      /* create new entry */
      q = arg[ac] = malloc((len = s - start) + 1);
      arg[++ac] = NULL;
      if (!q)
      {
        freep(arg);
        return NULL;
      }
      memcpy(q, start, len);
      q[len] = '\0';
    }
  }

  *args = ac;
  return arg;
}

/*
 * freep -- frees memory used for a call to split
 *
 */
void freep(char **p)
{
  char **q;

  assert(p);

  q = p;
  while (*q)
    free(*q++);
  free(p);
}

/*
 * internal function to get the first argument from the parameter list
 * and return a pointer to the beginning of the next argument, or NULL if
 * there are no more arguments
 *
 */
char *parse_firstarg(char *s)
{
  char *place;

  assert(s);

  /* skip over first argument */
  place = s;
  while (*place && !isspace(*place))
    place++;

  if (*place)
  {
    /* mark the end of the first parameter */
    *place++ = 0;

    /* skip over whitespace before next argument */
    while (isspace(*place))
      place++;

    /* if there is something here, return a pointer to it, else NULL */
    if (*place)
      return place;
    else
      return NULL;
  }
  else
    return NULL;
}

/*
 * Internally BEEP (getting the user's attention)
 * Note: One should consider implementing a _visual_ beep
 *   rather an audible beep and an option to switch between them.
 */
void beep(void)
{
  sound(900);
  delay(400);
  nosound();
  delay(100);
}

void beep_low(void)
{
  sound(900);
  delay(400);
  nosound();
  delay(100);
}


/*
 *	Check if the passed line begins with a specified word
 *	The word is matched case-insensitively.
 *	'len' is the length of the word.
 *	On success, the pointer is placed onto the next word.
 *	Return:	0: on failure
 */
int match_(char ** const Xp, const char * const word, int len)
{	char *p;

	assert(Xp);
	assert(word && *word);
	assert(len > 0);

	if(strncmpi(p = *Xp, word, len) == 0) {
		/* line begins with string, now test if it is a word */
		p += len;
		if(*p) {
			if(!isspace(*p))		/* no word boundary */
				return 0;
			/* skip whitespaces */
			while(isspace(*++p));
		}
		*Xp = p;
		return 1;		/* found */
	}

	return 0;
}

/*
 *	Return the absolute filename of the current COMMAND shell
 */
char *comFile(void)
{	char *fnam;

	if((fnam = getEnv("COMSPEC")) == NULL)
		return ComPath;
	return fnam;
}

/*
 * Construct the path of a file to be located in the same directory
 *	as COMMAND.
 *	Dynamically allocated.
 */
char *comPathFile(const char * fnam)
{	char *com, *h;
	int pathLen;

	assert(fnam);

	if((com = comFile()) == NULL)
		return strdup(fnam);

	h = malloc((pathLen = dfnfilename(com) - com) + strlen(fnam) + 1);
	if(!h)
		return NULL;

	memcpy(h, com, pathLen);
	strcpy(h + pathLen, fnam);

	return h;
}

/*
 * Name: ltrim() - left trims a string by removing leading spaces
 * Input: str - a pointer to a string
 * Output: returns a trimmed copy of str
 */
char *ltrim(char *str)
{	char c;

	assert(str);

  while ((c = *str++) != '\0' && isspace(c))
  	;
  
  return str - 1;
}

/*
 * Name: rtrim() - right trims a string by removing trailing spaces
 * Input: str - a pointer to a string
 * Output: str will have all spaces removed from the right.
 */
void rtrim(char * const str)
{	char *p;

	assert(str);

  p = strchr(str, '\0');
  while (--p >= str && isspace(*p))
  	;
  p[1] = '\0';
}

/*
 *	Combines ltrim() & rtrim()
 */
char *trim(char *str)
{	assert(str);
	rtrim(str);
	return ltrim(str);
}
