/*
 * MESSAGES.C - Read messages from strings database located in a file or
 *              at the end of command.com
 *
 *
 * Comments:
 *
 * 27 Jul 1998  John P. Price
 * - started.
 *
 * 03-Dec-1998  John P. Price
 * - Reduced the ammount of memory allocated by not reading in entire index,
 *   but only the index element needed.
 *
 * 1998/12/05 ska
 * - chg: split display_string() into two parts: 1) load string into
 *  local heap, 2) display string
 * - add: load messages into conventional far memory, high memory tried
 *  first
 */

//#define READ_FROM_FILE

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <alloc.h>
#include <dir.h>
#include <dos.h>
#include <io.h>

#include "config.h"
#include "command.h"
#include "strings.h"
#include "suppl.h"

struct indextype
{
  unsigned index;
  unsigned size;
};

#ifdef FEATURE_LOAD_MESSAGES
static unsigned msgSegm = 0;    /* strings segment if loaded */

void uninitModuleStrings(void)  /* called at exit */
{
  freeBlk(msgSegm);
  msgSegm = 0;
}
#endif

static FILE *openStrFile(void)
{
  FILE *f;
  char fdid[10];
  char *theid = STRINGS_ID;
  char *thisstr;

#ifdef READ_FROM_FILE
  if ((thisstr = malloc(MAXPATH)) == NULL)
  {
    printf("Out of memory accessing strings.dat file.\n");
    return NULL;
  }
  strcpy(thisstr, ComDir);
  strcat(thisstr, "STRINGS.DAT");
  if ((f = fopen(thisstr, "rb")) == NULL)
  {
    perror("Can not access strings.dat!");
    free(thisstr);
    return NULL;
  }
  free(thisstr);

#else
  unsigned offs;
  unsigned segs;

  if ((f = fopen(getEnv("COMSPEC"), "rb")) == NULL)
  {
    if ((thisstr = malloc(MAXPATH)) != NULL)
    {
      strcpy(thisstr, ComDir);
      strcat(thisstr, "COMMAND.COM");
      if ((f = fopen(thisstr, "rb")) == NULL)
      {
        perror("Can not open COMMAND.COM to read strings!");
        free(thisstr);
        return NULL;
      }
      free(thisstr);
    }
    else
      return NULL;
  }

//  printf("%s opened.\n",getEnv("COMSPEC"));

  fread(fdid, 2, 1, f);
  if (fdid[0] != 'M' || fdid[1] != 'Z')
  {
    printf("COMMAND.COM not a valid EXE file (MZ not found)!\n");
    fclose(f);
    return NULL;
  }

  fread(&offs, sizeof(unsigned), 1, f);
  fread(&segs, sizeof(unsigned), 1, f);

//  printf("segs = %u  offs = %u\n",segs,offs);

  if (offs != 0)
    segs--;

  fseek(f, (long)segs * 512 + offs, SEEK_SET);

//  printf("memory = %u\n", coreleft());

#endif

  fread(fdid, strlen(theid), 1, f);

  if (strncmp(fdid, theid, strlen(STRINGS_ID)))
  {
    printf("Strings ID not valid!\n");
    fclose(f);
    return NULL;
  }

  return f;
}

#ifdef FEATURE_LOAD_MESSAGES
int loadMsgs(void)              /* load messages into memory */
{
  FILE *f;
  size_t len;
  struct REGPACK r;

  if ((f = openStrFile()) == NULL)
    return 0;

  /* At this point f is positioned at the very first string index
     the data area is NUMBER_OF_STRINGS * sizeof(index) +
     SIZE_OF_STRINGS   */
  len = SIZE_OF_STRINGS + NUMBER_OF_STRINGS * sizeof(struct indextype);
        /* allocation mode: last fit, high first */
        if ((msgSegm = allocBlk(len, 0x82)) == 0)
        {
                fclose(f);
                return 0;
        }

        /* synchronize FILE* with file descriptor */
        lseek(fileno(f), ftell(f), SEEK_SET);
        r.r_ax = 0x3f00;              /* read from file descriptor */
        r.r_bx = fileno(f);           /* file descriptor */
        r.r_cx = len;                 /* size of block to read */
        r.r_ds = msgSegm;             /* segment of buffer to read block to */
        r.r_dx = 0;                   /* offset of buffer to read block to */
        intr(0x21, &r);               /* perform DOS API */
        if ((r.r_flags & 1) != 0      /* read failed */
                        || r.r_ax != len)
        {                             /* couldn't read everything */
                freeBlk(msgSegm);
                msgSegm = 0;
                fclose(f);
                return 0;
        }

        printf("Messages successfully loaded into memory.\n");
        /* strings read successfully */
  fclose(f);
  return 1;
}
#endif

static char *fetchString(unsigned id)
{
  struct indextype string_index;

  FILE *f;
  char *thisstr;

  if (id >= NUMBER_OF_STRINGS)
  {
    printf("Invalid string ID!\n");
    return NULL;
  }

#ifdef FEATURE_LOAD_MESSAGES
  if (msgSegm)
  {                             /* strings cached into memory */
    struct indextype far *idx;

    idx = MK_FP(msgSegm, id * sizeof(*idx));    /* pointer to id's control data */
    if ((thisstr = malloc(idx->size)) == NULL)
    {
      printf("Out of memory reading string in.\n");
      return NULL;
    }
    _fmemcpy(thisstr, MK_FP(msgSegm, NUMBER_OF_STRINGS * sizeof(*idx)
                            + idx->index), idx->size);
    return thisstr;
  }
#endif

  if ((f = openStrFile()) == NULL)
    /* error message already issued */
    return NULL;

  //read index value into string_index
  fseek(f, id * sizeof(string_index), SEEK_CUR);
  fread(&string_index, sizeof(struct indextype), 1, f);
  /* The file position is now (id+1) * sizeof(idx)
     --> to reach the first string NUM_STRINGS - (id+1)
     idx-entries must be skipped */
  fseek(f, (NUMBER_OF_STRINGS - 1 - id) * sizeof(string_index), SEEK_CUR);

  if ((thisstr = malloc(string_index.size)) == NULL)
  {
    printf("Out of memory reading string in.\n");
    fclose(f);
    return NULL;
  }

  fseek(f, string_index.index, SEEK_CUR);
  if (fread(thisstr, string_index.size, 1, f) != 1)
  {
    printf("Cannot read in string #%u.\n", id);
    free(thisstr);
    fclose(f);
    return NULL;
  }

  fclose(f);

  return thisstr;
}

void display_string(unsigned id,...)
{
  char *thisstr;
  va_list argptr;

  if ((thisstr = fetchString(id)) == NULL)
    return;

  va_start(argptr, id);
  vprintf(thisstr, argptr);
  va_end(argptr);
  free(thisstr);
}
