/* choice.c */

/* Waits for the user to press a key, which must be in a list of
   choices */

/*
   Copyright (C) 1994--2002 Jim Hall, jhall@freedos.org

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>			/* strtok  */
#include <ctype.h>			/* toupper */

#include <conio.h>			/* getch (Borland) */
#include <dos.h>			/* sleep (Borland) */

#include "getopt.h"			/* GNU getopt      */
#include "catgets.h"			/* catopen/catgets */


/* Symbolic constants */

/* #define DEBUG				/* debugging   */

#define beep() putch('\a')			/* make a beep */
/* #define beep() printf("**beep**")		/* fake a beep */


/* Function prototypes */

void usage (nl_catd cat);
int strchr_case (const char *list, int c, int case_sensitive);

extern int getopt (int argc, char *const *argv, const char *shortopts);


int
main (int argc, char **argv)
{
  char *s;
  char *choices;			/* the list of choices       */

  int i, opt, ret;

  int key;				/* the key that was pressed */
  int ext_key;				/* the key that was pressed */
  int def_key;				/* default key after wait   */

  int wait;				/* how long we have waited  */
  int def_wait = 0;			/* default time to wait     */

  int opt_beep = 0;			/* do we make a beep?       */
  int show_choices = 1;			/* do we show choices?      */
  int case_sensitive = 0;		/* are we case sensitive?   */

  nl_catd cat;				/* language catalog         */

  /* Open the message catalog */

  cat = catopen ("choice", 0);

  /* Scan command line */

  opterr = 0;					/* no error messages */
  choices = catgets (cat, 3, 0, "yn");		/* set default choices */

  while ((opt = getopt (argc, argv, "bBc:C:nNsSt:T:")) != EOF)
    {
      switch (opt)
	{
	case 'b':
	case 'B': /* Beep */
	  opt_beep = 1;
	  break;

	case 'c':
	case 'C': /* allowable chars */
	  choices = optarg;			/* pointer assignment */
	  if (choices[0] == ':')
	    {
	      /* possible workaround for a getopt bug? */
	      choices++;
	    }
	  break;

	case 'n':
	case 'N': /* do not display choices at end */
	  show_choices = 0;
	  break;

	case 's':
	case 'S': /* be case sensitive */
	  case_sensitive = 1;
	  break;

	case 't':
	case 'T': /* time to wait */
	  s = strtok (optarg, ":,");
	  def_key = s[0];

	  s = strtok (NULL, ":,");

	  /* Check that time to wait contains only numbers */

	  for (i = 0; s[i] != '\0'; i++)
	    {
	      if (!isdigit (s[i]))
		{
#if 0 /* debugging */
		  fprintf (stderr, "Time to wait is not a number\n");
#endif /* debugging */

		  /* show usage, and quit with error */

		  usage (cat);
		  catclose (cat);
		  exit (0);
		}
	    }

	  def_wait = atoi (s);

#ifdef DEBUG /* debugging */
	  printf ("/T triggered:\n");
	  printf ("def_wait=%d\n", def_wait);
	  printf ("def_key=%c\n", def_key);
#endif /* debugging */
	  break;

	default:
	  /* show usage, and quit with error */

	  usage (cat);
	  catclose (cat);
	  exit (0);
	  break;
	} /* switch */
    } /* while */

  /* Fixup stuff - We have to do this after the getopt loop, because
     /C may not have been given yet */

  if (def_wait)
    {
      if (def_wait < 0)
	{
	  /* Fix the def_wait, if it gets set funny */
	  def_wait = 0;
	}

      if (strchr_case (choices, def_key, 0) == 0)
	{
	  /* Make sure def_key is valid. */
	  usage (cat);
	  catclose (cat);
	  exit (0);
	}
    }
  else
    {
      def_key = choices[0];
    }

#ifdef DEBUG /* debugging */
  printf ("After fixup:\n");
  printf ("def_wait=%d\n", def_wait);
  printf ("def_key=%c\n", def_key);
#endif /* debugging */

  /* Display text */

  if (optind < argc)
    {
      for (i = optind; i < argc; i++)
	{
	  printf ("%s ", argv[i]);
	}
    }

  if (show_choices)
    {
      if (!case_sensitive)
	{
	  for (i = 0; choices[i]; i++)
	    {
	      /* Uppercase the choices */
	      choices[i] = toupper(choices[i]);
	    }
	}

      /* print the choices */

      printf ("[%s]? ", choices);
    }

  /* Make a beep */

  if (opt_beep)
    {
      beep();
    }

  /* Grab the key.  Don't exit until the key is in the choice list. */

  for (wait = 1, ret = 0; ret == 0; wait++ )
    {
      if (def_wait > 0)
	{
#ifdef DEBUG /* debugging */
	      printf ("sleep...");
#endif /* debugging */
	  sleep (1);			/* wait for 'n' seconds */
	}

      /* Check for a key */

      while (kbhit())
	{
	  key = getch();		/* grab the key */
	  if (!key)
	    {
	      ext_key = getch();	/* extended key */
	    }

	  /* Is the key in the list? */

	  if (ret = strchr_case (choices, key, 1))
	    {
	      break;			/* out of loop */
	    }

	  else if ((!case_sensitive) && (ret = strchr_case (choices, key, 0)))
	    {
	      break;			/* out of loop */
	    }

	  if (opt_beep)
	    {
#ifdef DEBUG /* debugging */
	      printf ("**BEEP**");
#endif /* debugging */
	      beep();
	    }
	} /* while kbhit */

      /* Did we run out of time yet? */

      if ( (def_wait > 0) && (wait >= def_wait) )
	{
	  ret = strchr_case (choices, def_key, 0);
	  break;			/* out of loop */
	}
    } /* for */

  /* Out of loop!  'ret' has the return value. */

  printf ("\n");			/* force a new line */
  catclose (cat);
  exit (ret);	       			/* ret starts counting at 1 */
}

void
usage (nl_catd cat)
{
  char *s1, *s2;

  s1 = catgets (cat, 0, 0, "Waits for the user to press a key, from a list of choices");
  printf ("CHOICE version 3.8, Copyright (C) 1994--2003 Jim Hall, jhall@freedos.org\n");
  printf ("%s\n", s1);

  s1 = catgets (cat, 1, 0, "Usage:");
  printf ("%s ", s1);

  s1 = catgets (cat, 0, 1, "choices");
  s2 = catgets (cat, 0, 2, "text");
  printf ("CHOICE [ /B ] [ /C[:]%s ] [ /N ] [ /S ] [ /T[:]c,nn ] [ %s ]\n", s1, s2);

  s2 = catgets (cat, 0, 8, "Sound an alert (beep) at prompt");
  printf ("  /B\t\t-  %s\n", s2);

  s2 = catgets (cat, 0, 3, "Specifies allowable keys.  Default is:");
  printf ("  /C[:]%s\t-  %s ", s1, s2);

  s2 = catgets (cat, 3, 0, "yn");
  printf ("%s\n", s2);

  s1 = catgets (cat, 0, 4, "Do not display the choices at end of prompt");
  printf ("  /N\t\t-  %s\n", s1);

  s1 = catgets (cat, 0, 5, "Be case-sensitive");
  printf ("  /S\t\t-  %s\n", s1);

  s1 = catgets (cat, 0, 6, "Automatically choose key c after nn seconds");
  printf ("  /T[:]c,nn\t-  %s\n", s1);

  s1 = catgets (cat, 0, 7, "The text to display as a prompt");
  s2 = catgets (cat, 0, 2, "text");
  printf ("  %s\t\t-  %s\n", s2, s1);
}

int
strchr_case (const char *list, int c, int case_sensitive)
{
  /* Very similar to strchr, except that I may want to be case sensitive */

  /* Scans a string in the forward direction, looking for a specific
     character.  Finds the first occurrence of the character 'c' in
     the string 'list'. */

  /* Returns: an index to the first occurrence of the character 'c' in
     'list' (starting at 1), or 0 if not found */

  int i;

  for (i = 0; list[i]; i++)
    {
      if ( ((!case_sensitive) && (toupper(c) == toupper(list[i]))) ||
	   ((case_sensitive) && (c == list[i])) )
	{
	  return (i+1);
	}
    } /* for */

  return (0);
}
