/*
   RtDots.c - root dots checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"
#include "..\errmsgs\errmsgs.h"

static BOOL DOTFinder(RDWRHandle handle, struct DirectoryPosition* pos,
                      void** structure);
static BOOL RootDirDropper(RDWRHandle handle,
                           struct DirectoryPosition* pos,
                           void** structure);


/*============================== Checking ==============================*/

/*************************************************************************
**                           RootDirDOTFinder
**************************************************************************
** Searches trough the root directory and looks for '.' and '..' entries.
**************************************************************************/

RETVAL RootDirDOTFinder(RDWRHandle handle)
{
    BOOL invalid = FALSE, *pinvalid = &invalid;

    if (!TraverseRootDir(handle, DOTFinder, (void**)&pinvalid, TRUE))
       return ERROR;

    return (invalid) ? FAILED : SUCCESS;
}

/*************************************************************************
**                              DOTFinder
**************************************************************************
** Takes a root directory entry and looks wether it is equal to '.' or '..'
**************************************************************************/

static BOOL DOTFinder(RDWRHandle handle, struct DirectoryPosition* pos,
                      void** structure)
{
    char* fixes = "";
    BOOL *invalid = *((BOOL**) structure);
    struct DirectoryEntry entry;

    if (!GetDirectory(handle, pos, &entry))
       return FAIL;

    if (IsDeletedLabel(entry) ||
        IsLFNEntry(&entry))
    {
       return TRUE;
    }
          
    if (IsCurrentDir(entry))                             /* '.' */
    {
       ReportFileSysError(". found in root directory",
                          0,
                          &fixes,
                          0,
                          FALSE);

       *invalid = TRUE;
    }

    if (IsPreviousDir(entry))                            /* '..' */
    {
       ReportFileSysError(".. found in root directory",
                          0,
                          &fixes,
                          0,
                          FALSE);            

       *invalid = TRUE;
    }
    
    return TRUE;
}

/*============================== Fixing ==============================*/

/*************************************************************************
**                           DropRootDirDOTs
**************************************************************************
** Searches trough the root directory and looks for '.' and '..' entries.
** If there is such an entry it is deleted.
**************************************************************************/

RETVAL DropRootDirDOTs(RDWRHandle handle)
{
    return (TraverseRootDir(handle, RootDirDropper, NULL, TRUE)) ?
           SUCCESS : ERROR;
}

/*************************************************************************
**                           RootDirDropper
**************************************************************************
** Checks wether the directory entry is '.' or '..'.
** If it is it is deleted.
**************************************************************************/

static BOOL RootDirDropper(RDWRHandle handle,
                           struct DirectoryPosition* pos,
                           void** structure)
{
    char* fixes = "";
    struct DirectoryEntry entry;

    structure = structure;

    if (!GetDirectory(handle, pos, &entry))
       return FAIL;
       
    if (IsDeletedLabel(entry) ||
        IsLFNEntry(&entry))
    {
       return TRUE;
    }
    
    if (IsCurrentDir(entry))
    {
       if (ReportFileSysError(". found in root directory",
                              0,
                              &fixes,
                              0,
                              TRUE) != REPORT_IGNORED)
       {                           
          MarkEntryAsDeleted(entry);
          if (!WriteDirectory(handle, pos, &entry))
             return FAIL;
       }
    }

    if (IsPreviousDir(entry))
    {
       if (ReportFileSysError(".. found in root directory",
                              0,
                              &fixes,
                              0,
                              TRUE) != REPORT_IGNORED)
       {                           
          MarkEntryAsDeleted(entry);
          if (!WriteDirectory(handle, pos, &entry))
             return FAIL;
       }
    }
    
    return TRUE;
}