/*
   dotpoint.c - '.' checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

static BOOL DotPointerChecker(RDWRHandle handle,
                              struct DirectoryPosition* pos,
			      struct DirectoryEntry* entry,
                              void** structure);

static BOOL CheckDots(RDWRHandle handle, CLUSTER firstcluster,
                      BOOL* invalid, BOOL fixit);

static BOOL DotPointerAdjuster(RDWRHandle handle,
                               struct DirectoryPosition* pos,
			       struct DirectoryEntry* entry,
                               void** structure);

/* Checking */

RETVAL CheckDotPointer(RDWRHandle handle)
{
   BOOL invalid = FALSE, *pinvalid = &invalid;

   if (!FastWalkDirectoryTree(handle, DotPointerChecker, (void**) &pinvalid))
      return ERROR;

   return (invalid) ? FAILED : SUCCESS;
}

static BOOL DotPointerChecker(RDWRHandle handle,
                              struct DirectoryPosition* pos,
			      struct DirectoryEntry* entry,
                              void** structure)
{
   CLUSTER firstcluster;
   BOOL* invalid = *((BOOL**) structure);

   pos = pos;

   if (IsLFNEntry(entry))
      return TRUE;

   if (IsDeletedLabel(*entry))
      return TRUE;

   if (entry->attribute & FA_DIREC)
   {
      firstcluster = GetFirstCluster(entry);
      if (firstcluster)
      {
	 if (!CheckDots(handle, firstcluster, invalid, FALSE))
	    return FAIL;
      }
   }
   
   return TRUE;
}

/* Fixing */

RETVAL AdjustDotPointer(RDWRHandle handle)
{
   return (FastWalkDirectoryTree(handle, DotPointerAdjuster, NULL)) ?
          SUCCESS : ERROR;
}

static BOOL DotPointerAdjuster(RDWRHandle handle,
                               struct DirectoryPosition* pos,
			       struct DirectoryEntry* entry,
                               void** structure)
{
   CLUSTER firstcluster;

   structure = structure, pos = pos;

   if (IsLFNEntry(entry))
      return TRUE;

   if (IsDeletedLabel(*entry))
      return TRUE;

   if (entry->attribute & FA_DIREC)
   {
      firstcluster = GetFirstCluster(entry);
      if (firstcluster)
      {
	 if (!CheckDots(handle, firstcluster, NULL, TRUE))
	    return FAIL;
      }
   }
   
   return TRUE;
}

/* Common */

static BOOL CheckDots(RDWRHandle handle, CLUSTER firstcluster,
                      BOOL* invalid, BOOL fixit)
{
   struct DirectoryPosition pos = {0, 0};
   struct DirectoryEntry entry;

   if (!GetNthDirectoryPosition(handle, firstcluster, 0, &pos))
      return FALSE;

   if ((pos.sector == 0) && (pos.offset == 0))
   {
      if (!fixit) printf("Found an empty directory\n");
      return TRUE;
   }

   if (!GetDirectory(handle, &pos, &entry))
      return FALSE;

   if (!IsCurrentDir(entry))
   {
      printf("Found a directory without '.' as first entry\n");
      return TRUE;
   }

   if (GetFirstCluster(&entry) != firstcluster)
   {
      printf("Found a directory for which the '.' entry is wrong\n");
      if (fixit)
      {
         SetFirstCluster(firstcluster, &entry);
         return WriteDirectory(handle, &pos, &entry);
      }
      else
      {  
         *invalid = TRUE;
      }
   }

   return TRUE;
}
