/***
*diskio.c - drive initialization/read-write module
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   Handles low-level DOS Sector reads/writes.
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
*******************************************************************************/

/*
   Versions:

   3.4   02-Jul-1998    __GNUC__
   3.3   07-Apr-1998    put back in check for remote drives; IOCTL read
                        failure during diskinit() will re-try with ABS
   3.2   18-Jan-1998    finalized the I/O function determination; logdisk()
                        no longer sets the dos drive
   3.1   29-Nov-1997    sec_buf checksum and pointer checks (a little late)

   3.0   28-Nov-1997    diskio() has been completely rewritten; added
                        diskinit() (and it's MUCH better!); now use
                        enums and data size typedefs; all globals
                        (except four) now have underscores in them;
                        INT 13h support; DOS and INT functions moved out

   2.6   21-Feb-1997    added underscores to a few non-ANSI things
   2.5   13-Sep-1996    removed some unused functions
   2.4   18-Nov-1995    added MID and get_media_id()
   2.3   23-Mar-1995    fixed logicalsector()
   2.2   20-Dec-1994    lots (lost track); cleaned up DISK_INIT
                        code; added underscores on non-ANSI stuff
   2.1   28-Nov-1993

   Release Notes:

   The heart, but not soul, of DISKED.

   Programming Notes:

   Lots of globals. I thought of putting all these in a struct but
   the amount of other modules that would need changing is just too
   great...

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <assert.h>
#ifdef __GNUC__
#include <malloc.h>
#endif
#include <dos.h>
#ifdef __WATCOMC__                  /* they fucked up! */
#define diskinfo_t _diskinfo_t
#endif
#include <bios.h>

#include "general.h"
#include "disked.h"
#include "diskio.h"
#include "alloc.h"
#include "error.h"
#include "dosio.h"
#include "int13.h"
#include "direct.h"

#define DIR_ENTRY_SIZE 32

int removable;          /* drive is removable */
UINT32 drive_size;      /* drive size in bytes */
int ext_drive;          /* if greater than 32MB */
UINT16 sec_size;        /* sector size, usually 512 bytes */
UINT16 max_head;        /* maximum disk head number */
UINT16 max_sector;      /* maximum sector number */
UINT16 max_track;       /* maximum track number */
UINT32 hidden_secs;     /* number of hidden sectors on a hard drive */
                        /*  Partition info, first track */
UINT16 reserved_secs;   /* number of reserved sectors */
                        /*  Boot sector(s) */
char dsk_format[8];

      /* logical (DOS) values */

UINT32 num_sectors;     /* total number of sectors */
UINT16 secs_cluster;    /* sectors per cluster */
UINT16 num_clusters;    /* maximum cluster no. */
UINT16 cluster_size;    /* cluster size in bytes */
UINT16 dir_sectors;     /* number of sectors of root */
UINT16 secs_fat;        /* sectors per FAT */
UINT16 num_fats;        /* number of FATs */
UINT16 dir_entries;     /* number of directory entries */
UINT16 data_sector;     /* first data sector */
UINT16 dir_sector;      /* start of root directory */
UINT16 avail_clusters;
MID mid;                /* media ID structure */
int fat_size;           /* 12 or 16 */

      /* variable drive values */

int cur_disk;           /* drive number 0+ */
int max_drive;          /* maximum drive number */
UINT32 log_sector;      /* logical sector no. */
UINT32 back_log_sector;
UINT16 phy_head;        /* physical head no. 0-max_head */
UINT16 phy_track;       /* physical track no. 0-max_track */
UINT16 phy_sector;      /* physical sector number 1-max_sector */
BYTE *sec_buf;          /* sector buffer */
UINT32 sec_bufp;        /* " pointer copy */
UINT32 sec_bufx;        /* " checksum */

      /* misc. stuff */

int disk_moved;         /* changed position */
int diskio_error;       /* DOS/BIOS function result */

DISK_FUNC diskio_function = DISK_ABS;

      /* static functions */

static int diskinit(int arg);
static void diskinc(long arg);
static int diskrdwr(DISKIO_COMMAND cmd, long arg, BYTE *buffer);
static int returnerror(int function, int error);
static int getdrivesize(int drv, UINT16 *secsize, UINT32 *drivesize, DPB *dpb);
static void setdiskfunction(DOS_ERROR dosstatus, int removable);
static int getbootsec(int drv, BYTE *buf);
static int checkboot(BYTE *tbuf, UINT16 secsize);


#pragma on (check_stack)      /* diskio() is recursive */

/***
*diskio  -  Disk Input/Output function.
*
*   Pass command, argument, and pointer to sector size buffer.
*
****/

extern int diskio(DISKIO_COMMAND command, long arg, BYTE *buffer)
{
static UINT32 saved_log_sector;
DISK_FUNC function;                 /* what we are doing */
DISKIO_STATUS status;               /* how it played out */

   status = DISK_OK;                /* I always assume things work */
   diskio_error = 0;

   switch (command)
   {
      case DISK_SAVE:
         saved_log_sector = log_sector;
         break;

      case DISK_REST:
         if (log_sector == saved_log_sector)
            break;
         log_sector = saved_log_sector;
         disk_moved = 1;
         break;

      case DISK_SET:
         if (arg < 0L || (UINT32)arg >= num_sectors)
            status = DISK_BOUNDS;
         else
         {
            back_log_sector = log_sector;
            log_sector = arg;
            disk_moved = 1;
         }
         break;

      case DISK_INC:
         diskinc(arg);
         disk_moved = 1;
         break;

      case DISK_READ:                     /* !MUST RETURN IF DISK_OK! */
         function = diskio_function;
         status = diskrdwr(DISK_READ,arg,buffer);
         if (status == DISK_OK)
            return status;
         break;

      case DISK_WRITE:                    /* !MUST RETURN IF DISK_OK! */
         function = diskio_function;
         status = diskrdwr(DISK_WRITE,arg,buffer);
         if (status == DISK_OK)
            return status;
         break;

      case DISK_INIT:                     /* !MUST RETURN IF DISK_OK! */
         function = DISK_LOG;
         status = diskinit((int)arg);
         if (status <= DISK_OK)           /* NOTE the <= here! */
         {
            disk_moved = 1;               /* force sector display */
            return status;
         }
         if (status == DISK_READ_ERR)
            status = returnerror(diskio_function,diskio_error);
         else if (status == DISK_DOS_ERR)
            status = returnerror(DISK_DOS,diskio_error);
         else
            status = returnerror(function,status);
         return status;
         break;

      default:
         assert(1);
         return status;
         break;
   }

   if (status == DISK_OK)
   {
      if (disk_moved)
      {
         physicalsector(log_sector,&phy_track,&phy_sector,&phy_head);
         return diskio(DISK_READ,-1L,buffer);
      }
      return DISK_OK;
   }

   /* here if error */

   diskio_error = status;
   return returnerror(function,status);
}

#pragma default (check_stack)

/***
*diskinit   -  check out and log in a disk
*
*
*  There are two status values, the result of the function (success,
*  warning, error), and the error number if error. The error can be
*  a DOS error (failure of a DOS function), disk error (failure of a
*  disk read), or no DPB and corrupt BOOT sector. The warnings can be
*  DPB not supported (it'll use the BOOT sector) or BOOT corrupt (it'll
*  use the DPB if available).
*
*  The result of the function is returned, any error will set
*  'diskio_error' directly.
*
*  Return Summary:            return         diskio_error
*
*              success        DISK_OK        DISK_OK
*              no DPB         DISK_USE_BOOT  DISK_OK
*              no BOOT        DISK_USE_DPB   DISK_OK
*              memory         DISK_MEMORY    n/a
*              network disk   DISK_REMOTE    n/a
*              no DPB, BOOT   DISK_BOOT_ERR  n/a
*              int 21h error  DISK_DOS_ERR   DOS error
*              read error     DISK_READ_ERR  READ error
*
*              DOS error is the function 59h value
*              READ error is the int 25h or int 13h return value
*              IOCTL errors are DOS errors
****/

static int diskinit(int drv)
{
BOOT boot;
BYTE *tbuf;
DPB dpb = { 0 };
UINT16 secsize;
UINT32 drivesize;
int extdrive;
DOS_ERROR dosstatus;
DISKIO_STATUS status;
DISKIO_STATUS drvstatus;

   if (is_drive_remote(drv))
      return DISK_REMOTE;

   status = DISK_OK;

   /* see what DOS says the drive is */
   /* we need to know the sector and drive sizes before reading */

   extdrive = ext_drive;   /* save */

   dosstatus = getdrivesize(drv,&secsize,&drivesize,&dpb);

   if (dosstatus != DOS_OK && dosstatus != DOS_EINVFNC)
   {
      diskio_error = dosstatus;
      return DISK_DOS_ERR;
   }

#ifdef __GNUC__         /* I've got to make a global sector buffer... */
   if (secsize != 512)
      return DISK_SECSIZE;
#endif

   if (dosstatus != DOS_OK)               /* RAM Drive */
      status = DISK_USE_BOOT;             /* to indicate that there is no DPB */

   ext_drive = (drivesize > 32L * 1024L * 1204L);

   /* for removable media we should read the BOOT sector */

   removable = non_removable(drv) == 0;

   /* set diskio_function */

   setdiskfunction(dosstatus,removable);

   /* get boot sector */

   if ((tbuf = alloc(secsize,sizeof(char))) == NULL)
   {
      ext_drive = extdrive;   /* restore */
      return DISK_MEMORY;
   }

   if ((drvstatus = getbootsec(drv,tbuf)) != DISK_OK)
   {
      freep(tbuf);
      diskio_error = drvstatus;
      ext_drive = extdrive;   /* restore */
      return DISK_READ_ERR;
   }

   if (checkboot(tbuf,secsize) != DISK_OK)
   {
      if (dosstatus == DOS_EINVFNC)
      {
         freep(tbuf);
         ext_drive = extdrive;   /* restore */
         return DISK_BOOT_ERR;
      }
      status = DISK_USE_DPB;           /* invalid BOOT sector, use DPB */
   }

   /* disk is ok */

   /* use DPB for non-removable */

   if (status == DISK_USE_BOOT || (removable && status != DISK_USE_DPB))
      memcpy(&boot,tbuf,sizeof(BOOT));
   else
      memcpy(&boot.sec_size,&dpb.bpb,sizeof(BPB));

   /* set all the globals */

   cur_disk = drv;
   sec_size = secsize;
   drive_size = drivesize;
   secs_cluster = boot.secs_cluster;
   reserved_secs = boot.reserved_secs;
   num_fats = boot.num_fats;
   dir_entries = boot.dir_entries;
   num_sectors = (UINT32)boot.num_sectors;
   secs_fat = boot.secs_fat;
   max_sector = boot.secs_track;
   max_head = boot.num_heads;

   /*
      ABBERATION: With DOS v6.0+ the BOOT field 'hidden_sectors'
      changed from WORD to DWORD and RAMDRIVE.SYS still thinks that
      it is WORD.
   */

   if (dosstatus != DOS_EINVFNC)
      hidden_secs = boot.hidden_sectors;
   else
      hidden_secs = 0;

   if (num_sectors == 0L)
      num_sectors = boot.huge_sectors;

   /*
      There needs to be much more error checking. Here's one for example.
      It's a kludge to get around DOS 3.0 and earlier BOOT sectors in
      which the hidden sectors field is a WORD and the possibility
      exists that the second WORD is non-zero.
   */

   if (hidden_secs > num_sectors)
      hidden_secs = 0;


   dir_sectors = dir_entries/(sec_size/DIR_ENTRY_SIZE);
   data_sector = (secs_fat*num_fats) + dir_sectors + reserved_secs-1;
   dir_sector = (secs_fat*num_fats) + reserved_secs;
   cluster_size = sec_size * secs_cluster;
   drive_size = (UINT32)sec_size * num_sectors;
   num_clusters =
      (UINT16)(((num_sectors-((UINT32)secs_fat*(UINT32)num_fats)-(UINT32)dir_sectors-1)
       / (UINT32)secs_cluster)+1);
   max_track =
      (UINT16)((((num_sectors+hidden_secs)/(UINT32)max_head)
       / (UINT32)max_sector)-1);

   fat_size = (num_clusters < 4096) ? 12 : 16;

   /* do volume specific stuff */

   memset(&mid,0,sizeof(MID));

   if (get_media_id(drv,&mid) != DOS_OK)  /* get volume data */
   {
      get_volume(drv,mid.vollabel);
      strcpy(mid.filesys,(fat_size == 12) ? "FAT12" : "FAT16");
   }

   memcpy(dsk_format,tbuf+3,8);           /* extract format; if BOOT is bad, */

   freep(tbuf);

   return status;
}

/***
*getdrivesize  -  ahhh, get drive size
*
****/

static int getdrivesize(int drv, UINT16 *secsize, UINT32 *drivesize, DPB *dpb)
{
DOS_ERROR status;
FREESPACE fs = { 0 };

   dpb->special = 1;
   status = get_block_info(drv,dpb);

   if (status == DOS_ENOFILE)             /* Windows 3.x kludge */
      status = DOS_EINVFNC;

   if (status == DOS_OK)                  /* standard DOS drive */
   {
      *secsize = dpb->bpb.sec_size;
      *drivesize = (UINT32)dpb->bpb.num_sectors;   /* (temp at first) */
      if (*drivesize == 0L)                        /* (need to do some) */
         *drivesize = dpb->bpb.total_sectors;      /* (multiplying) */
      *drivesize *= (UINT32)*secsize;
   }
   else if (status == DOS_EINVFNC)        /* non-standard */
   {
      int i;
      error.num = -1;                              /* (incase crit happens) */
      i = doscall(0x3600,0,0,drv,&fs);             /* (this reads the disk) */
      if (i >= 0xff || error.num != -1)
         status = exterror();
      else
      {
         *secsize = fs.sec_size;
         *drivesize = (UINT32)*secsize * fs.num_clusters * fs.secs_cluster;
      }
   }
   return status;
}

/***
*setdiskfunction  -
*
*  Sector I/O Functions by OS (F = floppy, H = hard, R = ramdrive)
*
*                                DOS   Win31/95 WinNT
*
*  dosread (INT 25h/26h)         FHR   FH       FH
*  biosread (INT 13h)            FH    FH       F
*  ioctlread (INT 21h/440Dh)     F     F        F
*
*  notes:   INT 26h is not supported by Win31/95
*           INT 13h is for physical drives only
*
****/

static void setdiskfunction(DOS_ERROR dosstatus, int removable)
{
   if (dosstatus != DOS_OK)               /* RAM Drive */
   {
      diskio_function = DISK_ABS;
   }
   else
   {
      if (removable)
      {
         if (Ioctl)
            diskio_function = DISK_IOCTL;
         else if (Int13)
            diskio_function = DISK_INT13;
         else
            diskio_function = DISK_ABS;
      }
      else
      {
         if (_osminor == 0x32)               /* WinNT */
            diskio_function = DISK_ABS;
         else if (nPartitions && Int13)
            diskio_function = DISK_INT13;
         else
            diskio_function = DISK_ABS;
      }
   }
}

/***
*getbootsec -  read BOOT sector
*
*  Not all drives are supported by INT 13h. The BOOT sector has
*  different physical positions depending on drive type.
*
****/

static int getbootsec(int drv, BYTE *buf)
{
DISKIO_STATUS status;

   /* Note: drive parameters are NOT known at this point. */

   if (diskio_function == DISK_ABS)
   {
      status = dosio(DOS_READ,ext_drive,drv,0,buf);
   }
   else
   {
      UINT16 t,h,s;
      t = 0;  h = 1;
      s = (drv <= 2) ? 0 : 1;                /* floppy or hard drive */

      if (diskio_function == DISK_INT13)
      {
         status = int13(BIOS_READ,drv,t,h,s,1,buf);

         /* if command invalid (not supported, like for partitions) */
         /* retry with DISK_ABS */

         if (status == 1)
            diskio_function = DISK_ABS;

         status = dosio(DOS_READ,ext_drive,drv,0,buf);
      }
      else if (diskio_function == DISK_IOCTL)
      {
         status = diskioctl(DOS_READ,drv,t,h,s,buf);

         /* if command invalid (not supported, like for Zip Drives) */
         /* retry with DISK_ABS */

         if (status != -1)
            diskio_function = DISK_ABS;

         status = dosio(DOS_READ,ext_drive,drv,0,buf);
      }
      else
      {
         status = DISK_FUNC_ERR;
      }
   }
   return status;
}

/***
*checkboot  -  check validity of BOOT sector
*
****/

static int checkboot(BYTE *tbuf, UINT16 secsize)
{
UINT16 tmp;

   /* quick and simple check for bad boot record */
   /* ramdrives might not have the AA55 signature */
   /*  well, RAMDRIVE.SYS does not have the AA55 signature */

   tmp = *(UINT16*)(tbuf+11);
   if (
      (tbuf[0]!=0xeb && tbuf[0]!=0xe9 && tbuf[0]!=0) ||
      (tbuf[2]!=0x90 && tbuf[2]!=0) ||
      tmp != secsize)
      return !DISK_OK;
   return DISK_OK;
}

/* end diskinit() functions */

/***
*diskrdwr   -  front end to either INT 25h, INT 13h or INT 440Dh
*
*  arg == -1 use current log_sector, else use arg
****/

static int diskrdwr(DISKIO_COMMAND cmd, long arg, BYTE *buf)
{
DOS_FUNCTION func;
BIOS_FUNCTION bfunc;
DISKIO_STATUS status;

   if (diskio_function == DISK_ABS)
   {
      UINT32 sector;
      sector = (arg >= 0) ? arg : log_sector;
      func = (cmd == DISK_WRITE) ? DOS_WRITE : DOS_READ;
      status = dosio(func,ext_drive,cur_disk,sector,buf);
   }
   else
   {
      UINT16 t,s,h;
      if (arg >= 0)
         physicalsector(arg,&t,&s,&h);
      else
         physicalsector(log_sector,&t,&s,&h);

      if (diskio_function == DISK_INT13)
      {
         bfunc = (cmd == DISK_WRITE) ? BIOS_WRITE : BIOS_READ;
         status = int13(bfunc,cur_disk,t,s,h,1,buf);
      }
      else if (diskio_function == DISK_IOCTL)
      {
         func = (cmd == DISK_WRITE) ? DOS_WRITE : DOS_READ;
         status = diskioctl(func,cur_disk,t,s,h,buf);
      }
      else
         status = DISK_FUNC_ERR;
   }

#ifdef DISKED
   if (cmd == DISK_READ && status == DISK_OK)
   {
      if (buf == sec_buf)
      {
         if (save_sec)                          /* copy of sector data */
            memcpy(save_sec,sec_buf,sec_size);
         sec_bufx = chksum(sec_buf,sec_size);   /* get checksum */
      }
   }
#endif
   return status;
}


/***
*diskiochk  -  simple sector buffer integrity checks
*
****/

extern int diskiochk(void)
{
   if ((UINT32)sec_buf != sec_bufp)             /* pointer changed? */
      return 2;
   if (chksum(sec_buf,sec_size) != sec_bufx)    /* checksum changed? */
      return 1;
   return 0;
}

/***
*chksum  -  checksum on buffer
*
****/

extern UINT32 chksum(BYTE *buf, UINT16 siz)
{
size_t i;
UINT32 c;

   for (c = 0, i = 0; i < siz; i++)
      c += buf[i];
   return c;
}

/***
*diskinc - Increment (or decrement) position, wrapping
*          around at the end or beginning.
****/

static void diskinc(long arg)
{
UINT32 tsector;

   back_log_sector = tsector = log_sector;         /* save current */

   log_sector += arg;                              /* set new position */
   if (log_sector >= num_sectors)                  /* gone past end? */
   {
      if (arg < 0L)
         log_sector = num_sectors + arg + tsector;
      else
         log_sector -= num_sectors;
   }
}

/* conversion/translation functions */

/****
*ncluster   -  next or previous cluster
*
*   Increment or decrement position by clusters.
*   If gone past the last cluster go to the first.
*   If gone before the first go to the last.
*
*   Returns the logical sector number of the first sector
*   of the cluster.
****/

extern UINT32 ncluster(register int arg)
{
UINT32 temp = log_sector;

   if (arg > 0)
      temp += (UINT32)secs_cluster;
   else
      temp -= (UINT32)secs_cluster;
   temp -= clustersector(log_sector);
   if (temp < data_sector || temp >= num_sectors)
   {
      if (arg > 0)                               /* first cluster */
         return(data_sector+1);
      else                                       /* last cluster */
         return(clustertosector(num_clusters));
   }
   return(temp);
}

/* Physical/Logical conversion routines.
 * I figured all of this stuff out by myself with the help of a
 * programmable calculator.  I have never found any of this stuff
 * documented anywhere -- it may be somewhere, I just never found it.
 */

/* convert physical parameters to logical sector number */

/*
   Whew, the bounds checking here is to get around the problem
   of when hidden_secs is not a multiple of sectors per track.
   See HISTORY.TXT for a bit more info.
*/

extern UINT32 logicalsector(UINT16 track, UINT16 sector, UINT16 head)
{
UINT32 trk;

   head *= (UINT16)max_sector;
   trk = (UINT32)track * (UINT32)max_head * (UINT32)max_sector;
   trk += ((UINT32)head + (UINT32)(sector-1)) - hidden_secs;

   if ((long)trk < 0)                  /* MUST test < 0 first! */
      trk = 0;
   else if (trk >= num_sectors)
      trk = num_sectors-1;
   return trk;
}

/* convert logical sector to physical parameters (sets our globals) */

extern void physicalsector(UINT32 sec, UINT16 *t, UINT16 *s, UINT16 *h)
{
   sec += hidden_secs;

   *t  = (UINT16)(sec / ((UINT32)max_sector * (UINT32)max_head));
   *s  = (UINT16)(sec % ((UINT32)max_sector)+1);
   *h  = (UINT16)(sec % ((UINT32)max_sector * (UINT32)max_head));
   *h /= max_sector;
}

/* convert logical sector number to the sector within the cluster number */

extern UINT16 clustersector(UINT32 sector)
{
   if (sector <= (UINT32)data_sector)
      return 0;
   return (UINT16) ((sector - (UINT32)(data_sector+1)) % (UINT32)secs_cluster);
}

/* convert cluster number to logical sector number */

extern UINT32 clustertosector(UINT16 cluster)
{
   return (UINT32)  (((UINT32)(cluster-2) * (UINT32)secs_cluster)
      + (UINT32)(data_sector+1));
}

/* convert the logical sector number to the cluster number */

extern UINT16 sectortocluster(UINT32 sector)
{
   if (sector <= (UINT32)data_sector)
      return 0;
   return (UINT16)
      ( ( (sector-(UINT32)(data_sector+1)) / (UINT32)secs_cluster ) + 2);
}


/* error message handler */

static char *module_name = "diskio";

static char *diskioerr_msg[] = {
   "BOOT sector invalid",
   "Cannot Log network drives",
   "Sector out of bounds (oh oh...)",
   "Not Enough Memory",
   "DISKIO Function error",
#ifdef __GNUC__
   "Sector size not 512",
#endif
   "unknown log error"
};

static char *diskio_func[] = {
   "INT 25h/26h",
   "INT 13h",
   "IOCTL"
};

extern const char *diskiofunc(void)
{
   if ((unsigned)diskio_function > 2)
      return NULL;
   return diskio_func[diskio_function];
}

static int returnerror(int function, int error)
{
const char *errmsg;
const char *iofunc;

   iofunc = NULL;

   /* get the interrupt error messages */

   if (function == DISK_ABS)
   {
      errmsg = dosioerror(error);
      iofunc = diskiofunc();
   }
   else if (function == DISK_INT13)
   {
      errmsg = int13error(error);
      iofunc = diskiofunc();
   }
   else if (function == DISK_LOG)   /* a diskio() error */
   {
      if (error < 0 || error > (int)(sizeof(diskioerr_msg)/sizeof(char*)))
         error = (sizeof(diskioerr_msg)/sizeof(char*))-1;
      errmsg = diskioerr_msg[error];
   }
   else
   {
      errmsg = doserror(error);                /* some other DOS error */
      if (strcmp(errmsg,"error") == 0)         /* or DISK_IOCTL error */
         set_err_arg("%02Xh",error);
      if (function == DISK_IOCTL)
         iofunc = diskiofunc();
   }
   set_error(errmsg,module_name,error,module_name);
   if (iofunc)
      set_err_info(iofunc);

   if (function == DISK_INT13)
      error += BIOS_ERROR_OFFSET;

   return error;
}
