/*
   ChkEntrs.c - directory entry checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"
#include "..\errmsgs\errmsgs.h"
#include "..\errmsgs\PrintBuf.h"

#define INVALIDSFNCHARS "\0x22\0x2a\0x2b\0x2c\0x2e\0x3a\0x3b\0x3c\0x3d" \
                        "\0x3e\0x3f\0x5b\0x5c\0x5d\0x7c"

static BOOL EntryChecker(RDWRHandle handle, struct DirectoryPosition* pos,
			 struct DirectoryEntry* entry, void** structure);
static BOOL EntryFixer(RDWRHandle handle, struct DirectoryPosition* pos,
		       struct DirectoryEntry* entry, void** structure);
static BOOL LookAtEntry(RDWRHandle handle, struct DirectoryPosition* pos,
			BOOL fixit);
static unsigned long CalculateFileSize(RDWRHandle handle,
                                       CLUSTER firstcluster); 
static BOOL CheckDots(RDWRHandle handle, CLUSTER firstcluster, BOOL fixit,
                      struct DirectoryPosition* pos, 
                      struct DirectoryEntry* entry); 
static BOOL FileSizeCalculator(RDWRHandle handle, CLUSTER label,
                               SECTOR datasector, void** structure);  
static BOOL CheckTheFileChain(RDWRHandle handle,
			      struct DirectoryPosition* filepos,
                              struct DirectoryEntry* entry,
                              CLUSTER startcluster,
			      BOOL fixit);

static char tempbuf[255];
                              
/*================================= Checking ==============================*/

/*******************************************************************
**                        CheckDirectoryEntries
********************************************************************
** Checks all the directory entries in the volume on validity.
********************************************************************/

RETVAL CheckDirectoryEntries(RDWRHandle handle)
{
    BOOL invalid = FALSE, *pinvalid = &invalid;

    if (!FastWalkDirectoryTree(handle, EntryChecker, (void**) &pinvalid))
       return ERROR;

    return (invalid) ? FAILED : SUCCESS;
}

/*******************************************************************
**                          EntryChecker
********************************************************************
** For every directory entry the entry is checked for invalid data.
********************************************************************/

static BOOL EntryChecker(RDWRHandle handle, struct DirectoryPosition* pos,
			 struct DirectoryEntry* entry, void** structure)
{
    BOOL *invalid = *((BOOL**) structure);

    entry = entry;

    switch (LookAtEntry(handle, pos, FALSE))
    {
      case FALSE:
           *invalid = TRUE;
      case TRUE:
           return TRUE;
      case FAIL:
           return ERROR;
    }

    return FAIL;                         /* Should never be executed. */
}

/*================================= Fixing ==============================*/

/*******************************************************************
**                        CheckDirectoryEntries
********************************************************************
** Checks all the directory entries in the volume on validity. And if 
** they contain invalid data, fills them with default values.
********************************************************************/
RETVAL FixDirectoryEntries(RDWRHandle handle)
{
    return (FastWalkDirectoryTree(handle, EntryFixer, NULL)) ?
           SUCCESS : ERROR;
}

/*******************************************************************
**                          EntryFixer
********************************************************************
** For every directory entry the entry is checked for invalid data. 
** If the entry contains invalid data it is replaced with default data.
********************************************************************/

static BOOL EntryFixer(RDWRHandle handle, struct DirectoryPosition* pos,
		       struct DirectoryEntry* entry, void** structure)
{
    structure = structure, entry = entry;

    if (LookAtEntry(handle, pos, TRUE) == FAIL)
       return FAIL;

    return TRUE;
}

/*================================= Common ==============================*/

/*******************************************************************
**                        LookAtEntry
********************************************************************
** Checks a certain entry for validity. The following checks are performed:
**
** - All the '.' and '..' must be directories.
** - Check the file chain for free or bad clusters.
** - The first cluster of an LFN entry must be 0.
** - The characters of a filename must be valid.
** - The attribute must be valid.
** - The time and date must be valid.
** - A directory may not have a length.
** - If this is a directory then check the '.' entry in it.
** - The size of a file must be correctly noted in the entry.
** - The first cluster must be valid
**
** The values in the directory entries are adjusted regardless of the 
** value of fixit. However the entry is only written if fixit is true.
**
** Returns: FALSE - an invalid entry was found
**          TRUE  - an invalid entry was not found
**          FAIL  - media error
********************************************************************/

static BOOL LookAtEntry(RDWRHandle handle, struct DirectoryPosition* pos,
			BOOL fixit)
{
    int i;
    BOOL invalidname = FALSE;
    BOOL invalid = FALSE;
    struct DirectoryEntry entry, preventry, tempentry;
    struct LongFileNameEntry* lfnentry = (struct LongFileNameEntry*) &entry;
    CLUSTER firstcluster;
    unsigned long labelsinfat;   
    unsigned long calculatedclusters, numclusters, bytespercluster;  
    struct DirectoryPosition prevpos, *pprevpos = &prevpos;
    
    memcpy(&tempentry, &entry, sizeof(struct DirectoryEntry));
           
    labelsinfat = GetLabelsInFat(handle);            
    if (!labelsinfat) return FAIL;
    
    bytespercluster = GetSectorsPerCluster(handle) * BYTESPERSECTOR;
    if (!bytespercluster) return FAIL;    
    
    if (!GetDirectory(handle, pos, &entry))
       return FAIL;

    /* Do not check deleted entries */
    if (IsDeletedLabel(entry))
       return TRUE;
       
    if (IsLFNEntry(&entry)) /* Checks on LFN entries */
    {
       /* Get the position of the parent directory. */
       switch (TracePreviousDir(handle, pos, pprevpos))
       {
          case FALSE:
               pprevpos = 0;
               break;
            
          case TRUE:
               /* Get the directory entry of the parent directory */
               if (!GetDirectory(handle, pprevpos, &preventry))
                  return FAIL;
               break;
            
          case FAIL:
               return FAIL;
       }

       if (lfnentry->firstclust)
       { 
          ShowDirectoryViolation(handle, &prevpos, &preventry,  
                                 "%s contains an invalid long file name entry");
          
          lfnentry->firstclust = 0;
          invalid = TRUE;
       }
    }
    else /* Checks on SFN entries */
    {
        /* Initialise some values we will need */
        firstcluster = GetFirstCluster(&entry);
            
       /* All the '.' and '..' entries must be directories */
       if ((IsPreviousDir(entry) || IsCurrentDir(entry)) &&
	   ((entry.attribute & FA_DIREC) == 0))
       {
          /* Get the position of the parent directory. */
          switch (TracePreviousDir(handle, pos, pprevpos))
          {
            case FALSE:
                 pprevpos = 0;
                 break;
            
            case TRUE:
                 /* Get the directory entry of the parent directory */
                 if (!GetDirectory(handle, pprevpos, &preventry))
                    return FAIL;
                 break;
            
            case FAIL:
                 return FAIL;
          }

       
          if (IsCurrentDir(entry))
          { 
             ShowDirectoryViolation(handle, &prevpos, &preventry,  
                                    "%s contains an '.' entry that is not a directory");
             memcpy(entry.filename, "DOT     ", 8); 
          }
          else
          {
             ShowDirectoryViolation(handle, &prevpos, &preventry,  
                                    "%s contains an '..' entry that is not a directory");
             memcpy(entry.filename, "DOTDOT     ", 8);    
          }   
             
          invalid = TRUE;           
       }       
       
       /* Check the file chain for free or bad clusters. */
       if (!IsCurrentDir(entry) && !IsPreviousDir(entry) && firstcluster)
       {
	  switch (CheckTheFileChain(handle, pos, &entry, firstcluster, fixit))
          {  
             case FAIL:
                  return FAIL;
             case FALSE:
                  invalid = TRUE;
          }
       }
            
       /* Check the file name */
       if (entry.filename[0] == ' ')
       {
          invalidname = TRUE;
          entry.filename[0] = 'A';
       }

       /* file name */
       for (i = 0; i < 8; i++)
       {
           if ((strchr(INVALIDSFNCHARS, entry.filename[i]))                ||
               (((unsigned char)(entry.filename[i]) < 0x20) && (entry.filename[i] != 0x05)) ||
               (islower(entry.filename[i])))
               
           {
              entry.filename[i] = 'A';
              invalidname = TRUE;
           }
       }

       /* extension */
       for (i = 0; i < 3; i++)
       {
           if ((strchr(INVALIDSFNCHARS, entry.extension[i]))               ||
               (((unsigned char)(entry.extension[i]) < 0x20) &&
                                             (entry.extension[i] != 0x05)) ||
               (islower(entry.extension[i])))
               
           {
              entry.extension[i] = 'A';
              invalidname = TRUE;
           }
       }

       if (invalidname)
       {
          ShowDirectoryViolation(handle, pos, &tempentry,  
                                 "%s contains invalid char(s)");
          invalid = TRUE;
       }
       
       /* Check the attribute */
       if (entry.attribute & 0xC0)
       {
          ShowDirectoryViolation(handle, pos, &tempentry,  
                                 "%s has an invalid attribute");
          
          if (entry.attribute & FA_DIREC)
          {
             entry.attribute = FA_DIREC;
          }
          
          entry.attribute = 0;
          invalid = TRUE;
       }

       /*
          Check the time stamps:
             Notice that year is valid over it's entire range.
       */

       /* Creation time */
       if ((memcmp(&entry.timestamp, "\0\0", 2) != 0 &&  /* Optional field */
            memcmp(&entry.timestamp, "\xff\xff", 2) != 0) &&  /* Optional field */
           ((entry.timestamp.second > 29)  ||
            (entry.timestamp.minute > 59)  ||
            (entry.timestamp.hours  > 23)))
       {
           sprintf(tempbuf, "%%s has an invalid creation time (%02u:%02u:%02u)",
                   entry.timestamp.hours, 
                   entry.timestamp.minute, 
                   entry.timestamp.second*2);
                                               
          ShowDirectoryViolation(handle, pos, &tempentry, tempbuf);                                               
          memset(&entry.timestamp, 0, sizeof(struct PackedTime));
          
          invalid = TRUE;
       }

       /* Creation date */
       if ((memcmp(&entry.datestamp, "\0\0", 2) != 0 &&  /* Optional field */
            memcmp(&entry.datestamp, "\xff\xff", 2) != 0) &&  /* Optional field */
           (((entry.datestamp.day   < 1) || (entry.datestamp.day > 31))  ||
            ((entry.datestamp.month < 1) || (entry.datestamp.month > 12))))
       {
          sprintf(tempbuf, "%%s has an invalid creation date (%02u/%02u)",
                  entry.datestamp.month, entry.datestamp.day);
                                               
          ShowDirectoryViolation(handle, pos, &tempentry, tempbuf); 
          memset(&entry.datestamp, 0, sizeof(struct PackedDate));
          
          invalid = TRUE;
       }

       /* Last access date */
       if ((memcmp(&entry.LastAccessDate, "\0\0", 2) != 0) &&
           (((entry.LastAccessDate.day   < 1) ||
                                         (entry.LastAccessDate.day > 31))  ||
            ((entry.LastAccessDate.month < 1) ||
                                         (entry.LastAccessDate.month > 12))))
       {
          sprintf(tempbuf, "%%s has an invalid access date (%02u/%02u)",
                  entry.LastAccessDate.month, entry.LastAccessDate.day);
                                               
          ShowDirectoryViolation(handle, pos, &tempentry, tempbuf); 
          memset(&entry.LastAccessDate, 0, sizeof(struct PackedDate));
          
          invalid = TRUE;
       }
       
       /* Last write time (mandatory) */
       if ((entry.LastWriteTime.second > 29)  ||
           (entry.LastWriteTime.minute > 59)  ||
           (entry.LastWriteTime.hours  > 23))
       {
           sprintf(tempbuf, "%%s has an invalid last write time (%02u:%02u:%02u)",
                   entry.LastWriteTime.hours, 
                   entry.LastWriteTime.minute, 
                   entry.LastWriteTime.second*2);               
               
          ShowDirectoryViolation(handle, pos, &tempentry, tempbuf); 
          memset(&entry.LastWriteTime, 0, sizeof(struct PackedTime));
          
          invalid = TRUE;
       }

       /* Last write date (mandatory) */
       if (((entry.LastWriteDate.day   < 1) || (entry.LastWriteDate.day > 31))  ||
           ((entry.LastWriteDate.month < 1) || (entry.LastWriteDate.month > 12)))
       {
	  sprintf(tempbuf, "%%s has an invalid last write date (%02u/%02u)",
                  entry.LastWriteDate.month, entry.LastWriteDate.day);
          
          entry.LastWriteDate.day = 1;
          entry.LastWriteDate.month = 1;
          
          invalid = TRUE;
       }

       /* A directory may not have a length */
       if ((entry.attribute & FA_DIREC) && (entry.filesize != 0))
       {
          ShowDirectoryViolation(handle, pos, &tempentry,  
                                 "%s is a directory with an invalid length");
          entry.filesize = 0;
          invalid = TRUE;
       }   
       
       /* The first cluster must be valid */
       if (((firstcluster < 2)                    ||
	   (firstcluster >= labelsinfat)) &&
           (!((firstcluster == 0) &&
	        	  (IsPreviousDir(entry) || entry.filesize == 0))))
       {
          ShowDirectoryViolation(handle, pos, &tempentry,  
                                 "%s has an invalid first cluster");
          MarkEntryAsDeleted(entry);
          invalid = TRUE;
       }
       
       /* If this is a directory then check the '.' entry in it. */
       if (entry.attribute & FA_DIREC)
       {
          if (firstcluster)
          {
	     if (CheckDots(handle, firstcluster, fixit,
                           pos, &tempentry) == FAIL)
                return FAIL;
          }
       }
       
       /* The size of a file must be correctly noted in the entry. */
       if (firstcluster &&
           ((entry.attribute & FA_LABEL) == 0) && ((entry.attribute & FA_DIREC) == 0))
       {
          calculatedclusters = CalculateFileSize(handle, firstcluster);
          if (calculatedclusters == FAIL) return FALSE;
       
          /* Calculate the number of clusters in the file. */
          numclusters = (entry.filesize / bytespercluster) +
                                   ((entry.filesize % bytespercluster) > 0);

          if (numclusters != calculatedclusters)   
          {
             sprintf(tempbuf, 
                     "%%s has an invalid size, the size should be %lu, but the entry says it's %lu",
                     calculatedclusters * bytespercluster, numclusters * bytespercluster);     

             ShowDirectoryViolation(handle, pos, &tempentry, tempbuf);
                                 
             entry.filesize = calculatedclusters * bytespercluster;
             invalid = TRUE;
          }
       }
    }

    if (invalid && fixit)       /* If we found an error and we have to fix it */
    {
       if (!WriteDirectory(handle, pos, &entry)) /* Write the changes to disk */
          return FAIL;
    }

    return !invalid;
}

/*******************************************************************
**                        CheckDots
********************************************************************
** Takes the first cluster of a directory and sees wether the directory
** contains '.' as first entry and if it does checks wether it points
** to the given cluster.
**
** Returns: FALSE - an invalid entry was found
**          TRUE  - an invalid entry was not found
**          FAIL  - media error
********************************************************************/

static BOOL CheckDots(RDWRHandle handle, CLUSTER firstcluster, BOOL fixit,
                      struct DirectoryPosition* dirpos, 
                      struct DirectoryEntry* direntry)
{
   struct DirectoryPosition pos = {0, 0};
   struct DirectoryEntry entry;

   if (!GetNthDirectoryPosition(handle, firstcluster, 0, &pos))
      return FAIL;

   if ((pos.sector == 0) && (pos.offset == 0))
   {
      ShowDirectoryViolation(handle, dirpos, direntry, "is an empty directory");    
      return FALSE;
   }

   if (!GetDirectory(handle, &pos, &entry))
      return FAIL;

   if (!IsCurrentDir(entry))
   {
      ShowDirectoryViolation(handle, dirpos, direntry, 
                             "doesn't contain an '.' as first entry");      
      return FALSE;
   }

   if (GetFirstCluster(&entry) != firstcluster)
   { 
      ShowDirectoryViolation(handle, dirpos, direntry, "has a wrong '.' entry");  
                                      
      if (fixit)
      {
         SetFirstCluster(firstcluster, &entry);
         return (WriteDirectory(handle, &pos, &entry) ? FALSE : FAIL);
      }

      return FALSE;
   }

   return TRUE;
}

/*******************************************************************
**                        CalculateFileSize
********************************************************************
** Calculates the size of a file in clusters
********************************************************************/

static unsigned long CalculateFileSize(RDWRHandle handle,
                                       CLUSTER firstcluster)
{
   unsigned long result=0, *presult = &result;

   if (!FileTraverseFat(handle, firstcluster, FileSizeCalculator,
                        (void**) &presult))
      return FAIL;

   return result;
}

static BOOL FileSizeCalculator(RDWRHandle handle, CLUSTER label,
                               SECTOR datasector, void** structure)
{
   unsigned long* presult = *((unsigned long**) structure);

   structure = structure,
   label = label,
   handle = handle,
   datasector=datasector;

   (*presult)++;

   return TRUE;
}

/*******************************************************************
**                        CheckTheFileChain
********************************************************************
**  Checks a file for free or bad clusters
**
**  Returns:
**      FALSE if there was an invalid cluster
**      TRUE  if there was no invalid cluster
**
**      FAIL  if there was an error
**
**  Note: we can not use FileTraverseFat here because then we could
**        not get to the free or bad clusters.
********************************************************************/

static BOOL CheckTheFileChain(RDWRHandle handle,
			      struct DirectoryPosition* filepos,
                              struct DirectoryEntry* entry,
			      CLUSTER startcluster, BOOL fixit)
{
    BOOL invalidfound = FALSE;
    CLUSTER current = startcluster, previous;

    while (!FAT_LAST(current))
    {
        previous = current;
        
        if (!GetNthCluster(handle, current, &current))
           return FAIL;
                           
        if (FAT_BAD(current) || FAT_FREE(current))
        {
           ShowDirectoryViolation(handle, filepos, entry,
                                  "%s contains a free or a bad cluster");
                               
           invalidfound = TRUE;   
             
           if (fixit)
           {
              if (FAT_BAD(current))
              {
                 if (!WriteFatLabel(handle, previous, FAT_LAST_LABEL))
                    return FAIL;
              }
              else
              {
                 if (!WriteFatLabel(handle, current, FAT_LAST_LABEL))
                    return FAIL;
              }
           }
           break;
        }
    }
    
    return !invalidfound;
}
