/*    
   Chkdsk.c - check disk utility.

   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>
#include <dir.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "fte.h"

#include "chkdrvr.h"
#include "checks.h"
#include "version.h"
#include "misc\switchch.h"
#include "summary\summary.h"
#include "struct\FstTrMap.h"
#include "fragfact\defrfact.h"
#include "surface\surface.h"

static void Usage(char switchch);
static void OnExit(void);

int main(int argc, char *argv[])
{
    time_t t;
    RDWRHandle handle;
    int retval;
    int thesame;
    BOOL fixerrors = FALSE, checkerrors = TRUE, scansurface = FALSE;
    BOOL param2switch = FALSE;
    char switchch = SwitchChar();
    char curdrive[3];
    char* curdir, *abspath;

    /* Show copyright message */
    printf("ChkDsk " VERSION "\n"
	   "Copyright 2002, 2003 Imre Leber under the GNU GPL\n\n");

    /* First check the parameters. */
    if ((argc == 2) &&
        (((argv[1][0] == switchch) || (argv[1][0] == '/'))  &&
	 (argv[1][1] == '?') && (argv[1][2] == '\0')))
    {
       Usage(switchch);
       return 0;
    }

    if (argc > 4)
    {
       printf("Invalid parameters (type %c? for help)\n", switchch);
       return 2;
    }

    if ((argc == 3) || (argc == 2))
    {
       if (((argv[argc-1][0] == switchch) || (argv[argc-1][0] == '/')) &&
           (argv[argc-1][1] == 'f') && (argv[argc-1][2] == '\0'))
       {
          param2switch = (argc == 2);
          fixerrors = TRUE;
       }
       else if (((argv[argc-1][0] == switchch) || (argv[argc-1][0] == '/')) &&
	   (argv[argc-1][1] == 'r') && (argv[argc-1][2] == '\0'))
       {
          param2switch = (argc == 2);
          scansurface = TRUE;
          checkerrors = FALSE;
       }
       else if (((argv[argc-1][0] == switchch) || (argv[argc-1][0] == '/')) &&
	   (argv[argc-1][1] == 's') && (argv[argc-1][2] == '\0'))
       {
          param2switch = (argc == 2);
          checkerrors = FALSE;
       }
       else if (((argv[argc-2][0] == switchch) || (argv[argc-2][0] == '/')) &&
		 (argv[argc-2][1] == 'd'))
       {
	  param2switch = TRUE;
       }
       else if (argc == 3)
       {
          printf("Invalid parameters (type %c? for help)\n", switchch);
          return 2;
       }
    }

    atexit(OnExit);

    InitSectorCache();
    StartSectorCache();

    if ((argc == 1) || param2switch)
    {
       curdrive[0] = getdisk() + 'A';
       curdrive[1] = ':';
       curdrive[2] = 0;

       if (!InitReadWriteSectors(curdrive, &handle))
       {
          printf("Cannot access %s\n", argv[1]);
          return 2;
       }
    }
    else
    {
       if (!InitReadWriteSectors(argv[1], &handle))
       {
          printf("Cannot access %s\n", argv[1]);
          return 2;
       }

       if (!IsWorkingOnImageFile(handle))
       {
	  curdrive[0] = argv[1][0];
       }
       else
       {
	  curdrive[0] = '?';
       }
    }

    if ((argc == 4) || (argc == 3))
    {
       int imagefile, canproceed=TRUE;
       
       if (((argv[argc-2][0] == switchch) || (argv[argc-2][0] == '/')) &&
	   (argv[argc-2][1] == 'd') && (argv[argc-2][2] == '\0'))
       {
	  abspath = (char*) malloc(MAXPATH);
	  curdir  = (char*) malloc(MAXPATH);

          imagefile = IsWorkingOnImageFile(handle);

	  if (!imagefile && (argc == 3) && (!getcwd(curdir, MAXPATH)))
	  {
	     printf("Cannot get current working dir\n");

	     free(abspath);
	     free(curdir);
	  }
	  else
	  { 
             if (imagefile || (argc == 4) ||
                 ((argv[argc-1][0] != 0) && (argv[argc-1][1] == ':')))
             {
		if ((argv[argc-1][0] != 0) && (argv[argc-1][1] == ':'))
		{
		   if (imagefile ||
                       (toupper(argv[argc-1][0]) != toupper(curdrive[0])))
		   {
		      printf("Invalid drive specification\n");
		      canproceed = FALSE;
		   }
		   else
		   {
		      strcpy(abspath, argv[argc-1] + 2);
		   }
		}
		else
		   strcpy(abspath, argv[argc-1]);
             }
	     else
             {
                if ((curdir[0] != 0) && (curdir[1] == ':'))
                   curdir+=2;               /* Take off the drive spec */

                canproceed = MakeAbsolutePath(curdir, argv[argc-1], abspath);
		if (!canproceed)
		{
		   printf("Incorrect relative path specification\n");
		}
             }

             if (canproceed)
	     {
                printf("%s\n\n", abspath);
                
                free(curdir);
                if (!PrintFileDefragFactors(handle, abspath))
                {
                   printf("Problem printing out defragmentation factors.\n");
	        }
                free(abspath);
             }
             else
             {
		free(curdir);
                free(abspath);
             }
	  }

          CloseReadWriteSectors(&handle);
	  return 0;
       }
       else if (argc == 4)
       {
	  printf("Invalid parameters (type %c? for help)\n", switchch);
	  CloseReadWriteSectors(&handle);
	  return 0;
       }
    }

    /* Check the BOOT */
    if (!DescriptorCheck(handle))
    {
       printf("Invalid BOOT sector\n");
       DestroyFastTreeMap();
       CloseReadWriteSectors(&handle);
       return 2;
    }

    /* See wether there are any differences in the FATs or the BOOTs
       if there are we are not starting the checking in this version. */
    thesame = MultipleFatCheck(handle);
    if ((thesame == TRUE) && (GetFatLabelSize(handle) == FAT32))
    {
       thesame = MultipleBootCheck(handle);
       if (thesame == FALSE)
       {
          printf("BOOTs are different\n");
       }
    }
    else if (thesame == FALSE)
    {
       printf("FATs are different\n");
    }
    else if (thesame == FAIL)
    {
       printf("Problem reading FAT(s)\n");
    }

    if (thesame == TRUE)
    {
       time(&t);

       /* Scan the data area (if required) */
       if (scansurface)
       {
          printf("Scanning, please wait . . .");
          if (!ScanSurface(handle))
             return FALSE;
          printf("\n");
       }

       /* Now try to check for (and fix) all the defects. */
       if (fixerrors)
       {
          switch (FixVolume(handle, TheChecks, AMOFCHECKS))
          {
              case TRUE:
                   retval = 0;
                   break;
      	      case FAIL:
	           printf("Error accessing the volume\n");
                   DestroyFastTreeMap();
                   CloseReadWriteSectors(&handle);
		   return 2;
          }
       }
       else if (checkerrors)
       {
          switch (CheckVolume(handle, TheChecks, AMOFCHECKS))
          {
              case TRUE:
                   retval = 0;
                   break;
              case FALSE:
 		   printf("\nErrors were found. You did not use the /F switch. "
		          "Errors are not corrected\n\n");
	           retval = 1;
                   break;
	      case FAIL:
	           printf("Error accessing the volume\n");
                   DestroyFastTreeMap();
                   CloseReadWriteSectors(&handle);
                   return 2;
          }
       }

       printf("Elapsed time: %lds\n", time(NULL)-t);
    }
    
    /* Print out the volume summary. */
    if (!ReportVolumeSummary(handle))
    {
       retval = 2;
    }

    if (thesame == FAIL) retval = 2;

    CloseReadWriteSectors(&handle);

    return retval;
}

static void Usage(char switchch)
{
    printf("Checks a volume and returns a status report\n"
           "Usage:\n"
           "\tChkdsk [<volume>] [%cf] [%cd <files>] [%cr] [%cs]\n"
           "\n"
           "%cf: attempt to fix any errors found.\n"
           "%cd: print out the indicated files and show the fragmentation factor.\n"
           "%cr: scan the data area and try to recover unreadable data.\n"
           "%cs: only show drive summary.\n"
           "\n"
           "Note: if volume is ommited, the current drive is assumed.\n",
           switchch,
           switchch,
           switchch,
           switchch,
           switchch,
           switchch,
           switchch,
           switchch);
}

static void OnExit(void)
{
    CloseSectorCache();
}
