/***
*dirent.c - disk directory maintenence
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   Handles the displaying and editing of directory sectors.
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
*******************************************************************************/

/*
   Versions:

   1.5   08-Aug-1998    fixed the `dosdate' and `dostime' edit fields
   1.4   04-Jan-1998    struct typedefs as defined in DIRENT.H; LFN support
                        (display only); ESCAPE in editing
   1.3   13-Nov-1993    more use of structures, cleaned up changedir()
   1.2   14-Jan-1994    =0 bug fix in changedir()
   1.1   13-Nov-1993    Started structures
   1.0   June 1993

   Release Notes:

   This is all "Brute Force" code.  Written in a couple a days after
   I wanted to arange all my directories is a certain non-sorted way
   easier than a touch program. It's ugly. But it works.

   Programming Notes:

   Keyboard input is done by scankey() which returns scancodes.

   LFN support is lame: it displays the LFN entries only and will
   not let you edit them. Poeple have said that Microsft's LFN
   kludge is "neat trick" (or some such thing). It remains,
   however, just a kludge that does not adhere to UNICODE standards
   as far as I can see.

*/

#include <stdio.h>
#include <conio.h>
#include <ctype.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include "mylib.h"               /* getnum() */
#include "keys.h"
#include "dirent.h"              /* DosTime, DosDate, dDIR */
#include "console.h"             /* input, output, cursor stuff */

/* NO globals referenced here */

/* NO globals defined here */

#define isexten(c) ( (c>0x7f) && (c<255) )

static int pdirent(DOSDIR *);
static int plfn(DOSLFN *lfn);
static void pattr(ATTRIB attr);
static void pchar(int c);
static char *ptime(DOSTIME time, char *b);
static char *pdate(DOSDATE date, char *b);
static int command(int c);
static int text(int len, unsigned char *b);
static int attrib(unsigned char *b);
static int dostime(DOSTIME time, DOSTIME *t);
static int dosdate(DOSDATE date, DOSDATE *d);

/*...filename.txt..attr. ...hh:mm:ss...mm/dd/yy...1234567..123456789 */
static char
*dir_header="   File          Attr     Date       Time       Cluster       Size\n";

/* dumpdir  -  display directory */

extern void dumpdir(unsigned char *buf, int sec_size)
{
int i;
DOSDIR *dir = (DOSDIR *)buf;

   output('\n');
   output('\n');
   print(dir_header);                        /* print header */

   for (i = 0; i < sec_size; i+=sizeof(DOSDIR),dir++)    /* and each entry */
   {
      print("\n   ");
      pdirent(dir);
   }
   output('\n');
}

/* change dir command structure */

/* ...filename.txt...ttri...mm/dd/yy...hh:mm:ss...1234567...1234567 */
struct fields {
   int beg;             /* starting column */
   int end;             /* ending column */
   int (*inp)();        /* function to do the change */
} field[] = {
   {4,7,  command},
   {9,16, text},
   {18,20,text},
   {23,28,attrib},
   {32,33,dosdate},
   {43,44,dostime},
};

#define NUMFIELDS    (sizeof(field)/sizeof(struct fields)-1)
#define LASTENTRY    ((int)(sec_size/sizeof(DOSDIR))-1)

/***
*changedir  -  edit directory entry(s) in sector buffer
*
****/

extern int changedir(unsigned char *buf,int sec_size)
{
KEY c;
DOSDIR *e;
unsigned int a,x;
int i;            /* index to entry */
unsigned int n;   /* for numerical input */
int r;            /* readonly flag */
int p;            /* absolute position */
int f;            /* field position */

   get_cursor(&a,&x);
   print("\n\n      %s",dir_header);

   i = r = 0;
   for (;;)
   {
      c = f = 0;                     /* command function, no key */
      p = field[f].beg;
      print("\n%03d      ",i+1);

      e = (DOSDIR *)(buf + (i * sizeof(DOSDIR)));
      r = pdirent(e);
get:
      set_cursor(a,p);

      switch (f)
      {
         case 0:                                /* command */
            c = (*field[f].inp)(r);
            break;
         case 1:                                /* file name */
            c = (*field[f].inp)(8,e->name);
            break;
         case 2:                                /* file extension */
            c = (*field[f].inp)(3,e->ext);
            break;
         case 3:                                /* attribute */
            c = (*field[f].inp)(&e->attr);
            break;
         case 4:                                /* dosdate */
            c = (*field[f].inp)(e->date,&e->date);
            break;
         case 5:                                /* dostime */
            c = (*field[f].inp)(e->time,&e->time);
            break;
      }

      /* deal with key returned from functions */

      if (c == ESCAPE)           /* abort changes */
      {
         if (f == 0)             /* first (command) field */
            c = '.';             /*  "fall through" to exit */
         else
            continue;
      }

      switch (c)                 /* convert scancodes to ASCII */
      {
         case RIGHT:             /* next field */
         case CRIGHT:
         case TABKEY:
            c = '\t';
            break;
         case SHTAB:             /* previous field */
         case CLEFT:
         case LEFT:
            c = 0x7f;
            break;
         case DOWN:              /* next entry */
            c = '\r';
            break;
         case UP:                /* previous entry */
            c = '\b';
            break;
         default:
            c &= 0xff;           /* convert to ASCII */
            break;
      }

      /* ASCII */

      switch (c)
      {
         case 0x7f:                                /* previous field */
            if (f > 0)
               p = field[--f].beg;
            break;
         case '\t':                                /* next field */
            if (f < NUMFIELDS)
               p = field[++f].beg;
            break;
         case '\r':
         case ' ':                                 /* next entry */
            if (i == LASTENTRY)                    /* if at end */
            {                                      /*  wrap */  
               i = 0;
               print("\n\n      %s",dir_header);   /* redisplay header */
            }
            else
               i++;
            continue;                              /* redisplay entry */
            break;
         case '\b':                                /* previous entry */
            if (i == 0)                            /* if at beginning */
            {                                      /*  wrap */
               i = LASTENTRY;
               print("\n\n      %s",dir_header);   /* redisplay header */
            }
            else
               i--;
            continue;
            break;
         case '=':                                 /* select entry */
            output(c);
            if (getnum(sec_size/sizeof(DOSDIR),&n,10) > 0 && n > 0)
               i = n-1;
            continue;
            break;
         case '/':
         case '?':
            print("\n\n\tCR/SP  next file");
            print("\n\tBS     previous file");
            print("\n\tTAB    next field");
            print("\n\tSHTAB  previous field");
            print("\n\t=n     goto file n");
            print("\n\t'C'    goto file first letter");
            print("\n\t.|ESC  exit\n");
            continue;
            break;
         case '.':
         case 'q':
            output('\n');
            return 1;
            break;
         case 0:
            break;
         default: 
            if (isalnum(c))                        /* letter? */
            {
               int j;
               c = toupper(c);
               for (j = i+1; ;j++)
               {
                  if (j > LASTENTRY)
                     j = 0;
                  if (*(buf+(j*sizeof(DOSDIR))) == c)
                     break;
                  if (j == i)
                     break;
               }
               if (j != i)
               {
                  i = j;
                  continue;
               }
            }
            break;
      }
      goto get;
   }
}

/***
*pdirent() - print directory entry
*
*  Returns: 0 unused, 1 normal, -1 for directory or erased
*
****/

static int pdirent(DOSDIR *dir)
{
int i,r;
char b[12];

   r = 1;
   if (dir->name[0] == 0)
   {
      print(" unused");
      r = 0;
   }
   else
   {
      /* set can't edit return flag */

      if (dir->name[0] == 0xE5 || dir->name[0] == '.')
         r = -1;

      if (dir->attr.rdonly && dir->attr.hidden &&
                              dir->attr.system && dir->attr.volume)
      {
         plfn((DOSLFN*)dir);
         return -1;
      }

      for (i = 0; i < 8; i++)
         pchar(dir->name[i]);

      output('.');

      for (i = 0; i < 3; i++)
         pchar(dir->ext[i]);

      print("  ");
      pattr(dir->attr);

      print("   %s",pdate(dir->date,b));
      print("   %s",ptime(dir->time,b));
      print("   % 7u",dir->start);
      if (!dir->attr.subdir)
         print("  % 9lu",dir->size);
   }
   return r;
}

/* plfn  -  print long file name entry (very limited) */

static int plfn(DOSLFN *lfn)
{
int i;
char *s;

   s = (char *)lfn->name1;
   for (i = 0; i < LNAME1*2; i+=2)
      pchar(s[i]);
   s = (char *)lfn->name2;
   for (i = 0; i < LNAME2*2; i+=2)
      pchar(s[i]);
   s = (char *)lfn->name3;
   for (i = 0; i < LNAME3*2; i+=2)
      pchar(s[i]);
   print(" ");

   pattr(lfn->attr);

   print("   ");
   for (i = (1<<7); i > 0; i >>= 1)
      output( lfn->ord & i ? '1' : '0');

   print("   %02x %02x",lfn->ord,lfn->cksum);

   return 0;
}


/* check for bad chars in case and non-dir sector is displayed */

static void pchar(int c)
{
   if ((isspace(c) && c != ' ' ) || !c || c==7)
      output(250);
   else if (c == 0xFF)
      output(249);
   else
      output(c);
}

static void pattr(ATTRIB attr)
{
   output(attr.rdonly ? 'R' : '_');
   output(attr.hidden ? 'H' : '_');
   output(attr.system ? 'S' : '_');
   output(attr.volume ? 'V' : '_');
   output(attr.subdir ? 'D' : '_');
   output(attr.archiv ? 'A' : '_');
}

static char *ptime(DOSTIME time, char *b)
{
   sprintf(b,"%02d:%02d:%02d",time.hours,time.minutes,time.seconds);
   return b;
}

static char *pdate(DOSDATE date, char *b)
{
   sprintf(b,"%02d/%02d/%02d",date.month,date.day,date.year+80);
   return b;
}

/* editing functions */


static int command(int c)
{
int i;

   i = scankey();
   if (c < 1 && (i==RIGHT || i==TABKEY))
      i = 0;
   return i;
}

/***
*text() -- edit a text entry (filename/ext)
*
*  Can enter ALL printable characters including illegal characters!
*
****/

static int text(int len, unsigned char *buf)
{
int c;
int i;
char bak[FILELEN];

   memcpy(bak,buf,len);

   for (i=0;;)
   {
      c=scankey();
      if (c==RIGHT)
      {
         if (i==len-1)
            break;
         curright();
         ++i;
         continue;
      }
      if (c==LEFT || c==BACKSP)
      {
         if (!i)
            break;
         --i;
         curleft();
         continue;
      }
      if (c==RETURN || c==TABKEY || c==SHTAB || c==DOWN || c==UP)
         break;
      if (c==ESCAPE)
      {
         memcpy(buf,bak,len);
         break;
      }
      if (c>0x352f) continue;
      c&=0xff;
      if (isgraph(c) || isexten(c))
      {
         *(buf+i)=(char)c;
         i++;
         output(c);
         if (i==len)
         {
            c=RIGHT;
            break;
         }
      }
   }
   return(c);
}

/***
*attrib() - toggle filename attributes, except for dir
*
****/

static int attrib(unsigned char *buf)
{
KEY i;
int bits,c,p,d;
unsigned char bak;
enum { R, H, S, V, D, A };

   /* 012345 */
   /* RHSVDA */

   bak = *buf;
   bits=d=p=0;                   /* no need to set d,bits but it gets rid of */
                              /*  MSC warning C4701 */
   for (;;)
   {
      i=scankey();
      if (i==RETURN || i==TABKEY || i==SHTAB || i==DOWN || i==UP)
         break;
      if (i==ESCAPE)
      {
         *buf = bak;
         break;
      }
      if (i==RIGHT)
      {
         if (p==A)      /* at end, exit */
            break;
         curright();
         ++p;
         continue;
      }
      if (i==LEFT || i==BACKSP)
      {
         if (p==R)
            break;
         --p;
         curleft();
         continue;
      }

      if ( p!=V && p!=D && (i&0xff)==' ')
      {
         if (p==R)
         {
            bits=1;
            d='R';
         }
         else if (p==H)
         {
            bits=2;
            d='H';
         }
         else if (p==S)
         {
            bits=4;
            d='S';
         }
         else if (p==A)
         {
            bits=0x20;
            d='A';
         }
         if (*buf&bits)
         {
            c='_';
            *buf&=~bits;
         }
         else
         {
            c=d;
            *buf|=bits;
         }
         output(c);
         curleft();
      }
   }
   return(i);
}

static int dostime(DOSTIME time, DOSTIME *t)
{
int i;
int h,m,s;
unsigned int r,x;
int c,p;
DOSTIME tak;

   /* 01234567 */
   /* hh:mm:ss */

   tak = *t;

   get_cursor(&r,&x);
   c = x;                     /* set starting column */
   p = c;                     /* set the working column pointer */

   h = time.hours;            /* set the working values */
   m = time.minutes;
   s = time.seconds;

   for (;;)
   {
      set_cursor(r,p);

      i=scankey();

      if (i==TABKEY || i==RETURN || i==SHTAB || i==DOWN || i==UP)
         return(i);

      if (i==ESCAPE)
      {
         *t = tak;
         break;
      }

      if (i==LEFT || i==BACKSP)
      {
         if (p==c && i==LEFT)
            return(i);
         if (p==c && i==BACKSP)
            continue;
         if (p==c+3 || p==c+6)
            p-=2;
         else
            p--;
         continue;
      }
right:
      if (i==RIGHT)
      {
         if (p==c+7)
            return(i);
         if (p==c+1 || p==c+4)
            p+=2;
         else
            p++;
         continue;
      }
      i&=0xff;
      if (isdigit(i))
      {
         output(i);
         i=i-'0';                   /* convert ascii to bin */

         /* based on column, */
         /*  calculate new time with the */
         /*  number input */

         if (p==c)
            h=(i*10)+(h%10);
         else if (p==c+1)
            h=i+((h/10)*10);
         else if (p==c+3)
            m=(i*10)+(m%10);
         else if (p==c+4)
            m=i+((m/10)*10);
         else if (p==c+6)
            s=(i*10)+(s%10);
         else if (p==c+7)
            s=i+((s/10)*10);

         time.hours = h;
         time.minutes = m;
         time.seconds = s;
         *t = time;

         i = RIGHT;
         goto right;
      }
   }
   return i;
}

/***
*dosdate() - edit date field
*
*  Not Y2K compliant.
****/

static int dosdate(DOSDATE date, DOSDATE *dt)
{
int i;
int m,d,y;
unsigned int r,x;
int c,p;
DOSDATE dak;


   /* 01234567 */
   /* mm/dd/yy */

   dak = *dt;

   get_cursor(&r,&x);
   c = x;
   p = c;
   m = date.month;
   d = date.day;
   y = date.year+80;

   for (;;)
   {
      set_cursor(r,p);
      i=scankey();

      if (i==TABKEY || i==RETURN || i==SHTAB || i==DOWN || i==UP)
         break;

      if (i==ESCAPE)
      {
         *dt = dak;
         break;
      }
      if (i==LEFT || i==BACKSP)
      {
         if (p==c && i==LEFT)
            return(i);
         if (p==c && i==BACKSP)
            continue;
         if (p==c+3 || p==c+6)
            p-=2;
         else
            p--;
         continue;
      }
right:
      if (i==RIGHT)
      {
         if (p==c+7)
            continue;
         if (p==c+1 || p==c+4)
            p+=2;
         else
            p++;
         continue;
      }
      i&=0xff;
      if (isdigit(i))
      {
         output(i);
         i=i-'0';

         if (p==c)
            m=(i*10)+(m%10);
         else if (p==c+1)
            m=i+((m/10)*10);
         else if (p==c+3)
            d=(i*10)+(d%10);
         else if (p==c+4)
            d=i+((d/10)*10);
         else if (p==c+6)
            y=(i*10)+(y%10);
         else if (p==c+7)
            y=i+((y/10)*10);
         if (y>80)
            y-=80;
         else
            y=0;

         date.month = m;
         date.day = d;
         date.year = y;
         y += 80;
         *dt = date;

         i=RIGHT;
         goto right;
      }
   }
   return i;
}
