/*
   Fileloop.c - file loops checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

static BOOL LoopChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                        struct DirectoryEntry* entry, void** structure);
static BOOL LoopTruncator(RDWRHandle handle, struct DirectoryPosition* pos,
                          struct DirectoryEntry* entry, void** structure);
static BOOL ScanForLoops(RDWRHandle handle, CLUSTER firstcluster,
                         CLUSTER* loopingcluster);

static BOOL IsLabelValid(RDWRHandle handle, CLUSTER label);

/* CHECKING */

RETVAL CheckFilesForLoops(RDWRHandle handle)
{
    BOOL invalid = FALSE, *pinvalid = &invalid;

    if (!FastWalkDirectoryTree(handle, LoopChecker, (void**) &pinvalid))
       return ERROR;

    return (invalid) ? FAILED : SUCCESS;
}

static BOOL LoopChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                        struct DirectoryEntry* entry, void** structure)
{
    CLUSTER firstcluster, dummy;
    BOOL *invalid = *((BOOL**) structure);

    pos = pos;

    firstcluster = GetFirstCluster(entry);
    if (firstcluster)
    {
       switch (ScanForLoops(handle, firstcluster, &dummy))
       {
          case FALSE:
               printf("A file was found with a loop\n");
               *invalid = TRUE;
               break;
          case TRUE:
               break;
          case FAIL:
               return FAIL;
       }
    }
    
    return TRUE;
}

/* FIXING */
RETVAL TruncateLoopingFiles(RDWRHandle handle)
{
    return (FastWalkDirectoryTree(handle, LoopTruncator, NULL)) ?
           SUCCESS : ERROR;
}

static BOOL LoopTruncator(RDWRHandle handle, struct DirectoryPosition* pos,
                          struct DirectoryEntry* entry, void** structure)
{
    CLUSTER firstcluster, offending;

    structure = structure, pos = pos;

    firstcluster = GetFirstCluster(entry);
    if (firstcluster)
    {
       switch (ScanForLoops(handle, firstcluster, &offending))
       {
          case FALSE:
               printf("A file was found with a loop\n");
               if (!WriteFatLabel(handle, offending, FAT_LAST_LABEL))
                  return FAIL;
               break;
          case TRUE:
               break;
          case FAIL:
               return FAIL;
       }
    }
    
    return TRUE;
}

/* COMMON */

/*
   Returns:
       TRUE if there is a loop
            then returns the second reference to the duplicated cluster
            in loopingcluster
       
       FALSE if there is no loop

       FAIL if there was an error accessing the volume

*/

static BOOL ScanForLoops(RDWRHandle handle, CLUSTER firstcluster,
                         CLUSTER* loopingcluster)
{
    char* bitfield;
    unsigned long ClustersInDataArea;
    CLUSTER current, label;

    ClustersInDataArea = GetClustersInDataArea(handle);
    if (!ClustersInDataArea) return FAIL;

    /* Check the given first cluster on validity */
    if (!FAT_NORMAL(firstcluster))
       return TRUE;

    bitfield = CreateBitField(ClustersInDataArea);
    if (!bitfield) return FAIL;

    SetBitfieldBit(bitfield, firstcluster);

    /* Take the first cluster of the file */
    if (!GetNthCluster(handle, firstcluster, &label))
    {
       DestroyBitfield(bitfield);
       return FAIL;
    }
       
    if (!FAT_NORMAL(label)) /* One cluster cannot contain a loop */
    {
       DestroyBitfield(bitfield);
       return TRUE;
    }

    if (!IsLabelValid(handle, label))
    {
       *loopingcluster = firstcluster;
       DestroyBitfield(bitfield);
       return FALSE;
    }

    if (GetBitfieldBit(bitfield, label)) /* First cluster points to itself */
    {
       *loopingcluster = firstcluster;
       DestroyBitfield(bitfield);
       return FALSE;
    }
    SetBitfieldBit(bitfield, label);

    while (FAT_NORMAL(label))
    {
       current = label;

       if (!GetNthCluster(handle, current, &label))
       {
          DestroyBitfield(bitfield);
          return FAIL;
       }

       if (!IsLabelValid(handle, label))
       {
          *loopingcluster = current;
          DestroyBitfield(bitfield);
          return FALSE;
       }

       if (FAT_NORMAL(label))
       {
          if (GetBitfieldBit(bitfield, label))
          {
             *loopingcluster = current;
             DestroyBitfield(bitfield);
             return FALSE;
          }
          SetBitfieldBit(bitfield, label);
       }
    }

    DestroyBitfield(bitfield);
    return TRUE;
}

static BOOL IsLabelValid(RDWRHandle handle, CLUSTER label)
{
    unsigned long labelsinfat;

    if (!FAT_NORMAL(label))
       return TRUE;

    /* To see wether the clusters that are returned are valid. */
    labelsinfat = GetLabelsInFat(handle);
    if (!labelsinfat) return FAIL;

    if (label >= labelsinfat) return FALSE;
    if (label < 2) return FALSE;

    return TRUE;
}


