/*
   Filelen.c - file length checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

static BOOL FileSizeChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                            struct DirectoryEntry* entry, void** structure);
static BOOL FileSizeAdjuster(RDWRHandle handle, struct DirectoryPosition* pos,
                             struct DirectoryEntry* entry, void** structure);
static BOOL LookForInvalidSizes(RDWRHandle handle,
                                struct DirectoryPosition* pos,
                                BOOL* invalid, BOOL fixit);
static unsigned long CalculateFileSize(RDWRHandle handle,
                                       CLUSTER firstcluster);
static BOOL FileSizeCalculator(RDWRHandle handle, CLUSTER label,
                               SECTOR datasector, void** structure);

/* Checking */

RETVAL CheckFileLengths(RDWRHandle handle)
{
   BOOL invalid = FALSE, *pinvalid = &invalid;

   if (!FastWalkDirectoryTree(handle, FileSizeChecker, (void**) &pinvalid))
      return ERROR;

   return (invalid) ? FAILED : SUCCESS;
}

static BOOL FileSizeChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                            struct DirectoryEntry* entry, void** structure)
{
   pos = pos, entry = entry;
   return LookForInvalidSizes(handle, pos, *((BOOL**) structure), FALSE);
}

/* Fixing */

RETVAL AdjustFileLengths(RDWRHandle handle)
{
   return (FastWalkDirectoryTree(handle, FileSizeAdjuster, NULL)) ?
          SUCCESS : ERROR;
}

static BOOL FileSizeAdjuster(RDWRHandle handle, struct DirectoryPosition* pos,
			     struct DirectoryEntry* entry, void** structure)
{
   structure = structure, entry = entry;

   return LookForInvalidSizes(handle, pos, NULL, TRUE);
}

/* Common */

static BOOL LookForInvalidSizes(RDWRHandle handle,
                                struct DirectoryPosition* pos,
                                BOOL* invalid, BOOL fixit)
{
   unsigned long calculatedclusters, numclusters, bytespercluster;

   struct DirectoryEntry entry;

   CLUSTER firstcluster;

   if (!GetDirectory(handle, pos, &entry))
      return FAIL;

   if (entry.attribute & FA_LABEL)
      return TRUE;

   if (IsLFNEntry(&entry))
      return TRUE;

   if (entry.attribute & FA_DIREC)
      return TRUE;

   if (IsDeletedLabel(entry))
      return TRUE;

   firstcluster = GetFirstCluster(&entry);
   if (!firstcluster) return TRUE;

   calculatedclusters = CalculateFileSize(handle, firstcluster);
   if (calculatedclusters == FAIL) return FALSE;

   /* Calculate the number of bytes per cluster */
   bytespercluster = GetSectorsPerCluster(handle) * BYTESPERSECTOR;
   if (!bytespercluster) return FAIL;

   /* Calculate the number of clusters in the file. */
   numclusters = (entry.filesize / bytespercluster) +
                       ((entry.filesize % bytespercluster) > 0);

   if (numclusters != calculatedclusters)
   {
      printf("Found a file with an invalid size\n");
      
      if (fixit)
      {
         if (entry.filesize == 0)
         {
            MarkEntryAsDeleted(entry);
         }
         else
         {
            entry.filesize = numclusters * bytespercluster;
         }
         
         if (!WriteDirectory(handle, pos, &entry))
            return FAIL;
      }
      else
      {    
         *invalid = TRUE;
      }
   }

   return TRUE;
}

static unsigned long CalculateFileSize(RDWRHandle handle,
                                       CLUSTER firstcluster)
{
   unsigned long result=0, *presult = &result;

   if (!FileTraverseFat(handle, firstcluster, FileSizeCalculator,
                        (void**) &presult))
      return FAIL;

   return result;
}

static BOOL FileSizeCalculator(RDWRHandle handle, CLUSTER label,
                               SECTOR datasector, void** structure)
{
   unsigned long* presult = *((unsigned long**) structure);

   structure = structure,
   label = label,
   handle = handle,
   datasector=datasector;

   (*presult)++;

   return TRUE;
}
