#include <stdio.h>
#include <dir.h>

#include "fte.h"

#include "chkdrvr.h"
#include "checks.h"
#include "version.h"
#include "misc\switchch.h"
#include "summary\summary.h"
#include "struct\FstTrMap.h"

void Usage(char switchch);

int main(int argc, char *argv[])
{
    time_t t;
    RDWRHandle handle;

    int retval;
    int thesame;
    BOOL fixerrors = FALSE, checkerrors = TRUE, scansurface = FALSE;
    char switchch = SwitchChar();

    char curdrive[3];

    /* Show copyright message */
    printf("ChkDsk " VERSION "\n"
	   "Copyright 2002, Imre Leber under the GNU GPL\n\n");

    /* First check the parameters. */
    if ((argc == 2) &&
        (((argv[1][0] == switchch) || (argv[1][0] == '/'))  &&
	 (argv[1][1] == '?') && (argv[1][2] == '\0')))
    {
       Usage(switchch);
       return 0;
    }

    if (argc > 4)
    {
       printf("Invalid parameters (type %c? for help)\n", switchch);
    }

    if (argc == 3)
    {
       if (((argv[2][0] == switchch) || (argv[2][0] == '/')) &&
           (argv[2][1] == 'f') && (argv[2][2] == '\0'))
       {
          fixerrors = TRUE;
       }
       else if (((argv[2][0] == switchch) || (argv[2][0] == '/')) &&
	   (argv[2][1] == 'r') && (argv[2][2] == '\0'))
       {
          scansurface = TRUE;
          checkerrors = FALSE;
       }
       else if (((argv[2][0] == switchch) || (argv[2][0] == '/')) &&
	   (argv[2][1] == 's') && (argv[2][2] == '\0'))
       {
          checkerrors = FALSE;
       }
       else
       {
	  printf("Invalid parameter(s) (type %c? for help)\n", switchch);
          return 2;
       }
    }

    if (argc == 1)
    {
       curdrive[0] = getdisk() + 'A';
       curdrive[1] = ':';
       curdrive[2] = 0;

       if (!InitReadWriteSectors(curdrive, &handle))
       {
          printf("Cannot access %s\n", argv[1]);
          return 2;
       }
    }
    else
    {
       if (!InitReadWriteSectors(argv[1], &handle))
       {
          printf("Cannot access %s\n", argv[1]);
          return 2;
       }
    }

    if (argc == 4)
    {
       if (((argv[2][0] == switchch) || (argv[2][0] == '/')) &&
           (argv[2][1] == 'd') && (argv[2][2] == '\0'))
       {
          if (!PrintFileDefragFactors(handle, argv[3]))
          {
             printf("Problem printing out defragmentation factors.\n");
          }
       }
       else
       {
          printf("Invalid parameters (type %c? for help)\n", switchch);
       }

       CloseReadWriteSectors(&handle);
       return 0;
    }

    /* Check the BOOT */
    if (!DescriptorCheck(handle))
    {
       printf("Invalid BOOT sector\n");
       DestroyFastTreeMap();
       CloseReadWriteSectors(&handle);
       return 2;
    }

    /* See wether there are any differences in the FATs or the BOOTs
       if there are we are not starting the checking in this version. */
    thesame = MultipleFatCheck(handle);
    if ((thesame == TRUE) && (GetFatLabelSize(handle) == FAT32))
    {
       thesame = MultipleBootCheck(handle);
       if (thesame == FALSE)
       {
          printf("BOOTs are different\n");
       }
    }
    else if (thesame == FALSE)
    {
       printf("FATs are different\n");
    }

    /* Create the fast tree map */
    if (fixerrors || checkerrors) /* don't create it if it is not required */
       CreateFastTreeMap(handle);

    if (thesame == TRUE)
    {
       time(&t);

       /* Scan the data area (if required) */
       if (scansurface)
       {
          printf("Scanning, please wait . . .");
          if (!ScanSurface(handle))
             return FALSE;
          printf("\n");
       }

       /* Now try to check for (and fix) all the defects. */
       if (fixerrors)
       {
          switch (FixVolume(handle, TheChecks, AMOFCHECKS))
          {
              case TRUE:
                   retval = 0;
                   break;
      	      case FAIL:
	           printf("Error accessing the volume\n");
                   DestroyFastTreeMap();
                   CloseReadWriteSectors(&handle);
                   return 2;
          }
       }
       else if (checkerrors)
       {
          switch (CheckVolume(handle, TheChecks, AMOFCHECKS))
          {
              case TRUE:
                   retval = 0;
                   break;
              case FALSE:
 		   printf("\nErrors were found. You did not use the /F switch. "
		          "Errors are not corrected\n\n");
	           retval = 1;
                   break;
	      case FAIL:
	           printf("Error accessing the volume\n");
                   DestroyFastTreeMap();
                   CloseReadWriteSectors(&handle);
                   return 2;
          }
       }

       printf("Elapsed time: %lds\n", time(NULL)-t);
    }
    
    /* Print out the volume summary. */
    if (!ReportVolumeSummary(handle))
    {
       retval = 2;
    }

    /* Destroy the fast tree map */
    if (fixerrors || checkerrors) /* don't create if it is not required */    
       DestroyFastTreeMap();

    if (thesame == FAIL) retval = 2;

    CloseReadWriteSectors(&handle);

    return retval;
}

void Usage(char switchch)
{
    printf("Checks a volume and returns a status report\n"
           "Usage:\n"
           "\tChkdsk [<volume>] [%cf] [%cd <files>] [%cr] [%cs]\n"
           "\n"
           "%cf: attempt to fix any errors found.\n"
           "%cd: print out the indicated files and show the fragmentation factor.\n"
           "%cr: scan the data area and try to recover unreadable data.\n"
           "%cs: only show drive summary.\n"
           "\n"
           "Note: if volume is ommited, the current drive is assumed.\n",
           switchch,
           switchch,
           switchch,
           switchch,
           switchch,
           switchch,
           switchch,
           switchch);
}
