/*
  Program:  ATTRIB
  Written By:  Phil Brutsche
  Copyright:  1998 by Phil Brutsche, under the terms of the GNU GPL.
  Maintained By:  Brian E. Reifsnyder


    This file is the source for an implementation of the DOS ATTRIB program.

    Copyright (C) 1998 by Phil Brutsche

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
Abbreviations:
plb - Phil Brutsche (the original author)
ber - Brian Reifsnyder (new maintainer as of 2001)
dal - David Lindauer (added subdirectory support)
*/

/*
/////////////////////////////////////////////////////////////////////////////
// V E R S I O N   H I S T O R Y  ///////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
*/

/*
2-12-1998 - Initial version - 0.1 (plb)
2-13-1998 - Started documentation.  Fixed bug in display_attribs where if
   there were no files found, the program exited regardless of any remaining
   file specifications in the argument list. (plb)
2-18-1998 - Finished documentation.  Changend calls to _dos_setfileattr and
   _dos_getfileattr to use the internal function setfileattr.
   This function was created by me for use in this program; It will be
   submitted to the maintainer of the FreeDOS C Library shortly. Version
   increased to 0.2. (plb)
2-22-1998 - Added MS undocumented function:  `attrib ,` will unset all
   file attributes.  (plb)
2-24-1998 - Added ability to change attributes on directories.  Increased
   to version 0.3.  (plb)
3-08-1998 - Rewrote to be able to use file lists (ie `attrib @filename` or
   `attrib @-`) per a suggestion from Charles Dye <raster@highfiber.com>.
   Moved up to version 0.4.  (plb)
3-16-1998 - Updated these friggin' comments :-).  Added a command-line
   parameter (`/?` in case yer interested.  `-?` works too.) to display
   online help.  Now version 0.5.  (plb)
7-19-1998 - Fixed bugs:
   * If files in another directory are specified, or are on another drive,
     it will set the attributes for the appropriately names file in the
     current directory.
   * Also: if only a drive is specified (ie `attrib d:`) it will assume `*.*`
     was meant.
   Now version 0.60.  (plb)
7-21-1998 - Fixed bugs:
   * In 4DOS attrib and MS-DOS attrib, if you give some attributes but
     no file names it will give all files and directories in the current
     directory those attributes.  (eg `attrib +r` would set all the files
     in the current directory as read-only).  This attrib didn't do that.
     (plb)
8-29-1998 - Changed behavior:
   * As per a suggestion from Charles Dye <raster@highfiber.com>, specifying
     only a directory on the command line will display the attributes for
     that directory.  (plb)
8-30-1998 - Changed behavior:
   * As per a suggestion from Charles Dye <raster@highfiber.com>, it now
     ignores the directory entries `..` and `.`.  (plb)
9-2-1998 - New version released as 0.63.  (plb)
9-4-1998 - When I released 0.63, I forgot to change the variable attrib_version
   from "0.60" to "0.63".  Oops!  Fixed now.  Now at 0.63a.  (plb)
9-26-1998 - Fixed a problem where ATTRIB didn't play nicely with FreeDOS.
   Now at 0.64.  (plb)
8-15-2000 - There appears that no major new bugs have been reported.  I'm
   releasing this as 1.0 and crossing my fingers... (plb)
7-25-2001 - Minor changes:
   * changed the appearance of the online help information.
   * Cleaned up and re-ordered source code.
   * Executable is now a packed with Apack 0.99b, written by Jorgen Ibsen.
     Greetings to Jorgen Ibsen!
   Now at 1.10.  (ber)
8-7-2001 - rewrite
   * Added subdir support.
   Now at 1.20. (dal)
   * Re-structured source code. (ber)
7-23-2002
   removed dos-versioning code (ber)
   now at 1.21. (ber)

*/

/*
/////////////////////////////////////////////////////////////////////////////
// D E F I N E S ////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
*/

#define RD_ONLY 0
#define HIDDEN 1
#define SYSTEM 2
#define ARCHIVE 3
#define INDENT "     "
#define SET_ATTRIBS 0
#define DISPLAY_ATTRIBS 1
#define PERROR(xx) { if (verbose) perror(xx) ; }

#define attrib_version "1.21"

/*
/////////////////////////////////////////////////////////////////////////////
// I N C L U D E S //////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
*/

#include <ctype.h>
#include <dir.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*
/////////////////////////////////////////////////////////////////////////////
// G L O B A L   V A R I A B L E S //////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
*/
extern unsigned int _stklen = 35000; /* Borland specific...sets stack size */

char version_string [] = "Version=" attrib_version;
char copyright_string [] = "Copyright=Copyright 1998, Phil Brutsche";

char *modifications [] = {
   "+S", "+H",
   "+R", "+A",
   "-S", "-H",
   "-R", "-A",
   NULL };

/*
/////////////////////////////////////////////////////////////////////////////
// P R O T O T Y P E S //////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
*/

int attribs_unchanged (int []);
int display_attribs (char *, int,int);
int display_attribs_traverse_subdir(char *, char *, int, int) ;
int set_file_attribs (int [], char *,int,int);
int setfileattr (char *, int);
int traverse_subdir(int attribs[], char *, char *, int, int) ;
int xsplit (char *, char *, char *, char*, char *);

void display_online_help (void);
void get_attrib_str (char [], char);
void invalid_attrib_change (char *);
void make_name (char *, char *, char *);
void process_file_list (int [], char *, int,int,int);
void secret_comma_change (int,int);
void set_attribs (int [], char *);

/*
/////////////////////////////////////////////////////////////////////////////
// F U N C T I O N S ////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
*/

int attribs_unchanged (int attribs [])
{
  return (attribs [0] == 0) && (attribs [1] == 0) &&
    (attribs [2] == 0) && (attribs [3] == 0);
}

int display_attribs (char *searchspec, int subdir, int verbose)
{
  struct ffblk file_info_block;
  int attributes = FA_RDONLY | FA_HIDDEN | FA_SYSTEM | FA_ARCH | FA_DIREC;
  unsigned int result;
  int rv = 0;
  char drive [3];
  char dir [240];
  char name [9];
  char ext [5];
  char searchpath [256];
  strcpy (searchpath, searchspec);
  xsplit (searchspec, drive, dir, name, ext);
  if (strcmp (name, "") == 0)
    strcat (searchpath, "\*.*");
  result = findfirst (searchpath, &file_info_block, attributes);
  while (result == 0)  /* There are files to process! */
    {
    char attrib_data [] = "----";
    char filename [256];
    if (strcmp (file_info_block.ff_name, "..") == 0
     || strcmp (file_info_block.ff_name, ".") == 0)
      {
      result = findnext (&file_info_block);
      continue;
      }
    get_attrib_str (attrib_data, file_info_block.ff_attrib);
    make_name (filename, searchspec, file_info_block.ff_name);
    if (verbose) fprintf (stdout, "[%s] %s\n", attrib_data, filename);
    rv = 1 ;
    result = findnext (&file_info_block);
    }
  if (subdir)
    {
    fnmerge (searchpath, drive, dir, "*", "*");
    result = findfirst (searchpath, &file_info_block, attributes);
    while (result == 0)  /* There are files to process! */
      {
      if (strcmp (file_info_block.ff_name, "..") == 0
       || strcmp (file_info_block.ff_name, ".") == 0)
	{
	result = findnext (&file_info_block);
	continue;
	}
      if (file_info_block.ff_attrib & FA_DIREC)
       rv |= display_attribs_traverse_subdir(searchspec,file_info_block.ff_name, subdir,verbose) ;
      result = findnext (&file_info_block);
      }
    }
  return rv ;
}
int display_attribs_traverse_subdir(char *search_spec, char *name, int subdir,int verbose)
{
  char buf[256], *p=search_spec,*q=buf, *r=p ;
  p = strrchr(search_spec,'\\') ;
  if (!p)
    {
    p = strrchr(search_spec,':') ;
    if (!p) p = search_spec;
    }
  if (r != p) while (r <= p) *q++=*r++;
  if (*p != '\\' && r != search_spec) *q++ = '\\';
  strcpy(q,name);
  q=q+strlen(q);
  *q++='\\';
  if (p != search_spec)	p++ ;
  strcpy(q,p) ;
  return display_attribs(buf,subdir,verbose) ;
}

void display_online_help (void)
{
  fprintf (stdout, "ATTRIB v%s:\n", attrib_version);
  fprintf (stdout, "Copyright 1998, Under the terms of the GNU GPL.\n\n");
  fprintf (stdout, "Displays or changes file attributes.\n\n");
  fprintf (stdout, "ATTRIB [+R | -R] [+A | -A] [+S | -S] [+H | -H] [[drive:][path]filename] [/S]\n");
  fprintf (stdout, "ATTRIB [+R | -R] [+A | -A] [+S | -S] [+H | -H] [@filelist] [/S]\n\n");
  fprintf (stdout, "  +   Sets an attribute.\n");
  fprintf (stdout, "  -   Clears an attribute.\n");
  fprintf (stdout, "  R   Read-only file attribute.\n");
  fprintf (stdout, "  A   Archive file attribute.\n");
  fprintf (stdout, "  S   System file attribute.\n");
  fprintf (stdout, "  H   Hidden file attribute.\n");
  fprintf (stdout, "  /S  Processes files in all directories in the specified path.\n");
}

void get_attrib_str (char string [], char attribs)
{
  if (FA_RDONLY & attribs) string [0] = 'R';
  if (FA_HIDDEN & attribs) string [1] = 'H';
  if (FA_SYSTEM & attribs) string [2] = 'S';
  if (FA_ARCH & attribs)   string [3] = 'A';
}

void invalid_attrib_change (char *invalid_attrib)
{
  int i = 0;
  fprintf (stdout, "Sorry.  \"%s\" is an invalid attribute change.\n",
   invalid_attrib);
  fprintf (stdout, "The valid attributes are:\n");
  while (modifications [i] != NULL)
  {
    fprintf (stdout, "%s%s%s%s\n", INDENT, modifications [i],
     INDENT, modifications [i + 1]);
    i += 2;
  }
}

void make_name (char *result, char *searchpath, char *filename)
{
  char drive [3];
  char dir [240];
  char name [9];
  char ext [5];
  xsplit (searchpath, drive, dir, name, ext);
  strcpy (result, drive);
  strcat (result, dir);
  strcat (result, filename);
}

void process_file_list (int attribs [], char *filelist, int mode, int subdir,int verbose)
{
  FILE *infile;
  char *result, data [256];
  if (strcmp (filelist + 1, "-") == 0) infile = stdin;
  else
    {
    infile = fopen (filelist + 1, "rt");
    if (infile == NULL)
      {
      fprintf (stderr, "Unable to open file list \"%s\"\n", filelist + 1);
      return;
      }
    }
  do
    {
    result = fgets (data, 256, infile);
    data [strlen (data) - 1] = '\0';
    if (result != NULL)
    if (mode == SET_ATTRIBS)
      {
      if (!set_file_attribs (attribs, data,subdir,verbose)) perror (data);
      } else if (!display_attribs (data,subdir,1)) perror (data);
    } while (result != NULL);
  if (infile != stdin) fclose (infile);
}

void secret_comma_change (int subdir, int verbose)
{
  int attribs [] = { -1, -1, -1, -1 };
  if (!set_file_attribs (attribs, "*.*",subdir,verbose)) perror("*.*");
}

int set_file_attribs (int attribs [], char *search_spec,int subdir,int verbose)
{
  struct ffblk file_info_block;
  int attributes = FA_RDONLY | FA_HIDDEN | FA_SYSTEM | FA_ARCH | FA_DIREC;
  unsigned int result;
  char drive [3];
  char dir [240];
  char name [9];
  char ext [5];
  char searchpath [256];
  int rv = 0 ;

  strcpy (searchpath, search_spec);
  xsplit (search_spec, drive, dir, name, ext);
  if (strcmp (name, "") == 0) strcat (searchpath, "\*.*");
  result = findfirst (searchpath, &file_info_block, attributes);
  if (result != 0) PERROR (search_spec);
  while (result == 0)  /* There are files to process! */
    {
    char old_attrib_data [] = "----";
    char new_attrib_data [] = "----";
    char filename [256];
    if (strcmp (file_info_block.ff_name, "..") == 0
     || strcmp (file_info_block.ff_name, ".") == 0)
      {
      result = findnext (&file_info_block);
      continue;
      }
    get_attrib_str (old_attrib_data, file_info_block.ff_attrib);
    set_attribs (attribs, &file_info_block.ff_attrib);
    get_attrib_str (new_attrib_data, file_info_block.ff_attrib);
    strcpy (filename, drive);
    strcat (filename, dir);
    strcat (filename, file_info_block.ff_name);
    result = setfileattr (filename, file_info_block.ff_attrib);
    if (result != 0) PERROR (filename)
    else if (verbose)
     fprintf (stdout, "[%s] -> [%s] %s\n", old_attrib_data, new_attrib_data,
     filename);
    rv = 1;
    result = findnext (&file_info_block);
    }
  if (subdir)
    {
    fnmerge (searchpath, drive, dir, "*", "*");
    result = findfirst (searchpath, &file_info_block, attributes);
    while (result == 0)  /* There are files to process! */
      {
      if (strcmp (file_info_block.ff_name, "..") == 0
       || strcmp (file_info_block.ff_name, ".") == 0)
	{
	result = findnext (&file_info_block);
	continue;
	}
      if (file_info_block.ff_attrib & FA_DIREC)
       rv |= traverse_subdir(attribs,search_spec, file_info_block.ff_name
       ,subdir,verbose);
      result = findnext (&file_info_block);
      }
    }
  return rv ;
}

void set_attribs (int attrib_list [], char *attrib)
{
  if ((attrib_list [RD_ONLY] > 0) && ((*attrib & FA_RDONLY) == 0))
   *attrib ^= FA_RDONLY;
  if ((attrib_list [RD_ONLY] < 0) && (*attrib & FA_RDONLY))
   *attrib ^= FA_RDONLY;

  if ((attrib_list [ARCHIVE] > 0) && ((*attrib & FA_ARCH) == 0))
   *attrib ^= FA_ARCH;
  if ((attrib_list [ARCHIVE] < 0) && (*attrib & FA_ARCH))
   *attrib ^= FA_ARCH;

  if ((attrib_list [HIDDEN] > 0) && ((*attrib & FA_HIDDEN) == 0))
   *attrib ^= FA_HIDDEN;
  if ((attrib_list [HIDDEN] < 0) && (*attrib & FA_HIDDEN))
   *attrib ^= FA_HIDDEN;

  if ((attrib_list [SYSTEM] > 0) && ((*attrib & FA_SYSTEM) == 0))
   *attrib ^= FA_SYSTEM;

  if ((attrib_list [SYSTEM] < 0) && (*attrib & FA_SYSTEM))
   *attrib ^= FA_SYSTEM;
}

int setfileattr (char *filename, int attributes)
{
  struct REGPACK regs;
  regs.r_ax = 0x4301;
  regs.r_cx = attributes;
  regs.r_ds = FP_SEG (filename);
  regs.r_dx = FP_OFF (filename);
  intr (0x21, &regs);
  if ((regs.r_flags & 0x0001) == 0) return 0;
  else return regs.r_ax;
}

int traverse_subdir(int attribs[], char *search_spec, char *name, int subdir,int verbose)
{
  char buf[256], *p=search_spec,*q=buf, *r=p;
  p = strrchr(search_spec,'\\');

  if (!p)
    {
    p = strrchr(search_spec,':');
    if (!p) p = search_spec;
    }
  if (r != p) while (r <= p) *q++=*r++ ;
  if (*p != '\\' && r != search_spec) *q++ = '\\';
  strcpy(q,name);
  q=q+strlen(q);
  *q++='\\';
  if (p != search_spec)	p++;
  strcpy(q,p) ;
  return set_file_attribs(attribs,buf,subdir,verbose) ;
}

int xsplit(char  *__path, char  *__drive,
 char  *__dir, char  *__name, char  *__ext )
{
  char buf[256],*p,*q =buf;
  strcpy(buf,__path);
  p = strrchr(buf,'.');
  if (*(p-1) != '.')
    {
    strcpy(__ext,p);
    *p=0;
    }
  else __ext[0] = 0;

  if (buf[1] == ':')
    {
    strncpy(__drive,buf,2);
    __drive[2] = 0;
    q += 2;
    }
  else __drive[0] = 0;

  p = strrchr(buf,'\\');
  if (p)
    {
    strcpy(__name,p+1);
    *(p+1) = 0;
    }
  else
    {
    strcpy(__name,q);
    q = 0;
    }
  if (q)
    strcpy(__dir,q);
  else __dir[0] = 0 ;
  return 0;
}

/*
/////////////////////////////////////////////////////////////////////////////
// M A I N   R O U T I N E //////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
*/

int main (int argc, char *argv [])
{
  int attr [4] = { 0, 0, 0, 0 };
  int i, has_filespec = 0,subdir = 0,verbose = 0;
  if (argc == 1)
    {
    if (!display_attribs ("*.*",0,1)) perror("*.*");
    return EXIT_SUCCESS;
    }
  for (i = 1; i < argc; i++)
    {
    strupr (argv [i]);
    /* Check for the command line help! */
    if (strcmp (argv [i], "/?") == 0 || strcmp (argv [i], "-?") == 0)
      {
      display_online_help ();
      return 0;
      }
    /* This is an `undocumented feature` from MS's ATTRIB.  It basically
       unsets all the attributes of all the files in a directory */
    if (strcmp (argv [i], ",") == 0)
      {
      secret_comma_change (subdir,verbose);
      return 0;
      }
    if (strlen (argv [i]) == 2)
      {
      /* Not so fast, bucko.  It may be an attribute, but it must be
	 a VALID attribute to continue */
      if (strchr ("-+/", argv [i][0]))
	{
	int change;
	if (argv [i][0] == '-') change = -1;
	else if (argv [i][0] == '+') change = 1;
	else if (argv [i][0] == '/') change = 0 ;

	switch (change == 0 ? 0 : argv [i][1])
	  {
	  case 'R':
	    attr [RD_ONLY] = change;
	    break;
	  case 'S':
	    attr [SYSTEM] = change;
	    break;
	  case 'A':
	    attr [ARCHIVE] = change;
	    break;
	  case 'H':
	    attr [HIDDEN] = change;
	    break;
	  default:
	    if (change == 0 && argv[i][1] == 'S') subdir = 1 ;
	    else if (change ==0 && argv[i][1] == 'V') verbose = 1 ;
	    else
	      {
	      invalid_attrib_change (argv [i]);
	      return EXIT_FAILURE;
	      }
	    break ;
	  }
	argv [i] = NULL;
	}
      }
    else has_filespec = 1;
    }
  if (! has_filespec )
    if (! attribs_unchanged (attr))
      {
      if (!set_file_attribs (attr, "*.*",subdir,verbose)) perror("*.*") ;
      return EXIT_SUCCESS;
      }
    else
      {
      if (!display_attribs("*.*",subdir,1)) perror("*.*") ;
      return EXIT_SUCCESS ;
      }
    for (i = 1; i < argc; i++)
      {
      if (argv [i] != NULL)
	{
	if (attribs_unchanged (attr))
	  {
	  if (argv [i][0] == '@')
	   process_file_list (attr, argv [i], DISPLAY_ATTRIBS,subdir,verbose);
	  else if (!display_attribs (argv [i],subdir,1)) perror(argv[i]) ;
	  }
	else
	  if (argv [i][0] == '@')
	  process_file_list (attr, argv [i], SET_ATTRIBS,subdir,verbose);
	  else
	    if (!set_file_attribs (attr, argv [i],subdir,verbose))
	      perror(argv[i]) ;
	}
      }
  return EXIT_SUCCESS;
}