/***************************************************************************
 Structure of the KL file:
	1 DWORD:    flags: needed modules
	1 WORD:     offset of COMBIs
	1 WORD:	offset of XString
	1 BYTE:	LastXStr
	xxx         Transtable starts here
----------------------------------------------------------------------------
 TransTable:   	1 BYTE:  Scancode
			1 BYTE:  Flags   0..2:  Length of columns
						  3:  Reserved
						  4:  Lock (cancel processing)
						  5:  Affected by NumLock
						  6:  Affected by CapsLock
						  7:  Replace scancode
		    [ 1 BYTE:  New scancode (optional) ]
			1 BYTE:  XBits for the columns
			n BYTEs: Chars/Xcodes

 Combi:   		1 BYTE:  Combinated character (')
			1 BYTE:  Number of pairs
			n WORD:  Pairs  character/diacritic  e

 XString:		1 BYTE:  length of XString (n)
			n WORD:  scancode/ASCII pairs
----------------------------------------------------------------------------
  File:
	TransTable:		[E]n[C][N][Sm][X]  n1   n2  ...
	Combi:		'  a  e  i ...  (spaces can be omitted; #can be used)
	XString:		Formatted string 1
				Formatted string 2
				...

	Specials in transtable:    !command
					   !Ccombi   			(200-)
					   !Llayout                   (100-)
					   !Ssubmapping               (120-)
					   !Aalternative mapping      (170-)
					   !Ouserdefinedlock          (188-)
					   !Huserdefined shifts       (180-)
					   #number                    character number
					   ##					character #
					   #!					character !
	XString:			   \Axxx         (0,ASCII)
					   \Kxxx,yyy     (xxx,yyy)
					   \Sxxx	     (ScanCode,0)
					   \n		     return (#13)
					   \\		     \ character
					   \[key]        HOME
							     END
							     PGUP
							     PGDN
							     UP
							     DOWN
							     LEFT
							     RIGHT
							     DEL
							     INS
							     Fx  (function)
							     SFx (shift+function)
							     CFx (control+function)
							     AFx (alt+function)
***************************************************************************/


#include<stdlib.h>
#include<stdio.h>
#include<string.h>
#include<ctype.h>

#define MaxBuffer 5000
#define MaxColumns 6

	   /*  Flags on the KEYTRANS section */
#define  LockKeyFlag  		 16
#define  ReplaceScanCodeFlag  128
#define  CapsLockFlag          64
#define  NumLockFlag           32

	   /*  Flags in the file */
#define  F_COMBI		0x0001
#define  F_BEEP		0x0002
#define  F_BasicCom     0x0004
#define  F_ExtCom       0x0008
#define  F_DefShift     0x0010
#define  F_UserShift    0x0020
#define  F_APM          0x0080
#define  F_Trans        0x0100
#define  F_XString      0x0200
#define  F_Full         0x0400

typedef unsigned char BYTE;
typedef unsigned short int WORD;
typedef unsigned long DWORD;
typedef BYTE BOOL;

/*////////// V A R I A B L E S  ////////////////////////////////////*/

BYTE   buffer[MaxBuffer];		/* buffer and pointer */
WORD   bufferPtr  = 0;

char   line[255]  = "";
BYTE   lineptr    = 0;
WORD   lineNumber = 0;

DWORD  flags      = 0;			/* flags */

char   finName[255]="";
char   foutName[255]="";	      /* file names */
FILE   *fin, *fout;			/* file variables */



/*////////// F U N C T I O N S  ////////////////////////////////////*/

void  SyntaxError  ( char *oline, BYTE colNumber, BYTE errorType)
{
	BYTE i;

	printf ("%s(%u): Error %u: ", finName, lineNumber, errorType );
	switch (errorType)
	{
		case 1:   puts ("Unknown flag");
			    break;
		case 2:   puts ("Syntax error");
			    break;
		case 3:   puts ("Too many columns");
			    break;
		case 4:   puts ("Invalid scancode number");
			    break;
		case 5:   puts ("Unexpected end of line");
			    break;
		case 6:   puts ("Too many COMBI pairs");
			    break;
		case 7:   puts ("Invalid key name");
			    break;
		case 8:   puts ("Invalid function key number");
			    break;
		default:  puts ("Undefined error");
	}

	puts (oline);
	for (i=1; i<colNumber; i++)  putchar(' ');
	puts ("^");

	exit (-1);
}

WORD  NoSpaces ( void )
{
	for (;line[lineptr]==32;lineptr++);
}


char *GetLine ( void )
{
	int i;
	do  {
		line[0] = 0;
		fgets (line, 255, fin);
		lineptr = 0;
		for (i = strlen(line)-1; (line[i]==10) || (line[i]==13) || (line[i]==' '); line[i--]=0);
		NoSpaces ();
		lineNumber++;
		if (!line[lineptr])
			*line = 0;
	} while ( ((line[0]==';') || !line[0]) && !feof(fin) );
	lineptr = 0;
	return (line);
}


WORD  WriteToBuffer (BYTE *str, WORD len)
{
	WORD i = 0;

	while  ( (i<len) && (bufferPtr<=MaxBuffer))
		buffer [bufferPtr++] = str[i++];

	return i;
}


BYTE  ReadNumber ( void )
{
	char cvtstr[8];
	BYTE i=0;

	while ( (line[lineptr]>='0') && (line[lineptr]<='9'))
		cvtstr[i++] = line[lineptr++];
	cvtstr[i] = 0;

	return (  (BYTE) (atoi (cvtstr) & 0xFF) );
}


BYTE ReadCharacter ( void )
{
	if (line[lineptr]=='#')
	{
		lineptr++;
		if ( (line[lineptr]=='#') || (line[lineptr]=='!'))
			return (line[lineptr++]);
		else
			return (ReadNumber());
	}
	else
		return (line[lineptr++]);
}


void UpdateFlags ( BYTE KEYcom )
{
	if ((200<=KEYcom) && (KEYcom<=234))   flags |= F_COMBI;
	if (162==KEYcom)				  flags |= F_BEEP;
	if ((80<=KEYcom) && (KEYcom<=199))    flags |= F_BasicCom;

	if ((80<=KEYcom) && (KEYcom<=99))     flags |= F_ExtCom;
	if ((140<=KEYcom) && (KEYcom<=159))   flags |= F_ExtCom;
	if ((162<=KEYcom) && (KEYcom<=199))   flags |= F_ExtCom;

	if ((80<=KEYcom) && (KEYcom<=89))     flags |= F_DefShift;
	if (165==KEYcom)                      flags |= F_DefShift;

	if ((80<=KEYcom) && (KEYcom<=97))     flags |= F_UserShift;
	if (166==KEYcom)                      flags |= F_UserShift;
	if ((180<=KEYcom) && (KEYcom<=187))   flags |= F_UserShift;

	if ((150<=KEYcom) && (KEYcom<=154))   flags |= F_UserShift;
}

void ParseTransTable ( void )
{
	BYTE tempbuffer [12] = {0,0,0,0,0,0,0,0,0,0,0,0},xbitptr=2,columnptr;

	NoSpaces ();

	/*** 1: preceeding E and Scancode ***/
	columnptr = lineptr;   /** temporal ussage of columnptr!! **/
	if (toupper(line[lineptr])=='E')
	{
		tempbuffer[0] = 0x80;
		lineptr ++;
	}
	tempbuffer[0] |= ReadNumber();
	if  ( !tempbuffer[0] ||
		((tempbuffer[0]>127) && (toupper(line[columnptr]!='E'))))
		SyntaxError (line, lineptr, 4);


	/*** 2: Flags ***/
	while (line[lineptr] != 32)
	{
		switch (toupper(line[lineptr]))
		{
			case 'C': tempbuffer[1] |= CapsLockFlag;
				    break;
			case 'N': tempbuffer[1] |= NumLockFlag;
				    break;
			case 'S': tempbuffer[1] |= ReplaceScanCodeFlag;
				    lineptr++;
				    tempbuffer[2]  = ReadNumber();
				    lineptr--;  /* compensate the one below */
				    xbitptr++;
				    break;
			case 'X': tempbuffer[1] |= LockKeyFlag;
				    break;
			default:  SyntaxError  ( line, lineptr, 1);
		}
		lineptr++;
	}

	NoSpaces ();

	/*** 3: columns ***/
	columnptr = xbitptr + 1;
	while ( line[lineptr] )
	{
		if (line[lineptr]=='!')	   /* XCommand */
		{
			lineptr++;
			tempbuffer[xbitptr] |= (1 << (columnptr-xbitptr-1));
			switch (toupper(line[lineptr++]))
			{
				case 'C':   tempbuffer[columnptr] = 199;
						break;
				case 'L':   tempbuffer[columnptr] = 99;
						break;
				case 'S':   tempbuffer[columnptr] = 119;
						break;
				case 'A':   tempbuffer[columnptr] = 169;
						break;
				case 'O':   tempbuffer[columnptr] = 179;
						break;
				case 'H':   tempbuffer[columnptr] =  89;
						break;
				default:    lineptr--;    /* there was no section specifier */
			}
			tempbuffer[columnptr] += ReadNumber();
			UpdateFlags ( tempbuffer[columnptr] );
		}
		else
			tempbuffer[columnptr] = ReadCharacter ();

		if (line[lineptr] && (line[lineptr] != 32))
			SyntaxError (line, lineptr+1, 2);

		NoSpaces ();

		if ((columnptr++)>xbitptr+MaxColumns)
			SyntaxError (line, lineptr, 3);


	}

	/*** 4: update the length ***/
	tempbuffer[1] |= (columnptr-xbitptr-1);

	/*** 5: Copy to buffer ***/
	WriteToBuffer (tempbuffer, 3 + ((tempbuffer[1] >> 7) & 1) +
						 (tempbuffer[1] & 0x07) );

}

void ParseCombi ( void )
{
	BYTE tempbuffer[130], count=0;

	/*** 1: leading character ***/
	NoSpaces ();
	tempbuffer[0] = ReadCharacter();
	NoSpaces ();

	/*** 2: pairs ***/
	while (line[lineptr] /* && (count<63) */ )
	{
		tempbuffer[(count+1)*2] = ReadCharacter();
		NoSpaces();

		if (!line[lineptr])
			SyntaxError (line,lineptr++,5);

		tempbuffer[(count+1)*2+1] = ReadCharacter();
		NoSpaces();

		if ((count++) >128)
			SyntaxError (line,lineptr,6);

	}

	tempbuffer[1] = count;

	/*** 3: copy to buffer ***/
	WriteToBuffer (tempbuffer, (count+1)<<1);
}

BYTE strpos (char *str1, char *str2)
{
	char *s = strstr (str1,str2);
	if (s==NULL)
		return (0);
	else
		return (s-str1+1);
}

BYTE ReadXKeyName ( void )
{
	BYTE b=0;

	switch (toupper(line[lineptr]))
	{
		case 'F': lineptr++;
			    switch (toupper(line[lineptr]))
			    {
				 case 'S': lineptr++;
					     b = ReadNumber ();
					     if (!b || (b>12))
						   SyntaxError (line,lineptr,8);
					     if ( (0<b) && (b<11) )
						     b += 83;
					     else
						     b += 124;
					     if (line[lineptr]!=']')
						   SyntaxError (line,lineptr,7);
					     lineptr++;
					     return (b);
				 case 'C': lineptr++;
					     b = ReadNumber ();
					     if (!b || (b>12))
						   SyntaxError (line,lineptr,8);
					     if ( (0<b) && (b<11) )
						     b += 93;
					     else
						     b += 126;
					     if (line[lineptr]!=']')
						   SyntaxError (line,lineptr,7);
					     lineptr++;
					     return (b);
				 case 'A': lineptr++;
					     b = ReadNumber ();
					     if (!b || (b>12))
						   SyntaxError (line,lineptr,8);
					     if ( (0<b) && (b<11) )
						     b += 103;
					     else
						     b += 128;
					     if (line[lineptr]!=']')
						   SyntaxError (line,lineptr,7);
					     lineptr++;
					     return (b);
				 default:  b = ReadNumber ();
					     if (!b || (b>12))
						   SyntaxError (line,lineptr,8);
					     if ( (0<b) && (b<11) )
						     b += 58;
					     else
						     b += 122;
					     if (line[lineptr]!=']')
						   SyntaxError (line,lineptr,7);
					     lineptr++;
					     return (b);
			    }
		default:  if (strpos( &line[lineptr], "HOME]") == 1 )
			    {
				  lineptr += 5;
				  return  ( 71 );
			    }
			    if (strpos( &line[lineptr], "END]") == 1 )
			    {
				  lineptr += 4;
				  return  ( 79 );
			    }
			    if (strpos( &line[lineptr], "PGUP]") == 1 )
			    {
				  lineptr += 5;
				  return  ( 73 );
			    }
			    if (strpos( &line[lineptr], "PGDN]") == 1 )
			    {
				  lineptr += 5;
				  return  ( 81 );
			    }
			    if (strpos( &line[lineptr], "UP]") == 1 )
			    {
				  lineptr += 3;
				  return  ( 72 );
			    }
			    if (strpos( &line[lineptr], "DOWN]") == 1 )
			    {
				  lineptr += 5;
				  return  ( 80 );
			    }
			    if (strpos( &line[lineptr], "LEFT]") == 1 )
			    {
				  lineptr += 5;
				  return  ( 75 );
			    }
			    if (strpos( &line[lineptr], "RIGHT]") == 1 )
			    {
				  lineptr += 6;
				  return  ( 77 );
			    }
			    if (strpos( &line[lineptr], "DEL]") == 1 )
			    {
				  lineptr += 4;
				  return  ( 83 );
			    }
			    if (strpos( &line[lineptr], "INS]") == 1 )
			    {
				  lineptr += 4;
				  return  ( 82 );
			    }
			    SyntaxError (line, lineptr, 2);
	}
}




/* NOTE: hi=scancode, lo=character */
WORD ReadXChar ( void )
{
	WORD w;

	if (line[lineptr]=='\\' )
	{
		lineptr++;
		switch (toupper(line[lineptr++]))
		{
			case 'A' : return (ReadNumber());
			case 'S' : return (ReadNumber() << 8);
			case 'K' : w = ReadNumber();
				     if (line[lineptr]==',')
					  lineptr++;
				     else
					  SyntaxError (line,lineptr,2);
				     return ( (w<<8) | ReadNumber() );
			case 'N' : return ( (28<<8) | 13 );
			case '\\': return ( '\\' );
			default  : return ( ReadXKeyName () << 8 );
		}
	}
	else
		return ( line[lineptr++] );
}

void ParseXString ( void )
{
	WORD tempbuffer[256];
	BYTE count=0;

	while (line[lineptr])
		tempbuffer[count++] = ReadXChar();

	WriteToBuffer ( &count, 1);
	WriteToBuffer ( (BYTE *) &tempbuffer, count << 1);
}


void ShowHelp ( void )
{
	puts ("KEYCOMP  filename  [outputfilename] [/T] [/F]");
	puts ("KEYCOMP  /?\n");
	puts ("FreeDOS KEYB 2 Prototype4-style .KEY files compiler.");
	puts ("filename       name of the source file to be opened. KEY extension");
	puts ("               is added if no extension is found");
	puts ("outputfilename name of the target compiled filename. If omitted,");
	puts ("               the same filename with KC extension is created");
	puts ("/T             specifies that the file remaps any of these:");
	puts ("		    - function keys F1..F12 (any combination)");
	puts ("		    - Ctrl + cursor keys ");
	puts ("		    - Alt + numbers above the letters");
	puts ("/F             specifies that the layout is FULL, and does not");
	puts ("               depend on previously loaded routines\n");

}


void  MainError  ( BYTE errorType, char *ExtraLine)
{
	BYTE i;

	printf ("Error %u: ", errorType );
	switch (errorType)
	{
		case 1:   puts ("Wrong number of parameters specified");
			    break;
		case 2:   puts ("Unable to open source file name:");
			    break;
		case 3:   puts ("[KEYS] section expected");
			    break;
		case 4:   puts ("[COMBI] section expected");
			    break;
		case 5:   puts ("[XSTRINGS] section expected");
			    break;
		case 6:   puts ("Syntax error");
			    break;
		case 7:   puts ("Unknown switch");
			    break;
		default:  puts ("Undefined error");
	}

	puts (ExtraLine);

	exit (-1);
}



void main ( BYTE argc, char *argv[] )
{
	/* Local vars */
	WORD     combiOffset   = 0; 		/* offsets and pointers to buffer */
	WORD     xstrOffset    = 0;
	WORD     *h            = (WORD *) buffer;
	DWORD    *bf  	     = (DWORD *) buffer;

	int	   i=0;				/* other indices */
	BYTE     numXstr    = 0;


	/* Initialize data */
	bufferPtr = 9;


/*/////////// PART 1: check parameters ////////////////////////////////*/

	if ( argc<2 )
		MainError (1, "");

	for (i=1; i<argc; i++)
	{
		if (argv[i][0]=='/')
		{
			if (strlen(argv[i])>2)
				MainError (6, argv[i]);
			switch ( toupper (argv[i][1]))
			{
				case 'T':  flags |= F_Trans;
					     break;
				case 'F':  flags |= F_Full;
					     break;
				case '?':  ShowHelp ();
					     return;
				default:   MainError (7, argv[i]);
			}
		}
		else
		{
			if (!finName[0])
				strcpy (finName, argv[i]);
			else if (!foutName[0])
				strcpy (foutName, argv[i]);
			else
				MainError (1, "");
		}
	}

	if (!finName[0])
		MainError (1, "");


/*/////////// PART 2: get filenames ///////////////////////////////////*/

	/* Check presence of extension */
	for ( i = strlen(finName)-1; (i>=0) && (finName[i]!='\\') && (finName[i]!='.'); i--);

	if (! ((i>=0) && (finName[i]=='.')) )
		strcat (finName, ".KEY");

	/* Get second argument */
	if (!foutName[0])
	{
		for ( i=strlen(finName)-1; finName[i]!='.'; i--);
		strncpy (foutName, finName,i+1);
		strcat (foutName, "KC");
	}

	/* Try to open the files */
	if ((fin = fopen (finName,"rt"))==NULL)
		MainError (2, finName);
	if ((fout = fopen (foutName,"wb"))==NULL)
		MainError (2, foutName);

/*/////////////////// PART 3: read sections /////////////////////*/

	/* Section 1: [KEYS] */

	if (strcmp (strupr (GetLine()),"[KEYS]"))
		MainError (3,"");

	while (!feof(fin) && strcmp (GetLine(), ""))
	{
		if (line[0]=='[') break;
		ParseTransTable ();
	}

	buffer[bufferPtr++]=0;		/* this 0 to indicate no more scancodes */

	if (!feof(fin))
	{

	/* Section 2: [COMBI] */


	if (!strcmp (strupr(line),"[COMBI]"))
	{
		combiOffset = bufferPtr;

		while (!feof(fin) && strcmp (GetLine(), ""))
		{
			if (line[0]=='[') break;
			ParseCombi ();
		}
	}

	buffer[bufferPtr++]=0;		/* this 0 to indicate no more COMBIs */
					  /* it is NOT required, but it is SAFE... */

	if (!feof(fin))
	{

	/* Section 3: [XSTRINGS] */

	xstrOffset = bufferPtr;

	if (strcmp (strupr(line),"[XSTRINGS]"))
		MainError (5,"");

	while (!feof(fin) && strcmp (GetLine(), ""))
	{
		ParseXString ();
		numXstr++;
	}

	/* No more sections */
	}
	}

	fclose (fin);


/*/////////////////// PART 4: fill in the header and copy the buffer ////*/

	if (numXstr>0)   flags |= F_XString;

	/* Update the header */
	*bf =flags;
	*(h+2)= combiOffset;
	*(h+3)= xstrOffset;
	buffer[8] = numXstr ;

	/* Copy the buffer */
	for (i=0; i<bufferPtr; i++)
		fputc (buffer[i], fout);

	fclose (fout);

	printf ("File %s succesfully compiled (0 Errors)\n", foutName);

}
