/*
 *  sf.c        FAT12/16; save unallocated clusters
 *
 *  This file is part of the D_DISK library.
 *  Copyright (C) 1998, Gregg Jennings.
 *
 *  See D_DISK.HTM for the library specifications.
 *  See D_DISK.TXT for overview the implementation.
 *  See NOTES.TXT for particulars about the source files.
 *
 *  30-Nov-1998 greggj  more comments
 *  06-Oct-1998 prozac  ds.d_units + 1
 *  26-Sep-1998 grejen  cleaned up and added options
 *  06-Jun-1998 grejen  fixed signed char bug in convert()
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>

#ifdef _WIN32
#include <direct.h>     /* _getdrive() */
#else
#include <dos.h>        /* _dos_getdrive() */
#endif
#include <conio.h>      /* kbhit(), getch() */

/*
    The sectors that make up the FAT of a typical GB drive can
    exceed the DOS/80x86 64K segment limit. Therefore, the __huge
    data C extensions must be used with 16-bit DOS compilers.
*/

#include "d_disk.h"
#include "d_lib.h"      /* clustertosector() */

#ifdef SMALL_DATA
#define lmalloc(n) halloc(n,1)
#define lfree(p)   hfree(p)
#else
#define lmalloc(n) malloc(n)
#define lfree(p)   free(p)
#endif

void usage(void);
long readclusters(int dh, struct d_stat *ds, int LARGE *clusters, char *file,
                 int test, long free);
int LARGE *diskfat(int dh, struct d_stat *ds);
int diskfat12(int dh, struct d_stat *ds, int LARGE *clusters);
int diskfat16(int dh, struct d_stat *ds, int LARGE *clusters);
long arraycnt(int LARGE *array, long max, int n);
void dumpbuf(char LARGE *buf, long n);
int displaymask(int c, int f);
unsigned int convert(unsigned char *buf, unsigned int size);
int percent(long m, long n);


int all,extended;


int main(int argc, char **argv)
{
int i,dh;
long nfree;
struct d_stat ds;
int LARGE *clusters;
unsigned fdisk,cdisk,sdisk;

    setvbuf(stdout,NULL,_IONBF,0);      /* flush stdout every printf */

    printf("\n");

    if (argc < 3 || argv[1][0] == '-' || argv[1][0] == '/')
        usage();

    all = extended = 0;

    while (--argc >= 3) {
        if (strcmp(argv[argc],"-e") == 0)
            extended = 1;
        if (strcmp(argv[argc],"-a") == 0)
            all = 1;
    }

#ifdef _WIN32                       /* TODO: make a getdrive() function */
    cdisk = _getdrive();            /*       for the library (there is  */
#else                               /*       one in DISKLIB)            */
    _dos_getdrive(&cdisk);
#endif

    sdisk = toupper(argv[1][0])-'@';
    fdisk = (argv[2][1] == ':') ? toupper(argv[2][0])-'@' : cdisk;

    if (fdisk == sdisk) {
        printf("Writing the file to the same disk defeats the purpose"
               " of this program.\n\n");
        usage();
    }

    printf("opening drive...\r");

    if ((dh = d_open(argv[1],D_RDONLY)) == -1) {
        d_perror("d_open");
        return 1;
    }

    printf("checking drive...\r");

    if (d_hstat(dh,&ds) == -1) {
        d_perror("d_stat");
        return 1;
    }

    printf("reading FAT...   \r");

    if ((clusters = diskfat(dh,&ds)) == NULL) {
        printf(" Bummer dude...\n");
        return 1;
    }

    /* count free clusters (free == 0) */

    nfree = arraycnt(clusters,ds.d_units+1,0);

    printf("%c: ",sdisk+'@');
    printf("%lu bytes per sector, ",ds.d_bsize);
    printf("%lu sectors per cluster. ",ds.d_usize);

    if (ds.d_usize > 1)
        printf("(%luK clusters)",(ds.d_bsize*ds.d_usize)/1024);

    printf("\n");

    printf("Of %ld clusters, ",ds.d_units+1);
    printf("%ld are free, ",nfree);
    printf("which would be %ld bytes.\n",nfree*ds.d_usize*ds.d_bsize);

    printf("(ASCII masking may reduce file size.)\n");

	printf("\nDo you want to save to a file? [y] Or test for actual size? [t] ");

    i = getchar();

    if (i == 'y' || i == 't') {
        printf("\n");
        readclusters(dh,&ds,clusters,argv[2],i == 't',nfree);
    }

    printf("\n");

    return 0;
}

/*
 *  readclusters    reads all unallocated clusters to a file
 *
 */

long readclusters(int dh, struct d_stat *ds, int LARGE *clusters, char *file,
                 int test, long nfree)
{
FILE *fh;
long i,s,t,n;
unsigned int j;
unsigned char *buf;
unsigned int clustersize;

    clustersize = (unsigned int)(ds->d_usize * ds->d_bsize);

    buf = malloc(clustersize);
    if (buf == NULL)
        return 1;

    if (!test && (fh = fopen(file,"wb")) == NULL) {
        perror(file);
        return 1;
    }

    n = 2;
    t = 0;
    for (i = 2; i <= ds->d_units+1; i++)
    {
        if (kbhit() && getch() == '\033')
            break;

        if (clusters[i] == 0)
        {
            s = clustertosector(i,ds);
            d_lseek(dh,s,SEEK_SET);

            d_read(dh,buf,(unsigned int)ds->d_usize);

            /*
                Errors are ignored; whole cluster lost... could read
                each sector of cluster but floppy clusters are usually
                only one sector.

                Note that there is no abort on errors, and that the
                buffer (buf) should be zeroed first.

                See DISKLIB's DF.C program.

            */

            if (all) {
                j = clustersize;
                if (!test)
                    fwrite(buf,1,j,fh);
            }
            else if ((j = convert(buf,clustersize)) != 0) {
                if (!test)
                    fwrite(buf,1,j,fh);
            }

            t += j;

            ++n;
            printf("cluster %5ld (%5u)",i,j);
            printf(" - %2d%% done",percent(n,nfree));
            printf(", %lu bytes so far",t);
            printf("\r");
        }
    }

    printf("cluster %5ld        ",i-1);
    printf(" - %2d%% done",percent(n,nfree));
    printf(", %lu bytes total ",t);
    printf("\r");

    if (!test)
        fclose(fh);
    free(buf);

    return 0;
}

/*
 *  diskfat     reads the FAT into an array of integers (front end)
 *
 *  Notes: ds.d_units == number of clusters - 1, and clusters start
 *  being counted from 0 (clusters 0 and 1 actually don't exist), so
 *  the array to hold the clusters must be ds.d_units + 2 and the
 *  loops must be from 2 to <= ds.d_units - 1.
 */

int LARGE *diskfat(int dh, struct d_stat *ds)
{
int i;
int LARGE *buf;

    if ((buf = lmalloc((ds->d_units+2)*sizeof(int))) == NULL)
        return NULL;

    d_lseek(dh,ds->d_bres,SEEK_SET);

    if (ds->d_units < 4096)
        i = diskfat12(dh,ds,buf);
    else
        i = diskfat16(dh,ds,buf);

    if (i != 1) {
        printf("diskfat: %s error",(i == -1) ? "memory" : "read");
        lfree(buf);
        return NULL;
    }

    return buf;
}

/*
 *  diskfat12   reads 12-bit FAT
 *
 *  Notes: ds.d_units == number of clusters - 1, and clusters start
 *  being counted from 0 (clusters 0 and 1 actually don't exist), so
 *  the array to hold the clusters must be ds.d_units + 2 and the
 *  loops must be from 2 to <= ds.d_units - 1.
 */

int diskfat12(int dh, struct d_stat *ds, int LARGE *clusters)
{
int h,i;
unsigned int *pc,c;
unsigned char *buf;

#ifdef SMALL_DATA
    if ((ds->d_bsize * ds->d_ftsize) > (long)UINT_MAX)
        return -1;
#endif

    if ((buf = malloc((size_t)(ds->d_bsize * ds->d_ftsize))) == NULL)
        return -1;

    if (d_read(dh,buf,(int)ds->d_ftsize) != (int)ds->d_ftsize) {
        free(buf);
        return 0;
    }

    buf += 3;
    for (h = 2; h <= ds->d_units+1; buf++) {
        for (i = 0; i < 2; i++) {
            pc = (unsigned int *)(buf++);
            if (i&1)
                c = *pc >> 4;               /* odd */
            else
                c = *pc & 0xFFF;            /* even */
            clusters[h] = c;
            if (++h > ds->d_units+1)
                break;
        }
    }

    free(buf);
    return 1;
}

/*
 *  diskfat16   reads 16-bit FAT
 *
 *  Notes: ds.d_units == number of clusters - 1, and clusters start
 *  being counted from 0 (clusters 0 and 1 actually don't exist), so
 *  the array to hold the clusters must be ds.d_units + 2 and the
 *  loops must be from 2 to <= ds.d_units - 1.
 */

int diskfat16(int dh, struct d_stat *ds, int LARGE *clusters)
{
long h,n;
unsigned int c;
unsigned short LARGE *buf;      /* NOTE: FAT sector buffer is 16-bit integers */

    if ((buf = lmalloc(ds->d_bsize * ds->d_ftsize)) == NULL)
        return -1;

    if (d_read(dh,buf,(int)ds->d_ftsize) != (int)ds->d_ftsize) {
        free(buf);
        return 0;
    }

#if 0       /* for testing only */
    printf("saving FAT...\n");
    dumpbuf((char LARGE *)buf,ds->d_bsize * ds->d_ftsize);
#endif

    n = 2;
    for (h = 2; h <= ds->d_units+1; h++) {
        c = buf[n++];
        clusters[h] = c;
    }

    lfree(buf);
    return 1;
}

/*
 * arraycnt     count occurences of a number in array
 *
 * Note: longs are used becuase *array may be __huge.
 *
 */

extern long arraycnt(int LARGE *array, long max, int n)
{
long i;
long c;

    c = 0;
    for (i = 0; i < max; i++) {
        if (array[i] == n) {
            ++c;
        }
    }

    return c;
}

/*
 *  convert     convert buffer, removes non printable characters
 *
 */

unsigned int convert(unsigned char *buf, unsigned int size)
{
int c;
unsigned char *b;
unsigned int i,j;

    b = buf;
    for (i = j = 0; i < size; i++) {
        c = buf[i];

        if (c < 0x7f) {
            if (isprint(c) || isspace(c))
                b[j++] = buf[i];
        }
        else if (extended) {
            b[j++] = buf[i];
        }
    }
    return j;
}


int percent(long m, long n)
{
float f;

    f = (float)m / n;
    f *= 100;
    return (int)f;
}

void usage(void)
{
    printf("usage: sf <d:> <file> [-e] [-a]\n\n"
           "Saves all unallocated clusters of drive `d:' to `file',\n"
           "stripping non-ASCII and non-whitespace control chracters.\n"
           "\n"
           "Options: -e, save extended characters; -a, save all characters.\n"
           "\n"
           "There is a run-time option to check for file size.\n"
           "\n"
           "Built with the free D_DISK library: "
           "http://www.diskwarez.com/d_disk.htm\n");
    exit(1);
}

#ifdef TESTING

void dumpbuf(char LARGE *buffer, long bytecnt)
{
FILE *fh;
long n;
int b;

   if ((fh = fopen("fat.dmp","w")) == NULL)
      return;

   for (n = 0; n < bytecnt; n+=16)
   {
      fprintf(fh,"%04x: ",(unsigned int)n);
      for (b = 0; b < 16; b++)
         fprintf(fh,"%02x ",buffer[n+b]);
      for (b = 0; b < 16; b++)
         fputc(displaymask(buffer[n+b],2),fh);
      fprintf(fh,"\n");
   }
   fclose(fh);
}

int displaymask(int c, int filter)
{
int i = '.';

   if (filter == 0)
   {
      if ((c < 7 || c > 10) && c != 0 && c != 13 && c != 255)
         i = c;
   }
   else if (filter == 1)
   {
      if (c >= ' ' && c <= 0xfe)
         i = c;
   }
   else if (filter == 2)
   {
      if (isprint(c))
         i = c;
   }
   return i;
}

#endif
