/***
*dos.c   - DOS calls
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   DOS calls
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
*******************************************************************************/

/*
   Versions:

   1.3   02-Jul-1998    __GNUC__
   1.2   10-Jan-1998    IOCTL enhancements; doserror()
   1.0   28-Nov-1997    created (moved from DISKIO.C)

   Release Notes:

   DOS specifc calls only. See also IOCTL.C, READ.C and WRITE.C for DJGPP.

   Programming Notes:

   The disk read/write functions return -1 for success.

*/

#include <stdio.h>
#include <assert.h>
#include <dos.h>

#ifdef DISKED              /* just so you can easily use this elsewhere */
#include "diskio.h"
#include "error.h"
#endif
#include "general.h"
#include "dosio.h"

/***
*exterror   -   Get DOS Extended Error information
*               (just to hide the non-standard function)
****/

extern int exterror(void)
{
#ifdef __GNUC__
struct _DOSERROR d;
   return _dosexterr(&d);
#else
   return _dosexterr(NULL);
#endif
}

/***
*doscall -  generic DOS calls
*
****/

extern int doscall(DOS_FUNCTION ax, int bx, int cx, int dx, void __far *data)
{
union _REGS regs;

   regs.x.ax = ax;            /* perform the call */
   regs.x.bx = bx;
   regs.x.cx = cx;
   regs.x.dx = dx;
   _intdos(&regs,&regs);

   switch (ax)                /* handle all special cases */
   {
      case DOS_GET_VER:
         regs.h.cl = regs.h.al;        /* trash CL */
         regs.h.al = regs.h.ah;
         regs.h.ah = regs.h.cl;
         break;
      case DOS_GET_VER_5:
         regs.h.al = regs.h.bh;
         regs.h.ah = regs.h.bl;
         break;
      case DOS_GET_FREE_SPACE:
#ifdef DISKED                                         /* with DISKED the */
         if (regs.x.ax >= 0xff || error.num != -1)    /* INT 24h will be */
#else                                                 /* trapped and */
         if (regs.x.ax >= 0xff)                       /* error.num set */
#endif
            return exterror();
         else
         {
            FREESPACE *fs = (FREESPACE *)data;
            fs->secs_cluster = regs.x.ax;
            fs->avail_clusters = regs.x.bx;
            fs->sec_size = regs.x.cx;
            fs->num_clusters = regs.x.dx;
         }
         break;

      default:
         break;
   }
   return regs.x.ax;
}

/***
*dosioctl   -  generic DOS IOCTL
*
*  Note: SUBFUNCTION 0x0C category codes are not supported
****/

#ifndef __GNUC__

extern int dosioctl(DOS_SUBFUNCTION subfunc, DOS_MINOR_CODE code, int device,
                    void __far *data)
{
union _REGS r;
struct _SREGS s;

   r.x.ax = 0x4400 + subfunc;
   r.x.bx = device;
   if (code != DOS_MINOR_NONE)
      r.x.cx = 0x0800 + code;

   if (data)
   {
      r.x.dx = _FP_OFF(data);
      s.ds = _FP_SEG(data);
   }
   _intdosx(&r,&r,&s);

   if (r.x.cflag)
      return exterror();

   /* handle special cases */

   if (subfunc == DOS_DEV_REMOVE)
      return r.x.ax;

   if (subfunc == DOS_DRV_REMOTE)
      return r.x.dx;

   return 0;
}

#endif

/*
   Actual sector IO functions.

   Returns -1 for OK.
*/

/***
*dosio   -  Absolute sector I/O
*
****/

#ifdef __GNUC__
extern int disk_read(int disk, long sector, void *buffer, int nsecs,
   int secsize);
extern int disk_read_ext(int disk, long sector, void *buffer, int nsecs,
   int secsize);
extern int disk_write(int disk, long sector, void *buffer, int nsecs,
   int secsize);
extern int disk_write_ext(int disk, long sector, void *buffer, int nsecs,
   int secsize);
#define __far
#endif


extern int dosio(DOS_FUNCTION func, int ext, int disk, UINT32 sector,
                 BYTE __far *buffer)
{
#ifdef __GNUC__
int i;

   i = 0;
   --disk;

   /*
      PORT NOTE: for DJGPP the sector size is hardcoded to 512, This
      should be changed.
   */

   if (func == DOS_READ) {
      if (ext)
		 i = disk_read_ext(disk,sector,buffer,1,512);
      else
		 i = disk_read(disk,sector,buffer,1,512);
   }
   else if (func == DOS_WRITE) {
      if (ext)
       i = disk_write_ext(disk,sector,buffer,1,512);
      else
       i = disk_write(disk,sector,buffer,1,512);
   }
#ifndef NDEBUG
   else
      assert(0)
#endif
   return i;

#else
union _REGS regs;
struct _SREGS sregs;
DCB Dcb;
DCB *dcb = &Dcb;

   regs.x.ax = disk - 1;

   if (!ext)
   {
      regs.x.dx = (unsigned int)sector;
      regs.x.cx = 1;
      regs.x.bx = _FP_OFF(buffer);
      sregs.ds = _FP_SEG(buffer);
   }
   else
   {
      dcb->sector = sector;
      dcb->number = 1;
      dcb->buffer = buffer;
      regs.x.cx = 0xffff;
      regs.x.bx = _FP_OFF(dcb);
      sregs.ds = _FP_SEG(dcb);
   }
   _int86x(func,&regs,&regs,&sregs);
   if (regs.x.cflag)
      return regs.h.al;
   return -1;
#endif
}

/***
*diskioctl  -  disk read/write track
*
****/

extern int diskioctl(DOS_FUNCTION cmd, int disk,
                     UINT16 track, UINT16 sec, UINT16 head, BYTE __far *buffer)
{
RWBLOCK Blk;
RWBLOCK *blk = &Blk;
int i;

   blk->special = 0;
   blk->head = head;
   blk->track = track;
   blk->sector = sec - 1;
   blk->nsecs = 1;
   blk->buffer = buffer;

   if (cmd == DOS_WRITE)
      i = dos_write_track(disk,blk);
   else
      i = dos_read_track(disk,blk);

   if (i == 0) i = -1;

   return i;
}

/* DOS INT 21/59 error messages */

static char *doserr_msg[] = {
   "Error 0",
   "Invalid function number",
   "File not found",
   "Path not found",
   "Memory blocks destroyed",
   "Too many open files",
   "Permission denied",
   "Bad file number",
   "Not enough core",
   "Invalid memory block address",
   "Invalid environment",
   "Invalid format",
   "Invalid access code",
   "Invalid data",
   "Unknown error",
   "Invalid drive specified",
   "Attempt to remove CurDir",
   "Not same device",
   "No more files",
   "Write protect",
   "Bad unit",
   "Drive not ready",
   "Bad Command",
   "CRC Error",
   "Bad Length",
   "Seek Error",
   "Not DOS Disk",
   "error"
};

/* INT 25h/26h error messages */

static char *diskerr_msg[] = {
   "Write Protected Disk",
   "Unknown Unit",
   "Drive Not Ready",
   "Unknown Command",
   "Data Error (CRC)",
   "Bad Request Structure Length",
   "Seek Error",
   "Unknown Media Type",
   "Sector Not Found",
   "Printer Out of Paper",
   "Write Fault",
   "Read Fault",
   "General Failure",
   "Reserved Error",
   "Reserved Error",
   "Invalid Disk Change",
   "unknown error"
};

extern const char *doserror(int error)
{
   if (error < 0 || error > (int)(sizeof(doserr_msg)/sizeof(char*)))
      error = (sizeof(doserr_msg)/sizeof(char*))-1;
   return doserr_msg[error];
}

extern const char *dosioerror(int error)
{
   if (error < 0 || error > (int)(sizeof(diskerr_msg)/sizeof(char*)))
      error = (sizeof(diskerr_msg)/sizeof(char*))-1;
   return diskerr_msg[error];
}
