/***
*arrays.c - array handling
*
*this file is part of DISKED
*Copyright (c) 1991-1998, Gregg Jennings.  All rights reserved.
*   P O Box 200, Falmouth, MA 02541-0200
*
*Purpose:
*   General array manipulation routines.
*
*Notice:
*   This program can be distributed only in accordance with, and
*   accompanied by, the DPU Software License. See COPYING.TXT or,
*   <http://www.diskwarez.com/dpu.htm>.
*******************************************************************************/

/*
   Versions:

   0.3   02-Jul-1998    ULONG_MAX_DIG
   0.2   21-Feb-1997    changed _itoa() to _ultoa()
   0.1   14-Mar-1995    added arraysub()
   0.0   15-Dec-1994    These routines were written to replace some
                        specific ones that in FILES.C.  They have
                        are very general purpose though.

   Release Notes:

   Except for print() and output() this module is independent. (That's
   nice and is what "reusable" really means... reusable by _ANY_ program.)

   Programming Notes:

   The use of __huge arrays are for  DISKED's clusters[] array. I have
   not used these for other arrays yet but the huge-ness should not be
   a cause of any problems nor any performance decrease.

*/

#include <stdlib.h>           /* for itoa() */

#include "console.h"          /* for output() (you can use putch()) */
#include "limits.h"           /* my additions to <limits.h> */

/*
#define output
#define print
*/

/* NO globals referenced here */

/* NO globals defined here */


/***
*arraymax   -  get max number of array
*
****/

extern int arraymax(int __huge *array, size_t max, int n)
{
size_t i;

   for (i = 0; i < max; i++)
      if (array[i] > n)
         n = array[i];
   return n;
}

/***
*arraymin   -  get min number of array
*
****/

extern int arraymin(int __huge *array, size_t max, int n)
{
size_t i;

   for (i = 0; i < max; i++)
      if (array[i] < n)
         n = array[i];
   return n;
}

/***
*arraycnt   -  count occurences
*
****/

extern int arraycnt(int __huge *array, size_t max, int n)
{
size_t i;
int c;

   for (c = i = 0; i < max; i++)
      if (array[i] == n)
         ++c;
   return c;
}

/***
*arrayfirst -  return offset of first occurance of n
*              max if not found
****/

extern size_t arrayfirst(int __huge *array, size_t max, int n)
{
size_t i;

   for (i = 0; i < max; i++)
      if (array[i] == n)
         break;
   return i;
}

/***
*arraylast  -  return offset of last occurance of n
*              0 if not found
****/

extern size_t arraylast(int __huge *array, size_t max, int n)
{
size_t i;

   for (i = max-1; i > 0; i--)
      if (array[i] == n)
         break;
   return i;
}

/***
*arrayfind  -  return offset of occurance of n
*              max if not found
****/

extern size_t arrayfind(int __huge *array, size_t max, int n)
{
size_t i;

   for (i = 0; i < max; i++)
      if (array[i] == n)
         break;
   return i;
}

/***
*arraytrav  -  find next/prev occurance of number at offset n
*
****/

extern size_t arraytrav(int __huge *array, size_t max, size_t n, int dir)
{
int num;

   num = array[n];

   for (n += dir; ; n += dir)
   {
      if (dir == -1 && n == 0xFFFF)
         n = max-1;
      if (dir == 1 && n > max)
         n = 0;
      if (array[n] == num)
         return n;
   }
}

/***
*arraysub   -  retreive array offsets of number in array
*
****/

extern size_t arraysub(int __huge *array, int *sub, size_t max, int n)
{
size_t i,j;

   for (i = j = 0; i < max; i++)
   {
      if (array[i] == n)
         sub[j++] = i;
   }
   return j;
}

/***
*arraymap -
*
*     2 3 4 5 6 7 8 9 10 11 12 13 14 15
*     2 2 2 2 2 0 0 3 3  0  0  0  0  4
*
*     7 8 11-14 ...
*
****/

extern int arraymap(int __huge *array, size_t max, int n, int base)
{
size_t i;
int c = 0;
int count = 0;
int indash = 0;
char buf[ULONG_MAX_DIG+1];

   for (i = 2; i < max; i++)
   {
      if (array[i] != n)
      {
         c = 1;
         if (count == 2)
            output(' ');
         if (count > 1)
         {
            print(_ultoa(i-1,buf,base));
            output(' ');
         }
         if (count == 1)
            output(' ');
         count = indash = 0;
      }
      else
      {
         if (count == 1)
            count++;
         else if (count == 0)
         {
            print(_ultoa(i,buf,base));
            count++;
         }
         else if (!indash)
         {
            output('-');
            indash = 1;
            count++;
         }
      }
   }
   if (array[i-1] == n && indash)
      print(_ultoa(i-1,buf,base));

   return c;
}

/* special for listing DISKED's tagged[] array */

extern void plist(long *array, size_t max, int base)
{
size_t i;
char buf[LONG_MAX_DIG+1];

   for (i = 1; i < max; i++)
      print("%d:%s ",i,_ltoa(array[i],buf,base));
}

#if 0

#include <stdio.h>
#include <stdlib.h>

#define output putchar
#define print printf

extern void arraymap(int *array, int max, int n, int base);

void main(void)
{
int array[] = { 0, 0, 1, 1, 0, 0, 2, 2, 2, 2, 0, 2, 0, -1 };

   arraymap(array,sizeof(array)/sizeof(int),2,10);
}

#endif
