	PAGE
;****************************************************
;*********** SOUND & GRAPHICS ROUTINES **************
;****************************************************

;INTRINSIC TO GENERATE SOUND USING THE 8253 CHIP
;The length of the sound is SLENG/18.2 in seconds
;The frequency is 1.19 million / SFREQ
;Volume= 0 = No sound, Other = Sound

PORTB	EQU	61H		;B PORT
COMREG	EQU	43H		;COMMAND REGISTER
LATCH2	EQU	42H		;

SFREQ	DW	0		;FREQUENCY
SLENG	DW	0		;LENGTH OF NOTE INITIALLY THEN TIME OF DAY

SOUND	PROC	NEAR
	POP	SFREQ		;SET FREQUENCY
	POP	SLENG		;SET TIME
	POP	AX		;GET ON/OFF
	OR	AX,AX		;ZERO?
	JE	SOUND2		;THEN DELAY, WITHOUT SOUND

	IN	AL,PORTB	;GET PORT VALUE
	OR	AL,03H		;ENABLE SPEAKER AND TIMER
	OUT	PORTB,AL
	MOV	AL,0B6H		;SET CHANNEL-2 FOR MODE 3
	OUT	COMREG,AL

;START THE NOTE

	MOV	AX,SFREQ	;GET FREQUENCY
	OUT	LATCH2,AL
	MOV	AL,AH
	OUT	LATCH2,AL
SOUND2:
	PUSH	ES		;set up ES to access BIOS's tick count
	XOR	AX, AX
	MOV	ES, AX
	MOV	AX, ES:[046CH]	;get tick count
	ADD	SLENG, AX	;add it to duration to get timeout time
SOUND5:	XCHG	DX, AX		;save tick count in dx
	MOV	AX, ES:[046CH]	;get tick count
	CMP	AX, DX		;did it suddenly go backwards?
	JNS	SOUND7		;skip if not
	 SUB	SLENG, 0B0H	;compensate for reset at midnight
SOUND7:	CMP	AX, SLENG	;loop until current time >= timeout time
	JS	SOUND5
	POP	ES

;TURN OFF SPEAKER

	IN	AL,PORTB	;GET CURRENT VALUE
	AND	AL,0FCH		;TURN OFF SPEAKER
	OUT	PORTB,AL
	JMP	CMLRET
SOUND	ENDP


;INTRINSIC TO SET THE VIDEO MODES

SETVID	PROC	NEAR
	XOR	AX,AX		;move pen to upper-left corner
	MOV	STARTX,AX
	MOV	STARTY,AX

	POP	AX		;GET THE MODE
	SUB	AH,AH		;SET VIDEO MODE FUNCTION (=0)
	INT	10H		;CALL BIOS
	JMP	CMLRET		;RETURN
SETVID	ENDP


;INTRINSIC TO CLEAR VIDEO SCREEN

VCLEAR	PROC	NEAR
	XOR	AX,AX		;move pen to upper-left corner
	MOV	STARTX,AX
	MOV	STARTY,AX
	CALL	CLEAR
	JMP	CMLRET
VCLEAR	ENDP


;INTRINSIC TO MOVE TO A NEW COORDINATE

MOVE	PROC	NEAR
	POP	STARTY		;GET COORDINATES
	POP	STARTX
	JMP	OPGO
MOVE	ENDP


;PLACE A DOT ON THE GRAPHIC SCREEN

DOT	PROC	NEAR
	POP	AX		;GET THE PIXEL VALUE
	POP	DX		;GET Y POSITION
	MOV	STARTY,DX	;SAVE IT
	POP	CX		;GET X POSITION
	MOV	STARTX,CX	;SAVE IT

;Test video mode and dispatch to proper dot routine. (BIOS is so incredibly
; slow so the dirty work is done here.)
	push	es			;save critical registers
	push	di
	mov	bl, al			;save color
	xor	ax, ax
	mov	es, ax
	mov	al, es:BIOMOD		;get video mode
	cmp	al, 0Ch
	jbe	dotcga			;jump if modes 0..0Ch
	cmp	al, 13h
	je	dot13			;jump if mode 13h
	jb	dotvga			;jump if modes 0Dh..12h
	cmp	al, 6Ah
	je	dotvga			;jump if mode 6Ah
dotcga:
	mov	al, bl			;get color
	MOV	AH,0CH			;GET PIXEL WRITE FUNCTION
	CALL	GETPAG			;GET CURRENT BIOS PAGE
	INT	10H			;DO THE FUNCTION
dot99:
	pop	di			;restore critical registers
	pop	es
	JMP	CMLRET			;RETURN

;Draw dot for mode 13h (320x200). (This is 100 times faster than BIOS!)
dot13:	mov	ax, 0A000h		;A000:(Y*320+X):= color
	mov	es, ax
	mov	ax, 320
	mul	dx			;dx:ax := ax*dx  (dx=STARTY)
	add	ax, cx
	mov	di, ax
	mov	al, bl			;get color
	stosb				;es:[di++]:= al
	jmp	dot99

;Draw dot for planar modes (0Dh..12h, 6Ah) (Several times faster than BIOS)
dotvga:	mov	ax, es:SYSCOL		;get number of bytes per scan line
	mul	dx			;Y*SYSCOL + X/8
	mov	dx, STARTX
	mov	cl, 3
	shr	dx, cl
	add	ax, dx
	push	ax

	mov	ax, es:SCROFF		;get offset to start of video buffer
	mov	cl, 4			;convert it to segment address
	shr	ax, cl
	add	ax, 0A000h		;add base segment address for video RAM
	mov	es, ax			;point to it with es so we can access it

	mov	dx, 03CEh		;access VGA registers
	mov	ax, 8008h		;set bit mask register
	mov	cx, STARTX
	and	cl, 07h
	shr	ah, cl			;80h >> (X&7)
	out	dx, ax			;(word writes are twice as fast)

	mov	ax, 1803h		;set data rotate register to 18h for xor
	test	bl, 80h			;is xor bit actually set?
	jne	dot20			;skip if so
	 mov	ah, 0			;else set for "replace" instead of xor
dot20:	out	dx, ax

	mov	ah, bl			;set/reset register:= color
	and	ax, 7F00h
	out	dx, ax

	mov	ax, 0F01h		;enable set/reset register:= 0Fh
	out	dx, ax

	pop	di			;address of byte to modify
	xchg	al, es:[di]		;latch old data & modify unmasked bits

;Set VGA registers to their default values
	mov	ax, 0FF08h		;enable all 8 bits in bit mask register
	out	dx, ax
	mov	ax, 0003h		;make sure xor mode is turned off
	out	dx, ax
	mov	ax, 0001h		;disable set/reset function
	out	dx, ax
	jmp	dot99
DOT	ENDP


;READ A DOT ON THE GRAPHIC SCREEN

REDDOT	PROC	NEAR
	POP	DX		;GET Y POSITION
	POP	CX		;GET X POSITION
	push	ds		;save critical registers
	xor	ax, ax
	mov	ds, ax
	mov	al, ds:BIOMOD	;get video mode
	cmp	al, 13h
	jne	red10		;jump if not mode 13h

;Read dot for mode 13h (320x200). (This is 100 times faster than BIOS!)
	mov	ax, 0A000h	;color:= A000:(Y*320+X)
	mov	ds, ax
	mov	ax, 320
	mul	dx		;dx:ax := ax*dx  (dx = Y position)
	add	ax, cx		;+ X position
	mov	si, ax
	lodsb			;al:= ds:[si++]
	pop	ds
	jmp	red20
red10:
	pop	ds
	MOV	AH,0DH		;GET PIXEL READ FUNCTION
	CALL	GETPAG		;GET CURRENT BIOS PAGE
	INT	10H		;DO THE FUNCTION
red20:	XOR	AH,AH		;ZERO HIGH BYTE
	PUSH	AX		;RETURN THE VALUE
	JMP	CMLRET		;RETURN
REDDOT	ENDP

	PAGE

BIOMOD	EQU	0449H		;BIOS RAM HOLDS CURRENT VIDEO MODE
SYSCOL	EQU	044AH		;BIOS RAM HOLDS NUMBER OF SCREEN COLUMNS

	IF SMALL		;use slow, but small, line drawing routines

;-------------------------------------------------------------------------------
;Intrinsic to draw a line on the graphic screen.
; LINE(X, Y, COLOR);
;
LINE	PROC	NEAR
	POP	AX		;GET CURRENT COLOR AND DOTTEDNESS
	mov	word ptr color,ax	;set color and dotdash
;	mov	bp,sp
;	mov	ax,[bp]		;set coordinates
;	mov	endy,ax
;	mov	ax,[bp+2]
;	mov	endx,ax

	pop	endy
	pop	endx

	call	lline			;draw the line
	mov	ax,endx			;copy start into end
	mov	startx,ax
	mov	ax,endy
	mov	starty,ax
	JMP	CMLRET		;RETURN
LINE	ENDP

;----------------------------------------------------------------------
;Routine to draw a straight line. This uses Bresenham's algorithm, and
; uses BIOS pixel routine for compatability, but is very s-l-o-w.
;
	align	2
startx	dw	0			;begining x position
endx	dw	0			;ending x position
starty	dw	0			;begining y position
endy	dw	0			;ending y position
daginx	dw	0			;diagonal x increment
daginy	dw	0			;diagonal y increment
dshort	dw	0			;shortest distance
strinx	dw	0			;straight x increment
striny	dw	0			;straight y increment
strcnt	dw	0			;straight count
dagcnt	dw	0			;diagonal count
color	db	0			;color attribute (must precede dotdash)
dotdash	db	0			;high byte of color for dotted lines

lline:	push	si			;save critical registers
	push	di

	call	getpag			;get currently displayed page in bh
	mov	cx,1			;set initial increments
	mov	dx,cx

;Calculate vertical difference
	mov	di,endy			;subtract start from end
	sub	di,starty
	jge	short llin10		;skip if positive
	neg	dx			;change sign of increment
	neg	di			;and difference
llin10:	mov	daginy,dx		;save y increment

;Calculate horizontal difference
	mov	si,endx			;subtract start from end
	sub	si,startx
	jge	short llin20		;skip if positive
	neg	cx			;change sign of increment
	neg	si			;and difference
llin20:	mov	daginx,cx		;save x incremnt

;See which difference is greater
	cmp	si,di			;is horizonal longer than vertical
	jge	short llin30		;skip if so
	sub	cx,cx			;no x change when straight
	xchg	si,di			;put longest in cx
	jmp	short llin40		;now save them
llin30:	sub	dx,dx			;no y change when straight
llin40:	mov	dshort,di		;save value of short distance
	mov	strinx,cx		;save increments
	mov	striny,dx

;Calculate adjustment factor
	mov 	ax,dshort		;get short value
	add	ax,ax			;double it
	mov	strcnt,ax		;save it
	sub	ax,si			;subtract long distance
	mov	di,ax			;use as loop counter
	sub	ax,si			;subtract long distance
	mov	dagcnt,ax		;save it

;Set up to draw the line
	mov	cx,startx		;set starting x
	mov	dx,starty		;set starting y
	inc	si			;inc long for endpoint

llin50:	dec	si			;decrement long distance
	js	short llin90		;exit if done

	rol	dotdash,1		;are we between dots?
	jc	short llin60		;jump if so

	mov	al,color		;get color code
	mov 	ah,0ch			;draw pixel function
	int	10h			;call BIOS
llin60:
	test	di,di			;straight segment?
	jge	short llin70		;do diagonal if not

;Draw straight line segment
	add	cx,strinx		;set up x and y increments
	add	dx,striny		;for straight segments
	add	di,strcnt		;add to adjustment factor
	jmp	short llin50		;do next pixel

;Draw diagonal line segment
llin70:	add	cx,daginx		;set x and y increments
	add	dx,daginy		;for diagonal segment
	add	di,dagcnt		;adjust adjustment
	jmp	short llin50		;do next pixel

llin90:	pop	di			;restore critical registers
	pop	si
	ret

	ELSE
	PAGE

;INTRINSIC TO DRAW A LINE

LINE	PROC	NEAR
	POP	AX		;GET CURRENT COLOR AND DOTTEDNESS
	NOT	AH		;MAKE 1'S PLOT POINTS IN DOTDASH
	MOV WORD PTR COLOR,AX
	MOV	BP,SP		;GET STACK FRAME
	MOV	AX,[BP]		;GET COORDINATES, BUT LEAVE ON STACK
	MOV	ENDY,AX
	MOV	AX,[BP+2]
	MOV	ENDX,AX
	PUSH	DI		;SAVE REGISTERS
	PUSH	SI
	PUSH	ES

;GET SETUP INFORMATION

	SUB	AX,AX		;GET NUMBER OF COLUMNS
	MOV	ES,AX
	MOV	AX,ES:SYSCOL	;GET NUMBER OF COLUMNS
	MOV	LINSIZ,AX
	MOV	AL,ES:BIOMOD	;GET VIDEO MODE
	MOV	VIDMOD,AL

;GET PAGE OFFSET AND CONVERT TO PARAGRAPH FORM

	MOV	BX,ES:SCROFF	;GET SCREEN OFFSET
	SHR	BX,1		;CONVERT TO SEGMENT
	SHR	BX,1
	SHR	BX,1
	SHR	BX,1
	MOV	SCRPAR,BX	;SAVE IT 

;TEST VIDEO MODES AND DISPATCH TO PROPER LINE ROUTINE

	CMP	AL,0DH		;MODES 0 - 0CH?
	JB	DOCGA		;SKIP IF YES
	CMP	AL,013H		;MODES 0DH - 12H
	JB	DOVGA		;SKIP IF YES
	CMP	AL,06AH		;MODES 13H - 69H?
	JB	DOCGA		;SKIP IF YES
	JE	DOVGA		;HANDLE VESA MODE 6A

DOCGA:	CALL	CLINE		;HANDLE CGA LINES
	JMP	SHORT XEXIT
DOVGA:	CALL	VLINE		;HANDLE VGA LINES

XEXIT:	POP	ES		;RESTORE REGISTERS
	POP	SI
	POP	DI
	POP	STARTY		;COPY END COORDINATES INTO START
	POP	STARTX
	JMP	CMLRET		;RETURN
LINE	ENDP
	PAGE

;LINE DRAWING ROUTINES

	EVEN
STARTX	DW	0		;BEGINING X POSITION
ENDX	DW	0		;ENDING X POSITION
STARTY	DW	0		;BEGINING Y POSITION
ENDY	DW	0		;ENDING Y POSITION
COLOR	DB	0		;COLOR ATRIBUTE
DOTDASH	DB	0		;HIGH BYTE OF COLOR FOR DOTTED AND DASHED LINES
LINSIZ	DW	80		;NUMBER OF BYTES ON A LINE
SCRPAR	DW	0		;PAGE PARAGRAPH
VIDMOD	DB	3		;CURRRENT VIDEO MODE

	EVEN
DAGINX	DW	0		;DIAGINAL X INCREMENT
DAGINY	DW	0		;DIAGINAL Y INCREMENT
STRINX	DW	0		;STRAIGHT X INCREMENT
STRINY	DW	0		;STRAIGHT Y INCREMENT
STRCNT	DW	0		;STRAIGHT COUNT
DAGCNT	DW	0		;DIAGONAL COUNT
COLORX2	DW	0		;PRE-ADJUSTED COLOR (NO XOR)

CLINE	PROC	NEAR
	MOV	AH,VIDMOD	;GET VIDEO MODE
;SET BP REGISTER TO POINT TO ROUTINE TO HANDLE LINE DRAW FOR THIS MODE
	CMP	AH,6		;TEST MODE
	JA	CL30		;SKIP IF GREATER THAN 6
	JE	CL10		;HANDLE MODE 6
	CMP	AH,4		;TEST MODE
	JB	CL35		;SKIP IF LESS THAN 4
	LEA	BP,LIN500	;USE CGA ROUTINE FOR 5
	MOV	AL,COLOR
	AND	AL,83H		;MAKE SURE BIT 2 IS CLEAR AND SET STATUS
	JNS	CL05		;JUMP IF NO XOR
	OR	AL,04H		;MOVE XOR BIT TO BIT 2
CL05:	AND	AX,0007H	;CLEAR OLD XOR BIT AND HIGH BYTE
	JMP	SHORT CL28	;ENTER COMMON CODE
CL10:
	LEA	BP,LIN600	;USE CGA ROUTINE FOR 6
	MOV	AL,COLOR
	AND	AL,81H		;MAKE SURE BIT 1 IS CLEAR AND SET STATUS
	JNS	CL25		;JUMP IF NO XOR
	OR	AL,02H		;MOVE XOR BIT TO BIT 1
CL25:	AND	AX,0003H	;CLEAR OLD XOR BIT AND HIGH BYTE
CL28:	SHL	AX,1		;TIMES TWO FOR WORD INDEX
	MOV	COLORX2,AX

	MOV	AX,0B800H	;POINT TO CGA SCREEN
	ADD	AX,SCRPAR	;ADD IN PAGE OFFSET
	MOV	ES,AX		;USE ES
	JMP	SHORT CL40
CL30:
	CMP	AH,13H		;MODE 13H (320x200x256)?
	JNE	CL35		;JUMP IF NOT
	LEA	BP,LIN1300	;USE ROUTINE FOR MODE 13H
	JMP	SHORT CL40
CL35:
	LEA	BP,LIN100	;USE BIOS ROUTINE
CL40:
	CALL	GETPAG		;GET CURRENT BIOS PAGE INTO BH REG
	MOV	CX,1		;SET INITIAL INCREMENTS
	MOV	DX,1		; CX = X AXIS,   DX = Y AXIS

;CALCULATE VERTICAL DIFFERENCE: SI = DELTA X,   DI = DELTA Y

	MOV	DI,ENDY		;SUBTRACT START FROM END
	SUB	DI,STARTY
	JGE	KEEPY		;SKIP IF POSITIVE
	NEG	DX		;CHANGE SIGN OF INCREMENT
	NEG	DI		;AND DIFFERENCE
KEEPY:	MOV	DAGINY,DX	;SAVE Y INCREMENT

;CALCULATE HORIZONAL DIFFERENCE

	MOV	SI,ENDX		;SUBTRACT START FROM END
	SUB	SI,STARTX
	JGE	KEEPX		;SKIP IF POSITIVE
	NEG	CX		;CHANGE SIGN OF INCREMENT
	NEG	SI		;AND DIFFERENCE
KEEPX:	MOV	DAGINX,CX	;SAVE X INCREMNT

;TEST LINE SEGMENTS

	CMP	SI,DI		;IS HORIZONAL LONGER THAN VERTICAL?
	JGE	HRZSEG		;SKIP IF SO
	SUB	CX,CX		;NO X CHANGE WHEN STRAIGHT
	XCHG	SI,DI		;PUT LONGER IN SI, SHORTER IN DI
	JMP	SHORT SAVALS	;NOW SAVE THEM
HRZSEG:	MOV	DX,0		;NO Y CHANGE WHEN STRAIGHT
SAVALS:	MOV	STRINX,CX	;SAVE INCREMENTS
	MOV	STRINY,DX

;CALCULATE ADJUSTMENT FACTORS FOR DECISION VARIABLE
; STRCNT:= 2*DI
; DAGCNT:= 2*DI - 2*SI
; DECISION:= 2*DI - SI

	MOV 	AX,DI		;GET SHORT VALUE
	SHL	AX,1		;DOUBLE IT
	MOV	STRCNT,AX	;SAVE IT
	SUB	AX,SI		;SUBTRACT LONG DISTANCE
	MOV	DI,AX		;USE AS DECISION VARIABLE
	SUB	AX,SI		;SUBTRACT LONG DISTANCE AGAIN
	MOV	DAGCNT,AX

;SETUP TO DRAW THE LINE

	MOV	CX,STARTX	;SET STARTING X
	MOV	DX,STARTY	;SET STARTING Y
	INC	SI		;INC LONG FOR ENDPOINT
	JMP	BP		;CALL SPECIFIC LINE ROUTINE
CLINE	ENDP

;DRAW LINES FOR GRAPHIC MODES WE DON'T HANDLE BY CALLING BIOS FOR POINT PLOT

LIN100:
LIN110:	ROL	DOTDASH,1	;ARE WE BETWEEN DOTS?
	JNC	LIN120		;SKIP PIXEL PLOT IF SO
	MOV	AH,0CH		;WRITE PIXEL FUNCTION
	MOV	AL,COLOR	;GET COLOR
	INT	10H		;CALL BIOS BH=PAGE, CX=X, DX=Y
LIN120:
	TEST	DI,DI		;STRAIGHT SEGMENT?
	JGE	LIN150		;DO DIAGONAL IF NOT

	ADD	CX,STRINX	;SET UP X AND Y INCREMENTS
	ADD	DX,STRINY	; FOR STRAIGHT MOVE
	ADD	DI,STRCNT	;ADJUST DECISION VARIABLE
	DEC	SI		;DECREMENT PIXEL COUNT
	JNE	LIN110		;NEXT PIXEL
	JMP	SHORT LIN190	;EXIT
LIN150:
	ADD	CX,DAGINX	;SET X AND Y INCREMENTS
	ADD	DX,DAGINY	; FOR DIAGONAL MOVE
	ADD	DI,DAGCNT	;ADJUST DECISION VARIABLE
	DEC	SI		;DECREMENT PIXEL COUNT
	JNE	LIN110		;NEXT PIXEL
LIN190:
	RET
	PAGE

;LINE DRAW FOR CGA MODES 4 AND 5 (320x200x4)

	EVEN
;PIXEL COLOR AND MASK TABLE
; LOW BYTE = MASK,   HIGH BYTE = COLOR
;		 ---------- NORMAL ---------     ----------- XOR -----------
;		  0       1       2       3       0       1       2       3
CTAB5	DW	0003FH, 0403FH, 0803FH, 0C03FH, 000FFH, 040FFH, 080FFH, 0C0FFH
	DW	000CFH, 010CFH, 020CFH, 030CFH, 000FFH, 010FFH, 020FFH, 030FFH
	DW	000F3H, 004F3H, 008F3H, 00CF3H, 000FFH, 004FFH, 008FFH, 00CFFH
	DW	000FCH, 001FCH, 002FCH, 003FCH, 000FFH, 001FFH, 002FFH, 003FFH

LIN500:
	MOV	AX,WORD PTR LIN514	;STUFF "SHL SI,1" INTO THE CODE
	MOV	WORD PTR LIN515,AX
	MOV	AX,OFFSET CTAB5		;MODIFY CODE FOR MODE 4 & 5 TABLE
	MOV	WORD PTR LIN516+2,AX
LIN502:	MOV	AX,AX			;STUFF THIS 2-BYTE NOP INTO THE CODE
	MOV	AX,WORD PTR LIN502
	MOV	WORD PTR LIN518,AX
LIN504:
;MULTIPLY Y DIRECTION VARIABLES BY 40 BYTES PER SCAN LINE
	MOV	AX,40			;DX:= DX *40
	IMUL	DX
	PUSH	AX
	MOV	AX,40			;STRINY:= STRINY *40
	IMUL	STRINY
	MOV	WORD PTR LIN546+2,AX	;SELF-MODIFING CODE
	MOV	AX,40			;DAGINY:= DAGINY *40
	IMUL	DAGINY
	MOV	WORD PTR LIN556+2,AX	;IMMEDIATES ARE MUCH FASTER
	POP	DX
	MOV	AX,STRINX
	MOV	WORD PTR LIN545+2,AX
	MOV	AX,STRCNT
	MOV	WORD PTR LIN547+2,AX
	MOV	AX,DAGINX
	MOV	WORD PTR LIN555+2,AX
	MOV	AX,DAGCNT
	MOV	WORD PTR LIN557+2,AX
	JMP	SHORT LIN505		;FLUSH INSTRUCTION CACHE
LIN505:
	MOV	BP,SI			;USE BP FOR LOOP COUNTER (LONG DISTANCE)
	MOV	BH,DOTDASH
	MOV	BL,BYTE PTR COLORX2	;GET ADJUSTED COLOR

	ALIGN	4		;TIME-CRITICAL LOOP...
LIN510:	ROL	BH,1		;ARE WE BETWEEN DOTS?
	JNC	LIN540		;SKIP PIXEL PLOT IF SO

	MOV	SI,CX		;GET X POSITION WITHIN BYTE
	SHL	SI,1		;FORM INDEX INTO TABLE
	SHL	SI,1		;IF MODE 5: TIMES 16
LIN514:	SHL	SI,1
LIN515:	SHL	SI,1		;IF MODE 6: TIMES 8
	ADD	SI,BX		;ADD IN ADJUSTED COLOR
	AND	SI,3FH
LIN516:	MOV	AX,CTAB5[SI]	;GET PIXEL COLOR AND MASK

	MOV	SI,CX		;IF MODE 5: SI:= X / 4
	SHR	SI,1
LIN517:	SHR	SI,1
LIN518:	MOV	AX,AX		;IF MODE 6: SI:= X / 8
	ADD	SI,DX		;ADD Y (*40) TO X
	TEST	DL,08H		;IS Y (*40) ODD OR EVEN?
	JE	LIN520		;JUMP IF EVEN
	ADD	SI,2000H-40	;USE INTERLACED BANK
LIN520:
	AND	AL,ES:[SI]	;GET OLD BYTE AND REMOVE PIXEL (UNLESS XOR)
	XOR	AL,AH		;INSERT NEW COLOR (OR DO XOR)
	MOV	ES:[SI],AL	;PUT UPDATED BYTE BACK
LIN540:
	TEST	DI,DI		;STRAIGHT MOVE?
	JGE	LIN550		;DO DIAGONAL IF NOT
LIN545:	ADD	CX,01234H	;SET UP X AND Y INCREMENTS
LIN546:	ADD	DX,01234H	; FOR STRAIGHT MOVE
LIN547:	ADD	DI,01234H	;ADJUST DECISION VARIABLE
	DEC	BP		;DECREMENT PIXEL COUNT
	JNE	LIN510		;NEXT PIXEL
	JMP	SHORT LIN590	;EXIT
LIN550:
LIN555:	ADD	CX,01234H	;SET X AND Y INCREMENTS
LIN556:	ADD	DX,01234H	; FOR DIAGONAL MOVE
LIN557:	ADD	DI,01234H	;ADJUST DECISION VARIABLE
	DEC	BP		;DECREMENT PIXEL COUNT
	JNE	LIN510		;NEXT PIXEL
LIN590:
	RET
	PAGE

;LINE DRAW FOR CGA MODE 6 (640x200x2)

	EVEN
;PIXEL COLOR AND MASK TABLE
; LOW BYTE = MASK,   HIGH BYTE = COLOR
;                 - NORMAL -       - XOR -
;		 OFF     ON      OFF      ON
CTAB6	DW	0007FH, 0807FH, 000FFH, 080FFH	;MSB = LEFT MOST PIXEL IN BYTE
	DW	000BFH, 040BFH, 000FFH, 040FFH
	DW	000DFH, 020DFH, 000FFH, 020FFH
	DW	000EFH, 010EFH, 000FFH, 010FFH
	DW	000F7H, 008F7H, 000FFH, 008FFH
	DW	000FBH, 004FBH, 000FFH, 004FFH
	DW	000FDH, 002FDH, 000FFH, 002FFH
	DW	000FEH, 001FEH, 000FFH, 001FFH	;LSB

LIN600:
LIN601:	MOV	AX,AX			;STUFF THIS 2-BYTE NOP INTO THE CODE
	MOV	AX,WORD PTR LIN601
	MOV	WORD PTR LIN515,AX
	MOV	AX,OFFSET CTAB6		;MODIFY CODE FOR MODE 6 TABLE
	MOV	WORD PTR LIN516+2,AX
	MOV	AX,WORD PTR LIN517	;STUFF "SHR SI,1" INTO THE CODE
	MOV	WORD PTR LIN518,AX
	JMP	LIN504
	PAGE

;LINE DRAW FOR MODE 13H (320x200x256)
; DRAWS FROM BOTH ENDS FOR MAXIMUM SPEED. THIS CAN DRAW THE PIXEL IN THE
; CENTER TWICE AND WOULD BE A PROBLEM IF THERE WERE AN XOR MODE.
;
;INPUTS:                   CX:= (SI+1)/2                 REGISTER USAGE:
; CX START X               SI:= STARTY *320 + STARTX      AX STRINC
; DX START Y               DX:= DI                        BL COLOR
; SI PIXEL COUNT           DI:= ENDY *320 + ENDX          BH DOTS
; DI DECISION VARIABLE     BP:= DAGINY *320 + DAGINX      CX PIXEL
; STRINX                   LIN1330+2:= STRCNT             DX DECISION
; STRINY                   LIN1360+2:= DAGCNT             SI INDEX 
; STRCNT                   AX:= STRINY *320 + STRINX      DI INDEX 
; DAGINX                   BX:= COLOR                     BP DIGINC
; DAGINY                   DS:= 0A000H                    DS 0A000H
; DAGCNT

LIN1300:XCHG	CX,SI		;CX:= (SI+1) /2
	INC	CX		;USE HALF THE PIXEL COUNT CUZ WE'RE DRAWING
	SHR	CX,1		; FROM BOTH ENDS
	MOV	AX,320		;SI:= STARTY *320 + STARTX
	IMUL	DX		;COMBINE X AND Y PORTIONS INTO ONE WORD
	ADD	SI,AX
	PUSH	DI		;DX:= DI
	MOV	AX,320		;DI:= ENDY *320 + ENDX
	IMUL	ENDY
	ADD	AX,ENDX
	MOV	DI,AX
	MOV	AX,320		;BP:= DAGINY *320 + DAGINX
	IMUL	DAGINY
	ADD	AX,DAGINX
	MOV	BP,AX
	MOV	AX,STRCNT		;LIN1330+2:= STRCNT
	MOV	WORD PTR LIN1330+2,AX	;SELF-MODIFYING CODE
	MOV	AX,DAGCNT		;LIN1360+2:= DAGCNT
	MOV	WORD PTR LIN1360+2,AX	;IMMEDIATES ARE MUCH FASTER
	JMP	SHORT LIN1305 		;FLUSH INSTRUCTION CACHE
LIN1305:MOV	AX,320			;AX:= STRINY *320 + STRINX
	IMUL	STRINY
	ADD	AX,STRINX
	MOV	BX,WORD PTR COLOR	;BL=COLOR, BH=DOTDASH
	MOV	DX,0A000H		;DS:=A000; POINT TO DISPLAY MEMORY
	MOV	DS,DX
	POP	DX		;DX:= DI
	TEST	DX,DX		;SET STATUS FOR DECISION VARIABLE

	ALIGN	4		;TIME-CRITICAL LOOP...
LIN1310:ROL	BH,1		;ARE WE BETWEEN DOTS?  (SIGN FLAG NOT AFFECTED)
	JNC	LIN1320		;SKIP PIXEL PLOT IF SO
	MOV	[SI],BL		;WRITE PIXEL FROM START END
	MOV	[DI],BL		;WRITE PIXEL FROM OTHER END
LIN1320:
	JNS	LIN1350		;JUMP ON DECISION VARIABLE (DX) STATUS
	ADD	SI,AX		;FOR STRAIGHT MOVE...
	SUB	DI,AX		;ADJUST OFFSETS FROM BOTH ENDS
LIN1330:ADD	DX,01234H	;ADJUST DECISION VARIABLE
	LOOP	LIN1310		;NEXT PIXEL (DOES NOT AFFECT SIGN FLAG)
	JMP	SHORT LIN1390	;EXIT
LIN1350:
	ADD	SI,BP		;FOR DIAGONAL MOVE...
	SUB	DI,BP		;ADJUST OFFSETS FROM BOTH ENDS
LIN1360:ADD	DX,01234H	;ADJUST DECISION VARIABLE
	LOOP	LIN1310		;NEXT PIXEL (DOES NOT AFFECT SIGN FLAG)
LIN1390:
	MOV	DX,CS		;RESTORE DATA SEGMENT REGISTER
	MOV	DS,DX
	RET
	PAGE

;EGA AND VGA LINE DRAWING ROUTINE
; BRESENHAM'S ALGORITHM IS USED FOR DIAGONAL LINES.
; THIS CODE IS BASED ON CODE BY SUTTY AND BLAIR IN THEIR BOOK ADVANCED
; PROGRAMMER'S GUIDE TO THE EGA/VGA

VLINE	PROC	NEAR
	PUSH	DS		;SAVE DS
	MOV	AX,STARTX	;MAKE SURE THAT ENDX >= STARTX
	MOV	CX,ENDX
	CMP	CX,AX
	JGE	VLIN10
	MOV	BP,STARTY	;SWAP ENDX & STARTX AND SWAP ENDY & STARTY
	MOV	DX,ENDY
	MOV	STARTX,CX
	MOV	STARTY,DX
	MOV	ENDX,AX
	MOV	ENDY,BP

;COMPUTE THE OFFSET (FROM A0000H) AND SAVE IT ON THE STACK
VLIN10:	MOV	AX,LINSIZ
	MUL	WORD PTR STARTY

	MOV	BX,STARTX	;+ X/8
	MOV	CL,3
	SHR	BX,CL
	ADD	AX,BX
	PUSH	AX		;SAVE OFFSET ON STACK, LATER POP TO DI

;COMPUTE THE BIT MASK FOR THE FIRST PIXEL AND SAVE IT ON THE STACK
	MOV	CX,STARTX	;COMPUTE WHICH BIT (X MOD 8) TO MODIFY
	AND	CL,7		;  0  1  2  3  4  5  6  7
	MOV	BX,80H		;  1  0  0  0  0  0  0  0  ---->
	SHR	BX,CL
	PUSH	BX		;SAVE MASK ON STACK, LATER POP TO SI

	MOV	DX,03CEH	;SELECT "BIT MAP" REGISTER TO APPEAR AT
	MOV	AL,08H		; PORT 3CFH
	OUT	DX,AL
	INC	DX		;SET "BIT MAP" REGISTER WITH BIT MASK
	MOV	AL,BL
	OUT	DX,AL

;SET UP REGISTERS FOR CURRENT COLOR
	MOV	BL,COLOR	;GET CURRENT COLOR
	MOV	DX,03CEH	;SELECT "ROTATE/FUNCTION" REGISTER
	MOV	AL,3
	OUT	DX,AL
	INC	DX		;POINT TO PORT 3CFH
	SUB	AL,AL		;START WITH WRITE MODE
	TEST	BL,080H		;XOR MODE?
	JZ	VLIN20		;SKIP IF NOT
	MOV	AL,18H		;SELECT XOR MODE
VLIN20:	OUT	DX,AL		;SET THE MODE	

	DEC	DX		;SELECT "SET/RESET" REGISTER TO APPEAR AT
	SUB	AL,AL		; PORT 3CFH
	OUT	DX,AL
	INC	DX		;ENABLE BIT PLANES CORRESPONDING TO COLOR
	MOV	AL,BL		;GET CURRENT COLOR
	AND	AL,07FH		;MASK OUT XOR BIT
	OUT	DX,AL

	DEC	DX		;SELECT "ENABLE SET/RESET" REGISTER TO
	MOV	AL,1		; APPEAR AT PORT 3CFH
	OUT	DX,AL
	INC	DX		;ENABLE 4 BIT PLANES
	MOV	AL,0FH
	OUT	DX,AL

	MOV	DX,03C4H	;SELECT "MAP MASK" REGISTER TO APPEAR
	MOV	AL,2		; AT PORT 3CFH
	OUT	DX,AL
	INC	DX		;ENABLE 4 BIT PLANES FOR MEMORY WRITE
	MOV	AL,0FH
	OUT	DX,AL

;SET UP SEGMENT REGISTERS
	MOV	DX,0A000H	;POINT DATA SEGMENT TO BASE OF DISPLAY
	ADD	DX,SCRPAR	;ADD IN SCREEN SEGMENT
	MOV	DS,DX
	MOV	ES,DX		;ALSO POINT EXTRA SEGMENT HERE

;DETERMINE IF HORIZONTAL, VERTICAL OR DIAGONAL LINE
	MOV	BX,CS:LINSIZ	;NUMBER OF BYTES PER RASTER LINE
	MOV	SI,CS:ENDX	;COMPUTE DX	       SI
	SUB	SI,CS:STARTX
	MOV	DI,CS:ENDY	;COMPUTE DY	       DI
	SUB	DI,CS:STARTY
	JGE	VLIN30
	NEG	BX
	NEG	DI
VLIN30:
	OR	SI,SI		;JUMP ACCORDING TO TYPE OF LINE
	JZ	VLIN40
	OR	DI,DI
	JZ	VLIN50
	JMP	VLIN60

;DRAW A VERTICAL LINE
VLIN40:	MOV	CX,DI		;SET UP COUNTER WITH THE NUMBER OF PIXELS
	INC	CX		;ADD ONE FOR END POINT
	MOV	BP,BX		;BYTES PER RASTER LINE

	POP	SI		;FETCH MASK
	MOV	DX,03CEH	;SET MASK
	MOV	AL,08H
	OUT	DX,AL
	INC	DX
	MOV	AX,SI
	OUT	DX,AL
	POP	DI		;FETCH OFFSET

	MOV	BL,CS:DOTDASH
	CMP	BL,0FFH		;IS THIS A SOLID LINE?
	JNE	VLIN45		;JUMP IF NOT

	ALIGN	4
VLIN43:	XCHG	AL,[DI]		;LATCH OLD DATA & MODIFY (ENABLED BITS)
	ADD	DI,BP		;MOVE TO NEXT SCAN LINE
	LOOP	VLIN43
	JMP	VLIN80

	ALIGN	4
VLIN45:	ROL	BL,1		;ARE WE BETWEEN DOTS?
	JNC	VLIN47		;JUMP IF SO
	XCHG	AL,[DI]		;LATCH OLD DATA & MODIFY (ENABLED BITS)
VLIN47:	ADD	DI,BP		;MOVE TO NEXT SCAN LINE
	LOOP	VLIN45
	JMP	VLIN80

;DRAW A HORIZONTAL LINE
VLIN50:	MOV	CX,SI		;SET UP COUNTER WITH THE NUMBER OF PIXELS
	INC	CX		;ADD ONE FOR END POINT
	POP	SI		;FETCH MASK
	POP	DI		;FETCH OFFSET

;DRAW PIXELS FROM THE LEADING PARTIAL BYTE
	MOV	AX,CS:STARTX
	AND	AX,07H		;CHECK FOR PARTIAL BYTE
	JZ	VLIN54		;JUMP IF NONE
	MOV	BP,0FFH		;COMPUTE THE MASK
	PUSH	CX
	MOV	CX,AX
	SHR	BP,CL
	POP	CX
	ADD	CX,AX		;UPDATE PIXEL COUNTER
	SUB	CX,08H
	JGE	VLIN52		;MODIFY MASK IF ONLY ONE BYTE
	NEG	CX
	SHR	BP,CL
	SHL	BP,CL
	SUB	CX,CX		;RESTORE COUNTER
VLIN52:
	MOV	DX,03CEH	;SET THE MASK
	MOV	AL,08H
	OUT	DX,AL
	INC	DX
	MOV	AX,BP
	AND	AL,CS:DOTDASH	;DRAW DOTTED LINE
	OUT	DX,AL
	XCHG	AL,[DI]		;LATCH DATA AND WRITE NEW DATA
	INC	DI		;UPDATE OFFSET

;DRAW PIXELS FROM THE MIDDLE COMPLETE BYTES
VLIN54:	MOV	BP,CX		;CHECK IF ANY BYTES TO SET
	CMP	BP,8
	JL	VLIN56
	SHR	CX,1		;COMPUTE COUNT
	SHR	CX,1
	SHR	CX,1
	MOV	DX,03CEH	;SET THE MASK
	MOV	AL,08H
	OUT	DX,AL
	INC	DX
	MOV	AL,CS:DOTDASH	;DRAW DOTTED LINE
	OUT	DX,AL
	MOV	SI,DI		;POINT SI AT SCREEN
	CLD			;FILL COMPLETE BYTES
	cmp	al, 0FFh	;solid line?
	jne	vlin55		;jump if not
	test	cs:color,80h	;jump if XOR line
	jne	vlin55
	mov	ah, al		;do two bytes at once because it's twice as fast
	shr	cx, 1
	rep stosw		;also it is not necessary to latch any existing
	jnc	vlin56		; pixels because they are all overwritten
	 stosb			;handle odd number of bytes
	jmp	vlin56
vlin55:
	rep movsb		;es:[di++] <- ds:[si++];  cx--
				;beware: there is only an 8-bit latch in the VGA
				; don't attempt to latch a word.
;DRAW PIXELS FROM THE TRAILING PARTIAL BYTE
VLIN56:	AND	BP,07H
	JZ	VLIN59
	MOV	AX,0FFFFH	;COMPUTE MASK
	MOV	CX,BP
	SHR	AX,CL
	XOR	AH,0FFH		;SET THE MASK
	MOV	DX,03CEH
	MOV	AL,08H
	OUT	DX,AL
	INC	DX
	MOV	AL,AH
	AND	AL,CS:DOTDASH	;DRAW DOTTED LINE
	OUT	DX,AL
	XCHG	AL,[DI]		;LATCH DATA AND SET NEW DATA
VLIN59:	JMP	VLIN80

;----------------------------------------------------------------------
;GENERATE A DIAGONAL LINE
;
; OCTANTS:
;	 ------------+------------
;	            /|\
;	    3     /  |  \     0
;	        /    |    \
;	      /   2  |  1   \
;	             |

;DETERMINE WHICH OCTANT THE LINE IS IN
VLIN60:	CMP	SI,DI		;IS DY > DX
	JLE	VLIN70		;BRANCH IF SO -- OCTANTS 1 AND 2

;COMPUTE CONSTANTS FOR OCTANT 0 AND 3
	MOV	CX,SI		;SET COUNTER TO DX+1
	INC	CX
	SAL	DI,1		;D1 = DY*2		DI
	MOV	BP,DI		;D  = DY*2-DX		BP
	SUB	BP,SI
	NEG	SI		;D2 = DY*2-DX-DX	SI
	ADD	SI,BP

	MOV WORD PTR CS:VLIN63+2,DI	;SAVE D1
	MOV WORD PTR CS:VLIN65+2,DI	;SAVE D1
	MOV WORD PTR CS:VLIN67+2,SI	;SAVE D2
	JMP	SHORT VLIN61	;FLUSH INSTRUCTION CACHE
VLIN61:	MOV	DX,03CEH	;SELECT BIT MASK REGISTER
	MOV	AL,08H
	OUT	DX,AL
	INC	DX
	POP	AX		;FETCH MASK
	POP	DI		;FETCH ADDRESS

;DRAW LINE IN OCTANTS 0 AND 3
	MOV	SI,BX		;BYTES PER RASTER LINE
	MOV	BL,CS:DOTDASH
	XCHG	AH,AL		;BIT MASK IN AH, AL = 0
	TEST	BP,BP		;TEST DECISION VARIABLE

	ALIGN	4
VLIN62:	JNS	VLIN66		;BRANCH BASED ON DECISION VARIABLE
	OR	AL,AH		;COMBINE BITS
	ROR	AH,1		;UPDATE MASK
	JC	VLIN64
VLIN63:	ADD	BP,01234H	;IF D < 0 THEN D = D + D1
	LOOP	VLIN62

	AND	AL,BL		;DRAW DOTTED LINE
	OUT	DX,AL		;OUTPUT FINAL MASK
	XCHG	AL,[DI]		;LATCH OLD DATA & MODIFY (ENABLED BITS)
	JMP	VLIN80

	ALIGN	4
VLIN64:	AND	AL,BL		;DRAW DOTTED LINE
	OUT	DX,AL		;ENABLE A BIT IN A BYTE
	XCHG	AL,[DI]		;LATCH OLD DATA & MODIFY (ENABLED BITS)
	XOR	AL,AL		;CLEAR AL
	INC	DI		;UPDATE BYTE ADDRESS
VLIN65:	ADD	BP,01234H	;IF D < 0 THEN D = D + D1
	LOOP	VLIN62
	JMP	VLIN80

	ALIGN	4
VLIN66:	OR	AL,AH		;COMBINE BITS
	AND	AL,BL		;DRAW DOTTED LINE
	OUT	DX,AL		;ENABLE A BIT IN A BYTE
	XCHG	AL,[DI]		;LATCH OLD DATA & MODIFY (ENABLED BITS)
	XOR	AL,AL		;CLEAR AL
	ROR	AH,1		;UPDATE MASK
	ADC	DI,SI		;MOVE TO NEXT SCAN LINE AND MAYBE NEXT BYTE
VLIN67:	ADD	BP,01234H	;IF D >= 0 THEN D = D + D2
	LOOP	VLIN62
	JMP	VLIN80

;COMPUTE CONSTANTS FOR OCTANT 1 AND 2
VLIN70:	MOV	CX,DI		;SET COUNTER TO DY+1
	INC	CX
	SAL	SI,1		;D1 = DX * 2
	MOV	BP,SI		;D  = DX * 2 - DY
	SUB	BP,DI
	NEG	DI		;D2 = -DY + DX * 2 - DY
	ADD	DI,BP

	MOV WORD PTR CS:VLIN75+2,SI	;SAVE D1
	MOV WORD PTR CS:VLIN77+2,DI	;SAVE D2
	JMP	SHORT VLIN71	;FLUSH INSTRUCTION CACHE
VLIN71:	MOV	DX,03CEH	;SELECT BIT MASK REGISTER
	MOV	AL,08H
	OUT	DX,AL
	INC	DX
	POP	AX		;FETCH MASK
	POP	DI		;FETCH ADDRESS
	OUT	DX,AL		;ENABLE A BIT IN A BYTE

;DRAW A LINE IN OCTANTS 1 AND 2
	MOV	SI,BX		;BYTES PER RASTER LINE
	MOV	BL,CS:DOTDASH
	TEST	BP,BP
	ALIGN	4
VLIN72:	ROL	BL,1		;ARE WE BETWEEN DOTS?
	JNC	VLIN74		;JUMP IF SO
	XCHG	AH,[DI]		;LATCH OLD DATA & MODIFY (ENABLED BITS)
VLIN74:	JNS	VLIN76
	ADD	DI,SI		;UPDATE OFFSET (Y = Y+1)
VLIN75:	ADD	BP,01234H	;IF D < 0 THEN D = D + D1
	LOOP	VLIN72
	JMP	SHORT VLIN80

	ALIGN	4
VLIN76:	ROR	AL,1		;UPDATE MASK (X = X+1)
	ADC	DI,SI		;UPDATE OFFSET (Y = Y+1)
	OUT	DX,AL		;ENABLE NEXT BIT WITHIN A BYTE
VLIN77:	ADD	BP,01234H	;IF D >= 0 THEN D = D + D2
	LOOP	VLIN72

;RESTORE PLANE ENABLE AND BIT MASK REGISTERS
VLIN80:	MOV	DX,03CEH	;ENABLE ALL 8 BITS IN A BYTE FOR WRITE
	MOV	AL,08H
	OUT	DX,AL
	INC	DX
	MOV	AL,0FFH
	OUT	DX,AL

	DEC	DX		;TURN OFF XOR MODE
	MOV	AL,3
	OUT	DX,AL
	INC	DX		;POINT TO PORT 3CFH
	SUB	AL,AL		;SET WRITE MODE
	OUT	DX,AL		;SET THE MODE	

	DEC	DX		;DISABLE SET/RESET FUNCTION
	MOV	AL,1
	OUT	DX,AL
	INC	DX
	XOR	AX,AX
	OUT	DX,AL
	POP	DS		;RESTORE DS
	RET

VLINE	ENDP
	ENDIF
