{
****************************************************************
****************************************************************
**                                                            **
**          FLISP,  a LISP interpreter for FreeDOS            **
**                                                            **
**          Copyright (C) 2005 by Francesco Zamblera          **
**           under the GNU General Public License             **
**                                                            **
**                       Version 1.0                          **
**                                                            **
****************************************************************
****************************************************************

    This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

    This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    vilnergoy@yahoo.it
}


{---------------------------------------}
PROGRAM F_Lisp_1_0;

USES Header,
     Utils,
     Dos,
     Crt,
     Scanner1;

{---------------------------------------}

PROCEDURE Eval (Code: PCellList; var Result: Cell); forward;

PROCEDURE setq  (Code: PCellList; var Result: Cell);
var p: PVarTable;
    Value: Cell;
    varname: string;
    cmdo : PVarTable;
begin
 if code^.data.flag <> _id
    then error ('setq: Variable expected')
    else begin
          varname := code^.data.varname;
          FindVar (varname,p);
          code := code^.next;
          eval(Code,Value);
          if exit_code then
             begin
              if p = nil then pushvar(GlobalVar, VarName, Value, cmdo)
                         else p^.value := Value;
              Result := Value
            end
         end
end;

{---------------------------------------}
PROCEDURE Compute_int (var Result: cell; term: cell; op: opcodes);
begin
 case op of
      _add: result.int := result.int + term.int;
      _mul: result.int := result.int * term.int;
      _sub: result.int := result.int - term.int;
      _div: if term.int <> 0
               then result.int := result.int div term.int
               else error ('Zero division')
      end
end;

{---------------------------------------}
FUNCTION min (i,j: integer): integer;
begin
 if i < j then min := i else min := j
end;

{---------------------------------------}
PROCEDURE Simplify (var num, denom: integer);
var i,k: integer;
begin
 k := abs(min(num,denom));
 for i := 2 to k do
     while ((num mod i) = 0) and ((denom mod i) = 0)
          do begin
              num := num div i;
              denom := denom div i
             end
end;

{---------------------------------------}
PROCEDURE compute_ratio (var Result: cell; term: cell; op: opcodes);
var num1, num2, denom1, denom2, denom3: integer;
begin
 case result.flag of
      _int: begin
             num1 := result.int;
             denom1 := 1;
             result.flag := _ratio
            end;
      _ratio: begin
               num1 := result.num; denom1 := result.denom
              end
     end;
 case term.flag of
       _int: begin
              num2 := term.int; denom2 := 1
             end;
       _ratio: begin
                num2 := term.num; denom2 := term.denom
               end
      end;
 case op of
      _add: begin
             result.denom := denom1 * denom2;
             result.num := num1 * denom2 + denom1 * num2
            end;
      _sub: begin
             result.denom := denom1 * denom2;
             result.num := num1 * denom2 - denom1 * num2
            end;
      _mul: begin
             result.num := num1 * num2;
             result.denom := denom1 * denom2
            end;
      _div: begin
             result.denom := denom1 * num2;
             result.num := num1 * denom2
            end
      end;
 simplify(result.num, result.denom);
 if result.denom = 1
    then begin
          num1 := result.num;
          result.flag := _int;
          result.int := num1
         end
end;

{---------------------------------------}
PROCEDURE compute_float (var Result: cell; term: cell; op: opcodes);
var term1, term2: real;
begin
 case result.flag of
      _int: begin
             term1 := result.int;
             result.flag := _float
            end;
      _ratio: begin
               term1 := result.num / result.denom;
               result.flag := _float
              end;
      _float: term1 := result.float
     end;
 case term.flag of
      _int: term2 := term.int;
      _ratio: term2 := term.num / term.denom;
      _float: term2 := term.float
     end;
 case op of
      _add: result.float := term1 + term2;
      _sub: result.float := term1 - term2;
      _mul: result.float := term1 * term2;
      _div: if term2 <> 0
               then result.float := term1 / term2
               else error ('Zero division')
     end
end;

{---------------------------------------}
PROCEDURE compute_next (var Result: cell; term: cell; op: opcodes);
var num1, num2, denom1, denom2: integer;
    float1, float2: real;
begin
 if (result.flag = _float) or (term.flag = _float)
    then compute_float (Result, Term, op)
    else if (result.flag = _ratio) or (term.flag = _ratio)
            then compute_ratio (Result, Term, Op)
            else  compute_int (Result, Term, Op)
end;

{---------------------------------------}
PROCEDURE compute (Code: PCellList; var Result: Cell; Op: OpCodes);
var total, args: integer;
    term: cell;
begin
 total := 0; args := 0;
 if (code = nil) or (code^.next = nil)
    then expected ('Arythmetic function: At least two arguments');
 while (Code <> nil) and exit_code
       do begin
           args := args + 1;
           eval (Code,term);
           if exit_code
              then begin
                    if not numeric (term)
                       then expected ('Numeric type for '+ KWList[ord(op)+1])
                       else begin
                             if args = 1
                                then Result := term
                                else compute_next (Result,term,op);
                             code := code^.next
                            end
                   end
          end
end;

{---------------------------------------}
PROCEDURE do_sqrt (Code: PCellList; var Result: Cell);
var n: real;
begin
 eval(Code,Result);
 if not numeric(result)
    then expected ('Function SQRT: Numeric operand')
    else begin
          case result.flag of
               _int: n := result.int;
               _float: n := result.float;
               _ratio: n := result.num / result.denom
              end;
          result.flag := _float;
          result.float := sqrt(n)
         end
end;

{---------------------------------------}
PROCEDURE do_sin (Code: PCellList; var Result: Cell);
var n: real;
begin
 eval(Code,Result);
 if not numeric(result)
    then expected ('Function SIN: Numeric operand')
    else begin
          case result.flag of
               _int: n := result.int;
               _float: n := result.float;
               _ratio: n := result.num / result.denom
              end;
          result.flag := _float;
          result.float := sin(n)
         end
end;

{---------------------------------------}
PROCEDURE do_cos (Code: PCellList; var Result: Cell);
var n: real;
begin
 eval(Code,Result);
 if not numeric(result)
    then expected ('Function COS: Numeric operand')
    else begin
          case result.flag of
               _int: n := result.int;
               _float: n := result.float;
               _ratio: n := result.num / result.denom
              end;
          result.flag := _float;
          result.float := cos(n)
         end
end;

{---------------------------------------}
FUNCTION WriteType (flag: SymType): string;
begin
 WriteType := TypeNames [ord(flag) + 1]
end;



{---------------------------------------}
PROCEDURE X_to_float (res: Cell; var Result: Cell);
begin
 Result.flag := _float;
 case res.flag of
      _int: result.float := res.int;
      _ratio: result.float := res.num / res.denom;
      _float: result.float := res.float
      else error ('Cannot coerce ' + writetype(res.flag) + ' into float')
      end
end;


{---------------------------------------}
PROCEDURE do_exp (Code: PCellList; var Result: Cell);
var res: cell;
begin
 if code = nil
    then expected ('Function EXP: Parameters')
    else begin
          eval(Code,Result);
          if not (result.flag in [_int, _float, _ratio])
                then Expected ('Function EXP: Numeric Type')
                else begin
                      X_to_float (code^.data,res);
                      result.flag := _float;
                      result.float := exp (res.float)
                     end
         end
end;

{---------------------------------------}
PROCEDURE do_random (Code: PCellList; var Result: Cell);
begin
 eval(Code,Result);
 if result.flag <> _int then Expected ('Function RANDOM: FIXNUM')
                        else result.int := random(result.int)
end;

{---------------------------------------}
PROCEDURE quote (Code: PCellList; var Result: Cell);
begin
 if code = nil
    then error ('Function QUOTE needs an operand')
    else if code^.next <> nil
             then error ('Too many arguments for QUOTE')
             else Result := code^.data
end;

{---------------------------------------}
PROCEDURE str_eq (Code: PCellList; var Result: Cell);
begin
 if (code = nil) or (code^.next = nil) or (code^.next^.next <> nil)
    then expected ('Function STRING=: Two parameters')
    else if (code^.data.flag <> _string)
            or (code^.next^.data.flag <> _string)
            then expected ('Function STRING=: String parameters')
            else begin
                  result.flag := _spec;
                  result.t := code^.data.str = code^.next^.data.str
                 end
end;

{---------------------------------------}
PROCEDURE str_ne (Code: PCellList; var Result: Cell);
begin
 if (code = nil) or (code^.next = nil) or (code^.next^.next <> nil)
    then expected ('Function STRING=: Two parameters')
    else if (code^.data.flag <> _string)
            or (code^.next^.data.flag <> _string)
            then expected ('Function STRING=: String parameters')
            else begin
                  result.flag := _spec;
                  result.t := not (code^.data.str = code^.next^.data.str)
                 end
end;

{---------------------------------------}
PROCEDURE str_lt (Code: PCellList; var Result: Cell);
begin
 if (code = nil) or (code^.next = nil) or (code^.next^.next <> nil)
    then expected ('Function STRING=: Two parameters')
    else if (code^.data.flag <> _string)
            or (code^.next^.data.flag <> _string)
            then expected ('Function STRING=: String parameters')
            else begin
                  result.flag := _spec;
                  result.t := code^.data.str < code^.next^.data.str
                 end
end;

{---------------------------------------}
PROCEDURE str_gt (Code: PCellList; var Result: Cell);
begin
 if (code = nil) or (code^.next = nil) or (code^.next^.next <> nil)
    then expected ('Function STRING=: Two parameters')
    else if (code^.data.flag <> _string)
            or (code^.next^.data.flag <> _string)
            then expected ('Function STRING=: String parameters')
            else begin
                  result.flag := _spec;
                  result.t := code^.data.str > code^.next^.data.str
                 end
end;

{---------------------------------------}
PROCEDURE Cdr (Code: PCellList; var Result: Cell);
var PartialRes: Cell;
begin
 Result.flag := _null;
 if code = nil
    then error ('CDR needs a parameter')
    else begin
          eval(code,PartialRes);
          if PartialRes.flag <> _list
             then error ('Parameter of CDR must be a list')
             else begin
                   if (PartialRes.list = nil)
                      then begin
                            Result.flag := _list;
                            Result.list := nil
                           end
                       else begin
                             Result.flag := _list;
                             Result.list := PartialRes.list^.next
                            end
                  end
         end
end;

{---------------------------------------}
PROCEDURE Car (Code: PCellList; var Result: Cell);
var PartialRes: Cell;
begin
 Result.flag := _null;
 if code = nil
    then error ('CAR needs a parameter')
    else begin
          eval(code,PartialRes);
          if PartialRes.flag <> _list
                     then error ('Parameter of CAR must be a list')
                     else begin
                           if PartialRes.list = nil
                              then begin
                                    Result.flag := _list;
                                    Result.list := nil
                                   end
                              else Result := PartialRes.list^.data
                          end
                 end
end;

PROCEDURE print (Datum: cell); forward;
{---------------------------------------}
PROCEDURE Cons (Code: PCellList; var Result: Cell);
var Partial1, Partial2: Cell;
begin
 Result.flag := _null;
 if (code = nil) or (code^.next = nil)
    then error ('CONS needs two parameters')
    else begin
          eval(code,Partial1);
          eval(code^.next,Partial2);
          if Partial2.flag <> _list
             then error ('Second parameter of CONS must be list')
             else begin
                   Result.flag := _list;
                   new(Result.list);
                   result.list^.data := Partial1;
                   result.list^.next := partial2.list
                  end
         end
end;

{---------------------------------------}
PROCEDURE AddToList (var p: PCellList; value: Cell);
var q: PCellList;
begin
 if p = nil then begin
                  new(p);
                  p^.next := nil;
                  p^.data := value
                 end
            else begin
                  q := p;
                  while q^.next <> nil do q := q^.next;
                  new(q^.next);
                  q^.next^.next := nil;
                  q^.next^.data := value
                 end
end;

{---------------------------------------}
PROCEDURE SpliceToList (var p: PCellList; q: PCellList);
begin
 if q^.data.flag <> _list
    then error('Argunemt of COMMA-AT must be a list');
 q := q^.data.list;
 while q <> nil do begin
                    AddToList(p,q^.data);
                    q := q^.next
                  end
end;

{---------------------------------------}
PROCEDURE BackQuote (Code: PCellList; var Result: Cell; Deref: boolean);
var p: PCellList;
    c: cell;
    Deref1: boolean;
begin
 case code^.data.flag of
      _id: if deref then eval(code, Result)
                    else Result := code^.data;
      _list: if code^.data.list = nil
                then result := code^.data
                else begin
                      p := nil;
                      code := code^.data.list;
                      CheckDeref(code,Deref1);
                      while code <> nil
                            do begin
                                BackQuote
                                    (Code, c, deref or Deref1);
                                if (c.flag = _list)
                                   and (c.list <> nil)
                                   and (c.list^.data.flag = _op)
                                   and (c.list^.data.opcode = _at)
                                   then SpliceToList
                                           (p,c.list^.next){^.data.list)}
                                   else AddToList (p,c);
                                code := code^.next
                               end;
                      if (p^.data.flag = _op)
                         and (p^.data.opcode = _comma)
                         then begin
                               p := p^.next;
                               result := p^.data
                              end
                         else begin
                               Result.flag := _list;
                               Result.list := p
                              end
                     end
     else Result := code^.data
    end
end;

{---------------------------------------}
PROCEDURE make_list (code: PCellList; var Result: Cell);
var Res: Cell;
begin
 Result.flag := _list;
 Result.list := nil;
 while code <> nil do begin
                       eval(Code,Res);
                       AddToList (Result.list,res);
                       code := code^.next
                      end
end;

{---------------------------------------}
PROCEDURE BinaryRelations (Code: PCellList; var Result: Cell; Op: OpCodes);
var term1, term2: cell;
    res: boolean;
begin
 if (code = nil) or (code^.next = nil)
    then error ('Two parameters required for binary relations')
    else begin
          eval(code,term1);
          eval(code^.next,term2);
          case op of
               _eq, _ne, _lt, _gt, _le, _ge :
                    begin
                     if (not numeric (term1))
                        or not numeric (term2)
                        then expected ('Numeric type')
                        else begin
                              case Op of
                                   _eq: res := term1.int = term2.int;
                                   _ne: res := term1.int <> term2.int;
                                   _lt: res := term1.int < term2.int;
                                   _gt: res := term1.int > term2.int;
                                   _le: res := term1.int <= term2.int;
                                   _ge: res := term1.int >= term2.int
                                  end;
                              result.flag := _spec;
                              result.t := res
                             end
                    end;
               _eq_atom: begin
                          result.flag := _spec;
                          result.t := ((term1.flag = _id)
                                      and (term2.flag = _id)
                                      and (term1.varname = term2.varname))
                              or      ((term1.flag = _op)
                                      and (term2.flag = _op)
                                      and (term1.opcode = term2.opcode))
                              or      ((term1.flag = _spec)
                                      and (term2.flag = _spec)
                                      and (term1.t = term2.t));
                         end
            end
         end
end;

{---------------------------------------}
FUNCTION IsFalse (var C: Cell): boolean;
begin
 IsFalse := ( (c.flag = _spec) and not c.t )
            or
            ( (c.flag = _list) and (c.list = nil) )
end;

{---------------------------------------}
PROCEDURE eval_cond (Code: PCellList; var Result: Cell);
var condition: cell; found: boolean;
begin
  if (code = nil) or (code^.next = nil)
    then expected ('Function COND: parameters')
    else begin
          found := false;
          while (code <> nil) and (not found)
                do begin
                    if code^.data.flag <> _list
                       then expected ('Function COND: list')
                       else begin
                             eval(code^.data.list, condition);
                             found := not IsFalse (Condition);
                             if not found then code := code^.next
                            end
                   end;
          if found then eval(code^.data.list^.next,result)
                   else begin
                         result.flag := _spec;
                         result.t := false
                        end
         end
end;

{---------------------------------------}
PROCEDURE eval_if (Code: PCellList; var Result: Cell);
var Condition: Cell;
begin
 if (code = nil) or (code^.next = nil) or (code^.next^.next = nil)
    then error ('IF wants three parameters')
    else begin
          eval(Code, Condition);
          if IsFalse (Condition)
             then eval (code^.next^.next, Result)
             else eval (code^.next, Result)
         end
end;

{---------------------------------------}
PROCEDURE Func_Eval (Code: PCellList; var Result: Cell);
var PartialRes: Cell;
    p: PCellList;
begin
 if (code = nil)
    then error ('EVAL needs a list as parameter')
    else begin
          eval(code, PartialRes);
          new (p);
          p^.data := PartialRes;
          p^.next := nil;
          eval(p,Result)
         end
end;


{---------------------------------------}
PROCEDURE Read_Eval_Print; Forward;

{---------------------------------------}
PROCEDURE LoadFile (filename: string; var Result: Cell; EchoOn: boolean);
begin
 InputFile := InputFile - 1;
 if inputfile < - maxfiles
    then error ('Too many open files')
    else begin
          assign(Files[InputFile].filevar, filename);
          reset(Files[InputFile].filevar);
          GetChar;
          while (not eof (Files[InputFile].filevar)) and exit_code
                do Read_Eval_Print;
          if exit_code and EchoOn then writeln(Filename + ' Loaded');
          close(Files[InputFile].filevar);
          result.flag := _null;
         end;
    inputfile := inputfile + 1
end;

{---------------------------------------}
PROCEDURE Load (code: PCellList; var Result: Cell);
var filename: Cell;
begin
 if code = nil
    then error ('Missing parameter')
    else begin
          eval (code, filename);
          if (not exit_code) or (filename.flag <> _id)
             then expected ('Function LOAD: Filename')
             else LoadFile(filename.varname,Result,true)
         end;
 if Look = ^Z then look := ' ';
 result.flag := _spec;
 result.t := true
end;

{---------------------------------------}
PROCEDURE readline (Code: PCellList; var Result: Cell);
var ID: integer; res: cell;
begin
 Result.flag := _string;
 if code = nil then begin
                     GetChar;
                     readln (Result.str)
                    end
               else begin
                     eval (code,res);
                     if res.flag <> _file
                        then expected ('Function READ-LINE: File Ident')
                        else begin
                              GetFileID (res.descriptor, ID);
                              if not eof (files[ID].filevar)
                                 then begin
                                       readln (files[ID].filevar, Result.str);
                                       inc (files[ID].pos)
                                      end
                                 else begin
                                       result.flag := _spec;
                                       result.t := false
                                      end
                             end
                    end
end;

{---------------------------------------}
PROCEDURE eval_read (Code: PCellList; var Result: Cell);
var p: PCellList; res: cell; ID, cmdo: integer;
begin
 cmdo := InputFile;
 if code <> nil
    then begin
          eval (Code, res);
          if res.flag <> _file
             then expected ('Function READ: File identifier')
             else begin
                   GetFileID (res.descriptor, ID);
                   InputFile := ID
                  end
         end;
 if not eof (files[ID].filevar)
        then begin
                scan(p);
                inc (files[ID].pos);
                result := p^.data;
                InputFile := cmdo
             end
        else begin
               result.flag := _spec;
               result.t := false
             end
end;

{---------------------------------------}
PROCEDURE PutForm (n1,n2: integer;
                   d: real;
                   s: string;
                   ch: char;
                   t: SymType);
var to_stout: boolean;
begin
 to_stout:= OutputFile = 0;
 case t of
      _int: if to_stout then write(n1)
                        else write (Files[OutputFile].FileVar, n1);
      _char: if to_stout then write ('#\' + ch)
                         else write (Files[OutputFile].FileVar, '#\' + ch);
      _float: if to_stout then write(d)
                          else write (Files[OutputFile].FileVar,d);
      _ratio: if to_stout then write(n1, '/', n2)
                          else write (Files[OutputFile].FileVar,n1,'/',n2);
      _string:
              if to_stout then write(s)
                          else write (Files[OutputFile].FileVar, s)
      end
end;

{---------------------------------------}
PROCEDURE eval_write (Code: PCellList; var Result: Cell);
var p: PCellList; ID: integer; res1, res2: cell; cmdo: byte;
begin
 cmdo := OutputFile;
 if code <> nil
    then begin
          eval(code,res1);
          code := code^.next;
          if code <> nil
             then begin
                   if (code^.data.flag <> _id)
                      and (code^.data.varname <> ':STREAM')
                      then expected ('Function WRITE: Keyword :STREAM')
                      else begin
                            code := code^.next;
                            eval (Code, res2);
                            if res2.flag <> _file
                               then expected ('Function WRITE: File id')
                               else begin
                                     GetFileID (res2.descriptor, ID);
                                     OutputFile := ID;
                                    end
                           end
                  end;
          if exit_code then begin
                             Print (res1);
                             PutForm (1,1,1,NL,' ',_string)
                            end
         end
    else expected ('Function WRITE: Parameter');
 result.flag := _spec; result.t := true;
 OutputFile := cmdo
end;

{---------------------------------------}
PROCEDURE SetParameters (code: PCellList; var check, out: boolean);
begin
 check := false; out := false;
 while (code <> nil) and exit_code
       do begin
           if code^.data.flag <> _id
              then expected ('Function OPEN: Keyword')
              else begin
                    if code^.data.varname = ':DIRECTION'
                       then begin
                             code := code^.next;
                             if (code = nil) or (code^.data.flag <> _id)
                                then expected ('OPEN: Keyword')
                                else begin
                                      check := check or
                                            (code^.data.varname = ':PROBE');
                                      out := out or
                                          (code^.data.varname = ':OUTPUT');
                                      if not (check or out)
                                         then error ('Invalid keyword')
                                     end
                            end;
                    if code^.data.varname = ':IF-DOES-NOT-EXIST'
                               then begin
                                     code := code^.next;
                                     if (code = nil) or
                                        (code^.data.flag <> _id)
                                        then expected ('OPEN: KEYWORD')
                                        else begin
                                              check := check or
                                              (code^.data.varname
                                              = ':CREATE');
                                              if not check
                                                 then expected (':CREATE')
                                             end
                                    end
                   end;
           code := code^.next
          end
end;

{---------------------------------------}
PROCEDURE openfile (Code: PCellList; var Result: Cell);
var Descriptor: string;
    out,check,exists: boolean;
    ID: integer;
begin
 out := false; check := false; exists := false;
 if (code = nil) or (code^.data.flag <> _string)
    then expected ('Function OPEN: Filename')
    else begin
          NewFile (code^.data.str,ID);
          if exit_code
             then begin
                   assign (Files[Id].FileVar, Files[Id].Name);
                   if code^.next <> nil
                      then SetParameters (code^.next,check,out);
                   if out then Files [ID].dir := _output
                          else Files [ID].dir := _input;
                   Files[ID].pos := 0;
                   if exit_code
                      then begin
                            {$I-}
                              reset(Files[Id].FileVar);
                              exists := IOResult = 0;
                              if exists then close (Files[Id].FileVar);
                            {$I+}
                            if (not check) and (not exists) and (not out)
                              then error ('File not found')
                              else begin
                                    if out
                                       then begin
                                             if exists
                                                then append
                                                     (Files[Id].FileVar)
                                                else rewrite
                                                     (Files[Id].FileVar)
                                            end
                                       else reset (Files[Id].FileVar);
                                    if check and (not exists) and (not out)
                                       then begin
                                             result.flag := _spec;
                                             result.t := false
                                            end
                                       else begin
                                             result.flag := _file;
                                             result.descriptor :=
                                              files[id].descriptor
                                            end
                                   end
                          end
                  end
         end
end;

{---------------------------------------}
PROCEDURE closefile (Code: PCellList; var Result: Cell);
var res: cell;
begin
 if code = nil
    then expected ('Function CLOSE: File Identifier')
    else begin
          eval (Code, Res);
          if res.flag <> _file
             then expected ('Function CLOSE: File Identifier')
             else begin
                   GetFileID (res.descriptor, FileID);
                   close (Files[FileID].FileVar);
                   Files[FileID].IsOpen := false
                  end;
          Result.flag := _spec;
          Result.t := false
         end
end;

{---------------------------------------}
PROCEDURE writeline (Code: PCellList; var Result: Cell);
var res,res2: cell; i: integer;
begin
 if code = nil
    then expected ('Function WRITELINE: String')
    else begin
          eval (Code, Res);
          if res.flag <> _string
             then expected ('Function WRITELINE: String')
             else begin
                   code := code^.next;
                   if code = nil
                      then writeln(res.str)
                      else begin
                            eval (Code, Res2);
                            if res2.flag <> _file
                               then expected ('File identifier')
                               else begin
                                     GetFileID (res2.descriptor,i);
                                     writeln (files[i].filevar, res.str)
                                    end
                           end
                  end
         end;
 result.flag := _spec; result.t := false
end;

{---------------------------------------}
PROCEDURE AddToGlobal (var LocalVar: PVarTable);
var p: PVarTable;
begin
 if LocalVar<> nil then begin
                         p := LocalVar;
                         while p^.next <> nil do p := p^.next;
                         p^.next := GlobalVar;
                         GlobalVar := LocalVar
                        end
end;

{---------------------------------------}
PROCEDURE LetParams (LetList: PCellList; var Names, Values: PCellList);
var _name, _value: cell;
    l: PCellList;
begin
 names := nil;
 values := nil;
 while LetList <> nil
       do begin
           case LetList^.data.flag of
                _list: begin
                        l := LetList^.data.list;
                        if l^.data.flag <> _id
                           then expected ('Identifier')
                           else begin
                                 _name := l^.data;
                                 _value := l^.next^.data;
                                 if l^.next^.next <> nil
                                    then error ('Let: Too many parameters');
                                end
                       end;
                _id: begin
                      _value.flag := _null;
                      _name := LetList^.data
                     end
                else error
                      ('Let: Bad Formal Parameter (neither ATOM nor LIST)')
               end;
           AddToList(names, _name);
           AddToList(values,_value);
           LetList := LetList^.next
      end
end;

{---------------------------------------}
PROCEDURE InitParam (FormalList: PCellList; Values: PCellList;
          var n: integer; macro: Boolean; var LocalList: PLocalVar);
var PartialResult, last: Cell;
    Variables, VarPointer: PVarTable;
    rest: boolean;
    p: PCellList;
begin
 n := 0; Variables := nil; rest := false; LocalList := nil;
 while (FormalList <> nil) and (Values <> nil) and exit_code
    do begin
        if FormalList^.data.flag <> _id
           then error ('Ident expected in param list');
        if exit_code
          then begin
                if formallist^.data.varname = '&OPTIONAL'
                   then formallist := formallist^.next
                   else if formallist^.data.varname = '&REST'
                           then begin
                                 if formallist^.next = nil
                                    then expected ('Identifier')
                                    else begin
                                          formallist := formallist^.next ;
                                          if formallist^.next <> nil
                                             then error('Too many params')
                                             else begin
                                                   rest := true;
                                                   last := formallist^.data;
                                                   formallist := nil;
                                                  end
                                         end
                                end
                           else if macro
                                   then PartialResult := Values^.data
                                   else begin
                                         eval(Values, PartialResult);
                                         n := n + 1
                                       end;
                if not rest
                   then begin
                         PushVar(Variables, FormalList^.data.varname,
                         PartialResult,VarPointer);
                         AddToLocal (LocalList,VarPointer);
                         FormalList := FormalList^.next;
                         Values := Values^.next
                        end
               end
       end;
 if (FormalList <> nil)
    and ((FormalList^.data.varname ='&OPTIONAL')
        or (FormalList^.data.varname ='&REST'))
    then begin
          FormalList := FormalList^.next;
          PartialResult.flag := _list; PArtialResult.list := nil;
          PushVar(Variables, Formallist^.data.varname,
                             PartialResult,VarPointer);
          AddToLocal (LocalList,VarPointer);
          formallist := formallist^.next;
         end
    else  if (FormalList <> nil) and (Values = nil)
             then error ('Too few params');
 if (Values <> nil) and rest
    then begin
          p := nil;
          while values <> nil do begin
                                  if macro
                                   then PartialResult := Values^.data
                                   else eval(Values, PartialResult);
                                  AddToList (p,PartialResult);
                                  values := values^.next
                                 end;
          PartialResult.flag := _list; PartialResult.list := p;
          PushVar(Variables,last.varname,PartialResult,VarPointer);
          AddToLocal (LocalList, VarPointer)
         end
    else if (Values <> nil) and (FormalList = nil)
            then error ('Too many params');
 if exit_code then AddToGlobal(Variables)
end;


{---------------------------------------}
PROCEDURE let (Code: PCellList; var Result: Cell);

var names, values: PCellList;
    Params, i: Integer;
    cmdo: cell;
    LocalList: PLocalVar;
begin
 if (code = nil) or (code^.next = nil)
    or (code^.data.flag <> _list)
    then expected ('Function LET: List of formal parameters')
    else begin
          LetParams (Code^.data.list, Names, Values);
          InitParam (Names,Values,Params,false,LocalList);
          code := code^.next;
          while code <> nil do begin
                                eval (code, Result);
                                code := code^.next
                               end;
          DisposeAll (LocalList)
         end
end;


{---------------------------------------}
PROCEDURE let_star (Code: PCellList; var Result: Cell);

var names, values: PCellList;
q: PVarTable;
    Params, i: Integer;
    cmdo,res: cell;
    LocalList, LocalVar: PLocalVar;
begin
 if (code = nil) or (code^.next = nil)
    or (code^.data.flag <> _list)
    then expected ('Function LET*: List of formal parameters')
    else begin
          LetParams (Code^.data.list, Names, Values);
          LocalList := nil;
          while names <> nil
                do begin
                    eval (values,res);
                    {Nella LET normale e nelle funzioni, il parametro e
                    valutato dalla InitParam}
                    pushvar (GlobalVar,names^.data.varname,res,q);
                    new (LocalVar);
                    LocalVar^.p := q;
                    LocalVar^.next := LocalList;
                    LocalList := LocalVar;
                    names := names^.next;
                    values := values^.next
                   end;
          code := code^.next;
          while code <> nil do begin
                                eval (code, Result);
                                code := code^.next
                               end;
          DisposeAll (LocalList)
         end
end;



{---------------------------------------}
PROCEDURE eval_print (code: PCellList; var Result: Cell);
begin
 eval(code, Result);
 print(Result); PutForm(1,1,1,NL,' ',_string);
 result.flag := _null
end;

{---------------------------------------}
PROCEDURE princ (code: PCellList; var Result: Cell);
begin
 eval(code, Result);
 print(Result);
 result.flag := _null
end;


{---------------------------------------}
PROCEDURE dotimes(Code: PCellList; var Result: Cell);
var c,res: Cell;
    p: PVarTable;
    i,lim:integer;
    code1: PCellList;
begin
 result.flag := _spec; result.t := false;
 if (code^.data.flag<>_list)
    then error('First parameter of DOTIMES must be a list')
    else if code^.data.list = nil
            then error ('First parameter of DOTIMES cannot be NIL')
            else begin
                  if code^.data.list^.data.flag <> _id
                     then expected('Function DOTIMES: Variable identifier')
                     else begin
                           c.flag := _int; c.int := 0;
                           pushvar (Globalvar,
                                   code^.data.list^.data.varname,
                                   c,p);
                           eval(code^.data.list^.next,Res);
                           if res.flag <> _int
                              then expected('FIXNUM')
                              else begin
                                    lim := res.int - 1;
                                    code := code^.next;
                                    i := 0;
                                    repeat
                                     code1 := code;
                                     p^.value.int := i;
                                     while (code1 <> nil) and not break do
                                           begin
                                            eval(Code1,Res);
                                            code1 := code1^.next
                                           end;
                                    i := i + 1
                                   until break or (i > lim)
                                 end;
                           DisposeVar (p);
                          end
                 end;
 break := false
end;

{---------------------------------------}
PROCEDURE loop(Code: PCellList; var Result: Cell);
var code1: PCellList;
begin
 repeat
  code1 := code;
  while (code1 <> nil) and not break do
        begin
         eval(Code1,Result);
         code1 := code1^.next
        end
 until break;
 break := false
end;

{---------------------------------------}
PROCEDURE DoList (Code: PCellList; var Result: Cell);
var c,res: Cell;
    p: PVarTable;
    q,code1: PCellList;
begin
 result.flag := _spec; result.t := false; break := false;
 if (code^.data.flag<>_list)
    then error('First parameter of DOLIST must be a list')
    else if code^.data.list = nil
            then error ('First parameter of DOLIST cannot be NIL')
            else begin
                  if code^.data.list^.data.flag <> _id
                     then expected('Function DOLIST: Variable identifier')
                     else begin
                           c.flag := _list; c.list := nil;
                           pushvar (Globalvar,
                                   code^.data.list^.data.varname,
                                   c,p);
                           eval(code^.data.list^.next,Res);
                           if res.flag <> _list
                              then expected('LIST')
                              else begin
                                    code := code^.next;
                                    q := res.list;
                                    while (q <> nil) and not break
                                          do begin
                                              code1 := code;
                                              p^.value := q^.data;
                                              while (code1 <> nil)
                                                    and not break
                                                    do begin
                                                        eval(Code1,Res);
                                                        Code1 := Code1^.next
                                                       end;
                                              q := q^.next
                                             end
                                 end;
                           DisposeVar (p)
                          end
                 end;
 break := false
end;

{---------------------------------------}
PROCEDURE do_help (Code: PCellList; var Result: Cell);
begin
end;

{---------------------------------------}
PROCEDURE boundp (Code: PCellList; var Result: Cell);
var p: PVarTable;
    res: cell;
begin
 eval(Code,Res);
 if res.flag <> _id then Expected ('Function BOUNDP: Identifier')
                    else begin
                          findvar(res.varname,p);
                          result.flag := _spec;
                          result.t := (p <> nil) and
                                      (p^.value.flag <> _null)
                         end
end;

{---------------------------------------}
Procedure GetType (typename: symbol; var t: symtype);
var i: byte; found: boolean;
begin
 i := 0; found := false;
 while (not found) and (i <= Types)
       do begin
           inc(i);
           found := typename = typenames [i]
          end;
 if found then t := Symtype (i-1) else error ('Invalid type')
end;


{---------------------------------------}
PROCEDURE X_to_int (res: Cell; var Result: Cell);
begin
 Result.flag := _int;
 case res.flag of
      _int: result.int := res.int;
      _ratio: result.int := res.num div res.denom;
      _float: result.int := trunc (res.float);
      _char: result.int := ord(res.ch)
      else error ('Cannot coerce ' + writetype(res.flag) + ' into fixnum')
      end
end;


{---------------------------------------}
PROCEDURE int_to_char (res: Cell; var Result: Cell);
begin
 result.flag := _char;
 if res.flag <> _int
    then error ('Cannot coerce ' + WriteType (res.flag) + 'into char')
    else result.ch := chr (res.int)
end;

{---------------------------------------}
PROCEDURE X_to_ratio (res: Cell; var Result: Cell);
begin
 Result.flag := _ratio;
 case res.flag of
      _int: begin
             result.num := res.int;
             result.denom := 1
            end;
      _ratio: result := res
      else error ('Cannot coerce ' + writetype(res.flag) + ' into rational')
      end
end;

{---------------------------------------}
PROCEDURE list_to_string (res: Cell; var Result: Cell);
var p: pcelllist; s: string;
begin
 Result.flag := _string;
 if res.flag <> _list
    then expected ('List')
    else begin
          s := '';
          p := res.list;
          while (p <> nil)
                do begin
                    if p^.data.flag <> _char
                       then expected ('COERCE: Char')
                       else s := s + p^.data.ch;
                    p := p^.next
                   end;
          result.str := s
        end
end;

{---------------------------------------}
PROCEDURE string_to_list (res: Cell; var Result: Cell);
var p: pcelllist; s: string; c: cell; i: byte;
begin
 if res.flag <> _string
    then error (Writetype(res.flag) + 'cannot be coerced to list')
    else begin
          p := nil; c.flag := _char;
          for i := 1 to length(res.str) do
              begin
               c.ch := res.str[i];
               AddToList(p,c);
              end
         end;
 result.flag := _list;
 result.list := p
end;

{---------------------------------------}
PROCEDURE Coerce (Code: PCellList; var Result: Cell);
var res,res2: cell; t: SymType;
begin
 eval(Code,Res);
 if code^.next = nil
    then error ('Function COERCE: missing type parameter')
    else begin
          code := code^.next;
          eval(code,res2);
          if res2.flag <>_id
             then expected ('Function COERCE: type identifier')
             else begin
                   gettype(res2.varname,t);
                   case t of
                        _int: X_to_int (res,result);
                        _float: X_to_float (res,result);
                        _ratio: X_to_ratio (res,result);
                        _string: list_to_string (res,result);
                        _char: int_to_char (res,result);
                        _list: string_to_list (res, result)
                        else error ('Function COERCE: invalid type')
                       end
                  end
         end
end;

{---------------------------------------}
PROCEDURE spelling (Code: PCellList; var Result: Cell);
var res: cell;
begin
 if (code = nil)
    then expected ('Function SYMBOL-NAME: parameter')
    else begin
          eval (code,res);
          if res.flag <> _id
             then expected ('Function SYMBOL-NAME: identifier')
             else begin
                   result.flag := _string;
                   result.str := res.varname
                  end
         end
end;

{---------------------------------------}
PROCEDURE defun (Code: PCellList; var Result: Cell; IsMacro: Boolean);
var p: PFuncTable; res: Cell;
begin
 if (code = nil) or (code^.next = nil) or (code^.next^.next = nil)
     then error ('DEFUN needs three parameters')
     else begin
           if code^.data.flag <> _id
              then error ('First parameter of DEFUN ' +
                         'must be the function identifier')
              else begin
                    new(p);
                    p^.FuncName := code^.data.varname;
                    result := code^.data;
                    code := code^.next;
                    if code^.data.flag <> _list
                       then error ('Second parameter of DEFUN ' +
                                  'must be parameter list')
                       else begin
                             p^.FuncParam := code^.data;
                             code := code^.next;
                             if code^.data.flag <> _list
                                then error ('Third parameter of DEFUN '+
                                           'must be a list')
                                else begin
                                      p^.FuncBody := code^.data;
                                      p^.IsMacro := IsMacro;
                                      pushfunc(p)
                                     end
                            end
                   end
          end
end;

{---------------------------------------}
PROCEDURE lambda (Code: PCellList; var Result: Cell);
var res: cell; p: PCellList;
begin
 result.flag := _id; result.varname := 'ANONYM:';
 new(p); p^.data := result; p^.next := code;
 code := p;
 defun (Code, Res, false)
end;

{---------------------------------------}
PROCEDURE MakeSymbol (Code: PCellList; var Result: Cell);
var res: cell; i: byte;
begin
 if (code = nil)
    then expected ('Function SYMBOL-NAME: parameter')
    else begin
          eval (code,res);
          if res.flag <> _string
             then expected ('Function SYMBOL-NAME: string')
             else begin
                   result.flag := _id;
                   result.varname := res.str;
                   for i := 1 to length (res.str)
                       do begin
                           if result.varname[i] = ' '
                              then result.varname [i] := '-'
                              else result.varname[i] :=
                                   uppercase (result.varname[i])
                          end
                  end
         end
end;

{---------------------------------------}
PROCEDURE MakeNewEntry (var GlobalResult: GlobalRes; name: string);
var p: GlobalRes;
begin
 new(p); p^.next := nil; p^.name := name; p^.value.flag := _null;
 if GlobalResult = nil then GlobalResult := p
                       else begin
                             p^.next := GlobalResult;
                             GlobalResult := p
                            end
end;

{---------------------------------------}
PROCEDURE FindValue (var p: GlobalRes; name: string);
var found: boolean;
begin
 p := GlobalResult;
 while (p <> nil) and not found
       do begin
           found := name = p^.name;
           if not found then p := p^.next
          end
end;

{---------------------------------------}
PROCEDURE throw (Code: PCellList; var Result: Cell);
var res, value: cell; p: GlobalRes;
begin
 if code = nil
    then expected ('Function THROW: Parameter')
    else begin
          eval (code,res);
          FindValue (p, res.varname);
          if res.varname = p^.name
             then begin
                   code := code^.next;
                   if code = nil
                      then expected ('Function THROW: Parameter')
                      else begin
                            eval (Code, res);
                            p^.value := res;
                            thrown := true;
                           end
                  end
         end
end;

{---------------------------------------}
PROCEDURE DisposeGlobal(var p: GlobalRes);
var q: GlobalRes;
begin
 if GlobalResult = p
    then begin
          GlobalResult := GlobalResult^.next;
          dispose(p)
         end
    else begin
          q := GlobalResult;
          while (q <> nil) and (q^.next <> p) do q := q^.next;
          if q^.next = p then begin
                               q^.next := q^.next^.next;
                               dispose(p)
                              end
         end
end;

{---------------------------------------}
PROCEDURE catch (Code: PCellList; var Result: Cell);
var res: cell; cmdoname: string; p: GlobalRes;
begin
 if code = nil
    then expected ('Function CATCH: Parameter')
    else begin
          eval(Code,res);
          if res.flag <> _id
             then expected ('Function CATCH: atom')
             else begin
                   MakeNewEntry (GlobalResult, res.varname);
                   cmdoname := res.varname;
                   code := code^.next; thrown := false;
                   while (code <> nil) and not thrown do
                         begin
                          eval(Code,Res);
                          if not thrown then code := code^.next
                         end;
                   if thrown then begin
                                   FindValue (p,cmdoname);
                                   result := p^.value;
                                   if result.flag = _null
                                      then error ('NULL type thrown');
                                   thrown := false;
                                   DisposeGlobal (p)
                                  end
                             else result := res
                  end
         end
end;

{---------------------------------------}
PROCEDURE PutFunction (q: PFuncTable);
begin
 if q^.IsMacro then PutForm (1,1,1,'(DEFMACRO ',' ',_string)
               else PutForm (1,1,1,'(DEFUN ', ' ', _string);
 PutForm (1,1,1,q^.FuncName + ' ',' ',_string);
 Print (q^.FuncParam);
 PutForm(1,1,1,NL+'    ',' ',_string);
 Print (q^.FuncBody);
 PutForm(1,1,1,')' + NL, ' ', _string)
end;

{---------------------------------------}
Procedure WriteFunc (Code: PCellList; var result: Cell);
var q: PFuncTable; res,filename: cell; ID: integer;
    cmdo: integer;
begin
 cmdo := OutputFile;
 eval (Code,Res);
 if res.flag <> _id
    then expected ('Function WRITE-FUNC: Identifier')
    else begin
          code := code^.next;
          if code <> nil
             then begin
                   eval(code,filename);
                   if filename.flag <> _file
                      then expected ('Function WRITE-FUNC: File identifier')
                      else GetFileID (filename.descriptor, ID);
                      OutputFile := ID
                  end;
          if InTable (res.varname,q)
             then PutFunction (q)
             else error ('Function WRITE-FUNC: Unknown function name');
          {if code^.next <> nil then PutForm(1,1,1,')' + NL, ' ', _string);}
         end;
 OutputFile := cmdo
end;

{---------------------------------------}
PROCEDURE initfile (Code: PCellList; var Result: Cell);
var Descriptor: string;
    exists: boolean;
    ID: integer;
begin
 exists := false;
 if (code = nil) or (code^.data.flag <> _string)
    then expected ('Function INIT-FILE: Filename')
    else begin
          NewFile (code^.data.str,ID);
          if exit_code
             then begin
                   assign (Files[Id].FileVar, Files[Id].Name);
                   if code^.next <> nil
                      then error ('Function INIT-FILE: Too Many Parameters');
                   if exit_code
                      then begin
                             Files [ID].dir := _output;
                             DeskFileID := ID;
                             AssignedDesktop := true;
                             Files[ID].pos := 0;
                            {$I-}
                              reset(Files[Id].FileVar);
                              exists := IOResult = 0;
                              if exists then close (Files[Id].FileVar);
                            {$I+}
                            if exists then append (Files[Id].FileVar)
                                      else rewrite (Files[Id].FileVar);
                            result.flag := _file;
                            result.descriptor := files[id].descriptor
                          end
                  end
         end
end;

{---------------------------------------}
PROCEDURE SaveFunc (Code: PcellList; Var Result: Cell);
var q: PFuncTable; res: cell; cmdo: integer;
begin
 if not AssignedDesktop
    then error ('Please set desktop file with INIT-FILE before using SAVE-FUNC')
    else begin
           q := nil;
           if (code = nil)
              then expected ('Function SAVE-FUNC: Parameter')
              else begin
                    eval(Code,Res);
                    if res.flag <> _id
                       then expected ('Function SAVE-FUNC: Function Identifier')
                       else begin
                             if InTable (res.varname,q)
                                then begin writeln('ci sono');
                                      cmdo := OutputFile;
                                      OutputFile := DeskFileID;
                                      PutFunction (q);
                                      OutputFile := cmdo
                                     end
                                else error ('Function SAVE-FUNC: ' +
                                            'Unknown function name')
                            end
                   end
          end;
 if exit_code then begin
                    result.flag := _spec;
                    result.t := true
                   end
              else begin
                    result.flag := _spec;
                    result.t := false
                   end
end;

{---------------------------------------}
PROCEDURE SaveVars;
var p: PVarTable; q: PCellList; res: cell; cmdo: integer;
begin
 p := GlobalVar;
 assign (files[1].filevar,'.\temp\$globvar.dat');
 rewrite (files[1].filevar);
 cmdo := outputfile; OutputFile := 1;
 while p <> nil do begin
                    PutForm (1,1,1,'(SETQ ' + p^.varname+' ',' ',_string);
                    if p^.value.flag in [_id, _list]
                       then begin
                             new(q);
                             q^.data := p^.value;
                             q^.next := nil;
                             addquote (q,'''');
                             p^.value := q^.data
                            end;
                    Print (p^.Value);
                    if p^.next <> nil
                       then PutForm (1,1,1,')' + NL + NL,' ',_string)
                       else PutForm (1,1,1,')' + NL,' ',_string);
                    p := p^.next
                   end;
 close (files[1].filevar);
 outputfile := cmdo
end;

{---------------------------------------}
PROCEDURE format (code: PCellList; var Result: Cell);
var s: string; cmdo,cmdo2: integer; res: cell; to_string : boolean;
    p: PCellList;
begin
 cmdo := outputfile; to_string := false;
 if (code = nil)
    then expected ('Function FORMAT: Parameters')
    else begin
          if code^.data.flag = _spec
             then begin
                   if code^.data.t
                        then outputfile := 0
                        else begin
                                NewFile('$cmdo.txt',outputfile);
                                Rewrite(files[outputfile].filevar);
                                to_string := true
                             end
                  end
             else if code^.data.flag = _file
                        then GetFileID (code^.data.descriptor, outputfile)
                        else expected ('Function FORMAT: File ID')
         end;
 if exit_code
    then begin
          code := code^.next;
          if code^.data.flag <> _string
             then expected ('Function FORMAT: STRING')
             else begin
                   s := code^.data.str;
                   code := code^.next;
                   while s <> ''
                      do begin
                          if s[1] <> '~'
                             then PutForm(1,1,1,s[1],' ',_string)
                             else begin
                                   delete(s,1,1);
                                   case s[1] of
                                       'D': PutForm (code^.data.int,1,1
                                                ,'',' ',_int);
                                       'A': PutForm (1,1,1,code^.data.ch,' ',
                                                _string);
                                       'E': PutForm (1,1,code^.data.float,
                                                '',' ',_float);
                                       'O': PutForm (1,1,1,
                                              octal(code^.data.int),
                                              ' ',_string);
                                       'S': print (Code^.data);
                                       '~': PutForm (1,1,1,'~',' ',_string);
                                       '%': PutForm (1,1,1,NL,' ',_string);
                                       end;
                                   if not (s[1] in ['~','%'])
                                        then code := code^.next
                                  end;
                              delete(s,1,1)
                         end;
                  PutForm(1,1,1,NL,' ',_string)
                 end
         end;
 if to_string then begin
                    close(files[outputfile].filevar);
                    reset(files[outputfile].filevar);
                    cmdo2 := inputfile;
                    inputfile := outputfile;
                    new(p);
                    scan(p);
                    result := p^.data;
                    close(files[outputfile].filevar);
                    outputfile := cmdo;
                    inputfile := cmdo2
                   end
              else begin
                    outputfile := cmdo;
                    result.flag := _spec;
                    result.t := false
                   end
end;

{---------------------------------------}
PROCEDURE SaveFunctions;
var p: PFuncTable;
begin
 assign (files[1].filevar,'.\temp\$func.dat'); rewrite (files[1].filevar);
 p := FuncTab; OutputFile := 1;
 while p <> nil do begin
                    PutFunction(p);
                    p := p^.next
                   end;
 close (files[1].filevar)
end;

{---------------------------------------}
PROCEDURE Star (i: byte; var result: cell);
begin
 result := PreviousRes[i]
end;

{---------------------------------------}
PROCEDURE Refresh;
begin
 CloseAll;
 SaveVars;
 SaveFunctions;
 OutputFile := 0;
 halt (10)
end;

{---------------------------------------}
PROCEDURE type_of (Code: PCellList; var Result: Cell);
var Res: cell;
begin
 Result.flag := _id;
 eval(Code,Res);
 case res.flag of
      _list: result.varname := 'LST';
      _int: result.varname := 'FIXNUM';
      _float: result.varname := 'FLOAT';
      _ratio: result.varname := 'RATIONAL';
      _char: result.varname := 'CHAR';
      _op: result.varname := 'FUNCTION';
      _id, _spec: result.varname := 'ATOM';
      _file: result.varname := 'STREAM';
      _string: result.varname := 'STRING'
   end
end;

{---------------------------------------}
PROCEDURE UpdateDO (incr: PVarTable);
var incr2,p: PVarTable; q: PCellList; res: cell;
begin
 while incr <> nil
        do begin
            p := nil;
            FindVar (incr^.varname,p);
            if p = nil
                then error ('Unknown identifier in DO increment list')
                else begin
                      new(q); q^.next := nil; q^.data := incr^.value;
                      eval (q,res);
                      dispose(q);
                      p^.value := res;
                     end;
            incr := incr^.next
           end;
end;

{---------------------------------------}
PROCEDURE Eval_do (Code: PCellList; var Result: Cell; star: boolean);
var termcond, returnvalue,p, names, values, code1: PCellList;
    n: integer;
    cmdo,res: cell;
    q,incr: PVarTable;
    LocalList, LocalVar: PLocalVar;
begin
 if (code = nil) or (code^.next = nil) or (code^.next^.next = nil)
    then expected ('Function DO: Three parameters')
    else begin
          if (code^.data.flag) <> _list
             then expected ('Function DO: LIST')
             else begin
                        {PARAMETER INITIALIZING}
                   if code^.data.list^.data.flag <> _list
                        then expected ('Function DO, Parameter list: list');
                   p := code^.data.list;
                   names := nil;
                   values := nil;
                   incr := nil;
                   while p <> nil
                        do begin
                                {INSERIRE CONTROLLO ERRORI}
                            AddToList (Names,p^.data.list^.data);
                            AddToList (Values,p^.data.list^.next^.data);
                            PutVarQueue (Incr, p^.data.list^.data.varname,
                                    p^.data.list^.next^.next^.data);
                            p := p^.next
                           end;
                        {TERMINAL CONDITION AND RETURN VALUE}
                   code := code^.next;
                   if (code = nil) or (code^.data.flag <> _list)
                      then expected
                           ('Function DO: Pair List (Terminal condition' +
                           ' + returned value')
                      else begin
                            TermCond := code^.data.list;
                            ReturnValue := code^.data.list^.next
                           end;
                        {CODE}
                   code := code^.next;
                   if code = nil
                        then expected ('Function DO: Code')
                        else begin
                                LocalList := nil;
                                if not star
                                   then InitParam (Names,Values,n,false,
                                        LocalList)
                                   else begin
                                         while names <> nil
                                                do begin
                                                    eval(values,res);
                                                    PushVar(GlobalVar,
                                                        names^.data.varname,
                                                        res,q);
                                                    new (LocalVar);
                                                    LocalVar^.p := q;
                                                    LocalVar^.next := LocalList;
                                                    LocalList := LocalVar;
                                                    names := names^.next;
                                                    values := values
                                                   end
                                        end;
                                code1 := code;
                                repeat
                                 eval(TermCond,res);
                                 if IsFalse (res)
                                    then begin
                                          code1 := code;
                                          while code1 <> nil
                                                do begin
                                                    eval(Code1,cmdo);
                                                    code1 := code1^.next
                                                   end;
                                          UpdateDo (incr)
                                         end
                                until not IsFalse (res);
                                eval (ReturnValue,Result);
                                DisposeAll (LocalList)
                            end
                  end
         end
end;

{---------------------------------------}
PROCEDURE Load_Font (Code: PCellList; var Result: Cell);
var res: cell; script: text;
begin
 if (code = nil)
    then expected ('Function LOAD-FONT: Parameters')
    else begin
          eval(Code,res);
          if res.flag <> _id
             then expected ('Function LOAD-FONT: Identifier')
             else begin
                   assign(script, '.\temp\script.bat'); rewrite(script);
                   writeln(script, '.\fonts\gnuchcp3 .\fonts\' +
                                    res.varname + '.fnt');
                   writeln(script,'xkeyb .\fonts\' + res.varname + '.key');
                   close(script);
                   refresh
                  end
         end
end;

{---------------------------------------}
PROCEDURE  UnLoad_keyb (Code: PCellList; var Result: Cell);
var script: text;
begin
  assign(script,'script.bat'); rewrite(script);
  writeln(script, 'xkeyb /U');
  close(script)
end;

{---------------------------------------}
PROCEDURE  UnLoad_font (Code: PCellList; var Result: Cell);
var script: text;
begin
  assign(script,'script.bat'); rewrite(script);
  writeln(script, 'gnuchcp3 -r');
  writeln(script,'xkeyb /U');
  close(script)
end;

{---------------------------------------
PROCEDURE Load_Keyb (Code: PCellList; var Result: Cell);
var res: cell;
begin
  if (code = nil)
    then expected ('Function LOAD-KEYB: Parameters')
    else begin
          eval(Code,res);
          if res.flag <> _id
             then expected ('Function LOAD-KEYB: Identifier')
             else begin
                   loadkeyb (res.varname);
                   result.flag := _spec; result.t := true
                  end
         end
end;


PROCEDURE Load_Font (Code: PCellList; var Result: Cell);
begin
 if (code = nil) or (code^.data.flag <> _id)
    then expected ('Function LOAD-FONT: Identifier')
    else begin
          LoadFontFile (code^.data.varname);
          LoadFont;
          LoadKeyb (code^.data.varname);
          result.flag := _spec; result.t := true
         end
end;
}

{---------------------------------------}
PROCEDURE Eval_op (Code: PCellList; var Result: Cell);
var opcode: opcodes;
begin
 opcode := code^.data.opcode;
 code := code^.next;
 if opcode in [_add, _sub, _mul, _div]
    then compute (Code, Result, OpCode)
    else if opcode in [_gt, _lt, _eq, _ge, _le, _ne, _eq_atom]
            then BinaryRelations (Code, Result, Opcode)
            else case opcode of
              _setq: setq(Code, Result);
              _quote: quote(Code, Result);
              _backquote: backquote (Code, Result, false);
              _defun: defun(Code, Result, false);
              _defmacro: defun (Code, Result, true);
              _comma, _at: error ('No BACKQUOTE to match');
              _load: load (Code, Result);
              _quit, _bye: stop := true;
              _car: car (Code, Result);
              _cdr: cdr (Code, Result);
              _cons: cons (Code, Result);
              _eval: func_eval (Code, Result);
              _if: eval_if (Code, Result);
              _lst: make_list (Code, Result);
              _let: let (Code, Result);
              _let_star: let_star (Code,Result);
              _print: eval_print (code, Result);
              _princ: princ (Code, Result);
              _terpri: writeln;
              _writeln: writeline (Code, Result);
              _write: eval_write (Code, Result);
              _read: eval_read (Code, Result);
              _readln: readline (Code, Result);
              _type_of: type_of(Code,Result);
              _dotimes: dotimes(Code, Result);
              _dolist, _doloop: dolist(Code, Result);
              _sqrt: do_sqrt(Code,Result);
              _sin: do_sin (Code, Result);
              _cos: do_cos (Code, Result);
              _random: do_random (Code, Result);
              _help: do_help (Code, Result);
              _boundp: boundp (Code, Result);
              _loop: loop (Code, Result);
              _return: break := true;
              _coerce: coerce (Code,Result);
              _str_eq: str_eq (Code, Result);
              _str_ne: str_ne (Code, Result);
              _str_lt: str_lt (Code, Result);
              _str_gt: str_gt (Code, Result);
              _open: openfile (Code, Result);
              _close: closefile (Code, Result);
              _symbol_name: spelling (Code, Result);
              _make_symbol: MakeSymbol (Code, Result);
              _lambda: lambda (Code, Result);
              _throw: throw (Code, Result);
              _catch: catch (Code, Result);
              _write_func: WriteFunc (Code, Result);
              _save_func, _sf: SaveFunc (Code,Result);
              _format: format (Code, Result);
              _cond: eval_cond (Code, Result);
              _do: eval_do (Code,Result,false);
              _do_star: eval_do (Code,Result,True);
              _load_font: Load_Font (Code, Result);
              _unload_font: Unload_Font (Code, Result);
              _unload_keyb: unload_keyb (Code, Result);
              _init_file: initfile (Code, Result);
              _exp: do_exp (Code,Result);
{              _load_keyb: Load_Keyb (Code, Result); }
              _refresh: Refresh
             end
end;

{---------------------------------------}
PROCEDURE Eval_func (Code: PCellList; var Result: Cell);
var q: PFuncTable;
    p: PCellList;
    i, Params: integer;
    LocalList: PLocalVar;
    cmdo: cell;

begin
 if not InTable(code^.data.varname, q)
    then error ('Undefined function ' + code^.data.varname);
 if exit_code
    then begin
          if q^.FuncParam.flag <> _list
             then error ('Internal error in param list');
          if exit_code
             then begin
                   InitParam (q^.FuncParam.list, code^.next, Params,
                             q^.IsMacro, LocalList);
                   new(p);
                   p^.next := nil;
                   p^.data := q^.FuncBody;
                   if q^.IsMacro then begin
                                       eval (p,cmdo);
                                       p^.next := nil;
                                       p^.data := cmdo
                                      end;
                   if exit_code then eval(p,Result);
                   DisposeAll (LocalList)
                  end
         end
end;

{---------------------------------------}
PROCEDURE Eval (Code: PCellList; var Result: Cell);
begin
 if (not thrown) and exit_code then begin
 case Code^.Data.Flag of
      _int, _float, _ratio: result := Code^.data;
      _id: lookup (Code^.Data.varname, Result);
      _op: case code^.data.opcode of
                _mul: star(1,result);
                _two_stars: star (2,result);
                _three_stars: star (3,result)
                else error ('Illegal function call')
             end;
      _char: result := code^.data;
      _spec: if code^.data.t
                then result := code^.data
                else begin
                      result.flag := _list;
                      result.list := nil
                     end;
      _string: result := code^.data;
      _list: begin
              if code^.data.list = nil
                 then Result := code^.data
                 else begin
                       case code^.data.list^.data.flag of
                            _op: eval_op (Code^.data.list, Result);
                            _id: eval_func (Code^.data.list, Result)
                           else error ('Undefined function')
                        end
                      end
            end;
      {IMPORTANTE PER LA LET}
      _null: result.flag := _null;
      _file: result := code^.data;
      _struct: result := code^.data
     end
     end
end;

{---------------------------------------}
PROCEDURE Print (Datum: Cell);
begin
 with Datum
      do begin
          case flag of
               _int:   PutForm (int,1,1,'',' ',_int);
               _char:  PutForm (1,1,1,'',ch, _char);
               _float: PutForm (1,1,float,'',' ',_float);
               _ratio: PutForm (num,denom,1,'',' ',_ratio);
               _id: PutForm (1,1,1,varname,' ',_string);
               _op: PutForm(1,1,1,KWList[ord(OpCode)+1],' ', _string);
               _spec: if t then PutForm(1,1,1,'T',' ',_string)
                           else PutForm(1,1,1,'NIL',' ',_string);
               _string: PutForm (1,1,1,'"' + datum.str + '"', ' ', _string);
               _list: begin
                       PutForm (1,1,1,'(',' ',_string);
                       while datum.list <> nil
                             do begin
                                 print (datum.list^.data);
                                 datum.list := datum.list^.next;
                                 if datum.list <> nil
                                    then PutForm(1,1,1,' ',' ', _string)
                                end;
                       PutForm (1,1,1,')',' ',_string)
                      end;
               _file: PutForm (1,1,1,descriptor,' ',_string);
               _struct: begin
                         PutForm (1,1,1,'#s(' + datum.strname,' ',_string);
                         while datum.r <> nil
                               do begin
                                   PutForm (1,1,1,' '+datum.r^.varname,' ',
                                           _string);
                                   print (datum.r^.value);
                                   datum.r := datum.r^.next;
                                  end;
                         PutForm (1,1,1,')',' ',_string)
                      end;
               end
       end
end;

{---------------------------------------}
PROCEDURE Restore;
begin
 openall;
 LoadFile ('.\temp\$GlobVar.dat',Result,false);
 LoadFile ('.\temp\$Func.dat',Result,false);
end;

{---------------------------------------}
PROCEDURE Welcome (prog_name, year: string);
begin
 writeln('  FLISP version 1.0, Copyright (C) 2005 Francesco Zamblera.');
 writeln;
 write('  FLISP comes with ABSOLUTELY NO WARRANTY; for details see the ');
 writeln('file COPYING.TXT.');
 write('  This is free software, and you are welcome to redistribute');
 writeln(' it under certain ');
 writeln('conditions.')
end;

{---------------------------------------}
PROCEDURE Init;
var i: integer;
begin
 UpCaseOn := true;
 InitFiles;
 thrown := false;
 Look := ' ';
 Code := nil;
 GlobalVar := nil;
 GlobalResult := nil;
 FuncTab := nil;
 exit_code := true;
 break := false;
 if ParamStr(1) = '-c'
    then Restore
    else begin
          LoadFile ('flisp.lib',Result,true);
          clrscr;
          Counter := 0;
          InputFile := 0;
          OutputFile := 0;
          DeskFileID := 0;
          FileId := 0;
          AssignedDesktop := false;
          for i := 1 to 3 do PreviousRes[i].flag := _null;
          Welcome ('MiniLISP for FreeDOS','2005')
         end;
 stop := false;
 WritePrompt;
 GetChar
end;


{---------------------------------------}
PROCEDURE Read_Eval_Print;
begin
  if look <> ^Z then Scan (Code)
                else stop := true;
  if Exit_Code
     then begin
           Eval (Code, Result);
           if not stop
              then begin
                    if exit_code
                       then begin
                              PreviousRes [3] := PreviousRes [2];
                              PreviousRes [2] := PreviousRes [1];
                              PreviousRes [1] := Result;
                              if FromKeyboard
                                 then begin
                                       Print (Result);
                                       WritePrompt
                                      end
                             end
                        else write('?..> ')
                   end
             end
        else begin
               write(' ? > ');
               GetChar
              end
end;

{---------------------------------------}
(* MAIN *)
begin
 Init;
 repeat
  exit_code := true;
  Read_Eval_Print
 until stop;
 CloseAll
end.

