/*
    SUPPL - a supplemental library for common useage
    Copyright (C) 1995,1996  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $RCSfile: ENV_MTCH.C $
   $Locker:  $	$Name:  $	$State: Exp $

	Functions: env_matchVar()

	Comments:

	int env_matchVar(word segm, char var[])

	Try to match the variable var in the environment segm.

	Normally all environment functions within this package are case-sensitive.
	This is uncommon to DOS and may lead to improper functionality, but this
	allows to process uncommon variables, such as "windir" and "winbootdir".

	First, this function tries to locate var as it is given; second, this
	function performs a case-insensitve match. The constents of the var[]
	array is modified to the string of the first found match that it can
	be used as the input to all the other functions.

	If segm == NULL, the current used environment is processed.

	Return:
		1: var == NULL
		2: no match found, var[] unaltered
		3: no environment at all, var[] unaltered
		Bit mask, if (return_value & 7) == 0:
		4: more than one case-sensitive match found, var[] unaltered
		8: more than one case-insensitive match found, var[] updated
		16: case-sensitve match found, var[] unaltered
		32: case-insensitve match found, var[] updated

	Note: If at least one of the bits 4 or 16 is set, the var[] is not
		modified.

	Target compilers: Micro-C v3.13, v3.14; Borland C v2, v3.1, v4.52

*/

#include "initsupl.inc"

#ifndef _MICROC_
#include <dos.h>
#include <string.h>
#endif
#include <portable.h>
#include "fmemory.h"

#include "suppl.h"
#include "dynstr.h"
#include "environ.h"

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: ENV_MTCH.C 1.5 1998/08/19 02:40:21 ska Exp $";
#endif

int env_matchVar(word segm, char var[])
{	int retVal, varLen;
	word firstOfs, ofs;

	if(!var) return 1;			/* no variable to search for */

	if(!segm && !(segm = env_glbSeg))
		return 3;				/* no environment at all */

	retVal = ofs = 0;
	varLen = strlen(var);		/* offset of '=' sign */
	while(peekb(segm, ofs)) {
		if(peekb(segm, ofs + varLen) == '=') {	/* length is OK */
			if(_fmemcmp(MK_FP(segm, ofs), TO_FP(var), varLen) == 0) {
				/* case-sensitive match */
				retVal |= (retVal & 16)? 4: 16;
			}
			else if(_fMemiCmp(MK_FP(segm, ofs), TO_FP(var), varLen) == 0) {
				/* case-insensitive match */
				if(!retVal) firstOfs = ofs;
				retVal |= (retVal & 32)? 8: 32;
			}
		}
		ofs += env_varlen(segm, ofs);
	}

	if(!retVal)								/* no match found */
		return 2;

	if(!(retVal & (16 | 4)))		/* no case-sensitive match -> update var */
		_fmemcpy(TO_FP(var), MK_FP(segm, firstOfs), varLen);

	return retVal & (32 | 16 | 8 | 4);
}
