/*
    SUPPL - a supplemental library for common useage
    Copyright (C) 1995,1996  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $RCSfile: DSTRAPPE.C $
   $Locker:  $	$Name:  $	$State: Exp $

	char *StrAppend_(char *dest[], const char src[]
	 , const char delim[], const char quotes[]);

	Append the string src[] at dest delimited by *delim and quoted by a quote.

	If src == NULL, this function performs no action.

	If *dest == NULL, src[] is quoted if necessary.

	If *dest does not end with one character out of delim[] and src[] does not
	start with one character out of delim and **dest != '\0', *delim is
	placed between *dest and src[].

	delim[] consists of two fields:
		"*\0#"
	- All characters out of "*" are valid delimiting characters.
		Its first character is used as the delimiter.
	- All characters out of "#" are characters that are no delimiters, but
		must be quoted.

	quotes[] consists of three fields delimited by '\0':
		"*\0#\0[]"
	- All characters out of "*" delimit a string like '"' in C.
	- All characters out of "#" delimit a character like '\\' in C.
		These characters also quote every other quotation character.
	- All character pairs out of "[]" delimit a string like this:
		'[' mark the start of a quoted string;
		']' mark its end.
	If src[] need to be quoted, this is done in this order:
		1) If there is a quote out of "*" is not used within *dest,
			the whole src[] is quoted.
		2) If there is a quote pair out of "[]" is not used within *dest,
			the whole src[] is quoted with this pair.
		3) If there is a quote in "#", each character to be quoted is quoted
			with this quotation character. It will also quote that characters,
			that are already quoted with "*" and "[]" quotes!
		4) src[] is not quoted.

	Return:
		NULL: malloc() failed; *dest is unchanged
		else: *dest

	Target compilers: Micro-C v3.13, v3.14; Borland C v2, v3.1, v4.52

*/

#include "initsupl.inc"

#ifndef _MICROC_
#include <string.h>
#include <stdlib.h>
#endif
#include <portable.h>
#include "dynstr.h"

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: DSTRAPPE.C 1.4 1998/08/19 02:40:21 ska Exp $";
#endif

char *StrAppend_(char *dest[], const char src[]
 , const char delim[], const char quotes[])
{	const char *delimiter;		/* characters that delimit two strings */
	const char *toquote;			/* characters that need to be quoted */
	const char *sQuote;			/* "*" quotes */
	const char *cQuote;			/* "#" quotes */
	const char *dQuote;			/* "[]" quotes */
	char *p;				/* temporary *dest pointer */
	const char *q;				/* temp pointer while scanning for quotes */
	char *h;				/* temp pointer */
	int ch;
	int quote;				/* current quote; NUL if none */

	if(!src || !*src) return *dest;

	if((p = *dest? StrDup(*dest): strdup("")) == NULL)
		return NULL;

	/* split the delim[] && quotes[] arrays */
	toquote = strchr(delimiter = delim, NUL) + 1;
	dQuote = strchr(cQuote = strchr(sQuote = quotes, NUL) + 1, NUL) + 1;

	/* check if the strings must be delimited */
	if(*p							/* the first string is not empty */
	 && !strchr(delimiter, *((char*)strchr(p, NUL)-1))	/* 1st str not delimited */
	 && !strchr(delimiter, *src)) {		/* both strings must be delimited */
	 	if(!StrAppChr(p, *delimiter))
	 		return StrFree(p);
	}

	/* check, if the string need be quoted */
	quote = NUL;
	ch = *(q = src);
	do {
		if(strchr(cQuote, ch)) {		/* skip the next character */
			if(q[1]) ++q;					/* cQuote to ignore */
		}
		else if(quote == ch)			/* current quote termintes */
			quote = 0;
		else if(!quote) {				/* unquoted part of the string */
			if(strchr(sQuote, ch))		/* single quote character */
				quote = ch;
			else if((h = strchr(dQuote, ch)) != NULL) {	/* double quote character */
				if(((h - dQuote) & 1) == 0)	/* '[' part found */
					quote = h[1];
			}
			else if(strchr(delimiter, ch) || strchr(toquote, ch))
				break;		/* src must be quoted! */
		}
	} while((ch = *++q) != NUL);

	/* append src[] */
	if(!ch) { 					/* src[] need not to be quoted */
		quote = 0;
		goto noQuote;
	}

/* determine how to quote src[] */

	/* check the single quotes */
	for(q = sQuote; (ch = *q++) != 0;)
		if(!strchr(src, quote = ch))
			goto quoteWhole;

	/* check the double quotes */
	for(q = dQuote - 1; (ch = *++q) != 0;)
		if(!(++q, strchr(src, ch)) && !strchr(src, quote = *q))
			goto quoteWhole;

	/* use single character quotes, if present */
	if(!*cQuote) goto noQuote;

	/* use the single character quote ch */
	if((h = realloc(p, strlen(p) + 2 * strlen(src) + 1)) == NULL)
		return StrFree(p);

	h = strchr(p = h, NUL);
	ch = *(q = src);
#if 0					/* quote only not-quoted characters */
	quote = 0;
	do {
		if(strchr(cQuote, ch)) {		/* skip the next character */
			if(q[1]) {
				*h++ = ch;
				ch = *++q;					/* cQuote to ignore */
			}
		}
		else if(quote == ch)			/* current quote termintes */
			quote = 0;
		else if(!quote) {				/* unquoted part of the string */
			if(strchr(sQuote, ch))		/* single quote character */
				quote = ch;
			else if((h = strchr(dQuote, ch)) != NULL) {	/* double quote character */
				if(((h - dQuote) & 1) == 0)	/* '[' part found */
					quote = h[1];
			}
			else if(strchr(delimiter, ch) || strchr(toquote, ch))
				*h++ = *cQuote;		/* character to be quoted */
		}
	} while((*h++ = ch, ch = *++q) != NUL);

#else			/* quote any character that is in delim[] */

	quote = *cQuote;
	do {
		if(strchr(cQuote, ch)) {		/* skip the next character */
			if(q[1]) {
				*h++ = ch;
				ch = *++q;					/* cQuote to ignore */
			}
		}
		else if(strchr(delimiter, ch) || strchr(toquote, ch))
			*h++ = quote;		/* character to be quoted */
		*h++ = ch;
	} while((ch = *++q) != NUL);
#endif

	*h = NUL;
	return StrRepl(*dest, StrTrim(p));

quoteWhole:
	if(!StrAppChr(p, ch))
		return StrFree(p);

noQuote:
	if(!StrCat(p, src) || !StrAppChr(p, quote))
		return StrFree(p);

	return StrRepl_(dest, p);
}
