/*
    SUPPL - a supplemental library for common useage
    Copyright (C) 1995,1996  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/*
    SUPPL - a supplemental library for common useage
    Copyright (C) 1995,1996  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $RCSfile: DFNMATCH.C $
   $Locker:  $	$Name:  $	$State: Exp $

	Match a filename against a pattern
	Supports >>DOS<< and Win32 patterns.

	Assumptions:
		for DOS:
			+ A path component contains one dot at maximum, except in
				"..". However, the dot may head or trail a string.
			+ The supplied strings are not checked for invalid characters.
			+ The drive separator ':' is considered as a normal delimiter
				of path components, however it does not match a slash.

	DOS patterns:
		"?" any character except dot and space
		"*..." none or more characters up to the next dot or end of name;
			any following characters are ignored to the next dot or end of name
			--> only one asterisk within name and/or extension honored
		"." delimiter of filename and fileextension components
		"/" or "\\" path component delimiter (any non-empty sequence)

		"*" or "*.*" match any file
		"*." matches any file without an extension

	Win32 patterns:
		"?" any character
		"*" none or more characters
		"/" or "\\" delimiter of filename components
		"*"	matches any filename
		"*.*" matches any filename with at least one dot
		"*." matches a sequence of characters without a dot in it
			More than one asterisks are honored, even so dots.

	The path components "." and ".." are treated as real components.
	The pattern may contain wildcards within the path components.
	Wildcards within the fielname are not recognized.

*/

#include "initsupl.inc"

#ifndef _MICROC_
#include <ctype.h>
#include "dir.inc"
#include <string.h>
#endif
#include "dfn.h"

#ifdef OS_DOS
#undef OS_DOS
#endif
#ifndef OS_WIN32		/* passed from DFNMATC2.C */
#define OS_DOS
#endif

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: DFNMATCH.C 1.4 1998/08/19 07:26:56 ska Exp $";
#endif

/*	no macro to avoid side effects */
static int isPathDelim(int ch)
{	return ch == '/' || ch == '\\';
}

/*
 *	Match a filename against a patterm
 */
static int match(char *fnam, char *pattern)
{	int pch, ch;

	while((pch = *pattern++) != '\0') {
		ch = *fnam++;
		switch(pch) {
		case '?':		/* any character except dot in DOS mode */
#ifdef OS_DOS
			if(ch == '.')
				return 0;
#endif
			if(!ch)
				return 0;
			break;
		case '*':
#ifdef OS_DOS
			/* skip any character upto the next dot or '\0' */
			while(*pattern && *pattern != '.')
				++pattern;
			if(ch && ch != '.')
				while(*fnam && *fnam != '.')
					++fnam;
#else
			/* an asterisk in Win32 does act more Unix-stylish,
				let's use a recursive algorithm */
			/* collapse multiple asterisks, they have no meaning
				anyway */
			while((pch = *pattern) == '*') ++pattern;
			if(!pch) 	/* asterisk at the end of name --> always match */
				return 1;

			--fnam;			/* was advanced above */
			if(pch == '.') {		/* often used patterns */
				if(!*++pattern)	/* any sequence, no dot in it */
								/* or end in a dot */
					return strchr(fnam, '.') == NULL
					 || (*fnam && ((char*)strchr(fnam, '\0'))[-1] == '.');

				/* optimization: advance to the next dor within 'fnam'.
					Because '.' is not case-depend, it's easy. */
				while((fnam = strchr(fnam, '.')) != NULL)
					if(match(++fnam, pattern))
						return 1;
				return 0;
			}

			if(!ch) return 0;

			/* Now try the remaining pattern against the 'fnam' with
				none to N characters strip from the beginning of fnam. */
			/* This routine is not that much called, therefore no
				special optimization are implemented */
			do {
				if(match(fnam, pattern))
					return 1;
			} while(*++fnam);
			return 0;
#endif
			break;
		case '.':		/* one trailing dot is ignored */
			if(!*pattern)
				goto ret;
			/* Fall through */

		default:
			if(toupper(ch) != toupper(pch))
				return 0;
		}
	}

ret:		/* ignore one trailing dot */
	return *fnam == '\0' || (*fnam == '.' && fnam[1] == '\0');
}


/*
 *	Copy the next path component and return the break character.
 */
static int copyComp(char *dst, char **src)
{	int ch;
	char *p;

	p = *src;
	while((ch = (*dst++ = *p++)) != ':' && ch && !isPathDelim(ch));
	if(ch) dst[-1] = '\0';
	*src = p;
	if(ch == '\\') ch = '/';

	return ch;
}

/*
 *	Break the filename into path components, then let them
 *	be matched individually
 */
int dfnmatch(char *fnam, char *pattern)
{	char n1[DFN_FILENAME_BUFFER_LENGTH], n2[DFN_FILENAME_BUFFER_LENGTH];
	int c1, c2;

#ifdef SUPPORT_UNC_PATHS
	if(isUNCpath(fnam) != isUNCpath(pattern))
		return 0;
#endif

	/* 1. Make sure both name start or don't start with a path delimiter */
	if(isPathDelim(*fnam)) {
		while(isPathDelim(*++fnam));
		if(!isPathDelim(*pattern))
			return 0;
		while(isPathDelim(*++pattern));
	}
	else if(isPathDelim(*pattern))
		return 0;
	/* Both names point to a non-path delimiter character */

	/* 2. break up the next path component */
	do {
		c1 = copyComp(n1, &fnam);
		c2 = copyComp(n2, &pattern);

		if(c1 != c2 || !match(n1, n2))
			return 0;

		if(c1 == '/') {		/* slash / backslash */
			/* skip multiple path component delimiters */
			while(isPathDelim(*fnam)) ++fnam;
			while(isPathDelim(*pattern)) ++pattern;
		}
	} while(c1);

	return 1;
}
