/* A DOS-version of the getopt_long function */

/* Copyright (C) 2001 Jim Hall, jhall@freedos.org */

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <string.h>

#define _GETOPT_L_C			/* getopt_l.h depends on this */

#include "getopt_l.h"

/* Global variables for getopt_long */

char *optarg;
int optind = 0;
int opterr = 1;
int optopt;


/*
  The getopt_long() function works like getopt() except that it also
  accepts long options, started out by a slash char.
*/

/*
  The getopt_long() function returns the option character if the
  option was found successfully, ':' if there was a missing parameter
  for one of the options, '?' for an unknown option character, or EOF
  for the end of the option list.

  getopt_long_only() returns the option character when a short option
  is recognized.  For a long option, they return val if flag is NULL,
  and 0 otherwise.  Error and EOF returns are the same as for getopt(),
  plus '?' for an ambiguous match or an extraneous parameter.
*/

/*
  CHANGES FROM THE GNU getopt_long() FUNCTION: This getopt_long does
  not honor arguments to options, so ':' in optstring is ignored, as
  is has_arg in longopts.  Note that this implies "/:" is a valid
  option for getopt_long (this is a bug).

  Also, flag is not used in longopts.  longindex is not yet used.

  These should be implemented in a future version of getopt_long.
*/

int getopt_long(int argc, char *const argv[],
		const char *optstring,
		const struct option *longopts, int *longindex)
{
  int i;
  int len;				/* length of token */
  char *tok;				/* token from ARGV */

  /* Scan the command line option starting at optind+1 */

  if (++optind == argc)
    {
      /* End of ARGV - we are done scanning */

      return (EOF);
    }

  if (argv[optind][0] != '/')
    {
      /* Does not start with a slash - we are done scanning */

      return (EOF);
    }

  /* Grab the token from the string, advance by 1 past the slash */

  tok = strtok (argv[optind], "=");
  len = strlen (++tok);

  switch (len)
    {
    case 0:
      /* This is an empty string */

      break;

    case 1:
      /* Look for the char in optstring */

      /* This is case sensitive, just like UNIX getopt */

      for (i = 0; optstring[i]; i++)
	{
	  /* Skip over the colon in optstring */

	  if ((optstring[i] != ':') && (tok[0] == optstring[i]))
	    {
	      /* We have a match */

	      /* Check if option takes an arg, and set optarg */

	      if (optstring[i+1] == ':')
		{
		  optarg = strtok (NULL, "\n");

		  if (!optarg)
		    {
		      if (opterr)
			{
			  /* Print a warning message */
			  fprintf (stderr, "%s: %s: Option requires an argument\n", argv[0], tok);
			}

		      optopt = tok[0];
		      return (':');
		    }
		}

	      return (tok[0]);
	    }
	} /* for */

      /* Continue to look in longopts */

    default:
      /* Look for this string in longopts */

      for (i = 0; longopts[i].name; i++)
	{
	  if (strcmpi (tok, longopts[i].name) == 0)
	    {
	      /* We have a match */

	      /* Check if option takes an arg, and set optarg */

	      if (longopts[i].has_arg)
		{
		  optarg = strtok (NULL, "\n");

		  if ((!optarg) && (longopts[i].has_arg == required_argument))
		    {
		      if (opterr)
			{
			  /* Print a warning message */
			  fprintf (stderr, "%s: %s: Option requires an argument\n", argv[0], tok);
			}

		      optopt = tok[0];
		      return (':');
		    }
		}

	      return (longopts[i].val);
	    }
	} /* for */

      break;
    } /* switch */

  /* We didn't find a match */

  if (opterr)
    {
      fprintf (stderr, "%s: %s: Option is not recognized\n", argv[0], tok);
    }

  optopt = tok[0];
  return ('?');
}
