/* catgets.c */

/* Functions that emulate UNIX catgets */

/*
  Copyright (C) 1999-2000 Jim Hall <jhall1@isd.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>			/* sprintf */
#include <stdlib.h>			/* getenv  */
#include <string.h>			/* strtok, strchr */

#include <dir.h>			/* fnmerge */

#include "db.h"
#include "catgets.h"


/* Symbolic constants */

#define MAXSTR (DB_KEYLEN + DB_VALLEN)	/* strings from the catalog file */


/* Globals */

nl_catd catalog = 0;


/* Functions */

char *
catgets(nl_catd  cat,  int set_number, int message_number,
	 char *message)
{
  /* get message from a message catalog */

  /* 'message' should really be const, but not when it is returned */

  /* On success, catgets() returns a pointer to an internal buffer
     area containing the null-terminated message string.  On failure,
     catgets() returns the value 'message'.  */

  char key[DB_KEYLEN];
  db_t *ptr;

  /* Is this the same catalog we have in memory? */

  if (cat != catalog)
    {
#ifndef NDEBUG
      printf ("*catgets* catalog not open\n");
#endif /* NDEBUG */

      return (message);
    }

  /* fetch the message that goes with the set/message number */

  sprintf (key, "%d.%d", set_number, message_number);
  ptr = db_fetch (key);

#ifndef NDEBUG
  printf ("*catgets* key,value is %s,%s\n", key, ptr->value);
#endif /* NDEBUG */

  if (ptr)
    {
      return (ptr->value);
    }

  /* else */

  return (message);
}

nl_catd
catopen(char *name, int flag)
{
  /* open a message catalog */

  /* 'flag' is completely ignored */

  /* catopen() returns a message catalog descriptor of type nl_catd on
     success.  On failure, it returns -1. */

  FILE *pfile;
  char *key;
  char *value;
  char str[MAXSTR];
  char catfile[MAXPATH];
  char tmpstr[MAXPATH];		/* this is a hack */
  int len;				/* this is a hack */

  /* Open the catalog file */

  if (catalog)
    {
      /* Already one open */

#ifndef NDEBUG
      printf ("*catopen* catalog already open\n");
#endif /* NDEBUG */

      return (-1);
    }

  if (strchr (name, '\\'))
    {
      /* first approximation: 'name' is a filename */

      strcpy (catfile, name);
    }

  else
    {
      /* Better approximation: 'name' is the base name of a catalog file */

#ifdef BUGGY_CODE
      /* this defines catfile like %NLSPATH%\%LANG%\name. */

      /* Note that "LANG=en" and "LANG=en-UK" imply that you need
         different directories to store your message catalogs */

      strcpy (tmpstr, getenv("NLSPATH"));
      len = strlen (tmpstr);

      if (tmpstr[len-1] != '\\')
	{
	  strcat (tmpstr, "\\");
	}

      strcat (tmpstr, getenv("LANG"));
      fnmerge (catfile, "", tmpstr, name, "");
#else
      /* this defines catfile like %NLSPATH%\name.%LANG% */

      /* Note that you lose the ability to define multiple dialects of
         a language if you use this method. */

      strcpy (tmpstr, getenv("LANG"));
      len = strlen (tmpstr);

      /* This is a hack to make a compromise between those who want
         to use 2-letter language codes, such as LANG=en, and those
         who want to use the long country codes, such as LANG=en-UK.
         I'll just chop it off at the appropriate length. */

      /* If a 2-letter code, then Cats keeps it.  If longer than
         that, and it is of the form "xx-YY" then I chop it off at
         the "-". Else, I just truncate to 3 characters to make a
         file ext. */

      if (len > 2)
	{
	  if (tmpstr[2] == '-')
	    {
	      /* Assumes LANG=xx-YY, so truncate to xx */
	      tmpstr[2] = '\0';
	    }
	  else
	    {
	      /* Assumes some other value, but truncate to be safe */
	      tmpstr[3] = '\0';
	    }
	}

      fnmerge (catfile, "", getenv("NLSPATH"), name, tmpstr);
#endif /* BUGGY CODE */
    }

#ifndef NDEBUG
  printf ("*catopen* opening catalog %s\n", catfile);
#endif /* NDEBUG */

  pfile = fopen (catfile, "r");
  if (!pfile)
    {
#ifndef NDEBUG
      printf ("*catopen* cannot open %s\n", name);
#endif /* NDEBUG */

      return (-1);
    }

  /* Read the file into memory */

  db_init (10);

  while (fgets (str, MAXSTR, pfile))
    {
      /* Break into parts.  Entries should be of the form:
	 "1.2:This is a message" */

#ifndef NDEBUG
      printf ("*catopen* str is %s\n", str);
#endif /* NDEBUG */

      /* A line that starts with '#' is considered a comment */

      if (str[0] != '#')
	{
	  key = strtok (str, ":");
	  value = strtok (NULL, "\n");

#ifndef NDEBUG
	  printf ("*catopen* key,value -> %s,%s\n", key, value);
#endif /* NDEBUG */

	  db_insert (key, value);
	} /* if comment */
    } /* while */

  fclose (pfile);

  /* Return success */

  catalog = 1;
  return (catalog);
}

void
catclose (nl_catd cat)
{
  /* close a message catalog */

  db_free();
  catalog = 0;
}
