%pure_parser
%{
/*
 * $Id: macro.y,v 1.26 2005/12/03 05:21:02 ronpinkas Exp $
 */

/*
 * Harbour Project source code:
 * Macro compiler YACC rules and actions
 *
 * Copyright 1999 Antonio Linares <alinares@fivetech.com>
 * www - http://www.harbour-project.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA (or visit the web site http://www.gnu.org/).
 *
 * As a special exception, the Harbour Project gives permission for
 * additional uses of the text contained in its release of Harbour.
 *
 * The exception is that, if you link the Harbour libraries with other
 * files to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the Harbour library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the Harbour
 * Project under the name Harbour.  If you copy code from other
 * Harbour Project or Free Software Foundation releases into a copy of
 * Harbour, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for Harbour, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.
 *
 */

/* TODO list
 * 1) jumps longer then 2^15 bytes
 * 2) Change the pcode generated by ::cVar from Self:cVar to QSELF():cVar
 * 3) Support this syntax: nPtr := @Hello()
 */

/* this #define HAVE TO be placed before all #include directives
 */
#define  HB_MACRO_SUPPORT

#include "hbmacro.h"
#include "hbcomp.h"

//JC1: yylex is not threadsafe, we need mutexes
#include "hbstack.h"    // that also includes thread.h

/* Compile using: bison -d -p hb_comp macro.y */

/* NOTE: these symbols are used internally in bison.simple
 */
#undef alloca
#define alloca  hb_xgrab
#undef malloc
#define malloc  hb_xgrab
#undef free
#define free hb_xfree

#define HB_MAX_PENDING_MACRO_EXP 16

static HB_EXPR_PTR s_Pending[ HB_MAX_PENDING_MACRO_EXP ];
static int s_iPending;

/* This is workaround of yyparse() declaration bug in bison.simple
*/

#if !defined(__GNUC__) && !defined(__IBMCPP__)
   #if 0
      /* This makes BCC 551 fail with Bison 1.30, even with the
         supplied harbour.simple file, which makes Bison 1.30 blow.
         [vszakats] */
      void __yy_memcpy ( char*, const char*, unsigned int ); /* to satisfy Borland compiler */
   #endif
#endif

/* yacc/lex related definitions
 */
#undef YYPARSE_PARAM
#define YYPARSE_PARAM HB_MACRO_PARAM    /* parameter passed to yyparse function - it have to be of 'void *' type */
#undef YYLEX_PARAM
#define YYLEX_PARAM   ( (HB_MACRO_PTR)YYPARSE_PARAM ) /* additional parameter passed to yylex */

extern int yyparse( void * );   /* to make happy some purist compiler */
extern void * hb_compFlexNew( HB_MACRO_PTR );
extern void hb_compFlexDelete( void * );

extern void yyerror( char * ); /* parsing error management function */

/* Standard checking for valid expression creation
 */
#define  HB_MACRO_CHECK( pExpr ) \
   if( ! ( HB_MACRO_DATA->status & HB_MACRO_CONT ) ) \
   { \
      hb_compExprDelete( pExpr, HB_MACRO_PARAM ); \
      YYABORT; \
   }

#define HB_MACRO_IFENABLED( pSet, pExpr, flag ) \
   if( HB_MACRO_DATA->supported & (flag) ) \
     { pSet = (pExpr); }\
   else \
   { \
      hb_compExprDelete( (pExpr), HB_MACRO_PARAM ); \
      YYABORT; \
   }

%}

%union                  /* special structure used by lex and yacc to share info */
{
   char *    string;       /* to hold a string returned by lex */
   int       iNumber;      /* to hold a temporary integer number */
   HB_LONG   lNumber;      /* to hold a temporary long number */
   struct
   {
      int    iNumber;      /* to hold a number returned by lex */
      char * szValue;
   } valInteger;
   struct
   {
      HB_LONG   lNumber;   /* to hold a long number returned by lex */
      char *    szValue;
   } valLong;
   struct
   {
      double dNumber;   /* to hold a double number returned by lex */
      /* NOTE: Intentionally using "unsigned char" instead of "BYTE" */
      unsigned char bWidth; /* to hold the width of the value */
      unsigned char bDec; /* to hold the number of decimal points in the value */
      char * szValue;
   } valDouble;
   HB_EXPR_PTR asExpr;
   void * pVoid;        /* to hold any memory structure we may need */
};

%{
/* This must be placed after the above union - the union is
 * typedef-ined to YYSTYPE
 */
int yylex( YYSTYPE *, HB_MACRO_PTR );
%}

%{
#ifdef __WATCOMC__
/* disable warnings for unreachable code */
#pragma warning 13 9
#endif
%}

%token IDENTIFIER NIL NUM_DOUBLE INASSIGN NUM_LONG
%token IIF IF LITERAL TRUEVALUE FALSEVALUE
%token AND OR NOT EQ NE1 NE2 INC DEC ALIASOP SELF
%token LE GE FIELD MACROVAR MACROTEXT
%token PLUSEQ MINUSEQ MULTEQ DIVEQ POWER EXPEQ MODEQ
%token HASHOP
%token CBMARKER
%token BITAND BITOR BITXOR BITSHIFTR BITSHIFTL

/*the lowest precedence*/
/*postincrement and postdecrement*/
%left  POST

/*assigment - from right to left*/
%right INASSIGN
%right PLUSEQ MINUSEQ
%right MULTEQ DIVEQ MODEQ
%right EXPEQ

/*logical operators*/
%right OR
%right AND
%right NOT

/* Bitwise */
%right BITOR
%right BITXOR
%right BITAND

/*relational operators*/
%right '=' EQ NE1 NE2
%right '<' '>' LE GE '$' LIKE MATCH

/* Bit shift */
%right BITSHIFTR BITSHIFTL

/*mathematical operators*/
%right  '+' '-'
%right  '*' '/' '%'
%right  POWER
%right UNARY

/*preincrement and predecrement*/
%right  PRE

/*special operators*/
%right  ALIASOP '&' '@'
%right ','
/*the highest precedence*/

%type <string>  IdentName IDENTIFIER LITERAL MACROVAR MACROTEXT
%type <valDouble>  NUM_DOUBLE
%type <valLong>    NUM_LONG
%type <asExpr>  ByRefArg Argument ArgList BlockExpList BlockVarList BlockNoVar SendId
%type <asExpr>  NumValue NumAlias
%type <asExpr>  NilValue
%type <asExpr>  LiteralValue
%type <asExpr>  CodeBlock
%type <asExpr>  Logical
%type <asExpr>  SelfValue
%type <asExpr>  Array
%type <asExpr>  ArrayAt
%type <asExpr>  Variable VarAlias
%type <asExpr>  MacroVar MacroVarAlias
%type <asExpr>  MacroExpr MacroExprAlias
%type <asExpr>  AliasId AliasVar AliasExpr
%type <asExpr>  VariableAt
%type <asExpr>  FunCall
%type <asExpr>  ObjectData
%type <asExpr>  ObjectMethod
%type <asExpr>  IfInline
%type <asExpr>  ExpList PareExpList PareExpListAlias AsParamList RootParamList
%type <asExpr>  Expression SimpleExpression
%type <asExpr>  EmptyExpression
%type <asExpr>  ExprAssign ExprOperEq ExprPreOp ExprPostOp
%type <asExpr>  ExprMath ExprBool ExprRelation ExprUnary
%type <asExpr>  ExprPlusEq ExprMinusEq ExprMultEq ExprDivEq ExprModEq ExprExpEq
%type <asExpr>  ArrayIndex IndexList
%type <asExpr>  FieldAlias FieldVarAlias
%type <asExpr>  PostOp
%type <asExpr>  WithData WithMethod
%type <asExpr>  Hash HashList
%%

Main : Expression '\n'  {
                           HB_MACRO_DATA->exprType = hb_compExprType( $1 );

                           if( HB_MACRO_DATA->Flags & HB_MACRO_GEN_PUSH )
                           {
                              hb_compExprDelete( hb_compExprGenPush( $1, HB_MACRO_PARAM ), HB_MACRO_PARAM );
                           }
                           else if( HB_MACRO_DATA->Flags & HB_MACRO_GEN_STATEMENT )
                           {
                              hb_compExprDelete( hb_compExprGenStatement( $1, HB_MACRO_PARAM ), HB_MACRO_PARAM );
                           }
                           else
                           {
                              hb_compExprDelete( hb_compExprGenPop( $1, HB_MACRO_PARAM ), HB_MACRO_PARAM );
                           }

                           hb_compGenPCode1( HB_P_ENDPROC, HB_MACRO_PARAM );
                        }
     | Expression       {
                           HB_MACRO_DATA->exprType = hb_compExprType( $1 );

                           if( HB_MACRO_DATA->Flags &  HB_MACRO_GEN_PUSH )
                           {
                              hb_compExprDelete( hb_compExprGenPush( $1, HB_MACRO_PARAM ), HB_MACRO_PARAM );
                           }
                           else if( HB_MACRO_DATA->Flags & HB_MACRO_GEN_STATEMENT )
                           {
                              hb_compExprDelete( hb_compExprGenStatement( $1, HB_MACRO_PARAM ), HB_MACRO_PARAM );
                           }
                           else
                           {
                              hb_compExprDelete( hb_compExprGenPop( $1, HB_MACRO_PARAM ), HB_MACRO_PARAM );
                           }

                           hb_compGenPCode1( HB_P_ENDPROC, HB_MACRO_PARAM );
                        }
     | ByRefArg         {
                           if( ! ( HB_MACRO_DATA->Flags & HB_MACRO_GEN_LIST ) )
                           {
                              HB_TRACE(HB_TR_DEBUG, ("macro -> invalid expression: %s", HB_MACRO_DATA->string));
                              hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );
                              hb_compExprDelete( $1, HB_MACRO_PARAM );
                              YYABORT;
                           }

                           $1 = ( HB_MACRO_DATA->Flags & HB_MACRO_GEN_PARE ) ? hb_compExprNewList( $1 ) : hb_compExprNewArgList( $1 );

                           HB_MACRO_DATA->exprType = hb_compExprType( $1 );

                           hb_compExprDelete( hb_compExprGenPush( $1, HB_MACRO_PARAM ), HB_MACRO_PARAM );

                           hb_compGenPCode1( HB_P_ENDPROC, HB_MACRO_PARAM );
                        }
     | AsParamList      {
                           HB_MACRO_DATA->exprType = hb_compExprType( $1 );

                           hb_compExprDelete( hb_compExprGenPush( $1, HB_MACRO_PARAM ), HB_MACRO_PARAM );

                           hb_compGenPCode1( HB_P_ENDPROC, HB_MACRO_PARAM );
                        }
     | Expression error {
                           HB_TRACE(HB_TR_DEBUG, ("macro -> invalid expression: %s", HB_MACRO_DATA->string));

                           //printf( "Macro: %s\n", HB_MACRO_DATA->string );

                           hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );
                           hb_compExprDelete( $1, HB_MACRO_PARAM );

                           while ( s_iPending )
                           {
                              hb_compExprDelete( s_Pending[ --s_iPending ], HB_MACRO_PARAM );
                           }

                           if( yychar == IDENTIFIER && yylval.string )
                           {
                              hb_xfree( yylval.string );
                              yylval.string = NULL;
                           }

                           YYABORT;
                        }
     | error            {
                           // This case is when error maybe nested in say a CodeBlock.
                           HB_TRACE(HB_TR_DEBUG, ("macro -> invalid syntax: %s", HB_MACRO_DATA->string));

                           //printf( "2-Macro: %s\n", HB_MACRO_DATA->string );

                           hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );

                           while ( s_iPending )
                           {
                              hb_compExprDelete( s_Pending[ --s_iPending ], HB_MACRO_PARAM );
                           }

                           if( yychar == IDENTIFIER && yylval.string )
                           {
                              hb_xfree( yylval.string );
                              yylval.string = NULL;
                           }

                           YYABORT;
                        }
     ;

IdentName  : IDENTIFIER      { $$ = $1; }
           ;

/* Numeric values
 */
NumValue   : NUM_DOUBLE      { $$ = hb_compExprNewDouble( $1.dNumber, $1.bWidth, $1.bDec ); }
           | NUM_LONG        { $$ = hb_compExprNewLong( $1.lNumber ); }
           ;

NumAlias   : NUM_LONG ALIASOP      { $$ = hb_compExprNewLong( $1.lNumber ); }
;

/* NIL value
 */
NilValue   : NIL             { $$ = hb_compExprNewNil(); }
;

/* Literal string value
 */
LiteralValue : LITERAL       { $$ = hb_compExprNewString( $1 ); }
;

/* Logical value
 */
Logical    : TRUEVALUE       { $$ = hb_compExprNewLogical( TRUE ); }
           | FALSEVALUE      { $$ = hb_compExprNewLogical( FALSE ); }
           ;

/* SELF value and expressions
 */
SelfValue  : SELF            { $$ = hb_compExprNewSelf(); }
;

/* Literal array
 */
Array      : '{' ArgList '}'     {
                                   $$ = hb_compExprNewArray( $2 );

                                   if( s_iPending && s_Pending[ s_iPending - 1 ] == $2 )
                                   {
                                      s_iPending--;
                                   }
                                 }
           ;

Hash       : '{' HashList '}'         { $$ = hb_compExprNewFunCall( hb_compExprNewFunName( hb_strdup( "HASH" ) ), $2, HB_MACRO_PARAM ); }
           ;

HashList   : /* nothing => nil */ HASHOP /* nothing => nil */    { $$ = NULL; }
           | Expression HASHOP EmptyExpression                   { $$ = hb_compExprAddListExpr( hb_compExprNewArgList( $1 ), $3 ); }
           | HashList ',' EmptyExpression HASHOP EmptyExpression { $$ = hb_compExprAddListExpr( hb_compExprAddListExpr( $1, $3 ), $5 ); }
           ;

/* Literal array access
 */
ArrayAt     : Array ArrayIndex   { $$ = $2; }
;

/* Variables
 */
Variable    : IDENTIFIER         { $$ = hb_compExprNewVar( $1 ); }
;

VarAlias    : IDENTIFIER ALIASOP      { $$ = hb_compExprNewAlias( $1 ); }
;

/* Macro variables - this can signal compilation errors
 */
MacroVar    : MACROVAR        { $$ = hb_compExprNewMacro( NULL, '&', $1 );
                                HB_MACRO_CHECK( $$ );
                              }
            | MACROTEXT       {  ULONG ulLen = strlen( $1 );
                                 char * szVarName = hb_macroTextSubst( $1, &ulLen );
                                 if( hb_macroIsIdent( szVarName ) )
                                 {
                                    $$ = hb_compExprNewVar( szVarName );
                                    hb_xfree( $1 );
                                    HB_MACRO_CHECK( $$ );
                                 }
                                 else
                                 {
                                    /* invalid variable name
                                     */
                HB_TRACE(HB_TR_DEBUG, ("macro -> invalid variable name: %s", $1));

                                    hb_xfree( $1 );
                                    YYABORT;
                                 }
                              }
;

MacroVarAlias  : MacroVar ALIASOP   { $$ = $1; }
;

/* Macro expressions
 */
MacroExpr  : '&' PareExpList       { $$ = hb_compExprNewMacro( $2, 0, NULL ); }
;

MacroExprAlias : MacroExpr ALIASOP     { $$ = $1; }
;

/* Aliased variables
 */
/* special case: _FIELD-> and FIELD-> can be nested
 */
FieldAlias  : FIELD ALIASOP               { $$ = hb_compExprNewAlias( hb_strdup( "FIELD") ); }
            | FIELD ALIASOP FieldAlias    { $$ = $3; }
            ;

/* ignore _FIELD-> or FIELD-> if a real alias is specified
 */
FieldVarAlias  : FieldAlias VarAlias            { hb_compExprDelete( $1, HB_MACRO_PARAM ); $$ = $2; }
               | FieldAlias NumAlias            { hb_compExprDelete( $1, HB_MACRO_PARAM ); $$ = $2; }
               | FieldAlias PareExpListAlias    { hb_compExprDelete( $1, HB_MACRO_PARAM ); $$ = $2; }
               | FieldAlias MacroVarAlias       { hb_compExprDelete( $1, HB_MACRO_PARAM ); $$ = $2; }
               | FieldAlias MacroExprAlias      { hb_compExprDelete( $1, HB_MACRO_PARAM ); $$ = $2; }
               ;

AliasId     : IDENTIFIER      { $$ = hb_compExprNewVar( $1 ); }
            | MacroVar        { $$ = $1; }
            ;

AliasVar   : NumAlias AliasId          { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | MacroVarAlias AliasId     { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | MacroExprAlias AliasId    { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | PareExpListAlias AliasId  { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | VarAlias AliasId          { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | FieldAlias AliasId        { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           | FieldVarAlias AliasId     { $$ = hb_compExprNewAliasVar( $1, $2 ); }
           ;

/* Aliased expressions
 */
/* NOTE: In the case:
 * alias->( Expression )
 * alias always selects a workarea at runtime
 */
AliasExpr  : NumAlias PareExpList         { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           | VarAlias PareExpList         { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           | MacroVarAlias PareExpList    { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           | MacroExprAlias PareExpList   { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           | PareExpListAlias PareExpList { $$ = hb_compExprNewAliasExpr( $1, $2 ); }
           ;

/* Array expressions access
 */
VariableAt  : NilValue      ArrayIndex    { $$ = $2; }
            | LiteralValue  ArrayIndex    { $$ = $2; }
            | CodeBlock     ArrayIndex    { $$ = $2; }
            | Logical       ArrayIndex    { $$ = $2; }
            | SelfValue     ArrayIndex    { $$ = $2; }
            | Variable      ArrayIndex    { $$ = $2; }
            | AliasVar      ArrayIndex    { $$ = $2; }
            | AliasExpr     ArrayIndex    { $$ = $2; }
            | MacroVar      ArrayIndex    { $$ = $2; }
            | MacroExpr     ArrayIndex    { $$ = $2; }
            | ObjectData    ArrayIndex    { $$ = $2; }
            | ObjectMethod  ArrayIndex    { $$ = $2; }
            | WithData      ArrayIndex    { $$ = $2; }
            | WithMethod    ArrayIndex    { $$ = $2; }
            | FunCall       ArrayIndex    { $$ = $2; }
            | IfInline      ArrayIndex    { $$ = $2; }
            | PareExpList   ArrayIndex    { $$ = $2; }
            ;

/* Function call
 */
FunCall    : IDENTIFIER '(' ArgList ')'   {
                                            $$ = hb_compExprNewFunCall( hb_compExprNewFunName( $1 ), $3, HB_MACRO_PARAM );
                                            HB_MACRO_CHECK( $$ );

                                            if( s_iPending && s_Pending[ s_iPending - 1 ] == $3 )
                                            {
                                               s_iPending--;
                                            }
                                          }
           | MacroVar '(' ArgList ')'     {
                                            $$ = hb_compExprNewFunCall( $1, $3, HB_MACRO_PARAM );
                                            HB_MACRO_CHECK( $$ );

                                            if( s_iPending && s_Pending[ s_iPending - 1 ] == $3 )
                                            {
                                               s_iPending--;
                                            }
                                          }
            | IDENTIFIER '(' error        {
                                            hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );

                                            if( yychar == IDENTIFIER && yylval.string )
                                            {
                                               hb_xfree( yylval.string );
                                               yylval.string = NULL;
                                            }

                                            while ( s_iPending )
                                            {
                                               hb_compExprDelete( s_Pending[ --s_iPending ], HB_MACRO_PARAM );
                                            }

                                            YYABORT;
                                          }
;

ArgList    : Argument                     {
                                            $$ = hb_compExprNewArgList( $1 );

                                            if( s_iPending <= HB_MAX_PENDING_MACRO_EXP )
                                            {
                                               s_Pending[ s_iPending++ ] = $$;
                                            }
                                          }
           | ArgList ',' Argument         { $$ = hb_compExprAddListExpr( $1, $3 ); }
           ;

ByRefArg   : '@' IDENTIFIER               { $$ = hb_compExprNewVarRef( $2 ); }
           ;

Argument   : EmptyExpression           { $$ = $1; }
           | '@' Expression            {
                                          switch( $2->ExprType )
                                          {
                                             case HB_ET_VARIABLE:
                                               $$ = $2;
                                               $$->ExprType = HB_ET_VARREF;
                                               $$->ValType = HB_EV_VARREF;
                                               break;

                                             case HB_ET_ALIASVAR:
                                             {
                                               char *szAlias = $2->value.asAlias.pAlias->value.asSymbol;

                                               if( strcmp( szAlias, "M" ) == 0 || strncmp( szAlias, "MEMVAR", 4 > strlen( szAlias ) ? 4 : strlen( szAlias ) ) == 0 )
                                               {
                                                  $$ = $2->value.asAlias.pVar;

                                                  $2->value.asAlias.pVar = NULL;
                                                  hb_compExprDelete( $2, HB_MACRO_PARAM  );

                                                  if( $$->ExprType == HB_ET_MACRO )
                                                  {
                                                     $$->value.asMacro.SubType = HB_ET_MACRO_VAR_REF;
                                                  }
                                                  else
                                                  {
                                                     $$->ExprType = HB_ET_MEMVARREF;
                                                     $$->ValType = HB_EV_VARREF;
                                                  }
                                               }
                                               break;
                                             }

                                             case HB_ET_FUNCALL:
                                                $$ = $2->value.asFunCall.pFunName;

                                                $2->value.asFunCall.pFunName = NULL;
                                                hb_compExprDelete( $2, HB_MACRO_PARAM  );

                                                $$->ExprType = HB_ET_FUNREF;
                                                $$->ValType = HB_EV_FUNREF;
                                                break;

                                             case HB_ET_SEND:
                                               $$ = $2;
                                               $$->value.asMessage.bByRef = TRUE;
                                               break;

                                             case HB_ET_MACRO:
                                               $$ = $2;
                                               $$->value.asMacro.SubType = HB_ET_MACRO_VAR_REF;
                                               break;

                                             case HB_ET_ARRAYAT:
                                               $$ = $2;
                                               $$->value.asList.bByRef = TRUE;
                                               break;

                                             default:
                                               hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );
                                               $$ = $2;
                                          }
                                       }
           ;

/* Object's instance variable
 */
SendId      : IdentName      { $$ = hb_compExprNewFunName( $1 ); }
            | MacroVar       { $$ = $1; $1->value.asMacro.SubType = HB_ET_MACRO_SYMBOL; }
            | MacroExpr      { $$ = $1; $1->value.asMacro.SubType = HB_ET_MACRO_SYMBOL; }
            ;

ObjectData  : NumValue ':' SendId        { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | NilValue ':' SendId        { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | LiteralValue ':' SendId    { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | CodeBlock ':' SendId       { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | Logical ':' SendId         { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | SelfValue ':' SendId       { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | Array ':' SendId           { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | Hash ':' SendId            { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | ArrayAt ':' SendId         { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | Variable ':' SendId        { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | AliasVar ':' SendId        { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | AliasExpr ':' SendId       { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | MacroVar ':' SendId        { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | MacroExpr ':' SendId       { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | FunCall ':' SendId         { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | PareExpList ':' SendId     { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | VariableAt ':' SendId      { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | ObjectMethod ':' SendId    { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | ObjectData ':' SendId      { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | WithMethod ':' SendId      { $$ = hb_compExprNewSendExp( $1, $3 ); }
            | WithData ':' SendId        { $$ = hb_compExprNewSendExp( $1, $3 ); }
            ;

WithData    : ':' SendId                 {
                                            $$ = hb_compExprNewWithSendExp( $2 );
                                         }
           ;

/* Object's method
 */
ObjectMethod : ObjectData '(' ArgList ')'    {
                                               $$ = hb_compExprNewMethodCall( $1, $3 );

                                               if( s_iPending && s_Pending[ s_iPending - 1 ] == $3 )
                                               {
                                                  s_iPending--;
                                               }
                                             }
            ;

WithMethod   : WithData '(' ArgList ')'  {
                                            $$ = hb_compExprNewWithMethodCall( $1, $3 );
                                         }
             ;

SimpleExpression :
             NumValue
           | NilValue                         { $$ = $1; }
           | LiteralValue                     { $$ = $1; }
           | CodeBlock                        { $$ = $1; }
           | Logical                          { $$ = $1; }
           | SelfValue                        { $$ = $1; }
           | Array                            { $$ = $1; }
           | Hash                             { $$ = $1; }
           | ArrayAt                          { $$ = $1; }
           | AliasVar                         { $$ = $1; }
           | MacroVar                         { $$ = $1; }
           | MacroExpr                        { $$ = $1; }
           | Variable                         { $$ = $1; }
           | VariableAt                       { $$ = $1; }
           | FunCall                          { $$ = $1; }
           | IfInline                         { $$ = $1; }
           | ObjectData                       { $$ = $1; }
           | ObjectMethod                     { $$ = $1; }
           | WithData                         { $$ = $1; }
           | WithMethod                       { $$ = $1; }
           | AliasExpr                        { $$ = $1; }
           | ExprAssign                       { $$ = $1; }
           | ExprOperEq                       { HB_MACRO_IFENABLED( $$, $1, HB_SM_HARBOUR ); }
           | ExprPostOp                       { HB_MACRO_IFENABLED( $$, $1, HB_SM_HARBOUR ); }
           | ExprPreOp                        { HB_MACRO_IFENABLED( $$, $1, HB_SM_HARBOUR ); }
           | ExprUnary                        { $$ = $1; }
           | ExprMath                         { $$ = $1; }
           | ExprBool                         { $$ = $1; }
           | ExprRelation                     { $$ = $1; }
;

Expression : SimpleExpression                 { $$ = $1; HB_MACRO_CHECK( $$ ) }
           | PareExpList                      { $$ = $1; HB_MACRO_CHECK( $$ ) }
;

RootParamList : Argument ',' {
                                if( !(HB_MACRO_DATA->Flags & HB_MACRO_GEN_LIST) )
                                {
                                   HB_TRACE(HB_TR_DEBUG, ("macro -> invalid expression: %s", HB_MACRO_DATA->string));
                                   hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );
                                   hb_compExprDelete( $1, HB_MACRO_PARAM );
                                   YYABORT;
                                }
                             }
                Argument     {
                                HB_MACRO_DATA->iListElements = 1;
                                $$ = hb_compExprAddListExpr( ( HB_MACRO_DATA->Flags & HB_MACRO_GEN_PARE ) ? hb_compExprNewList( $1 ) : hb_compExprNewArgList( $1 ), $4 );
                             }
              ;

AsParamList  : RootParamList             { $$ = $1; }
             | AsParamList ',' Argument  { HB_MACRO_DATA->iListElements++; $$ = hb_compExprAddListExpr( $1, $3 ); }
;

EmptyExpression: /* nothing => nil */    { $$ = hb_compExprNewEmpty(); }
           | Expression
;

/* NOTE: PostOp can be used in one context only - it uses $0 rule
 *    (the rule that stands before PostOp)
 */
PostOp      : INC    { $$ = hb_compExprNewPostInc( $<asExpr>0 ); }
            | DEC    { $$ = hb_compExprNewPostDec( $<asExpr>0 ); }
            ;

/* NOTE: We cannot use 'Expression PostOp' because it caused
 * shift/reduce conflicts
 */
ExprPostOp  : NumValue     PostOp %prec POST  { $$ = $2; }
            | NilValue     PostOp %prec POST  { $$ = $2; }
            | LiteralValue PostOp %prec POST  { $$ = $2; }
            | CodeBlock    PostOp %prec POST  { $$ = $2; }
            | Logical      PostOp %prec POST  { $$ = $2; }
            | SelfValue    PostOp %prec POST  { $$ = $2; }
            | Array        PostOp %prec POST  { $$ = $2; }
            | Hash         PostOp %prec POST  { $$ = $2; }
            | ArrayAt      PostOp %prec POST  { $$ = $2; }
            | Variable     PostOp %prec POST  { $$ = $2; }
            | MacroVar     PostOp %prec POST  { $$ = $2; }
            | MacroExpr    PostOp %prec POST  { $$ = $2; }
            | AliasVar     PostOp %prec POST  { $$ = $2; }
            | AliasExpr    PostOp %prec POST  { $$ = $2; }
            | VariableAt   PostOp %prec POST  { $$ = $2; }
            | PareExpList  PostOp %prec POST  { $$ = $2; }
            | IfInline     PostOp %prec POST  { $$ = $2; }
            | FunCall      PostOp %prec POST  { $$ = $2; }
            | ObjectData   PostOp %prec POST  { $$ = $2; }
            | ObjectMethod PostOp %prec POST  { $$ = $2; }
            | WithData     PostOp %prec POST  { $$ = $2; }
            | WithMethod   PostOp %prec POST  { $$ = $2; }
            ;

ExprPreOp   : INC Expression  %prec PRE      { $$ = hb_compExprNewPreInc( $2 ); }
            | DEC Expression  %prec PRE      { $$ = hb_compExprNewPreDec( $2 ); }
            ;

ExprUnary   : NOT Expression                 { $$ = hb_compExprNewNot( $2 ); }
            | '-' Expression  %prec UNARY    { $$ = hb_compExprNewNegate( $2 ); }
            | '+' Expression  %prec UNARY    { $$ = $2; }
            ;

ExprAssign  : NumValue     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | NilValue     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | LiteralValue INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | CodeBlock    INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | Logical      INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | SelfValue    INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | Array        INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | Hash         INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | ArrayAt      INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | Variable     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | MacroVar     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | MacroExpr    INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | AliasVar     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | AliasExpr    INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | VariableAt   INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | PareExpList  INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | IfInline     INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | FunCall      INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | ObjectData   INASSIGN Expression   { HB_MACRO_IFENABLED( $$, hb_compExprAssign( $1, $3 ), HB_SM_HARBOUR ); }
            | ObjectMethod INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            | WithData     INASSIGN Expression   { HB_MACRO_IFENABLED( $$, hb_compExprAssign( $1, $3 ), HB_SM_HARBOUR ); }
            | WithMethod   INASSIGN Expression   { $$ = hb_compExprAssign( $1, $3 ); }
            ;

ExprPlusEq  : NumValue     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Hash         PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithData     PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithMethod   PLUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewPlusEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprMinusEq : NumValue     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Hash         MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithData     MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithMethod   MINUSEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMinusEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprMultEq  : NumValue     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Hash         MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithData     MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithMethod   MULTEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMultEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprDivEq   : NumValue     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Hash         DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithData     DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithMethod   DIVEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewDivEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprModEq   : NumValue     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Hash         MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithData     MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithMethod   MODEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewModEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprExpEq   : NumValue     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | NilValue     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | LiteralValue EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | CodeBlock    EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Logical      EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | SelfValue    EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Array        EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Hash         EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ArrayAt      EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | Variable     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroVar     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | MacroExpr    EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasVar     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | AliasExpr    EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | VariableAt   EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | PareExpList  EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | IfInline     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | FunCall      EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectData   EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | ObjectMethod EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithData     EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            | WithMethod   EXPEQ Expression   { $$ = hb_compExprSetOperand( hb_compExprNewExpEq( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprOperEq  : ExprPlusEq      { $$ = $1; }
            | ExprMinusEq     { $$ = $1; }
            | ExprMultEq      { $$ = $1; }
            | ExprDivEq       { $$ = $1; }
            | ExprModEq       { $$ = $1; }
            | ExprExpEq       { $$ = $1; }
            ;

ExprMath    : Expression '+'       Expression { $$ = hb_compExprSetOperand( hb_compExprNewPlus( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '-'       Expression { $$ = hb_compExprSetOperand( hb_compExprNewMinus( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '*'       Expression { $$ = hb_compExprSetOperand( hb_compExprNewMult( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '/'       Expression { $$ = hb_compExprSetOperand( hb_compExprNewDiv( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '%'       Expression { $$ = hb_compExprSetOperand( hb_compExprNewMod( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression POWER     Expression { $$ = hb_compExprSetOperand( hb_compExprNewPower( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression BITAND    Expression { $$ = hb_compExprSetOperand( hb_compExprNewBitAnd( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression BITOR     Expression { $$ = hb_compExprSetOperand( hb_compExprNewBitOr( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression BITXOR    Expression { $$ = hb_compExprSetOperand( hb_compExprNewBitXOr( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression BITSHIFTR Expression { $$ = hb_compExprSetOperand( hb_compExprNewBitShiftR( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression BITSHIFTL Expression { $$ = hb_compExprSetOperand( hb_compExprNewBitShiftL( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprBool    : Expression AND Expression   { $$ = hb_compExprSetOperand( hb_compExprNewAnd( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression OR  Expression   { $$ = hb_compExprSetOperand( hb_compExprNewOr( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ExprRelation: Expression EQ    Expression   { $$ = hb_compExprSetOperand( hb_compExprNewEQ( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '<'   Expression   { $$ = hb_compExprSetOperand( hb_compExprNewLT( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '>'   Expression   { $$ = hb_compExprSetOperand( hb_compExprNewGT( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression LE    Expression   { $$ = hb_compExprSetOperand( hb_compExprNewLE( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression GE    Expression   { $$ = hb_compExprSetOperand( hb_compExprNewGE( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression NE1   Expression   { $$ = hb_compExprSetOperand( hb_compExprNewNE( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression NE2   Expression   { $$ = hb_compExprSetOperand( hb_compExprNewNE( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '$'   Expression   { $$ = hb_compExprSetOperand( hb_compExprNewIN( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression '='   Expression   { $$ = hb_compExprSetOperand( hb_compExprNewEqual( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression LIKE  Expression   { $$ = hb_compExprSetOperand( hb_compExprNewLike( $1 ), $3, HB_MACRO_PARAM ); }
            | Expression MATCH Expression   { $$ = hb_compExprSetOperand( hb_compExprNewMatch( $1 ), $3, HB_MACRO_PARAM ); }
            ;

ArrayIndex : IndexList ']'                   { $$ = $1; }
           ;

/* NOTE: $0 represents the expression before ArrayIndex
 *    Don't use ArrayIndex in other context than as an array index!
 */
IndexList  : '[' Expression               { $$ = hb_compExprNewArrayAt( $<asExpr>0, $2, HB_MACRO_PARAM ); }
           | IndexList ',' Expression     { $$ = hb_compExprNewArrayAt( $1, $3, HB_MACRO_PARAM ); }
           | IndexList ']' '[' Expression { $$ = hb_compExprNewArrayAt( $1, $4, HB_MACRO_PARAM ); }
           ;

CodeBlock  : '{' CBMARKER
                  {
                    $$ = hb_compExprNewCodeBlock();

                    if( s_iPending <= HB_MAX_PENDING_MACRO_EXP )
                    {
                       s_Pending[ s_iPending++ ] = $$;
                    }
                  }
             BlockNoVar CBMARKER BlockExpList '}'
                  {
                    $$ = $<asExpr>3;

                    if( s_iPending && s_Pending[ s_iPending - 1 ] == $$ )
                    {
                       s_iPending--;
                    }
                  }
           | '{' CBMARKER
                  {
                    $$ = hb_compExprNewCodeBlock();

                    if( s_iPending <= HB_MAX_PENDING_MACRO_EXP )
                    {
                       s_Pending[ s_iPending++ ] = $$;
                    }
                  }
             BlockVarList CBMARKER BlockExpList '}'
                  {
                    $$ = $<asExpr>3;

                    if( s_iPending && s_Pending[ s_iPending - 1 ] == $$ )
                    {
                       s_iPending--;
                    }
                  }
           ;

/* NOTE: This uses $-2 then don't use BlockExpList in other context
 */
BlockExpList : Expression                { $$ = hb_compExprAddListExpr( $<asExpr>-2, $1 ); }
           | BlockExpList ',' Expression { $$ = hb_compExprAddListExpr( $<asExpr>-2, $3 ); }
           ;

/* NOTE: This is really not needed however it allows the use of $-2 item
 * in BlockExpList to refer the same rule defined in Codeblock
 */
BlockNoVar : /* empty list */    { $$ = NULL; }
;

BlockVarList : IDENTIFIER                 { $$ = hb_compExprCBVarAdd( $<asExpr>0, $1, HB_MACRO_PARAM ); }
           | BlockVarList ',' IDENTIFIER  { $$ = hb_compExprCBVarAdd( $<asExpr>0, $3, HB_MACRO_PARAM ); HB_MACRO_CHECK( $$ ); }
           ;

ExpList    : '(' EmptyExpression          { $$ = hb_compExprNewList( $2 ); }
           | ExpList ',' EmptyExpression  { $$ = hb_compExprAddListExpr( $1, $3 ); }
;

PareExpList : ExpList ')'                 { $$ = $1; }
            | ExpList error               {
                                            hb_macroError( EG_SYNTAX, HB_MACRO_PARAM );

                                            hb_compExprDelete( $1, HB_MACRO_PARAM );

                                            while ( s_iPending )
                                            {
                                               hb_compExprDelete( s_Pending[ --s_iPending ], HB_MACRO_PARAM );
                                            }

                                            if( yychar == IDENTIFIER && yylval.string )
                                            {
                                               hb_xfree( yylval.string );
                                               yylval.string = NULL;
                                            }

                                            YYABORT;
                                          }
;

PareExpListAlias : PareExpList ALIASOP     { $$ = $1; }
;

IfInline   : IIF '(' Expression ',' EmptyExpression ','
             { $<asExpr>$ = hb_compExprAddListExpr( hb_compExprNewList( $3 ), $5 ); }
             EmptyExpression ')'
             { $$ = hb_compExprNewIIF( hb_compExprAddListExpr( $<asExpr>7, $8 ) ); }

           | IF '(' Expression ',' EmptyExpression ','
             { $<asExpr>$ = hb_compExprAddListExpr( hb_compExprNewList( $3 ), $5 ); }
             EmptyExpression ')'
             { $$ = hb_compExprNewIIF( hb_compExprAddListExpr( $<asExpr>7, $8 ) ); }

           | IIF '(' Expression ',' EmptyExpression
             { $<asExpr>$ = hb_compExprAddListExpr( hb_compExprNewList( $3 ), $5 ); }
             ')'
             { $$ = hb_compExprNewIIF( hb_compExprAddListExpr( $<asExpr>6, hb_compExprNew( HB_ET_NONE ) ) ); }

           | IF '(' Expression ',' EmptyExpression
             { $<asExpr>$ = hb_compExprAddListExpr( hb_compExprNewList( $3 ), $5 ); }
             ')'
             { $$ = hb_compExprNewIIF( hb_compExprAddListExpr( $<asExpr>6, hb_compExprNew( HB_ET_NONE ) ) ); }
           ;

%%

#ifdef __WATCOMC__
/* enable warnings for unreachable code */
#pragma warning 13 1
#endif

/*
 ** ------------------------------------------------------------------------ **
 */

int hb_macroYYParse( HB_MACRO_PTR pMacro )
{
   int iResult;
   void * lexBuffer;

   #ifdef HB_THREAD_SUPPORT
      HB_CRITICAL_LOCK( hb_macroMutex );
   #endif

   // Reset
   s_iPending = 0;

   lexBuffer = hb_compFlexNew( pMacro );

   pMacro->status = HB_MACRO_CONT;
   /* NOTE: bison requires (void *) pointer
    */
   iResult = yyparse( ( void * ) pMacro );

   hb_compFlexDelete( lexBuffer );

   #ifdef HB_THREAD_SUPPORT
      HB_CRITICAL_UNLOCK( hb_macroMutex );
   #endif

   return iResult;
}

/* ************************************************************************* */

void yyerror( char * s )
{
   HB_SYMBOL_UNUSED( s );
}
