/****************************************************************************
*
*                            Open Watcom Project
*
*    Portions Copyright (c) 1983-2002 Sybase, Inc. All Rights Reserved.
*
*  ========================================================================
*
*    This file contains Original Code and/or Modifications of Original
*    Code as defined in and that are subject to the Sybase Open Watcom
*    Public License version 1.0 (the 'License'). You may not use this file
*    except in compliance with the License. BY USING THIS FILE YOU AGREE TO
*    ALL TERMS AND CONDITIONS OF THE LICENSE. A copy of the License is
*    provided with the Original Code and Modifications, and is also
*    available at www.sybase.com/developer/opensource.
*
*    The Original Code and all software distributed under the License are
*    distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
*    EXPRESS OR IMPLIED, AND SYBASE AND ALL CONTRIBUTORS HEREBY DISCLAIM
*    ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF
*    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR
*    NON-INFRINGEMENT. Please see the License for the specific language
*    governing rights and limitations under the License.
*
*  ========================================================================
*
* Description:  WHEN YOU FIGURE OUT WHAT THIS FILE DOES, PLEASE
*               DESCRIBE IT HERE!
*
****************************************************************************/


// MSGENCOD -- encode messages to be generated by GML program
//
// 91/05/24 -- J.W.Welch        -- defined
// 91/05/29 -- E.Sutherland     -- start on project
// 91/06/28 -- E.Sutherland     -- complete with encoding/decoding words
// 91/07/05 -- A.F.Scian        -- removed use of flt-pt in statistics
//                               - cleaned up output of stats
// 91/07/09 -- E.Sutherland     -- added non-overlapping phrase encoding
// 91/10/03 -- J.W.Welch        -- cleaned up, use ENCODIO.C
// 92/03/03 -- J.W.Welch        -- output maximum for message size
// 93/03/05 -- J.W.Welch        -- extend for level vector
// 93/03/30 -- A.F.Scian        -- used unsigned long for byte counts
// 93/11/23 -- A.F.Scian        -- accept 2 digit warning levels
// 95/01/18 -- J.W.Welch        -- emit '\'' instead of '''

#include <stdio.h>
#include <process.h>
#include <direct.h>
#include <ctype.h>
#define process_text not_used
#include "msgencod.h"
#undef  process_text
#include "watlibio.h"

#define ENC_BIT   0x80
#define LARGE_BIT 0x40

typedef struct word WORD;
typedef struct message MESSAGE;

typedef enum                    // TYPES OF MESSAGES
{   MSG_TYPE_ERROR              // - error
,   MSG_TYPE_WARNING            // - warning
,   MSG_TYPE_INFO               // - informational
,   MSG_TYPE_ANSI               // - warning when extensions, error otherwise
,   MSG_TYPE_ANSIERR            // - ignored when extensions, error otherwise
,   MSG_TYPE_ANSIWARN           // - ignored when extensions, warn otherwise
,   MSG_TYPE_ANSICOMP           // - error, can be changed to warning when extensions
,   MSG_TYPE_EXTWARN            // - warn when extensions, ignored otherwise
} MSG_TYPE;


struct word                     // WORD ENTRY
{   WORD *next;                 // - next in ring
    unsigned short code;        // - encoding byte
    unsigned short count;       // - occurence count
    char text[1];               // - text for word
};

struct message                  // MESSAGE/PHRASE ENTRY
{   MESSAGE *next;              // - next in ring
    unsigned short number;      // - message number within group
    unsigned short count;       // - occurence count
    unsigned short codes;       // - number of words/phrases
    unsigned char level;        // - error/warning level
    MSG_TYPE type;              // - type of message
    WORD *words[1];             // - vector of WORD pointers
};

static WORD *RingWords;         // - Ring of Words
static WORD *RingGroups;        // - Ring of Groups
static MESSAGE *RingMessages;   // - Ring of Messages
static MESSAGE *RingPhrases;    // - Ring of Phrases
static unsigned long enc_words; // - number of words encoded
static unsigned long enc_phrases;//- number of phrases encoded
static unsigned msg_size;       // - maximum message size
static unsigned long lenwords;  // - cum. string length
static unsigned long numwords;  // - number of words in group
static char delim[3] = "  ";    // - delimiter for output
static unsigned long space_read_in;// - size of input file
static unsigned long space_put_out;// - size of output file

                                // IO FILES
static char *source;            // - source file
static char *messages;          // - C output file
static char *codes;             // - header output file
static char *levels;            // - levels output file
static FILE_DEFN *filepres;     // - source file, when opened

typedef enum
{   REC_GRP                     // - :MSGGRP token
,   REC_SYM                     // - :MSGSYM token
,   REC_TXT                     // - :MSGTXT token
,   REC_MOD                     // - modifiers
,   REC_EOF                     // - end of file
,   MAX_REC                     // - maximum record
,   REC_NONE                    // - no record read in
} REC_TYPE;                     // TYPE OF RECORD

typedef enum
{   START_LINE                  // - beginning of program
,   FOUND_EOF                   // - found EOF
,   GROUP_LINE                  // - :MSGGRP found
,   SYM_GROUP                   // - :MSGSYM within a group
,   TEXT_GROUP                  // - :MSGTXT within a group
,   SYM_NO_GRP                  // - :MSGSYM first in file
,   TEXT_NO_GRP                 // - :MSGTXT without a group
,   MOD_GROUP                   // - modifier within group
,   MOD_NO_GRP                  // - modifier without group
,   MAX_STATE                   // - highest legal state possible
,   PRE_EOF                     // - premature end-of-dile
,   GRP_NOT_FRST                // - Group not first file
,   MISPL_TXT                   // - misplaced text token
,   MISPL_SYM                   // - misplaced symbol token
,   MISPL_MOD                   // - misplaced modifier
,   DUP_MOD                     // - more than one modifier
}STATE;                         // STATE OF LOGIC

static char state_table[MAX_STATE][MAX_REC] =
//  GROUP      SYMBOL     TEXT        MODIFIER   EOF
{ GROUP_LINE  ,SYM_NO_GRP,MISPL_TXT  ,MISPL_MOD ,PRE_EOF   // START_LINE
, FOUND_EOF   ,FOUND_EOF ,FOUND_EOF  ,FOUND_EOF ,FOUND_EOF // FOUND_EOF
, GROUP_LINE  ,SYM_GROUP ,MISPL_TXT  ,MISPL_MOD ,PRE_EOF   // GROUP_LINE
, MISPL_SYM   ,MISPL_SYM ,TEXT_GROUP ,MISPL_MOD ,PRE_EOF   // SYM_GROUP
, GROUP_LINE  ,SYM_GROUP ,MISPL_TXT  ,MOD_GROUP ,FOUND_EOF // TEXT_GROUP
, GRP_NOT_FRST,MISPL_SYM ,TEXT_NO_GRP,MISPL_MOD ,PRE_EOF   // SYM_NO_GRP
, GRP_NOT_FRST,SYM_NO_GRP,MISPL_TXT  ,MOD_NO_GRP,FOUND_EOF // TEXT_NO_GRP
, GROUP_LINE  ,SYM_GROUP ,MISPL_TXT  ,DUP_MOD   ,FOUND_EOF // MOD_GROUP
, GRP_NOT_FRST,SYM_NO_GRP,MISPL_TXT  ,DUP_MOD   ,FOUND_EOF // MOD_NO_GRP
};


static void scan_name(          // SCAN NAME FROM TEXT LINE
    const char **scanner,       // - line scanner
    const char **word,          // - word on line
    unsigned *length )          // - length of name
{
    const char *cptr;           // - line pointer
    const char *name;           // - name on line

    cptr = scan_over_white_space( *scanner );
    name = cptr;
    cptr = scan_over_black_space( cptr );
    *length = cptr - name;
    *word = name;
    *scanner = cptr;
}


static int write_c_delim()      // WRITE DELIMITER ( '{' OR ',' ) TO C FILE
{
    int retn;                   // - return

    retn = write_c_str( delim );
    delim[0] = ',';
    return( retn );
}


static void substitute_phrases( // SUBSTITUTE PHRASES IN A SENTENCE
    WORD **sentence,            // - sentence
    unsigned *count )           // - addr( # phrases )
{
    unsigned words_left;        // - # words left to scan
    unsigned words_sent;        // - # words in sentence
    unsigned phrase_size;       // - size of current phrase
    MESSAGE *phrase;            // - current phrase
    WORD **word;                // - addr( word in phrase )
    WORD **sent;                // - scan position in "sentence"

    IF( RingPhrases != NULL )
        words_sent = *count;
        LOOP :: SUB_LOOP
            sent = sentence;
            GUESS :: NO_SUBST
                FOR( words_left = words_sent
                   ; words_left > 0
                   ; --words_left, ++sent )
                    phrase = RingPhrases;
                    LOOP
                        phrase = phrase->next;
                        phrase_size = phrase->codes;
                        QUITIF( ( phrase_size <= words_left )
                              &&( 0 ==
                                 memcmp( sent
                                       , phrase->words
                                       , sizeof( WORD * ) * phrase_size ) )
                          ) :: NO_SUBST
                    UNTIL( phrase == RingPhrases );
                ENDFOR
                QUIT :: SUB_LOOP
            ADMIT :: DO_SUBST
                words_left -= phrase_size - 1;
                words_sent -= phrase_size - 1;
                memcpy( sent + 1
                      , sent + phrase_size
                      , sizeof( WORD * ) * ( words_left - 1 ) );
                *sent = (WORD *)phrase;
                FOR( word = phrase->words
                   ; phrase_size > 0
                   ; -- phrase_size, ++ word )
                    -- (*word)->count;
                ENDFOR
                ++ phrase->count;
            ENDGUESS
        ENDLOOP
        *count = words_sent;
    ENDIF
}


static int alloc_ring(          // ALLOCATE RING SPACE
    void **hdr ,                // - ring header
    size_t size ,               // - size to be allocated
    void **element )            // - element allocated
{
    int retn;                   // - return value

    *element = (void *)_ring_alloc( (void ***)hdr, size );
    IF( *element == NULL )
        retn = msgerr( "Out Of Memory" );
    ELSE
        retn = 0;
    ENDIF
    return( retn );
}


static int print_name(          // - SPLIT NAME AND PRINT
    const char *name )          // - name to be parsed
{
    int retn;                   // - return value

    GUESS :: NO_ERRS
        LOOP
            retn = write_c_delim();
            QUITIF( retn != 0 ) :: NO_ERRS
            retn = write_c_chr( '\'' );
            QUITIF( retn != 0 ) :: NO_ERRS
            IF( *name == '\'' )
                retn = write_c_chr( '\\' );
                QUITIF( retn != 0 ) :: NO_ERRS
            ENDIF
            retn = write_c_chr( *name );
            QUITIF( retn != 0 ) :: NO_ERRS
            IF( *name == '\\' )
                retn = write_c_chr( *name );
                QUITIF( retn != 0 ) :: NO_ERRS
            ENDIF
            retn = write_c_chr( '\'' );
            QUITIF( retn != 0 ) :: NO_ERRS
            ++name;
            QUITIF( *name == '\0' );
        ENDLOOP
        retn = write_c_eol();
    ENDGUESS
    return( retn );
}


static int beg_part(            // PUT OUT STARTING PART
    const char *text )          // - text for start
{
    delim[0] = '{';
    return( write_c_line( text ) );
}


static int end_part()           // PUT SPACE BETWEEN PARTS
{
    int retn;                   // - return value

    GUESS :: NO_ERRS
        retn = write_c_line( "};" );
        QUITIF( retn != 0 );
        retn = write_c_eol();
    ENDGUESS
    return( retn );
}


static int print_group_name(    // PRINT NAME
    WORD *curr )                // - current value
{
    return( print_name( curr->text ) );
}


static int print_group_prefix() // PRINT GROUP PREFIXES
{
    int retn;                   // - return value

    GUESS :: NO_ERRS
      retn = beg_part( "MSG_SCOPE unsigned char MSG_MEM group_prefix[] = {" );
      QUITIF( retn != 0 );
      retn = _ring_walk_int( (void **)RingGroups , print_group_name , 0 );
      QUITIF( retn != 0 ) :: NO_ERRS;
      retn = end_part();
    ENDGUESS
    return( retn );
}


static int write_raw(           // write text in unencoded format
    const char *curr )          // - current word
{
    int retn;                   // - return value
    int length;                 // - length of a word
    char len[5];                // - string of length

    GUESS :: NO_ERRS
        length = strlen( curr );
        space_put_out += length + 1;
        utoa( length , len , 10 );
        retn = write_c_delim();
        QUITIF( retn != 0 );
        retn = write_c_str( len );
        QUITIF( retn != 0 );
        retn = print_name( curr );
    ENDGUESS
    return( retn );
}


static int print_msg_word(      // PRINT ENCODING FOR A WORD/PHRASE
    WORD *word )                // - word in sentence
{
    int retn;                   // - return value

    IF( word->code & ENC_BIT )
        retn = write_encode( word->code );
    ELSE
        retn = write_raw( word->text );
    ENDIF
    return( retn );
}


static int print_msg_words(     // PRINT WORDS IN A MESSAGE
    WORD **text,                // - text (words)
    unsigned count,             // - number of words
    unsigned number,            // - number in comment
    const char *comment )       // - comment
{
    int retn;                   // - return value
    char buf[10];

    GUESS :: NO_ERRS
        utoa( number, buf, 10 );
        retn = write_c_strs_eol( "/" "* "
                               , comment
                               , buf
                               , " *" "/"
                               , NULL );
        QUITIF( retn != 0 );
        FOR( ; count > 0; -- count )
            retn = print_msg_word( *text++ );
            QUITIF( retn != 0 ) :: NO_ERRS
        ENDFOR
        retn = write_c_line( ", 0" );
        space_put_out += 1;
    ENDGUESS
    return( retn );
}


static int print_phrase_encod(  // PRINT PHRASE
    MESSAGE *curr )             // - current phrase
{
    return( print_msg_words( curr->words
                           , curr->codes
                           , ( curr->number & 0x3f )
                           + ( ( curr->number & 0xff00 ) >> 2 )
                           , "PHRASE " ) );
}


static int print_sentence(      // PRINT SENTENCE IN A MESSAGE
    MESSAGE *curr )             // - current message
{
    return( print_msg_words( curr->words
                           , curr->codes
                           , curr->number
                           , "MESSAGE " ) );
}


static int print_text_encod(    // PRINT ENCODED TEXT
    WORD *curr )                // - current record
{
    int retn;                   // - return value

    IF( curr->code & ENC_BIT )
        retn = write_raw( curr->text );
    ELSE
        retn = 0;
    ENDIF
    return( retn );
}


static int write_encode(        // WRITE ENCODED PARTS
    unsigned int code )         // - given code
{
    char code1[10];             // - byte[1] of code
    char code2[10];             // - byte[2] of code (optional)
    int retn;                   // - return value

    GUESS :: NO_ERRS
        retn = write_c_delim();
        QUITIF( retn != 0 );
        space_put_out += 1;
        stxcpy( code1 , code & 0x3f );
        IF( code & LARGE_BIT )
            space_put_out += 1;
            stxcpy( code2 , code >> 8 );
            retn = write_c_strs_eol( "ENC_BIT | "
                                   , "LARGE_BIT | "
                                   , code1
                                   , ", "
                                   , code2
                                   , NULL );
        ELSE
            retn = write_c_strs_eol( "ENC_BIT | "
                                   , code1
                                   , NULL );
        ENDIF
    ENDGUESS
    return( retn );
}


static int print_encod_text()   // PRINT ENCODED TEXT
{
    int retn;                   // - return value

    GUESS :: NO_ERRS
        retn = beg_part( "MSG_SCOPE unsigned char MSG_MEM encoded_text[] =" );
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = _ring_walk_int( (void ***)RingWords, print_text_encod, 0 );
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = _ring_walk_int( (void ***)RingPhrases, print_phrase_encod, 0 );
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = _ring_walk_int( (void **)RingMessages, &print_sentence, 0 );
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = end_part();
    ENDGUESS
    return( retn );
}


static int print_word_count()   // PRINT WORD COUNT
{
    int retn;                   // - return value
    char encwrds[10];           // - string of number of encoded words

    GUESS :: NO_ERRS
        utoa( enc_words , encwrds , 10 );
        retn = write_c_strs_eol
                        ( "MSG_SCOPE unsigned short MSG_MEM word_count = "
                        , encwrds
                        , ";"
                        , NULL );
        QUITIF( retn != 0 );
        retn = write_c_eol();
    ENDGUESS
    return( retn );
}


static int print_offset()       // PRINT OFFSET OF WORD, PHRASE, OR MESSAGE
{
    int retn;                   // - return value
    char len[10];               // - length in text form

    GUESS :: NO_ERRS
        space_put_out += 2;
        utoa( lenwords , len , 10 );
        retn = write_c_delim();
        QUITIF( retn != 0 );
        retn = write_c_line( len );
    ENDGUESS
    return( retn );
}


static int prt_msg_offset(      // PRINT OFFSET OF A MESSAGE
    MESSAGE *msg )              // - the message
{
    int retn;                   // - return value
    WORD *word;                 // - current word/phrase
    WORD **wordp;               // - points in vector of words
    unsigned count;             // - counts words

    GUESS :: NO_ERRS
        retn = print_offset();
        QUITIF( retn != 0 );
        FOR( wordp = msg->words, count = msg->codes
           ; count > 0
           ; --count )
            word = *wordp++;
            IF( word->code & ENC_BIT )
                IF( word->code & LARGE_BIT )
                    ++ lenwords;
                ENDIF
            ELSE
                lenwords += strlen( word->text );
            ENDIF
            ++ lenwords;
        ENDFOR
        ++ lenwords;
    ENDGUESS
    return( retn );
}


static int prt_enc_word_offset( // PRINT OFFSET OF ENCODED WORD
    WORD *curr )                // - current record
{
    int retn;                   // - return value

    IF( curr->code & ENC_BIT )
        retn = print_offset();
        lenwords += strlen( curr->text ) + 1;
    ELSE
        retn = 0;
    ENDIF
    return( retn );
}


static int print_text_table()   // PRINT TEXT TABLE IN FILE
{
    int retn;                   // - return value

    GUESS :: NO_ERRS
        retn = beg_part( "MSG_SCOPE unsigned short MSG_MEM text_table[] = " );
        QUITIF( retn != 0 );
        lenwords = 0;
        retn = _ring_walk_int( (void **)RingWords , prt_enc_word_offset, 0 );
        QUITIF( retn != 0 );
        retn = _ring_walk_int( (void **)RingPhrases , prt_msg_offset, 0 );
        QUITIF( retn != 0 );
        retn = _ring_walk_int( (void **)RingMessages , prt_msg_offset, 0 );
        QUITIF( retn != 0 );
        retn = end_part();
    ENDGUESS
    return( retn );
}


static int print_group_num(     // PRINT GROUP NUMBER
    MESSAGE *curr )             // - current message
{
    int retn;                   // - return value
    char putwrds[5];            // - string of numwords;

    GUESS :: NO_ERRS
        retn = 0;
        QUITIF( curr->number != 0 )
        utoa( numwords , putwrds , 10 );
        retn = write_c_delim();
        QUITIF( retn != 0 ) :: NO_ERRS
        retn = write_c_line( putwrds );
    ENDGUESS
    ++numwords;
    return( retn );
}


static int print_group_table(   // PRINT GROUP_TABLE
    int index)                  // - group indexer
{
    int retn;                   // - return value

    GUESS :: NO_ERRS
        retn = beg_part( "MSG_SCOPE unsigned short MSG_MEM group_table[]= " );
        QUITIF( retn != 0 );
        numwords = index;
        retn = _ring_walk_int( (void **)RingMessages , print_group_num, 0 );
        QUITIF( retn != 0 );
        retn = end_part();
    ENDGUESS
    return( retn );
}


static void encode_phrases()    // ENCODE PHRASES
{
    unsigned number;            // - phrase number
    MESSAGE *phrase;            // - current phrase

    phrase = RingPhrases;
    IF( phrase != NULL )
        number = enc_words;
        LOOP
            phrase = phrase->next;
            phrase->number = ( number & 0x3f )
                           | ( ( number & 0xffc0 ) << 2 )
                           | ENC_BIT;
            IF( number++ >= 64 )
                phrase->number |= LARGE_BIT;
            ENDIF
        UNTIL( phrase == RingPhrases );
    ENDIF
}


static void encode_dec(         // DECIDE ENCODE
    WORD *curr )                // - current record
{
    IF( ( enc_words < 64 )
      &&( ( ( curr->count - 1 )*( strlen( curr->text ) ) ) > 3 ) )
        curr->code = enc_words + ENC_BIT;
        ++enc_words;
    ELSEIF ( ( enc_words >= 64 )&&
     ( ( ( curr->count - 1 )*( strlen( curr->text ) - 1 ) ) > 4 ) )
        curr->code = ( enc_words & 0x3f ) + ( (enc_words & 0x3fc0 ) << 2 ) +
                       ENC_BIT  + LARGE_BIT;
        ++enc_words;
    ENDIF
}


static void encode_words()      // ENCODE WORDS
{
    enc_words = 0;
    _ring_walk( (void **)RingWords , encode_dec );
}


static int compare_word_counts( // COMPARE WORDS, USING COUNTS, FOR SORT
    WORD *op1,                  // - word(1) (current)
    WORD *op2 )                 // - word(2) (being inserted)
{
    return( op2->count - op1->count );
}


static void sort_words()        // SORT WORDS
{
    _ring_sort( (void ***)&RingWords, &compare_word_counts );
}


static int compare_phr_counts(  // COMPARE WORDS, USING COUNTS, FOR SORT
    MESSAGE *op1,               // - phrase(1) (current)
    MESSAGE *op2 )              // - phrase(2) (being inserted)
{
    return( op2->count - op1->count );
}


static void sort_phrases()      // SORT WORDS
{
    _ring_sort( (void ***)&RingPhrases, &compare_phr_counts );
}


static int write_define(        // WRITE DEFINE STATEMENTS
    int (*writer)               // - writer
        ( const char * ) )      // - - string
{
    unsigned fmt_size;
    int retn;                   // - return
    char msg[80];               // - message buffer

    GUESS :: NO_ERRS
        retn = (*writer)( "#define ENC_BIT 0x80" );
        QUITIF( retn != 0 );
        retn = (*writer)( "#define LARGE_BIT 0x40" );
        QUITIF( retn != 0 );
        // messages involve an extra ' ' at the end
        // so we'll add 16 and round up to mos 16 to be absolutely safe
        fmt_size = ( ( msg_size + 16 ) + 0x0f ) & ~ 0x0f;
        stucpy( stpcpy( msg, "#define MAX_MSG " ), fmt_size );
        retn = (*writer)( msg );
        QUITIF( retn != 0 );
        retn = (*writer)( "" );
        QUITIF( retn != 0 );
    ENDGUESS
    return( retn );
}


static int create_output()      // CREATE OUTPUT
{
    int retn;                   // - return value

    GUESS :: NO_OPEN
        retn = open_c_file( messages );
        QUITIF( retn == 0 );
    ADMIT :: ENCODE_TEXT
        sort_words();
        encode_words();
        sort_phrases();
        encode_phrases();
        write_define( &write_c_line );
        retn = print_group_table( enc_words + enc_phrases );
        QUITIF( retn != 0 );
        retn = print_text_table();
        QUITIF( retn != 0 );
        retn = print_word_count( );
        QUITIF( retn != 0 );
        retn = print_encod_text();
        QUITIF( retn != 0 );
        IF( RingGroups != NULL )
            retn = print_group_prefix();
            QUITIF( retn != 0 );
        ENDIF
        retn = close_c_file();
    ENDGUESS
    return( retn );
}


static void create_level(       // CREATE A LEVEL ENTRY
    MESSAGE *msg )              // - current entry
{
    char buffer[16];            // - record buffer

    buffer[0] = delim[0];
    delim[0] = ',';
    stucpy( buffer+1, ( msg->type << 4 ) + msg->level );
    write_h_line( buffer );
}


static int create_levels()      // CREATE LEVELS FILE
{
    int retn;                   // - return value

    GUESS :: NO_OPEN
        retn = open_h_file( levels );
        QUITIF( retn == 0 );
    ADMIT :: LEVELS
        write_h_line( "typedef enum" );
        write_h_line( "{   MSG_TYPE_ERROR" );
        write_h_line( ",   MSG_TYPE_WARNING" );
        write_h_line( ",   MSG_TYPE_INFO" );
        write_h_line( ",   MSG_TYPE_ANSI" );
        write_h_line( ",   MSG_TYPE_ANSIERR" );
        write_h_line( ",   MSG_TYPE_ANSIWARN" );
        write_h_line( ",   MSG_TYPE_ANSICOMP" );
        write_h_line( ",   MSG_TYPE_EXTWARN" );
        write_h_line( "} MSG_TYPE;" );
        write_h_line( " " );
        write_h_line( "static unsigned char msg_level[] =" );
        delim[0] = '{';
        _ring_walk( (void**)RingMessages, &create_level );
        write_h_line( "};" );
        retn = close_h_file();
    ENDGUESS
    return retn;
}


static int process_group(       // GRP OPTION FOUND
    const char *scanner )       // - line scanner
{
    int retn;                   // - return value
    char *name;                 // - group name
    unsigned length;            // - length of given word
    WORD *new;                  // - new group name

    GUESS :: NO_GROUP
        scanner = scan_over_black_space( scanner );
        scan_name( &scanner , &name , &length );
        QUITIF( length > 0 ) :: NO_GROUP;
        retn = msgerr( "No group name present" );
    ADMIT :: NO_ERRS
        retn = alloc_ring( &RingGroups , sizeof( struct word ) + length ,
                                                    &new );
        QUITIF( retn != 0 ) :: NO_ERRS;
        stvcpy( new->text , name , length );
    ENDGUESS
    return( retn );
}


static int alloc_message(       // ALLOCATE A MESSAGE ENTRY (OR PHRASE)
    void **ring,                // - ring
    WORD **words,               // - words vector
    unsigned count,             // - number of words
    unsigned number )           // - message number
{
    int retn;                   // - return code
    MESSAGE *msg;               // - new message

    GUESS :: NO_ERRS
        substitute_phrases( words, &count );
        retn = alloc_ring( ring, sizeof( struct message ) +
                 (count - 1 )*( sizeof( WORD * ) ) , &msg );
        QUITIF( retn != 0 );
        memcpy( msg->words, words, count * sizeof( WORD * ) );
        msg->codes = count;
        msg->count = 0;
        msg->number = number;
        msg->type = MSG_TYPE_ERROR;
        msg->level = 0;
    ENDGUESS
    return( retn );
}


static int scan_level(          // SCAN LEVEL
    const char *cptr,           // - line scanner
    unsigned *level )           // - level scanned
{
    int retn;                   // - return code

    GUESS :: NO_LEVEL
        cptr = scan_over_white_space( cptr );
        if( *cptr == '.' ) {
            cptr = scan_over_white_space( cptr + 1 );
        }
        QUITIF( *cptr != '\0' );
        retn = msgerr( "No warning level" );
    ADMIT :: BAD_LEVEL
        QUITIF( isdigit( *cptr ) );
        retn = msgerr( "Invalid warning level" );
    ADMIT :: MORE_JUNK
        *level = *cptr - '0';
        ++cptr;
        if( *cptr != '\0' && isdigit( *cptr ) ) {
            *level *= 10;
            *level += *cptr - '0';
            ++cptr;
        }
        cptr = scan_over_white_space( cptr );
        QUITIF( *cptr == '\0' );
        retn = msgerr( "Junk after warning level" );
    ADMIT :: OK
        retn = 0;
    ENDGUESS
    return retn;
}


static int scan_words(          // SCAN WORDS
    const char *cptr,           // - line scanner
    WORD *tgt[50],              // - sentence ring
    int *counter,               // - number of words
    int *a_size )               // - size of sentence/phrase
{
    int retn;                   // - return value
    int count;                  // - words in sentence
    const char *name;           // - word after text
    char text[80];              // - word alone
    unsigned length;            // - length of name
    WORD *word;                 // - word
    int size;                   // - size of sentence/phrase

    GUESS :: NO_ERRS
        count = 0;
        size = 0;
        LOOP :: GET_WORDS
            scan_name( &cptr , &name , &length );
            QUITIF( length == 0 ) :: GET_WORDS;
            size += length + 1;
            stvcpy( text , name , length );
            word = (WORD *)_ring_lookup( (void *)RingWords
                                       , offsetof( struct word , text )
                                       , text );
            IF( word == NULL )
                retn = alloc_ring( &RingWords
                                 , sizeof( struct word ) + length
                                 , &word );
                QUITIF( retn != 0 ) :: NO_ERRS;
                word->count = 0;
                word->code = 0;
                stvcpy( word->text , name , length );
            ENDIF
            ++word->count;
            *tgt++ = word;
            ++count;
        ENDLOOP
        *counter = count;
        *a_size = size;
        retn = 0;
    ENDGUESS
    return( retn );
}


static int process_text(        // TXT OPTION FOUND
    const char *scanner,        // - line scanner
    int *symbol_num )           // - number within group
{
    int size;                   // - size of sentence
    int retn;                   // - return value
    int count;                  // - words in sentence
    WORD *sentence[50];         // - vector of words

    GUESS :: NO_TEXT
        scanner = scan_over_white_space( scan_over_black_space( scanner ) );
        QUITIF( *scanner != '\0' );
        retn = msgerr( "No message given" );
    ADMIT :: NO_GET_ERRS
        space_read_in += strlen( scanner ) + 4;
        retn = scan_words( scanner, &sentence, &count, &size );
        QUITIF( retn == 0 ) :: NO_GET_ERRS;
    ADMIT :: NO_SPACE
        retn = alloc_message( &RingMessages, sentence, count, *symbol_num );
        QUITIF( retn == 0 ) :: NO_SPACE
    ADMIT :: OK
        IF( msg_size < size )
            msg_size = size;
        ENDIF
        *symbol_num += 1;
    ENDGUESS
    return( retn );
}


static int process_symbol(      // SYM OPTION FOUND
    const char *scanner,        // - line scanner
    int group_num,              // - group number
    int symbol_num )            // - symbol number
{
    int retn;                   // - return value
    char *symbol;               // - symbol after MSGSYM
    unsigned symsize;           // - size of symbol
    char symnum[10];            // - hexadecimal symbol number
    char sym[80];               // - the symbol

    GUESS :: NO_SYMBOL
        scanner = scan_over_black_space( scanner );
        scan_name( &scanner , &symbol, &symsize );
        QUITIF( symsize != 0 ) :: NO_SYMBOL;
        retn = msgerr( "Missing symbol name after :MSGSYM" );
    ADMIT :: PRINT_FILE
        stvcpy( sym, symbol, symsize );
        stxcpy( symnum, ( group_num << 10 ) + symbol_num );
        GUESS :: NO_ERRS
            retn = write_h_str( "#define " );
            QUITIF( retn != 0 );
            retn = write_h_str( sym );
            QUITIF( retn != 0 );
            retn = write_h_str( " " );
            QUITIF( retn != 0 );
            retn = write_h_line( symnum );
        ENDGUESS
    ENDGUESS
    return( retn );
}


static int process_state(       // PROCESS CURRENT STATE
    STATE state ,               // - current state
    int *symbol_num ,           // - symbol number
    int *group_num ,            // - group number
    const char *lineptr )       // - current line pointer
{
    int retn;                   // - return value
    int grpnum;                 // - group number
    int symbol;                 // - symbol number

    grpnum = *group_num;
    symbol = *symbol_num;
    SWITCH( state )
        CASE( GROUP_LINE )
            grpnum++;
            symbol = 0;
            retn = process_group( lineptr );
        CASE( SYM_GROUP )
            retn = process_symbol( lineptr, grpnum, symbol  );
        CASE( TEXT_GROUP )
            retn = process_text( lineptr, &symbol );
        CASE( SYM_NO_GRP )
            retn = process_symbol( lineptr, 0, symbol );
        CASE( TEXT_NO_GRP )
            retn = process_text( lineptr, &symbol );
        CASE( PRE_EOF )
            retn = msgerr( "Premature End-of-File" );
        CASE( GRP_NOT_FRST )
            retn = msgerr( "Group is not first token in file" );
        CASE( MISPL_TXT )
            retn = msgerr( "Misplaced text token" );
        CASE( MISPL_SYM )
            retn = msgerr( "Misplaced symbol token" );
        CASE( MISPL_MOD )
            retn = msgerr( "Misplaced message modifier" );
        CASE( DUP_MOD )
            retn = msgerr( "More than one message modifier" );
        DEFAULT
            retn = msgerr( "Error when reading new state" );
    ENDSWITCH
    *group_num = grpnum;
    *symbol_num = symbol;
    return( retn );
}


static int add_phrase(          // ADD PHRASE TO RING
    const char *lineptr )       // - pointer to line
{
    int retn;                   // - return value
    int counter;                // - counter
    WORD *tgt[50];              // - phrase sentence
    int size;                   // - not used

    GUESS :: NO_ERRS
        retn = scan_words( scan_over_black_space( lineptr )
                        , tgt
                        , &counter
                        , &size );
        QUITIF( retn != 0 );
        retn = alloc_message( &RingPhrases, tgt, counter, ENC_BIT );
        QUITIF( retn != 0 );
        ++ enc_phrases;
    ENDGUESS
    return( retn );
}


static int process_modifier(    // PROCESS A MODIFIER
    MSG_TYPE type,              // - message type
    unsigned level )            // - error/warning level
{
    RingMessages->type = type;
    RingMessages->level = level;
    return 0;
}


static int process_input()      // PROCESS INPUT FILE
{
    int retn;                   // - return value
    char *lineptr;              // - pointer to line
    int symbol_num;             // - symbol number
    int group_num;              // - group number
    REC_TYPE record;            // - record just read in
    STATE state;                // - new state;
    MSG_TYPE type;              // - type of modifier
    unsigned level;             // - error/warning level
    unsigned line_ctr;          // - line counter
    char line_ctr_buf[16];      // - buffer for line counter

    GUESS :: OPEN_SOURCE
        filepres = io_open_read_file( source );
        QUITIF( filepres != NULL );
        retn = msgerr( "Could not open file for reading" );
    ADMIT :: OPEN_HEADER
        retn = open_h_file( codes );
        QUITIF( retn == 0 );
    ADMIT :: NO_ERRS
        symbol_num = 0;
        group_num = -1;         // - No group found yet
        state = START_LINE;
        RingWords = NULL;
        RingGroups = NULL;
        RingMessages = NULL;
        RingPhrases = NULL;
        line_ctr = 0;
        LOOP :: PROCESS_LINE
            ++ line_ctr;
            IF( !io_getrec( filepres ) )
                IF( io_eof( filepres ) )
                    record = REC_EOF;
                ELSE
                    retn = msgerr( "IO Error when reading file" );
                    QUIT :: PROCESS_LINE;
                ENDIF
            ELSE
                lineptr = (char *)scan_over_white_space( filepres->record_ptr );
                IF( *lineptr != ':' )
                    record = REC_NONE;
                ELSEIF( memicmp( lineptr , ":MSGSYM" , 7 ) == 0 )
                    record = REC_SYM;
                ELSEIF( memicmp( lineptr , ":MSGTXT" , 7 ) == 0 )
                    record = REC_TXT;
                ELSEIF( memicmp( lineptr , ":WARNING", 8 ) == 0 )
                    type = MSG_TYPE_WARNING;
                    retn = scan_level( lineptr + 8, &level );
                    QUITIF( retn != 0 ) :: PROCESS_LINE;
                    record = REC_MOD;
                ELSEIF( memicmp( lineptr , ":ANSICOMP", 9) == 0 )
                    type = MSG_TYPE_ANSICOMP;
                    level = 0;
                    record = REC_MOD;
                ELSEIF( memicmp( lineptr , ":ANSIERR", 8 ) == 0 )
                    type = MSG_TYPE_ANSIERR;
                    level = 0;
                    record = REC_MOD;
                ELSEIF( memicmp( lineptr , ":ANSIWARN", 9 ) == 0 )
                    type = MSG_TYPE_ANSIWARN;
                    retn = scan_level( lineptr + 9, &level );
                    QUITIF( retn != 0 ) :: PROCESS_LINE;
                    record = REC_MOD;
                ELSEIF( memicmp( lineptr , ":ANSI", 5 ) == 0 )
                    type = MSG_TYPE_ANSI;
                    retn = scan_level( lineptr + 5, &level );
                    QUITIF( retn != 0 ) :: PROCESS_LINE;
                    record = REC_MOD;
                ELSEIF( memicmp( lineptr , ":EXTWARN", 8 ) == 0 )
                    type = MSG_TYPE_EXTWARN;
                    retn = scan_level( lineptr + 8, &level );
                    QUITIF( retn != 0 ) :: PROCESS_LINE;
                    record = REC_MOD;
                ELSEIF( memicmp( lineptr , ":INFO", 5 ) == 0 )
                    type = MSG_TYPE_INFO;
                    level = 0;
                    record = REC_MOD;
                ELSEIF( memicmp( lineptr , ":MSGGRP" , 7 ) == 0 )
                    record = REC_GRP;
                ELSEIF( memicmp( lineptr , ":MSGPHR" , 7 ) == 0 )
                    record = REC_NONE;
                    retn = add_phrase( lineptr );
                    QUITIF( retn != 0 ) :: PROCESS_LINE;
                ELSE
                    record = REC_NONE;
                ENDIF
            ENDIF
            IF( record != REC_NONE )
                state = state_table[ state ][ record ];
                QUITIF( state == FOUND_EOF ) :: PROCESS_LINE;
                IF( ( state == MOD_GROUP ) || ( state == MOD_NO_GRP ) )
                    retn = process_modifier( type, level );
                ELSE
                    retn = process_state( state
                                        , &symbol_num
                                        , &group_num
                                        , lineptr );
                ENDIF
                QUITIF( retn != 0 ) :: PROCESS_LINE;
            ENDIF
        ENDLOOP
        IF( ( retn != 0 ) && ( state != PRE_EOF ) )
            stucpy( line_ctr_buf, line_ctr );
            retn = msgerr_list( "line["
                              , line_ctr_buf
                              , "] "
                              , filepres->record_ptr
                              , NULL );
        ENDIF
        write_define( &write_h_line );
        close_h_file();
        io_close_read_file( filepres );
    ENDGUESS
    return( retn );
}


static int scan_filename(       // SCAN FILE NAME FROM TEXT LINE
    const char **scanner,       // - text line
    char **tgt )                // - target name
{
    int retn;                   // - return value
    char *new;                  // - name to be put into target
    const char *cptr;           // - scanner pointer
    unsigned size;              // - size of filename
    char *name;                 // - name read

    GUESS :: NO_NAME
        cptr = *scanner;
        scan_name( &cptr , &name , &size );
        QUITIF( size > 0 );
        retn = msgerr( "Missing file name" );
        *tgt = NULL;
    ADMIT :: NO_SPACE
        new = malloc( size + 1 + 3 );
        QUITIF( new != NULL ) :: NO_SPACE;
        retn = msgerr( "Out of memory" );
        *tgt = NULL;
    ADMIT :: ALL_OK
        stvcpy( new , name , size );
        *scanner = cptr;
        retn = 0;
        *tgt = new;
    ENDGUESS
    return( retn );
}


static int parse_filenames(     // GET ALL FILENAMES
    const char *cptr )          // - command line
{
    int retn;                   // - return value

    GUESS :: NO_ERRS
        retn = scan_filename( &cptr , &source );
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = scan_filename( &cptr , &messages );
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = scan_filename( &cptr , &codes );
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = scan_filename( &cptr , &levels );
        QUITIF( retn != 0 ) :: NO_ERRS;
        cptr = (char *)scan_over_white_space( cptr );
        QUITIF( *cptr == '\0' );
        retn = msgerr( "Extra junk on command line" );
    ENDGUESS
    return( retn );
}


static void display_help()      // DISPLAY HELP TEXT
{
    static char *text[] =       // - help text
    { "Usage: msgencod source message codes levels" ,
      "" ,
      "Example: msgencod ftnmsgs ftnerrs ftnhead levels" ,
      "" ,
      "The program takes the \"source\" (in GML format) and produces a" ,
      "\"message.c\" file, a \"codes.h\" file, and a \"levels.h\" file." ,
      "" ,
      "The \"message.c\" file contains all the messages (encoded).",
      "The \"codes.h\" file contains tags for the messages." ,
      "The \"levels.h\" file contains message levels for the messages." ,
      NULL };

    display_messages( text );
}


int main(                       // MAIN-LINE
    int argc,                   // - # arguments
    char *argv[] )              // - arguments
{
    int retn;                   // - return value
    char cmd_line[128];         // - command line
    char *cptr;                 // - command line ptr
    unsigned percentage;        // - percentage

    msg_progname( "MSGENCOD" );
    msgerr_return( -1 );
    GUESS :: HELP
        QUITIF( argc < 2 );
        QUITIF( 0 != stricmp( argv[1], "?" ) );
        display_help();
    ADMIT :: NO_ERRS
        cptr = getcmd( cmd_line );
        retn = parse_filenames( cptr );
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = process_input();
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = create_output();
        QUITIF( retn != 0 ) :: NO_ERRS;
        retn = create_levels();
        QUITIF( retn != 0 ) :: NO_ERRS;
        percentage = ( space_put_out * 100L + space_read_in - 1 )
                   / space_read_in;
        printf( "Compression: %u%% input(%lu) output(%lu)\n"
              , percentage
              , space_read_in
              , space_put_out );
    ENDGUESS
    return( retn );
}
