/* Pong Arcade Game
 *
 * What a better way to kick off the LRG library then to re-invent pong?
 *
 * Simple to play, just use LEFT-SHIFT to move up, ALT to move down, and
 * ESC to quit. For player2 it's RIGHT-SHIFT for up, and CTRL for down.
 * The game will end when one player gets 10 points.
 *
 * Some improvements:  Smoother movement for ball and paddles.  Maybe more
 * dynamics to the way the ball moves (i.e if the paddle is moving down,
 * induce a downward motion on ball).
 *
 * Contributed by Tom St Denis
 * Use of SHIFT/ALT/CTRL, Sound effects, and some clean up by Dave Dunfield.
 *
 * Compile command: cc pong -fop
 */

#include <stdio.h>
#include <lrg.h>

#define P_WIDTH 10		/* x size */
#define P_DEPTH 30		/* y size */
#define	BALL_R	14		/* Ball size */

/* players positions (basically their top Y value) */
int player1_p, player2_p;

/* scores */
int score1, score2;

/* Ball X and Y position */
int ball_x, ball_y;

/* Ball X and Y deltas */
int ball_dx, ball_dy;

extern unsigned RAND_SEED;

draw_screen()
{
	/* Draw paddles */
	lrg_fbox(0, player1_p, P_WIDTH, P_DEPTH, RED);
	lrg_fbox(310, player2_p, P_WIDTH, P_DEPTH, RED);

	/* draw ball */
	lrg_fcircle(ball_x, ball_y, 10, B_YELLOW);

	/* draw scores */
	lrg_printf(  0, 0, BLUE<<8|B_WHITE, "Player1: %-2d", score1);
	lrg_printf(230, 0, BLUE<<8|B_WHITE, "Player2: %-2d", score2);
}

clear_screen()
{
	/* erase paddles */
	lrg_fbox(0, player1_p, P_WIDTH, P_DEPTH, BLACK);
	lrg_fbox(310, player2_p, P_WIDTH, P_DEPTH, BLACK);

	/* erase ball */
	lrg_fcircle(ball_x, ball_y, 10, BLACK);
}

/* setup the random number */
randomize()
{
	int a, b, c;

	get_time(&a, &b, &c);

	RAND_SEED = (a*1800)+(b*60)+c;
}

show_help()
{
	puts("PC-PONG re-born with Micro-C LRG ");
	puts("Game written by Tom St Denis.\n");
	puts("Game is played against other human, or dog or animal.  The game");
	puts("ends when either a player gets 10 points, or someone hits ESCAPE");
	puts("\nKeys are:");
	puts("  Left-SHIFT- Up Player One\t\tRight-SHIFT- Up Player Two");
	puts("  ALT       - Down Player One\t\tCTRL       - Down Player Two");
	puts("  ESCAPE    - Quit");
	puts("\nPress any key (try enter) to continue.");
	while(!kbhit());
	kbget();		/* clear key */
}

main()
{
	int quit, key, a;

	show_help();

	randomize();

	if(lrg_open())
		abort("VGA required.");

	ball_x = 160 + (random(40) - 20);	/* +- 20 */
	ball_y = 100 + (random(40) - 20);

	a = random(4);
	if(a < 2)
		ball_dx = 1;
	else
		ball_dx = -1;

	a = random(4);
	if(a > 2)
		ball_dy = 1;
	else
		ball_dy = -1;

	player1_p = player2_p = 100;
	score1 = score2 = quit = 0;

	do {
		clear_screen();

		/* handle several key presses per update */
		for(a = 0; a < 5; ++a) {
			/* handle keyboard input */
			key = peek(0, 0x417);
			if(key & 0x0002) {	/* L-SHIFT - Player 1 up */
				if(player1_p > 12)
					player1_p -= 2; }
			if(key & 0x0008) {	/* ALT - Player 1 down */
				if(player1_p < (198 - P_DEPTH))
					player1_p += 2; }
			if(key & 0x0001) {	/* R-SHIFT - Player 2 up */
				if(player2_p > 12)
					player2_p -= 2; }
			if(key & 0x0004) {	/* CTRL - Player 2 down */
				if(player2_p < (198 - P_DEPTH))
					player2_p += 2; }
			if(kbtst() == 0x1B)
				quit = -1; }

		for(a = 0; a < 5; ++a) {
			/* Move ball, and adjust deltas if needed */
			ball_x += ball_dx;
			ball_y += ball_dy;

			if(ball_x < 20) {
				if((player1_p-BALL_R > ball_y) || (player1_p+(P_DEPTH+BALL_R) < ball_y)) {
					++score2;
					beep(100, 700); }
				else
					sound(1000);
				ball_dx = 1; }
			else if (ball_x > 300) {
				if((player2_p-BALL_R > ball_y) || (player2_p+(P_DEPTH+BALL_R) < ball_y)) {
					++score1;
					beep(100, 700); }
				else
					sound(1200);
				ball_dx = -1; }

			if(ball_y < 20) {
				ball_dy = 1;
				sound(1300); }
			else if (ball_y > 180) {
				ball_dy = -1;
				sound(1300); } }

			draw_screen();

			/* delay */
			delay(30);
			sound_off(); }
		while (!quit && (score1 < 10) && (score2 < 10));

	lrg_close();

	if(score1 == 10)
		puts("Player 1 wins!!!");
	else if(score2 == 10)
		puts("Player 2 wins!!!");
	else
		puts("Hmm.. quitters are we?  Oh well, enjoy :)");
}
