//
//     SVGA Driver
//

#define FALSE  0
#define TRUE   1

#define VIDEO_MEMORY        0xA0000
#define SVGA640x400x256     0x0100
#define SVGA640x480x256     0x0101
#define SVGA800x600x256     0x0103
#define SVGA1024x768x256    0x0105
#define SVGA1280x1024x256   0x0107
#define VGA320x200x256      0x0013
#define TEXTMODE            0x0003

typedef struct{
   unsigned char VbeSignature[4];
   unsigned short VbeVersion;
   unsigned short OemStringPtrOff;
   unsigned short OemStringPtrSeg;
   unsigned char Capabilities[4];
   unsigned long VideoModePtr;
   unsigned short TotalMemory;
   unsigned short OemSoftwareRev;
   unsigned long OemVendorNamePtr;
   unsigned long OemProductNamePtr;
   unsigned long OemProductRevPtr;
   unsigned char Reserved[222];
   unsigned char OemData[256];
}VbeInfoBlock;

typedef struct{
   unsigned short    ModeAttributes;
   unsigned char    WinAAttributes;
   unsigned char    WinBAttributes;
   unsigned short    WinGranularity;
   unsigned short    WinSize;
   unsigned short    WinASegment;
   unsigned short    WinBSegment;
   unsigned long WinFuncPtr;
   unsigned short  BytesPerScanLine;
   unsigned short  XResolution;
   unsigned short  YResolution;
   unsigned char  XCharSize;
   unsigned char  YCharSize;
   unsigned char  NumberOfPlanes;
   unsigned char  BitsPerPixel;
   unsigned char  NumberOfBanks;
   unsigned char  MemoryModel;
   unsigned char  BankSize;
   unsigned char  NumberOfImagePages;
   unsigned char  Reserved;
   unsigned char  RedMaskSize;
   unsigned char  RedFieldPosition;
   unsigned char  GreenMaskSize;
   unsigned char  GreenFieldPosition;
   unsigned char  BlueMaskSize;
   unsigned char  BlueFieldPosition;
   unsigned char  RsvdMaskSize;
   unsigned char  DirectColorModeInfo;
   unsigned long PhysBasePtr;
   unsigned long OffScreenMemOffset;
   unsigned short  OffScreenMemSize;
   unsigned char  Reserveds[206];
}ModeInfoBlock;

unsigned short VID_Init(unsigned short mode);
void VID_GetVbeInfo(VbeInfoBlock * info);
void VID_GetModeInfo(unsigned short mode, ModeInfoBlock *info);
void VID_SetMode(unsigned short mode);
void VID_PutBuffer(unsigned char *buffer);

//Declared Extern
ModeInfoBlock MODEINFO;
char CANUSELFB=TRUE;
unsigned long MEMORYSIZE=0;

//Static global variables
static VbeInfoBlock VBEINFO;
static char USELFB=0;
static unsigned long LASTSIZE=0;
static unsigned long NUMBANKS=0;
static unsigned long LINEARFRAMEBUFFER=0;


//Initialize the whole Super VGA system
unsigned short VID_Init(unsigned short mode){
   __dpmi_meminfo meminfo;
   unsigned char name[256];

   USELFB=CANUSELFB;
   VID_GetVbeInfo(&VBEINFO);

   //Display information
   dosmemget(VBEINFO.OemStringPtrSeg*16+VBEINFO.OemStringPtrOff, 256, name);
   //printf("VESA driver version %i.%i installed (%s)\n",
   //   VBEINFO.VbeVersion>>8, VBEINFO.VbeVersion&0x00FF,name);
   //printf("Press any key to launch the application...\n");
   //while(!kbhit());  getch();

  if (!VBEINFO.VbeVersion)return FALSE;
  if (strncmp(VBEINFO.VbeSignature,"VESA",4)!=0) return FALSE;
   
   //Setup the video mode
   if(mode<0x100){
       if(mode==0x13){
         MODEINFO.XResolution=320; MODEINFO.YResolution=200;
         MEMORYSIZE=MODEINFO.XResolution*MODEINFO.YResolution;
         LINEARFRAMEBUFFER=VIDEO_MEMORY;
         USELFB=TRUE;
      }
      else return FALSE;
   }
   else{
      VID_GetModeInfo(mode, &MODEINFO);
      MEMORYSIZE=MODEINFO.XResolution*MODEINFO.YResolution;
      if(VBEINFO.VbeVersion<512)USELFB=FALSE;
      if(USELFB){
         meminfo.size = MEMORYSIZE; meminfo.address = MODEINFO.PhysBasePtr;
         if(__dpmi_physical_address_mapping(&meminfo) == -1)return FALSE;
         LINEARFRAMEBUFFER=meminfo.address;
         }
      else{  NUMBANKS=MEMORYSIZE/(64 *1024); LASTSIZE=MEMORYSIZE%(64 *1024);
             if(!LASTSIZE)NUMBANKS--;
           }
   }
   VID_SetMode(mode);
return TRUE; }

/**** VESA functions ****/
//VESA FUNCTION 00
// ==================
// = VID_GetVbeInfo =
// ==================
void VID_GetVbeInfo(VbeInfoBlock * info){
   __dpmi_regs r;
   r.x.ax = 0x4F00; r.x.es = __tb / 16; r.x.di = 0;
   __dpmi_int(0x10, &r);
   dosmemget(__tb, sizeof(VbeInfoBlock), info);
return; }

//VESA FUNCTION 01
// ===================
// = VID_GetModeInfo =
// ===================
void VID_GetModeInfo(unsigned short mode, ModeInfoBlock *info){
   __dpmi_regs r;
   r.x.ax = 0x4F01; r.x.cx = mode;  r.x.es = __tb / 16; r.x.di = 0;
   __dpmi_int(0x10, &r);
   dosmemget(__tb, sizeof(ModeInfoBlock), info);
return; }

//VESA FUNCTION 02
// =================
// =  VID_SetMode  =
// =================
void VID_SetMode(unsigned short mode){
   __dpmi_regs r;
   if(mode<0x100) r.x.ax = mode;
   else{//Set mode
        r.x.ax = 0x4F02;
        if(USELFB) r.x.bx = 0x4000+mode;
        else r.x.bx = mode;
        }
   __dpmi_int(0x10, &r);
return; }

//Display function using VESA FUNCTION 05 for bank switching
//=================
//= VID_PutBuffer =
//=================
void VID_PutBuffer(unsigned char *buffer){
   unsigned char bank;
   __dpmi_regs r;
   if(USELFB) _dosmemputl(buffer,MEMORYSIZE/4,LINEARFRAMEBUFFER);
   else {
       r.x.ax=0x4F05; r.x.bx=0x0000; r.x.dx=0x0000;
       for(bank=0; bank<NUMBANKS; bank++){
         __dpmi_int(0x10, &r);
         r.x.ax=0x4F05;
         _dosmemputl(buffer+bank*64 *1024,16 *1024,VIDEO_MEMORY);
         r.x.dx+=64/MODEINFO.WinGranularity;
      }
      r.x.ax=0x4F05;
      __dpmi_int(0x10, &r);
      _dosmemputl(buffer+bank*64 *1024,LASTSIZE/4,VIDEO_MEMORY);
   }
return; }
