;========================================================
; SCROLL -- By Eric Tauck
;
; This is a TSR to pause the screen and wait for a key-
; stroke after a certain number of consecutive lines have
; been displayed.  SCROLL is only enabled when scroll
; lock is activated.  The scroll lock key may be pressed
; to unpause the screen and disable SCROLL.
;
; This program works by counting the number of linefeeds
; displayed with the teletype function of INT 10.  SCROLL
; will not work with any device drivers or programs that
; bypass INT 10, like ANSI.SYS.
;
; Usage: SCROLL [lines]
;
; The default number lines is 24.
;
; This program is written for WASM and uses several of
; the WASM library files.

        jmp     install

BEGIN_RESIDENT

NEWLOC  EQU     80H
RELOC   EQU     $ - NEWLOC

;========================================
; Resident data.

page    DB      ?               ;paused page number
cols    DB      ?               ;paused columns

lines   DW      24              ;lines to show before pause
scroll  DW      ?               ;current lines shown

old10   LABEL   DWORD           ;original interrupt 10H
        DW      ?, ?

old16   LABEL   DWORD           ;original interrupt 16H
        DW      ?, ?

prompt  DB      10,13,'Press any key to continue...',0

;========================================
; Reset the scroll count.

Reset   PROC    NEAR
        seg     cs
        push    lines - RELOC
        seg     cs
        pop     scroll - RELOC
        ret
        ENDP

;========================================
; Return scroll lock state.
;
; Out: ZF= set if not locked.

Lock    PROC    NEAR
        push    ax
        mov     ah, 2           ;get shift status
        pushf
        seg     cs
        call    old16 - RELOC   ;execute
        test    al, 10H         ;set ZF
        pop     ax
        ret
        ENDP

;========================================
; Wait for key or scroll off. Assume
; DS = CS.

Wait    PROC    NEAR
wait1   call    Lock            ;check scroll lock
        jz      wait2           ;exit if cleared
        mov     ah, 1           ;keyboard status
        pushf
        call    old16 - RELOC   ;execute
        jz      wait1
        sub     ah, ah          ;read keystroke
        pushf
        call    old16 - RELOC   ;execute
wait2   ret
        ENDP

;========================================
; Type a string.  Assume DS = CS.
;
; In: SI= string.

Message PROC    NEAR
        jmps    type2

type1   push    si
        mov     ah, 0EH         ;TTY function
        mov     bl, 7           ;color (graphics mode only)
        pushf
        call    old10 - RELOC   ;execute
        pop     si

type2   cld
        lodsb                   ;load next byte
        or      al, al          ;check if end of string
        jnz     type1           ;loop back if not

        ret
        ENDP

;========================================
; Home cursor and clear the line.  Assume
; DS = CS.

Clear   PROC    NEAR

;--- goto start of line

        mov     ax, 0E0DH               ;TTY carriage return
        mov     bl, 7                   ;color (graphics mode only)
        pushf
        call    old10 - RELOC           ;execute

;--- clear the characters

        mov     ax, 0920H               ;write spaces
        mov     bl, 7                   ;color (graphics mode only)
        mov     bh, page - RELOC        ;page
        mov     cl, cols - RELOC        ;columns
        sub     ch, ch
        pushf
        call    old10 - RELOC           ;execute
        ret
        ENDP

;========================================
; Pause the display.

Pause   PROC    NEAR
        push    ax
        push    bx
        push    cx
        push    dx
        push    di
        push    si
        push    bp
        push    ds

        push    cs                              ;load data segment
        pop     ds                              ;

        mov     ah, 0FH                         ;get info function
        pushf
        call    old10 - RELOC                   ;execute
        mov     cols - RELOC, ah                ;save columns
        mov     page - RELOC, bh                ;save active page

        mov     si, OFFSET prompt - RELOC       ;pause prompt
        call    Message                         ;display
        call    Wait                            ;wait for keystroke
        call    Clear                           ;clear this line

        pop     ds
        pop     bp
        pop     si
        pop     di
        pop     dx
        pop     cx
        pop     bx
        pop     ax
        ret
        ENDP

;========================================
; Interrupt 16H handler.  Reloads scroll
; count if input function.

New16   PROC    FAR
        or      ah, ah                  ;check if standard input function
        jz      new161
        cmp     ah, 10H                 ;check if extended input function
        jne     new162
new161  call    Reset                   ;reset scroll count
new162  seg     cs
        jmp     old16 - RELOC
        ENDP

;========================================
; Interrupt 10H handler.

New10   PROC    FAR

;--- check video function

        cmp     ah, 05H         ;set video page
        je      new103
        cmp     ah, 07H         ;scroll down
        je      new103
        cmp     ah, 06H         ;scroll up
        jne     new101
        cmp     al, 1           ;check if not one line
        jne     new103
new101  cmp     ah, 0EH         ;teletype character
        jne     new104
        cmp     al, 10          ;check if linefeed
        jne     new104

;--- teletype linefeed

        seg     cs
        cmp     scroll - RELOC, 0       ;check if count expired
        je      new102
        seg     cs
        dec     scroll - RELOC          ;decrement count
new102  call    Lock                    ;check if scroll lock
        jz      new104
        seg     cs
        cmp     scroll - RELOC, 0       ;check if count expired
        jnz     new104

        call    Pause                   ;pause
        call    Reset                   ;reset count
        iret

;--- no pause

new103  call    Reset                   ;reset count
new104  seg     cs
        jmp     old10 - RELOC
        ENDP

END_RESIDENT

;========================================
; Installation.

;--- installation entry point

install INCLUDE '..\..\library\case1.asm'
        INCLUDE '..\..\library\convert.asm'
        INCLUDE '..\..\library\parms.asm'
        INCLUDE '..\..\library\message1.asm'

;--- banner

        mov     ax, OFFSET banner
        call    MesPutL                 ;display banner

;--- read parameter

        call    ParGet                  ;get first parameter
        jc      skip
        mov     si, ax

        mov     cx, OFFSET help         ;response if '?'
        cmp     BYTE [si], '?'          ;check if question mark
        je      termx
        mov     ax, si
        mov     cx, 10
        call    Str2Num                 ;convert to number
        mov     cx, OFFSET error        ;response if error
        jc      termx                   ;jump if error
        or      dx, dx                  ;check if 32 bit number
        jnz     termx                   ;exit if so

        mov     lines, ax               ;save number of lines
        jmps    skip

termx   jmps    term                    ;terminate relay

;--- display scroll lines

skip    mov     ax, OFFSET status
        call    MesPut                  ;display prompt
        mov     ax, lines
        sub     dx, dx
        mov     cx, 10
        mov     bx, OFFSET buffer
        call    Num2Str                 ;convert lines to string
        mov     ax, OFFSET buffer
        call    MesPutL                 ;display lines

;--- install

        mov     ax, lines                       ;copy scroll
        mov     scroll, ax                      ;

        push    es
        mov     ax, 3510H                       ;get interrupt 10H function
        int     21H                             ;execute
        mov     WORD old10, bx                  ;save offset
        mov     WORD old10 + 2, es              ;save segment

        mov     ax, 3516H                       ;get interrupt 16H function
        int     21H                             ;execute
        mov     WORD old16, bx                  ;save offset
        mov     WORD old16 + 2, es              ;save segment

        mov     ah, 49H                         ;release memory
        mov     es, [2CH]                       ;environment segment
        int     21H                             ;execute
        pop     es

        mov     si, OFFSET BEGIN_RESIDENT       ;start of resident code
        mov     di, NEWLOC                      ;new location
        mov     cx, OFFSET END_RESIDENT         ;end
        sub     cx, si                          ;bytes of code
        cld
        rep
        movsb                                   ;copy code

        mov     ax, 2510H                       ;set interrupt 10H function
        mov     dx, OFFSET New10 - RELOC        ;offset, segement in DS
        int     21H                             ;execute

        mov     ax, 2516H                       ;set interrupt 16H function
        mov     dx, OFFSET New16 - RELOC        ;offset, segement in DS
        int     21H                             ;execute

        mov     ax, OFFSET END_RESIDENT - RELOC ;end of resident code
        mov     cl, 4                           ;bits to shift
        mov     dx, ax
        shr     dx, cl                          ;convert to paragraph
        and     ax, 0FH                         ;mask remainder
        sub     ah, al                          ;set carry if non-zero
        adc     dx, 0                           ;update paragraphs

        mov     ax, 3100H                       ;TSR
        int     21H                             ;execute

;--- terminate

term    mov     ax, cx
        call    MesPutL                 ;display message

        mov     ax, 4CFFH               ;error terminate
        int     21H                     ;execute

;--- data

banner  DB      13,10,'Scroll  Version 1.00  By Eric Tauck',0
status  DB      'Installed with scroll = ',0
error   DB      'Error in parameters, run SCROLL ? for help',0
help    DB      'Usage: SCROLL [lines]  default = 24',0

buffer  LABEL   BYTE
        ORG     +6
