;****************************************;
; WASM Multiplex Support, Transient Code ;
; By Eric Tauck                          ;
;                                        ;
; Defines:                               ;
;                                        ;
;   PlxChn  chain (install) interrupt    ;
;   PlxQue  query resident TSR           ;
;   PlxChk  check if can unhook          ;
;   PlxRel  release (unhook) interrupt   ;
;                                        ;
; Requires:                              ;
;                                        ;
;   INTR2.ASM                            ;
;   MULPLX1.ASM                          ;
;                                        ;
; This code is the transient portion of  ;
; the multiplex support.  These routines ;
; initialize this code and data in       ;
; MULPLX2.ASM. This file should included ;
; within the transient portion of the    ;
; TSR, while MULPLX1.ASM should be       ;
; included in the resident portion.      ;
;****************************************;

; RES_OFF and RES_ID must be EQUated by the user.  RES_ID is a 16 bit value
; that should be unique for all programs using MULPLX1.ASM and MULPLX2.ASM.
; RES_OFF is the byte offset to subtract from all memory references if the
; TSR is shifted to a lower memory location.  If the TSR is not shifted,
; RES_OFF should be set to zero.

        jmps    _mulplex2_end

;================================================
; Chain interrupt.
;
; In: AX= multiplex number; CX= version number;
;     DX= handler (can be zero if none); 
;
; Note: If the resident code is relocated, the
; relocation should be performed BEFORE this
; routine is called (i.e. the variables are
; assumed to be at their relocated addresses).

PlxChn  PROC    NEAR

;--- define data

        mov     _PlxNum - RES_OFF, ax   ;save number
        mov     _PlxVer - RES_OFF, cx   ;save version
        or      dx, dx                  ;check if handler defined
        jz      _plxch1
        sub     dx, RES_OFF             ;adjust
        mov     _PlxHan - RES_OFF, dx   ;save handler

;--- hook interrupts

_plxch1 mov     al, 2FH
        mov     bx, OFFSET PlxOld - RES_OFF
        mov     cx, OFFSET _PlxNew - RES_OFF
        call    IntChn                  ;chain interrupt
        ret
        ENDP

;================================================
; Query resident TSR.
;
; In: AX= multiplex number.
;
; Out: DX= resident segment (zero if not
;      resident or another TSR hooked); CX=
;      resident version; CY= set another TSR
;      using multiplex number.

PlxQue  PROC    NEAR
        push    di
        push    si
        push    bp
        push    ds
        push    es
        sub     bx, bx                  ;zero id
        mov     cx, bx                  ;zero version
        mov     dx, bx                  ;zero segment
        push    ax
        int     2FH                     ;query
        pop     di
        cmp     di, ax                  ;check if anyone hooked
        je      _plxqu2
        cmp     bx, RES_ID              ;verify owner
        jne     _plxqu3
        clc
_plxqu1 pop     es
        pop     ds
        pop     bp
        pop     si
        pop     di
        ret

;--- not resident

_plxqu2 or      cx, cx                  ;verify unchanged
        jnz     _plxqu3
        or      bx, bx                  ;verify unchanged
        jnz     _plxqu3
        or      dx, dx                  ;verify unchanged
        jz      _plxqu1

;--- other program using multiplex number

_plxqu3 sub     dx, dx                  ;zero segment
        stc
        jmps    _plxqu1
        ENDP

;================================================
; Check if interrupt has been changed.
;
; In: AX= multiplex number.
;
; Out: ZF= set if unchanged.

PlxChk  PROC    NEAR
        call    PlxQue                  ;resident query
        or      dx, dx                  ;check if segment returned
        jz      _plxck1

        mov     al, 2FH                 ;interrupt
        mov     cx, OFFSET _PlxNew - RES_OFF ;handler offset
        call    IntChk                  ;check interrupt 2F
        ret

_plxck1 dec     dx                      ;clear ZF
        ret
        ENDP

;================================================
; Release interrupt.
;
; In: AX= multiplex number.

PlxRel  PROC    NEAR
        call    PlxQue                  ;get resident segment
        push    ds
        mov     ds, dx
        mov     al, 2FH                 ;interrupt
        mov     bx, OFFSET PlxOld-RES_OFF ;variable
        call    IntRel                  ;release interrupt 21
        pop     ds
        ret
        ENDP

_mulplex2_end
