/* Windows Cryptographic utilities DLL, (c) 1994 Andy Brown
	 This package is freeware. You may do whatever you like with it so long
	 as you agree that the author assumes no responsibility for any
	 consequences of your use of this package */


#include <windows.h>
#include "md5.h"
#include "des.h"
#include "idea.h"
#include "diamond.h"
#include "mdc.h"
#include "crutils.h"

/* static declarations */

static BYTE key[96];												/* largest key is 768 bits for MDC */
static BYTE iv[32];													/* largest block size is 256 bits for MDC */
static des_key_schedule ks1,ks2;						/* the DES key schedule */
static BYTE idea_key[104];									/* the full 832 bit IDEA key */
static BYTE inv_idea_key[104];			  			/* inverted full IDEA key */
static DIAMONDKEY dmkey;										/* diamond key schedule */
static WORD diamond_keybytes;								/* size of the diamond key */

/* static prototypes */

int FAR PASCAL LibMain(HANDLE hInstance,WORD wDataSeg,WORD wHeapSize,LPSTR lpszCmdLine);
static VOID generate_keys(LPBYTE passphrase);
static VOID encrypt_buffer(BYTE huge *buffer,DWORD len,UINT alg,UINT mode);
static VOID decrypt_buffer(BYTE huge *buffer,DWORD len,UINT alg,UINT mode);
static VOID xor(LPBYTE dest,LPBYTE source,WORD size);
static VOID encrypt_block_des(LPBYTE block);
static VOID encrypt_block_3des(LPBYTE block);
static VOID encrypt_block_mdc(LPBYTE block);
static VOID decrypt_block_mdc(LPBYTE block);
static VOID decrypt_block_des(LPBYTE block);
static VOID decrypt_block_3des(LPBYTE block);
static VOID mymemcpy(LPBYTE dest,LPBYTE src,WORD size);
static VOID myhmemcpy(BYTE huge *dest,BYTE huge *src,DWORD size);
static VOID mybzero(LPBYTE buf,WORD size);
static WORD mystrlen(LPBYTE text);
static VOID set_keys(LPBYTE ukey,WORD keysize,LPBYTE uiv,UINT alg,UINT mode);
static VOID zero_vars(VOID);


/******************************/
/* LibMain DLL initialisation */
/******************************/

int FAR PASCAL LibMain(HANDLE hInstance,WORD wDataSeg,WORD wHeapSize,LPSTR lpszCmdLine)
{
	if(wHeapSize>0) UnlockData(0);
	return 1;
}


/*******************************************************/
/* Encrypt a huge buffer with user supplied passphrase */
/*******************************************************/

VOID FAR PASCAL _export EncryptWithPassphrase(BYTE huge *buffer,DWORD len,LPBYTE passphrase,UINT alg,UINT mode)
{
	diamond_keybytes=256;													/* nice secure size */
	generate_keys(passphrase);
	encrypt_buffer(buffer,len,alg,mode);
	zero_vars();
}


/***********************************************************/
/* Decrypt a huge buffer with the user supplied passphrase */
/***********************************************************/

VOID FAR PASCAL _export DecryptWithPassphrase(BYTE huge *buffer,DWORD len,LPBYTE passphrase,UINT alg,UINT mode)
{
	diamond_keybytes=256;													/* nice secure size */
	generate_keys(passphrase);
	decrypt_buffer(buffer,len,alg,mode);
	zero_vars();
}


/***********************************************************/
/* Encrypt a huge buffer with the user supplied key and IV */
/***********************************************************/

VOID FAR PASCAL _export EncryptWithKey(BYTE huge *buffer,DWORD len,LPBYTE ukey,WORD keysize,LPBYTE uiv,UINT alg,UINT mode)
{
	set_keys(ukey,keysize,uiv,alg,mode);
	encrypt_buffer(buffer,len,alg,mode);
  zero_vars();
}


/***********************************************************/
/* Decrypt a huge buffer with the user supplied key and IV */
/***********************************************************/

VOID FAR PASCAL _export DecryptWithKey(BYTE huge *buffer,DWORD len,LPBYTE ukey,WORD keysize,LPBYTE uiv,UINT alg,UINT mode)
{
	set_keys(ukey,keysize,uiv,alg,mode);
	decrypt_buffer(buffer,len,alg,mode);
  zero_vars();
}


/************************************/
/* Set the user-supplied key and IV */
/************************************/

static VOID set_keys(LPBYTE ukey,WORD keysize,LPBYTE uiv,UINT alg,UINT mode)
{
	if(alg==ALG_DIAMOND || alg==DIAMOND_LITE)
		diamond_keybytes=keysize;

	if(mode!=MODE_ECB)
	{
		WORD ivsize;

		if(alg==ALG_MDC)
			ivsize=32;
		else if(alg==ALG_DIAMOND)
			ivsize=16;
		else
			ivsize=8;

		mymemcpy(iv,uiv,ivsize);
	}
	mymemcpy(key,ukey,keysize);
}


/***********************/
/* Zero the key and IV */
/***********************/

static VOID zero_vars(VOID)
{
	mybzero(key,96);
	mybzero(iv,32);
	mybzero((LPBYTE)&ks1,sizeof(des_key_schedule));
	mybzero((LPBYTE)&ks2,sizeof(des_key_schedule));
	mybzero(idea_key,104);
	mybzero(inv_idea_key,104);
  mybzero((LPBYTE)&dmkey,sizeof(DIAMONDKEY));
}


/****************************/
/* Generate our crypto keys */
/****************************/

static VOID generate_keys(LPBYTE passphrase)
{
/* first generate a key by multiple hashing */

	MD5(passphrase,mystrlen(passphrase),&key[0]);
	MD5(&key[0],16,&key[16]);
	MD5(&key[16],16,&key[32]);
	MD5(&key[32],16,&key[48]);
	MD5(&key[48],16,&key[64]);
	MD5(&key[64],16,&key[80]);

/* now generate an IV by further hashing */

	MD5(&key[80],16,&iv[0]);
	MD5(&iv[0],16,&iv[16]);
}


/***********************************************/
/* Encrypt a huge buffer, keys are already set */
/***********************************************/

static VOID encrypt_buffer(BYTE huge *buffer,DWORD len,UINT alg,UINT mode)
{
static BYTE block[32],temp[32],civ[32],prev[32];
DWORD i,blocksize;

/* cipher initialisation */

	switch(alg)
	{
		case ALG_IDEA:	
			ExpandUserKey((LPWORD)key,(LPWORD)idea_key);
      blocksize=8;
			break;
		case ALG_3DES:													/* drop through... */
    	des_key_sched((des_cblock FAR *)&key[8],&ks2);
		case ALG_DES:
			des_key_sched((des_cblock FAR *)key,&ks1);
      blocksize=8;
			break;
		case ALG_DIAMOND:
			set_diamond_key(key,diamond_keybytes,15,FALSE,&dmkey,DIAMOND_FULL);
      blocksize=16;
			break;
		case ALG_DIAMONDLITE:
			set_diamond_key(key,diamond_keybytes,30,FALSE,&dmkey,DIAMOND_LITE);
      blocksize=8;
			break;
		case ALG_MDC:
			blocksize=32;
      break;
	}

	if(mode!=MODE_ECB)
		mymemcpy(civ,iv,blocksize);

	for(i=0;i<len;i+=blocksize)
	{
		myhmemcpy(block,&buffer[i],blocksize);
		switch(mode)
		{
			case MODE_CBC:
				xor(block,civ,blocksize);
				break;
			case MODE_PCBC:
      	mymemcpy(temp,block,blocksize);
				xor(block,civ,blocksize);
				if(i) xor(block,prev,blocksize);
        mymemcpy(prev,temp,blocksize);
				break;
			case MODE_CFB:
				mymemcpy(temp,block,blocksize);
				if(i)
					myhmemcpy(block,&buffer[i-blocksize],blocksize);
				else
        	mymemcpy(block,iv,blocksize);
				break;
			case MODE_OFB:
				mymemcpy(temp,block,blocksize);
        mymemcpy(block,i ? prev : iv,blocksize);
        break;
    }
		switch(alg)
		{
			case ALG_IDEA:
				Idea((LPWORD)block,(LPWORD)block,(LPWORD)idea_key);
				break;
			case ALG_DES:
				encrypt_block_des(block);
				break;
			case ALG_3DES:
				encrypt_block_3des(block);
				break;
			case ALG_DIAMOND:
				diamond_encrypt_block(block,block,&dmkey);
				break;
			case ALG_DIAMONDLITE:
				lite_encrypt_block(block,block,&dmkey);
        break;
			case ALG_MDC:
				encrypt_block_mdc(block);
        break;
		}
		switch(mode)
		{
			case MODE_CBC:
			case MODE_PCBC:
				mymemcpy(civ,block,blocksize);
				break;
			case MODE_CFB:
				xor(block,temp,blocksize);
				break;
			case MODE_OFB:
      	mymemcpy(prev,block,blocksize);
				xor(block,temp,blocksize);
        break;
		}
    myhmemcpy(&buffer[i],block,blocksize);
	}

	if(alg==ALG_DIAMOND || alg==ALG_DIAMONDLITE)
		diamond_done(&dmkey);
}


/***********************************************/
/* Decrypt a huge buffer, keys are already set */
/***********************************************/

static VOID decrypt_buffer(BYTE huge *buffer,DWORD len,UINT alg,UINT mode)
{
static BYTE block[32],temp[32],civ[32],prev[32];
DWORD i,blocksize;

/* cipher initialisation */

	switch(alg)
	{
		case ALG_IDEA:	
			ExpandUserKey((LPWORD)key,(LPWORD)idea_key);
      InvertIdeaKey((LPWORD)idea_key,(LPWORD)inv_idea_key);
      blocksize=8;
			break;
		case ALG_3DES:													/* drop through... */
    	des_key_sched((des_cblock FAR *)&key[8],&ks2);
		case ALG_DES:
			des_key_sched((des_cblock FAR *)key,&ks1);
      blocksize=8;
			break;
		case ALG_DIAMOND:
			set_diamond_key(key,diamond_keybytes,15,TRUE,&dmkey,DIAMOND_FULL);
      blocksize=16;
			break;
		case ALG_DIAMONDLITE:
			set_diamond_key(key,diamond_keybytes,30,TRUE,&dmkey,DIAMOND_LITE);
      blocksize=8;
			break;
		case ALG_MDC:
			blocksize=32;
      break;
	}

	if(mode!=MODE_ECB)
		mymemcpy(civ,iv,blocksize);

	for(i=0;i<len;i+=blocksize)
	{
		myhmemcpy(block,&buffer[i],blocksize);
		switch(mode)
		{
			case MODE_CBC:
    	case MODE_PCBC:
				mymemcpy(temp,block,blocksize);
				break;
			case MODE_CFB:
				mymemcpy(temp,block,blocksize);
        mymemcpy(block,i ? prev : iv,blocksize);
				myhmemcpy(prev,&buffer[i],blocksize);
				break;
			case MODE_OFB:
				mymemcpy(temp,block,blocksize);
				mymemcpy(block,i ? prev : iv,blocksize);
				break;
		}
		switch(alg)
		{
			case ALG_IDEA:
				if(mode==MODE_CFB || mode==MODE_OFB)
					Idea((LPWORD)block,(LPWORD)block,(LPWORD)idea_key);
				else
					Idea((LPWORD)block,(LPWORD)block,(LPWORD)inv_idea_key);
				break;
			case ALG_DES:
				if(mode==MODE_CFB || mode==MODE_OFB)
					encrypt_block_des(block);
				else
					decrypt_block_des(block);
				break;
			case ALG_3DES:
				if(mode==MODE_CFB || mode==MODE_OFB)
					encrypt_block_3des(block);
				else
					decrypt_block_3des(block);
				break;
			case ALG_DIAMOND:
				if(mode==MODE_CFB || mode==MODE_OFB)
					diamond_encrypt_block(block,block,&dmkey);
				else
					diamond_decrypt_block(block,block,&dmkey);
				break;
			case ALG_DIAMONDLITE:
				if(mode==MODE_CFB || mode==MODE_OFB)
					lite_encrypt_block(block,block,&dmkey);
				else
					lite_decrypt_block(block,block,&dmkey);
				break;
			case ALG_MDC:
				if(mode==MODE_CFB || mode==MODE_OFB)
					encrypt_block_mdc(block);
				else
					decrypt_block_mdc(block);
				break;
		}
		switch(mode)
		{
			case MODE_CBC:
				xor(block,civ,blocksize);
				mymemcpy(civ,temp,blocksize);
				break;
			case MODE_PCBC:
				xor(block,civ,blocksize);
				mymemcpy(civ,temp,blocksize);
				if(i) xor(block,prev,blocksize);
				mymemcpy(prev,block,blocksize);
				break;
			case MODE_CFB:
				xor(block,temp,blocksize);
				break;
			case MODE_OFB:
				mymemcpy(prev,block,blocksize);
				xor(block,temp,blocksize);
				break;
    }
		myhmemcpy(&buffer[i],block,blocksize);
	}

	if(alg==ALG_DIAMOND || alg==ALG_DIAMONDLITE)
		diamond_done(&dmkey);
}


/****************/
/* XOR function */
/****************/

static VOID xor(LPBYTE dest,LPBYTE source,WORD size)
{
WORD i;

	for(i=0;i<size;i++)
  	dest[i]^=source[i];
}


/****************************/
/* Encrypt a block with DES */
/****************************/

static VOID encrypt_block_des(LPBYTE block)
{
static des_cblock in,out;

	mymemcpy(in,block,8);
	des_ecb_encrypt(&in,&out,&ks1,TRUE);
	mymemcpy(block,out,8);
}


/*****************************/
/* Encrypt a block with 3DES */
/*****************************/

static VOID encrypt_block_3des(LPBYTE block)
{
static des_cblock block1,block2;

	mymemcpy(block1,block,8);
	des_ecb_encrypt(&block1,&block2,&ks1,TRUE);			/* DES(k1) */
	des_ecb_encrypt(&block2,&block1,&ks2,FALSE);		/* DES^-1(k2) */
  des_ecb_encrypt(&block1,&block2,&ks1,TRUE);			/* DES(k1) */
	mymemcpy(block,block2,8);
}


/****************************/
/* Encrypt a block with MDC */
/****************************/

static VOID encrypt_block_mdc(LPBYTE block)
{
static BYTE ciphertext[32];

	mdc_encrypt(block,ciphertext,key);
  mymemcpy(block,ciphertext,32);
}


/***********************/
/* Decrypt a DES block */
/***********************/

static VOID decrypt_block_des(LPBYTE block)
{
static des_cblock outbuf;
static des_cblock inbuf;

	mymemcpy(inbuf,block,sizeof(des_cblock));
	des_ecb_encrypt(&inbuf,&outbuf,&ks1,FALSE);
	mymemcpy(block,outbuf,sizeof(des_cblock));
}


/************************/
/* Decrypt a 3DES block */
/************************/

static VOID decrypt_block_3des(LPBYTE block)
{
static des_cblock block1;
static des_cblock block2;

	mymemcpy(block1,block,sizeof(des_cblock));
	des_ecb_encrypt(&block1,&block2,&ks1,FALSE);			/* DES^-1(k1) */
	des_ecb_encrypt(&block2,&block1,&ks2,TRUE);				/* DES(k2) */
	des_ecb_encrypt(&block1,&block2,&ks1,FALSE);			/* DES^-1(k1) */
	mymemcpy(block,block2,sizeof(des_cblock));
}


/************************/
/* Decrypt an MDC block */
/************************/

static VOID decrypt_block_mdc(LPBYTE block)
{
static BYTE plaintext[32];

	mdc_decrypt(block,plaintext,key);
  mymemcpy(block,plaintext,32);
}


/**********************************/
/* memcpy function (FAR pointers) */
/**********************************/

static VOID mymemcpy(LPBYTE dest,LPBYTE src,WORD size)
{
	while(size--)
  	*dest++=*src++;
}


/***********************************/
/* memcpy function (huge pointers) */
/***********************************/

static VOID myhmemcpy(BYTE huge *dest,BYTE huge *src,DWORD size)
{
	while(size--)
  	*dest++=*src++;
}


/********************************/
/* bzero function (FAR pointer) */
/********************************/

static VOID mybzero(LPBYTE buf,WORD size)
{
	while(size--)
  	*buf++='\0';
}


/*********************************/
/* strlen function (FAR pointer) */
/*********************************/

static WORD mystrlen(LPBYTE text)
{
WORD length;

	length=0;

	while(*text++)
		length++;

  return length;
}
