# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2009 Free Software Foundation
#
# $Id: pdf.py 9958 2009-10-11 18:56:59Z reinhard $
#
# DESCRIPTION:
# Transforms adapter for direct
#
# NOTES:
#  DocTemplates: contain one or more PageTemplates
#  PageTemplates: contain one or more Frames
#  Frames: contain flowables, like Paragraph or Table
#

import sys

try:
  from reportlab.pdfgen import canvas
except ImportError:
  raise "Unable to load reportlab. You can get reportlab from"+\
        " www.reportlab.org. If you have debian just do a apt-get install "+\
        " python-reportlab."

from reportlab.platypus.doctemplate import PageTemplate, BaseDocTemplate
from reportlab.platypus import Table, Paragraph
from reportlab.platypus.frames import Frame
from reportlab.lib.styles import getSampleStyleSheet
from reportlab.rl_config import defaultPageSize
from reportlab.lib import colors
from reportlab.lib.units import cm

# Truetype font support
try:
  from reportlab.pdfbase import pdfmetrics
  from reportlab.pdfbase.ttfonts import TTFont
  from reportlab.lib.fonts import addMapping
  ttf_supported = 1
except ImportError:
  ttf_supported = 0

PAGE_WIDTH = defaultPageSize[0]
PAGE_HEIGHT = defaultPageSize[1]

font_to_filename = { 'ArialUnicodeMS' : 'arialuni.ttf',
                     'Arial'          : 'Arial.ttf',
                     'ArialI'         : 'Arial_Italic.ttf',
                     'ArialB'         : 'Arial_Italic.ttf',
                     'ArialBI'        : 'Arial_Bold_Italic.ttf',
                     'PMingLiU'       : 'pmingliu.ttf',
                     'SimSun'         : 'simsun.ttf',
                     'ZenKai-Medium'  : 'arphic/bkai00mp.ttf'
# TODO: add more fonts / move to other place, possibly join with code in
#       Universal/ps
}


class DirectFormatter:

  def __init__(self, destination, stylesheet, options={}):
    self.dest = destination

    self._reporttitle = ''
    self._sectiontitle = ''

    # TODO: make that all configurable
    self.fontdir = '/usr/share/fonts/truetype/'

    # standart settings
    self.font = 'Helvetica'
    self.boldfont = 'Helvetica-Bold'

    #self.font = 'Arial'
    #self.boldfont = 'Arial'

    #self.font = 'ArialUnicodeMS'
    #self.boldfont = 'ArialUnicodeMS'

    #self.font = 'SimSun'

    #self.font = 'ZenKai-Medium'
    #self.boldfont = 'ZenKai-Medium'

    self.loadFont(self.font)
    self.loadFont(self.boldfont)

  def loadFont(self,name):
    if not ttf_supported:
      return

    if font_to_filename.has_key(name):
      try:
        pdfmetrics.registerFont(TTFont(name, self.fontdir + \
                                       font_to_filename[name]))
        addMapping(name, 0, 0, name)
      except Exception,msg:
        print _("Error occured loading font %s (%s).") % (name,msg)
        sys.exit(0)

    # TODO: do we need more mappings?
    #addMapping('Arial', 0, 1, 'ArialI')
    #addMapping('Arial', 1, 0, 'ArialB')
    #addMapping('Arial', 1, 1, 'ArialBI')

  # Used to paint non-flowing (i.e standard) parts of pages
  def myPages(self, canvas, doc):
    canvas.saveState()

    # first frame for title and sectiontitle
    f1 = Frame(self._doc.leftMargin-cm, self._doc.bottomMargin+self._doc.height-0.4*cm,
                    self._doc.width+2*cm, 2.5*cm,
                    leftPadding=0, bottomPadding=0, rightPadding=0, topPadding=0,
                    id='title', showBoundary=0)

    titlestyle = getSampleStyleSheet()["Title"]
    titlestyle.fontName=self.font
    p1 = Paragraph(self._reporttitle, titlestyle)

    canvas.setStrokeColor(colors.grey)
    canvas.line(self._doc.leftMargin-cm, self._doc.bottomMargin+self._doc.height+1.0*cm,
                    self._doc.leftMargin+self._doc.width+cm, self._doc.bottomMargin+self._doc.height+1.0*cm)
    p2 = Paragraph(self._sectiontitle, getSampleStyleSheet()["Heading2"])
    f1.addFromList([p1, p2], canvas)

    # second frame for table heading
    f2 = Frame(self._doc.leftMargin, self._doc.bottomMargin+self._doc.height-cm,
                    self._doc.width, 0.65*cm,
                    leftPadding=0, bottomPadding=0, rightPadding=0, topPadding=0,
                    id='head', showBoundary=0)

    # white on black for table heading
    Style = [elem for elem in self._tableStyle]
    Style.append(('BACKGROUND', (0,0), (-1,-1), colors.black))
    Style.append(('TEXTCOLOR', (0,0), (-1,-1), colors.white))
    Style.append(('FONTNAME', (0,0), (-1,-1), self.font))

    t = Table([self._tableHead], colWidths=self._colwidth, style=Style)
    f2.addFromList([t], canvas)

    # page numbering
    canvas.drawCentredString(PAGE_WIDTH/2, 0.5*cm, "-%d-" % doc.page)
    canvas.restoreState()


  def BeginReport(self):
    self._doc = BaseDocTemplate(self.dest)
    self._body = []


  def EndReport(self):
    # third frame for table body
    frame = Frame(self._doc.leftMargin, self._doc.bottomMargin,
                         self._doc.width, self._doc.height-cm,
                         leftPadding=0, bottomPadding=0, rightPadding=0, topPadding=0,
                         id='body', showBoundary=0)
    pt = PageTemplate(id='All', frames=frame, onPage=self.myPages, pagesize=self._doc.pagesize)
    self._doc.addPageTemplates(pt)
    self._doc.build(self._body)


  def BeginReportTitle(self):
    pass


  def EndReportTitle(self, reporttitle):
    self._reporttitle = reporttitle.encode('utf-8')


  def BeginSection(self):
    pass


  def EndSection(self):
    pass


  def BeginSectionTitle(self):
    pass


  def EndSectionTitle(self, sectiontitle):
    self._sectiontitle = sectiontitle.encode('utf-8')


  def BeginTable(self):
    self._odd = 1


  def EndTable(self):
    pass


  def BeginTableHead(self):
    self._tableAlign = []
    self._tableHead = []
    self._colwidth = []
    self._headwidth = 0
    self._numCols = 0

  def EndTableHead(self):
    self._tableStyle = []
    for i in range(self._numCols):
      self._tableStyle.append(('ALIGN', (i, 0), (i, 0), self._tableAlign[i].upper()))
      self._tableStyle.append(('FONTNAME', (0,0), (-1,-1), self.font))
      self._colwidth[i] = (self._doc.width/self._headwidth)*self._colwidth[i]

    self._oddStyle = [elem for elem in self._tableStyle]
    self._oddStyle.append(('BACKGROUND', (0,0), (-1,-1), colors.lightgrey))
    self._oddStyle.append(('TEXTCOLOR', (0,0), (-1,-1), colors.black))

    self._subtotalStyle = [elem for elem in self._tableStyle]
    self._subtotalStyle.append(('LINEABOVE', (0,0), (-1,-1), 0.25, colors.black))
    self._subtotalStyle.append(('FONTNAME', (0,0), (-1,-1), self.boldfont))

  def BeginColHead(self):
    pass


  def EndColHead(self, colhead, width, alignment=None):

    # TODO: temporary... should be calculated
    if not width:
      width = 15

    self._tableHead.append(colhead.encode('utf-8'))
    self._tableAlign.append(alignment)
    self._colwidth.append(width)
    self._headwidth += width
    self._numCols += 1


  def BeginRow(self, rowtype=None):
    self._row = []
    if rowtype == 'subtotal':
      self._rowStyle = self._subtotalStyle
    else:
      if self._odd:
        self._rowStyle = self._oddStyle
      else:
        self._rowStyle = self._tableStyle


  def EndRow(self):
    t = Table([self._row], colWidths=self._colwidth, style=self._rowStyle)
    self._body.append(t)
    self._odd = not self._odd


  def BeginCol(self):
    pass


  def EndCol(self, col, index):
    self._row.append(col.encode('utf-8'))




