# GNU Enterprise Application Server - Generators - Form Generator
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: form.py 9953 2009-10-11 18:50:17Z reinhard $

import classdef
import layout
import sys
import operator

# =============================================================================
# This class implements the form generator
# =============================================================================

class FormGenerator:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, sess, klass, lang, connection, formWidth, formHeight):
    """
    Create a new instance of a form generator.

    @param sess: language interface Session instance
    @param klass: language interface object of the class to generate a form for
    @param lang: language to generate the form for, i.e. 'de_AT'
    @param connection: name of the connection to use for datasources
    @param formWidth: maximum width of the generated form
    @param formHeight: maximum height of the generated form
    """
    
    self.__session    = sess
    self.__class      = klass
    self.__language   = lang
    self.__connection = connection
    self.__maxWidth   = formWidth
    self.__maxHeight  = formHeight

    self.__classDef   = classdef.ClassDef (sess, klass, lang)

    self.__updateLogic ()

    self.layoutManager = self.__createManager ()

    self.__options = self.__xml ('options',
        {None: self.__xml ('author',
          {None: [u_("GNU Enterprise Application Server")]}) + \
               self.__xml ('version', {None: ['1.0']}) + \
               self.__xml ('description',
                 {None: [u_('Generated form for class "%s"') \
                     % self.__classDef.classname]})})


  # ---------------------------------------------------------------------------
  # Run the generator
  # ---------------------------------------------------------------------------

  def run (self):
    """
    This function executes the generator and returns a string with the form's
    XML code.

    @return: string with XML code of the generated form
    """

    try:
      code = [u'<?xml version="1.0" encoding="utf-8"?>']
      code.extend (self.__xml ('form', {'title': self.__classDef.classLabel,
                                        None: self.__options}, "", True))

      for item in self.__sources.values ():
        code.extend (self.__xml ('datasource', item, "  "))

      code.extend (self.__xml ('logic', {None: self.__blocksToXML ()}, "  "))

      tabPos = len (self.layoutManager.visualPages) > 1 and 'top' or 'none'
      code.extend (self.__xml ('layout', {'xmlns:c': 'GNUe:Layout:Char',
          'c:height': self.layoutManager.maxPageHeight,
          'c:width' : self.layoutManager.maxPageWidth,
          'tabbed'  : tabPos,
          None: self.__pagesToXML ()}, "  "))

      code.append ('</form>')

      return "\n".join (code)

    finally:
      self.__classDef.release ()


  # ---------------------------------------------------------------------------
  # Update the logic of the form (blocks and datasources)
  # ---------------------------------------------------------------------------

  def __updateLogic (self):
    """
    This function updates both dictionaries __sources and __blocks based on the
    class definition.
    """

    self.__sources = {}
    self.__blocks  = {}

    attrs = {}
    order = self.__classDef.sortColumn ()
    if order is not None:
      attrs [None] = self.__xml ('sortorder',
            {None: self.__xml ('sortfield', {'name': order})})

    self.__addToSources ('dtsMaster', self.__classDef.classname, attrs)

    for item in self.__classDef.properties + self.__classDef.specials:
      # First process the property 
      #if item.reference: # is None or item.reference.isLookup:
      self.__addToBlocks ('blkMaster', 'dtsMaster', item)
      item.block = 'blkMaster'

      # and if it is a reference all parts of the reference
      if item.reference is not None:
        for refItem in item.reference.properties:
          table = item.reference.classname

          # dropdown widgets need an additional datasource for their values
          source = "dts%s%s" % (item.fullName, item.reference.classname)
          refItem.block  = 'blkMaster'
          refItem.source = source

          self.__addToSources (source, table, {'prequery': 'Y'})



  # ---------------------------------------------------------------------------
  # Add/Update a datasource
  # ---------------------------------------------------------------------------

  def __addToSources (self, name, table, attributes = {}):
    """
    This function adds or updates the given datasource in the datasource
    dictionary.

    @param name: name of the datasource
    @param table: name of the table the datasource is bound to
    @param attributes: dictionary with additional tags for the datasource
    """

    if not self.__sources.has_key (name):
      self.__sources [name] = {}

    self.__sources [name].update (attributes)
    self.__sources [name] ['name']       = name
    self.__sources [name] ['table']      = table
    self.__sources [name] ['connection'] = self.__connection


  # ---------------------------------------------------------------------------
  # Add a property to the given block
  # ---------------------------------------------------------------------------

  def __addToBlocks (self, name, source, field):
    """
    This function appends the given field to the blocks dictionary

    @param name: name of the block to be extended
    @param source: name of the datasource the block is bound to
    @param field: property instance of the field to be added
    """

    if not self.__blocks.has_key (name):
      self.__blocks [name] = {'source': source, 'fields': []}

    self.__blocks [name] ['fields'].append (field)


  # ---------------------------------------------------------------------------
  # Create an apropriate layout manager
  # ---------------------------------------------------------------------------

  def __createManager (self):
    """
    This function instanciates a layout manager. If a class definition has only
    a single page and all items don't exceed available horizontal space a
    tabular otherwise a vertical layout manager will be used.

    @return: layout manager instance
    """

    result = layout.Vertical

    if len (self.__classDef.virtualPages) == 1:
      width = []

      for item in self.__classDef.properties:
        width.append (max (len (item.label), item.widgetWidth ()))

      if sum (width) + len (width) - 1 <= self.__maxWidth:
        result = layout.Tabular

    return result (self.__classDef, self.__maxWidth, self.__maxHeight)


  # ---------------------------------------------------------------------------
  # Get an XML representation of all blocks and their fields
  # ---------------------------------------------------------------------------

  def __blocksToXML (self):
    """
    This function returns a XML sequence describing the blocks and their
    fields.

    @return: sequence with XML code for the blocks
    """

    code = []

    for (blockName, block) in self.__blocks.items ():
      fCode = []

      for field in block ['fields']:
        if field.reference is None:
          fDef = {'name': field.fieldName, 'field': field.dbField}
          if field.typecast is not None:
            fDef ['typecast'] = field.typecast
          if field.propDef.gnue_length and field.typecast != 'number':
            fDef ['maxLength'] = field.propDef.gnue_length

          fCode.extend (self.__xml ('field', fDef))

        else:
          for refItem in field.reference.properties:
            fDef = {'name'          : refItem.fieldName,
                    'field'         : field.dbField,
                    'fk_key'        : 'gnue_id',
                    'fk_source'     : refItem.source,
                    'fk_description': refItem.dbField}
            fCode.extend (self.__xml ('field', fDef))

        bDef = {'name'      : blockName,
                'datasource': block ['source'],
                None        : fCode}

      code.extend (self.__xml ('block', bDef))

    return code


  # ---------------------------------------------------------------------------
  # Transform the visual pages into XML code
  # ---------------------------------------------------------------------------

  def __pagesToXML (self):
    """
    This function generates the XML code sequence for all visual pages, created
    by the layout manager.

    @return: sequence with XML code containing all <page> tags
    """

    code     = []
    setFocus = True

    for (page, properties) in self.layoutManager.visualPages:
      pageCode = []
      
      # add the labels first
      for item in properties + self.__classDef.specials:
        if item.isSpecial:
          pData = self.layoutManager.getSpecialLocation (page, item)

          item.labelPos = pData ['labelPos']

        if item.labelPos is not None:
          pageCode.extend (self.__xml ('label', \
              {'c:width' : len (item.label) + 1,
               'c:height': 1,
               'c:x'     : item.labelPos [0],
               'c:y'     : item.labelPos [1],
               'text'    : "%s:" % item.label}))

      # add the entries
      for item in properties + self.__classDef.specials:
        if item.isSpecial:
          pData = self.layoutManager.getSpecialLocation (page, item)
          item.left     = pData ['left']
          item.row      = pData ['row']

        if item.reference is None:
          pageCode.extend (self.__xml ('entry',
                                      self.__entryAttributes (item, setFocus)))

        else:
          for refItem in item.reference.properties:
            pageCode.extend (self.__xml ('entry',
                                   self.__entryAttributes (refItem, setFocus)))

          if not item.reference.isLookup:
            first = item.reference.properties [0]
            button = {'c:height': 1,
                      'c:width' : 3,
                      'c:x': first.left + first.width,
                      'c:y': item.row,
                      'label': '...',
                      None: self.__getSearchTriggerCode (first)}
            pageCode.extend (self.__xml ('button', button))

        setFocus = False
                          
      code.extend (self.__xml ('page', {'name': page, None: pageCode}))

    return code


  # ---------------------------------------------------------------------------
  # Create a dictionary describing an entry
  # ---------------------------------------------------------------------------

  def __entryAttributes (self, item, setFocusOrder):
    """
    This function creates a dictionary with attributes describing the given
    item.

    @param item: property instance with the item to get a dictionary for
    @param setFocusOrder: if True the result will get a key 'focusOrder'
    @return: dictionary with all needed keys for creating a form-entry
    """

    result = {'c:width' : item.width,
              'c:height': item.height,
              'c:x'     : item.left,
              'c:y'     : item.row,
              'field'   : item.fieldName,
              'block'   : item.block}

    if item.displaymask is not None:
      result ['displaymask'] = item.displaymask

    if item.inputmask is not None:
      result ['inputmask'] = item.inputmask

    if item.style is not None:
      result ['style'] = item.style

    if setFocusOrder:
      result ['focusorder'] = 1

    if hasattr (item, 'rows') and item.rows:
      result ['rows'] = item.rows

    return result


  # ---------------------------------------------------------------------------
  # create a trigger code for calling a search dialog
  # ---------------------------------------------------------------------------

  def __getSearchTriggerCode (self, item):
    """
    This function generates a button trigger for calling a search-dialog. 
    NOTE: this feature is not *yet* implemented by appserver

    @param item: the search item

    @return: XML code sequence for the trigger
    """

    return self.__xml ('trigger', {'type': 'ON-ACTION',
     None: ["params = {'result': None, 'OK': False}",
            'runForm ("appserver://appserver/search/%s", params)' % \
                item.parent.classname,
            "if params.get ('OK'):",
            "  %s.%s.set (params ['result'])" % (item.block, item.fieldName)]})



  # ---------------------------------------------------------------------------
  # Create a porition of XML code
  # ---------------------------------------------------------------------------

  def __xml (self, tag, attrs, indent = "", keep = False):
    """
    This function create a sequence of XML code. If the attribute dictionary
    has a None-key, this sequence will be treated as contents of the tag. Such
    contents will be indented by another level.

    @param tag: name of the XML tag
    @param attrs: dictionary with attributes of the tag
    @param indent: indentation for each line
    @param keep: if True the tag is not closed, although attrs has no contents.
    """

    result = []
    parts  = []
    gap    = "  "

    if attrs.has_key (None):
      contents = []
      for element in attrs [None]:
        contents.extend (element.splitlines ())
      del attrs [None]

    else:
      contents = None

    keys = attrs.keys ()
    keys.sort ()
    if 'name' in keys:
      keys.remove ('name')
      keys.insert (0, 'name')

    close = (contents is None and not keep) and "/" or ""

    xmlString = "%s<%s" % (indent, tag)
    nextIndent = len (xmlString)

    for key in keys:
      add = '%s="%s"' % (key, attrs [key])
      if len (xmlString) + len (add) > 76:
        result.append (xmlString)
        xmlString = " " * nextIndent

      xmlString = "%s %s" % (xmlString, add)

    xmlString = "%s%s>" % (xmlString, close)
    result.append (xmlString)

    if contents is not None:
      iLen = len (indent) + len (gap)
      cString = "\n".join (["%s%s" % (" " * iLen, i) for i in contents])

      result.append (cString)
      if not keep:
        result.append ("%s</%s>" % (indent, tag))

    return result
