# GNU Enterprise Application Server - Instance Object
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: geasInstance.py 9953 2009-10-11 18:50:17Z reinhard $

import sys
import datetime
import mx.DateTime

from gnue import appserver
from gnue.appserver.language import Object, Session
from gnue.common.logic import language
from gnue.common.apps import errors
from gnue.common.utils import GDateTime

# =============================================================================
# Exceptions
# =============================================================================

class DbValueError (errors.AdminError):
  def __init__ (self, propertyName, value):
    message = u_("Database returned invalid value '%(value)s' for property "
                 "'%(property)s'") \
              % {"value"   : repr (value),
                 "property": propertyName}
    errors.AdminError.__init__ (self, message)

class PropertyValueError (errors.UserError):
  def __init__ (self, propertyName, value):
    message = u_("Invalid value '%(value)s' for property '%(property)s'") \
              % {"value": repr (value),
                 "property": propertyName}
    errors.UserError.__init__ (self, message)

class ParameterValueError (errors.UserError):
  def __init__ (self, parameterName, value):
    message = u_("Invalid value '%(value)s' for parameter '%(parameter)s'") \
              % {"value"    : repr (value),
                 "parameter": parameterName}
    errors.UserError.__init__ (self, message)

class ResultTypeError (errors.ApplicationError):
  def __init__ (self, procedure, typename, result):
    msg = u_("Procedure '%(procedure)s' expected a result of type "
             "'%(type)s', but got type '%(resulttype)s'") \
             % {"procedure" : procedure,
                "type"      : typename,
                "resulttype": type (result).__name__}
    errors.ApplicationError.__init__ (self, msg)

class PropertyAccessError (errors.ApplicationError):
  def __init__ (self, classname, propertyname):
    msg = u_("Calculated field '%(property)s' is a read-only field") \
          % {"property": "%s.%s" % (classname, propertyname)}
    errors.ApplicationError.__init__ (self, msg)

class ReferenceError (errors.ApplicationError):
  def __init__ (self, name, part):
    msg = u_("Cannot resolve propertyname '%(name)s' because '%(part)s' is " 
             "not a reference property") \
          % {'name': name,
             'part': part}
    errors.ApplicationError.__init__ (self, msg)

class OrderBySequenceError (errors.ApplicationError):
  pass

# =============================================================================
# Instance class
# =============================================================================

class geasInstance:

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, session, connection, record, classdef):
    self.__session    = session
    self.__connection = connection
    self.__record     = record
    self.__classdef   = classdef

    self.__connection.setConstraints (classdef.table, classdef.masters)


  # ---------------------------------------------------------------------------
  # Convert a value into the given type if possible
  # ---------------------------------------------------------------------------

  def __convert (self, value, propertydef, exception):

    # TODO: use gnue-common's datatypes.convert() function for this.

    if value is None:
      # "None" is always valid, independent of data type
      return None

    elif propertydef.dbType == "string":
      # String property: we expect the database to deliver a unicode string
      if isinstance (value, unicode):
        return value
      elif isinstance (value, str):
        return unicode (value)
      else:
        try:
          return unicode (str (value))
        except:
          raise exception (propertydef.fullName, value)

    elif propertydef.dbType == "number":
      # Number property: Must be something that can be converted to an int or
      # float
      try:
        if propertydef.gnue_scale:
          # ... with fractional part
          return float (value)
        else:
          # ... without fractional part
          return int (value)
      except ValueError:
        raise exception (propertydef.fullName, value)
      
    elif propertydef.dbType == "date":
      # Date property: Must be a datetime.date or at least parseable as such
      if isinstance (value, basestring):
        value = GDateTime.parseISO (value)

      if isinstance (value, datetime.datetime):
        return value.date ()

      elif isinstance (value, datetime.date):
        return value

      elif isinstance (value, mx.DateTime.DateTimeType):
        return datetime.date (value.year, value.month, value.day)

      else:
        raise exception (propertydef.fullName, value)

    elif propertydef.dbType == "time":
      # Time property: Must be a datetime.time or at least parseable as such
      if isinstance (value, basestring):
        value = GDateTime.parseISO (value)

      if isinstance (value, datetime.datetime):
        return value.time ()

      elif isinstance (value, datetime.time):
        return value

      elif isinstance (value, datetime.timedelta):
        return (datetime.datetime (1, 1, 1) + value).time ()

      elif isinstance (value, mx.DateTime.DateTimeType) or \
         isinstance (value, mx.DateTime.DateTimeDeltaType):
        return datetime.time (value.hour, value.minute, int (value.second),
            int ((value.second - int (value.second)) * 1000000))

      else:
        raise exception (propertydef.fullName, value)

    elif propertydef.dbType == "datetime":
      # Date/Time property: Must be a datetime.datetime or at least parseable
      # as such
      if isinstance (value, basestring):
        value = GDateTime.parseISO (value)

      if isinstance (value, datetime.datetime):
        return value

      elif isinstance (value, datetime.date):
        return datetime.datetime (value.year, value.month, value.day)

      elif isinstance (value, mx.DateTime.DateTimeType):
        return datetime.datetime (value.year, value.month, value.day,
            value.hour, value.minute, int (value.second),
            int ((value.second - int (value.second)) * 1000000))

      else:
        raise exception (propertydef.fullName, value)

    elif propertydef.dbType == "boolean":
      # Boolean property: Must be something meaningful
      if value in [0, "0", "f", "F", "false", "FALSE", "n", "N", "no", "NO",
                   False]:
        return False
      elif value in [1, "1", "t", "T", "true", "TRUE", "y", "Y", "yes", "YES",
                     True]:
        return True
      else:
        raise exception (propertydef.fullName, value)

    else:
      # Unknown property type
      raise exception (propertydef.fullName, value)

  # ---------------------------------------------------------------------------
  # Get the value of a property
  # ---------------------------------------------------------------------------

  def __getValue (self, propertyname):
    classdef = self.__classdef
    record   = self.__record

    # resolve indirect properties
    elements = propertyname.split (u'.')
    if len (elements) > 1:
      for e in elements [:-1]:
        propertydef = classdef.properties [e]
        if not propertydef.isReference:
          raise ReferenceError, (propertyname, propertydef.fullName)
        classdef = propertydef.referencedClass
        key = record.getField (propertydef.column)
        # if any reference on the way through the property name is None, assume
        # the indirect property is None.
        if key is None:
          return None
        record = self.__connection.findRecord (classdef.table, key, [])

    propertydef = classdef.properties [elements [-1]]
    if propertydef.isCalculated:
      if len (elements) > 1:
        instance = geasInstance (self.__session, self.__connection, record,
                                 classdef)
      else:
        instance = self
      value = instance.call (propertydef.procedure, None)
    else:
      value = record.getField (propertydef.column)

    return self.__convert (value, propertydef, DbValueError)


  # ---------------------------------------------------------------------------
  # Get the values of a list of properties
  # ---------------------------------------------------------------------------

  def get (self, propertylist):
    return [self.__getValue (aProperty) for aProperty in propertylist]

  # ---------------------------------------------------------------------------
  # Set the value of a property
  # ---------------------------------------------------------------------------

  def __putValue (self, propertyname, value, regular = True):

    propertydef = self.__classdef.properties [propertyname]
    if propertydef.isCalculated:
      raise PropertyAccessError, (self.__classdef.fullName, propertyname)

    __value = self.__convert (value, propertydef, PropertyValueError)

    if propertydef.isReference and __value is not None:
      # check wether the referenced object exists or not
      table = propertydef.referencedClass.table
      record = self.__connection.findRecord (table, __value, [u'gnue_id'])
      if record is None or record.getField (u'gnue_id') != __value:
        raise PropertyValueError, (propertydef.fullName, value)

    # Do not call OnChange triggers while in OnInit code and when setting
    # time/user stamp fields
    if regular and self.state () != 'initializing':
      for trigger in self.__classdef.OnChange:
        self.call (trigger, {}, {'oldValue': self.__getValue (propertyname),
                                 'newValue': __value,
                                 'propertyName': propertydef.fullName})

    self.__record.putField (propertydef.column, __value)


  # ---------------------------------------------------------------------------
  # Set the values of a list of properties
  # ---------------------------------------------------------------------------

  def put (self, propertylist, valuelist):

    for i in range (0, len (propertylist)):
      self.__putValue (propertylist [i], valuelist [i])

  # ---------------------------------------------------------------------------
  # Call a procedure
  # ---------------------------------------------------------------------------

  def call (self, proceduredef, params, namespace = None):

    # TODO: This should run in a separate process so that a segfaulting
    # procedure doesn't kill appserver.
    # (needs to be implemented as an option in gnue-common)
    checktype (namespace, [None, dict])
    if namespace is None:
        namespace = {}

    # Create a session object which with the actual session id
    sess = Session.Session (self.__session, self.__session.parameters)

    # set context for the procedure
    sess.setcontext (proceduredef.gnue_module.gnue_name)

    # Create an object representing the current business object
    obj = Object.Object (sess, self.__session, self.__classdef.fullName,
                         self.__getValue (u'gnue_id'))

    # check and convert the parameters
    parameters = {}
    if params is not None:
      for name in [x.encode ('ascii') for x in params.keys ()]:
        paramDef = proceduredef.parameters [name]
        parameters [name] = self.__convert (params [name], paramDef,
                                              ParameterValueError)

    local_namespace = {'self': obj}
    local_namespace.update(namespace)

    execution_context = language.create_execution_context(
            language = proceduredef.gnue_language,
            name = '%s.%s' % (self.__classdef.fullName, proceduredef.fullName),
            local_namespace = local_namespace,
            global_namespace = {
                'session': sess,
                'find':  sess.find,
                'setcontext': sess.setcontext,
                'new': sess.new},
            builtin_namespace = {
                'message': sess.message})

    method = execution_context.build_function(
            name = proceduredef.gnue_name,
            parameters = parameters.keys(),
            code = proceduredef.gnue_code)
    result = method(**parameters)

    if (proceduredef.gnue_type is None) != (result is None):
      if result is not None or not proceduredef.gnue_nullable:
        raise ResultTypeError, (proceduredef.fullName, proceduredef.gnue_type,
                                result)
    return result

  # ---------------------------------------------------------------------------
  # Validate all properties of an instance
  # ---------------------------------------------------------------------------

  def validate (self):
    """
    This function checks all properties marked as 'not nullable' to have a
    value other than None. If a none value is encountered a PropertyValueError
    is raised.
    """

    for trigger in self.__classdef.OnValidate:
      self.call (trigger, None)
    
    # after finishing all OnValidate calls, have a look at the required fields
    for prop in self.__classdef.properties.values ():
      if not prop.isCalculated:
        if prop.gnue_nullable is not None and not prop.gnue_nullable:
          value = self.__record.getField (prop.column)
          if value is None:
            raise PropertyValueError, (prop.fullName, None)


  # ---------------------------------------------------------------------------
  # Dictionary-like access methods
  # ---------------------------------------------------------------------------
  def __getitem__ (self, key):
    return self.__getValue (key)


  # ---------------------------------------------------------------------------
  # Check wether a classdef has a property or not
  # ---------------------------------------------------------------------------

  def has_key (self, key):
    return self.__classdef.properties.has_key (key)


  # ---------------------------------------------------------------------------
  # Update instance-stamps (creation- or modification-date/user )
  # ---------------------------------------------------------------------------

  def updateStamp (self, creation = False):
    (datefield, userfield) = [('gnue_modifydate', 'gnue_modifyuser'),
                              ('gnue_createdate', 'gnue_createuser')][creation]
    if self.has_key (datefield):
      self.__putValue (datefield, datetime.datetime.now (), False)

    if self.has_key (userfield) and self.__session.user is not None:
      self.__putValue (userfield, self.__session.user, False)


  # ---------------------------------------------------------------------------
  # Get the state of the instances record in the cache
  # ---------------------------------------------------------------------------

  def state (self):
    """
    This function returns the current state of the instance.

    @return: state of the instance:
      'initializing': new instance, OnInit still running
      'initialized': new instance, OnInit finished, but no other modifications
      'inserted': new instance, already modified
      'changed': existing instance with modifications
      'deleted': deleted instance
      'clean': existing instance without any modifications
    """

    return self.__record.state ()


  # ---------------------------------------------------------------------------
  # Get the fully qualified classname of an instance
  # ---------------------------------------------------------------------------

  def getClassname (self):
    """
    This function returns the fully qualified classname of the instance
    @return: fully qualified classname of the instance
    """

    return self.__classdef.fullName


  # ---------------------------------------------------------------------------
  # Set a sequence of fields to order instances
  # ---------------------------------------------------------------------------

  def setSort (self, order):
    """
    Set a sequence of tuples used for comparing this instance with another one.

    @param order: sequence of tuples (value, descending), where data is
        the value to be compared and 'descending' specifies the sort-direction.
    """

    checktype (order, [None, list])
    self.__order = order


  # ---------------------------------------------------------------------------
  # Compare two instances
  # ---------------------------------------------------------------------------

  def __cmp__ (self, other):

    # If this or the other instance has no order-by rule, just do the
    # default-compare for instances
    if self.__order is None or other.__order is None:
      return cmp (self, other)

    # If both instance have an order-by rule, they must match in length
    if len (self.__order) != len (other.__order):
      raise OrderBySequenceError, \
          u_("Order-by sequence mismatch: '%(self)s' and '%(other)s'") \
          % {'self': self.__order, 'other': other.__order}

    for ix in xrange (len (self.__order)):
      (left, descending)  = self.__order [ix]
      (right, rightOrder) = other.__order [ix]

      if descending != rightOrder:
        raise OrderBySequenceError, \
            u_("Order-by sequence element has different directions: "
               "'%(self)s', '%(other)s'") \
            % {'self': self.__order [ix], 'other': other.__order [ix]}

      noneOpt = self.__session.nullFirstAsc
      if descending:
        (left, right) = (right, left)
        noneOpt = not self.__session.nullFirstDsc

      # NOTE: we use equality (==) here just for the case left or right is an
      # instance of NullObject which also evaluates to None
      if left == None or right == None:
        if not noneOpt:
          (left, right) = (right, left)
        result = cmp (left, right)
      else:
        result = cmp (left, right)

      if result != 0:
        return result

    # If no field gave a result, the two instances are treated to be equal
    return 0
