// This file is part of The New Aspell
// Copyright (C) 2001 by Kevin Atkinson under the GNU LGPL license
// version 2.0 or 2.1.  You should have received a copy of the LGPL
// license along with this library if you did not you can find
// it at http://www.gnu.org/.

//13-Sep-04
// changed memory_size and null_len_ from a static const size_t to an enum.
//the original was an error in VC.
/*
30-Oct-04
  Removed the memory data member from Convert and added getEncode() and getDecode()
  member functions.
  The orginal code had encode_ and decode_ members that could point to the
  default values (encode_d and decode_d respectivly) or objects that it
  constructed it's self. It used inplace construction using the data member
  memory. The encode_ and decode_ objects where not and could not be deleted
  since they could point to default objects which this class does not own.
  Since the destructor for Decode and Encode were not called, memory allocated
  to their members (aka String key) was never freed, resulting in a memory leak.
  Now encode_ is zero, unless there is a non-default value constructed.
  Now if encode_ is non-zero, the destructor for Convert deletes it.
  The same is true of decode_.
*/

#ifndef ASPELL_CONVERT__HPP
#define ASPELL_CONVERT__HPP

#include "aspellroot.h"
#include "string.hpp"
#include "posib_err.hpp"
#include "char_vector.hpp"
#include "filter_char.hpp"
#include "filter_char_vector.hpp"
#include "stack_ptr.hpp"
#include "filter.hpp"
#include "cache.hpp"

namespace acommon {

  class OStream;
  class Config;

  struct ConvBase : public Cacheable {
    typedef const Config CacheConfig;
    typedef const char * CacheKey;
    String key;
    bool cache_key_eq(const char * l) const  {return key == l;}
    ConvBase() {}
  private:
    ConvBase(const ConvBase &);
    void operator=(const ConvBase &);
  };

  struct Decode : public ConvBase {
    virtual PosibErr<void> init(ParmStr code, const Config &) {return no_err;}
    virtual void decode(const char * in, int size,
			FilterCharVector & out) const = 0;
    virtual PosibErr<void> decode_ec(const char * in, int size,
                                     FilterCharVector & out, ParmStr orig) const = 0;
    static PosibErr<Decode *> get_new(const String &, const Config *);
    virtual ~Decode() {}
  };
  struct Encode : public ConvBase {
    // null characters should be treated like any other character
    // by the encoder.
    virtual PosibErr<void> init(ParmStr, const Config &) {return no_err;}
    virtual void encode(const FilterChar * in, const FilterChar * stop, 
                        CharVector & out) const = 0;
    virtual PosibErr<void> encode_ec(const FilterChar * in, const FilterChar * stop, 
                                     CharVector & out, ParmStr orig) const = 0;
    // may convert inplace
    virtual bool encode(FilterChar * & in, FilterChar * & stop, 
                        FilterCharVector & buf) const {return false;}
    static PosibErr<Encode *> get_new(const String &, const Config *);
    virtual ~Encode() {}
  };
  struct DirectConv { // convert directly from in_code to out_code.
    // should not take ownership of decode and encode.
    // decode and encode guaranteed to stick around for the life
    // of the object.
    virtual PosibErr<void> init(const Decode *, const Encode *, 
				const Config &) {return no_err;}
    virtual void convert(const char * in, int size, 
			 CharVector & out) const = 0;
    virtual PosibErr<void> convert_ec(const char * in, int size, 
                                      CharVector & out, ParmStr orig) const = 0;
    virtual ~DirectConv() {}
  };

  template <class T> struct NormTable;
  struct FromUniNormEntry;
  struct ToUniNormEntry;
  struct NormTables : public Cacheable {
    typedef const Config CacheConfig;
    typedef const char * CacheKey;
    String key;
    bool cache_key_eq(const char * l) const  {return key == l;}
    static PosibErr<NormTables *> get_new(const String &, const Config *);
    NormTable<FromUniNormEntry> * internal;
    NormTable<FromUniNormEntry> * strict_d;
    NormTable<FromUniNormEntry> * strict;
    struct ToUniTable {
      String name;
      NormTable<ToUniNormEntry> * data;
      NormTable<ToUniNormEntry> * ptr;
      ToUniTable() : data(0), ptr(0) {} //13-Sep-04
    };
    typedef Vector<ToUniTable> ToUni;
    Vector<ToUniTable> to_uni;
    ~NormTables();
  };

  typedef FilterCharVector ConvertBuffer;

  class Convert {
  private:
    CachePtr<Decode> decode_d;
    Decode * decode_;
    CachePtr<Encode> encode_d;
    Encode * encode_;
    CachePtr<NormTables> norm_tables_;
    DirectConv * conv_;
//30-Oct-04#if 0
//30-Oct-04    static const size_t memory_size = 96;
//30-Oct-04#else
//30-Oct-04	 enum {memory_size = 96}; //13-Sep-04
//30-Oct-04#endif
//30-Oct-04    char memory[memory_size];

    ConvertBuffer buf_;
#if 0
    static const unsigned int null_len_ = 4; // POSIB FIXME: Be more precise
#else
	 enum { null_len_ = 4 }; //13-Sep-04
#endif

    Decode * getDecode() const { //30-Oct-04
		if (decode_) 
       return decode_ ;
		return decode_d.get();
	 }
	 Encode * getEncode() const { //30-Oct-04
		 if (encode_)
			 return encode_ ;
		 return encode_d.get();
	 }
    Convert(const Convert &);
    void operator=(const Convert &);

  public:
	  Convert() :decode_(0),encode_(0),conv_(0) 
	  {}
	 ~Convert() {
		 if (decode_)
			 delete decode_; //30-Oct-04
		 if (encode_)
			 delete encode_; //30-Oct-04
		 if (conv_)
			 delete conv_; //30-Oct-04
	 }


    // This filter is used when the convert method is called.  It must
    // be set up by an external entity as this class does not set up
    // this class in any way.
    Filter filter;

    PosibErr<void> init(const Config &, ParmStr in, ParmStr out);
    PosibErr<void> init_norm_to(const Config &, ParmStr in, ParmStr out);
    PosibErr<void> init_norm_from(const Config &, ParmStr in, ParmStr out);
    
    const char * in_code() const   {return getDecode()->key.c_str();}
    const char * out_code() const  {return getEncode()->key.c_str();}

    void append_null(CharVector & out) const
    {
      const char nul[4] = {0,0,0,0}; // 4 should be enough
      out.write(nul, null_len_);
    }

    unsigned int null_len() const {return null_len_;}
  
    // this filters will generally not translate null characters
    // if you need a null character at the end, add it yourself
    // with append_null

    void decode(const char * in, int size, FilterCharVector & out) const 
      {getDecode()->decode(in,size,out);}
    
    void encode(const FilterChar * in, const FilterChar * stop, 
		CharVector & out) const
      {getEncode()->encode(in,stop,out);}

    bool encode(FilterChar * & in, FilterChar * & stop, 
                FilterCharVector & buf) const
      {return getEncode()->encode(in,stop,buf);}

    // does NOT pass it through filters
    // DOES NOT use an internal state
    void convert(const char * in, int size, CharVector & out, ConvertBuffer & buf) const
    {
      if (conv_) {
	conv_->convert(in,size,out);
      } else {
        buf.clear();
        getDecode()->decode(in, size, buf);
        getEncode()->encode(buf.pbegin(), buf.pend(), out);
      }
    }

    // does NOT pass it through filters
    // DOES NOT use an internal state
    PosibErr<void> convert_ec(const char * in, int size, CharVector & out, 
                              ConvertBuffer & buf, ParmStr orig) const
    {
      if (conv_) {
	RET_ON_ERR(conv_->convert_ec(in,size,out, orig));
      } else {
        buf.clear();
        RET_ON_ERR(getDecode()->decode_ec(in, size, buf, orig));
       RET_ON_ERR(getEncode()->encode_ec(buf.pbegin(), buf.pend(), 
                                      out, orig));
      }
      return no_err;
    }


    // convert has the potential to use internal buffers and
    // is therefore not const.  It is also not thread safe
    // and I have no intention to make it thus.

    void convert(const char * in, int size, CharVector & out) {
      if (filter.empty()) {
        convert(in,size,out,buf_);
      } else {
        generic_convert(in,size,out);
      }
    }

    void generic_convert(const char * in, int size, CharVector & out);
    
  };

  bool operator== (const Convert & rhs, const Convert & lhs);

  ASPELL_API const char * fix_encoding_str(ParmStr enc, String & buf);

  // also returns true if the encoding is unknown
  bool ascii_encoding(const Config & c, ParmStr enc0);

  enum Normalize {NormNone, NormFrom, NormTo};

  ASPELL_API PosibErr<Convert *> internal_new_convert(const Config & c, 
                                           ParmString in, ParmString out,
                                           bool if_needed,
                                           Normalize n);
  
  static inline PosibErr<Convert *> new_convert(const Config & c,
                                                ParmStr in, ParmStr out,
                                                Normalize n)
  {
    return internal_new_convert(c,in,out,false,n);
  }
  
  static inline PosibErr<Convert *> new_convert_if_needed(const Config & c,
                                                          ParmStr in, ParmStr out,
                                                          Normalize n)
  {
    return internal_new_convert(c,in,out,true,n);
  }

  struct ConvObj {
    Convert * ptr;
    ConvObj(Convert * c = 0) : ptr(c) {}
    ~ConvObj() {delete ptr;}
    PosibErr<void> setup(const Config & c, ParmStr from, ParmStr to, Normalize norm)
    {
      delete ptr;
      ptr = 0;
      PosibErr<Convert *> pe = new_convert_if_needed(c, from, to, norm);
      if (pe.has_err()) return pe;
      ptr = pe.data;
      return no_err;
    }
    operator const Convert * () const {return ptr;}
  private:
    ConvObj(const ConvObj &);
    void operator=(const ConvObj &);
  };

  struct ConvP {
    const Convert * conv;
    ConvertBuffer buf0;
    CharVector buf;
    operator bool() const {return conv;} //13-Sep-04
    ConvP(const Convert * c = 0) : conv(c) {}
    ConvP(const ConvObj & c) : conv(c.ptr) {}
    ConvP(const ConvP & c) : conv(c.conv) {}
    void operator=(const ConvP & c) { conv = c.conv; }
    PosibErr<void> setup(const Config & c, ParmStr from, ParmStr to, 
                         Normalize norm)
    {
      delete conv;
      conv = 0;
      PosibErr<Convert *> pe = new_convert_if_needed(c, from, to, norm);
      if (pe.has_err()) return pe;
      conv = pe.data;
      return no_err;
    }
    char * operator() (char * str, size_t sz)
    {
      if (conv) {
        buf.clear();
        conv->convert(str, sz, buf, buf0);
#ifndef NULLTERMINATE
        buf.ensure_null_end();
#endif
        return buf.data();
      } else {
        return str;
      }
    }
    const char * operator() (const char * str, size_t sz)
    {
      if (conv) {
        buf.clear();
        conv->convert(str, sz, buf, buf0);
        return buf.str();
      } else {
        return str;
      }
    }
    char * operator() (MutableString str)
    {
      return operator()(str.str, str.size);
    }
    char * operator() (char * str)
    {
      if (conv) {
        buf.clear();
        conv->convert(str, -1, buf, buf0);
#ifndef NULLTERMINATE
        buf.ensure_null_end();
#endif
        return buf.data();
      } else {
        return str;
      }
    }
    const char * operator() (ParmStr str)
    {
      if (conv) {
        buf.clear();
        conv->convert(str, -1, buf, buf0);
#ifndef NULLTERMINATE
        buf.ensure_null_end();
#endif
        return buf.data();
      } else {
        return str;
      }
    }
    const char * operator() (char c)
    {
      buf.clear();
      if (conv) {
        char str[2] = {c, 0};
        conv->convert(str, 1, buf, buf0);
      } else {
        buf.append(c);
      }
#ifndef NULLTERMINATE
      buf.ensure_null_end();
#endif
      return buf.data();
    }
  };

  struct Conv : public ConvP
  {
    ConvObj conv_obj;
    Conv(Convert * c = 0) : ConvP(c), conv_obj(c) {}
    PosibErr<void> setup(const Config & c, ParmStr from, ParmStr to, Normalize norm)
    {
      RET_ON_ERR(conv_obj.setup(c,from,to,norm));
      conv = conv_obj.ptr;
      return no_err;
    }
  };

  struct ConvECP {
    const Convert * conv;
    ConvertBuffer buf0;
    CharVector buf;
    operator bool() const {return conv ? true: false;} //13-Sep-04
    ConvECP(const Convert * c = 0) : conv(c) {}
    ConvECP(const ConvObj & c) : conv(c.ptr) {}
    ConvECP(const ConvECP & c) : conv(c.conv) {}
    void operator=(const ConvECP & c) { conv = c.conv; }
    PosibErr<void> setup(const Config & c, ParmStr from, ParmStr to, Normalize norm)
    {
      delete conv;
      conv = 0;
      PosibErr<Convert *> pe = new_convert_if_needed(c, from, to, norm);
      if (pe.has_err()) return pe;
      conv = pe.data;
      return no_err;
    }
    PosibErr<char *> operator() (char * str, size_t sz)
    {
      if (conv) {
        buf.clear();
        RET_ON_ERR(conv->convert_ec(str, sz, buf, buf0, str));
#ifndef NULLTERMINATE
        buf.ensure_null_end();
#endif
        return buf.data();
      } else {
        return str;
      }
    }
    PosibErr<char *> operator() (MutableString str)
    {
      return operator()(str.str, str.size);
    }
    PosibErr<char *> operator() (char * str)
    {
      if (conv) {
        buf.clear();
        RET_ON_ERR(conv->convert_ec(str, -1, buf, buf0, str));
#ifndef NULLTERMINATE
        buf.ensure_null_end();
#endif
        return buf.data();
      } else {
        return str;
      }
    }

    PosibErr<const char *> operator() (ParmStr str)
    {
      if (conv) {
        buf.clear();
        RET_ON_ERR(conv->convert_ec(str, -1, buf, buf0, str));
#ifndef NULLTERMINATE
        buf.ensure_null_end();
#endif
        return buf.data();
      } else {
        return str.str();
      }
    }
    PosibErr<const char *> operator() (char c)
    {
      char buf2[2] = {c, 0};
      return operator()(ParmString(buf2,1));
    }
  };

  struct ConvEC : public ConvECP
  {
    ConvObj conv_obj;
    ConvEC(Convert * c = 0) : ConvECP(c), conv_obj(c) {}
    PosibErr<void> setup(const Config & c, ParmStr from, ParmStr to, Normalize norm)
    {
      RET_ON_ERR(conv_obj.setup(c,from,to,norm));
      conv = conv_obj.ptr;
      return no_err;
    }
  };

  struct MBLen 
  {
    enum Encoding {Other, UTF8, UCS2, UCS4} encoding;
    MBLen() : encoding(Other) {}
    ASPELL_API PosibErr<void> setup(const Config &, ParmStr enc);
    ASPELL_API unsigned operator()(const char * str, const char * stop);
    unsigned operator()(const char * str, unsigned byte_size) {
      return operator()(str, str + byte_size);}
  };

}

#endif
