/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: pgpTestPublicKey.c,v 1.4 1999/03/10 02:59:11 heller Exp $
____________________________________________________________________________*/

#include "pgpUtilities.h"
#include "pgpPublicKey.h"
#include "pgpKeys.h"
#include "pgpErrors.h"
#include "pgpTest.h"

#define GOOD 0
#define FAIL 1

#define KEYNAME "<test@pgp.com>"
#define KEYPASS "test"
#define HASHALG kPGPHashAlgorithm_SHA

static PGPByte inbuf[] = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,
						  0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10};

	static int
sMemoryCompare (PGPByte *b1, PGPByte *b2, PGPSize s)
{
	while (s--) {
		if (*b1++ != *b2++) {
			return (*--b1 - *--b2);
		}
	}
	return 0;
}


	static void
sTestEncrypt( PGPContextRef context, PGPMemoryMgrRef mgr, PGPKeyRef key,
	PGPPublicKeyMessageFormat fmt, PGPBoolean dofail,
	PGPOptionListRef passopt )
{
	PGPPublicKeyContextRef			pubContext;
	PGPPrivateKeyContextRef			privContext;
	PGPByte							*outbuf = NULL;
	PGPByte							*outbuf2 = NULL;
	PGPSize							inSize, outSize;
	PGPSize							outSize2;
	PGPError						err	= kPGPError_NoErr;

	/* Encrypt inbuf into outbuf */

	err = PGPNewPublicKeyContext( key, fmt, &pubContext );
	pgpTestAssertNoErr( err );

	err = PGPGetPublicKeyOperationSizes( pubContext, &inSize, &outSize, NULL );
	pgpTestAssertNoErr( err );

	pgpTestAssert (sizeof(inbuf) <= inSize);
	outbuf = (PGPByte *) PGPNewData( mgr, outSize, 0);
	pgpTestAssert ( IsntNull ( outbuf ) );

	err = PGPPublicKeyEncrypt( pubContext, inbuf, sizeof(inbuf), outbuf,
							   &outSize );
	pgpTestAssertNoErr( err );

	PGPFreePublicKeyContext( pubContext );
	pubContext = NULL;


	/* Optionally cause a failure */
	if (dofail)
		outbuf[0]++;

	/* Decrypt outbuf into outbuf2 */

	err = PGPNewPrivateKeyContext( key, fmt, &privContext, passopt,
									PGPOLastOption( context));
	pgpTestAssertNoErr( err );

	err = PGPGetPrivateKeyOperationSizes( privContext, &outSize2, NULL, NULL );
	pgpTestAssertNoErr( err );

	outbuf2 = (PGPByte *) PGPNewData( mgr, outSize2, 0);
	pgpTestAssert ( IsntNull ( outbuf2 ) );

	err = PGPPrivateKeyDecrypt( privContext, outbuf, outSize,
								outbuf2, &outSize2 );
	if( dofail ) {
		pgpTestAssert( IsPGPError( err ) );
		PGPFreePrivateKeyContext( privContext );
		PGPFreeData( outbuf );
		PGPFreeData( outbuf2 );
		return;
	}

	pgpTestAssertNoErr( err );
	PGPFreePrivateKeyContext( privContext );
	privContext = NULL;


	/* Compare outbuf2 with inbuf */

	pgpTestAssert( sizeof(inbuf) == outSize2 );
	pgpTestAssert( 0 == sMemoryCompare( inbuf, outbuf2, sizeof(inbuf) ) );

	/* Done */

	PGPFreeData( outbuf );
	PGPFreeData( outbuf2 );
}

	static void
sTestSignature( PGPContextRef context, PGPMemoryMgrRef mgr, PGPKeyRef key, 
	PGPPublicKeyMessageFormat fmt, PGPBoolean dofail,
	PGPOptionListRef passopt )
{
	PGPPublicKeyContextRef			pubContext;
	PGPPrivateKeyContextRef			privContext;
	PGPHashContextRef				hashContext;
	PGPByte							*sigbuf = NULL;
	PGPSize							sigSize;
	PGPError						err	= kPGPError_NoErr;

	/* Sign inbuf into outbuf */

	err = PGPNewPrivateKeyContext( key, fmt, &privContext, passopt,
									PGPOLastOption(context));
	pgpTestAssertNoErr( err );

	err = PGPGetPrivateKeyOperationSizes( privContext, NULL, NULL, &sigSize );
	pgpTestAssertNoErr( err );

	sigbuf = (PGPByte *) PGPNewData( mgr, sigSize, 0);
	pgpTestAssert ( IsntNull ( sigbuf ) );

	err	= PGPNewHashContext( mgr, HASHALG, &hashContext );
	PGPContinueHash( hashContext, inbuf, sizeof(inbuf) );

	err = PGPPrivateKeySign( privContext, hashContext, sigbuf, &sigSize );
	pgpTestAssertNoErr( err );

	PGPFreePrivateKeyContext( privContext );
	privContext = NULL;


	/* Optionally cause a failure */
	if (dofail)
		sigbuf[0]++;

	/* Verify signature in sigbuf */

	err = PGPNewPublicKeyContext( key, fmt, &pubContext );
	pgpTestAssertNoErr( err );

	err	= PGPNewHashContext( mgr, HASHALG, &hashContext );
	PGPContinueHash( hashContext, inbuf, sizeof(inbuf) );

	err = PGPPublicKeyVerifySignature( pubContext, hashContext, sigbuf,
									   sigSize );
	if( dofail )
		pgpTestAssert( IsPGPError( err ) );
	else
		pgpTestAssertNoErr( err );

	PGPFreePublicKeyContext( pubContext );
	pubContext = NULL;

	/* Done */

	PGPFreeData( sigbuf );
}


/*
 * This works with its own context so changing passphrase has no
 * side effects
 */
	void
TestLowLevelPublicKey( )
{
	PGPContextRef					context;
	PGPMemoryMgrRef					mgr;
	PGPKeyRef						key;
	PGPSubKeyRef					subkey;
	PGPKeySetRef					keys = NULL;
	PGPKeySetRef					keySet = NULL;
	PGPFilterRef					filter = NULL;
	PGPKeyListRef					klist = NULL;
	PGPKeyIterRef					kiter = NULL;
	PGPInt32						passbufferBits;
	PGPSize							passbufferBytes;
	PGPByte							*passbuffer = NULL;
	PGPError						err	= kPGPError_NoErr;

	err	= PGPNewContext( kPGPsdkAPIVersion, &context );
	pgpTestAssertNoErr( err );

	mgr = PGPGetContextMemoryMgr( context );

	/* Open key rings and find key we will use */

	err = PGPOpenDefaultKeyRings( context, kPGPKeyRingOpenFlags_Mutable,
								  &keys );
	if( IsPGPError( err ) )
		goto error;

	err	= PGPNewUserIDStringFilter( context, KEYNAME,
									kPGPMatchSubString, &filter );
	if ( IsPGPError( err ) )
		goto error;

	err	= PGPFilterKeySet( keys, filter, &keySet);
	if ( IsPGPError( err ) )
		goto error;
	PGPFreeFilter( filter );
	filter	= NULL;

	PGPOrderKeySet( keySet, kPGPAnyOrdering, &klist );
	PGPNewKeyIter( klist, &kiter );
	
	err = PGPKeyIterNext( kiter, &key );
	if ( IsPGPError( err ) )
		goto error;
	err = PGPKeyIterNextSubKey( kiter, &subkey );
	if ( IsPGPError( err ) )
		goto error;

	PGPFreeKeyIter( kiter );
	kiter = NULL;
	PGPFreeKeyList( klist );
	klist = NULL;

	PGPFreeKeySet( keySet );
	keySet = NULL;


	/* Test encryption */
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_PGP, GOOD,
				  PGPOPassphrase (context, KEYPASS));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_PGP, FAIL,
				  PGPOPassphrase (context, KEYPASS));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_PKCS1, GOOD,
				  PGPOPassphrase (context, KEYPASS));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_PKCS1, FAIL,
				  PGPOPassphrase (context, KEYPASS));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_IKE, GOOD,
				  PGPOPassphrase (context, KEYPASS));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_IKE, FAIL,
				  PGPOPassphrase (context, KEYPASS));
#if 0
/* ElGamal encryption not supported in X509 mode */
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_X509, GOOD,
				  PGPOPassphrase (context, KEYPASS));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_X509, FAIL,
				  PGPOPassphrase (context, KEYPASS));
#endif

	/* Test signature */
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_PGP, GOOD,
				  PGPOPassphrase (context, KEYPASS));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_PGP, FAIL,
				  PGPOPassphrase (context, KEYPASS));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_PKCS1, GOOD,
				  PGPOPassphrase (context, KEYPASS));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_PKCS1, FAIL,
				  PGPOPassphrase (context, KEYPASS));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_IKE, GOOD,
				  PGPOPassphrase (context, KEYPASS));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_IKE, FAIL,
				  PGPOPassphrase (context, KEYPASS));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_X509, GOOD,
				  PGPOPassphrase (context, KEYPASS));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_X509, FAIL,
				  PGPOPassphrase (context, KEYPASS));

	/* Change passphrase to a splitkey type buffer */
    if( IsPGPError( PGPGetKeyNumber( key, kPGPKeyPropLockingBits,
                                     &passbufferBits ) ) )
        goto error;
    passbufferBytes = (passbufferBits + 7) / 8;
    passbuffer = (PGPByte *) PGPNewData(mgr, passbufferBytes, 0);
    PGPContextGetRandomBytes( context, passbuffer, passbufferBytes );

	err = PGPChangePassphrase( key,
                    PGPOPassphrase ( context, KEYPASS ),
                    PGPOPasskeyBuffer (context, passbuffer, passbufferBytes),
                    PGPOLastOption (context) );
	pgpTestAssertNoErr( err );
	if( IsntNull( subkey ) ) {
		err = PGPChangeSubKeyPassphrase( subkey,
					PGPOPassphrase ( context, KEYPASS ),
					PGPOPasskeyBuffer (context, passbuffer, passbufferBytes),
					PGPOLastOption (context) );
		pgpTestAssertNoErr( err );
	}

	/* Test encryption */
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_PGP, GOOD,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_PGP, FAIL,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_PKCS1, GOOD,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_PKCS1, FAIL,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_IKE, GOOD,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_IKE, FAIL,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
#if 0
/* ElGamal encryption not supported in X509 mode */
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_X509, GOOD,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestEncrypt( context, mgr, key, kPGPPublicKeyMessageFormat_X509, FAIL,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
#endif

	/* Test signature */
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_PGP, GOOD,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_PGP, FAIL,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_PKCS1, GOOD,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_PKCS1, FAIL,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_IKE, GOOD,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_IKE, FAIL,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_X509, GOOD,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));
	sTestSignature( context, mgr, key, kPGPPublicKeyMessageFormat_X509, FAIL,
				  PGPOPasskeyBuffer (context, passbuffer, passbufferBytes));

error:

	if( IsntNull( filter ) )
		PGPFreeFilter( filter );
	if( IsntNull( kiter ) )
		PGPFreeKeyIter( kiter );
	if( IsntNull( klist ) )
		PGPFreeKeyList( klist );
	if( IsntNull( keySet ) )
		PGPFreeKeySet( keySet );
	if( IsntNull( keys ) )
		PGPFreeKeySet( keys );
	if( IsntNull( passbuffer ) )
		PGPFreeData( passbuffer );

	(void)PGPFreeContext( context );
}


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
