/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: CPhotoUserID.cp,v 1.12 1999/03/10 02:38:54 heller Exp $
____________________________________________________________________________*/

#include <Folders.h>
#include <Scrap.h>

#include <LDragTask.h>
#include <PP_Messages.h>
#include <UDrawingState.h>
#include <UDrawingUtils.h>
#include <UGAColorRamp.h>
#include <UGraphicUtils.h>

#include "MacErrors.h"
#include "MacFiles.h"
#include "MacQuickDraw.h"
#include "PGPMacMemory.h"
#include "pgpEndianConversion.h"
#include "pgpJPEG.h"
#include "pgpClientShared.h"
#include "pgpUtilities.h"

#include "CPhotoUserID.h"
#include "CPGPKeys.h"

#define	kMaxPhotoIDWidth	120
#define kMaxPhotoIDHeight	144

const ResIDT	kInvalidPhotoIDResID 			= 2102;
const ResIDT	kRevokedPhotoIDResID 			= 2104;
const ResIDT	kExpiredPhotoIDResID 			= 2106;

CPhotoUserID::CPhotoUserID(LStream *inStream) :
		LPane( inStream ),
		LDragAndDrop( GetMacPort(), this )
{
	mLocked = FALSE;
	mGWorld	= NULL;
	mStatus	= kPGPPhotoUserIDStatus_Normal;
}

CPhotoUserID::~CPhotoUserID(void)
{
	FreeGWorld();
}

	Point
CPhotoUserID::CalcScaledImageSize(Point imageSize)
{
	Point	scaledSize;
	float	scaleFactor;
	
	scaledSize = imageSize;
	
	// Compute scaled picture 
	if( scaledSize.h > kMaxPhotoIDWidth )
	{
		scaleFactor = (float) kMaxPhotoIDWidth / (float) scaledSize.h;
		
		scaledSize.h = kMaxPhotoIDWidth;
		scaledSize.v = scaledSize.v * scaleFactor;
	}

	if( scaledSize.v > kMaxPhotoIDHeight )
	{
		scaleFactor = (float) kMaxPhotoIDHeight / (float) scaledSize.v;
		
		scaledSize.v = kMaxPhotoIDHeight;
		scaledSize.h = scaledSize.h * scaleFactor;
	}
	
	return( scaledSize );
}

	OSStatus
CPhotoUserID::AllocateGWorld(Point imageSize)
{
	OSStatus	err = noErr;
	
	if( IsntNull( mGWorld ) )
	{
		if( mGWorldSize.v != imageSize.v ||
			mGWorldSize.h != imageSize.h )
		{
			FreeGWorld();
		}
	}
	
	if( IsNull( mGWorld ) )
	{
		Rect	imageRect;
		
		SetRect( &imageRect, 0, 0, imageSize.h, imageSize.v );
		
		err = NewGWorld( &mGWorld, 32, &imageRect, NULL, NULL, useTempMem );
		if( IsntErr( err ) )
		{
			mGWorldSize = imageSize;
		}
	}
	
	SetUpdateCommandStatus( TRUE );
	
	return( err );
}

	void
CPhotoUserID::FreeGWorld(void)
{
	if( IsntNull( mGWorld ) )
	{
		DisposeGWorld( mGWorld );
		mGWorld = NULL;
		
		SetUpdateCommandStatus( TRUE );
	}
}

	void
CPhotoUserID::FindCommandStatus(
	CommandT	inCommand,
	Boolean		&outEnabled,
	Boolean		&outUsesMark,
	Char16		&outMark,
	Str255		outName)
{
	switch( inCommand )
	{
		case cmd_Copy:
			outEnabled = HavePicture();
			break;

		case cmd_Clear:
			outEnabled = HavePicture() && ! IsLocked();
			break;
			
		case cmd_Paste:
		{
			long	offset;
			
			outEnabled = ( GetScrap( NULL, 'PICT', &offset ) >= 0 &&
							! IsLocked() );
			break;
		}
		
		default:
			LCommander::FindCommandStatus( inCommand, outEnabled,
					outUsesMark, outMark, outName );
			break;
	}
}

	Boolean
CPhotoUserID::ObeyCommand(
	CommandT	inCommand,
	void*		ioParam)
{
	Boolean		cmdHandled = FALSE;
	
	switch( inCommand )
	{
		case cmd_Paste:
		{
			PicHandle	picture;
			long		offset;
			
			picture = (PicHandle) pgpNewHandle( 0, kMacMemory_PreferTempMem |
											kMacMemory_UseApplicationHeap );
			
			if( GetScrap( (Handle) picture, 'PICT', &offset ) > 0 )
			{
				OSStatus	err;
				
				err = SetPictureData( picture, TRUE );
				pgpAssertNoErr( err );
			}
			
			pgpDisposeHandle( (Handle) picture );
			
			cmdHandled = TRUE;
			break;
		}
		
		case cmd_Copy:
		{
			if( HavePicture() )
			{
				PicHandle	picture;
				OSStatus	err;
				
				err = GetPictureData( &picture );
				if( IsntErr( err ) )
				{
					HLock( (Handle) picture );
					
					ZeroScrap();
					err = PutScrap( GetHandleSize( (Handle) picture ),
										'PICT', *picture );
					pgpAssertNoErr( err );
					
					DisposeHandle( (Handle) picture );
				}
				
				pgpAssertNoErr( err );
			}
			
			cmdHandled = TRUE;
			break;
		}
		
		case cmd_Clear:
			ClearPicture();
			cmdHandled = TRUE;
			break;
			
		case msg_TabSelect:
			cmdHandled = TRUE;
			break;
			
		default:
			LCommander::ObeyCommand( inCommand, ioParam );
			break;
	}
	
	return cmdHandled;
}

	void
CPhotoUserID::ClearPicture(void)
{
	FreeGWorld();
	Refresh();
}
	
	Boolean
CPhotoUserID::ItemIsAcceptable(
	DragReference	inDragRef,
	ItemReference	inItemRef)
{
	Boolean		isAcceptable = FALSE;
	
	if( ! IsLocked() )
	{
		ushort		numFlavors;
		OSStatus	status;

		status = CountDragItemFlavors(inDragRef, inItemRef, &numFlavors);
		if(IsntErr(status))
		{
			ushort flavorIndex;
			
			for( flavorIndex = 1; flavorIndex <= numFlavors; flavorIndex++ )
			{
				FlavorType	flavorType;
				
				status = GetFlavorType(inDragRef, inItemRef, flavorIndex,
										&flavorType);
				if(IsntErr(status))
				{
					if(flavorType == 'PICT')
					{
						isAcceptable = TRUE;
					}
					else if(flavorType == flavorTypeHFS)
					{
						HFSFlavor	flavorData;
						ByteCount	dataSize;
						ByteCount	dataOffset;
						
						dataSize 	= sizeof(flavorData);
						dataOffset	= 0;
						
						status = GetFlavorData(inDragRef, inItemRef,
										flavorType, &flavorData,
										(Int32 *) &dataSize,
										dataOffset);
						if(IsntErr(status))
						{
							// Check for dragged folders and/or disks
							
							if(flavorData.fileType == 'PICT')
							{
								isAcceptable = true;
							}
						}
					}
					
					if(isAcceptable)
						break;
				}
			}
		}
	}
	
	return(isAcceptable);
}

	void
CPhotoUserID::DoDropAreaHilite(
	DragReference 	inDragRef,
	Boolean			showHilite)
{
	RGBColor		tempColor;
	Rect			dropRect;
	StColorState	saveColorState;
	
	tempColor = UGAColorRamp::GetBlackColor();
	RGBForeColor( &tempColor );

	tempColor = UGAColorRamp::GetWhiteColor();
	RGBBackColor( &tempColor );

	CalcLocalFrameRect( dropRect );
	InsetRect( &dropRect, 1, 1 );

	StRegion	dropRgn( dropRect );
	
	if( showHilite )
	{
		ShowDragHilite( inDragRef, dropRgn, TRUE );
	}
	else
	{
		HideDragHilite( inDragRef );
	}
}

	void
CPhotoUserID::HiliteDropArea(DragReference inDragRef)
{
	DoDropAreaHilite( inDragRef, TRUE );
}

	void
CPhotoUserID::UnhiliteDropArea(DragReference inDragRef)
{
	DoDropAreaHilite( inDragRef, FALSE );
}
	
	OSStatus
CPhotoUserID::GetPictureData(PicHandle *picturePtr)
{
	OSStatus	err		= noErr;
	PicHandle	picture = NULL;
	
	if( HavePicture() )
	{
		GWorldPtr	tempGWorld;
		Rect		imageRect;
		
		SetRect( &imageRect, 0, 0, mGWorldSize.h, mGWorldSize.v );
		
		err = NewGWorld( &tempGWorld, 32, &imageRect, NULL, NULL, useTempMem );
		if( IsntErr( err ) )
		{
			CGrafPtr		savePort;
			GDHandle		saveDevice;
			OpenCPicParams	picParams;
			
			GetGWorld( &savePort, &saveDevice );
			SetGWorld( tempGWorld, NULL );
			
			LockPixels( GetGWorldPixMap( tempGWorld ) );
			
			pgpClearMemory( &picParams, sizeof( picParams ) );
			
			picParams.srcRect 	= imageRect;
			picParams.hRes		= 0x00480000;	// 72 DPI
			picParams.vRes		= 0x00480000;	// 72 DPI
			picParams.version	= -2;
			
			picture = OpenCPicture( &picParams );
			if( IsntNull( picture ) )
			{
				CopyBits( 	&((GrafPtr) mGWorld)->portBits,
							&((GrafPtr) tempGWorld)->portBits,
							&imageRect, &imageRect, srcCopy, NULL );
				ClosePicture();
			}
			
			SetGWorld( savePort, saveDevice );
			
			UnlockPixels( GetGWorldPixMap( tempGWorld ) );
			DisposeGWorld( tempGWorld );
		}
	}
	else
	{
		err = eofErr;
	}
	
	*picturePtr = picture;
	
	return( err );
}

	OSStatus
CPhotoUserID::SetPictureData(
	PicHandle 	picture,
	Boolean		scaleToFit)
{
	OSStatus	err = noErr;
	CGrafPtr	savePort;
	GDHandle	saveDevice;
	Rect		imageRect;
	Point		imageSize;
	
	GetGWorld( &savePort, &saveDevice );
	
	// Image the picture into a GWorld.
	imageRect = (**picture).picFrame;
	
	imageSize.h = UGraphicUtils::RectWidth( imageRect );
	imageSize.v = UGraphicUtils::RectHeight( imageRect );
	
	if( scaleToFit )
		imageSize = CalcScaledImageSize( imageSize );
	
	err = AllocateGWorld( imageSize );
	if( IsntErr( err ) )
	{		
		SetRect( &imageRect, 0, 0, imageSize.h, imageSize.v );
		
		SetGWorld( mGWorld, NULL );
		LockPixels( GetGWorldPixMap( mGWorld ) );
		DrawPicture( picture, &imageRect );
		UnlockPixels( GetGWorldPixMap( mGWorld ) );
	}
	
	SetGWorld( savePort, saveDevice );
	Refresh();

	return( err );
}

	OSStatus
CPhotoUserID::AddPictureFromFile(const FSSpec *fileSpec)
{
	OSStatus	err;
	CInfoPBRec	cpb;
	
	err = FSpGetCatInfo( fileSpec, &cpb );
	if( IsntErr( err ))
	{
		short	fileRef;
		
		err = FSpOpenDF( fileSpec, fsRdPerm, &fileRef );
		if( IsntErr( err ) )
		{
			long	fileSize;
			
			err = GetEOF( fileRef, &fileSize );
			if( IsntErr( err ) )
			{
				if( fileSize > 512L )
				{
					err = SetFPos( fileRef, fsFromStart, 512L );
					if( IsntErr( err ) )
					{
						PicHandle	picture;
						PGPUInt32	bufferSize;
						
						bufferSize = fileSize - 512L;
						
						picture = (PicHandle) pgpNewHandle( bufferSize,
										kMacMemory_PreferTempMem |
										kMacMemory_UseApplicationHeap );
									
						if( IsntNull( picture ) )
						{
							long	count = bufferSize;
							
							HLock( (Handle) picture );
							
							err = FSRead( fileRef, &count, *picture );
							if( IsntErr( err ) )
							{
								err = SetPictureData( picture, TRUE );
							}
							
							pgpDisposeHandle( (Handle) picture );
						}
						else
						{
							err = memFullErr;
						}
					}
				}
				else
				{
					err = eofErr;
				}
			}
		
			FSClose( fileRef );
		}
	}
	
	return( err );
}

	void
CPhotoUserID::ReceiveDragItem(
	DragReference	inDragRef,
	DragAttributes	inDragAttrs,
	ItemReference	inItemRef,
	Rect&			inItemBounds)
{
	if( ( inDragAttrs & kDragInsideSenderWindow ) == 0 )
	{
		ushort		numFlavors;
		OSStatus	err;

		err = CountDragItemFlavors(inDragRef, inItemRef, &numFlavors);
		if( IsntErr( err ) )
		{
			for( ushort flavorIndex = 1; flavorIndex <= numFlavors;
						flavorIndex++ )
			{
				FlavorType	flavorType;
				
				err = GetFlavorType( inDragRef, inItemRef, flavorIndex,
										&flavorType );
				if( IsntErr( err ) )
				{
					if( flavorType == 'PICT' )
					{
						Size	dataSize;
						
						err = GetFlavorDataSize( inDragRef, inItemRef,
									flavorType, &dataSize );
						if( IsntErr( err ) )
						{
							PicHandle	picture;
							
							picture = (PicHandle) pgpNewHandle( dataSize,
										kMacMemory_PreferTempMem |
										kMacMemory_UseApplicationHeap );
							if( IsntNull( picture ) )
							{
								HLock( (Handle) picture );
								
								err = GetFlavorData( inDragRef, inItemRef,
											flavorType, *picture, &dataSize,
											0 );
								if( IsntErr( err ) )
								{
									err = SetPictureData( picture, TRUE );
								}
								
								pgpDisposeHandle( (Handle) picture );
							}
							else
							{
								err = memFullErr;
							}
						}
					}
					else
					{
						HFSFlavor	flavorData;
						ByteCount	dataSize;
						
						dataSize = sizeof( flavorData );
						
						err = GetFlavorData(inDragRef, inItemRef, flavorType,
										&flavorData, (Int32 *) &dataSize, 0 );
						if( IsntErr( err ) && flavorData.fileType == 'PICT' )
						{
							err = AddPictureFromFile( &flavorData.fileSpec );
							if( IsErr( err ) )
							{
								// Cannot display a dialog from the drag
								// manager
								SysBeep( 1 );
							}
						}
					}
				}
			}
		}
	}
}

	void
CPhotoUserID::DrawPhoto(
	const Rect 	*photoRect,
	SInt16		inBitDepth)
{
	StColorState	saveColors;
	ResIDT			pictResID = 0;
	ResIDT			maskResID = 0;
	
	ForeColor( blackColor );
	BackColor( whiteColor );
	
	if( HavePicture() )
	{
		Rect	sourceRect;
		Point	destSize;
		
		if( mGWorldSize.h > kMaxPhotoIDWidth ||
			mGWorldSize.v > kMaxPhotoIDHeight )
		{
			
			// Our GWorld is larger than the display area.
			// Scale the image
			
			destSize = CalcScaledImageSize( mGWorldSize );
		}
		else
		{
			destSize = mGWorldSize;
		}

		SetRect( &sourceRect, 0, 0, mGWorldSize.h, mGWorldSize.v );
		SetRect( &mImageRect, 0, 0, destSize.h, destSize.v );

		AlignRect( photoRect, &mImageRect, kAlignAtAbsoluteCenter );
		
		RGBForeColor( &UGAColorRamp::GetBlackColor() );
		RGBBackColor( &UGAColorRamp::GetWhiteColor() );
		
		CopyBits( &((GrafPtr)mGWorld)->portBits,
				&GetMacPort()->portBits,
				&sourceRect, &mImageRect, srcCopy | ditherCopy, NULL );

		switch( mStatus )
		{
			case kPGPPhotoUserIDStatus_Normal:
				/* No overlay picture */
				break;

			case kPGPPhotoUserIDStatus_NoValidity:
				pictResID = kInvalidPhotoIDResID;
				break;

			case kPGPPhotoUserIDStatus_Revoked:
				pictResID = kRevokedPhotoIDResID;
				break;

			case kPGPPhotoUserIDStatus_Expired:
				pictResID = kExpiredPhotoIDResID;
				break;
			
			default:
				pgpDebugMsg( "DrawPhoto: Unknown status");
				break;
		}
	
		if( inBitDepth >= 8 )
		{
			maskResID = pictResID + 1;
		}
	}
	
	if( pictResID != 0 )
	{
		PicHandle	thePicture;

		thePicture = GetPicture( pictResID );
		if( IsntNull( thePicture ) )
		{
			Rect	imageRect;
			
			HLock( (Handle) thePicture );
			
			imageRect = (**thePicture).picFrame;
			AlignRect( photoRect, &imageRect, kAlignAtAbsoluteCenter );

			if( maskResID != 0 )
			{
				PicHandle	maskPicture;
				CGrafPtr	savePort;
				GDHandle	saveDevice;
				Rect		gWorldRect;
				QDErr		err;
				
				/* Draw picture transparently using mask */
				GetGWorld( &savePort, &saveDevice );
				
				gWorldRect = imageRect;
				OffsetRect( &gWorldRect, -gWorldRect.left, -gWorldRect.top );
					
				maskPicture = GetPicture( maskResID );
				if( IsntNull( maskPicture ) )
				{
					GWorldPtr	pictGWorld;
					
					err = NewGWorld( &pictGWorld, 16, &gWorldRect, NULL, NULL,
											useTempMem );
					pgpAssertNoErr( err );
					if( IsntErr( err ) )
					{
						GWorldPtr	maskGWorld;

						SetGWorld( pictGWorld, NULL );
						ForeColor( blackColor );
						BackColor( whiteColor );
						
						LockPixels( GetGWorldPixMap( pictGWorld ) );
						DrawPicture( thePicture, &gWorldRect );
						
						err = NewGWorld( &maskGWorld, 16, &gWorldRect, NULL,
												NULL, useTempMem );
						pgpAssertNoErr( err );
						if( IsntErr( err ) )
						{
							SetGWorld( maskGWorld, NULL );
							ForeColor( blackColor );
							BackColor( whiteColor );
							
							LockPixels( GetGWorldPixMap( maskGWorld ) );
							DrawPicture( maskPicture, &gWorldRect );

							SetGWorld( savePort, saveDevice );

							CopyDeepMask(  	&((GrafPtr) pictGWorld)->portBits,
										&((GrafPtr) maskGWorld)->portBits,
										&GetMacPort()->portBits,
										&gWorldRect, &gWorldRect, &imageRect, srcCopy, NULL );
							
							UnlockPixels( GetGWorldPixMap( maskGWorld ) );
							DisposeGWorld( maskGWorld );
						}
						
						UnlockPixels( GetGWorldPixMap( pictGWorld ) );
						DisposeGWorld( pictGWorld );
					}
					
					ReleaseResource( (Handle) maskPicture );
				}

				SetGWorld( savePort, saveDevice );
			}
			else
			{
				/* Draw picture directly */			
				DrawPicture( thePicture, &imageRect );
			}
			
			ReleaseResource( (Handle) thePicture );
		}
	}
}

	void
CPhotoUserID::DrawSelf(void)
{
	Rect frame;

	CalcLocalFrameRect( frame );
	
	{
		StDeviceLoop	devLoop(frame);		
		SInt16			depth;

		while( devLoop.NextDepth( depth ) )
		{
			DrawPhoto( &frame, depth );
		}
	}
}

	void
CPhotoUserID::ClickSelf(
	const SMouseDownEvent & inMouseDown)
{
	LPane::ClickSelf( inMouseDown );

	if( ! IsTarget() )
		SwitchTarget( this );
		
	if( HavePicture() &&
		LDropArea::DragAndDropIsPresent() )
	{
		if( WaitMouseMoved( inMouseDown.macEvent.where ) )
		{
			PicHandle	picture;
			
			if( IsntErr( GetPictureData( &picture ) ) )
			{
				Rect				frameRect;
				Rect				dragRect = mImageRect;
				StPortOriginState	saveState( GetMacPort() );
				
				SetOrigin( 0, 0 );
				HLock( (Handle) picture );
				
				CalcPortFrameRect( frameRect );
				AlignRect( &frameRect, &dragRect, kAlignAtAbsoluteCenter );
				
				LDragTask	drag( inMouseDown.macEvent, dragRect,
									1, 'PICT', *picture,
									GetHandleSize( (Handle) picture), 0 );
									
				DisposeHandle( (Handle) picture );
			}
		}
	}
}

	OSStatus
CPhotoUserID::CreateTempJPEGFile(FSSpec *fileSpec)
{
	OSStatus	err;
	
	if( IsErr( FindFolder( kOnSystemDisk, kTemporaryFolderType,
				kCreateFolder, &fileSpec->vRefNum, &fileSpec->parID ) ) )
	{
		fileSpec->vRefNum 	= -1;
		fileSpec->parID		= fsRtDirID;
	}

	err = FSGetUniqueName( fileSpec->vRefNum, fileSpec->parID,
				"\pPGPkeys TEMP", fileSpec->name );
	if( IsntErr( err ) )
	{
		err = FSpCreate( fileSpec, 'pgpK', 'TEMP', 0 );
	}
	
	return( err );
}

	OSStatus
CPhotoUserID::JPEGToGWorld(JPEGDecompressionContext *jpegContext)
{
	OSStatus	err = noErr;

	// Setup JPEG decompression parameters
							
	jpegContext->dInfo.quantize_colors 	= FALSE;
	jpegContext->dInfo.out_color_space	= JCS_RGB;

	err = JPEGStartDecompress( jpegContext );
	if( err == noErr )
	{
		UInt32		bufferSize;
		void		*buffer;
		Point		imageSize;
		
		imageSize.h = jpegContext->dInfo.image_width;
		imageSize.v = jpegContext->dInfo.image_height;
		
		// Create buffer for one scan line
		bufferSize = imageSize.h * jpegContext->dInfo.out_color_components;

		buffer = pgpAllocMac( bufferSize, kMacMemory_PreferTempMem |
						kMacMemory_UseApplicationHeap );
		if( IsntNull( buffer ) )
		{
			UInt32		remainingLines;
			UInt16		gWorldRowBytes;
			char		*destBaseAddr;
			UInt32		scanLineIndex;
			
			destBaseAddr 	= GetPixBaseAddr( mGWorld->portPixMap );
			gWorldRowBytes 	= (**mGWorld->portPixMap).rowBytes & 0x7FFF;
			remainingLines 	= jpegContext->dInfo.output_height;
			
			for( scanLineIndex = 0; scanLineIndex < remainingLines;
					scanLineIndex++ )
			{
				JDIMENSION	count;
				
				err = PGPErrorToMacError( JPEGReadScanlines( jpegContext,
								(JSAMPROW *) &buffer, 1, &count ) );
				if( IsntErr( err ) )
				{
					UInt32	*curPixel	= (UInt32 *) destBaseAddr;
					char	*cur		= (char *) buffer;
					char	*lastPlus1	= cur + bufferSize;
				
					// back up one byte to account for alpha channel.
					--cur;
					--lastPlus1;
					
					while( cur < lastPlus1 )
					{
						// Clear the alpha channel
						*curPixel++ = (( *(UInt32 *) cur ) & 0x00FFFFFF);
						cur += 3;
					}

					destBaseAddr += gWorldRowBytes;
				}
				
				if( IsErr( err ) )
					break;
			}
			
			pgpAssert(	jpegContext->dInfo.output_scanline ==
						jpegContext->dInfo.output_height );
			
			pgpFreeMac( buffer );
		}
		
		JPEGFinishDecompress( jpegContext );
	}
	else
	{
		err = memFullErr;
	}
	
	return( err );
}

	OSStatus
CPhotoUserID::SetPhotoData(
	PGPContextRef	context,
	void			*data,
	PGPSize			dataSize,
	Boolean			scaleToFit)
{
	OSStatus				err = noErr;
	FSSpec					fileSpec;
	PGPFileSpecRef			fileSpecRef = kInvalidPGPFileSpecRef;
	PGPPhotoUserIDHeader	*header;
	PGPByte					*dataStart;
	PGPUInt16				headerLength;
	
	// Save data to a temporary file
	
	pgpFixBeforeShip( "Reject invalid JPEG color spaces" );

	header 			= (PGPPhotoUserIDHeader *) data;
	headerLength	= PGPStorageToUInt16( (PGPByte *) &header->headerLength );
	dataStart		= (PGPByte *) data + headerLength;
	
	if( header->headerVersion != kPGPPhotoIDHeaderVersion ||
		header->photoIDFormat != kPGPPhotoIDFormat_JPEG )
	{
		pgpDebugMsg( "SetPhotoData(): Invalid photo user ID data" );
		err = paramErr;
	}
	
	if( IsntErr( err ) )
	{
		err = CreateTempJPEGFile( &fileSpec );
		if( IsntErr( err ) )
		{
			err = PGPErrorToMacError( PGPNewFileSpecFromFSSpec( context,
											&fileSpec, &fileSpecRef ) );
		}
	}
	
	if( IsntErr( err ) )
	{
		short			fileRef;
		
		err = FSpOpenDF( &fileSpec, fsRdWrPerm, &fileRef );
		if( IsntErr( err ) )
		{
			long	count = dataSize - headerLength;
			
			err = FSWrite( fileRef, &count, dataStart );
			
			FSClose( fileRef );
		}
			
		if( IsntErr( err ) )
		{
			JPEGDecompressionContext	jpegContext;

			JPEGInitDecompressionContext( &jpegContext );

			err = JPEGCreateDecompress( &jpegContext );
			if( IsntErr( err ) )
			{
				err = JPEGOpenDecompressionFile( &jpegContext, fileSpecRef,
								TRUE );
				if( IsntErr( err ) )
				{
					err = JPEGReadHeader( &jpegContext, TRUE );
					if( IsntErr( err ) )
					{
						Point	imageSize;
					
						imageSize.h = jpegContext.dInfo.image_width;
						imageSize.v = jpegContext.dInfo.image_height;
						
						if( scaleToFit )
							imageSize = CalcScaledImageSize( imageSize );
							
						err = AllocateGWorld( imageSize );
						if( IsntErr( err ) )
						{
							err = JPEGToGWorld( &jpegContext );
							
							if( IsErr( err ) )
								FreeGWorld();	
						}
					}
					
					JPEGCloseDecompressionFile( &jpegContext );
				}
				
				JPEGDestroyDecompress( &jpegContext );
			}
			
		}
	
		FSpDelete( &fileSpec );
	}
	
	if( PGPFileSpecRefIsValid( fileSpecRef ) )
		PGPFreeFileSpec( fileSpecRef );

	Refresh();

	return( err );
}

	OSStatus
CPhotoUserID::GWorldToJPEG(JPEGCompressionContext *jpegContext)
{
	OSStatus	err = noErr;
	UInt32		bufferSize;
	void		*buffer;
	
	// specify image characteristics
	jpegContext->cInfo.image_width 		= mGWorldSize.h;
	jpegContext->cInfo.image_height 	= mGWorldSize.v;
	jpegContext->cInfo.input_components = 3;	
	jpegContext->cInfo.in_color_space 	= JCS_RGB;
	
	// Create buffer for one scan line
	bufferSize = mGWorldSize.h * jpegContext->cInfo.input_components;

	buffer = pgpAllocMac( bufferSize, kMacMemory_PreferTempMem |
					kMacMemory_UseApplicationHeap );
	if( IsntNull( buffer ) )
	{
		err = JPEGSetDefaults( jpegContext );
		if( IsntErr( err ) )
		{
			err = JPEGSetQuality( jpegContext, kPGPDefaultJPEQQuality, TRUE );
		}
		
		if( IsntErr( err ) )
		{
			err = JPEGStartCompress( jpegContext, TRUE );
			if( err == noErr )
			{
				UInt32		remainingLines;
				UInt16		gWorldRowBytes;
				char		*sourceBaseAddr;
				UInt32		scanLineIndex;
				
				sourceBaseAddr 	= GetPixBaseAddr( mGWorld->portPixMap );
				gWorldRowBytes 	= (**mGWorld->portPixMap).rowBytes & 0x7FFF;
				remainingLines 	= mGWorldSize.v;
				
				for( scanLineIndex = 0; scanLineIndex < remainingLines;
						scanLineIndex++ )
				{
					JDIMENSION	count;
					char		*curPixel		= sourceBaseAddr;
					char		*cur			= (char *) buffer;
					char		*lastPlusOne	= cur + bufferSize;
					
					// Copy the GWorld row to the buffer.
					while( cur < lastPlusOne )
					{
						curPixel++;		// Skip past the alpha channel
					
						*cur++ = *curPixel++;
						*cur++ = *curPixel++;
						*cur++ = *curPixel++;
					}
					
					sourceBaseAddr += gWorldRowBytes;

					err = PGPErrorToMacError( JPEGWriteScanlines( jpegContext,
									(JSAMPROW *) &buffer, 1, &count ) );
					if( IsErr( err ) )
						break;
				}
				
				JPEGFinishCompress( jpegContext );
			}
		}
		
		pgpFreeMac( buffer );
	}
	
	return( err );
}

	OSStatus
CPhotoUserID::GetPhotoData(
	PGPContextRef	context,
	void 			**data,
	PGPSize			*dataSize)
{
	OSStatus		err = noErr;
	FSSpec			fileSpec;
	PGPFileSpecRef	fileSpecRef = kInvalidPGPFileSpecRef;
	
	pgpAssert( HavePicture() );
	
	*data 		= NULL;
	*dataSize	= 0;
	
	// Compress to a temporary file
	
	err = CreateTempJPEGFile( &fileSpec );
	if( IsntErr( err ) )
	{
		err = PGPErrorToMacError( PGPNewFileSpecFromFSSpec( context,
										&fileSpec, &fileSpecRef ) );
	}

	if( IsntErr( err ) )
	{
		JPEGCompressionContext	jpegContext;

		JPEGInitCompressionContext( &jpegContext );

		err = JPEGCreateCompress( &jpegContext );
		if( IsntErr( err ) )
		{
			err = JPEGOpenCompressionFile( &jpegContext, fileSpecRef );
			if( IsntErr( err ) )
			{
				err = GWorldToJPEG( &jpegContext );
					
				JPEGCloseCompressionFile( &jpegContext );
			}
			
			JPEGDestroyCompress( &jpegContext );
		}
		
		if( IsntErr( err ) )
		{
			short	fileRef;
		
			err = FSpOpenDF( &fileSpec, fsRdPerm, &fileRef );
			if( IsntErr( err ) )
			{
				long	fileSize;
				
				err = GetEOF( fileRef, &fileSize );
				if( IsntErr( err ) )
				{
					PGPPhotoUserIDHeader	*header;
					
					*dataSize = fileSize + sizeof( *header );
					
					*data = PGPNewData( PGPGetContextMemoryMgr( context ),
											*dataSize, 0 );
					if( IsntNull( *data ) )
					{
						PGPByte		*dataStart;
						PGPUInt16	headerLength;
						
						header 			= (PGPPhotoUserIDHeader *) *data;
						headerLength 	= sizeof( *header );
						
						pgpClearMemory( header, sizeof( *header ) );
						
						header->headerVersion 	= kPGPPhotoIDHeaderVersion;
						header->photoIDFormat	= kPGPPhotoIDFormat_JPEG;
						
						PGPUInt16ToStorage( headerLength,
								(PGPByte *) &header->headerLength );
								
						dataStart = (PGPByte *) *data + headerLength;

						err = FSRead( fileRef, &fileSize, dataStart );
						if( IsErr( err ) )
						{
							PGPFreeData( *data );
							
							*data 		= NULL;
							*dataSize 	= 0;
						}
					}
					else
					{
						*dataSize = 0;
						
						err = memFullErr;
					}
				}
				
				FSClose( fileRef );
			}
		}	
		
		FSpDelete( &fileSpec );
	}
	
	if( PGPFileSpecRefIsValid( fileSpecRef ) )
		PGPFreeFileSpec( fileSpecRef );

	return( err );
}

	void
CPhotoUserID::SetStatus(PGPPhotoUserIDStatus status)
{
	if( status != mStatus )
	{
		mStatus = status;
		Refresh();
	}
}