program PGPSort;

{                                                                           }
{ PGPSORT v1.03 by Stle Schumacher/Felix Softworks 1994                    }
{                                                                           }
{ Syntax  : PGPSORT [+UserID] [<keyring>] - tries to sort on last name      }
{           PGPSORT +K[eyID]  [<keyring>] - sorts on 8-digit key ID (v2.6)  }
{           PGPSORT +K[eyID]6 [<keyring>] - sorts on 6-digit key ID (v2.3)  }
{           PGPSORT +S[ize]   [<keyring>] - sorts on key size               }
{           PGPSORT +D[ate]   [<keyring>] - sorts on date of creation       }
{                                                                           }
{           Use '-' instead of '+' to sort in descending order              }
{                                                                           }
{ Synopsis: Sorts PGP key rings.                                            }
{                                                                           }
{ History : v1.03 - Now sorts secret keyrings, ascending/descending order   }
{           v1.02 - Now sorts key IDs according to last 8 digits (was 6)    }
{           v1.01 - Improved sorting of 'unstandard' user IDs               }
{           v1.00 - Original version                                        }
{                                                                           }
{ Examples: PGPSORT                - Sorts your main public key ring        }
{                                    (PUBRING.PGP) according to the last    }
{                                    names in the user IDs on the keys      }
{           PGPSORT -d BIGRING.PGP - Sorts the key ring BIGRING.PGP         }
{                                    according to the date of creation of   }
{                                    the keys, newest keys first            }
{                                                                           }
{ The files PGPSORT.PAS and PGPSORT.EXE are placed in the public domain and }
{ may be freely distributed and modified. Any questions should be addressed }
{ to the author at:                                                         }
{                                                                           }
{                 Internet       : staalesc@ifi.uio.no                      }
{                                                                           }
{                 SoundServer BBS: +47 22 57 16 00                          }
{                                  Stle Schumacher                         }
{                                                                           }
{                 Snail mail     : Stle Schumacher                         }
{                                  Gyldenlovesgate 24                       }
{                                  N-0260 Oslo                              }
{                                  NORWAY                                   }
{                                                                           }

{$A+,B-}
{$M 65520,0,655360}

uses
  Crt,Dos;

const
  Version      = '1.03';
  RevisionDate = '1994/08/07';

  MaxKeys      = 10000;

type
  KeyPtr = ^KeyRec;
  KeyRec = record
             fPos,
             length    : longint;
             keyID     : longint;
             userID    : string[25];
             size      : integer;
             date      : longint;
           end;

var
  SortCriterion: (KeyID8,KeyID6,UserID,Size,Date);
  SortOrder    : (Ascending,Descending);

  keys         : integer;
  key          : array[0..MaxKeys] of KeyPtr;

procedure Error(const msg: string);
  begin
    WriteLn(msg);
    Halt(1);
  end;

function FileExists(const fileName: PathStr): boolean;
  var
    DirInfo: SearchRec;
  begin
    FindFirst(fileName,Archive,DirInfo);
    FileExists:=(DosError=0) and (fileName<>'');
  end;

function NoDirInName(const fileName: PathStr): boolean;
  var
    i: Integer;
  begin
    NoDirInName:=True;
    for i:=1 to Length(fileName) do
      if fileName[i] in [':','\'] then
        NoDirInName:=False;
  end;

function DirWithSlash(const dir: DirStr): DirStr;
  begin
    if (dir<>'') and (Copy(dir,Length(dir),1)<>'\') then
      DirWithSlash:=dir+'\'
    else
      DirWithSlash:=dir;
  end;

function UpperCase(s: string): string; near; assembler;
  asm
    PUSH    DS
    LDS     SI,[BP+4]
    LES     DI,[BP+8]
    CLD
    LODSB
    STOSB
    XOR     CH,CH
    MOV     CL,AL
    JCXZ    @3
  @1:
    LODSB
    CMP     AL,'a'
    JB      @2
    CMP     AL,'z'
    JA      @2
    SUB     AL,'a'-'A'
  @2:
    STOSB
    LOOP    @1
  @3:
    POP     DS
  end;

procedure QuickSort;

  function Sorted(a,b: integer): boolean;
    begin
      if SortOrder=Ascending then
        case SortCriterion of
          KeyID6,
          KeyID8: if (key[a]^.keyID < 0) and (key[b]^.keyID > 0) then
                    Sorted := false
                  else if (key[a]^.keyID > 0) and (key[b]^.keyID < 0) then
                    Sorted := true
                  else
                    Sorted := key[a]^.keyID < key[b]^.keyID;
          UserID: Sorted := key[a]^.userID < key[b]^.userID;
          Size  : Sorted := key[a]^.size < key[b]^.size;
          Date  : Sorted := key[a]^.date < key[b]^.date;
        end
      else
        case SortCriterion of
          KeyID6,
          KeyID8: if (key[a]^.keyID > 0) and (key[b]^.keyID < 0) then
                    Sorted := false
                  else if (key[a]^.keyID < 0) and (key[b]^.keyID > 0) then
                    Sorted := true
                  else
                    Sorted := key[a]^.keyID > key[b]^.keyID;
          UserID: Sorted := key[a]^.userID > key[b]^.userID;
          Size  : Sorted := key[a]^.size > key[b]^.size;
          Date  : Sorted := key[a]^.date > key[b]^.date;
        end;
    end;

  procedure SwapKeys(a,b: integer);
    var
      x: KeyPtr;
    begin
      x:=key[a];
      key[a]:=key[b];
      key[b]:=x;
    end;

  procedure Sort(left,right: integer);
    var
      i,j: integer;
    begin
      i:=left; j:=right;
      key[0]^:=key[(left+right) div 2]^;
      repeat
        while Sorted(i,0) do inc(i);
        while Sorted(0,j) do dec(j);
        if i<=j then
          begin
            SwapKeys(i,j);
            inc(i); dec(j);
          end;
      until i>j;
      if left<j then
        Sort(left,j);
      if i<right then
        Sort(i,right);
    end;

  begin
    Sort(1,keys);
  end;

function ReadPacketLength(var f: file; CTB: byte): longint;
  const
    LengthOfLengthField: array[0..3] of byte = (1,2,4,0);
  var
    i,l,b : byte;
    Length: longint;
  begin
    l:=LengthOfLengthField[CTB and 3];
    Length:=0;
    for i:=1 to l do
       begin
         BlockRead(f,b,1);
         Length:=(Length SHL 8)+b;
       end;
    ReadPacketLength:=Length;
  end;

function SwapLong(l: longint): longint; near; assembler;
  asm
    MOV DH,[BP+4];
    MOV DL,[BP+5];
    MOV AH,[BP+6];
    MOV AL,[BP+7];
  end;

function ReadLongint(var f: file): longint;
  var
    l : longint;
  begin
    BlockRead(f,l,4);
    ReadLongint:=SwapLong(l);
  end;

procedure SortKeyRing(const keyRing: PathStr);
  var
    f,newF       : file;
    b,CTB        : byte;
    fileLength,
    fPos,keyPos,
    PacketLength : longint;
    UserID       : string;
    firstUserID  : boolean;
    dir          : DirStr;
    name         : NameStr;
    ext          : ExtStr;
    buf          : array[1..2048] of byte;
    bakName      : string;
    i,
    bytes        : integer;

  procedure ShowProgress(fPos,fLen: longint);
    begin
      GotoXY(10,WhereY);
      Write((fPos * 100) div fLen:3,'%');
    end;

  begin
    keys:=0;
    fPos:=0;
    GetMem(key[0],SizeOf(KeyRec));
    Assign(f,KeyRing); {$I-} Reset(f,1); {$I+}
    If IOResult<>0 then Error(keyring+': access denied.');
    FileLength:=FileSize(f);
    Write('Reading: ');
    while fPos<FileLength do
      begin
        if fPos mod 20=0 then
          ShowProgress(fPos,FileLength);
        Seek(f,fPos); keyPos:=fPos;
        BlockRead(f,CTB,1);
        PacketLength:=ReadPacketLength(f,CTB); fPos:=FilePos(f);
        CTB:=CTB and 60;

        case CTB of
          20,24:                  {Public/Secret key packet}
            begin
              inc(keys);
              if keys>MaxKeys then
                Error('The keyring '+keyRing+' is too long to sort.');
              GetMem(key[keys],SizeOf(KeyRec));
              if key[keys]=nil then
                Error('The keyring '+keyRing+' is too long to sort.');
              key[keys]^.fPos:=keyPos;
              key[keys-1]^.length:=keyPos-key[keys-1]^.fPos;
              firstUserID:=true;
              Seek(f,FilePos(f)+1);
              key[keys]^.date:=ReadLongint(f);
              Seek(f,FilePos(f)+3);
              BlockRead(f,key[keys]^.size,2); key[keys]^.size:=Swap(key[keys]^.size);
              Seek(f,FilePos(f)+((key[keys]^.size+7) div 8)-4);
              key[keys]^.keyID:=ReadLongint(f);
              if SortCriterion=KeyID6 then
                key[keys]^.keyID:=key[keys]^.keyID and $FFFFFF;
              key[keys]^.UserID:='';
            end;
          52:                     {User ID packet}
            begin
              UserID[0]:=char(PacketLength);
              BlockRead(f,UserID[1],PacketLength);
              if firstUserID then
                begin
                  firstUserID:=false;
                  UserID:=UpperCase(UserID);
                  b:=2;
                  while (b<=Length(UserID)) and
                  not (((UserID[b] in ['0'..'9']) and (UserID[b-1]<>'-'))
                       or (UserID[b] in ['[','(','{','<','>','+','#','*',':','/','|'])
                       or (Copy(UserID,b,2)=' -')
                       or (Copy(UserID,b,3)=' II')
                       or (Copy(UserID,b,4)=' JR.')
                       or (Copy(UserID,b,5)=' M.D.')) do
                    inc(b);
                  UserID[0]:=CHAR(b-1);
                  if UserID[1] in ['<','*'] then UserID:=Copy(UserID,2,255);
                  while UserID[Length(UserID)] in [' ',',','.'] do dec(UserID[0]);

                  {Derive name from internet address?}
                  if (Pos(' ',UserID)=0) and (Pos('@',UserID)>0) then
                    begin
                      UserID[0]:=char(Pos('@',UserID)-1);
                      b:=Pos('.',UserID);
                      if b>0 then UserID[b]:=' ';
                    end
                  else
                    for b:=1 to Length(UserID)-1 do
                      if (UserID[b]='.') and (UserID[b+1]<>' ') then
                        UserID:=Copy(UserID,1,b)+' '+Copy(UserID,b+1,255);

                  {Split first and last names}
                  if Pos(' ',UserID)=0 then
                    key[keys]^.userID:=UserID
                  else
                    begin
                      b:=Pos(', ',UserID);
                      if (b>0) and (b+1=Pos(' ',UserID)) then
                        key[keys]^.userID:=UserID
                      else
                        begin
                          b:=Length(UserID);
                          while (UserID[b]<>' ') do dec(b);
                          key[keys]^.userID:=Copy(UserID,b+1,Length(UserID))+', '+Copy(UserID,1,b-1);
                        end;
                    end;
                  if (Copy(UserID,1,2)='!!') or (Copy(UserID,1,2)='**') then
                    firstUserID:=true;
                end;
            end;
          48:                     {Keyring trust packet}
            {Nothing to do};
          8:                      {Signature packet}
            {Nothing to do};
          else                    {Unknown packet}
            begin
              WriteLn(CTB);
              Error(keyRing+' is not a key ring.');
            end;
        end;
        fPos:=fPos+PacketLength;
      end;
    key[keys]^.length:=FileSize(f)-key[keys]^.fPos;
    Close(f);
    if keys=0 then
      Error(keyRing+' is not a key ring.');

    {Sort keys}
    GotoXY(1,WhereY); Write('Sorting:   0%');
    QuickSort;

    {Backup old keyring}
    FSplit(KeyRing,Dir,Name,Ext);
    bakName:=Dir+Name+'.BAK';
    Assign(f,bakName); {$I-} Erase(f); {$I+}
    if IOResult<>0 then {Old backup not found};
    Assign(f,KeyRing); Rename(f,bakName);

    {Write new keyring}
    GotoXY(1,WhereY); Write('Writing:');
    Assign(f,bakName); Reset(f,1);
    Assign(newF,KeyRing); Rewrite(newF,1);
    for i:=1 to keys do
      begin
        if i mod 20=0 then
          ShowProgress(i,keys);
        Seek(f,key[i]^.fPos);
        while key[i]^.length>0 do
          begin
            bytes:=key[i]^.length;
            if bytes>SizeOf(buf) then bytes:=SizeOf(buf);
            BlockRead(f,buf,bytes);
            BlockWrite(newF,buf,bytes);
            dec(key[i]^.length,bytes);
          end;
      end;
    Close(f); Close(newF);
    GotoXY(1,WhereY);

    for i:=0 to keys do
      FreeMem(key[i],SizeOf(KeyRec));
  end;

procedure WriteSyntax;
  begin
    WriteLn('Syntax: PGPSORT [+UserID] [<keyring>] - tries to sort on last name');
    WriteLn('        PGPSORT +K[eyID]  [<keyring>] - sorts on 8-digit key ID (PGP 2.6)');
    WriteLn('        PGPSORT +K[eyID]6 [<keyring>] - sorts on 6-digit key ID (PGP 2.3)');
    WriteLn('        PGPSORT +S[ize]   [<keyring>] - sorts on key size');
    WriteLn('        PGPSORT +D[ate]   [<keyring>] - sorts on date of creation');
    WriteLn;
    WriteLn('        Use ''-'' instead of ''+'' to sort in descending order');
    Halt(1);
  end;

var
  mode,
  KeyRing: string;

begin
  WriteLn;
  WriteLn('PGPSORT v',Version,' (C) 1994 Felix Softworks');
  WriteLn('Written by Stle Schumacher ',RevisionDate);
  WriteLn;

  KeyRing:='PUBRING.PGP';
  SortCriterion:=UserID;
  SortOrder:=Ascending;

  if ParamCount in [1,2] then
    begin
      mode:=ParamStr(1);
      if mode[1] in ['+','-'] then
        begin
          case mode[1] of
            '+': SortOrder:=Ascending;
            '-': SortOrder:=Descending;
          end;
          mode:=UpperCase(Copy(mode,2,255));
          if (mode='K') or (mode='KEYID') or (mode='K8') or (mode='KEYID8') then
            SortCriterion:=KeyID8
          else if (mode='K6') or (mode='KEYID6') then
            SortCriterion:=KeyID6
          else if (mode='U') or (mode='USERID') then
            SortCriterion:=UserID
          else if (mode='S') or (mode='SIZE') then
            SortCriterion:=Size
          else if (mode='D') or (mode='DATE') then
            SortCriterion:=Date
          else
            WriteSyntax
        end
      else if ParamCount=2 then
        WriteSyntax
      else
        KeyRing:=UpperCase(ParamStr(1));
      if ParamCount=2 then
        KeyRing:=UpperCase(ParamStr(2));
    end
  else if ParamCount<>0 then
    WriteSyntax;

  if not FileExists(KeyRing) then
    begin
      if NoDirInName(KeyRing) then
        KeyRing:=DirWithSlash(UpperCase(GetEnv('PGPPATH')))+KeyRing;
      if not FileExists(KeyRing) then
        Error(KeyRing+' not found.');
    end;

  SortKeyRing(KeyRing);

  Write(KeyRing,' sorted on ');
  case SortCriterion of
    KeyID6,
    KeyID8: WriteLn('key ID.');
    UserID: WriteLn('user ID.');
    Size  : WriteLn('size.');
    Date  : WriteLn('date.');
  end;
end.
