/*
 * Copyright (C) 1995, 1996 Systemics Ltd (http://www.systemics.com/)
 * All rights reserved.
 *
 * This library and applications are FREE FOR COMMERCIAL AND NON-COMMERCIAL USE
 * as long as the conditions within the COPYRIGHT file are adhered to.
 *
 */

package cryptix.crypt;

/**
 * This class implements the DES block cipher.
 *
 * <p>Copyright (C) 1995, 1996 Systemics Ltd (http://www.systemics.com/)
 * All rights reserved.
 */
public final class DES extends BlockCipher
{
	private static final int REQUIRED_LIB_MAJOR_VER = 1;
	private static final int REQUIRED_LIB_MINOR_VER = 11;
	
	private static final String LIBRARY_NAME = "des";

	private static boolean native_link_ok = false;
	private static boolean native_lib_loaded = false;
	private static String native_link_err = "Class never loaded";

	static 
	{
		// load the DLL or shared library that contains the native code
		// implementation of the des block cipher.
		try
		{
			System.loadLibrary( LIBRARY_NAME );
			native_lib_loaded = true;
			try
			{
				if ( ( getLibMajorVersion() != REQUIRED_LIB_MAJOR_VER ) || 
					( getLibMinorVersion() < REQUIRED_LIB_MINOR_VER ) )
				{
					native_link_err = "The " + LIBRARY_NAME + " native library " + 
										getLibMajorVersion() + "." + getLibMinorVersion() +
										" is too old to use. Version " + 
										REQUIRED_LIB_MAJOR_VER + "." + REQUIRED_LIB_MINOR_VER + 
										" required.";
				}
				else
				{
					native_link_ok = true;
					native_link_err = null;
				}
			}
			catch ( UnsatisfiedLinkError ule )
			{
				native_link_err = "The " + LIBRARY_NAME + " native library is too old to use. Version " + 
									REQUIRED_LIB_MAJOR_VER + "." + REQUIRED_LIB_MINOR_VER + 
									" required.";
			}
		}
		catch ( UnsatisfiedLinkError ule )
		{
			native_link_err = "The " + LIBRARY_NAME + " native library was not found.";
		}
	}

	public final static boolean
	hasFileLibraryLoaded()
	{
		return native_lib_loaded;
	}

	public final static boolean
	isLibraryCorrect()
	{
		return native_link_ok;
	}

	public final static String
	getLinkErrorString()
	{
		return native_link_err;
	}

    /**
     * This is the length of a block.
     */
    public static final int BLOCK_LENGTH = 8;

    /**
     * This is the length of a the user key.
     */
    public static final int KEY_LENGTH = 8;

    /**
     * This is the length of the internal buffer for the native code.
     */
    private static final int INTERNAL_KEY_LENGTH = 128;

    /**
     * This is the internal buffer with the expanded encrypt key.
     */
    private byte ks[] = null;
 
    /**
     * This creates a DES block cipher from a byte array of the correct length.
     * @param userKey    the user key.
	 * @throws UnsatisfiedLinkError if the library is not of the correct version
     */
    public DES( byte userKey[] ) 
    {
		if ( !native_link_ok )
			throw new UnsatisfiedLinkError( native_link_err );
        ks = new byte[INTERNAL_KEY_LENGTH];
        if ( userKey.length != KEY_LENGTH )
            throw new CryptoError( "DES: User key length wrong" );

        set_ks( userKey );
    }

    /**
     * Encrypt a block.
	 * The in and out buffers can be the same.
     * @param in The data to be encrypted.
     * @param in_offset   The start of data within the in buffer.
     * @param out The encrypted data.
     * @param out_offset  The start of data within the out buffer.
     */
    protected void 
    blockEncrypt( byte in[], int in_offset, byte out[], int out_offset )
    {
        if ( ks == null )
            throw new CryptoError( "DES: User key not set" );

        do_des( in, in_offset, out, out_offset, true );
    }

    /**
     * Decrypt a block.
	 * The in and out buffers can be the same.
     * @param in The data to be decrypted.
     * @param in_offset   The start of data within the in buffer.
     * @param out The decrypted data.
     * @param out_offset  The start of data within the out buffer.
     */
    protected void
    blockDecrypt( byte in[], int in_offset, byte out[], int out_offset )
    {
        if ( ks == null )
            throw new CryptoError( "DES: User key not set." );

        do_des( in, in_offset, out, out_offset, false );
    }

    /**
     * Return the block length of this cipher.
     * @returns the block length of this cipher.
     */
    public int 
    blockLength()
    {
        return BLOCK_LENGTH;
    }

    /**
     * Return the key length for this cipher.
     * @returns the key length of this cipher.
     */
    public int 
    keyLength()
    {
        return KEY_LENGTH;
    }

	// The native functions that get the library version.

 	private native static int getLibMajorVersion();
	private native static int getLibMinorVersion();

    // these are the native functions that implement the DES algorithm.

   /**
     * This function expands the user key to an internal form.
     */
    private native void set_ks( byte userKey[] );

    /**
     * Encrypt/decrypt the block (depending upon encrypt flag)
     */
    private native void do_des( byte in[], int in_offset, byte out[], int out_offset, boolean encrypt );
}
