/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: CWipeFreeSpaceWizardDialog.cp,v 1.20 1999/05/17 15:36:08 heller Exp $
____________________________________________________________________________*/

#include <HFSVolumes.h>

#include <LEditText.h>
#include <LPopupButton.h>
#include <LProgressBar.h>
#include <LPushButton.h>
#include <LStaticText.h>
#include <PP_Messages.h>
#include <UReanimator.h>
#include <URegistrar.h>

#include "MacErrors.h"
#include "MacFiles.h"
#include "MacStrings.h"
#include "pflPrefs.h"
#include "pflPrefTypes.h"
#include "pgpClientPrefs.h"
#include "pgpMacVolumeWiping.h"
#include "pgpMem.h"
#include "pgpMemoryMgr.h"
#include "PGPOpenPrefs.h"
#include "pgpClientErrors.h"
#include "pgpDiskWiper.h"
#include "pgpToolsUtils.h"

#include "CWipeFreeSpaceWizardDialog.h"
#include "CPGPtoolsWipeFreeSpaceTask.h"
#include "PGPtoolsResources.h"

const ResIDT		view_Introduction			= 6100;
const ResIDT		view_SelectVolume			= 6200;
const ResIDT		view_Wiping					= 6300;
const ResIDT		view_Conclusion				= 6400;
const ResIDT		view_UnmountHFSFailure		= 6500;
const ResIDT		view_UnmountHFSPlusFailure	= 6600;
const ResIDT		view_Error					= 6700;
const ResIDT		view_IntegrityFailure		= 6800;
const ResIDT		view_VersionFailure			= 6900;

const ResIDT		kVolumeMenuResID			= 129;

const MessageT		kNextViewMessage			= 'Next';
const MessageT		kPreviousViewMessage		= 'Prev';
const MessageT		kAllowNextViewSwitchMessage	= 'ANxt';
const MessageT		kAllowPrevViewSwitchMessage	= 'APrv';
const MessageT		kDenyNextViewSwitchMessage	= 'DNxt';
const MessageT		kDenyPrevViewSwitchMessage	= 'DPrv';
const MessageT		kDenyCancelMessage			= 'DCan';

const PaneIDT		kNextButtonPaneID			= 'bNxt';
const PaneIDT		kPreviousButtonPaneID		= 'bPrv';
const PaneIDT		kCancelButtonPaneID			= 'bCan';

WipeFreeSpaceView	CWipeFreeSpaceWizardDialog::mNewView =
							kWipeFreeSpaceView_Invalid;
PGPError			CWipeFreeSpaceWizardDialog::mError =
							kPGPError_NoErr;

#pragma mark --- CWipeFreeSpaceWizardView ---

class CWipeFreeSpaceWizardView : 	public LView,
									public LCommander,
									public LBroadcaster,
									public LListener
{
public:

	enum { class_ID = 'vWFS' };
	
					CWipeFreeSpaceWizardView(LStream *inStream);
	virtual			~CWipeFreeSpaceWizardView();

	virtual	void	ListenToMessage(MessageT inMessage, void *ioParam);
};

CWipeFreeSpaceWizardView::CWipeFreeSpaceWizardView(LStream *inStream)
	: LView(inStream)
{
}

CWipeFreeSpaceWizardView::~CWipeFreeSpaceWizardView()
{
}

	void
CWipeFreeSpaceWizardView::ListenToMessage(
	MessageT 	inMessage,
	void 		*ioParam)
{
	(void) inMessage;
	(void) ioParam;
}

#pragma mark --- CSelectVolumeParamsView ---

class CSelectVolumeParamsView : public CWipeFreeSpaceWizardView
{
public:

	enum
	{
		class_ID = 'vSVP',
		
		kVolumePopupMenuPaneID		= 'pVol',
		kWipePassesEditTextPaneID	= 'ePas'
	};
	
					CSelectVolumeParamsView(LStream *inStream);
	virtual			~CSelectVolumeParamsView();

	void			GetParams(WipeVolumeInfo *wipeVolumeInfo,
							PGPUInt32 *wipePasses);
	virtual	void	ListenToMessage(MessageT inMessage, void *ioParam);
	void			SetParams(short wipeVRefNum, PGPUInt32 wipePasses);

protected:

	virtual void	FinishCreateSelf(void);
	
private:

	LArray			*mVolumeInfoList;
	LPopupButton	*mVolumePopup;
	LEditText		*mWipePassesEdit;
	
	void			GetCurrentWipeVolumeInfo(WipeVolumeInfo *wipeVolumeInfo);
};

CSelectVolumeParamsView::CSelectVolumeParamsView(LStream *inStream)
	: CWipeFreeSpaceWizardView(inStream)
{
	mVolumeInfoList = NULL;
}

CSelectVolumeParamsView::~CSelectVolumeParamsView()
{
	delete mVolumeInfoList;
}

	void
CSelectVolumeParamsView::FinishCreateSelf(void)
{
	CWipeFreeSpaceWizardView::FinishCreateSelf();
	
	mVolumePopup = (LPopupButton *) FindPaneByID( kVolumePopupMenuPaneID );
	pgpAssertAddrValid( mVolumePopup, VoidAlign );
	
	mWipePassesEdit = (LEditText *) FindPaneByID( kWipePassesEditTextPaneID );
	pgpAssertAddrValid( mWipePassesEdit, VoidAlign );
	
	mVolumePopup->AddListener( this );
	mWipePassesEdit->AddListener( this );
}

	void
CSelectVolumeParamsView::ListenToMessage(
	MessageT 	inMessage,
	void 		*ioParam)
{
	switch( inMessage )
	{
		case kVolumePopupMenuPaneID:
		case kWipePassesEditTextPaneID:
		{
			WipeVolumeInfo	info;
			
			GetCurrentWipeVolumeInfo( &info );
			if( info.canWipe && mWipePassesEdit->GetValue() != 0 )
			{
				BroadcastMessage( kAllowNextViewSwitchMessage, NULL );
			}
			else
			{
				BroadcastMessage( kDenyNextViewSwitchMessage, NULL );
			}
			
			break;
		}
		
		default:
			CWipeFreeSpaceWizardView::ListenToMessage( inMessage, ioParam );
			break;
	}
}

	void
CSelectVolumeParamsView::GetCurrentWipeVolumeInfo(WipeVolumeInfo *wipeVolumeInfo)
{
	mVolumeInfoList->FetchItemAt( mVolumePopup->GetValue(), wipeVolumeInfo );
}

	void
CSelectVolumeParamsView::GetParams(
	WipeVolumeInfo	*wipeVolumeInfo,
	PGPUInt32		*wipePasses)
{
	*wipePasses = mWipePassesEdit->GetValue();
	
	GetCurrentWipeVolumeInfo( wipeVolumeInfo );
}

	void
CSelectVolumeParamsView::SetParams(
	short		wipeVRefNum,
	PGPUInt32	wipePasses)
{
	Str255			volumeName;
	short			defaultItem = 1;
	MenuHandle		volumeMenu;
	OSErr			err = noErr;
	
	mVolumeInfoList = new LArray( sizeof( WipeVolumeInfo ) );
	
	volumeMenu = GetMenu( kVolumeMenuResID );
	AssertHandleIsValid( volumeMenu, nil );
	
	if( volumeMenu != nil )
	{
		short			numMenuItems;
		short			menuItemIndex;
		XVolumeParam	xpb;
	
		// Remove existing menu items, if any.
		numMenuItems = CountMItems( volumeMenu );
		for( menuItemIndex = numMenuItems; menuItemIndex >= 1; menuItemIndex-- )
		{
			DeleteMenuItem( volumeMenu, menuItemIndex );
		}

		// Loop over all mounted volumes and add them to the menu.
		// Volumes which cannot be wiped are dimmed.
		
		menuItemIndex = 1;
		
		pgpClearMemory( &xpb, sizeof( xpb ) );
	
		xpb.ioNamePtr = volumeName;
		
		do
		{
			xpb.ioVolIndex = menuItemIndex;

			err = pgpPBXGetVolInfo( &xpb );
			if( IsntErr( err ) )
			{
				WipeVolumeInfo	info;
				const VCB		*vcb;
				
				CopyPString( volumeName, info.name );

				info.vRefNum		= xpb.ioVRefNum;
				info.driveNumber	= xpb.ioVDrvInfo;
				info.driverRefNum	= xpb.ioVDRefNum;
				
				vcb = GetVCBForVolume( info.vRefNum );
				pgpAssertAddrValid( vcb, VCB );
				
				info.signature = vcb->vcbSigWord;
				
				if( IsntErr( pgpCanWipeVolumeFreeSpace( info.vRefNum ) ) )
				{	
					info.canWipe = TRUE;
				}
				else
				{
					info.canWipe = FALSE;
				}
				
				mVolumeInfoList->AddItem( &info );
			
				AppendMenu( volumeMenu, "\px" );
				SetMenuItemText( volumeMenu, menuItemIndex, volumeName );
				
				if( info.vRefNum == wipeVRefNum )
					defaultItem = menuItemIndex;
				
				if( ! info.canWipe )
				{
					DisableItem( volumeMenu, menuItemIndex );
				}
				
				++menuItemIndex;
			}
		
		} while( IsntErr( err ) );
		
		mVolumePopup->SetMacMenuH( volumeMenu, TRUE );
		mVolumePopup->SetValue( defaultItem );
	}
	
	mWipePassesEdit->SetValue( wipePasses );
	SwitchTarget( mWipePassesEdit );
}

#pragma mark --- CWipingProgressView ---

class CWipingProgressView : public CWipeFreeSpaceWizardView
{
public:

	enum
	{
		class_ID = 'vWPV',
		
		kStartButtonPaneID	= 'bWip'
	};
	
					CWipingProgressView(LStream *inStream);
	virtual			~CWipingProgressView();

	virtual void	ListenToMessage(MessageT inMessage, void *ioParam);
	void			SetParams(PGPContextRef context,
							const WipeVolumeInfo *wipeVolumeInfo,
							PGPUInt32 numPasses);

protected:

	virtual void	FinishCreateSelf(void);

private:

	PGPContextRef	mContext;
	LPushButton		*mStartButton;
	WipeVolumeInfo	mWipeVolInfo;
	PGPUInt32		mNumPasses;
	
	void			DoWipe(void);
	OSStatus		WipeVolumeEventHandler(const VolumeWipeEvent *event);

	static OSStatus	WipeVolumeEventHandlerShell(const VolumeWipeEvent *event,
							PGPUserValue userValue);
};

CWipingProgressView::CWipingProgressView(LStream *inStream)
	: CWipeFreeSpaceWizardView(inStream)
{
}

CWipingProgressView::~CWipingProgressView()
{
}

	void
CWipingProgressView::FinishCreateSelf(void)
{
	CWipeFreeSpaceWizardView::FinishCreateSelf();
	
	mStartButton = (LPushButton *) FindPaneByID( kStartButtonPaneID );
	pgpAssertAddrValid( mStartButton, VoidAlign );
	
	mStartButton->AddListener( this );
}

	void
CWipingProgressView::ListenToMessage(
	MessageT 	inMessage,
	void 		*ioParam)
{
	switch( inMessage )
	{
		case kStartButtonPaneID:
		{
			DoWipe();
			break;
		}
		
		default:
			CWipeFreeSpaceWizardView::ListenToMessage( inMessage, ioParam );
			break;
	}
}

	void
CWipingProgressView::SetParams(
	PGPContextRef			context,
	const WipeVolumeInfo 	*wipeVolumeInfo,
	PGPUInt32				numPasses)
{
	mContext		= context;
	mWipeVolInfo 	= *wipeVolumeInfo;
	mNumPasses		= numPasses;
}

	void
CWipingProgressView::DoWipe(void)
{
	CToolsError					err;
	CPGPtoolsWipeFreeSpaceTask	*task;
	
	BroadcastMessage( kDenyPrevViewSwitchMessage, NULL );
	BroadcastMessage( kDenyCancelMessage, NULL );

	mSuperView->Draw( NULL );

	task = new CPGPtoolsWipeFreeSpaceTask;
	if( IsntNull( task ) )
	{
		err = task->SetParams( mContext, mWipeVolInfo.vRefNum, mNumPasses );
		if( err.IsntError() )
		{
			err = task->WipeFreeSpace();
		}
		
		delete task;		
	}
	else
	{
		err.pgpErr = kPGPError_OutOfMemory;
	}
	
	if( err.IsntError() )
	{
		CWipeFreeSpaceWizardDialog::SetNewView( kWipeFreeSpaceView_Conclusion );
	}
	else if( err.IsCancelError() )
	{
		CWipeFreeSpaceWizardDialog::SetNewView( kWipeFreeSpaceView_Exit );
	}
	else if( err.err == kWipingError_InvalidVolumeDataStructure )
	{
		CWipeFreeSpaceWizardDialog::SetNewView(
					kWipeFreeSpaceView_IntegrityFailure );
	}
	else if( err.err == kWipingError_UnsupportedVolumeFormat )
	{
		CWipeFreeSpaceWizardDialog::SetNewView(
					kWipeFreeSpaceView_VersionFailure );
	}
	else 
	{
		CWipeFreeSpaceWizardDialog::SetError( err.ConvertToPGPError() );
		CWipeFreeSpaceWizardDialog::SetNewView( kWipeFreeSpaceView_Error );
	}
}

#pragma mark --- CWipeFreeSpaceWizardDialog ---

CWipeFreeSpaceWizardDialog::CWipeFreeSpaceWizardDialog(LStream *inStream)
	: LGADialog(inStream)
{
	mCurrentView 	= kWipeFreeSpaceView_Invalid;
	mCurrentViewObj	= NULL;
	
	pgpClearMemory( &mWipeVolInfo, sizeof( mWipeVolInfo ) );
}

CWipeFreeSpaceWizardDialog::~CWipeFreeSpaceWizardDialog()
{
}

	void
CWipeFreeSpaceWizardDialog::SetParams(PGPContextRef context)
{
	PGPPrefRef	prefRef;
	PGPError	err;
	
	mContext				= context;
	mWipeVolInfo.vRefNum 	= -1;
	mWipingPasses			= 1;
	
	err = PGPOpenClientPrefs( PGPGetContextMemoryMgr( context ), &prefRef );
	if( IsntPGPError( err ) )
	{
		err = PGPGetPrefNumber( prefRef, kPGPPrefDiskWipePasses,
					&mWipingPasses );
		pgpAssert( IsntErr( err ) );

		(void) PGPClosePrefFile( prefRef );
	}
}

	void
CWipeFreeSpaceWizardDialog::FinishCreateSelf()
{
	LGADialog::FinishCreateSelf();
	
	RegisterClass_( CWipeFreeSpaceWizardView );
	RegisterClass_( CSelectVolumeParamsView );
	RegisterClass_( CWipingProgressView );
	
	SetThemeWindowBackground( GetMacPort(),
				kThemeActiveDialogBackgroundBrush, FALSE );

	mNextButton = (LPushButton *) FindPaneByID( kNextButtonPaneID );
	pgpAssertAddrValid( mNextButton, VoidAlign );

	mPreviousButton = (LPushButton *) FindPaneByID( kPreviousButtonPaneID );
	pgpAssertAddrValid( mPreviousButton, VoidAlign );

	mCancelButton = (LPushButton *) FindPaneByID( kCancelButtonPaneID );
	pgpAssertAddrValid( mCancelButton, VoidAlign );

	// mNextButton already added by the modal dialog handler.
	mPreviousButton->AddListener( this );
	mCancelButton->AddListener( this );

	mNewView 	= kWipeFreeSpaceView_Invalid;
	mError		= kPGPError_NoErr;
	
	SwitchToView( kWipeFreeSpaceView_Introduction );
	StartIdling();
}

	void
CWipeFreeSpaceWizardDialog::ApplyForeAndBackColors(void) const
{
	SetThemeWindowBackground( GetMacPort(),
				kThemeActiveDialogBackgroundBrush, FALSE );
}

	void
CWipeFreeSpaceWizardDialog::ListenToMessage(
	MessageT 	inMessage,
	void 		*ioParam)
{
	(void) ioParam;
	
	switch( inMessage )
	{
		case kNextViewMessage:
			SetNewView( GetNextView( mCurrentView ) );
			break;
			
		case kPreviousViewMessage:
			SetNewView( GetPreviousView( mCurrentView ) );
			break;
			
		case msg_Cancel:
			DoClose();
			break;
		
		case kAllowNextViewSwitchMessage:
			mNextButton->Enable();
			break;
					
		case kAllowPrevViewSwitchMessage:
			mPreviousButton->Enable();
			break;
			
		case kDenyNextViewSwitchMessage:
			mNextButton->Disable();
			break;
			
		case kDenyPrevViewSwitchMessage:
			mPreviousButton->Disable();
			break;

		case kDenyCancelMessage:
			mCancelButton->Disable();
			break;
			
		default:
			LGADialog::ListenToMessage( inMessage, ioParam );
			break;
	}
}

	void
CWipeFreeSpaceWizardDialog::AdjustControlsForView(WipeFreeSpaceView theView)
{
	short	titleIndex = kNextButtonTitleStrIndex;
	Str255	buttonTitle;
	
	mNextButton->Enable();
	mPreviousButton->Enable();

	switch( theView )
	{
		case kWipeFreeSpaceView_Introduction:
		{
			mPreviousButton->Disable();
			
			if( FileSharingIsActive() )
				mNextButton->Disable();
				
			break;
		}
			
		case kWipeFreeSpaceView_Conclusion:
		case kWipeFreeSpaceView_Error:
		case kWipeFreeSpaceView_IntegrityFailure:
		case kWipeFreeSpaceView_VersionFailure:
			titleIndex = kDoneButtonTitleStrIndex;
			mPreviousButton->Disable();
			mCancelButton->Disable();
			break;
			
		case kWipeFreeSpaceView_Wiping:
			mNextButton->Disable();
			break;
	}
	
	GetIndString( buttonTitle, kPGPtoolsMiscStringsResID, titleIndex );
	mNextButton->SetDescriptor( buttonTitle );
}

	WipeFreeSpaceView
CWipeFreeSpaceWizardDialog::GetNextView(WipeFreeSpaceView theView)
{
	WipeFreeSpaceView	nextView;
	
	switch( theView )
	{
		case kWipeFreeSpaceView_Introduction:
			nextView = kWipeFreeSpaceView_SelectVolume;
			break;
		
		case kWipeFreeSpaceView_SelectVolume:
		case kWipeFreeSpaceView_UnmountHFSFailure:
		case kWipeFreeSpaceView_UnmountHFSPlusFailure:
			nextView = kWipeFreeSpaceView_Wiping;
			break;

		case kWipeFreeSpaceView_Wiping:
			nextView = kWipeFreeSpaceView_Conclusion;
			break;

		case kWipeFreeSpaceView_Conclusion:
		case kWipeFreeSpaceView_Error:
		case kWipeFreeSpaceView_IntegrityFailure:
		case kWipeFreeSpaceView_VersionFailure:
			nextView = kWipeFreeSpaceView_Exit;
			break;

		default:
			pgpDebugMsg( "Missing next view" );
			nextView = theView;
			break;
	}
	
	return( nextView );
}

	WipeFreeSpaceView
CWipeFreeSpaceWizardDialog::GetPreviousView(WipeFreeSpaceView theView)
{
	WipeFreeSpaceView	previousView;
	
	switch( theView )
	{
		case kWipeFreeSpaceView_SelectVolume:
			previousView = kWipeFreeSpaceView_Introduction;
			break;

		case kWipeFreeSpaceView_UnmountHFSFailure:
		case kWipeFreeSpaceView_UnmountHFSPlusFailure:
		case kWipeFreeSpaceView_Wiping:
			previousView = kWipeFreeSpaceView_SelectVolume;
			break;

		default:
			pgpDebugMsg( "Missing previous view" );
			previousView = theView;
			break;
	}
	
	return( previousView );
}

	WipeFreeSpaceView
CWipeFreeSpaceWizardDialog::LeaveCurrentView(WipeFreeSpaceView newView)
{
	switch( mCurrentView )
	{
		case kWipeFreeSpaceView_SelectVolume:
		{
			PGPUInt32				wipePasses;
			CSelectVolumeParamsView	*viewObj;
			
			viewObj = (CSelectVolumeParamsView *) mCurrentViewObj;
				
			viewObj->GetParams( &mWipeVolInfo, &wipePasses );
			
			if( wipePasses != mWipingPasses )
			{
				PGPError	err;
				PGPPrefRef	prefRef;

				mWipingPasses = wipePasses;
				
				err = PGPOpenClientPrefs( PGPGetContextMemoryMgr( mContext ),
							&prefRef );
				if( IsntPGPError( err ) )
				{
					err = PGPSetPrefNumber( prefRef, kPGPPrefDiskWipePasses,
								mWipingPasses );
					pgpAssert( IsntErr( err ) );

					(void) PGPSavePrefFile( prefRef );
					(void) PGPClosePrefFile( prefRef );
				}
			}
			
			/* Fall through */
		}
		
		case kWipeFreeSpaceView_UnmountHFSPlusFailure:
		{
			if( newView == kWipeFreeSpaceView_Wiping )
			{
				if( mWipeVolInfo.canWipe )
				{
					OSErr	err;
					
					/*
					** Try to unmount volume. If we cannot, go to unmount
					** warning page.
					*/
					
					err = UnmountVol( NULL, mWipeVolInfo.vRefNum );
					if( IsErr( err ) )
					{
						if( mWipeVolInfo.signature == kHFSSigWord )
						{
							newView = kWipeFreeSpaceView_UnmountHFSFailure;
						}
						else if( mWipeVolInfo.signature == kHFSPlusSigWord )
						{
							newView = kWipeFreeSpaceView_UnmountHFSPlusFailure;
						}
						else
						{
							pgpDebugMsg( "Unknown volume signature" );
							newView = mCurrentView;
						}
					}
					else
					{
						ParamBlockRec	pb;
						
						pgpClearMemory( &pb, sizeof( pb ) );
						
						pb.volumeParam.ioVRefNum = mWipeVolInfo.driveNumber;
						
						err = PBMountVol( &pb );
						if( IsntErr( err ) )
						{
							mWipeVolInfo.vRefNum = pb.volumeParam.ioVRefNum;
						}
						else
						{
							mError 	= MacErrorToPGPError( err );
							newView	= kWipeFreeSpaceView_Error;
						}
					}
				}
				else
				{
					newView = kWipeFreeSpaceView_SelectVolume;
				}
			}
			
			break;
		}
	}

	if( newView != kWipeFreeSpaceView_Invalid &&
		IsntNull( mCurrentViewObj ) )
	{
		Rect	viewRect;
		
		mCurrentViewObj->CalcPortFrameRect( viewRect );
		delete mCurrentViewObj;				// delete it
		mCurrentViewObj = NULL;
		
		FocusDraw();
		SetThemeWindowBackground( GetMacPort(),
				kThemeActiveDialogBackgroundBrush, FALSE );
		EraseRect( &viewRect );
	}
	
	return( newView );
}

	void
CWipeFreeSpaceWizardDialog::EnterView(WipeFreeSpaceView newView)
{
	ResIDT	newViewResID;
	
	switch( newView )
	{
		case kWipeFreeSpaceView_Error:
			newViewResID = view_Error;
			break;
		
		case kWipeFreeSpaceView_IntegrityFailure:
			newViewResID = view_IntegrityFailure;
			break;
		
		case kWipeFreeSpaceView_VersionFailure:
			newViewResID = view_VersionFailure;
			break;
			
		case kWipeFreeSpaceView_Introduction:
			newViewResID = view_Introduction;
			break;

		case kWipeFreeSpaceView_SelectVolume:
			newViewResID = view_SelectVolume;
			break;

		case kWipeFreeSpaceView_Wiping:
			newViewResID = view_Wiping;
			break;

		case kWipeFreeSpaceView_UnmountHFSFailure:
			newViewResID = view_UnmountHFSFailure;
			break;

		case kWipeFreeSpaceView_UnmountHFSPlusFailure:
			newViewResID = view_UnmountHFSPlusFailure;
			break;
			
		case kWipeFreeSpaceView_Conclusion:
			newViewResID = view_Conclusion;
			break;
			
		default:
			newViewResID = 0;
			break;
	}

	if( newViewResID != 0 )
	{
		AdjustControlsForView( newView );

		LCommander::SetDefaultCommander( this );
		LPane::SetDefaultView( this );
		
		mCurrentViewObj = (CWipeFreeSpaceWizardView *)
							UReanimator::ReadObjects( 'PPob', newViewResID );
		mCurrentViewObj->FinishCreate();
		mCurrentViewObj->SetSuperCommander( this );
		mCurrentViewObj->AddListener( this );
		
		switch( newView )
		{
			case kWipeFreeSpaceView_Introduction:
			{
				if( FileSharingIsActive() )
				{
					LPane	*fileSharingWarning;
			
					fileSharingWarning = mCurrentViewObj->FindPaneByID( 1 );
					if( IsntNull( fileSharingWarning ) )
						fileSharingWarning->Show();
				}
				
				break;
			}
			
			case kWipeFreeSpaceView_SelectVolume:
			{
				CSelectVolumeParamsView	*viewObj =
						(CSelectVolumeParamsView *) mCurrentViewObj;
				
				viewObj->SetParams( mWipeVolInfo.vRefNum, mWipingPasses );
				break;
			}
			
			case kWipeFreeSpaceView_UnmountHFSFailure:
			{
				LStaticText	*textObj;
				Str255		text;

				SysBeep( 1 );
				
				textObj = (LStaticText *) mCurrentViewObj->FindPaneByID( 1 );
				pgpAssert( IsntNull( textObj ) );
				
				textObj->GetDescriptor( text );
				PrintPString( text, text, mWipeVolInfo.name );
				textObj->SetDescriptor( text );
				
				textObj = (LStaticText *) mCurrentViewObj->FindPaneByID( 2 );
				pgpAssert( IsntNull( textObj ) );
				
				textObj->GetDescriptor( text );
				PrintPString( text, text, mWipeVolInfo.name );
				textObj->SetDescriptor( text );
				
				break;
			}

			case kWipeFreeSpaceView_UnmountHFSPlusFailure:
			{
				LStaticText	*textObj;
				LStaticText	*hiddenTextObj;
				Str255		text;

				SysBeep( 1 );
				
				textObj = (LStaticText *) mCurrentViewObj->FindPaneByID( 1 );
				pgpAssert( IsntNull( textObj ) );
				
				textObj->GetDescriptor( text );
				PrintPString( text, text, mWipeVolInfo.name );
				textObj->SetDescriptor( text );
				
				if( mWipeVolInfo.vRefNum == -1 )
				{
					textObj 		= (LStaticText *) mCurrentViewObj->FindPaneByID( 3 );
					hiddenTextObj	= (LStaticText *) mCurrentViewObj->FindPaneByID( 2 );
				}
				else
				{
					textObj 		= (LStaticText *) mCurrentViewObj->FindPaneByID( 2 );
					hiddenTextObj	= (LStaticText *) mCurrentViewObj->FindPaneByID( 3 );
				}
				
				pgpAssert( IsntNull( textObj ) );
				pgpAssert( IsntNull( hiddenTextObj ) );
				
				textObj->Show();
				hiddenTextObj->Hide();
				
				textObj->GetDescriptor( text );
				PrintPString( text, text, mWipeVolInfo.name );
				textObj->SetDescriptor( text );
				
				break;
			}
			
			case kWipeFreeSpaceView_Error:
			{
				LStaticText	*textObj;
				Str255		text;
				Str255		errorStr;
				
				SysBeep( 1 );
				
				PGPGetClientErrorString( mError, sizeof( errorStr ),
							(char *) errorStr );
				CToPString( (char *) errorStr, errorStr );
				
				textObj = (LStaticText *) mCurrentViewObj->FindPaneByID( 1 );
				pgpAssert( IsntNull( textObj ) );
				
				textObj->GetDescriptor( text );
				PrintPString( text, text, mWipeVolInfo.name );
				PrintPString( text, text, errorStr );
				
				textObj->SetDescriptor( text );
				break;
			}

			case kWipeFreeSpaceView_IntegrityFailure:
			case kWipeFreeSpaceView_VersionFailure:
				SysBeep( 1 );
				SysBeep( 1 );
				SysBeep( 1 );
				
				/* Fall through */
				
			case kWipeFreeSpaceView_Conclusion:
			{
				LStaticText	*textObj;
				Str255		text;
				
				
				textObj = (LStaticText *) mCurrentViewObj->FindPaneByID( 1 );
				pgpAssert( IsntNull( textObj ) );
				
				textObj->GetDescriptor( text );
				PrintPString( text, text, mWipeVolInfo.name );
				textObj->SetDescriptor( text );
				
				break;
			}
			
			case kWipeFreeSpaceView_Wiping:
			{
				CWipingProgressView	*viewObj =
						(CWipingProgressView *) mCurrentViewObj;
				
				viewObj->SetParams( mContext, &mWipeVolInfo, mWipingPasses );
				break;
			}
		}
		
		mCurrentViewObj->Show();
		mCurrentViewObj->Draw( NULL );
		mCurrentViewObj->DontRefresh();
	}
	
	mCurrentView = newView;
}


	void
CWipeFreeSpaceWizardDialog::SwitchToView(WipeFreeSpaceView newView)
{
	newView = LeaveCurrentView( newView );
	if( newView != kWipeFreeSpaceView_Invalid )
	{ 
		EnterView( newView );
	}
}

	void
CWipeFreeSpaceWizardDialog::SetNewView(WipeFreeSpaceView newView)
{
	mNewView = newView;
}

	void
CWipeFreeSpaceWizardDialog::SetError(PGPError err)
{
	mError = err;
}

	void
CWipeFreeSpaceWizardDialog::FindCommandStatus(
	CommandT	inCommand,
	Boolean		&outEnabled,
	Boolean		&outUsesMark,
	Char16		&outMark,
	Str255		outName)
{
	(void) inCommand;
	(void) outUsesMark;
	(void) outMark;
	(void) outName;
	
	outEnabled = false;
}

	void
CWipeFreeSpaceWizardDialog::SpendTime(const EventRecord &inMacEvent)
{
	(void) inMacEvent;
	
	if( mNewView == kWipeFreeSpaceView_Exit )
	{
		DoClose();
	}
	else if( mNewView != kWipeFreeSpaceView_Invalid )
	{
		WipeFreeSpaceView	newView;
		
		newView 	= mNewView;
		mNewView	= kWipeFreeSpaceView_Invalid;
		
		SwitchToView( newView );
	}
	
	if( mCurrentView == kWipeFreeSpaceView_Introduction )
	{
		if( FileSharingIsActive() )
		{
			if( mNextButton->IsEnabled() )
				mNextButton->Disable();
		}
		else if( ! mNextButton->IsEnabled() )
		{
			mNextButton->Enable();
		}
	}
			
}