/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: CKeyTable.cp,v 1.192.6.1 1999/07/09 00:02:38 heller Exp $
____________________________________________________________________________*/

#include <Fonts.h>
#include <Drag.h>
#include <Processes.h>
#include <Scrap.h>

#include <UGAColorRamp.h>
#include <UDrawingState.h>
#include <UDrawingUtils.h>
#include <UGraphicUtils.h>
#include <UMemoryMgr.h>
#include <LEditText.h>
#include <LDropFlag.h>
#include <LProgressBar.h>
#include <LTableArrayStorage.h>
#include <LTableMultiGeometry.h>
#include <LTableMultiSelector.h>
#include <PP_Messages.h>
#include <PP_KeyCodes.h>
#include <UDesktop.h>

#include <string.h>

#include "CKeyTable.h"
#include "CKeyTableLabels.h"
#include "CKeyView.h"
#include "CSearchWindow.h"
#include "CSignDialog.h"
#include "CSplitWindow.h"
#include "CKeyInfoWindow.h"
#include "CCertProperties.h"
#include "CKeyDragTask.h"
#include "CPGPKeys.h"
#include "CPhotoUserID.h"
#include "CCAServer.h"
#include "GetPassphrase.h"
#include "CWarningAlert.h"
#include "CPGPStDialogHandler.h"
#include "ResourceConstants.h"
#include "MacStrings.h"
#include "MacDialogs.h"
#include "UDragAndDropUtils.h"
#include "CKeyGenWizardDialog.h"
#include "CSecureMemory.h"
#include "CGAProgressDialog.h"

#include "pgpClientLib.h"

// SDK Includes
#include "pgpErrors.h"
#include "pgpMem.h"
#include "pgpFeatures.h"
#include "pgpUtilities.h"
#include "pgpClientPrefs.h"
#include "pgpAdminPrefs.h"
#include "pgpVersionHeader.h"
#include "pgpSDKPrefs.h"

const Int16		kDefaultColumnWidth		=	16;
const Int16		kRowHeight				=	18;
const Int16		kBarHeight				=	10;
const Int16 	kBarRightBorder			=	10;
const Int16	 	kBarLeftBorder			=	4;
const Int16		kMaxTrust				=	3;
const Int16		kMaxValidity			=	2;
const Int16		kTextIndent				=	3;
const Int32		kMaxKeyBuffer			=	MAX_PGPInt32;
const ResIDT	kBarberIconID			=	4000;
const ResIDT	kExportCustomPutDialog	=	1004;
const Int16		kUserIDStringLength		=	256;

const Int16		kUserValueMarked		=	1;
const Int16		kUserValueUnmarked		=	0;

const ResIDT	dialog_AddUserID		= 138;
const ResIDT	dialog_KeyGenWizard		= 150;
const ResIDT	dialog_AddPhotoUserID	= 158;

const UInt32	kDefaultKeyBufferLength	= 32767;

static char		hexDigit[] = "0123456789ABCDEF";

NMRec				sImportNotification;
ProcessSerialNumber	sThisProcess;
Str255				sNotifyString;

KeyTableColumnInfo	CKeyTable::kDefaultColumnInfo	=
	{
#if PGP_BUSINESS_SECURITY
		4,
		kUserIDColumnID,
		kValidityColumnID,
		kKeySizeColumnID,
		kDescriptionColumnID,
		0,0,0,0,0,
#else
		5,
		kUserIDColumnID,
		kValidityColumnID,
		kTrustColumnID,
		kKeySizeColumnID,
		kDescriptionColumnID,
		0,0,0,0,
#endif
		
		286, 53, 53, 81, 71, 71, 71, 35, 100
	};

Int16 MatchUserIDs(char *uid1, char *uid2);

// The following small class fixes a PowerPlant bug in
// LTableArrayStorage.  The PowerPlant class does not
// properly support LHierarchyTable

class CHierarchyTableArrayStorage	:	public LTableArrayStorage
{
public:
						CHierarchyTableArrayStorage(
								LTableView			*inTableView,
								Uint32				inDataSize);
	virtual void		InsertCols(
								Uint32				inHowMany,
								TableIndexT			inAfterCol,
								const void			*inDataPtr,
								Uint32				inDataSize);
	virtual void		RemoveCols(
								Uint32				inHowMany,
								TableIndexT			inFromCol);
};

CHierarchyTableArrayStorage::CHierarchyTableArrayStorage(
	LTableView			*inTableView,
	Uint32				inDataSize )
		:	LTableArrayStorage(inTableView, inDataSize)
{
}

	void
CHierarchyTableArrayStorage::RemoveCols(
	Uint32		inHowMany,
	TableIndexT	inFromCol)
{
	TableIndexT			rows,
						cols;
	CPGPHierarchyTable	*hTable;
	
	hTable = (CPGPHierarchyTable *)mTableView;
	hTable->GetWideOpenTableSize(rows, cols);
	
	STableCell	theCell(0, inFromCol);
	
	for (theCell.row = 1; theCell.row <= rows; theCell.row++) {
		TableIndexT	startIndex;
		mTableView->CellToIndex(theCell, startIndex);
		mDataArray->RemoveItemsAt(inHowMany, startIndex);
	}
}

	void
CHierarchyTableArrayStorage::InsertCols(
	Uint32		inHowMany,
	TableIndexT	inAfterCol,
	const void	*inDataPtr,
	Uint32		inDataSize)
{
	TableIndexT			rows,
						cols;
	CPGPHierarchyTable	*hTable;
	
	hTable = (CPGPHierarchyTable *)mTableView;
	hTable->GetWideOpenTableSize(rows, cols);
	
	STableCell	theCell(0, inAfterCol + 1);
	
	for (theCell.row = 1; theCell.row <= rows; theCell.row++) {
		TableIndexT	startIndex;
		mTableView->CellToIndex(theCell, startIndex);
		mDataArray->InsertItemsAt(inHowMany, startIndex, inDataPtr,
								inDataSize);
	}
}

#pragma mark --- Initialization ---

CKeyTable::CKeyTable(
	LStream	*inStream)
		:	CPGPHierarchyTable(inStream),
			LDragAndDrop(UQDGlobals::GetCurrentPort(), this)
{
	mTableGeometry	= new LTableMultiGeometry(this,	kDefaultColumnWidth,
													kRowHeight);
	mTableSelector	= new LTableMultiSelector(this);
	mTableStorage	= new CHierarchyTableArrayStorage(this, (ulong)0);
	mLastKeypress	= 0;
	mKeySet			= NULL;
	mKeyList		= NULL;
	mKeyLabels		= NULL;
	mDefaultKey		= NULL;
	mImports		= NULL;
	mDefaultRing	= FALSE;
	mMutableRing	= FALSE;
	mNotified		= FALSE;
	mSendingDrag	= FALSE;
	mTargetKeyServer= NULL;
	mDragHiliteWorld= NULL;
	mDragHiliteRgn	= NULL;
	mNotifyIconSuite= NULL;
	PGPRSASupported( &mHasRSA );
}

void
CKeyTable::FinishCreateSelf()
{
	PGPError			err;
	PGPSize				len;
	KeyTableColumnInfo	*prefColumnInfo;
	
	::GetCurrentProcess(&sThisProcess);
	
	mBarberPixPat = ::GetPixPat(kBarberIconID);
	pgpAssertAddrValid(mBarberPixPat, VoidAlign);
	
	// Load Prefs for column display
	len = sizeof(KeyTableColumnInfo);
	err = PGPGetPrefData(gPrefRef, kPGPPrefPGPkeysMacColumnData,
							&len, &prefColumnInfo);
	if(IsPGPError(err))
	{
		::BlockMoveData(&kDefaultColumnInfo, &mColumnInfo,
							sizeof(KeyTableColumnInfo));
	}
	else
	{
		::BlockMoveData(prefColumnInfo, &mColumnInfo,
							sizeof(KeyTableColumnInfo));
		PGPDisposePrefData(gPrefRef, prefColumnInfo);
	}
	SetupColumns();
	
	mKeyIter = NULL;
	
	SwitchTarget(this);
	
	BuildServerSubmenu();
}

CKeyTable::~CKeyTable()
{
	Reset1();
	DisposePixPat( mBarberPixPat );
	if( IsntNull( mNotifyIconSuite ) )
		DisposeIconSuite( mNotifyIconSuite, TRUE );
	SaveColumnInfo();
}

	void
CKeyTable::SetKeyLabels( CKeyTableLabels *keyLabels )
{
	mKeyLabels = keyLabels;
	mKeyLabels->AdjustLabelSizes();
}

	void
CKeyTable::SaveColumnInfo()
{
	PGPError err;
	
	err = PGPSetPrefData(gPrefRef, kPGPPrefPGPkeysMacColumnData,
							sizeof(KeyTableColumnInfo), &mColumnInfo);
	pgpAssertNoErr(err);
}

	void
CKeyTable::ListenToMessage(	MessageT inMessage,
								void */*ioParam*/)
{
	switch(inMessage)
	{
		case kKeyTableRedrawMessage:
			RedrawTable();
			break;
		case kKeyTableResyncMessage:
			ResyncTable(FALSE, TRUE);
			RedrawTable();
			break;
		case kRebuildServersMessage:
			BuildServerSubmenu();
			break;
	}
}

	void
CKeyTable::SetupColumns()
{
	Int16		columnIndex;

	RemoveAllCols(false);
	InsertCols(mColumnInfo.numActiveColumns, 1, NULL, 0, false);
	for(columnIndex = 0;columnIndex < mColumnInfo.numActiveColumns;
		columnIndex++)
	{
		SetColWidth(
			mColumnInfo.columnWidths[mColumnInfo.columns[columnIndex]],
			columnIndex + 1, columnIndex + 1);
	}
}	

	void
CKeyTable::GetKeyColumnInfo(KeyTableColumnInfo *columnInfo)
{
	::BlockMoveData(&mColumnInfo, columnInfo, sizeof(KeyTableColumnInfo));
}

	Boolean
CKeyTable::ColumnVisible(Int16	columnID)
{
	Int16		columnIndex;

	for(columnIndex = 0;columnIndex < mColumnInfo.numActiveColumns;
		columnIndex++)
	{
		if(mColumnInfo.columns[columnIndex] == columnID)
			return true;
	}
	return false;
}

	void
CKeyTable::MoveColumn(Int16 oldCol, Int16 newCol)
{
	Int16 colType;
	
	colType = mColumnInfo.columns[oldCol];
	if(newCol < oldCol)
		newCol++;
	if(oldCol < mColumnInfo.numActiveColumns - 1)
		::BlockMoveData(&mColumnInfo.columns[oldCol + 1],
						&mColumnInfo.columns[oldCol],
						sizeof(Int16) *
						(mColumnInfo.numActiveColumns - oldCol - 1));
	if(newCol < mColumnInfo.numActiveColumns - 1)
		::BlockMoveData(&mColumnInfo.columns[newCol],
					&mColumnInfo.columns[newCol + 1],
					sizeof(Int16) *
					(mColumnInfo.numActiveColumns - newCol - 1));
	mColumnInfo.columns[newCol] = colType;
	RemoveCols(1, oldCol + 1, false);
	InsertCols(1, newCol, NULL, 0, false);
	SetColWidth(mColumnInfo.columnWidths[colType], newCol + 1, newCol + 1);
	Refresh();
}

	void
CKeyTable::ResizeColumn( Int16 col, Int16 newSize )
{
	mColumnInfo.columnWidths[mColumnInfo.columns[col]] = newSize;
	SetColWidth( mColumnInfo.columnWidths[mColumnInfo.columns[col]],
		col + 1, col + 1 );
	Refresh();
}

	void
CKeyTable::Reset()
{
	PGPError err;
	
	if(PGPKeySetRefIsValid(mKeySet))
	{
		if(mKeyIter)
			PGPFreeKeyIter(mKeyIter);
		err = PGPNewKeyIter(mKeyList, &mKeyIter);
		pgpAssertNoErr(err);
		pgpAssertAddrValid(mKeyIter, VoidAlign);
		
		mDefaultKey = NULL;
	}
}

	void
CKeyTable::Reset1()
{
	PGPError err;
	
	CKeyInfoWindow::CloseAll();
	CCertPropertiesWindow::CloseAll();
	if(PGPKeyIterRefIsValid(mKeyIter))
	{
		err = PGPFreeKeyIter(mKeyIter);
		pgpAssertNoErr(err);
	}
	mKeyIter = kInvalidPGPKeyIterRef;
	if(PGPKeyListRefIsValid(mKeyList))
	{
		err = PGPFreeKeyList(mKeyList);
		pgpAssertNoErr(err);
	}
	mKeyList = kInvalidPGPKeyListRef;
	CPGPKeys::TheApp()->SetKeyList(mKeyList);
	if(PGPKeySetRefIsValid(mKeySet))
	{
		if(mDefaultRing)
			CPGPKeys::TheApp()->CommitDefaultKeyrings();
		else if(mMutableRing)
		{
			err = PGPCommitKeyRingChanges(mKeySet);
			if(IsPGPError(err))
				ReportPGPError(err);
		}
		PGPFreeKeySet(mKeySet);
	}
	mKeySet = kInvalidPGPKeySetRef;
}

	void
CKeyTable::Empty()
{
	ClearTable();
	Reset();
	Refresh();
}

	void
CKeyTable::CommitKeySet()
{
	PGPError	err;
	
	if(mDefaultRing)
		CPGPKeys::TheApp()->CommitDefaultKeyrings();
	else if(mMutableRing)
	{
		err = PGPCommitKeyRingChanges(mKeySet);
		if(IsPGPError(err))
			ReportPGPError(err);
	}
}

	void
CKeyTable::SetKeyDBInfo(	PGPKeySetRef	keySet,
							Boolean			writable,
							Boolean			defaultRings )
{
	PGPBoolean	generatedKey;
	PGPBoolean	pairFound,
				allowKG = TRUE;
	PGPError	err;

	mKeySet			= keySet;
	mMutableRing	= writable;
	mDefaultRing	= defaultRings;

	err = PGPOrderKeySet(mKeySet, kPGPUserIDOrdering, &mKeyList);
	pgpAssert( IsntPGPError( err ) && PGPKeyListRefIsValid( mKeyList ) );
	CPGPKeys::TheApp()->SetKeyList( mKeyList );

	ClearTable();
	Reset();
	
	pairFound = ResyncTable(TRUE, TRUE);
	
	if(defaultRings)
	{
#if PGP_BUSINESS_SECURITY
		PGPSize			defaultKeysSize;
		char			*defaultKeys;
		PGPKeySetRef	defaultKeysSet = kInvalidPGPKeySetRef;
		
		err = PGPGetPrefData(gAdminPrefRef, kPGPPrefDefaultKeys,
								&defaultKeysSize, &defaultKeys);
		if(IsntPGPError(err))
		{
			if(defaultKeysSize > 10)
			{
				err = PGPImportKeySet(gPGPContext, &defaultKeysSet,
						PGPOInputBuffer(gPGPContext, defaultKeys,
										defaultKeysSize),
						PGPOLastOption(gPGPContext));
				if(IsntPGPError(err) && PGPKeySetRefIsValid(defaultKeysSet))
				{
					err = PGPAddKeys(defaultKeysSet, mKeySet);
					pgpAssertNoErr(err);
					CommitKeySet();
					PGPFreeKeySet(defaultKeysSet);
					pairFound = ResyncTable(TRUE, TRUE);
				}
				err = PGPSetPrefString(gAdminPrefRef,
										kPGPPrefDefaultKeys, "");
			}
			PGPDisposePrefData(gPrefRef, defaultKeys);
		}

		err = PGPGetPrefBoolean(gAdminPrefRef, kPGPPrefAllowKeyGen,
								&allowKG);
		pgpAssertNoErr(err);
#endif
		
		err = PGPGetPrefBoolean(gPrefRef, kPGPPrefFirstKeyGenerated,
								&generatedKey);
		pgpAssertNoErr(err);
		if(!generatedKey && allowKG)
		{
			err = PGPSetPrefBoolean(gPrefRef, kPGPPrefFirstKeyGenerated,
									TRUE);
			pgpAssertNoErr(err);
			
			if(!pairFound)
			{
				CKeyGenWizardDialog *kgw;
				
				kgw = (CKeyGenWizardDialog *)
					LWindow::CreateWindow(dialog_KeyGenWizard,
											CPGPKeys::TheApp());
				kgw->SetKeyTable(this);
			}
		}
	}
}

#pragma mark --- Drawing ---

	void
CKeyTable::DrawSelf()
{
	StColorPenState	colorState;
	RgnHandle		localUpdateRgnH = GetLocalUpdateRgn();
	Rect			updateRect = (**localUpdateRgnH).rgnBBox;
	SInt16			depth;
	Rect			frame,
					sortFrame,
					noSortFrame,
					cellRect;
	STableCell		cell,
					topLeftCell,
					botRightCell;
	
	DisposeRgn( localUpdateRgnH );
	CalcLocalFrameRect(frame);
	FetchIntersectingCells( updateRect, topLeftCell, botRightCell );
	noSortFrame = updateRect;
	sortFrame.left = mImageLocation.h;
	sortFrame.right = sortFrame.left + GetColWidth( 1 );
	sortFrame.top = noSortFrame.top;
	sortFrame.bottom = noSortFrame.bottom;
	noSortFrame.left = sortFrame.right;
	
	colorState.Normalize();
	TextFont( kFontIDGeneva );
	TextSize( 9 );
	TextFace( 0 );
	TextMode( srcOr );
	{
		StDeviceLoop	devLoop(frame);		
		while(devLoop.NextDepth(depth))
		{
			SetThemeBackground( kThemeListViewBackgroundBrush, depth,
								( depth > 1 ) );
			EraseRect( &noSortFrame );
			if( sortFrame.right > 0 )
			{
				SetThemeBackground( kThemeListViewSortColumnBackgroundBrush,
									depth, ( depth > 1 ) );
				EraseRect( &sortFrame );
			}
			mLowVertUpdate = mHighVertUpdate = 0;

			for(cell.row = topLeftCell.row; cell.row <= botRightCell.row;
				cell.row++)
			{
				for(cell.col = topLeftCell.col; cell.col <= botRightCell.col;
					cell.col++)
				{
					GetLocalCellRect( cell, cellRect );
					DrawCell( cell, cellRect );
				}
			}
			if( mHighVertUpdate > 0 && mLowVertUpdate > 0 )
			{
				SetThemePen( kThemeListViewSeparatorBrush, depth,
							( depth > 1 ) );
				do
				{
					MoveTo( frame.left, mLowVertUpdate );
					LineTo( frame.right - 1, mLowVertUpdate );
					mLowVertUpdate += kRowHeight;
				} while( mLowVertUpdate <= mHighVertUpdate );
			}
		}
	}
	HiliteSelection(IsActive(), true);
}

	void
CKeyTable::DrawDropFlag(
	const STableCell	&inCell,
	TableIndexT			inWideOpenRow)
{
	if (mCollapsableTree->IsCollapsable(inWideOpenRow))
	{
		Rect				flagRect;
		
		CalcCellFlagRect( inCell, flagRect );
		if( UEnvironment::HasFeature( env_HasAppearance11 ) )
		{
			ThemeButtonDrawInfo		theDrawingInfo;

			theDrawingInfo.adornment = kThemeAdornmentDrawIndicatorOnly;
			
			if( IsExpanded( inWideOpenRow ) )
				theDrawingInfo.value = kThemeDisclosureDown;
			else
				theDrawingInfo.value = kThemeDisclosureRight;
			
			//if( IsActive() )
				theDrawingInfo.state = kThemeStateActive;
			//else
			//	theDrawingInfo.state = kThemeStateInactive;
			DrawThemeButton(	&flagRect,
							  	kThemeDisclosureButton,
							  	&theDrawingInfo,
							  	NULL, NULL, NULL, NULL );
		}
		else
			LDropFlag::Draw(flagRect, mCollapsableTree->IsExpanded(inWideOpenRow));
	}
}

	void
CKeyTable::DrawCell1(
	const STableCell	&inCell,
	const Rect			&inLocalRect,
	Boolean				hilite,
	Boolean				inactive )
{
	TableIndexT		woRow = mCollapsableTree->GetWideOpenIndex(inCell.row);
	Rect			frame;

	CalcLocalFrameRect( frame );
	{
		StDeviceLoop	devLoop(frame);		
		SInt16			depth;

		while(devLoop.NextDepth(depth))
		{
			StClipRgnState	clip;
			STableCell		woCell(woRow, 1);
			Uint32			dataSize = sizeof(KeyTableRef);
			KeyTableRef		keyRef;
			
			clip.ClipToIntersection( inLocalRect );
			GetCellData(woCell, &keyRef, dataSize);
			SetThemeTextColor( kThemeListViewTextColor, depth, ( depth > 1 ) );
			switch( mColumnInfo.columns[inCell.col-1] )
			{
				case kUserIDColumnID:
					if( DrawUserIDColumnCell(&keyRef, inLocalRect, depth,
										hilite, inactive ) )
						SetCellData(woCell, &keyRef, dataSize);
					DrawDropFlag(inCell, woRow);
					break;
				case kValidityColumnID:
					DrawValidityColumnCell(&keyRef, inLocalRect, depth);
					break;
				case kTrustColumnID:
					DrawTrustColumnCell(&keyRef, inLocalRect, depth);
					break;
				case kKeySizeColumnID:
					DrawSizeColumnCell(&keyRef, inLocalRect, depth);
					break;
				case kKeyIDColumnID:
					DrawKeyIDColumnCell(&keyRef, inLocalRect, depth);
					break;
				case kCreationColumnID:
					DrawCreationColumnCell(&keyRef, inLocalRect, depth);
					break;
				case kExpirationColumnID:
					DrawExpirationColumnCell(&keyRef, inLocalRect, depth);
					break;
				case kMRKColumnID:
					DrawMRKColumnCell(&keyRef, inLocalRect, depth);
					break;
				case kDescriptionColumnID:
					DrawDescriptionColumnCell(&keyRef, inLocalRect, depth);
					break;
			}
			if( !mLowVertUpdate ||
				( mLowVertUpdate > inLocalRect.bottom - 1 ) )
				mLowVertUpdate = inLocalRect.bottom - 1;
			if( !mHighVertUpdate ||
				( mHighVertUpdate < inLocalRect.bottom - 1 ) )
				mHighVertUpdate = inLocalRect.bottom - 1;
		}
	}
}

	void
CKeyTable::DrawCell(
	const STableCell	&inCell,
	const Rect			&inLocalRect)
{
	DrawCell1( inCell, inLocalRect, false, false );
}

	Boolean
CKeyTable::DrawUserIDColumnCell(
	KeyTableRef		*ktr,
	Rect			cellRect,
	Int16			depth,
	Boolean			hilite,
	Boolean			inactive )
{
	PGPError		err;
	char			cstr[kUserIDStringLength];
	Str255			pstr;
	PGPSize			len;
	Int16			width;
	Rect			hiliteRect;
	ResIDT			iconID;

	switch( ktr->type )
	{
		case kKey:
		{
			PGPBoolean	expired,
						revoked,
						disabled,
						secret;
			PGPInt32	algorithm;
			PGPKeyRef	key;
			

			key = ktr->u.key;
			PGPGetKeyBoolean( key, kPGPKeyPropIsExpired, &expired );
			PGPGetKeyBoolean( key, kPGPKeyPropIsRevoked, &revoked );
			PGPGetKeyBoolean( key, kPGPKeyPropIsDisabled, &disabled );
			PGPGetKeyBoolean( key, kPGPKeyPropIsSecret, &secret );
			PGPGetKeyNumber( key, kPGPKeyPropAlgID, &algorithm );
			if( revoked )
				iconID = (algorithm == kPGPPublicKeyAlgorithm_DSA) ?	
							kDSARevokedKeyID : kRSARevokedKeyID;
			else if( expired )
				iconID = (algorithm == kPGPPublicKeyAlgorithm_DSA) ?	
							kDSAExpiredKeyID : kRSAExpiredKeyID;
			else if( disabled )
			{
				if(algorithm == kPGPPublicKeyAlgorithm_DSA)
					iconID = kDSADisabledKeyID;
				else
					iconID = kRSADisabledKeyID;
			}
			else if( secret )
			{
				PGPBoolean	secShared;
				
				PGPGetKeyBoolean( key, kPGPKeyPropIsSecretShared,
								&secShared);
				if(algorithm == kPGPPublicKeyAlgorithm_DSA)
					iconID = secShared ?	kDSASplitKeyPairIconID :
											kDSAKeyPairIconID;
				else
					iconID = secShared ?	kRSASplitKeyPairIconID :
											kRSAKeyPairIconID;
			}
			else if(algorithm == kPGPPublicKeyAlgorithm_DSA)
				iconID = kDSAKeyIconID;
			else
				iconID = kRSAKeyIconID;
			DrawIcon(iconID, cellRect, 1, hilite ?	kTransformSelected :
													kTransformNone );
			
			len = kUserIDStringLength;
			err = PGPGetPrimaryUserIDNameBuffer( key, kUserIDStringLength,
												cstr, &len );
			if(err != kPGPError_BufferTooSmall)
				pgpAssertNoErr(err);
			CToPString(cstr, pstr);
			if( mDefaultKey == key )
				TextFace( bold );
			else if( disabled )
				TextFace( italic );
			if( !mHasRSA && ( algorithm == kPGPPublicKeyAlgorithm_RSA ) )
				TextFace( italic );
			TruncString(	mColumnInfo.columnWidths[kUserIDColumnID] -
							kDropFlagSlop - kLeftIndent -
							kIconWidth - kLeftBorder,
							pstr, truncMiddle);
			hiliteRect.left = cellRect.left + kDropFlagSlop +
								kIconWidth + kLeftBorder;
			break;
		}
		case kUserID:
		{
			PGPUserIDRef	user;
			PGPInt32		algorithm;
			PGPBoolean		isAttributeID;
			
			user = ktr->u.user;
			
			err = PGPGetUserIDBoolean( user, kPGPUserIDPropIsAttribute,
						&isAttributeID );
			pgpAssertNoErr( err );
			
			if( isAttributeID )
			{
				PGPInt32	attributeType;
				short		strIndex;
				
				err = PGPGetUserIDNumber( user,
						kPGPUserIDPropAttributeType, &attributeType );
				
				if( attributeType == kPGPAttribute_Image )
				{
					strIndex 	= kPhotographID;
					iconID		= kPhotoUserIDIconID;
				}
				else
				{
					strIndex 	= kUnknownID;
					iconID		= kUnknowUserIDIconID;
				}

				GetIndString( pstr, kStringListID, strIndex);
			}
			else
			{
				err = PGPGetKeyNumber(ktr->ownerKey, kPGPKeyPropAlgID,
										&algorithm);
				pgpAssertNoErr(err);
				
				if( algorithm == kPGPPublicKeyAlgorithm_DSA )
					iconID = kDHUserIDIconID;
				else
					iconID = kRSAUserIDIconID;

				err = PGPGetUserIDStringBuffer(user, kPGPUserIDPropName,
							kUserIDStringLength, cstr, &len);
				CToPString(cstr, pstr);
			}
			
			pgpAssertNoErr(err);

			DrawIcon(iconID, cellRect, 2, hilite ?	kTransformSelected :
													kTransformNone );
			TruncString(	mColumnInfo.columnWidths[kUserIDColumnID] -
							kDropFlagSlop - kLeftIndent -
							kLevelIndent - kIconWidth - kLeftBorder,
							pstr, truncMiddle);
			hiliteRect.left = cellRect.left + kDropFlagSlop +
							kLevelIndent + kIconWidth + kLeftBorder;
			break;
		}
		case kSignature:
		{
			Str255		str2;
			size_t		len;
			PGPKeyRef	certKey;
			PGPBoolean	revoked,
						expired,
						verified,
						tried,
						notCorrupt,
						exportable,
						x509;
			PGPInt32	trustLevel;
			PGPSigRef	sig;

			sig = ktr->u.sig;
			PGPGetSigBoolean(sig, kPGPSigPropIsRevoked, &revoked);
			PGPGetSigBoolean(sig, kPGPSigPropIsVerified, &verified);
			PGPGetSigBoolean(sig, kPGPSigPropIsTried, &tried);
			PGPGetSigBoolean(sig, kPGPSigPropIsNotCorrupt, &notCorrupt);
			PGPGetSigBoolean(sig, kPGPSigPropIsExportable, &exportable);
			PGPGetSigBoolean(sig, kPGPSigPropIsExpired, &expired);
			PGPGetSigBoolean(sig, kPGPSigPropIsX509, &x509);
			PGPGetSigNumber(sig, kPGPSigPropTrustLevel, &trustLevel);
			
			if( x509 )
			{
				if( revoked )
					iconID = kX509CertRevokedIconID;
				else if( expired )
					iconID = kX509CertExpiredIconID;
				else
					iconID = kX509CertIconID;
			}
			else if(revoked)
				iconID = 	kRevokedSigID;
			else if(expired)
				iconID =	kSignatureExpiredIconID;
			else if(verified)
			{
				if(exportable)
				{
					if(trustLevel == 1)
						iconID =	kExportMetaSigIconID;
					else
						iconID =	kExportSignatureIconID;
				}
				else
				{
					if(trustLevel == 2)
						iconID =	kNoExportMetaSigIconID;
					else
						iconID =	kNoExportSignatureIconID;
				}
			}
			else if(tried)
				iconID =	kBadSignatureID;
			else if(notCorrupt)
			{
				if( exportable )
					iconID =	kExportSignatureIconID;
				else
					iconID =	kNoExportSignatureIconID;
			}
			else
				iconID =	kBadSignatureID;
			DrawIcon(iconID, cellRect, 3, hilite ?	kTransformSelected :
													kTransformNone );
			err = PGPGetSigCertifierKey(sig, mKeySet, &certKey);
			if(!PGPKeyRefIsValid(certKey) && !mDefaultRing)
			{
				err = PGPGetSigCertifierKey(sig,
						CPGPKeys::TheApp()->GetKeySet(),
						&certKey);
			}
			if(IsntPGPError(err) && PGPKeyRefIsValid(certKey))
			{
				err = PGPGetPrimaryUserIDNameBuffer(certKey,
					kUserIDStringLength, cstr, &len);
				pgpAssertNoErr(err);
				CToPString(cstr, pstr);
			}
			else if( x509 )
			{
				err = PGPGetSigPropertyBuffer( sig,
						kPGPSigPropX509IssuerLongName,
						sizeof(cstr) - 1, cstr, &len );
				pgpAssertNoErr(err);
				CToPString(cstr, pstr);
				TextFace( italic );
			}
			else
			{
				PGPKeyID		keyID;
				char			keyIDString[ kPGPMaxKeyIDStringSize ];
				
				::GetIndString(	pstr,
								kStringListID,
								kSigUserUnavailableID);
				err = PGPGetKeyIDOfCertifier(sig, &keyID);
				pgpAssertNoErr(err);
				err = PGPGetKeyIDString( &keyID,
										kPGPKeyIDString_Abbreviated,
										keyIDString);
				pgpAssertNoErr(err);
				CToPString(keyIDString, str2);
				if(str2[0] > 10)
					str2[0] = 10;
				AppendPString(str2, pstr);
				TextFace( italic );
			}
			TruncString(	mColumnInfo.columnWidths[kUserIDColumnID] -
							kDropFlagSlop - kLeftIndent -
							kLevelIndent * 2 - kIconWidth - kLeftBorder,
							pstr, truncMiddle );
			hiliteRect.left = cellRect.left + kDropFlagSlop +
							kLevelIndent * 2 + kIconWidth + kLeftBorder;
			break;
		}
	}
	MoveTo( hiliteRect.left + kLeftIndent, cellRect.bottom - kBottomBorder );
	width = StringWidth( pstr );
	if( hilite && !inactive )
		TextMode(srcCopy);
	DrawString( pstr );
	if( hilite )
	{
		hiliteRect.right = hiliteRect.left + width + kLeftIndent;
		hiliteRect.top = cellRect.top;
		hiliteRect.bottom = cellRect.bottom - 1;
		if( inactive )
		{
			RGBColor	hiliteColor;
			
			LMGetHiliteRGB( &hiliteColor );
			RGBForeColor( &hiliteColor );
			FrameRect( &hiliteRect );
		}
		else
		{
			UDrawingUtils::SetHiliteModeOn();
			InvertRect(&hiliteRect);
		}
	}
	TextFace( 0 );
	{
		Int16	sl, sr;
		
		sl = hiliteRect.left - kIconWidth - kLeftBorder;
		sr = hiliteRect.left + width;
		if( sr != ktr->selectRight || sl != ktr->selectLeft ||
			iconID != ktr->iconID )
		{
			ktr->selectLeft		= sl;
			ktr->selectRight	= sr;
			ktr->iconID			= iconID;
			return true;
		}
	}
	return false;
}

	void
CKeyTable::DrawValidityColumnCell(	KeyTableRef	*ktr,
									Rect		cellRect,
									Int16		depth)
{
	PGPError	err;
	PGPValidity	validity	= kPGPValidity_Unknown;
	PGPBoolean	axiomatic;
	
	switch(ktr->type)
	{
		PGPInt32	myValidityNumber	= 0;
		
		case kKey:
		case kUserID:
			if(ktr->type == kKey)
			{
				PGPKeyRef	key;
	
				key = ktr->u.key;
				err = PGPGetPrimaryUserIDValidity(key, &validity);
				pgpAssertNoErr(err);
				PGPGetKeyBoolean(key, kPGPKeyPropIsAxiomatic, &axiomatic);
			}
			else
			{
				PGPUserIDRef	user;
				PGPInt32		number;

				user = ktr->u.user;
				err = PGPGetUserIDNumber(user, kPGPUserIDPropValidity,
										&number);
				validity	= (PGPValidity)number;
				pgpAssertNoErr(err);
				PGPGetKeyBoolean(ktr->ownerKey, kPGPKeyPropIsAxiomatic,
									&axiomatic);
			}
			switch(validity)
			{
				default:
				case kPGPValidity_Unknown:
				case kPGPValidity_Invalid:
					myValidityNumber = 0;
					break;
				case kPGPValidity_Marginal:
					myValidityNumber = 1;
					break;
				case kPGPValidity_Complete:
					myValidityNumber = 2;
					break;
			}
			DrawValidity(axiomatic ? (kMaxValidity + 1) : myValidityNumber,
							cellRect, depth);
			break;
		case kSignature:
			break;
	}
}

	void
CKeyTable::DrawTrustColumnCell(		KeyTableRef	*ktr,
									Rect		cellRect,
									Int16		depth)
{
	PGPError err;
	
	switch(ktr->type)
	{
		case kKey:
			{
				PGPKeyRef	key;
				PGPInt32	trust;
	
				key = ktr->u.key;
				err = PGPGetKeyNumber(key, kPGPKeyPropTrust, &trust);
				pgpAssertNoErr(err);
				switch(trust)
				{
					default:
					case kPGPKeyTrust_Undefined:
					case kPGPKeyTrust_Unknown:
					case kPGPKeyTrust_Never:
						trust = 0;
						break;
					case kPGPKeyTrust_Marginal:
						trust = 1;
						break;
					case kPGPKeyTrust_Complete:
						trust = 2;
						break;
					case kPGPKeyTrust_Ultimate:
						trust = 3;
						break;
				}
				DrawTrust(trust, cellRect, depth);
			}
			break;
		case kUserID:
		case kSignature:
			break;
	}
}

	void
CKeyTable::DrawSizeColumnCell(	KeyTableRef	*ktr,
									Rect		cellRect,
									Int16		depth)
{
	PGPError err;
	
	switch(ktr->type)
	{
		case kKey:
		{
			PGPKeyRef		key;
			PGPSubKeyRef	subKey;
			PGPInt32		bits,
							algorithm;
			Str255			str,
							str2;

			key = ktr->u.key;
			PGPGetKeyNumber(key, kPGPKeyPropBits, &bits);
			PGPGetKeyNumber(key, kPGPKeyPropAlgID, &algorithm);
			NumToString(bits, str);
			if( algorithm == kPGPPublicKeyAlgorithm_DSA )
			{
				PGPKeyIterSeek(mKeyIter, key);
				err = PGPKeyIterNextSubKey(mKeyIter, &subKey);
				if( IsntPGPError( err ) && IsntNull( subKey ) )
				{
					err = PGPGetSubKeyNumber(subKey, kPGPKeyPropBits,
												&bits);
					pgpAssertNoErr(err);
					NumToString(bits, str2);
					AppendPString("\p/", str2);
					AppendPString(str, str2);
					CopyPString(str2, str);
				}
			}
			MoveTo(	cellRect.left + kLeftIndent + kTextIndent,
					cellRect.bottom - kBottomBorder);
			DrawString(str);
			break;
		}
		case kUserID:
		case kSignature:
			break;
	}
}

	void
CKeyTable::DrawKeyIDColumnCell(		KeyTableRef	*ktr,
									Rect		cellRect,
									Int16		depth)
{
	PGPError	err;
	Str255		pstr;
	PGPKeyID	keyID;
	char		keyIDString[ kPGPMaxKeyIDStringSize ];
	
	switch( ktr->type )
	{
		case kKey:
		{
			PGPKeyRef	key;

			key = ktr->u.key;
			err = PGPGetKeyIDFromKey(key, &keyID );
			pgpAssertNoErr(err);
			err = PGPGetKeyIDString( &keyID, kPGPKeyIDString_Abbreviated,
									keyIDString);
			pgpAssertNoErr(err);
			CToPString(keyIDString, pstr);

			MoveTo( cellRect.left + kLeftIndent + kTextIndent,
						cellRect.bottom - kBottomBorder );
			DrawString( pstr );
			break;
		}
		case kUserID:
			break;
		case kSignature:
		{
			PGPSigRef	sig;

			sig = ktr->u.sig;
			err = PGPGetKeyIDOfCertifier(sig, &keyID);
			pgpAssertNoErr(err);
			if( IsntPGPError( err ) )
			{
				err = PGPGetKeyIDString( &keyID, kPGPKeyIDString_Abbreviated,
										keyIDString);
				pgpAssertNoErr(err);
				CToPString(keyIDString, pstr);

				MoveTo( cellRect.left + kLeftIndent + kTextIndent,
							cellRect.bottom - kBottomBorder );
				DrawString( pstr );
			}
			break;
		}
	}
}

	void
CKeyTable::DrawCreationColumnCell(	KeyTableRef	*ktr,
									Rect		cellRect,
									Int16		depth)
{
	PGPTime		creation;
	Str255		str;
	
	switch(ktr->type)
	{
		case kKey:
		{
			PGPKeyRef	key;

			key = ktr->u.key;
			PGPGetKeyTime( key, kPGPKeyPropCreation, &creation );
			creation = PGPTimeToMacTime( creation );
			DateString( creation, shortDate, str, NULL );
			MoveTo( cellRect.left + kLeftIndent + kTextIndent,
					cellRect.bottom - kBottomBorder );
			DrawString( str );
			break;
		}
		case kUserID:
			break;
		case kSignature:
		{
			PGPSigRef	sig;

			sig = ktr->u.sig;
			PGPGetSigTime( sig, kPGPSigPropCreation, &creation );
			creation = PGPTimeToMacTime( creation );
			DateString( creation, shortDate, str, NULL );
			MoveTo( cellRect.left + kLeftIndent + kTextIndent,
					cellRect.bottom - kBottomBorder );
			DrawString( str );
			break;
		}
	}
}

	void
CKeyTable::DrawExpirationColumnCell(	KeyTableRef	*ktr,
										Rect		cellRect,
										Int16		depth)
{
	PGPTime		expiration;
	Str255		str;

	switch( ktr->type )
	{
		case kKey:
		{
			PGPKeyRef	key;

			key = ktr->u.key;
			PGPGetKeyTime( key, kPGPKeyPropExpiration, &expiration );
			if( expiration != 0 )
			{
				expiration = PGPTimeToMacTime(expiration);
				DateString( expiration, shortDate, str, NULL );
			}
			else
				GetIndString( str, kStringListID, kNeverExpiresID );
			MoveTo(	cellRect.left + kLeftIndent + kTextIndent,
						cellRect.bottom - kBottomBorder);
			DrawString( str );
			break;
		}
		case kSignature:
		{
			PGPSigRef	sig;

			sig = ktr->u.sig;
			PGPGetSigTime(sig, kPGPSigPropExpiration, &expiration);
			if( expiration != 0 )
			{
				expiration = PGPTimeToMacTime(expiration);
				DateString(expiration, shortDate, str, NULL);
			}
			else
				GetIndString( str, kStringListID, kNeverExpiresID);
			MoveTo( cellRect.left + kLeftIndent + kTextIndent,
						cellRect.bottom - kBottomBorder);
			DrawString( str );
			break;
		}
		case kUserID:
			break;
	}
}

	void
CKeyTable::DrawMRKColumnCell(		KeyTableRef	*ktr,
									Rect		cellRect,
									Int16		depth)
{
	switch( ktr->type )
	{
		case kKey:
		{
			PGPKeyRef		key;
			PGPError		err;
			PGPUInt32		numADKs;

			key = ktr->u.key;
			err = PGPCountAdditionalRecipientRequests( key, &numADKs );
			if( IsntPGPError( err ) && ( numADKs > 0 ) )
				DrawIcon(kMRKActiveIconID, cellRect, -1);
			else
				DrawIcon(kKeyAttrOffIconID, cellRect, -1);
			break;
		}
		case kUserID:
		case kSignature:
			break;
	}
}

	void
CKeyTable::DrawDescriptionColumnCell(	KeyTableRef	*ktr,
										Rect		cellRect,
										Int16		depth)
{
	Str255		str;
	PGPInt32	algorithm;
	Int16		strIndex;
	
	switch( ktr->type )
	{
		case kKey:
		{
			PGPKeyRef	key;
			PGPBoolean	secret,
						shared;

			key = ktr->u.key;
			PGPGetKeyBoolean(key, kPGPKeyPropIsSecret, &secret);
			PGPGetKeyNumber(key, kPGPKeyPropAlgID, &algorithm);
			PGPGetKeyBoolean( key, kPGPKeyPropIsSecretShared, &shared );
			if( shared )
				strIndex = kSplitKeyPairDescriptionID;
			else if( secret )
			{
				if( algorithm == kPGPPublicKeyAlgorithm_DSA )
					strIndex = kDHDSSKeyPairDescriptionID;
				else
					strIndex = kRSAKeyPairDescriptionID;
			}
			else
			{
				if( algorithm == kPGPPublicKeyAlgorithm_DSA )
					strIndex = kDHDSSKeyDescriptionID;
				else
					strIndex = kRSAKeyDescriptionID;
			}
			break;
		}
		case kUserID:
			strIndex = kUserIDDescriptionID;
			break;
		case kSignature:
		{
			PGPBoolean	x509;
			
			PGPGetSigNumber( ktr->u.sig, kPGPSigPropAlgID, &algorithm );
			PGPGetSigBoolean( ktr->u.sig, kPGPSigPropIsX509, &x509 );
			if( x509 )
			{
				if( algorithm == kPGPPublicKeyAlgorithm_DSA )
					strIndex = kX509DSSCertDescriptionID;
				else if( algorithm == kPGPPublicKeyAlgorithm_RSA )
					strIndex = kX509RSACertDescriptionID;
				else
					strIndex = kUnknownSignatureDescriptionID;
			}
			else
			{
				if( algorithm == kPGPPublicKeyAlgorithm_DSA )
					strIndex = kDSSSignatureDescriptionID;
				else if( algorithm == kPGPPublicKeyAlgorithm_RSA )
					strIndex = kRSASignatureDescriptionID;
				else
					strIndex = kUnknownSignatureDescriptionID;
			}
			break;
		}
	}
	GetIndString( str, kKeyDescriptionsStringListID, strIndex );
	MoveTo(	cellRect.left + kLeftIndent + kTextIndent,
				cellRect.bottom - kBottomBorder );
	DrawString( str );
}

	void
CKeyTable::DrawTrust(Int32 trustValue, Rect cellRect, Int16 depth)
{
	RGBColor	trustColor = { 52223, 52223, 65535 };
	Int16		width;
	
	width = mColumnInfo.columnWidths[kTrustColumnID] -
				kBarRightBorder - kBarLeftBorder;
	cellRect.left 	+=	kBarLeftBorder;
	cellRect.right	= 	cellRect.left + width;
	cellRect.bottom	=	cellRect.bottom - kBottomBorder;
	cellRect.top	=	cellRect.bottom - kBarHeight;
	
	RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
	FrameRect(&cellRect);
	InsetRect(&cellRect, 1, 1);
	if(trustValue >= kMaxTrust)
	{
		PenPixPat(mBarberPixPat);
		PaintRect(&cellRect);
		PenNormal();
	}
	else
	{
		if(depth >= 8)
		{
			RGBForeColor(&trustColor);
			PaintRect(&cellRect);
		}
		
		cellRect.right -= width * ((kMaxTrust - 1) - trustValue) /
									(kMaxTrust - 1);
		RGBForeColor( &UGAColorRamp::GetColor(colorRamp_Gray9) );
		PaintRect(&cellRect);
	}
	ApplyForeAndBackColors();
}

	void
CKeyTable::DrawValidity(Int32 trustValue, Rect cellRect, Int16 depth)
{
	RGBColor	trustColor = { 52223, 52223, 65535 };
	Int16		width;
	PGPBoolean	showMarginal;
	PGPError	err;
	
	err = PGPGetPrefBoolean(gPrefRef,
							kPGPPrefDisplayMarginalValidity, &showMarginal);
	pgpAssertNoErr(err);
	if(!showMarginal)
	{
		ResIDT		iconID;

		if(trustValue > kMaxValidity)
		{
			iconID = kAxiomaticIconID;
		}
		else if(trustValue > 1)
		{
			iconID = kValidKeyIconID;
		}
		else if(trustValue == 1)
		{
			PGPBoolean	marginalIsInvalid;
		
			err = PGPGetPrefBoolean(gPrefRef,
							kPGPPrefMarginalIsInvalid,
							&marginalIsInvalid);
			pgpAssertNoErr(err);
			
			if( marginalIsInvalid )
			{
				iconID = kKeyAttrOffIconID;
			}
			else
			{
				iconID = kValidKeyIconID;
			}
		}
		else
		{
			iconID = kKeyAttrOffIconID;
		}
		
		DrawIcon(iconID, cellRect, -2);
	}
	else
	{
		width = mColumnInfo.columnWidths[kValidityColumnID] - kBarRightBorder
				- kBarLeftBorder;
		cellRect.left 	+=	kBarLeftBorder;
		cellRect.right	= 	cellRect.left + width;
		cellRect.bottom	=	cellRect.bottom - kBottomBorder;
		cellRect.top	=	cellRect.bottom - kBarHeight;
		
		::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
		::FrameRect(&cellRect);
		::InsetRect(&cellRect, 1, 1);
		if(trustValue > kMaxValidity)
		{
			::PenPixPat(mBarberPixPat);
			::PaintRect(&cellRect);
			::PenNormal();
		}
		else
		{
			if(depth >= 8)
			{
				::RGBForeColor(&trustColor);
				::PaintRect(&cellRect);
			}
			
			cellRect.right -= width * (kMaxValidity - trustValue) /
					kMaxValidity;
			::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Gray9));
			::PaintRect(&cellRect);
		}
		ApplyForeAndBackColors();
	}
}

	void
CKeyTable::DrawIcon(
	ResIDT				iconID,
	Rect				cellRect,
	UInt32				indentLevel,
	IconTransformType	transform )
{
	Rect				iconRect;

	switch(indentLevel)
	{
		case -2:
			iconRect.left 	= 	cellRect.left + 10;
			break;
		case -1:
			iconRect.left 	= 	cellRect.left + kLeftIndent;
			break;
		case 0:
			iconRect.left 	= 	cellRect.left + kDropFlagSlop;
			break;
		case 1:
			iconRect.left 	= 	cellRect.left + kDropFlagSlop + kLeftIndent;
			break;
		case 2:
			iconRect.left 	= 	cellRect.left + kDropFlagSlop + kLeftIndent
									+ kLevelIndent;
			break;
		case 3:
			iconRect.left 	= 	cellRect.left + kDropFlagSlop + kLeftIndent
									+ kLevelIndent * 2;
			break;
		default:
			pgpAssert(0);
			break;
	}
			
	iconRect.top	=	cellRect.top + 1;
	iconRect.right 	= 	iconRect.left + 16;//kIconWidth;
	iconRect.bottom = 	iconRect.top + 16;//kIconHeight;
	
	PlotIconID(&iconRect, kAlignNone, transform, iconID);
}

	void
CKeyTable::RedrawTable()
{
	Rect frame;
	
	CalcPortFrameRect(frame);
	InvalPortRect(&frame);
}

	void
CKeyTable::HiliteCellActively(
	const STableCell	&inCell,
	Boolean				inHilite )
{
	StColorPenState	saveColorPen;
	Rect			cellFrame,
					insideFrame;
	
	if( inCell.col != 1 )
		return;
	if( GetLocalCellRect(inCell, cellFrame) && FocusExposed() )
	{
		insideFrame = cellFrame;
		insideFrame.bottom--;
		TextFont( kFontIDGeneva );
		TextSize( 9 );
		TextFace( 0 );
		TextMode( srcOr );
		if( !inHilite )
		{
			StDeviceLoop	devLoop(insideFrame);		
			SInt16			depth;

			while(devLoop.NextDepth(depth))
			{
				SetThemeBackground( kThemeListViewSortColumnBackgroundBrush,
									depth, ( depth > 1 ) );
				EraseRect( &insideFrame );
			}
		}
		DrawCell1( inCell, cellFrame, inHilite, false );
	}
}

	void
CKeyTable::HiliteCellInactively(
	const STableCell	&inCell,
	Boolean			 	inHilite )
{
	StColorPenState	saveColorPen;
	Rect			cellFrame,
					insideFrame;
	
	if( inCell.col != 1 )
		return;
	if( GetLocalCellRect(inCell, cellFrame) && FocusExposed() )
	{
		insideFrame = cellFrame;
		insideFrame.bottom--;
		TextFont( kFontIDGeneva );
		TextSize( 9 );
		TextFace( 0 );
		TextMode( srcOr );
		if( !inHilite )
		{
			StDeviceLoop	devLoop(insideFrame);		
			SInt16			depth;

			while(devLoop.NextDepth(depth))
			{
				SetThemeBackground( kThemeListViewSortColumnBackgroundBrush,
									depth, ( depth > 1 ) );
				EraseRect( &insideFrame );
			}
		}
		DrawCell1( inCell, cellFrame, inHilite, true );
	}
}

#pragma mark --- Table Management ---

	void
CKeyTable::UnmarkKey(PGPKeyRef inKey)
{
	PGPKeyRef		key;
	PGPUserIDRef	user;
	PGPSigRef		sig;
	PGPError		err;

	PGPKeyIterRewind(mKeyIter);
	while(IsntPGPError(err = PGPKeyIterNext(mKeyIter, &key)) && IsntNull(key))
	{
		if(key == inKey)
		{
			err = PGPSetKeyUserVal(key, kUserValueUnmarked);
			pgpAssert(!err);
			while(IsntPGPError(err = PGPKeyIterNextUserID(mKeyIter, &user)) &&
					IsntNull(user))
			{
				err = PGPSetUserIDUserVal(user, kUserValueUnmarked);
				pgpAssert(!err);
				while(IsntPGPError(
					err = PGPKeyIterNextUIDSig(mKeyIter, &sig)) &&
					IsntNull(sig))
				{
					err = PGPSetSigUserVal(sig, kUserValueUnmarked);
					pgpAssert(!err);
				}
			}
			break;
		}
	}
}

	void
CKeyTable::RemoveKeyFromTable(PGPKeyRef inKey)
{
	KeyTableRef		tableRef;
	STableCell		cell(1,1);
	TableIndexT		rows,
					cols,
					rowIndex;
	Uint32			dataSize = sizeof(KeyTableRef);
	
	GetWideOpenTableSize(rows, cols);
	for(rowIndex = 1;rowIndex < rows;rowIndex++)
	{
		cell.row = rowIndex;
		GetCellData(cell, &tableRef, dataSize);
		if((tableRef.type == kKey) && (tableRef.u.key == inKey))
		{
			RemoveRows(1, rowIndex, true);
			break;
		}
	}
}

	void
CKeyTable::TerminateKey(PGPKeyRef inKey)
{
	KeyTableRef		tableRef;
	STableCell		cell(1,1);
	TableIndexT		rows,
					cols,
					rowIndex;
	Uint32			dataSize = sizeof(KeyTableRef);
	
	GetWideOpenTableSize(rows, cols);
	for(rowIndex = 1;rowIndex < rows;rowIndex++)
	{
		cell.row = rowIndex;
		GetCellData(cell, &tableRef, dataSize);
		if((tableRef.type == kKey) && (tableRef.u.key == inKey))
		{
			tableRef.terminated = TRUE;
			SetCellData(cell, &tableRef, dataSize);
			break;
		}
	}
}

	void
CKeyTable::RemoveTerminatedKeys()
{
	KeyTableRef		tableRef;
	STableCell		cell(1,1);
	TableIndexT		rows,
					cols,
					rowIndex;
	Uint32			dataSize = sizeof(KeyTableRef);
	
	GetWideOpenTableSize(rows, cols);
	for(rowIndex = rows;rowIndex > 0;rowIndex--)
	{
		cell.row = rowIndex;
		GetCellData(cell, &tableRef, dataSize);
		if((tableRef.type == kKey) && tableRef.terminated)
			RemoveRows(1, rowIndex, true);
	}
}

	void
CKeyTable::ClearTable()
{
	RemoveAllRows(false);
}

/*
ResyncTable

This is a meaty function.  It runs through the KeyDB list of keys and checks
each key, userID, and signature to see whether its userVal has been set to
non-zero.  If it hasn't, then the function assumes that entry is not yet in
the table and adds it.  It also sets its userVal to 1 so that it wont be
added again later.  This function is used in many places, for example when
signing key(s), this function is called to merge all the new signature(s)
into the displayed list.

Returns TRUE if at least one key pair was found while iterating over the
list
*/

	Boolean
CKeyTable::ResyncTable(
	Boolean		collapse,
	Boolean		onlyUnmarked,
	Boolean		selectUnmarkedKeys )
{
	PGPKeyRef 		key;
	PGPUserIDRef	user;
	PGPSigRef		sig;
	PGPError		err;
	KeyTableRef		tableRef;
	Int32			userVal;
	TableIndexT		uidSubRow,
					sigSubRow,
					lastKeyRow,
					lastRow;
	STableCell		selectCell(0,0);
	Int16			uidIndex,
					sigIndex;
	Boolean			pairFound = FALSE;
	PGPBoolean		secret;
	
	lastRow = lastKeyRow = 0;
	if(mDefaultRing)
	{
		PGPGetDefaultPrivateKey(mKeySet, &mDefaultKey);
	}
	PGPKeyIterRewind(mKeyIter);
	while(IsntPGPError(err = PGPKeyIterNext(mKeyIter, &key)) && IsntNull(key))
	{
		err = PGPGetKeyBoolean(key, kPGPKeyPropIsSecret, &secret);
		pgpAssert(!err);
		if(secret)
			pairFound = TRUE;
		err = PGPGetKeyUserVal(key, (PGPUserValue *)&userVal);
		pgpAssert(!err);
		++lastRow;
		if(userVal != kUserValueMarked || !onlyUnmarked)
		{
			tableRef.type		=	kKey;
			tableRef.ownerKey	=	key;
			tableRef.terminated =	FALSE;
			tableRef.u.key		=	key;
			lastKeyRow = InsertSiblingRows(1, lastKeyRow, &tableRef,
									sizeof(KeyTableRef), TRUE, TRUE);
			err = PGPSetKeyUserVal(key, (PGPUserValue)kUserValueMarked);
			pgpAssertNoErr(err);
			if( selectUnmarkedKeys )
			{
				selectCell.col = 1;	selectCell.row = GetExposedIndex( lastKeyRow );
			}
		}
		else if(!lastKeyRow)
			lastKeyRow = 1;
		if(lastRow > 1)
			lastKeyRow = lastRow;
		for(uidIndex = 1;IsntPGPError(err = PGPKeyIterNextUserID(mKeyIter,
							&user)) && IsntNull(user);uidIndex++)
		{
			err = PGPKeyIterNextUIDSig(mKeyIter, &sig);
			err = PGPGetUserIDUserVal(user, (PGPUserValue *)&userVal);
			pgpAssert(!err);
			++lastRow;
			if(userVal != kUserValueMarked || !onlyUnmarked)
			{
				tableRef.type		=	kUserID;
				tableRef.terminated =	FALSE;
				tableRef.ownerKey	=	key;
				tableRef.u.user		=	user;
				if(uidIndex == 1)	// First user ID
				{
					InsertChildRows(1, lastKeyRow, &tableRef,
										sizeof(KeyTableRef), TRUE, TRUE);
					uidSubRow = lastKeyRow + 1;
				}
				else
					uidSubRow = InsertSiblingRows(1, uidSubRow, &tableRef,
										sizeof(KeyTableRef), TRUE, TRUE);
				err = PGPSetUserIDUserVal(user,
										(PGPUserValue)kUserValueMarked);
				pgpAssert(!err);
			}
			else
			{
				if(uidIndex == 1)
					uidSubRow = lastKeyRow + 1;
				else
					uidSubRow = lastRow;
			}
			for(sigIndex = 1; (sigIndex == 1 && IsntNull(sig)) ||
				IsntPGPError(err = PGPKeyIterNextUIDSig(mKeyIter, &sig)) &&
				IsntNull(sig);sigIndex++)
			{
				err = PGPGetSigUserVal(sig, (PGPUserValue *)&userVal);
				pgpAssert(!err);
				++lastRow;
				if(userVal != kUserValueMarked || !onlyUnmarked)
				{
					tableRef.type		=	kSignature;
					tableRef.ownerKey	=	key;
					tableRef.terminated =	FALSE;
					tableRef.u.sig		=	sig;
					if(sigIndex == 1)	// First signature
					{
						InsertChildRows(1, uidSubRow, &tableRef,
										sizeof(KeyTableRef), FALSE, TRUE);
						sigSubRow = uidSubRow + 1;
					}
					else
						sigSubRow = InsertSiblingRows(1, sigSubRow,
										&tableRef, sizeof(KeyTableRef),
										FALSE, TRUE);
					err = PGPSetSigUserVal(sig,
											(PGPUserValue)kUserValueMarked);
					pgpAssert(!err);
				}
				else
				{
					if(sigIndex == 1)
						sigSubRow = uidSubRow + 1;
					else
						sigSubRow = lastRow;
				}
			}
		}
	}
	if(collapse)
		CollapseKeys(TRUE);
	if( !selectCell.IsNullCell() )
	{
		ScrollCellIntoFrame( selectCell );
		SelectCell( selectCell );
	}
	return pairFound;
}

	PGPError
CKeyTable::ReverifySigsHandler(
	PGPContextRef	context,
	PGPEvent		*event,
	PGPUserValue	userValue )
{
	CPGPStDialogHandler		*dialog = (CPGPStDialogHandler *) userValue;
	
	if( dialog->DoDialog() == msg_Cancel )
		return kPGPError_UserAbort;
	else
		return kPGPError_NoErr;
}

	void
CKeyTable::ReverifySigs()
{
	CPGPStDialogHandler		dialogHandler( kProgressDialogResID, nil );
	CGAProgressDialog *		dialogObj;
	Str255					pstr;
	PGPError				err = kPGPError_NoErr;
	PGPKeySetRef			checkKeys = kInvalidPGPKeySetRef,
							singleKeySet = kInvalidPGPKeySetRef;
	PGPUInt32				numKeys;
	TableIndexT				numRows, numCols;
		
	dialogObj = (CGAProgressDialog *) dialogHandler.GetDialog();
	GetIndString( pstr, kStringListID, kVerifyingSignaturesID );
	dialogObj->SetCaption( pstr );
	dialogObj->SetIndeterminateFlag( TRUE );
	dialogObj->Show();
	
	GetTableSize(numRows, numCols);
	STableCell	topCell(1, 1);
	STableCell 	botCell(numRows, 1);
	STableCell	cell;				// Loop thru all cells

	err = PGPNewEmptyKeySet(mKeySet, &checkKeys);
	if( IsPGPError( err ) )
		goto done;
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		if(CellIsSelected(cell)) 
		{
			TableIndexT		inWideOpenRow =
							mCollapsableTree->GetWideOpenIndex(cell.row);
			STableCell		woCell(inWideOpenRow, 1);
			KeyTableRef		keyRef;
			PGPKeySetRef	oldKeys = checkKeys;
			Uint32			dataSize = sizeof(KeyTableRef);
			PGPKeySetRef	singleKeySet;
			
			GetCellData(woCell, &keyRef, dataSize);
			err = PGPNewSingletonKeySet(keyRef.ownerKey, &singleKeySet);
			if( IsPGPError( err ) )
				goto done;
			err = PGPUnionKeySets(oldKeys, singleKeySet, &checkKeys);
			if( IsPGPError( err ) )
				goto done;
			PGPFreeKeySet( oldKeys );
			PGPFreeKeySet( singleKeySet );
			singleKeySet = kInvalidPGPKeySetRef;
			if( dialogHandler.DoDialog() == msg_Cancel )
			{
				err = kPGPError_UserAbort;
				goto done;
			}
		}
	}
	PGPCountKeys( checkKeys, &numKeys );
	if( numKeys > 0 )
	{
		err = PGPCheckKeyRingSigs( checkKeys, CPGPKeys::TheApp()->GetKeySet(),
								TRUE, ReverifySigsHandler, &dialogHandler );
		if( IsPGPError( err ) )
			goto done;
		RedrawTable();
	}
done:
	if( PGPKeySetRefIsValid( checkKeys ) )
		PGPFreeKeySet( checkKeys );
	if( PGPKeySetRefIsValid( singleKeySet ) )
		PGPFreeKeySet( singleKeySet );
	if( IsPGPError( err ) && err != kPGPError_UserAbort )
		ReportPGPError( err );
}

	void
CKeyTable::Sign()
{
	TableIndexT		numRows, numCols;
	PGPError		err;
	KeyTableRef		*userIDs;
	Int32			numUserIDs;
	PGPBoolean		ksSync;
		
	GetTableSize(numRows, numCols);
	STableCell	topCell(1, 1);
	STableCell 	botCell(numRows, 1);
	STableCell	cell;				// Loop thru all cells
	
	err = PGPGetPrefBoolean(gPrefRef,
				kPGPPrefKeyServerSyncOnKeySign, &ksSync);
	pgpAssertNoErr(err);
	numUserIDs = 0;
	userIDs = (KeyTableRef	*)pgpAlloc(sizeof(KeyTableRef));
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		if(CellIsSelected(cell)) 
		{
			TableIndexT		inWideOpenRow =
							mCollapsableTree->GetWideOpenIndex(cell.row);
			STableCell		woCell(inWideOpenRow, 1);
			KeyTableRef		keyRef;
			Uint32			dataSize = sizeof(KeyTableRef);
			
			GetCellData(woCell, &keyRef, dataSize);
			if(keyRef.type == kSignature)
				continue;
			err = pgpRealloc(	&userIDs,
								(numUserIDs + 1) * sizeof(KeyTableRef));
			pgpAssertNoErr(err);
			if(IsntPGPError(err))
			{
				userIDs[numUserIDs] = keyRef;
				numUserIDs ++;
				if(ksSync)
					GetCellOnServer(cell);
			}
		}
	}
	if(numUserIDs > 0)
	{
		CSignDialog *mSignDialog;
		
		mSignDialog = (CSignDialog *)
			LWindow::CreateWindow(kSignDialogResID, CPGPKeys::TheApp());
		mSignDialog->AddListener(this);
		mSignDialog->SetUserIDs(mDefaultKey, userIDs, numUserIDs);
		if(ksSync)
		{
			ResyncTable(FALSE, TRUE);
			RedrawTable();
		}
	}
	else
		pgpFree(userIDs);
}

	Boolean
CKeyTable::Delete(STableCell& inCell)
{
	TableIndexT		inWideOpenRow =
						mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell		woCell(inWideOpenRow, 1);
	KeyTableRef		keyRef;
	Uint32			dataSize = sizeof(KeyTableRef);
	Boolean			result = TRUE;
	PGPError		err;
	
	GetCellData(woCell, &keyRef, dataSize);

	switch(keyRef.type)
	{
		case kKey:
		{
			PGPKeySetRef		singleKeySet;
			
			err = PGPNewSingletonKeySet(keyRef.u.key, &singleKeySet);
			if(IsntPGPError(err) && IsntNull(singleKeySet))
			{
				err = PGPRemoveKeys( singleKeySet, mKeySet );
				pgpAssertNoErr(err);
				PGPFreeKeySet(singleKeySet);
				if(IsntPGPError(err))
				{
					RemoveRows(1, inWideOpenRow, true);
					if(mDefaultKey == keyRef.u.key)
						mDefaultKey = NULL;
					if(mDefaultRing)
						CPGPKeys::TheApp()->InvalidateCaches();
				}
			}
			break;
		}
		case kUserID:
			// This procedure could resort the table, so we need to
			// remove the whole key from the table and resync with
			// the sorted iterator
			
			UnmarkKey(keyRef.ownerKey);
			TerminateKey(keyRef.ownerKey);
			
			err = PGPRemoveUserID(keyRef.u.user);
			if(err != kPGPError_BadParams)	// tried to delete primary user id
				pgpAssertNoErr(err);
			break;
		case kSignature:
			err = PGPRemoveSig(keyRef.u.sig);
			pgpAssertNoErr(err);
			if(IsntPGPError(err))
				RemoveRows(1, inWideOpenRow, true);
			break;
	}
	return result;
}

	Boolean
CKeyTable::EnableKey(STableCell& inCell)
{
	TableIndexT		inWideOpenRow =
						mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell		woCell(inWideOpenRow, 1);
	KeyTableRef		keyRef;
	Uint32			dataSize = sizeof(KeyTableRef);

	GetCellData(woCell, &keyRef, dataSize);
	if( keyRef.type == kKey )
		PGPEnableKey( keyRef.u.key );
	return TRUE;
}

	Boolean
CKeyTable::DisableKey(STableCell& inCell)
{
	TableIndexT		inWideOpenRow =
						mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell		woCell(inWideOpenRow, 1);
	KeyTableRef		keyRef;
	Uint32			dataSize = sizeof(KeyTableRef);

	GetCellData(woCell, &keyRef, dataSize);
	if( keyRef.type == kKey )
		PGPDisableKey( keyRef.u.key );
	return TRUE;
}

	Boolean
CKeyTable::ShowKeyInfo(STableCell& inCell)
{
	TableIndexT		inWideOpenRow =
						mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell		woCell(inWideOpenRow, 1);
	KeyTableRef		keyRef;
	Uint32			dataSize = sizeof(KeyTableRef);
	Boolean			result = TRUE;
	
	GetCellData(woCell, &keyRef, dataSize);
	if(keyRef.type == kKey)
	{
		if(!CKeyInfoWindow::OpenKeyInfo(keyRef.u.key, this))
			result = FALSE;	// Too many key windows, break out
	}
	else if(keyRef.type == kUserID)
	{
		if(!CKeyInfoWindow::OpenKeyInfo(keyRef.ownerKey, this))
			result = FALSE;	// Too many key windows, break out
	}
	else if(keyRef.type == kSignature)
	{
		if(!CCertPropertiesWindow::OpenCertProp(keyRef.u.sig,
			keyRef.ownerKey, this))
			result = FALSE;	// Too many cert windows, break out
	}
	
	return result;
}

	Boolean
CKeyTable::Revoke(STableCell& inCell)
{
	PGPError			err = kPGPError_NoErr;
	TableIndexT			inWideOpenRow =
							mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell			woCell(inWideOpenRow, 1);
	KeyTableRef			keyRef;
	Uint32				dataSize = sizeof(KeyTableRef);
	CSecureCString256	passphrase;
	PGPByte				*passKey = NULL;
	PGPSize				passKeySize;
	Boolean				result = TRUE;
	PGPBoolean			ksSync,
						secret,
						split = FALSE;
	PGPKeyRef			revokerKey = kInvalidPGPKeyRef;
	
	err = PGPGetPrefBoolean(gPrefRef,
				kPGPPrefKeyServerSyncOnRevocation, &ksSync);	CKERR;
	GetCellData(woCell, &keyRef, dataSize);
	if(keyRef.type == kKey)
	{
		if(CWarningAlert::Display(kWACautionAlertType, kWAOKCancelStyle,
						kStringListID, kRevokeConfirmID) == msg_OK)
		{
			if(ksSync)
				GetCellOnServer(inCell);
			
			PGPGetKeyBoolean( keyRef.u.key, kPGPKeyPropIsSecret, &secret );
			if( secret )
				revokerKey = keyRef.u.key;
			else
			{
				PGPUInt32	numRevokers,
							revIndex;
				PGPBoolean	revSecret,
							revSplit;
				PGPKeyRef	key;
				
				err = PGPCountRevocationKeys( keyRef.u.key, &numRevokers );
				CKERR;
				for( revIndex = 0; revIndex < numRevokers; revIndex++ )
				{
					err = PGPGetIndexedRevocationKey( keyRef.u.key, mKeySet,
								revIndex, &key, NULL ); CKERR;
					if( PGPKeyRefIsValid( key ) )
					{
						PGPGetKeyBoolean( key, kPGPKeyPropIsSecret,
							&revSecret );
						PGPGetKeyBoolean( key, kPGPKeyPropIsSecretShared,
							&revSplit );
						if( revSecret )
						{
							if( !PGPKeyRefIsValid( revokerKey ) || !revSplit )
								revokerKey = key;
							if( !revSplit )
								break;
						}
					}
				}
			}
			if( PGPKeyRefIsValid( revokerKey ) )
				err = GetPassForKey( revokerKey, &split, passphrase,
									&passKey, &passKeySize );
			else
				err = kPGPError_ItemNotFound;

			if( IsntPGPError( err ) )
			{
				/* Revoke all subkeys first */
				
				if( PGPKeyIterSeek( mKeyIter, keyRef.u.key ) >= 0 )
				{
					PGPSubKeyRef	subKey;
					
					err = PGPKeyIterNextSubKey( mKeyIter, &subKey );
					while( IsntPGPError( err ) )
					{
						CPGPKeys::TheApp()->GetMinimumRandomData();
						
						err = PGPRevokeSubKey( subKey, split ?
									PGPOPasskeyBuffer( gPGPContext, passKey,
										passKeySize ) :
									PGPOPassphrase( gPGPContext, passphrase ),
									PGPOLastOption( gPGPContext ) );
					
						if( IsntPGPError( err ) )
							err = PGPKeyIterNextSubKey( mKeyIter, &subKey );
					}
					
					if( err == kPGPError_EndOfIteration )
						err = kPGPError_NoErr;
				}
				else
				{
					err = kPGPError_KeyInvalid;	/* Better error? */
				}
			}
			
			if( IsntPGPError( err ) )
			{
				CPGPKeys::TheApp()->GetMinimumRandomData();
				err = PGPRevokeKey( keyRef.u.key, split ?
					PGPOPasskeyBuffer( gPGPContext, passKey, passKeySize ) :
					PGPOPassphrase( gPGPContext, passphrase ),
					PGPOLastOption( gPGPContext ) );
				if(ksSync)
					SendKeyToServer(inCell);
			}
		}
	}
	else if( keyRef.type == kSignature )
	{
		PGPKeyRef	certKey;
		PGPBoolean	x509;
		
		PGPGetSigBoolean( keyRef.u.sig, kPGPSigPropIsX509, &x509 );
		if( x509 )
		{
			CWarningAlert::Display( kWAStopAlertType, kWAOKStyle,
						kStringListID, kNoX509Revocation );
		}
		else
		{
			err = PGPGetSigCertifierKey( keyRef.u.sig, mKeySet, &certKey );
			if( IsntPGPError( err ) && PGPKeyRefIsValid( certKey ) )
			{
				if(ksSync)
					GetKeyOnServer(certKey);
				err = GetPassForKey( certKey, &split, passphrase,
										&passKey, &passKeySize );
				if( IsntPGPError( err ) )
				{	
					CPGPKeys::TheApp()->GetMinimumRandomData();
					err = PGPRevokeSig( keyRef.u.sig, mKeySet, split ?
						PGPOPasskeyBuffer( gPGPContext, passKey, passKeySize ) :
						PGPOPassphrase( gPGPContext, passphrase),
						PGPOLastOption( gPGPContext ) );
					if( IsntPGPError( err ) && ksSync)
					{
						err = PGPSendKeyToServer(gPGPContext, gTLSContext,
									certKey,  NULL);
					}
				}
			}
		}
	}
done:
	if( err == kPGPError_ItemNotFound )
	{
		CWarningAlert::Display( kWAStopAlertType, kWAOKStyle,
						kStringListID, kNoRevokerKeyID );
	}
	else if( IsPGPError( err ) )
		ReportPGPError( err );
	if( IsntNull( passKey ) )
		PGPFreeData( passKey );
	return ( err == kPGPError_NoErr );
}

typedef struct CAServerComm
{
	PGPkeysCAOp				op;
	CPGPStDialogHandler		*dialog;
	PGPBoolean				abort;
	PGPUInt32				lastState;
	CKeyTable *				table;
} CAServerComm;

	PGPError
CKeyTable::CAServerHandler(
	PGPContextRef			context,
	PGPEvent				*event,
	PGPUserValue			userValue )
{
	CAServerComm *			comm = (CAServerComm *) userValue;
	CGAProgressDialog *		dialogObj;
	PGPError				err = kPGPError_NoErr;
	PGPKeySetRef			remoteSet = kInvalidPGPKeySetRef;
	PGPKeyID *				authKeyID = NULL;
	
	LThread::Yield();
	dialogObj = (CGAProgressDialog *) comm->dialog->GetDialog();
	
	switch( event->type )
	{
		case kPGPEvent_KeyServerEvent:
			if( comm->lastState != event->data.keyServerData.state)
			{
				Str255	caption;
				
				GetIndString( caption, kServerStatusSTR,
								event->data.keyServerData.state);
				dialogObj->SetCaption( caption );
				comm->lastState = event->data.keyServerData.state;
			}
			break;
		case kPGPEvent_KeyServerTLSEvent:
		{
			PGPtlsSessionRef		tls = event->data.keyServerTLSData.tlsSession;
			PGPKeyRef				remoteKey;
			PGPtlsCipherSuiteNum	cipher;
			PGPKeyID				remoteKeyID;
			PGPInt32				remoteAlg;
			PGPClientPrefsNumber	authAlgPref;
			PGPClientPrefsByte		authKeyIDPref;
			PGPClientPrefsString	serverURLPref;
			PGPUInt32				authAlg;
			PGPSize					keyIDSize;
			PGPBoolean				confirm = TRUE;
			
			err = PGPtlsGetRemoteAuthenticatedKey( tls, &remoteKey, NULL );	CKERR;
			err = PGPNewSingletonKeySet( remoteKey, &remoteSet );			CKERR;
			err = PGPtlsGetNegotiatedCipherSuite( tls, &cipher );			CKERR;
			err = PGPGetKeyIDFromKey( remoteKey, &remoteKeyID );			CKERR;
			err = PGPGetKeyNumber( remoteKey, kPGPKeyPropAlgID, &remoteAlg );CKERR;
			if( comm->op == kPGPCA_OP_UpdateCRL )
			{
				authAlgPref		= kPGPPrefCARevocationServerAuthKeyAlg;
				authKeyIDPref	= kPGPPrefCARevocationServerAuthKeyID;
				serverURLPref	= kPGPPrefCARevocationServerURL;
			}
			else
			{
				authAlgPref		= kPGPPrefCAServerAuthKeyAlg;
				authKeyIDPref	= kPGPPrefCAServerAuthKeyID;
				serverURLPref	= kPGPPrefCAServerURL;
			}
			PGPGetPrefNumber( gPrefRef, authAlgPref, &authAlg );
			PGPGetPrefData( gPrefRef, authKeyIDPref, &keyIDSize, &authKeyID );
			if( IsntNull( authKeyID ) && ( authAlg == remoteAlg ) &&
				!PGPCompareKeyIDs( &remoteKeyID, authKeyID ) )
				confirm = FALSE;
			if( confirm )
			{
				char	serverURL[256];

				err = PGPGetPrefStringBuffer( gPrefRef, serverURLPref,
										sizeof(serverURL), serverURL );		CKERR;
				err = PGPConfirmRemoteKeyDialog( "", serverURL, remoteKey,
						cipher, TRUE );										CKERR;
				PGPSetPrefNumber( gPrefRef, authAlgPref, remoteAlg );
				PGPSetPrefData( gPrefRef, authKeyIDPref,
					sizeof(PGPKeyID), &remoteKeyID );
				comm->table->ImportKeysFromKeySet( remoteSet );
				remoteSet = kInvalidPGPKeySetRef;
			}
			break;
		}
	}
done:
	if( comm->abort )
		err = kPGPError_UserAbort;
	if( PGPKeySetRefIsValid( remoteSet ) )
		PGPFreeKeySet( remoteSet );
	if( IsntNull( authKeyID ) )
		PGPFreeData( authKeyID );
	return err;
}

	Boolean
CKeyTable::CAServerOp(STableCell& inCell)
{
	TableIndexT			inWideOpenRow =
							mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell			woCell(inWideOpenRow, 1);
	KeyTableRef			keyRef;
	Uint32				dataSize = sizeof(KeyTableRef);
	PGPError			err = kPGPError_NoErr;
	Boolean				result = TRUE;
	PGPBoolean			secret = TRUE;
	PGPKeyRef			key = kInvalidPGPKeyRef;
	PGPUserIDRef		userID = kInvalidPGPUserIDRef;
	PGPKeyServerClass	serverClass;
	PGPAttributeValue *	avList = NULL;
	PGPUInt32			numAVs = 0;
	
	if( inCell.col > 0 )
	{
		GetCellData(woCell, &keyRef, dataSize);
		if( keyRef.type == kKey )
		{
			key = keyRef.u.key;
			PGPGetPrimaryUserID( key, &userID );
		}
		else if( keyRef.type == kUserID )
		{
			userID = keyRef.u.user;
			key = PGPGetUserIDKey( userID );
		}
		else
			result = FALSE;
	}
	if( result )
	{
		PGPGetPrefNumber( gPrefRef, kPGPPrefCAType,
			(PGPUInt32 *)&serverClass );
		if( PGPKeyRefIsValid( key ) )
			PGPGetKeyBoolean( key, kPGPKeyPropIsSecret, &secret );
		if(secret)
		{
			if( mCAOp == kPGPCA_OP_RequestCert )
			{
				err = PGPGetCACertReqAVList( gPGPContext, FALSE, userID, serverClass,
					&avList, &numAVs );
			}
			if( IsntPGPError( err ) )
			{
				CPGPStDialogHandler		dialogHandler( kProgressDialogResID, nil );
				CGAProgressDialog *		dialogObj;
				CAServerComm			comm;
				Str255					caption;
				
				comm.op			= mCAOp;
				comm.abort		= FALSE;
				comm.dialog		= &dialogHandler;
				comm.table		= this;
				dialogObj = (CGAProgressDialog *) dialogHandler.GetDialog();
				GetIndString( caption, kServerStatusSTR, kPGPKeyServerState_Opening);
				dialogObj->SetCaption( caption );
				dialogObj->SetIndeterminateFlag( TRUE );
				dialogObj->Show();

				CAThread *	caThread = new CAThread(
								mCAOp, key, userID, mKeySet, avList, numAVs,
								CAServerHandler, &comm, &err );
				
				err = kPGPCA_InProgress;
				caThread->Resume();
				while( err == kPGPCA_InProgress )
				{
					if( dialogHandler.DoDialog() == msg_Cancel )
						comm.abort = TRUE;
					LThread::Yield();
				}
			}
		}
		if( IsntPGPError( err ) )
		{
			PGPInt16	stringID;
			
			switch( mCAOp )
			{
				case kPGPCA_OP_RequestCert:
					stringID = kCertReqSuccessfulID;
					break;
				case kPGPCA_OP_RetrieveCert:
					stringID = kCertRetrieveSuccessID;
					break;
				case kPGPCA_OP_UpdateCRL:
					stringID = kCRLUpdateSuccessID;
					break;
			}
			CWarningAlert::Display( kWANoteAlertType, kWAOKStyle,
							kStringListID, stringID );
		}
		else
			ReportPGPError( err );
	}
	if( IsntNull( avList ) )
		PGPFreeCACertReqAVList( avList, numAVs );
	return result;
}

	Boolean
CKeyTable::Split(STableCell& inCell)
{
	TableIndexT			inWideOpenRow =
							mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell			woCell(inWideOpenRow, 1);
	KeyTableRef			keyRef;
	Uint32				dataSize = sizeof(KeyTableRef);
	PGPError			err;
	Boolean				result = TRUE;
	PGPBoolean			secret;
	
	GetCellData(woCell, &keyRef, dataSize);
	if(keyRef.type == kKey)
	{
		err = PGPGetKeyBoolean(keyRef.u.key, kPGPKeyPropIsSecret, &secret);
		pgpAssertNoErr(err);
		if(secret)
		{
			CSplitWindow *mSplitWindow;
			
			mSplitWindow = (CSplitWindow *)
				LWindow::CreateWindow(kSplitDialogResID, this);
			mSplitWindow->AddListener(this);
			mSplitWindow->SetSplitKey(keyRef.u.key);
		}
	}
	return result;
}

	Boolean
CKeyTable::SendKeyToServer(STableCell& inCell)
{
	TableIndexT		inWideOpenRow =
						mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell		woCell(inWideOpenRow, 1);
	KeyTableRef		tableRef;
	UInt32			dataSize = sizeof(KeyTableRef);
	PGPError		err = kPGPError_NoErr;

	GetCellData(woCell, &tableRef, dataSize);

	err = PGPSendKeyToServer(gPGPContext, gTLSContext, tableRef.ownerKey,
							mTargetKeyServer);
	if( err == kPGPError_ServerKeyFailedPolicy )
	{
		CWarningAlert::Display( kWANoteAlertType, kWAOKStyle,
						kStringListID, kFailedServerPolicyID );
	}
	else if( IsPGPError(err) && ( err != kPGPError_UserAbort ) )
		ReportPGPError(err);

	return IsntPGPError(err);
}

	Boolean
CKeyTable::GetKeyOnServer(PGPKeyRef key)
{
	PGPKeySetRef	keySet;
	PGPError		err;

	err = PGPGetKeyFromServer(gPGPContext, gTLSContext, key, &keySet);
	if(IsPGPError(err) && (err != kPGPError_UserAbort))
		ReportPGPError(err);

	if(PGPKeySetRefIsValid(keySet))
	{
		err = PGPAddKeys(keySet, mKeySet);
		pgpAssertNoErr(err);
		CommitKeySet();
		err = PGPFreeKeySet(keySet);
		pgpAssertNoErr(err);
	}
	return IsntPGPError(err);
}

	Boolean
CKeyTable::GetCellOnServer(STableCell& inCell)
{
	TableIndexT		inWideOpenRow =
						mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell		woCell(inWideOpenRow, 1);
	KeyTableRef		tableRef;
	Uint32			dataSize = sizeof(KeyTableRef);
	PGPKeyID		keyID;
	PGPKeySetRef	keySet;
	PGPKeyRef		key;
	PGPError		err;
	
	keySet = NULL;
	GetCellData(woCell, &tableRef, dataSize);
	if(tableRef.type == kKey)
	{
		err = PGPGetKeyFromServer( gPGPContext, gTLSContext,
					tableRef.u.key, &keySet);
		if(IsPGPError(err) && (err != kPGPError_UserAbort))
			ReportPGPError(err);
	}
	else if(tableRef.type == kSignature)
	{
		err = PGPGetKeyIDOfCertifier(tableRef.u.sig,
										&keyID);
		pgpAssertNoErr(err);
		if(IsntPGPError(err) )
		{
			err = PGPGetSigCertifierKey(tableRef.u.sig,
										mKeySet,
										&key);
			// intentionally don't report error here
			// key may not be on the ring
			if( IsntPGPError( err ) )
			{
				err = PGPGetKeyFromServer( gPGPContext, gTLSContext,
							key, &keySet);
			}
			else
			{
				err = PGPGetKeyIDFromServer( gPGPContext, gTLSContext,
							&keyID, &keySet);
			}
			
			if(IsPGPError(err) && (err != kPGPError_UserAbort))
				ReportPGPError(err);
		}
	}
	if(PGPKeySetRefIsValid(keySet))
	{
		err = PGPAddKeys(keySet, mKeySet);
		pgpAssertNoErr(err);
		CommitKeySet();
		err = PGPFreeKeySet(keySet);
		pgpAssertNoErr(err);
	}
	return IsntPGPError(err);
}

	UInt32
CKeyTable::SelectionAction(Boolean (CKeyTable::*function)(STableCell&))
{	
	UInt32	cellsPerformedCount = 0;
	
	TableIndexT	numRows, numCols;
	GetTableSize(numRows, numCols);
	
	STableCell	topCell(1, 1);
	STableCell 	botCell(numRows, 1);
	STableCell	cell;
	
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		if(CellIsSelected(cell)) 
		{
			if((this->*function)(cell) == true)
				cellsPerformedCount++;
			else
				goto error;
		}
	}
error:
	return cellsPerformedCount;
}

	void
CKeyTable::ClickCell(
	const STableCell&		inCell,
	const SMouseDownEvent&	inMouseDown)
{
	if( GetClickCount() == 2 )
		SelectionAction(&CKeyTable::ShowKeyInfo);
	else if ( LDropArea::DragAndDropIsPresent() && 
						WaitMouseMoved( inMouseDown.macEvent.where ) )
	{
		LGWorld		*dragWorld = NULL;
		RgnHandle	dragImageRgn = NULL;
		RgnHandle	litCellRgn = NULL;
		
		mSendingDrag = TRUE;
		
		CKeyDragTask	drag(*this, inMouseDown.macEvent);
#ifdef	__POWERPC__
		OSErr		err;

		if( UEnvironment::HasGestaltAttribute(gestaltDragMgrAttr,
									gestaltDragMgrHasImageSupport ) )
		{
			StColorPenState	colorState;
			Rect			frame;
			Int16			depth;
			Rect			cellRect,
							zCellRect,
							zFrame;
			
			CalcLocalFrameRect( frame );
			GetLocalCellRect( inCell, cellRect );
			
			StDeviceLoop	devLoop(frame);		
			if(	devLoop.NextDepth( depth ) &&
				IsntNull( dragImageRgn = NewRgn() ) &&
				IsntNull( litCellRgn = NewRgn() ) )
			{
				// Get selected region in image (local) coords
				GetDragHiliteRgn( dragImageRgn, inCell, false );
				GetCellDragRgn( inCell, litCellRgn, false );
				OffsetRgn( litCellRgn, cellRect.left, cellRect.top );
				UnionRgn( litCellRgn, dragImageRgn, dragImageRgn );
				
				// Offset the offscreen rect to 0 for size efficiency
				zFrame = (**dragImageRgn).rgnBBox;
				OffsetRect( &zFrame, -zFrame.left, -zFrame.top );
				
				try
				{
					// Create the offscreen GWorld
					dragWorld = new LGWorld( zFrame, depth, useTempMem );
					
					// Safety check
					OutOfFocus( NULL );
					FocusDraw();
					
					// Draw the hit cell and frame the rest offscreen
					// Note that some of this may not even be visible,
					// so this must be done offscreen
					if( dragWorld->BeginDrawing() )
					{
						TableIndexT		woRow =
							mCollapsableTree->GetWideOpenIndex(inCell.row);
						STableCell		woCell(woRow, 1);
						Uint32			dataSize = sizeof(KeyTableRef);
						KeyTableRef		keyRef;
						PixMapHandle	dragImage;
						GrafPtr			thisPort;
						Point			offsetPt;
						
						GetPort( &thisPort );
						StColorPenState::Normalize();
						
						// Frame the non-hit cells
						offsetPt = topLeft( (**dragImageRgn).rgnBBox );
						OffsetRgn( dragImageRgn, -offsetPt.h, -offsetPt.v );
						PenSize( 2, 2 );
						FrameRgn( dragImageRgn );
						PenNormal();
						
						// Draw the hit cell
						TextFont( kFontIDGeneva );
						TextSize( 9 );
						TextFace( 0 );
						SetThemeTextColor( kThemeListViewTextColor, depth,
											( depth > 1 ) );
						zCellRect = cellRect;
						OffsetRect( &zCellRect, -offsetPt.h, -offsetPt.v );
						GetCellData( woCell, &keyRef, dataSize );
						DrawUserIDColumnCell( &keyRef, zCellRect, depth,
										true, false );
						dragWorld->EndDrawing();
						
						dragImage =
							GetGWorldPixMap( dragWorld->GetMacGWorld() );
						
						LocalToPortPoint( offsetPt );
						PortToGlobalPoint( offsetPt );
						err = SetDragImage( drag.GetDragReference(),
								dragImage, dragImageRgn,
								offsetPt, kDragStandardTranslucency );
					}
				}
				catch(...)
				{
					// Oh well, not enough memory for a ghosted drag
				}
			}
		}
#endif
		drag.DoDrag();
		if( IsntNull( dragWorld ) )
			delete dragWorld;
		if( IsntNull( dragImageRgn ) )
			DisposeRgn( dragImageRgn );
		if( IsntNull( litCellRgn ) )
			DisposeRgn( litCellRgn );
		if(UDragAndDropUtils::DroppedInTrash(drag.GetDragReference()) &&
			mMutableRing)
		{
			if(UDragAndDropUtils::CheckForOptionKey(drag.GetDragReference()))
				LCommander::GetTarget()->ProcessCommand(cmd_OptionClear);		
			else
				LCommander::GetTarget()->ProcessCommand(cmd_Clear);
		}
		
		mSendingDrag = FALSE;
	}
	else if(inCell.col == 1)
	{
		if( !IsTarget() )
			SwitchTarget( this );
		CPGPHierarchyTable::ClickCell( inCell, inMouseDown );
	}
}

	void
CKeyTable::ClickValidityCell(
	const STableCell&		inCell)
{
	STableCell	selCell;
	SPoint32	imagePt;
	Rect		cellRect;
	Point		curMouse;
	Boolean		pressed = FALSE;
	PGPBoolean	showMarginal,
				axiomatic;
	TableIndexT	woRow = mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell	woCell(woRow, 1);
	Uint32		dataSize = sizeof(KeyTableRef);
	KeyTableRef	keyRef;
	PGPError	err;
			
	GetCellData(woCell, &keyRef, dataSize);
	err = PGPGetPrefBoolean(gPrefRef,
		kPGPPrefDisplayMarginalValidity, &showMarginal);
	pgpAssertNoErr(err);
	if(!showMarginal && (keyRef.type != kSignature))
	{
		err = PGPGetKeyBoolean(keyRef.ownerKey, kPGPKeyPropIsAxiomatic,
							&axiomatic);
		pgpAssertNoErr(err);
		FocusDraw();
		GetLocalCellRect(inCell, cellRect);
		while(::StillDown())
		{
			::GetMouse(&curMouse);
			LocalToImagePoint(curMouse, imagePt);
			if(GetCellHitBy(imagePt, selCell) &&
				(selCell == inCell))
			{
				if(!pressed)
					DrawIcon(axiomatic? kPressedAxiomaticIconID : 
										kPressedValidityIconID,
										cellRect, -2);
				pressed = TRUE;
			}
			else if(pressed)
			{
				DrawCell(inCell, cellRect);
				pressed = FALSE;
			}
		}
		if(pressed)
		{
			DrawCell(inCell, cellRect);
			selCell = inCell;
			selCell.col = 1;
			if (!CellIsSelected(selCell)) {
				UnselectAllCells();
				SelectCell(selCell);
			}
			Sign();
		}
	}
}

	void
CKeyTable::ClickSelf(
	const SMouseDownEvent	&inMouseDown)
{
	STableCell	hitCell;
	SPoint32	imagePt;
	Point		curMouse;
	
	LocalToImagePoint(inMouseDown.whereLocal, imagePt);
	
	if (GetCellHitBy(imagePt, hitCell)) {
										// Click is inside hitCell
										// Check if click is inside DropFlag
		TableIndexT	woRow = mCollapsableTree->GetWideOpenIndex(hitCell.row);
		Rect	flagRect;
		CalcCellFlagRect(hitCell, flagRect);
		
		if( ( hitCell.col == 1 ) &&
			mCollapsableTree->IsCollapsable(woRow) &&
			::MacPtInRect(inMouseDown.whereLocal, &flagRect))
		{
			SInt16	depth;
			Boolean	expanded = mCollapsableTree->IsExpanded(woRow);
			
			FocusDraw();
			depth = (**((**GetMainDevice()).gdPMap)).pixelSize;
			SetThemeBackground( kThemeListViewSortColumnBackgroundBrush,
									depth, ( depth > 1 ) );
			if( UEnvironment::HasFeature(env_HasAppearance11) )
			{
				Rect 				trackRect = flagRect;
				Boolean 			drawnSelected = false;
				ThemeButtonDrawInfo	previousDrawInfo;
				ThemeButtonDrawInfo currentDrawInfo;

				MacInsetRect(&trackRect, 1, 1);
				
				currentDrawInfo.state		= kThemeStateActive;
				currentDrawInfo.value		= ( expanded ?	kThemeDisclosureDown :
															kThemeDisclosureRight );
				currentDrawInfo.adornment	= kThemeAdornmentDrawIndicatorOnly;
				previousDrawInfo			= currentDrawInfo;

				while( StillDown() )
				{
					Point	mouseLoc;
					
					GetMouse(&mouseLoc);				
					Boolean isInIcon = MacPtInRect( mouseLoc, &trackRect );
					previousDrawInfo.state = currentDrawInfo.state;

					if( isInIcon != drawnSelected )
					{
						if (isInIcon)
						{
							PlayThemeSound(kThemeSoundDisclosureEnter);
							currentDrawInfo.state = kThemeStatePressed;
						}
						else
						{
							PlayThemeSound(kThemeSoundDisclosureExit);
							currentDrawInfo.state = kThemeStateActive;
						}			
						drawnSelected = isInIcon;
						DrawThemeButton(	&flagRect,
										  	kThemeDisclosureButton,
										  	&currentDrawInfo,
										 	 &previousDrawInfo,
										 	 NULL, NULL, NULL );
					}
				}
				PlayThemeSound( kThemeSoundDisclosureRelease );

				EventRecord macEvent;
				OSEventAvail( mouseUp, &macEvent );
				GlobalToPortPoint( macEvent.where );
				PortToLocalPoint( macEvent.where );
				drawnSelected = MacPtInRect( macEvent.where, &trackRect );

				// If it ended up in the control, flip it.

				if( drawnSelected )
				{
					// Draw the icon flipped.

					//mFlipping = true;
					RefreshRect( flagRect );
					UpdatePort();
					
					// Wait 3 ticks.

					unsigned long ignore;
					::Delay(3, &ignore);
					
					// Change the expand status.
					
					//mFlipping = false;
					if( inMouseDown.macEvent.modifiers & optionKey )
					{
						if( expanded )
							DeepCollapseRow(woRow);
						else
							DeepExpandRow(woRow);
					}
					else
					{
						if( expanded )
							CollapseRow(woRow);
						else
							ExpandRow(woRow);
					}

					// Force a redraw.
					RefreshRect( flagRect );
					UpdatePort();
				}	
			}
			else
			{
				if (LDropFlag::TrackClick(flagRect, inMouseDown.whereLocal,
										expanded))
				{
					if( inMouseDown.macEvent.modifiers & optionKey )
					{
						if( expanded )
							DeepCollapseRow(woRow);
						else
							DeepExpandRow(woRow);
					}
					else
					{
						if( expanded )
							CollapseRow(woRow);
						else
							ExpandRow(woRow);
					}
				}
			}
		}
		else
		{
			if(hitCell.col == 1)
			{
				TableIndexT	woRow =
					mCollapsableTree->GetWideOpenIndex(hitCell.row);
				STableCell	woCell(woRow, 1);
				Uint32		dataSize = sizeof(KeyTableRef);
				KeyTableRef	keyRef;

				GetCellData(woCell, &keyRef, dataSize);
				if( inMouseDown.whereLocal.h >= keyRef.selectLeft &&
					inMouseDown.whereLocal.h <= keyRef.selectRight )
				{
					if(ClickSelect(hitCell, inMouseDown))
						ClickCell(hitCell, inMouseDown);
				}
				else
				{
					LMarqueeTask	marquee( *this, inMouseDown );
					
					if( !inMouseDown.macEvent.modifiers )
						UnselectAllCells();
					marquee.DoMarquee();
					if( !inMouseDown.macEvent.modifiers )
					{
						GetMouse( &curMouse );
						if( ( curMouse.h == inMouseDown.whereLocal.h ) &&
							( curMouse.v == inMouseDown.whereLocal.v ) )
							UnselectAllCells();
					}
				}
			}
			else
			{
				if(hitCell.col <= mColumnInfo.numActiveColumns &&
					mColumnInfo.columns[hitCell.col - 1] == kValidityColumnID)
				{
					ClickValidityCell(hitCell);
				}
				else
				{
					LMarqueeTask marquee( *this, inMouseDown );
					
					if( !inMouseDown.macEvent.modifiers )
						UnselectAllCells();
					marquee.DoMarquee();
					if( !inMouseDown.macEvent.modifiers )
					{
						GetMouse( &curMouse );
						if( ( curMouse.h == inMouseDown.whereLocal.h ) &&
							( curMouse.v == inMouseDown.whereLocal.v ) )
							UnselectAllCells();
					}
				}
			}
		}
	} else {							// Click is outside of any Cell
		UnselectAllCells();
	}
}

	const LView *
CKeyTable::GetTargetView() const
{
	return this;
}

	void
CKeyTable::FocusMarqueeArea()
{
	FocusDraw();
}

	Boolean
CKeyTable::CheckSelectionChanged(
	const LMarqueeTask& inMarqueeTask,
	const Rect&			inOldMarqueeRect,
	const Rect&			inNewMarqueeRect)
{
	STableCell			oldTR,
						oldBL,
						newTR,
						newBL;
	
	FetchIntersectingCells( inOldMarqueeRect, oldTR, oldBL );
	FetchIntersectingCells( inNewMarqueeRect, newTR, newBL );
	if( ( oldTR.row != newTR.row ) || ( oldBL.row != newBL.row ) ||
		( oldTR.col != newTR.col ) || ( oldBL.col != newBL.col ) )
		return TRUE;
	else
		return FALSE;
}

	void
CKeyTable::UpdateSelection(
	const LMarqueeTask& inMarqueeTask,
	const Rect&			inOldMarqueeRect,
	const Rect&			inNewMarqueeRect)
{
	STableCell			oldTR,
						oldBL,
						newTR,
						newBL,
						top,
						bottom,
						cell;
	Boolean				newMarquee = (	!inOldMarqueeRect.left &&
										!inOldMarqueeRect.right &&
										!inOldMarqueeRect.top &&
										!inOldMarqueeRect.bottom );
	
	FetchIntersectingCells( inOldMarqueeRect, oldTR, oldBL );
	FetchIntersectingCells( inNewMarqueeRect, newTR, newBL );
	
	
	if( !newMarquee && ( oldTR.row < newTR.row ) )
		top = oldTR;
	else
		top = newTR;
	if( !newMarquee && oldBL.row > newBL.row )
		bottom = oldBL;
	else
		bottom = newBL;
	cell.col = 1;
	for( cell.row = top.row; cell.row <= bottom.row; cell.row++ )
	{
		if( ( cell.row >= newTR.row ) && ( cell.row <= newBL.row ) )
		{
			SelectCell( cell );
		}
		else
		{
			UnselectCell( cell );
		}
	}
	Point 	localPoint;
	Rect	frame;

	inMarqueeTask.GetCurrentPoint(localPoint);
	CalcLocalFrameRect( frame );
	
	if( localPoint.v < frame.top )
	{
		ScrollPinnedImageBy( 0, -kRowHeight, TRUE );
	}
	else if( localPoint.v > frame.bottom )
	{
		ScrollPinnedImageBy( 0, kRowHeight, TRUE );
	}
	if( localPoint.h < frame.left )
	{
		ScrollPinnedImageBy( -8, 0, TRUE );
	}
	else if( localPoint.h > frame.right )
	{
		ScrollPinnedImageBy( 8, 0, TRUE );
	}
}

	void
CKeyTable::Click(
	SMouseDownEvent	&inMouseDown)
{
									// Check if a SubPane of this View
									//   is hit
	LPane	*clickedPane = FindSubPaneHitBy(inMouseDown.wherePort.h,
											inMouseDown.wherePort.v);
											
	if (clickedPane != nil) {		// SubPane is hit, let it respond to
									//   the Click
		clickedPane->Click(inMouseDown);
		
	} else {						// No SubPane hit. Inherited function
		PortToLocalPoint(inMouseDown.whereLocal);
		UpdateClickCount(inMouseDown);
		
		if (ExecuteAttachments(msg_Click, &inMouseDown)) {
			ClickSelf(inMouseDown);
		}
	}
}

	PGPBoolean
CKeyTable::GetSelectionStats(
	PGPUInt32		*numKeys,
	PGPUInt32		*numUserIDs,
	PGPUInt32		*numSigs )
{
	PGPError		err = kPGPError_NoErr;
	TableIndexT		numRows, numCols;
	
	GetTableSize(numRows, numCols);
	STableCell		topCell(1, 1);
	STableCell 		botCell(numRows, 1);
	STableCell		cell;				// Loop thru all cells

	*numKeys		= 0;
	*numUserIDs		= 0;
	*numSigs		= 0;
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		if(CellIsSelected(cell)) 
		{
			TableIndexT		inWideOpenRow =
							mCollapsableTree->GetWideOpenIndex(cell.row);
			STableCell		woCell(inWideOpenRow, 1);
			KeyTableRef		keyRef;
			Uint32			dataSize = sizeof(KeyTableRef);
			
			GetCellData(woCell, &keyRef, dataSize);
			switch( keyRef.type )
			{
				case kKey:
					++*numKeys;
					break;
				case kUserID:
					++*numUserIDs;
					break;
				case kSignature:
					++*numSigs;
					break;
			}
		}
	}
	if( *numKeys > 0 || *numUserIDs > 0 || *numSigs > 0 )
		return TRUE;
	return FALSE;
}

	void
CKeyTable::GetSelectedKeySet(
	PGPKeySetRef *selectedKeys )
{
	PGPError				err = kPGPError_NoErr;
	PGPKeySetRef			selKeys = kInvalidPGPKeySetRef,
							singleKeySet = kInvalidPGPKeySetRef;
	PGPUInt32				numKeys;
	TableIndexT				numRows, numCols;
		
	*selectedKeys = kInvalidPGPKeySetRef;
	GetTableSize(numRows, numCols);
	STableCell	topCell(1, 1);
	STableCell 	botCell(numRows, 1);
	STableCell	cell;				// Loop thru all cells

	err = PGPNewEmptyKeySet(mKeySet, &selKeys);
	if( IsPGPError( err ) )
		goto done;
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		if(CellIsSelected(cell)) 
		{
			TableIndexT		inWideOpenRow =
							mCollapsableTree->GetWideOpenIndex(cell.row);
			STableCell		woCell(inWideOpenRow, 1);
			KeyTableRef		keyRef;
			Uint32			dataSize = sizeof(KeyTableRef);
			
			GetCellData(woCell, &keyRef, dataSize);
			if( keyRef.type == kKey )
			{
				PGPKeySetRef	oldKeys = selKeys,
								singleKeySet;
				
				err = PGPNewSingletonKeySet(keyRef.ownerKey, &singleKeySet);
				if( IsPGPError( err ) )
					goto done;
				err = PGPUnionKeySets(oldKeys, singleKeySet, &selKeys);
				if( IsPGPError( err ) )
					goto done;
				PGPFreeKeySet( oldKeys );
				PGPFreeKeySet( singleKeySet );
				singleKeySet = kInvalidPGPKeySetRef;
			}
		}
	}
	PGPCountKeys( selKeys, &numKeys );
	if( !numKeys )
	{
		PGPFreeKeySet( selKeys );
		selKeys = kInvalidPGPKeySetRef;
	}
	else
		*selectedKeys = selKeys;
done:
	if( PGPKeySetRefIsValid( singleKeySet ) )
		PGPFreeKeySet( singleKeySet );
	if( IsPGPError( err ) && err != kPGPError_UserAbort )
		ReportPGPError( err );
}

	Boolean
CKeyTable::SelectionExists()
{
	if(mTableSelector->GetFirstSelectedRow() > 0)
		return TRUE;
	else
		return FALSE;
}

	void
CKeyTable::CollapseKeys(Boolean all)
{
	TableIndexT	rows,
				cols,
				rowIndex;
	STableCell	cell(1, 1);
	Boolean		selectedOnly;
	
	if(all)
		selectedOnly = FALSE;
	else
		selectedOnly = SelectionExists();
	GetWideOpenTableSize(rows, cols);
	for(rowIndex = rows;rowIndex > 0;rowIndex--)
	{
		cell.row = GetExposedIndex(rowIndex);
		if(!selectedOnly || CellIsSelected(cell))
			if(IsCollapsable(rowIndex))
				DeepCollapseRow(rowIndex);
	}
	cell.row = cell.col = 1;
	ScrollCellIntoFrame(cell);
	RedrawTable();
}

	void
CKeyTable::ExpandKeys()
{
	TableIndexT	rows,
				cols,
				rowIndex;
	STableCell	cell(1, 1);
	Boolean		selectedOnly;
	
	selectedOnly = SelectionExists();
	GetWideOpenTableSize(rows, cols);
	for(rowIndex = rows;rowIndex > 0;rowIndex--)
	{
		cell.row = GetExposedIndex(rowIndex);
		if(!selectedOnly || CellIsSelected(cell))
			if(IsCollapsable(rowIndex))
				DeepExpandRow(rowIndex);
	}
	RedrawTable();
}

	Boolean
CKeyTable::AskUserForUserID(Str255 userIDStr)
{
	CPGPStDialogHandler addUserDialog(dialog_AddUserID, CPGPKeys::TheApp());
	LWindow				*window;
	MessageT			dialogMessage;
	Str255				stringValue;
	
	window = addUserDialog.GetDialog();
tryAgain:
	do
	{
		dialogMessage = addUserDialog.DoDialog();
	} while( dialogMessage != msg_OK && dialogMessage != msg_Cancel );
	if(dialogMessage == msg_OK)
	{
		((LEditText *)window->FindPaneByID('eNam'))->
									GetDescriptor(stringValue);

		if (stringValue[ 0 ] == '\0')
		{
			CWarningAlert::Display(kWACautionAlertType, kWAOKStyle,
							kStringListID, kBadNameID);
			goto tryAgain;
		}
		else
			CopyPString(stringValue, userIDStr);

		((LEditText *)window->FindPaneByID('eEma'))->
									GetDescriptor(stringValue);
		if (stringValue[ 0 ] != '\0')
		{
			AppendPString("\p <", userIDStr);
			AppendPString(stringValue, userIDStr);
			AppendPString("\p>", userIDStr);
		}
		return TRUE;
	}
	return FALSE;
}

	Boolean
CKeyTable::AskUserForPhotoID(
	void 	**photoData,
	PGPSize	*photoDataSize)
{
	CPGPStDialogHandler addPhotoDialog(dialog_AddPhotoUserID,
										CPGPKeys::TheApp());
	LWindow				*window;
	MessageT			dialogMessage;
	CPhotoUserID		*photoIDPane;
	LPushButton			*okButton;
	Boolean				haveData = FALSE;
	
	window = addPhotoDialog.GetDialog();

	photoIDPane = (CPhotoUserID *) window->FindPaneByID( 'pPID' );
	pgpAssertAddrValid( photoIDPane, VoidAlign );
	
	okButton = (LPushButton *) window->FindPaneByID( 'okay' );
	pgpAssertAddrValid( okButton, VoidAlign );
	
	SwitchTarget( photoIDPane );
	
	do
	{
		if( photoIDPane->HavePicture() )
		{
			okButton->Enable();
		}
		else
		{
			okButton->Disable();
		}
		
		dialogMessage = addPhotoDialog.DoDialog();

		if( dialogMessage == 'File' )
		{
			StandardFileReply	sfReply;
			SFTypeList			typeList;
			
			typeList[0] = 'PICT';
			
			UDesktop::Deactivate();
				StandardGetFile( NULL, 1, typeList, &sfReply );
			UDesktop::Activate();
			
			if( sfReply.sfGood )
			{
				photoIDPane->AddPictureFromFile( &sfReply.sfFile );
			}
		}
		
	} while( dialogMessage != msg_OK && dialogMessage != msg_Cancel );
	
	if( dialogMessage == msg_OK )
	{
		photoIDPane->GetPhotoData( gPGPContext, photoData, photoDataSize );
		haveData = photoDataSize != 0;
	}
	
	return( haveData );
}

	void
CKeyTable::AddUserID(
	PGPBoolean addPhotoID )
{
	PGPError		err;
	STableCell		cell,
					wideCell;
	Uint32			dataSize = sizeof(KeyTableRef);
	KeyTableRef		keyRef;
	PGPBoolean		secret;
	Boolean			firstPass = TRUE;
	
	cell = GetFirstSelectedCell();
	wideCell.col = 1;
	wideCell.row = mCollapsableTree->GetWideOpenIndex(cell.row);
	GetCellData(wideCell, &keyRef, dataSize);
	secret = FALSE;
	if(keyRef.type == kKey)
	{
		err = PGPGetKeyBoolean( keyRef.u.key, kPGPKeyPropIsSecret, &secret );
		pgpAssertNoErr( err );
	}
	if( addPhotoID )
	{
		PGPUserIDRef	userID;
		PGPBoolean		isAttributeID;
		
		PGPKeyIterSeek( mKeyIter, keyRef.u.key );
		while( IsntPGPError( PGPKeyIterNextUserID( mKeyIter, &userID ) ) )
		{
			err = PGPGetUserIDBoolean( userID, kPGPUserIDPropIsAttribute,
						&isAttributeID );
			pgpAssertNoErr( err );
			
			if( isAttributeID )
			{
				PGPInt32	attributeType;
				
				err = PGPGetUserIDNumber( userID,
						kPGPUserIDPropAttributeType, &attributeType );
				
				if( attributeType == kPGPAttribute_Image )
				{
					CWarningAlert::Display(kWACautionAlertType, kWAOKStyle,
									kStringListID, kOnlyOnePhotoID);
					return;
				}
			}
		}
	}
	
	if( ( keyRef.type == kKey ) && secret )
	{
		PGPBoolean	ksSync;
		PGPBoolean	needsPass;
		Str255		userIDStr;
		void		*photoData;
		PGPSize		photoDataSize;
		
		needsPass 		= FALSE;
		userIDStr[0] 	= 0;
		photoData		= NULL;
		photoDataSize	= 0;
		
		err = PGPGetPrefBoolean(gPrefRef,
					kPGPPrefKeyServerSyncOnAdd, &ksSync);
		pgpAssertNoErr(err);
		if(ksSync)
		{
			GetCellOnServer(cell);
			ResyncTable(FALSE, TRUE);
		}
		
		if( addPhotoID )
		{
			needsPass = AskUserForPhotoID( &photoData, &photoDataSize );
		}
		else
		{
			needsPass = AskUserForUserID( userIDStr );
		}
		
		if( needsPass )
		{
			CSecureCString256	passphrase;
			PGPByte				*passKey = NULL;
			PGPSize				passKeySize;
			PGPBoolean			split = FALSE;
			
			passphrase[0] = 0;

			err = GetPassForKey( keyRef.u.key, &split, passphrase,
									&passKey, &passKeySize );
			if( IsntPGPError( err ) )
			{
				CPGPKeys::TheApp()->GetMinimumRandomData();
				
				if( addPhotoID )
				{
					err = PGPAddAttributeUserID( keyRef.u.key,
							kPGPAttribute_Image, (PGPByte *) photoData,
							photoDataSize, split ?
							PGPOPasskeyBuffer( gPGPContext, passKey,
							passKeySize ) :
							PGPOPassphrase( gPGPContext, passphrase),
							PGPOLastOption( gPGPContext ) );
				}
				else
				{
					char	cStringUsrID[ 256 ];

					PToCString( userIDStr, cStringUsrID );
					
					err = PGPAddUserID(keyRef.u.key, cStringUsrID, split ?
								PGPOPasskeyBuffer( gPGPContext, passKey,
								passKeySize ) :
								PGPOPassphrase( gPGPContext, passphrase),
								PGPOLastOption( gPGPContext ));
				}
				
				if(err == kPGPError_DuplicateUserID)
				{
					CWarningAlert::Display(kWACautionAlertType, kWAOKStyle,
									kStringListID, kDupeUserID);
				}
				else
					pgpAssert(!err);
				if(!err)
				{
					UnmarkKey(keyRef.u.key);
					RemoveKeyFromTable(keyRef.u.key);
					ResyncTable(FALSE, TRUE);
					CommitKeySet();
					pgpAssertNoErr(err);
				}
			}
			if( IsntNull( passKey ) )
				PGPFreeData( passKey );
			if(ksSync)
			{
				err = PGPSendKeyToServer(gPGPContext, gTLSContext,
							keyRef.u.key, NULL);
				if(IsPGPError(err) && (err != kPGPError_UserAbort))
					ReportPGPError(err);
			}
		}
		
		if( IsntNull( photoData ) )
			PGPFreeData( photoData );
	}
	else
		CWarningAlert::Display(kWACautionAlertType, kWAOKStyle,
						kStringListID, kAddNameOnlyPairsID);
}

	void
CKeyTable::AddRevoker()
{
	PGPError			err = kPGPError_NoErr;
	STableCell			cell,
						wideCell;
	Uint32				dataSize = sizeof(KeyTableRef);
	KeyTableRef			keyRef;
	PGPKeySetRef		revokerSet = kInvalidPGPKeySetRef;
	PGPFilterRef		filter = kInvalidPGPFilterRef;
	PGPKeySetRef		noRSAKeySet = kInvalidPGPKeySetRef;
	CSecureCString256	passphrase;
	PGPByte				*passKey = NULL;
	PGPSize				passKeySize;
	PGPBoolean			ksSync;
	
	cell = GetFirstSelectedCell();
	wideCell.col = 1;
	wideCell.row = mCollapsableTree->GetWideOpenIndex(cell.row);
	GetCellData(wideCell, &keyRef, dataSize);
	if(keyRef.type == kKey)
	{
		Str255			prompt;
		char			cstr[256];
		PGPUInt32		numRevokers;
		
		err = PGPGetPrefBoolean( gPrefRef,
					kPGPPrefKeyServerSyncOnAdd, &ksSync );
		pgpAssertNoErr(err);
		if( ksSync )
		{
			GetCellOnServer( cell );
			ResyncTable( FALSE, TRUE );
		}
		GetIndString( prompt, kStringListID, kAddRevokerID );
		PToCString( prompt, cstr );
		err = PGPNewKeySigAlgorithmFilter( gPGPContext,
											kPGPPublicKeyAlgorithm_DSA,
											&filter );	CKERR;
		err = PGPFilterKeySet( mKeySet, filter, &noRSAKeySet );	CKERR;
		UDesktop::Deactivate();
		err = PGPSelectKeysDialog( gPGPContext, kPGPSelectKeysDefault,
									cstr,
									noRSAKeySet,
									mKeySet, &revokerSet );
		UDesktop::Activate();	CKERR;
		PGPCountKeys( revokerSet, &numRevokers );
		
		if( numRevokers > 0 && ( msg_OK ==
			CWarningAlert::Display( kWACautionAlertType, kWAOKCancelStyle,
							kStringListID, kAddRevokerConfirmationID ) ) )
		{
			PGPBoolean	split;
			
			err = GetPassForKey( keyRef.u.key, &split, passphrase,
									&passKey, &passKeySize );	CKERR;
			err = PGPAddKeyOptions( keyRef.u.key, split ?
					PGPOPasskeyBuffer( gPGPContext, passKey, passKeySize ) :
					PGPOPassphrase( gPGPContext, passphrase ),
					PGPORevocationKeySet( gPGPContext, revokerSet ),
					PGPOLastOption( gPGPContext ) ); CKERR;
			if(ksSync)
			{
				err = PGPSendKeyToServer(gPGPContext, gTLSContext,
							keyRef.u.key, NULL);
				if( IsPGPError(err) && ( err != kPGPError_UserAbort ) )
					ReportPGPError(err);
			}
			CWarningAlert::Display( kWANoteAlertType, kWAOKStyle,
							kStringListID, kAddRevokerFinishedID );
		}
	}
done:
	if( PGPFilterRefIsValid( filter ) )
		PGPFreeFilter( filter );
	if( PGPKeySetRefIsValid( noRSAKeySet ) )
		PGPFreeKeySet( noRSAKeySet );
	if( PGPKeySetRefIsValid( revokerSet ) )
		PGPFreeKeySet( revokerSet );
	if( IsntNull( passKey ) )
		PGPFreeData( passKey );
	if( IsPGPError( err ) )
		ReportPGPError( err );
}

	Boolean
CKeyTable::SetDefault()
{
	PGPError		err;
	STableCell		cell;
	Uint32			dataSize = sizeof(KeyTableRef);
	KeyTableRef		keyRef;
	PGPBoolean		secret = FALSE;
	
	cell = GetFirstSelectedCell();
	cell.col = 1;
	cell.row = mCollapsableTree->GetWideOpenIndex(cell.row);
	GetCellData(cell, &keyRef, dataSize);
	if(keyRef.type == kKey)
	{
		err = PGPGetKeyBoolean(keyRef.u.key, kPGPKeyPropIsSecret, &secret);
		pgpAssertNoErr(err);
	}
	if((keyRef.type == kKey) && secret)
	{
		PGPSetDefaultPrivateKey(keyRef.u.key);
		mDefaultKey = keyRef.u.key;
		PGPsdkSavePrefs(gPGPContext);
		RedrawTable();
		return TRUE;
	}
	else if(keyRef.type == kUserID)
	{
		err = PGPSetPrimaryUserID(keyRef.u.user);
		pgpAssertNoErr(err);
		CommitKeySet();
		UnmarkKey(keyRef.ownerKey);
		RemoveKeyFromTable(keyRef.ownerKey);
		ResyncTable(FALSE, TRUE);
		RedrawTable();
		return TRUE;
	}
	else
		return FALSE;
}

	Boolean
CKeyTable::WeAreFront()
{
	ProcessSerialNumber	process;
	Boolean				weFront = FALSE;

	::GetFrontProcess(&process);
	::SameProcess(&process, &sThisProcess, &weFront);
	
	return weFront;
}

	void
CKeyTable::ImportHandler()
{
	PGPKeySetRef		newKeySet = kInvalidPGPKeySetRef,
						selectedKeySet = kInvalidPGPKeySetRef;
	PGPFileSpecRef		pgpFileRef;
	PGPUInt32			keyCount;
	Str255				str;
	PGPError			err;
	KeyImportStruct		*savedKis;
	Boolean				abort = FALSE;
	
	while(IsntNull(mImports))
	{
		if(mImports->savedImportIsFile)
		{
			err = PGPNewFileSpecFromFSSpec(gPGPContext,
									&mImports->savedImportSpec,
									&pgpFileRef);
			pgpAssertNoErr(err);
			if(IsntPGPError(err))
			{
				if( PStringHasSuffix( mImports->savedImportSpec.name,
					"\p.p12", FALSE ) ||
					PStringHasSuffix( mImports->savedImportSpec.name,
					"\p.pfx", FALSE ) )
				{
					char		*outputPassphrase;
					
					UDesktop::Deactivate();
					err = PGPPassphraseDialog( gPGPContext, 
						PGPOUIOutputPassphrase( gPGPContext, &outputPassphrase ),
						PGPOLastOption( gPGPContext ) );
					UDesktop::Activate();
					if( IsntPGPError( err ) )
					{
						err = PGPImportKeySet( gPGPContext, &newKeySet,
								PGPOInputFormat( gPGPContext, kPGPInputFormat_PKCS12 ),
								PGPOInputFile( gPGPContext, pgpFileRef ),
								PGPOPassphrase( gPGPContext, outputPassphrase ),
								PGPOLastOption( gPGPContext ) );
						PGPFreeData( outputPassphrase );
						ReportPGPError( err );
					}
					
				}
				else
				{
					err = PGPImportKeySet( gPGPContext, &newKeySet,
								PGPOInputFile( gPGPContext, pgpFileRef ),
								PGPOLastOption( gPGPContext ) );
				}
				PGPFreeFileSpec(pgpFileRef);
			}
		}
		else
		{
			err = PGPImportKeySet( gPGPContext, &newKeySet,
					PGPOInputBuffer( gPGPContext,
							mImports->savedImportBuffer,
							mImports->savedImportBufferSize ),
					PGPOLastOption( gPGPContext ) );
			pgpFree(mImports->savedImportBuffer);
		}
		if(IsntPGPError(err) && PGPKeySetRefIsValid(newKeySet))
		{
			err = PGPCountKeys(newKeySet, &keyCount);
			if(IsntPGPError(err) && keyCount > 0)
			{
				if(!mImports->bypassImportDialog)
				{
					char	cstring[128];
					
					::GetIndString(	str,
									kStringListID,
									kSpecifyKeysToImportID);
					PToCString(str, cstring);
					UDesktop::Deactivate();
					err = PGPSelectKeysDialog(gPGPContext,
											kPGPSelectKeysImportVariation,
											cstring, newKeySet, mKeySet,
											&selectedKeySet);
					UDesktop::Activate();
				}
				else
					selectedKeySet = newKeySet;
				if(IsntPGPError(err) && PGPKeySetRefIsValid(selectedKeySet))
				{
					err = PGPAddKeys(selectedKeySet, mKeySet);
					pgpAssertNoErr(err);
					CommitKeySet();
					if(!mImports->bypassImportDialog)
						PGPFreeKeySet(selectedKeySet);
				}
				else if(err == kPGPError_UserAbort)
				{
					abort = TRUE;
				}
				else
					ReportPGPError(err);
			}
			else
				CWarningAlert::Display(kWACautionAlertType, kWAOKStyle,
							kStringListID, kNoKeysFoundID);
			PGPFreeKeySet(newKeySet);
		}
		savedKis = mImports;
		mImports = mImports->next;
		pgpFree(savedKis);
		if(abort)
		{
			// user has aborted import, dispose all queued imports
			while(IsntNull(mImports))
			{
				if(!mImports->savedImportIsFile)
					pgpFree(mImports->savedImportBuffer);
				savedKis = mImports;
				mImports = mImports->next;
				pgpFree(savedKis);
			}
		}
	}
	ResyncTable(FALSE, TRUE);
	RedrawTable();
	if(mDefaultRing)
		CPGPKeys::TheApp()->InvalidateCaches();
}

	void
CKeyTable::SpendTime(const EventRecord		&/*inMacEvent*/)
{
	if(!WeAreFront())
		return;
	else if(mBlockDialogDelays > 0)
	{
		mBlockDialogDelays--;
		return;
	}
	if(mNotified)
	{
		::NMRemove(&sImportNotification);
		mNotified = FALSE;
	}
	StopIdling();
	ImportHandler();
}

	void
CKeyTable::ImportKeysFromFile(FSSpec *keyFile, Boolean noDialog)
{
	KeyImportStruct	*kis;
	
	pgpAssertAddrValid(keyFile, FSSpec);
	
	kis = (KeyImportStruct	*)pgpAlloc(sizeof(KeyImportStruct));
	if(IsntNull(kis))
	{
		kis->savedImportIsFile = TRUE;
		kis->bypassImportDialog = noDialog;
		::BlockMoveData(keyFile, &kis->savedImportSpec, sizeof(FSSpec));
		kis->next = mImports;
		mImports = kis;
		StartIdling();
		StepForward();
	}
}

	void
CKeyTable::ImportKeysFromHandle(Handle data)
{
	KeyImportStruct	*kis;
	
	if(IsntNull(data))
	{
		kis = (KeyImportStruct	*)pgpAlloc(sizeof(KeyImportStruct));
		if(IsntNull(kis))
		{
			long	importSize;
			
			importSize = GetHandleSize(data);
			kis->savedImportIsFile = FALSE;
			kis->bypassImportDialog = TRUE;
			kis->savedImportBuffer = (PGPByte *)pgpAlloc(importSize);
			if(IsntNull(kis->savedImportBuffer))
			{
				::BlockMoveData(*data, kis->savedImportBuffer, importSize);
				kis->savedImportBufferSize = importSize;
				kis->next = mImports;
				mImports = kis;
				ImportHandler();
			}
			else
				// +++++ out of memory
				pgpFree(kis);
		}
	}
}

	void
CKeyTable::ImportKeysFromKeySet(PGPKeySetRef keySet)
{
	if(PGPKeySetRefIsValid(keySet))
	{
		PGPError	err;
		
		err = PGPAddKeys(keySet, mKeySet);
		pgpAssertNoErr(err);
		CommitKeySet();
		PGPFreeKeySet(keySet);
		ResyncTable(FALSE, TRUE);
		RedrawTable();
		if(mDefaultRing)
			CPGPKeys::TheApp()->InvalidateCaches();
	}
}

	void
CKeyTable::CopyKeys()
{
	TableIndexT	numRows, numCols;
	GetTableSize(numRows, numCols);
	
	STableCell		topCell(1, 1);
	STableCell 		botCell(numRows, 1);
	STableCell		cell;
	
	PGPKeySetRef	keysToExport;
	PGPError		err;
	Int16			keyCount = 0;
	
	::ZeroScrap();
	err = PGPNewKeySet(gPGPContext, &keysToExport);
	pgpAssertNoErr(err);
	pgpAssertAddrValid(keysToExport, VoidAlign);
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		if(CellIsSelected(cell)) 
		{
			TableIndexT		inWideOpenRow = mCollapsableTree->
								GetWideOpenIndex(cell.row);
			STableCell		woCell(inWideOpenRow, cell.col);
			KeyTableRef		keyRef;
			Uint32			dataSize = sizeof(KeyTableRef);
			PGPKeySetRef	singleKeySet;
			
			GetCellData(woCell, &keyRef, dataSize);
			if(keyRef.type == kKey)
			{
				err = PGPNewSingletonKeySet(keyRef.u.key, &singleKeySet);
				pgpAssertNoErr(err);
				pgpAssertAddrValid(singleKeySet, VoidAlign);
				err = PGPAddKeys(singleKeySet, keysToExport);
				pgpAssertNoErr(err);
				err = PGPCommitKeyRingChanges(keysToExport);
				if(IsPGPError(err))
					ReportPGPError(err);
				PGPFreeKeySet(singleKeySet);
				keyCount++;
			}
		}
	}
	if((keyCount > 0) )
	{
		void			*keyBuffer;
		UInt32			keyBufferLen;
		char			comment[256];
		PGPBoolean		compat;
		
		comment[0] = 0;
#if PGP_BUSINESS_SECURITY
		PGPGetPrefStringBuffer(
				gAdminPrefRef,
				kPGPPrefComments,
				sizeof(comment), comment);
#endif	// PGP_BUSINESS_SECURITY
		if(comment[0] == 0)
			PGPGetPrefStringBuffer(
					gPrefRef,
					kPGPPrefComment,
					sizeof( comment ), comment);
		
		PGPGetPrefBoolean( gPrefRef, kPGPPrefExportKeysCompatible,
							&compat );
		
		/* let SDK export to a buffer it creates itself */
		err = PGPExportKeySet(keysToExport,
			PGPOAllocatedOutputBuffer( gPGPContext,
				&keyBuffer, MAX_PGPUInt32, &keyBufferLen ),
			PGPOVersionString( gPGPContext, pgpVersionHeaderString),
			PGPOCommentString( gPGPContext, comment),
			compat ?
				PGPOExportFormat( gPGPContext, kPGPExportFormat_Basic ) :
				PGPOExportFormat( gPGPContext, kPGPExportFormat_Complete ),
			PGPOLastOption( gPGPContext ));
		pgpAssert(!err);
		if( IsntPGPError( err ) )
		{
			::PutScrap(keyBufferLen, 'TEXT', keyBuffer);
			
			PGPFreeData( keyBuffer );
		}
	}
	PGPFreeKeySet(keysToExport);
}

	void
CKeyTable::ToggleColumn(Int16 columnID)
{
	Int16	columnIndex,
			oldColumn;
	Boolean	found = false;
	
	for(columnIndex = 0; columnIndex < mColumnInfo.numActiveColumns;
		columnIndex++)
	{
		if(mColumnInfo.columns[columnIndex] == columnID)
		{
			oldColumn = columnIndex;
			found = true;
			break;
		}
	}
	
	if(found)
	{
		found = false;
		for(columnIndex = 0; columnIndex < mColumnInfo.numActiveColumns;
			columnIndex++)
		{
			if(found)
				mColumnInfo.columns[columnIndex - 1] =
					mColumnInfo.columns[columnIndex];
			else if(mColumnInfo.columns[columnIndex] == columnID)
				found = true;
		}
		mColumnInfo.numActiveColumns --;
		RemoveCols(1, oldColumn + 1, false);
	}
	else
	{
		mColumnInfo.columns[mColumnInfo.numActiveColumns] =
			columnID;
		InsertCols(1, mColumnInfo.numActiveColumns, NULL, 0, false);
		mColumnInfo.numActiveColumns ++;
		mColumnInfo.columnWidths[columnID] =
				kDefaultColumnInfo.columnWidths[columnID];
		SetColWidth(mColumnInfo.columnWidths[columnID],
			mColumnInfo.numActiveColumns, mColumnInfo.numActiveColumns);
	}
	RedrawTable();
	mKeyLabels->AdjustLabelSizes();
	SaveColumnInfo();
}

typedef struct SFInfo
{
	StandardFileReply			sfReply;
	DialogPtr					dialog;
	Boolean						exportPrivate;
	Boolean						complete;
} SFInfo;

enum
{
	kExportPrivateCheckboxItem	= 13,
	kCompleteExportCheckboxItem	= 14
};

	static pascal short
ExportDialogHook(short item, DialogPtr dialog, SFInfo *sfInfo)
{
	pgpAssertAddrValid( dialog, DialogRecord );
	pgpAssertAddrValid( sfInfo, SFInfo );

	if( IsNull( sfInfo->dialog ) || ( sfInfo->dialog == dialog ) )
		switch( item )
		{
			case kExportPrivateCheckboxItem:
			{
				sfInfo->exportPrivate = ToggleDialogCheckbox( dialog,
											kExportPrivateCheckboxItem );
				item = sfHookNullEvent;
				break;
			}
			case kCompleteExportCheckboxItem:
			{
				sfInfo->complete = ToggleDialogCheckbox( dialog,
											kCompleteExportCheckboxItem );
				item = sfHookNullEvent;
				break;
			}
			case sfHookFirstCall:
				if( IsNull( sfInfo->dialog ) )
				{
					sfInfo->dialog = dialog;
					if( sfInfo->complete )
						ToggleDialogCheckbox( dialog,
												kCompleteExportCheckboxItem );
					item = sfHookNullEvent;
				}
				break;
		}
	return item;
}

	void
CKeyTable::ExportSelected()
{
	PGPError			err;
	TableIndexT			numRows, numCols;
	DlgHookYDUPP		cbHook;
	static Point		where={0,0};
	GetTableSize(numRows, numCols);
	
	STableCell	topCell(1, 1);
	STableCell 	botCell(numRows, 1);
	STableCell	cell;
	
	SFInfo				sfInfo;
	Str255				exportPrompt,
						exportFilename;
	PGPBoolean			boolPref;
	
	GetIndString(exportPrompt, kStringListID, kExportKeysPromptID);
	GetIndString(exportFilename, kStringListID, kExportKeysDefaultNameID);
	
	PGPGetPrefBoolean( gPrefRef, kPGPPrefExportKeysCompatible, &boolPref );
	sfInfo.complete			= !boolPref;
	sfInfo.exportPrivate	= FALSE;
	sfInfo.dialog			= NULL;
	
	cbHook 		= NewDlgHookYDProc( ExportDialogHook );

	UDesktop::Deactivate();
	CustomPutFile(exportPrompt, exportFilename, &sfInfo.sfReply,
					kExportCustomPutDialog, where, cbHook, NULL, NULL,
					NULL, &sfInfo);
	UDesktop::Activate();

	DisposeRoutineDescriptor( cbHook );
	
	
	if(sfInfo.sfReply.sfGood)
	{
		PGPFileSpecRef		pgpFileRef = kInvalidPGPFileSpecRef;
		PGPKeySetRef		keysToExport = kInvalidPGPKeySetRef;
		Int32				keysExported = 0;
		
		FSpDelete( &sfInfo.sfReply.sfFile );
		err = PGPNewFileSpecFromFSSpec(gPGPContext, &sfInfo.sfReply.sfFile,
										&pgpFileRef);
		pgpAssertNoErr(err);
		if(PGPFileSpecRefIsValid(pgpFileRef))
		{
			err = PGPNewEmptyKeySet(mKeySet, &keysToExport);
			pgpAssertNoErr(err);
			if(PGPKeySetRefIsValid(keysToExport))
			{
				for(cell = botCell; cell.row >= topCell.row; cell.row--) 
				{
					if(CellIsSelected(cell)) 
					{
						TableIndexT		inWideOpenRow =
											GetWideOpenIndex(cell.row);
						STableCell		woCell(inWideOpenRow, 1);
						KeyTableRef		keyRef;
						Uint32			dataSize = sizeof(KeyTableRef);
						PGPKeySetRef	oldKeys = keysToExport;
						PGPKeySetRef	oneKey;
						
						GetCellData(woCell, &keyRef, dataSize);
						err = PGPNewSingletonKeySet(keyRef.ownerKey, &oneKey);
						pgpAssertNoErr(err);
						err = PGPUnionKeySets(oldKeys, oneKey, &keysToExport);
						pgpAssertNoErr(err);
						PGPFreeKeySet(oldKeys);
						PGPFreeKeySet(oneKey);
						keysExported++;
					}
				}
				if(keysExported > 0)
				{
					char comment[256];
					
					comment[0] = 0;
	#if PGP_BUSINESS_SECURITY
					PGPGetPrefStringBuffer(
							gAdminPrefRef,
							kPGPPrefComments,
							sizeof(comment), comment);
	#endif	// PGP_BUSINESS_SECURITY
					if(comment[0] == 0)
						PGPGetPrefStringBuffer(
								gPrefRef,
								kPGPPrefComment,
								sizeof( comment ), comment);
					err = PGPExportKeySet( keysToExport,
						PGPOOutputFile(gPGPContext, pgpFileRef),
						PGPOExportPrivateKeys(gPGPContext,
											sfInfo.exportPrivate),
						PGPOVersionString(gPGPContext,
											pgpVersionHeaderString),
						PGPOCommentString(gPGPContext, comment),
						sfInfo.complete ?
							PGPOExportFormat(gPGPContext,
								kPGPExportFormat_Complete ) :
							PGPOExportFormat( gPGPContext,
								kPGPExportFormat_Basic ),
						PGPOLastOption(gPGPContext));
					pgpAssertNoErr(err);
				}
				else
					CWarningAlert::Display(kWANoteAlertType, kWAOKStyle,
									kStringListID, kNoKeyToExportID);
				PGPFreeKeySet(keysToExport);
			}
			PGPFreeFileSpec(pgpFileRef);
		}
	}
}

	void
CKeyTable::BuildServerSubmenu()
{
	MenuHandle			serverMenu;
	Int16				numItems;
	PGPKeyServerEntry	*ksEntries;
	PGPUInt32			ksCount;
	PGPError			err;
	
	serverMenu = ::GetMenuHandle(kSendToServerMENUID);
	if(IsntNull(serverMenu))
	{
		numItems = ::CountMItems(serverMenu);
		while(numItems > 2)
			::DeleteMenuItem(serverMenu, numItems--);
		err = PGPGetKeyServerPrefs(gPrefRef, &ksEntries, &ksCount);
		pgpAssertNoErr(err);
		if(IsntPGPError(err))
		{
			for(Int16 ksInx = 0; ksInx < ksCount; ksInx++)
			{
				if(IsKeyServerListed(ksEntries[ksInx].flags))
				{
					Str255	menuStr;
					char	url[kMaxServerNameLength + 1];
					
					PGPGetKeyServerURL( &ksEntries[ksInx], url );
					CToPString( url, menuStr );
					::InsertMenuItem(serverMenu, "\pA", 0x6FFF);
					::SetMenuItemText(serverMenu, ++numItems, menuStr);
				}
			}
			PGPDisposePrefData(gPrefRef, ksEntries);
		}
	}
}

	PGPError
CKeyTable::GetMetaIntroducers(
	PGPKeySetRef		*metaKeys )
{
	PGPError			err = kPGPError_NoErr;
	PGPBoolean			noMargValid;
	PGPUInt32			validThreshold;
	PGPKeySetRef		metaSet = kInvalidPGPKeySetRef,
						singleSet = kInvalidPGPKeySetRef;
	PGPKeyListRef		keyList = kInvalidPGPKeyListRef;
	PGPKeyIterRef		keyIter = kInvalidPGPKeyIterRef;
	PGPKeyRef			key;
	
	*metaKeys = NULL;
	PGPGetPrefBoolean( gPrefRef, kPGPPrefMarginalIsInvalid, &noMargValid );
	if( noMargValid )
		validThreshold = kPGPValidity_Complete;
	else
		validThreshold = kPGPValidity_Marginal;
	err = PGPNewKeySet( gPGPContext, &metaSet );	CKERR;
	err = PGPOrderKeySet( mKeySet, kPGPAnyOrdering, &keyList ); CKERR;
	err = PGPNewKeyIter( keyList, &keyIter ); CKERR;
	
	while(	IsntPGPError( PGPKeyIterNext( keyIter, &key ) ) &&
			PGPKeyRefIsValid( key ) )
	{
		PGPBoolean		expired,
						revoked,
						disabled;
		PGPValidity		validity;
		PGPUserIDRef	userID;
		
		PGPGetKeyBoolean( key, kPGPKeyPropIsExpired, &expired );
		PGPGetKeyBoolean( key, kPGPKeyPropIsRevoked, &revoked );
		PGPGetKeyBoolean( key, kPGPKeyPropIsDisabled, &disabled );
		PGPGetPrimaryUserIDValidity( key, &validity );
		
		if( !expired && !revoked && !disabled && validity >= validThreshold )
		{
			while(	IsntPGPError(
					PGPKeyIterNextUserID( keyIter, &userID ) ) &&
					PGPUserIDRefIsValid( userID ) )
			{
				PGPSigRef	sig;
				
				while( IsntPGPError(
						PGPKeyIterNextUIDSig( keyIter, &sig ) ) &&
						PGPSigRefIsValid( sig ) )
				{
					PGPBoolean	exportable;
					PGPInt32	sigTrust;
					
					PGPGetSigBoolean( sig, kPGPSigPropIsRevoked, &revoked );
					PGPGetSigBoolean( sig, kPGPSigPropIsExportable,
									&exportable );
					PGPGetSigNumber( sig, kPGPSigPropTrustLevel, &sigTrust );

					if ( !revoked && !exportable && ( sigTrust == 2 ) )
					{
						err = PGPNewSingletonKeySet( key, &singleSet ); CKERR;
						err = PGPAddKeys( singleSet, metaSet ); CKERR;
						err = PGPCommitKeyRingChanges( metaSet ); CKERR;
						PGPFreeKeySet( singleSet );
						singleSet = kInvalidPGPKeySetRef;
					}
				}
			}
		}
	}
	*metaKeys = metaSet;
	metaSet = kInvalidPGPKeySetRef;
done:
	if( PGPKeySetRefIsValid( metaSet ) )
		PGPFreeKeySet( metaSet );
	if( PGPKeySetRefIsValid( singleSet ) )
		PGPFreeKeySet( singleSet );
	if( PGPKeyListRefIsValid( keyList ) )
		PGPFreeKeyList( keyList );
	if( PGPKeyIterRefIsValid( keyIter ) )
		PGPFreeKeyIter( keyIter );
	return err;
}

	void
CKeyTable::UpdateIntroducers()
{
	PGPError						err = kPGPError_NoErr;
	PGPKeySetRef					metaSet		= kInvalidPGPKeySetRef,
									foundSet	= kInvalidPGPKeySetRef,
									singleKey	= kInvalidPGPKeySetRef;
	PGPKeyListRef					keyList		= kInvalidPGPKeyListRef,
									signedList	= kInvalidPGPKeyListRef;
	PGPKeyIterRef					keyIter		= kInvalidPGPKeyIterRef,
									signedIter	= kInvalidPGPKeyIterRef;
	PGPFilterRef					filter		= kInvalidPGPFilterRef;
	PGPKeyServerRef					rootRef		= kInvalidPGPKeyServerRef;
	PGPKeyRef						key;
	PGPKeyServerEntry				rootServer;
	PGPKeyServerSpec				ksSpec;
	PGPUInt32						keyCount;
	StPGPPreserveKeyServerStorage	keyserverStorage;

	
	err = PGPGetRootKeyServer( gPrefRef, &rootServer );
	if( err == kPGPError_ItemNotFound )
	{
		CWarningAlert::Display( kWACautionAlertType, kWAOKStyle,
						kStringListID, kNoRootServerID );
		err = kPGPError_NoErr;
		goto done;
	}
	CKERR;
	err = PGPNewKeyServerFromHostName( gPGPContext,
				rootServer.serverDNS,
				rootServer.serverPort,
				rootServer.protocol,
				kPGPKeyServerAccessType_Normal,
				kPGPKeyServerKeySpace_Normal,
				&rootRef );	CKERR;
	pgpClearMemory( &ksSpec, sizeof(PGPKeyServerSpec) );
	ksSpec.server		= rootRef;
	
	err = GetMetaIntroducers( &metaSet ); CKERR;
	
	err = PGPCountKeys( metaSet, &keyCount );	CKERR;
	// Update each meta-introducer from the appropriate server
	err = PGPOrderKeySet( metaSet, kPGPAnyOrdering, &keyList ); CKERR;
	err = PGPNewKeyIter( keyList, &keyIter ); CKERR;
	while(	IsntPGPError( PGPKeyIterNext( keyIter, &key ) ) &&
			PGPKeyRefIsValid( key ) )
	{
		err = PGPSearchKeyServerDialog( gPGPContext,
				1, &ksSpec, gTLSContext, FALSE, &foundSet,
				PGPOUIKeyServerSearchKey( gPGPContext, key ),
				PGPOLastOption( gPGPContext ) );
		if( err == kPGPError_UserAbort )
		{
			err = kPGPError_NoErr;
			goto done;
		}
		if( IsPGPError( err ) )
			ReportPGPError( err );
		if( PGPKeySetRefIsValid( foundSet ) )
		{
			err = PGPAddKeys( foundSet, mKeySet );	CKERR;
			CommitKeySet();
			PGPFreeKeySet( foundSet );
			foundSet = kInvalidPGPKeySetRef;
		}
	}
	CommitKeySet();
	
	// Find all the trust==1 keys signed by each meta-introducer
	err = PGPKeyIterRewind( keyIter );	CKERR;
	while(	IsntPGPError( PGPKeyIterNext( keyIter, &key ) ) &&
			PGPKeyRefIsValid( key ) )
	{
		PGPKeyID			keyIDMeta;
		
		err = PGPGetKeyIDFromKey( key, &keyIDMeta );	CKERR;
		err = PGPNewSigKeyIDFilter( gPGPContext, &keyIDMeta, &filter ); CKERR;
		
		err = PGPSearchKeyServerDialog( gPGPContext, 1, &ksSpec, gTLSContext,
						FALSE, &foundSet,
						PGPOUIKeyServerSearchFilter( gPGPContext, filter ),
						PGPOLastOption( gPGPContext ) );
		if( err == kPGPError_UserAbort )
		{
			err = kPGPError_NoErr;
			goto done;
		}
		if( IsPGPError( err ) )
			ReportPGPError( err );
		PGPFreeFilter( filter );
		filter = kInvalidPGPFilterRef;
		if( PGPKeySetRefIsValid( foundSet ) )
		{
			err = PGPOrderKeySet( foundSet, kPGPAnyOrdering, &signedList );
			CKERR;
			err = PGPNewKeyIter( signedList, &signedIter ); CKERR;
			
			while(	IsntPGPError( PGPKeyIterNext( signedIter, &key ) ) &&
					PGPKeyRefIsValid( key ) )
			{
				PGPUserIDRef	userID;
				Boolean			metaSigned = FALSE;
				
				while(	!metaSigned && IsntPGPError(
						PGPKeyIterNextUserID( signedIter, &userID ) ) &&
						PGPUserIDRefIsValid( userID ) )
				{
					PGPSigRef	sig;
					
					while( IsntPGPError(
							PGPKeyIterNextUIDSig( keyIter, &sig ) ) &&
							PGPSigRefIsValid( sig ) )
					{
						PGPKeyID	signerID;
						PGPBoolean	exportable;
						PGPInt32	sigTrust;
						
						err = PGPGetKeyIDOfCertifier( sig, &signerID ); CKERR;
						err = PGPGetSigBoolean( sig, kPGPSigPropIsExportable,
												&exportable );	CKERR;
						err = PGPGetSigNumber( sig, kPGPSigPropTrustLevel,
												&sigTrust );	CKERR;
						
						if( exportable && ( sigTrust == 1 ) &&
							!PGPCompareKeyIDs( &keyIDMeta, &signerID ) )
						{
							metaSigned = TRUE;
							err = PGPNewSingletonKeySet( key, &singleKey );
							CKERR;
							err = PGPAddKeys( singleKey, mKeySet );	CKERR;
							PGPFreeKeySet( singleKey );
							singleKey = kInvalidPGPKeySetRef;
							break;
						}
					}
				}
			}
			CommitKeySet();
			
			PGPFreeKeyList( signedList );
			signedList = kInvalidPGPKeyListRef;
			PGPFreeKeyIter( signedIter );
			signedIter = kInvalidPGPKeyIterRef;
			PGPFreeKeySet( foundSet );
			foundSet = kInvalidPGPKeySetRef;
		}
	}
done:
	if( PGPKeySetRefIsValid( metaSet ) )
		PGPFreeKeySet( metaSet );
	if( PGPKeySetRefIsValid( foundSet ) )
		PGPFreeKeySet( foundSet );
	if( PGPKeySetRefIsValid( singleKey ) )
		PGPFreeKeySet( singleKey );
	if( PGPKeyListRefIsValid( keyList ) )
		PGPFreeKeyList( keyList );
	if( PGPKeyIterRefIsValid( keyIter ) )
		PGPFreeKeyIter( keyIter );
	if( PGPKeyListRefIsValid( signedList ) )
		PGPFreeKeyList( signedList );
	if( PGPKeyIterRefIsValid( signedIter ) )
		PGPFreeKeyIter( signedIter );
	if( PGPFilterRefIsValid( filter ) )
		PGPFreeFilter( filter );
	if( PGPKeyServerRefIsValid( rootRef ) )
		PGPFreeKeyServer( rootRef );
	if( IsPGPError( err ) )
		ReportPGPError( err );
}

	Boolean
CKeyTable::ObeyCommand(
	CommandT	inCommand,
	void		*ioParam)
{
	Boolean		cmdHandled = true;
	ResIDT		theMenuID;
	Int16		theMenuItem;
	PGPError	err;

	if(IsSyntheticCommand(inCommand, theMenuID, theMenuItem))
	{
		switch(theMenuID)
		{
			case kSendToServerMENUID:
				if(gServerCallsPresent)
				{
					PGPKeyServerEntry	*ksEntries;
					PGPUInt32			ksCount;
					
					err = PGPGetKeyServerPrefs(gPrefRef, &ksEntries,
												&ksCount);
					pgpAssertNoErr(err);
					if(IsntPGPError(err))
					{
						theMenuItem -= 3;
						if(theMenuItem < ksCount)
						{
							UInt32	successfulSends;
							Str255	numSendsString;
							
							mTargetKeyServer = &ksEntries[theMenuItem];
							successfulSends = 
								SelectionAction(&CKeyTable::SendKeyToServer);
							if(successfulSends > 0)
							{
								NumToString(successfulSends, numSendsString);
								CWarningAlert::Display(kWANoteAlertType,
										kWAOKStyle, kStringListID,
										kServerSendSuccessful,
										numSendsString);
							}
							mTargetKeyServer = NULL;
						}
						PGPDisposePrefData(gPrefRef, ksEntries);
					}
				}
				break;
			default:
				cmdHandled = LCommander::ObeyCommand(inCommand, ioParam);
				break;
		}
	}
	else switch (inCommand) 
	{
		case cmd_ViewValidity:
			ToggleColumn(kValidityColumnID);
			break;
		case cmd_ViewTrust:
			ToggleColumn(kTrustColumnID);
			break;
		case cmd_ViewSize:
			ToggleColumn(kKeySizeColumnID);
			break;
		case cmd_ViewKeyID:
			ToggleColumn(kKeyIDColumnID);
			break;
		case cmd_ViewCreationDate:
			ToggleColumn(kCreationColumnID);
			break;
		case cmd_ViewExpirationDate:
			ToggleColumn(kExpirationColumnID);
			break;
		case cmd_ViewMRK:
			ToggleColumn(kMRKColumnID);
			break;
		case cmd_ViewDescription:
			ToggleColumn(kDescriptionColumnID);
			break;
		case cmd_Copy:
			CopyKeys();
			break;
		case cmd_Paste:
		{
			Handle	pastedTextH;
			Int32	offset,
					length;
			
			length = ::GetScrap(nil, 'TEXT', &offset);
			pastedTextH = ::NewHandle(length);
			HNoPurge(pastedTextH);
			if((length = ::GetScrap(pastedTextH, 'TEXT', &offset)) > 0)
			{
				PGPKeySetRef	newKeySet = kInvalidPGPKeySetRef;
				char *			pastedChars;
				PGPBoolean		isPGP = FALSE;
				char			*t, *fe;
				
				HLock(pastedTextH);
				pastedChars = *(char **)pastedTextH;
				{
					char		pgpStr[] = "-----BEGIN";
					char		*s;
					
					s = pgpStr;
					t = pastedChars;
					fe = pastedChars + length;
					while( t < fe )
					{
						if( *t == *s )
							s++;
						else
							s = pgpStr;
						t++;
						if( *s == '\0' )
						{
							if( ( *t == ' ' ) &&
								( *(t+1) == 'P' ) &&
								( *(t+2) == 'G' ) &&
								( *(t+3) == 'P' ) )
								isPGP = TRUE;
							t -= strlen(pgpStr);
							break;
						}
					}
				}
				if( fe - t )
					err = PGPImportKeySet( gPGPContext, &newKeySet,
						PGPOInputBuffer( gPGPContext, t, fe - t ),
						PGPOInputFormat( gPGPContext,
							isPGP ? kPGPInputFormat_PGP :
									kPGPInputFormat_PEMEncodedX509Cert ),
						PGPOLastOption( gPGPContext ) );
				
				if(IsntPGPError(err) && IsntNull(newKeySet))
				{
					PGPKeySetRef	finalKeySet = kInvalidPGPKeySetRef;
				
					err = PGPSelectKeysDialog( gPGPContext,
							kPGPSelectKeysImportVariation,
							"", newKeySet, mKeySet, &finalKeySet );
					if( IsntPGPError( err ) )
					{
						err = PGPAddKeys( finalKeySet, mKeySet );
						pgpAssertNoErr(err);
						CommitKeySet();
					}
					PGPFreeKeySet(newKeySet);
					if( PGPKeySetRefIsValid( finalKeySet ) )
						PGPFreeKeySet( finalKeySet );
					ResyncTable(FALSE, TRUE);
					if(mDefaultRing)
						CPGPKeys::TheApp()->InvalidateCaches();
				}
			}
			::DisposeHandle(pastedTextH);
			break;
		}
		case cmd_NewKey:
		{
			CKeyGenWizardDialog *kgw;
			
			kgw = (CKeyGenWizardDialog *)
				LWindow::CreateWindow(dialog_KeyGenWizard,
										CPGPKeys::TheApp());
			kgw->SetKeyTable(this);
			break;
		}
		case cmd_AddCertificate:
		case cmd_RetrieveCertificate:
			if( inCommand == cmd_AddCertificate )
				mCAOp = kPGPCA_OP_RequestCert;
			else
				mCAOp = kPGPCA_OP_RetrieveCert;
			SelectionAction(&CKeyTable::CAServerOp);
			ResyncTable(FALSE, TRUE);
			RedrawTable();
			CommitKeySet();
			break;
		case cmd_UpdateRevocations:
		{
			STableCell cell(0,0);
			mCAOp = kPGPCA_OP_UpdateCRL;
			CAServerOp(cell);
			break;
		}
		case cmd_OptionClear:
			SelectionAction(&CKeyTable::Delete);
			RemoveTerminatedKeys();
			ResyncTable(FALSE, TRUE);
			RedrawTable();
			CommitKeySet();
			break;
		case cmd_Clear:
			if(CWarningAlert::Display(kWACautionAlertType,
							kWAOKCancelStyle, kStringListID,
							kClearQuestion) == msg_OK)
			{
				SelectionAction(&CKeyTable::Delete);
				RemoveTerminatedKeys();
				ResyncTable(FALSE, TRUE);
				RedrawTable();
				CommitKeySet();
			}
			break;
		case cmd_Revoke:
			SelectionAction(&CKeyTable::Revoke);
			ResyncTable(FALSE, TRUE);
			RedrawTable();
			CommitKeySet();
			break;
		case cmd_SplitKey:
			SelectionAction(&CKeyTable::Split);
			break;
		case cmd_CollapseAll:
			CollapseKeys(FALSE);
			break;
		case cmd_ExpandAll:
			ExpandKeys();
			break;
		case cmd_SelectAll:
			UnselectAllCells();
			SelectAllCells();
			break;
		case cmd_SetDefault:
			if(!SetDefault())
				CWarningAlert::Display(kWACautionAlertType, kWAOKStyle,
								kStringListID, kCantDefaultStringID);
			break;
		case cmd_Info:
			SelectionAction(&CKeyTable::ShowKeyInfo);
			break;
		case cmd_Sign1:
			Sign();
			break;
		case cmd_EnableKeys:
			SelectionAction(&CKeyTable::EnableKey);
			RedrawTable();
			CommitKeySet();
			break;
		case cmd_DisableKeys:
			SelectionAction(&CKeyTable::DisableKey);
			RedrawTable();
			CommitKeySet();
			break;
		case cmd_AddName:
		case cmd_AddPhotoUserID:
			AddUserID( inCommand == cmd_AddPhotoUserID );
			RedrawTable();
			break;
		case cmd_AddRevoker:
			AddRevoker();
			break;
		case cmd_ReverifySignatures:
			ReverifySigs();
			break;
		case cmd_KSGetKeys:
			if(gServerCallsPresent)
			{
				if(SelectionAction(&CKeyTable::GetCellOnServer))
				{
					ResyncTable(FALSE, TRUE);
					RedrawTable();
				}
			}
			break;
		case cmd_KSSendKeyToDomain:
			if(gServerCallsPresent)
			{
				UInt32	successfulSends;
				Str255	numSendsString;
				
				mTargetKeyServer = NULL;
				successfulSends = 
					SelectionAction(&CKeyTable::SendKeyToServer);
				if(successfulSends > 0)
				{
					NumToString(successfulSends, numSendsString);
					CWarningAlert::Display(kWANoteAlertType,
							kWAOKStyle, kStringListID,
							kServerSendSuccessful,
							numSendsString);
				}
			}
			break;
		case cmd_ImportKeys:
			{
				FSSpec				fsSpec;
				SFTypeList			typeList;
				
				typeList[0] = kPGPMacFileType_ArmorFile;
				typeList[1] = kPGPMacFileType_PrivRing;
				typeList[2] = kPGPMacFileType_PubRing;
				
				if( CustomGetFileWithShowAll( 3, typeList, &fsSpec ) )
					ImportKeysFromFile( &fsSpec, FALSE );
			}
			break;
		case cmd_ExportKeys:
			ExportSelected();
			break;
		case cmd_UpdateIntroducers:
			UpdateIntroducers();
			break;
		default:
			cmdHandled = LCommander::ObeyCommand(inCommand, ioParam);
			break;
	}
	
	return cmdHandled;
}

	void
CKeyTable::GetColumnCheck(	Int16		columnID,
							Boolean		&outEnabled,
							Boolean		&outUsesMark,
							Char16		&outMark)
{
	if(ColumnVisible(columnID))
	{
		outUsesMark = true;
		outMark = checkMark;
	}
	else
	{
		outUsesMark = true;
		outMark = ' ';
	}
	outEnabled = true;
}

	void
CKeyTable::FindCommandStatus(
	CommandT	inCommand,
	Boolean		&outEnabled,
	Boolean		&outUsesMark,
	Char16		&outMark,
	Str255		outName)
{
	PGPError	err;
	PGPBoolean	canSign;
				
	switch(inCommand) 
	{
		case cmd_ViewValidity:
			GetColumnCheck(kValidityColumnID,
							outEnabled, outUsesMark, outMark);
			break;
		case cmd_ViewTrust:
			GetColumnCheck(kTrustColumnID,
							outEnabled, outUsesMark, outMark);
			break;
		case cmd_ViewSize:
			GetColumnCheck(kKeySizeColumnID,
							outEnabled, outUsesMark, outMark);
			break;
		case cmd_ViewKeyID:
			GetColumnCheck(kKeyIDColumnID, outEnabled, outUsesMark, outMark);
			break;
		case cmd_ViewCreationDate:
			GetColumnCheck(kCreationColumnID,
							outEnabled, outUsesMark, outMark);
			break;
		case cmd_ViewExpirationDate:
			GetColumnCheck(kExpirationColumnID,
							outEnabled, outUsesMark, outMark);
			break;
		case cmd_ViewMRK:
			GetColumnCheck(kMRKColumnID, outEnabled, outUsesMark, outMark);
			break;
		case cmd_ViewDescription:
			GetColumnCheck(kDescriptionColumnID,
							outEnabled, outUsesMark, outMark);
			break;
		case cmd_Paste:
			{	// Check if TEXT is in the Scrap
				Int32	offset;
				outEnabled = (::GetScrap(nil, 'TEXT', &offset) > 0);
			}
			break;
		case cmd_KSSendKeys:
		case cmd_KSSendKeyToDomain:
		case cmd_KSGetKeys:
			if(SelectionExists() && gServerCallsPresent)
				outEnabled = TRUE;
			break;
		case cmd_AddCertificate:
		case cmd_RetrieveCertificate:
#if	PGP_BUSINESS_SECURITY
			PGPBoolean		allowCRS;
			
			err = PGPGetPrefBoolean(gAdminPrefRef,
					kPGPPrefAllowManualX509CertRequest, &allowCRS);
			pgpAssertNoErr(err);
			if( !allowCRS )
				break;
#endif
			{
				char serverURL[256];
				
				err = PGPGetPrefStringBuffer( gPrefRef, kPGPPrefCAServerURL,
							sizeof(serverURL), serverURL );
				if( IsPGPError(err) || ( serverURL[0] < 1 ) )
					break;
			}
		case cmd_SplitKey:
		case cmd_AddToKey:
		case cmd_AddName:
		case cmd_AddPhotoUserID:
		case cmd_AddRevoker:
		case cmd_Revoke:
			if(!mMutableRing || !mDefaultRing)
			{
				outEnabled = FALSE;
				break;
			}
		case cmd_SetDefault:
			{
				STableCell	cell,
							ocell;
				KeyTableRef	tableRef;
				UInt32		dataSize = sizeof(KeyTableRef);
				
				outEnabled = FALSE;
				
				if(inCommand == cmd_SetDefault)
					GetIndString( outName, kStringListID,
									kSetDefaultMenuItemID);
				if(SelectionExists())
				{
					ocell = cell = GetFirstSelectedCell();
					if( ! GetNextSelectedCell(ocell) )
					{
						cell.col = 1;
						cell.row = GetWideOpenIndex(cell.row);
						GetCellData(cell, &tableRef, dataSize);
						if( ( tableRef.type == kKey ) ||
							( ( tableRef.type == kUserID ) &&
								( ( inCommand == cmd_AddCertificate ) ||
								( inCommand == cmd_RetrieveCertificate ) ) ) )
						{
							if( inCommand == cmd_Revoke )
							{
								PGPBoolean	revocable;
								
								PGPGetKeyBoolean(tableRef.u.key,
										kPGPKeyPropIsRevocable, &revocable);
								if( revocable )
									outEnabled = TRUE;
							}
							else
							{
								PGPGetKeyBoolean(tableRef.ownerKey,
										kPGPKeyPropCanSign, &canSign);
								if( canSign )
								{
									if( ( inCommand == cmd_AddPhotoUserID )
										|| ( inCommand == cmd_AddRevoker ) )
									{
										PGPInt32	algorithm;
										
										PGPGetKeyNumber( tableRef.u.key,
											kPGPKeyPropAlgID,
											&algorithm );
										if( algorithm !=
											kPGPPublicKeyAlgorithm_RSA )
											outEnabled = TRUE;
									}
									else
										outEnabled = TRUE;
								}
							}
						}
						else if( inCommand == cmd_Revoke &&
									tableRef.type == kSignature )
						{
							// Could make sure the certifying key is a
							// secret key here, but that would slow
							// things down a lot
							outEnabled = TRUE;
						}
						else if( ( tableRef.type == kUserID ) &&
									( inCommand == cmd_SetDefault ) )
						{
							PGPBoolean	isAttributeID;
							
							err = PGPGetUserIDBoolean( tableRef.u.user,
											kPGPUserIDPropIsAttribute,
											&isAttributeID);
							pgpAssertNoErr( err );
							
							if( ! isAttributeID )
							{
								outEnabled = TRUE;
								GetIndString(	outName,
												kStringListID,
												kSetPrimaryNameMenuItemID);
							}
						}
					}
				}
			}
			break;
		case cmd_EnableKeys:
		case cmd_DisableKeys:
		case cmd_Clear:
		case cmd_OptionClear:
			if(!mMutableRing)
				break;
		case cmd_Info:	
		case cmd_Copy:
		case cmd_ExportKeys:
		case cmd_ReverifySignatures:
		case cmd_Sign1:
			outEnabled = SelectionExists();
			break;
		case cmd_CollapseAll:
			outEnabled = true;
			if(SelectionExists())
				GetIndString(	outName,
								kStringListID,
								kCollapseSelectionID);
			else
				GetIndString(	outName,
								kStringListID,
								kCollapseAllID);
			break;
		case cmd_ExpandAll:
			outEnabled = true;
			if(SelectionExists())
				GetIndString(	outName,
								kStringListID,
								kExpandSelectionID);
			else
				GetIndString(	outName,
								kStringListID,
								kExpandAllID);
			break;
		case cmd_UpdateIntroducers:
#if PGP_BUSINESS_SECURITY
			if(!mMutableRing || !mDefaultRing)
				break;
			outEnabled = true;
#endif
			break;
		case cmd_NewKey:
#if	PGP_BUSINESS_SECURITY
			PGPBoolean		allowKG;
			
			err = PGPGetPrefBoolean(gAdminPrefRef, kPGPPrefAllowKeyGen,
									&allowKG);
			pgpAssertNoErr(err);
			if(!allowKG)
				break;
#endif
		case cmd_ImportKeys:
		case cmd_UpdateRevocations:
			if(!mMutableRing || !mDefaultRing)
				break;
		case cmd_SelectAll:
			outEnabled = true;
			break;
		default:
			LCommander::FindCommandStatus(inCommand, outEnabled,
										outUsesMark, outMark, outName);
			break;
	}
}

	Boolean
CKeyTable::HandleKeyPress(const EventRecord&	inKeyEvent)
{
	Boolean		keyHandled	= true;
	Int16		theKey		= inKeyEvent.message & charCodeMask,
				charInx;
	STableCell	cell;
	TableIndexT	rows, cols;
	
	if(inKeyEvent.modifiers & cmdKey)
		keyHandled = LCommander::HandleKeyPress(inKeyEvent);
	else
	{
		SetUpdateCommandStatus(TRUE);
		switch(theKey)
		{
			default:
				if(((theKey >= ' ') && (theKey <= 'Z')) ||
					((theKey >= 'a') && (theKey <= 'z')))
				{
					if(inKeyEvent.when - kMaxKeypressInterval <=
						mLastKeypress)
					{
						if((charInx = strlen(mSearchString)) <
							kMaxSearchStringSize - 1)
						{
							mSearchString[charInx]		= theKey;
							mSearchString[charInx+1]	= 0;
						}
					}
					else
					{
						mSearchString[0] = theKey;
						mSearchString[1] = 0;
					}
					mLastKeypress = inKeyEvent.when;
					SelectSearchString();
				}
				else
					keyHandled = LCommander::HandleKeyPress(inKeyEvent);
				break;
			case char_UpArrow:
				cell = GetFirstSelectedCell();
				cell.col = 1;
				if(cell.row > 1)
					cell.row--;
				if(cell.row < 1)
					cell.row = 1;
				if( ! ( inKeyEvent.modifiers & shiftKey ) )
					UnselectAllCells();
				ScrollCellIntoFrame(cell);
				SelectCell(cell);
				break;
			case char_DownArrow:
				GetTableSize(rows, cols);
				cell.col = 1;
				for( cell.row = rows; ( cell.row > 0 ) && !CellIsSelected( cell );
					cell.row-- )
					;
				cell.row++;
				if(cell.row > rows)
					cell.row = rows;
				if( ! ( inKeyEvent.modifiers & shiftKey ) )
					UnselectAllCells();
				ScrollCellIntoFrame(cell);
				SelectCell(cell);
				break;
			case char_LeftArrow:
			case char_RightArrow:
				break;
			case char_Home:
				cell.row = cell.col = 1;
				ScrollCellIntoFrame(cell);
				break;
			case char_End:
				GetTableSize(rows, cols);
				cell.col = 1;
				cell.row = rows;
				ScrollCellIntoFrame(cell);
				break;
			case char_PageUp:
			{
				Rect	frameRect;
				
				CalcLocalFrameRect(frameRect);
				ScrollPinnedImageBy( 0,
					-UGraphicUtils::RectHeight(frameRect), true );
				break;
			}
			case char_PageDown:
			{
				Rect	frameRect;
				
				CalcLocalFrameRect(frameRect);
				ScrollPinnedImageBy( 0,
					UGraphicUtils::RectHeight(frameRect), true );
				break;
			}
			case char_Tab:
			case char_Return:
			case char_Enter:
				keyHandled = LCommander::HandleKeyPress(inKeyEvent);
				break;
			case char_FwdDelete:
			case char_Backspace:
				if( SelectionExists() )
				{
					if(inKeyEvent.modifiers & optionKey)
						LCommander::GetTarget()->ProcessCommand(cmd_OptionClear);
					else
						LCommander::GetTarget()->ProcessCommand(cmd_Clear);
				}
				break;
		}
	}
	return keyHandled;
}

	Int16
MatchUserIDs(char *uid1, char *uid2)
{
	Int16	matchChars = 0;
	char	a, b;
	
	while(*uid1 && *uid2)
	{
		a = *uid1++;	b = *uid2++;
		if((a >= 'a') && (a <= 'z'))
			a -= ('a' - 'A');
		if((b >= 'a') && (b <= 'z'))
			b -= ('a' - 'A');
		if(a == b)
			matchChars++;
		else
			break;
	}
	return matchChars;
}

	void
CKeyTable::SelectSearchString()
{
	STableCell	cell;
	TableIndexT	rows,
				cols,
				row,
				lastMarkerRow=0;
	KeyTableRef	tableRef;
	Uint32		dataSize = sizeof(KeyTableRef);
	PGPError	err;
	UInt32		len;
	char		userIDStr[kUserIDStringLength];
	Int16		searchScore,
				bestSearchScore = 0;
	
	UnselectAllCells();
	GetWideOpenTableSize(rows, cols);
	cell.col = 1;
	for(row=1;row<=rows;row++)
	{
		cell.row = row;
		GetCellData(	cell,
						&tableRef,
						dataSize);
		if(tableRef.type == kKey)
		{
			err = PGPGetPrimaryUserIDNameBuffer(tableRef.u.key,
						kUserIDStringLength - 1, userIDStr, &len);
			pgpAssertNoErr(err);
			userIDStr[len] = 0;
			searchScore = MatchUserIDs(userIDStr, mSearchString);
			if(searchScore > bestSearchScore)
			{
				bestSearchScore = searchScore;
				lastMarkerRow = row;	
			}
		}
	}
	cell.row = lastMarkerRow;
	if(!lastMarkerRow)
		mSearchString[0] = 0;
	else if(rows)
	{
		cell.row = GetExposedIndex(cell.row);
		ScrollCellIntoFrame(cell);
		SelectCell(cell);
	}
}

	void
CKeyTable::ScrollImageBy(	Int32		inLeftDelta,
							Int32		inTopDelta,
							Boolean		inRefresh)
{
	if(inLeftDelta != 0)
	{
		mKeyLabels->NotifyScrolled(inLeftDelta);
		if(inRefresh)
			mKeyLabels->Refresh();
	}
	CPGPHierarchyTable::ScrollImageBy(inLeftDelta, inTopDelta, inRefresh);
}

#pragma mark --- Drag and Drop ---

	void
CKeyTable::GetCellDragRgn(
	const STableCell&	inCell,
	RgnHandle			dragRgn,
	Boolean				cutout )
{
	StColorPenState		colorState;
	TableIndexT			woRow =
							mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell			woCell(woRow, 1);
	Uint32				dataSize = sizeof(KeyTableRef);
	KeyTableRef			keyRef;
	Rect				grabRect;
	IconSuiteRef		iconHandle;
	RgnHandle			iconRgn,
						innerRgn;
			
	pgpAssert( IsntNull( dragRgn ) );
	
	GetCellData( woCell, &keyRef, dataSize );
		
	OpenRgn();
	PenSize( 2, 2 );
	grabRect.left	= keyRef.selectLeft +
						kIconWidth + kLeftBorder +
						kLeftIndent;
	grabRect.right	= keyRef.selectRight + 1;
	grabRect.top	= 3;
	grabRect.bottom	= kRowHeight - 3;
	FrameRect( &grabRect );
	CloseRgn( dragRgn );

	if( !GetIconSuite( &iconHandle,
						keyRef.iconID,
						kSelectorAllSmallData ) )
	{
		iconRgn = NewRgn();
		if( IsntNull( iconRgn ) )
		{
			grabRect.top = 1;
			grabRect.bottom = grabRect.top + kIconHeight;
			grabRect.left = keyRef.selectLeft + kLeftIndent;
			grabRect.right = grabRect.left + kIconWidth;
			
			if( !IconSuiteToRgn( iconRgn, &grabRect, kAlignNone,
									iconHandle ) )
			{
				UnionRgn( iconRgn, dragRgn, dragRgn );
			}
			DisposeRgn( iconRgn );
		}
		DisposeIconSuite( iconHandle, false );
	}
	
	if( cutout )
	{
		innerRgn = NewRgn();
		if( IsntNull( innerRgn ) )
		{
			CopyRgn( dragRgn, innerRgn );
			InsetRgn( innerRgn, 2, 2 );
			DiffRgn( dragRgn, innerRgn, dragRgn );
			DisposeRgn( innerRgn );
		}
	}
}

	void		
CKeyTable::GetDragHiliteRgn(
	RgnHandle			ioHiliteRgn,
	const STableCell&	skipCell,
	Boolean				globalize )
{
	StColorPenState	colorState;
	STableCell		cell,
					topLCell,
					botRCell;
	RgnHandle		cellRgn;
	Rect			cellRect,
					frame;

	CalcLocalFrameRect( frame );
	FetchIntersectingCells( frame, topLCell, botRCell );
	cell.col = 1;
	for( cell.row = 0; cell.row <= mRows; cell.row++ ) 
	{
		if( CellIsSelected( cell )  && cell != skipCell &&
			( cell.row >= topLCell.row ) && (cell.row <= botRCell.row ) &&
			( cell.col >= topLCell.col ) && (cell.col <= botRCell.col ) )
		{
			cellRgn = NewRgn();
			if( IsntNull( cellRgn ) )
			{
				GetCellDragRgn( cell, cellRgn, true );
				GetLocalCellRect( cell, cellRect );
				if( globalize )
				{
					LocalToPortPoint( topLeft( cellRect ) );
					PortToGlobalPoint( topLeft( cellRect ) );
				}
				OffsetRgn( cellRgn, cellRect.left, cellRect.top );
				UnionRgn(ioHiliteRgn, cellRgn, ioHiliteRgn);
				DisposeRgn( cellRgn );
			}
		}
	}
}

	void	
CKeyTable::AddFlavors(CKeyDragTask	*dragTask)
{
	pgpAssertAddrValid(dragTask, VoidAlign);
	mDragTask = dragTask;
	SelectionAction(&CKeyTable::AddDrag);
}

	Boolean	
CKeyTable::GetKeyRef(
	ItemReference	item, 
	PGPKeyRef *		key)
{
	TableIndexT		inWideOpenRow =
						mCollapsableTree->GetWideOpenIndex(item);
	STableCell		woCell(inWideOpenRow, 1);
	KeyTableRef		keyRef;
	Uint32			dataSize = sizeof(KeyTableRef);

	GetCellData(woCell, &keyRef, dataSize);
	
	*key = keyRef.ownerKey;
	return TRUE;
}

	Boolean	
CKeyTable::GetTextBlock(ItemReference	item, 
						void			**outData,
						Int32			*len)
{
	TableIndexT		inWideOpenRow =
						mCollapsableTree->GetWideOpenIndex(item);
	STableCell		woCell(inWideOpenRow, 1);
	KeyTableRef		keyRef;
	Uint32			dataSize = sizeof(KeyTableRef);
	PGPKeySetRef	singleKeySet;
	PGPError		err;

	GetCellData(woCell, &keyRef, dataSize);
	
	// This cell has been dragged.  We will now
	// create a text buffer of the ASCII armored
	// version of the key to hand back.
	
	*outData = NULL;
	if(keyRef.type == kKey)
	{
		err = PGPNewSingletonKeySet(keyRef.u.key, &singleKeySet);
		if(IsntPGPError(err) && IsntNull(singleKeySet))
		{
			PGPByte		*keyBuf;
			PGPSize		keyBufLen;
			char		comment[256];
			PGPBoolean	compat;
							
			comment[0] = 0;
#if PGP_BUSINESS_SECURITY
			PGPGetPrefStringBuffer(
					gAdminPrefRef,
					kPGPPrefComments,
					sizeof(comment), comment);
#endif	// PGP_BUSINESS_SECURITY
			if(comment[0] == 0)
				PGPGetPrefStringBuffer(
						gPrefRef,
						kPGPPrefComment,
						sizeof(comment), comment);
			
			PGPGetPrefBoolean( gPrefRef, kPGPPrefExportKeysCompatible,
								&compat );
			
			err = PGPExportKeySet( singleKeySet,
					PGPOAllocatedOutputBuffer(gPGPContext, &keyBuf,
									kMaxKeyBuffer, &keyBufLen),
					PGPOCommentString(gPGPContext, comment),
					PGPOVersionString(gPGPContext, pgpVersionHeaderString),
					compat ?
					PGPOExportFormat(gPGPContext, kPGPExportFormat_Basic) :
					PGPOExportFormat(gPGPContext, kPGPExportFormat_Complete),
					PGPOLastOption(gPGPContext));
			pgpAssertNoErr(err);
			PGPFreeKeySet(singleKeySet);
			if(IsntPGPError(err))
			{
				*outData = (void *)keyBuf;
				*len = keyBufLen;
				return TRUE;
			}
			else
				ReportPGPError(err);
		}
	}
	return FALSE;
}

		Boolean
CKeyTable::GetHFS(
	ItemReference	item,
	DragReference	dragRef,
	FSSpec			*spec)
{
	OSErr		theErr;
	AEDesc		droppedLocation;
	CInfoPBRec	catInfo;
	
	pgpAssertAddrValid(spec, FSSpec);
	theErr = ::GetDropLocation(dragRef, &droppedLocation);
	pgpAssert(theErr == noErr);
	if(theErr == noErr)
	{
		Boolean	isChanged;

		pgpAssert(droppedLocation.descriptorType == typeAlias);
		StHandleLocker	lock(droppedLocation.dataHandle);
		theErr = ::ResolveAlias(	nil, 
									AliasHandle(droppedLocation.dataHandle),
									spec,
									&isChanged);
		
		pgpAssert(theErr == noErr);
	}
	
	if (theErr == noErr)
	{
		catInfo.hFileInfo.ioCompletion 	= nil;
		catInfo.hFileInfo.ioNamePtr 	= spec->name;
		catInfo.hFileInfo.ioVRefNum		= spec->vRefNum;
		catInfo.hFileInfo.ioFDirIndex	= 0;
		catInfo.hFileInfo.ioDirID		= spec->parID;
		
		theErr = PBGetCatInfoSync(&catInfo);
		

		spec->parID 						= catInfo.hFileInfo.ioDirID;
	
		pgpAssert(theErr == noErr);
	}

	if (theErr == noErr)
	{
		TableIndexT		inWideOpenRow =
							mCollapsableTree->GetWideOpenIndex(item);
		STableCell		woCell(inWideOpenRow, 1);
		KeyTableRef		keyRef;
		Uint32			dataSize = sizeof(KeyTableRef),
						len;
		uchar			userIDStr[kUserIDStringLength];
		PGPKeySetRef	singleKeySet;
		PGPError		err;
		
		GetCellData(woCell, &keyRef, dataSize);
		// This cell has been dragged to the Finder.  We will now
		// create a file of the ASCII armored version of the key.
		
		if(keyRef.type == kKey)
		{
			err = PGPNewSingletonKeySet(keyRef.u.key, &singleKeySet);
			if(IsntPGPError(err) && IsntNull(singleKeySet))
			{
				err = PGPGetPrimaryUserIDNameBuffer(keyRef.u.key,
						kUserIDStringLength - 1, (char *)userIDStr, &len);
				pgpAssert(!err);
				userIDStr[len] = 0;
				if(len > 31)
				{
					userIDStr[30] = 0xC9; /* Elipses */
					userIDStr[31] = 0;
				}
				pgpAssert(0);
				// Temporarily commented out following line
				//err = pgpExportKeyFile(singleKeySet, userIDStr);
				pgpAssert(!err);
				//pgpFixBeforeShip("handles dupe filenames??");
				PGPFreeKeySet(singleKeySet);
				return TRUE;
			}
		}
	}
	
	return theErr == noErr;
}

	Boolean	
CKeyTable::AddDrag(STableCell& aCell)
{
	pgpAssertAddrValid(mDragTask, VoidAlign);
	mDragTask->AddThisFlavor(aCell.row);
	return true;
}

	void
CKeyTable::FocusDropArea()
{
	OutOfFocus(nil);
	FocusDraw();
}

	void
CKeyTable::HiliteDropArea(
	DragReference	inDragRef)
{
	Rect			frame;
	Int16			depth;
	
	pgpAssert( IsNull( mDragHiliteWorld ) && IsNull( mDragHiliteRgn ) );
	
	CalcLocalFrameRect( frame );
	
	StDeviceLoop	devLoop(frame);		
	if(devLoop.NextDepth(depth))
	{
		RgnHandle	innerRgn;
		RGBColor	opColor;
		
		try
		{
			mDragHiliteWorld = new LGWorld( frame, depth, useTempMem );
			mDragHiliteRgn = NewRgn();
			innerRgn = NewRgn();
			RectRgn( mDragHiliteRgn, &frame );
			CopyRgn( mDragHiliteRgn, innerRgn );
			InsetRgn( innerRgn, 3, 3 );
			DiffRgn( mDragHiliteRgn, innerRgn, mDragHiliteRgn );
			DisposeRgn( innerRgn );
			
			if( mDragHiliteWorld->BeginDrawing() )
			{
				GrafPtr	thisPort;
				
				GetPort( &thisPort );
				StColorState::Normalize();
				CopyBits( &GetMacPort()->portBits,
							&thisPort->portBits,
							&frame, &frame, srcCopy, mDragHiliteRgn );
				mDragHiliteWorld->EndDrawing();
				
				SetThemePen( kThemeDragHiliteBrush, 16, 1 );
				opColor.red = opColor.green = opColor.blue = 0x7FFF;
				OpColor( &opColor );
				PenMode( blend );
				PaintRgn( mDragHiliteRgn );
			}
		}
		catch(...)
		{
			// Oh well, we ran out of memory, no problem
		}
	}
}

	void
CKeyTable::UnhiliteDropArea(
	DragReference	inDragRef)
{
	Rect	frame;
	CalcLocalFrameRect(frame);
	StColorState::Normalize();
	if( IsntNull( mDragHiliteWorld ) && IsntNull( mDragHiliteRgn ) )
	{
		mDragHiliteWorld->CopyImage(GetMacPort(), frame, srcCopy,
									mDragHiliteRgn);
		DisposeRgn( mDragHiliteRgn );
		mDragHiliteRgn = NULL;
		delete mDragHiliteWorld;
		mDragHiliteWorld = NULL;
	}
}

	Boolean
CKeyTable::ItemIsAcceptable(DragReference inDragRef, ItemReference inItemRef)
{
	Boolean			isAcceptable = FALSE;
	ushort			numFlavors;
	OSStatus		status;
	
	if(!mMutableRing || mSendingDrag)
		return false;
	status = CountDragItemFlavors(inDragRef, inItemRef, &numFlavors);
	if(IsntErr(status))
	{
		for(ushort flavorIndex = 1; flavorIndex <= numFlavors; flavorIndex++)
		{
			FlavorType	flavorType;
			
			status = GetFlavorType(inDragRef, inItemRef, flavorIndex,
									&flavorType);
			if(IsntErr(status))
			{
				if(flavorType == kKeyDragFlavor)
				{
					isAcceptable = TRUE;
				}
				else if(flavorType == 'TEXT')
				{
					isAcceptable = TRUE;
				}
				else if(flavorType == flavorTypeHFS)
				{
					HFSFlavor	flavorData;
					ByteCount	dataSize;
					ByteCount	dataOffset;
					
					dataSize 	= sizeof(flavorData);
					dataOffset	= 0;
					
					status = GetFlavorData(inDragRef, inItemRef, flavorType,
									&flavorData, (Int32 *) &dataSize,
									dataOffset);
					if(IsntErr(status))
					{
						// Check for dragged folders and/or disks
						
						if(flavorData.fileType != 'disk' &&
							flavorData.fileType != 'fold')
						{
							isAcceptable = true;
						}
					}
				}

				if(isAcceptable)
					break;
			}
		}
	}
	return(isAcceptable);
}

	void
CKeyTable::StepForward()
{
	if(!WeAreFront())
	{
		if(!mNotified)
		{
			GetIndString(	sNotifyString,
							kStringListID,
							kStepForwardID);
			GetIconSuite( &mNotifyIconSuite, 128, svAllSmallData );
			HNoPurge( mNotifyIconSuite );
			sImportNotification.qType = nmType;
			sImportNotification.nmMark = 1;
			sImportNotification.nmIcon = mNotifyIconSuite;
			sImportNotification.nmSound = NULL;
			sImportNotification.nmStr = sNotifyString;
			sImportNotification.nmResp = NULL;
			sImportNotification.nmRefCon = 0;
			NMInstall(&sImportNotification);
			mBlockDialogDelays = 3;
			mNotified = TRUE;
		}
	}
	else
	{
		mBlockDialogDelays = 0;
		mNotified = FALSE;
	}
}

	void
CKeyTable::ReceiveDragItem(
	DragReference	inDragRef,
	DragAttributes	inDragAttrs,
	ItemReference	inItemRef,
	Rect			&/*inItemBounds*/)	// In Local coordinates
{
	Boolean			received = FALSE,
					localSend;
	ushort			numFlavors;
	OSStatus		status;
	
	localSend = (inDragAttrs & kDragInsideSenderApplication);
	
	status = CountDragItemFlavors(inDragRef, inItemRef, &numFlavors);
	if(IsntErr(status))
	{
		for(ushort flavorIndex = 1; flavorIndex <= numFlavors; flavorIndex++)
		{
			FlavorType	flavorType;
			Size		expectedSize = 0,
						droppedSize;
			
			status = GetFlavorType(inDragRef, inItemRef, flavorIndex,
									&flavorType);
			if(IsntErr(status))
			{
				if(flavorType == kKeyDragFlavor )
				{
					PGPKeySetRef	keySet;
					PGPKeyRef		key;
					
					status = GetFlavorDataSize(inDragRef, inItemRef,
												flavorType, &expectedSize);
					if( expectedSize == sizeof(PGPKeyRef) )
					{
						droppedSize = expectedSize;
						status = GetFlavorData(inDragRef, inItemRef,
								flavorType, &key, &droppedSize, 0);
						if( IsntErr( status ) &&
							IsntPGPError( PGPNewSingletonKeySet( key, &keySet ) ) )
						{
							ImportKeysFromKeySet( keySet );
							received = TRUE;
						}
					}
				}
				else if(flavorType == 'TEXT')
				{
					PGPByte			*droppedBuf;
					
					status = GetFlavorDataSize(inDragRef, inItemRef,
												flavorType, &expectedSize);
					if(IsntErr(status) && IsntNull(droppedBuf =
								(PGPByte *)pgpAlloc(expectedSize)))
					{
						droppedSize = expectedSize;
						status = GetFlavorData(inDragRef, inItemRef,
								flavorType, droppedBuf, &droppedSize, 0);
						pgpAssertNoErr(status);
						pgpAssert(droppedSize == expectedSize);
						if(IsntErr(status))
						{
							KeyImportStruct *kis;
							
							kis = (KeyImportStruct *)
									pgpAlloc(sizeof(KeyImportStruct));
							if(IsntNull(kis))
							{
								kis->savedImportIsFile = FALSE;
								kis->bypassImportDialog = localSend;
								kis->savedImportBuffer = droppedBuf;
								kis->savedImportBufferSize = droppedSize;
								kis->next = mImports;
								mImports = kis;
								StartIdling();
								StepForward();
							}
						}
						else
							pgpFree(droppedBuf);
					}
					received = TRUE;
				}
				else if(flavorType == flavorTypeHFS)
				{
					HFSFlavor	flavorData;
					ByteCount	dataSize;
					
					dataSize 	= sizeof(flavorData);
					
					status = GetFlavorData(inDragRef, inItemRef, flavorType,
							&flavorData, (Int32 *) &dataSize, 0);
					if(IsntErr(status))
					{
						ImportKeysFromFile(&flavorData.fileSpec, localSend);
						received = TRUE;
					}
				}

				if(received)
					break;
			}
		}
	}
}

