/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc.
	All rights reserved.

	$Id: CAuthenticationPanel.cp,v 1.22 1999/05/17 05:46:40 wprice Exp $
____________________________________________________________________________*/
#include <LPushButton.h>

#include "pgpClientLib.h"
#include "WarningAlert.h"
#include "pflPrefTypes.h"
#include "pgpMem.h"

#include "CString.h"
#include "CIconPane.h"
#include "CPGPnet.h"
#include "CAuthenticationPanel.h"
#include "pgpNetPrefs.h"
#include "MacStrings.h"
#include "PGPnetIPC.h"

namespace
{
	const PaneIDT	caption_PGPKey				=	1000;
	const PaneIDT	button_SelectKey			=	1001;
	const PaneIDT	caption_X509Certificate		=	1002;
	const PaneIDT	button_SelectCertificate	=	1003;
	const PaneIDT	icon_Key					=	1006;
	const PaneIDT	button_ClearKey				=	1007;
	const PaneIDT	button_ClearCertificate		=	1008;
	const PaneIDT	icon_Cert					=	1009;
	const PaneIDT	group_509					=	1010;

	enum
	{
		STRx_AuthenticationPanel	=	1002,
		kSelectPGPKeyID				=	1,
		kNoPGPKeysID				=	2
	};
}


CAuthenticationPanel::CAuthenticationPanel(
	LStream *	inStream)
	: CPanel(inStream)
{
	mPGPAuthKey		= kInvalidPGPKeyRef;
	m509AuthKey		= kInvalidPGPKeyRef;
	m509AuthCert	= kInvalidPGPSigRef;
}

CAuthenticationPanel::~CAuthenticationPanel()
{
	
}

	void
CAuthenticationPanel::FinishCreateSelf()
{
	// Listen to buttons
	((LPushButton *) FindPaneByID(button_SelectKey))->AddListener(this);
	((LPushButton *) FindPaneByID(button_SelectCertificate))->AddListener(this);
	( mClearKey = (LPushButton *) FindPaneByID(button_ClearKey))->
		AddListener(this);
	( mClearCert = (LPushButton *) FindPaneByID(button_ClearCertificate))->
		AddListener(this);
#if PGP_FREEWARE
	FindPaneByID( group_509 )->Disable();
#endif

	Load( NULL );
}

	void
CAuthenticationPanel::ListenToMessage(
	MessageT	inMessage,
	void *	)
{
	PGPError		pgpErr;
	PGPKeySetRef	tempKeySet;
	
	switch (inMessage)
	{
		case button_SelectKey:
		{
			StPGPKeySetRef	theKeySet;
			PGPUInt32		count;
			
			CPGPnet::PostPNEvent( kPGPnetHLEvent_ClearPhrasePGP );
			theKeySet = PGPAuthenticationKeyFilter(
											CPGPnet::GetDefaultKeyrings());
			PGPCountKeys(theKeySet, &count);
			if( count == 0 )
			{
				WarningAlert(	kWAStopAlertType,
								kWAOKStyle,
								STRx_AuthenticationPanel,
								kNoPGPKeysID);
			}
			else
			{
				pgpErr = PGPSelectKeysDialog(
							CPGPnet::GetPGPContext(),
							kPGPSelectKeysSingleKey,
							CString(CStringFromSTRx(STRx_AuthenticationPanel, kSelectPGPKeyID)),
							theKeySet,
							NULL,
							&tempKeySet);
				if( IsntPGPError( pgpErr) )
				{
					CString			userID;
					PGPSize			userIDSize = userID.GetMinStorage();
					StPGPKeyListRef	keyList;
					StPGPKeyIterRef	keyIter;
					
					pgpErr = PGPOrderKeySet( tempKeySet,
								kPGPAnyOrdering, &keyList );
					pgpAssertNoErr( pgpErr );
					pgpErr = PGPNewKeyIter( keyList, &keyIter );
					pgpAssertNoErr( pgpErr );
					pgpErr = PGPKeyIterNext( keyIter, &mPGPAuthKey );
					pgpAssertNoErr( pgpErr );
					pgpErr = PGPGetPrimaryUserIDNameBuffer(	mPGPAuthKey, 
															userIDSize,
															userID,
															&userIDSize);
					pgpAssertNoErr( pgpErr );
					FindPaneByID(caption_PGPKey)->SetDescriptor(userID);
					
					Handle						iconSuite;
					
					pgpErr = PGPGetKeyIcon( mPGPAuthKey, NULL, &iconSuite );
					pgpAssertNoErr( pgpErr );
					((CIconPane *) FindPaneByID(icon_Key))->
						SetIconSuite(iconSuite, true);
					((CIconPane *) FindPaneByID(icon_Key))->Show();
					if( PGPKeySetRefIsValid( mPGPKeySet ) )
						mPGPKeySet.Free();
					mPGPKeySet = tempKeySet;
					mClearKey->Enable();
				}
			}
			break;
		}
		case button_SelectCertificate:
		{
			PGPKeyRef	x509Key;
			PGPSigRef	x509Cert;
			Str255		pstr;
			char		cstr[256];
			PGPSize		strSize = sizeof(cstr) - 1;
			
			CPGPnet::PostPNEvent( kPGPnetHLEvent_ClearPhrase509 );
			pgpErr = PGPSelect509Dialog(
						CPGPnet::GetPGPContext(),
						kPGPSelect509CanSignOnly + kPGPSelect509NoSplit,
						NULL,
						CPGPnet::GetDefaultKeyrings(),
						&tempKeySet, &x509Key, &x509Cert );
			if( pgpErr != kPGPError_UserAbort )
			{
				pgpErr = PGPGetSigPropertyBuffer( x509Cert,
							kPGPSigPropX509LongName, strSize,
							(char *)&cstr, &strSize );
				if( IsntPGPError( pgpErr ) )
				{
					cstr[strSize] = '\0';
					CToPString( cstr, pstr );
					FindPaneByID(caption_X509Certificate)->SetDescriptor( pstr );
					( (CIconPane *) FindPaneByID( icon_Cert ) )->Show();
					
					if( PGPKeySetRefIsValid( m509KeySet ) )
						m509KeySet.Free();
					m509KeySet		= tempKeySet;
					m509AuthKey		= x509Key;
					m509AuthCert	= x509Cert;
					mClearCert->Enable();
				}
				else
					PGPFreeKeySet( tempKeySet );
			}
			break;
		}
		case button_ClearKey:
			FindPaneByID(caption_PGPKey)->SetDescriptor( "\p" );
			((CIconPane *) FindPaneByID(icon_Key))->Hide();
			mPGPAuthKey = kInvalidPGPKeyRef;
			mClearKey->Disable();
			CPGPnet::PostPNEvent( kPGPnetHLEvent_ClearPhrasePGP );
			break;
		case button_ClearCertificate:
			FindPaneByID(caption_X509Certificate)->SetDescriptor( "\p" );
			((CIconPane *) FindPaneByID(icon_Cert))->Hide();
			m509AuthKey		= kInvalidPGPKeyRef;
			m509AuthCert	= kInvalidPGPSigRef;
			mClearCert->Disable();
			CPGPnet::PostPNEvent( kPGPnetHLEvent_ClearPhrase509 );
			break;
	}
}

	void
CAuthenticationPanel::Load(
	void * )
{
	PGPByte *				keyID = NULL;
	PGPSize					len;
	PGPError				err;
	PGPSize					iasnSize = 0;
	PGPByte *				iasn = NULL;
	PGPUInt32				algNum;
	
	err = PGPGetPrefData( CPGPnet::GetNetPrefs(),
			kPGPNetPrefPGPAuthKeyID,
			&len, &keyID );
	if( IsntPGPError( err ) )
	{
		PGPGetPrefNumber( CPGPnet::GetNetPrefs(),
			kPGPNetPrefPGPAuthKeyAlgorithm, &algNum );
		PGPGetKeyByKeyID( CPGPnet::GetDefaultKeyrings(),
			(PGPKeyID *)keyID, (PGPPublicKeyAlgorithm)algNum, &mPGPAuthKey );
		if( PGPKeyRefIsValid( mPGPAuthKey ) )
		{
			CString			userID;
			PGPSize			userIDSize = userID.GetMinStorage();
			
			err = PGPGetPrimaryUserIDNameBuffer(	mPGPAuthKey, 
													userIDSize,
													userID,
													&userIDSize);
			if( IsntPGPError( err ) )
			{
				FindPaneByID(caption_PGPKey)->SetDescriptor(userID);
				
				Handle						iconSuite;
				
				err = PGPGetKeyIcon( mPGPAuthKey, NULL, &iconSuite );
				pgpAssertNoErr( err );
				if( IsntNull( iconSuite ) )
				{
					( (CIconPane *) FindPaneByID( icon_Key ) )->
						SetIconSuite( iconSuite, true );
				}
				( (CIconPane *) FindPaneByID( icon_Key ) )->Show();
				mClearKey->Enable();
			}
		}
		PGPFreeData( keyID );
		keyID = NULL;
	}
	
	PGPGetPrefData( CPGPnet::GetNetPrefs(),
			kPGPNetPrefX509AuthKeyID,
			&len, &keyID );
	PGPGetPrefNumber( CPGPnet::GetNetPrefs(),
		kPGPNetPrefX509AuthKeyAlgorithm, &algNum );
	PGPGetPrefData( CPGPnet::GetNetPrefs(),
			kPGPNetPrefX509AuthCertIASN,
			&iasnSize, &iasn );
	if( IsntNull( iasn )  && len )
	{
		err = PGPX509CertFromExport( CPGPnet::GetPGPContext(),
					algNum, keyID, iasn, iasnSize,
					CPGPnet::GetDefaultKeyrings(),
					&m509AuthKey, &m509AuthCert );
		if( IsntPGPError( err ) )
		{
			Str255		pstr;
			char		cstr[256];
			PGPSize		strSize = sizeof(cstr) - 1;
			
			PGPGetSigPropertyBuffer( m509AuthCert,
					kPGPSigPropX509LongName, strSize,
					(char *)&cstr, &strSize );
			cstr[strSize] = '\0';
			CToPString( cstr, pstr );
			FindPaneByID(caption_X509Certificate)->
				SetDescriptor( pstr );
			( (CIconPane *) FindPaneByID( icon_Cert ) )->
				Show();
			mClearCert->Enable();
		}
	}
	if( IsntNull( iasn ) )
		PGPFreeData( iasn );
	if( IsntNull( keyID ) )
		PGPFreeData( keyID );
}

	void
CAuthenticationPanel::Save(
	void * )
{
	PGPKeyID				keyID;
	PGPUInt32				alg = kPGPPublicKeyAlgorithm_DSA;
	PGPSize					len;
	PGPUInt32				iasnSize = 0;
	PGPByte *				iasn;
	PGPBoolean				good = FALSE;
	
	if( PGPKeyRefIsValid( mPGPAuthKey ) )
	{
		if( IsntPGPError( PGPGetKeyIDFromKey( mPGPAuthKey, &keyID ) ) )
		{
			PGPGetKeyNumber( mPGPAuthKey, kPGPKeyPropAlgID, (PGPInt32 *)&alg );
			PGPSetPrefNumber( CPGPnet::GetNetPrefs(),
				kPGPNetPrefPGPAuthKeyAlgorithm, alg );
			len = sizeof(PGPKeyID);
			PGPSetPrefData( CPGPnet::GetNetPrefs(),
				kPGPNetPrefPGPAuthKeyID, len, &keyID );
			good = TRUE;
		}
	}
	if( !good )
		PGPRemovePref( CPGPnet::GetNetPrefs(), kPGPNetPrefPGPAuthKeyID );
	good = FALSE;
	if( PGPSigRefIsValid( m509AuthCert ) )
	{
		if( IsntPGPError( PGPX509CertToExport( CPGPnet::GetPGPContext(),
								m509AuthKey, m509AuthCert,
								&alg, (PGPByte *)&keyID,
								&iasn, &iasnSize ) ) )
		{
			PGPSetPrefNumber( CPGPnet::GetNetPrefs(),
				kPGPNetPrefX509AuthKeyAlgorithm, alg );
			PGPSetPrefData( CPGPnet::GetNetPrefs(),
				kPGPNetPrefX509AuthCertIASN, iasnSize, iasn );
			len = sizeof(PGPKeyID);
			PGPSetPrefData( CPGPnet::GetNetPrefs(),
				kPGPNetPrefX509AuthKeyID, len, &keyID );
			PGPFreeData( iasn );
			good = TRUE;
		}
	}
	if( !good )
		PGPRemovePref( CPGPnet::GetNetPrefs(), kPGPNetPrefX509AuthKeyID );
}

	PGPKeySetRef
CAuthenticationPanel::PGPAuthenticationKeyFilter(
	PGPKeySetRef	inAllKeys) const
{
	PGPError		pgpErr;
	PGPKeySetRef	result;
	StPGPKeySetRef	keySet;
	StPGPKeySetRef	oldSet;
	StPGPKeySetRef	singleKeySet;
	StPGPKeyListRef	keyList;
	StPGPKeyIterRef	keyIter;
	PGPKeyRef		key;
	PGPBoolean		canSign;
	PGPBoolean		isSecretShared;
	
	pgpErr = PGPNewEmptyKeySet(inAllKeys, &keySet);
	pgpAssertNoErr( pgpErr );
	pgpErr = PGPOrderKeySet(inAllKeys, kPGPAnyOrdering, &keyList);
	pgpAssertNoErr( pgpErr );
	pgpErr = PGPNewKeyIter(keyList, &keyIter);
	pgpAssertNoErr( pgpErr );
	
	while (PGPKeyIterNext(keyIter, &key) == kPGPError_NoErr) {
		pgpErr = PGPGetKeyBoolean(key, kPGPKeyPropCanSign, &canSign);
		pgpAssertNoErr( pgpErr );
		PGPGetKeyBoolean(	key,	kPGPKeyPropIsSecretShared,
									&isSecretShared);
		if (canSign && !isSecretShared)
		{
			pgpErr = PGPNewSingletonKeySet(key, &singleKeySet);
			pgpAssertNoErr( pgpErr );
			oldSet = keySet;
			pgpErr = PGPUnionKeySets(oldSet, singleKeySet, &keySet);
			pgpAssertNoErr( pgpErr );
			oldSet.Free();
			singleKeySet.Free();
		}
	}
	
	result = keySet;
	keySet.Set(kInvalidPGPKeySetRef);
	
	return result;
}

