/*____________________________________________________________________________
	Copyright (C) 1998 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: CNewDiskWizardDialog.cp,v 1.11 1999/03/15 21:30:18 heller Exp $
____________________________________________________________________________*/

#include <LCheckBox.h>
#include <LEditText.h>
#include <LProgressBar.h>
#include <LPushButton.h>
#include <LRadioButton.h>
#include <LStaticText.h>
#include <PP_Messages.h>
#include <UDesktop.h>
#include <URegistrar.h>
#include <UReanimator.h>

#if PGP_BUSINESS_SECURITY
#include "pgpKeys.h"
#include "pgpUtilities.h"
#endif

#include "CPassphraseEdit.h"
#include "CWarningAlert.h"
#include "MacDialogs.h"
#include "MacFiles.h"
#include "MacStrings.h"
#include "pgpMem.h"
#include "pgpPassphraseUtils.h"
#include "pgpRandomPool.h"

#include "CNewDiskWizardDialog.h"
#include "CPGPDiskApplication.h"
#include "PGPDiskFile.h"
#include "PGPDiskPreferences.h"
#include "PGPDiskResources.h"
#include "PGPDiskUtils.h"

// The number of random data points needed is approximately the number of
// bits we need. Right now that is 64 bits for the salt and 128 bits for
// the encrypted key. We use 300 here to give ourselves a margin.

#define kRandomDataSampleCount	300

const ResIDT		view_Introduction			= 8100;
const ResIDT		view_GetInfo				= 8200;
const ResIDT		view_GetPassphrase			= 8300;
const ResIDT		view_GatherRandomData		= 8400;
const ResIDT		view_Formatting				= 8500;
const ResIDT		view_Error					= 8600;
const ResIDT		view_Conclusion				= 8700;

#if PGP_BUSINESS_SECURITY
const ResIDT		view_ADKWarning				= 8350;
#endif

const MessageT		kNextViewMessage			= 'Next';
const MessageT		kPreviousViewMessage		= 'Prev';
const MessageT		kAllowNextViewSwitchMessage	= 'ANxt';
const MessageT		kAllowPrevViewSwitchMessage	= 'APrv';
const MessageT		kDenyNextViewSwitchMessage	= 'DNxt';
const MessageT		kDenyPrevViewSwitchMessage	= 'DPrv';
const MessageT		kDenyCancelMessage			= 'DCan';

const PaneIDT		kNextButtonPaneID			= 'bNxt';
const PaneIDT		kPreviousButtonPaneID		= 'bPrv';
const PaneIDT		kCancelButtonPaneID			= 'bCan';

const PaneIDT		kGetInfoDescTextPaneID		= 'cDes';
const PaneIDT		kDiskNameEditTextPaneID		= 'eNam';
const PaneIDT		kDiskSizeEditTextPaneID		= 'eSiz';
const PaneIDT		kKBRadioButtonPaneID		= 'rKB ';
const PaneIDT		kMBRadioButtonPaneID		= 'rMB ';

const PaneIDT		kADKUserIDTextPaneID		= 'cADK';

const PaneIDT		kPassphraseEditTextPaneID	= 'ePas';
const PaneIDT		kConfirmationEditTextPaneID	= 'eCon';
const PaneIDT		kHideTypingCheckboxPaneID	= 'xHid';
const PaneIDT		kQualityProgressBarPaneID	= 'pbPQ';

const PaneIDT		kRandomProgressBarPaneID	= 'pbRD';
const PaneIDT		kClickNextTextPaneID		= 'cNxt';

const PaneIDT		kFormattingNameTextPaneID	= 'cFmt';
const PaneIDT		kFormattingProgressBarPaneID	= 'pbFM';

const PaneIDT		kErrorTextPaneID			= 'cErr';

const MessageT		msg_ChangedDiskName			= kDiskNameEditTextPaneID;
const MessageT		msg_ChangedDiskSize			= kDiskSizeEditTextPaneID;
const MessageT		msg_HideTyping				= kHideTypingCheckboxPaneID;
const MessageT		msg_ChangedPassphrase		= kPassphraseEditTextPaneID;
const MessageT		msg_ChangedConfirmation		= kConfirmationEditTextPaneID;

NewDiskView		CNewDiskWizardDialog::mNewView 		= kNewDiskView_Invalid;
CComboError		CNewDiskWizardDialog::mNewDiskError;

LProgressBar	*CNewDiskWizardDialog::mFormattingProgressBar = NULL;

#pragma mark ------GetNewPGPDiskFSSpec--------

enum	
{
	kNewPGPDiskSFDialogResID		= 2002,
	kNewPGPDiskMinimumSizeInK		= 400,
	kNewPGPDiskMaximumSizeInMB		= 2047L,
	kNewPGPDiskMaximumSizeInK		= ( kNewPGPDiskMaximumSizeInMB * 1024L ),
	
	kAvailableFreeSpaceStaticText	= 13
};

typedef struct NewPGPDiskSFData
{
	short				curVRefNum;
	StandardFileReply	*sfReply;

} NewPGPDiskSFData;

	static UInt32
GetVolumeFreeSpaceInK(short vRefNum)
{
	UInt32	freeAllocBlocks;
	UInt32	allocBlockSize;
	UInt32	freeKilobytes = 0;
	
	if( GetVolumeBlockCounts( vRefNum, &freeAllocBlocks, NULL,
				&allocBlockSize ) == noErr )
	{
		UInt32	freeBlocks;
		
		// Compute free space
		freeBlocks		= freeAllocBlocks * ( allocBlockSize / 512L );
		freeKilobytes	= freeBlocks / 2;
	}
	
	return( freeKilobytes );

}

	static pascal short
NewPGPDiskSFHook(short item, DialogPtr dialog, void *dataPtr)
{
	pgpAssertAddrValid( dialog, DialogRecord );
	pgpAssertAddrValid( dataPtr, NewPGPDiskSFData );
	
	if ( GetWRefCon( dialog ) == sfMainDialogRefCon )
	{
		NewPGPDiskSFData	*sfData = (NewPGPDiskSFData *) dataPtr;
		
		pgpAssertAddrValid( sfData->sfReply, StandardFileReply );
		
		switch( item )
		{
			case sfHookNullEvent:
			{
				// Check to see if the disk has changed. If so, recompute
				// the available free space on the disk and update the text.
				// Round the free space up so that the user can enter this
				// number in order to use all the space
			
				if( sfData->curVRefNum != sfData->sfReply->sfFile.vRefNum )
				{
					Str255	freeSpaceStr;
					Str255	tempStr;
					UInt32	freeMB;
					UInt32	freeKilobytes;
					
					sfData->curVRefNum = sfData->sfReply->sfFile.vRefNum;
					
					freeSpaceStr[0] = 0;
					
					freeKilobytes = GetVolumeFreeSpaceInK( sfData->curVRefNum );
					if ( freeKilobytes > kNewPGPDiskMaximumSizeInK )
						freeKilobytes = kNewPGPDiskMaximumSizeInK;
						
					freeMB = freeKilobytes / 1024UL;
						
					if ( freeMB >= 10 )
					{
						GetIndString( freeSpaceStr, kDialogStringListResID,
								kAvailableFreeSpaceMegabytesFormatStrIndex );
						NumToString( freeMB, tempStr );				
						PrintPString( freeSpaceStr, freeSpaceStr, tempStr );
					}
					else
					{
						GetIndString( freeSpaceStr, kDialogStringListResID,
								kAvailableFreeSpaceKilobytesFormatStrIndex );
						NumToString( freeKilobytes, tempStr );				
						PrintPString( freeSpaceStr, freeSpaceStr, tempStr );
					}
					
					SetIndDialogItemText( dialog,
							kAvailableFreeSpaceStaticText, freeSpaceStr );
				}
				
				break;
			}
		}
	}
	
	return item;
}

//	Presents a Standard File put dialog for saving the new PGP disk. Also
// 	prompts the user for the size of the disk. Default values come from the
//	preferences. Returns userCanceledErr if the user cancelled the dialog.

	static OSStatus
GetNewPGPDiskFSSpec(
	FSSpec 		*fileSpec,
	ScriptCode	*fileScript)
{
	OSStatus			status = noErr;
	Point				where;
	Str255				prompt;
	Str255				defaultFileName;
	DlgHookYDUPP		dialogHookUPP;
	StandardFileReply	sfReply;
	NewPGPDiskSFData	sfData;
	
	pgpAssertAddrValid( fileSpec, FSSpec );
	pgpAssertAddrValid( fileScript, ScriptCode );
	
	where.h	= 0;
	where.v	= 0;
	
	GetIndString( prompt, kDialogStringListResID, kNewPGPDiskPromptStrIndex );
	GetIndString( defaultFileName, kDialogStringListResID,
			kNewPGPDiskFileNameStrIndex );
	
	dialogHookUPP = NewDlgHookYDProc( NewPGPDiskSFHook );
	if( dialogHookUPP == nil )
	{
		pgpDebugMsg( "GetNewPGPDiskFSSpec: Cannot allocate dialog hook UPP" );
		
		status = memFullErr;
		goto Exit;
	}

	pgpClearMemory( &sfData, sizeof( sfData ) );
	
	sfData.sfReply = &sfReply;
	
TryAgain:

	CustomPutFile( prompt, defaultFileName, &sfReply,
					kNewPGPDiskSFDialogResID, where, dialogHookUPP, NULL,
					NULL, NULL, &sfData );
	
	if( sfReply.sfGood )
	{
		(void) FSMakeFSSpec( sfReply.sfFile.vRefNum, sfReply.sfFile.parID,
					sfReply.sfFile.name, fileSpec );
		*fileScript	= sfReply.sfScript;
	
		AssertSpecIsValid( fileSpec, "CPGPDiskApplication::"
				"GetNewPGPDiskFSSpec: bad spec from sfReply.sfFile" );
		
		status = CheckForLocalPGPDiskVolume( fileSpec->vRefNum );
		if( IsErr( status ) )
		{
			CInfoPBRec	cpb;
			Boolean		fileAlreadyExists = FALSE;
			
			if( status == kPGPDiskRemoteVolumeTypeError ||
				status == kPGPDiskForeignVolumeTypeError )
			{
				Str255	msg;
				Str255	volumeName;
				
				GetIndString( msg, kErrorStringListResID,
						kPGPDisksMustBeOnLocalVolumeStrIndex );
				PrintPString( msg, msg, fileSpec->name );

				GetVolumeName( sfReply.sfFile.vRefNum, volumeName );
				PrintPString( msg, msg, volumeName );

				SysBeep( 1 );
				CWarningAlert::Display( kWACautionAlertType, kWAOKStyle, msg );
				
				goto TryAgain;
			}
			else
			{
				pgpDebugMsg( "GetNewPGPDiskFSSpec: Unexpected error from "
						"CheckForLocalPGPDiskVolume" );
				// punt
			}
		
			if( IsntErr( FSpGetCatInfo( fileSpec, &cpb ) ) )
			{
				fileAlreadyExists = TRUE;
				
				// Standard file will not let the user try to replace a
				// folder, so we don't need to handle that case.
				
				if( ( cpbFileAttributes( &cpb ) &
					( kFileAttributeFileLockedMask |
						kFileAttributeFileOpenMask ) ) != 0 )
				{
					Str255	msg;
					
					// File is locked or open. Inform the user and try again.
					GetIndString( msg, kErrorStringListResID,
							kCantReplaceLockedOrBusyFileStrIndex );
					PrintPString( msg, msg, fileSpec->name );
					
					SysBeep( 1 );
					CWarningAlert::Display( kWAStopAlertType, kWAOKStyle,
							msg );
					goto TryAgain;
				}
				else if( IsPGPDiskFileType( cpbFileCreator( &cpb ),
								cpbFileType( &cpb ) ) )
				{
					Str255		msg;
					MessageT	result;
					
					// If it is a PGPDisk file, get double confirmation
					// before replacing the file.
					
					GetIndString( msg, kErrorStringListResID,
							kConfirmReplacePGPDiskFileStrIndex );
					PrintPString( msg, msg, fileSpec->name );
					PrintPString( msg, msg, fileSpec->name );
								
					SysBeep( 1 );
					result = CWarningAlert::Display( kWACautionAlertType,
										kWACancelOKStyle, msg );
					if( result == msg_Cancel )
						goto TryAgain;
				}
			}
		}
	}
	else
	{
		status = userCanceledErr;
	}
		
Exit:

	if( dialogHookUPP != nil )
		DisposeRoutineDescriptor( dialogHookUPP );
	
	return( status );
}

#pragma mark --- CNewDiskWizardDialog ---

CNewDiskWizardDialog::CNewDiskWizardDialog(LStream *inStream)
	: CPGPDiskDialog(inStream)
{
	mCurrentView 			= kNewDiskView_Invalid;
	mCurrentViewObj			= NULL;
	mDiskSize				= gPGPDiskPrefs.newDiskDefaultSize;
	mDiskSizeIsMB			= gPGPDiskPrefs.newDiskDefaultIsMB;
	mCollectingRandomData	= FALSE;
	mFormattingNewDisk		= FALSE;
	
#if PGP_BUSINESS_SECURITY
	mContext	= kInvalidPGPContextRef;
	mAllKeys	= kInvalidPGPKeySetRef;
	mADKRef		= kInvalidPGPKeyRef;
#endif
}

CNewDiskWizardDialog::~CNewDiskWizardDialog()
{
	if( gPGPDiskPrefs.newDiskDefaultSize != mDiskSize ||
		gPGPDiskPrefs.newDiskDefaultIsMB != mDiskSizeIsMB )
	{
		WritePreferences( &gPGPDiskPrefs );
	}

#if PGP_BUSINESS_SECURITY	
	if( PGPKeySetRefIsValid( mAllKeys ) )
		PGPFreeKeySet( mAllKeys );
#endif
}

#if PGP_BUSINESS_SECURITY		 // [

	void
CNewDiskWizardDialog::SetADK(
	PGPKeySetRef 	keySet,
	PGPKeyRef 		adkRef)
{
	mAllKeys	= keySet;
	mADKRef		= adkRef;
}

#endif	// ]

	void
CNewDiskWizardDialog::FinishCreateSelf()
{
	CPGPDiskDialog::FinishCreateSelf();
	
	mNextButton = (LPushButton *) FindPaneByID( kNextButtonPaneID );
	pgpAssertAddrValid( mNextButton, VoidAlign );

	mPreviousButton = (LPushButton *) FindPaneByID( kPreviousButtonPaneID );
	pgpAssertAddrValid( mPreviousButton, VoidAlign );

	mCancelButton = (LPushButton *) FindPaneByID( kCancelButtonPaneID );
	pgpAssertAddrValid( mCancelButton, VoidAlign );

	// mNextButton already added by the modal dialog handler.
	mPreviousButton->AddListener( this );

	mNewView = kNewDiskView_Invalid;
	
	SwitchToView( kNewDiskView_Introduction );
	StartIdling();
}

	void
CNewDiskWizardDialog::ListenToMessage(
	MessageT 	inMessage,
	void 		*ioParam)
{
	switch( inMessage )
	{
		case msg_ChangedDiskName:
		case msg_ChangedDiskSize:
		case msg_ChangedPassphrase:
		case msg_ChangedConfirmation:
			AdjustControlsForView( mCurrentView );
			break;
		
		case msg_HideTyping:
		{
			Boolean	hideTyping;
			
			pgpAssertAddrValid( mPassphraseEdit, VoidAlign );
			pgpAssertAddrValid( mConfirmationEdit, VoidAlign );
			
			hideTyping = (*(UInt32 *) ioParam) != 0;
			
			mPassphraseEdit->SetHideTyping( hideTyping );
			mConfirmationEdit->SetHideTyping( hideTyping );
			
			break;
		}
		
		case kNextViewMessage:
			SetNewView( GetNextView( mCurrentView ) );
			break;
			
		case kPreviousViewMessage:
			SetNewView( GetPreviousView( mCurrentView ) );
			break;
			
		case msg_Cancel:
		{
			if( mFormattingNewDisk )
			{
				mNewDiskError.err = userCanceledErr;
			}
			else
			{
				DoClose();
			}
			
			break;
		}
		
		case kAllowNextViewSwitchMessage:
			mNextButton->Enable();
			break;
					
		case kAllowPrevViewSwitchMessage:
			mPreviousButton->Enable();
			break;
			
		case kDenyNextViewSwitchMessage:
			mNextButton->Disable();
			break;
			
		case kDenyPrevViewSwitchMessage:
			mPreviousButton->Disable();
			break;

		case kDenyCancelMessage:
			mCancelButton->Disable();
			break;
			
		default:
			CPGPDiskDialog::ListenToMessage( inMessage, ioParam );
			break;
	}
}

	void
CNewDiskWizardDialog::AdjustControlsForView(NewDiskView theView)
{
	Boolean	enableNextButton	= TRUE;
	Boolean	enablePrevButton	= TRUE;
	Boolean	enableCancelButton	= TRUE;
	
	switch( theView )
	{
		case kNewDiskView_Introduction:
			enablePrevButton = FALSE;
			break;
		
		case kNewDiskView_GetInfo:
		{
			Str255		str;
			
			mDiskNameEdit->GetDescriptor( str );
			if( str[0] == 0 )
			{
				enableNextButton = FALSE;
			}
			else
			{
				mDiskSizeEdit->GetDescriptor( str );
				if( str[0] == 0 )
				{
					enableNextButton = FALSE;
				}
			}

			break;
		}
		
		case kNewDiskView_GetPassphrase:
		{
			ushort				passphraseQuality;
			CSecureCString256	cPassphrase;
			
			mPassphraseEdit->GetDescriptor( mPassphrase );
			if( mPassphrase[0] < 8 )
			{
				enableNextButton = FALSE;
			}
			
			PToCString( mPassphrase, cPassphrase );
			
			passphraseQuality = pgpEstimatePassphraseQuality( cPassphrase );
			mQualityProgressBar->SetValue( passphraseQuality );
			
			break;
		}
		
		case kNewDiskView_GatherRandomData:
		{
			enableNextButton = FALSE;
			break;
		}
		
		case kNewDiskView_Formatting:
		{
			enableNextButton 	= FALSE;
			enablePrevButton 	= FALSE;
			break;
		}

		case kNewDiskView_Conclusion:
		case kNewDiskView_Error:
		{
			Str255	buttonTitle;
			
			GetIndString( buttonTitle, kMiscStringsListResID,
						kDoneButtonTitleStrIndex );
			mNextButton->SetDescriptor( buttonTitle );
			
			enablePrevButton 	= FALSE;
			enableCancelButton	= FALSE;
			break;
		}
	}
	
	if( enableNextButton )
	{
		mNextButton->Enable();
	}
	else
	{
		mNextButton->Disable();
	}

	if( enablePrevButton )
	{
		mPreviousButton->Enable();
	}
	else
	{
		mPreviousButton->Disable();
	}

	if( enableCancelButton )
	{
		mCancelButton->Enable();
	}
	else
	{
		mCancelButton->Disable();
	}
}

	NewDiskView
CNewDiskWizardDialog::GetNextView(NewDiskView theView)
{
	NewDiskView	nextView;
	
	switch( theView )
	{
		case kNewDiskView_Introduction:
			nextView = kNewDiskView_GetInfo;
			break;

		case kNewDiskView_GetInfo:
			nextView = kNewDiskView_GetPassphrase;
	#if PGP_BUSINESS_SECURITY
			if( PGPKeyRefIsValid( mADKRef ) )
				nextView = kNewDiskView_ADKWarning;
	#endif
			break;
		
	#if PGP_BUSINESS_SECURITY
		case kNewDiskView_ADKWarning:
			nextView = kNewDiskView_GetPassphrase;
			break;
	#endif
			
		case kNewDiskView_GetPassphrase:
			nextView = kNewDiskView_GatherRandomData;
			break;

		case kNewDiskView_GatherRandomData:
			nextView = kNewDiskView_Formatting;
			break;

		case kNewDiskView_Formatting:
			nextView = kNewDiskView_Conclusion;
			break;

		case kNewDiskView_Conclusion:
		case kNewDiskView_Error:
			nextView = kNewDiskView_Exit;
			break;

		default:
			pgpDebugMsg( "Missing next view" );
			nextView = theView;
			break;
	}
	
	return( nextView );
}

	NewDiskView
CNewDiskWizardDialog::GetPreviousView(NewDiskView theView)
{
	NewDiskView	previousView;
	
	switch( theView )
	{
		case kNewDiskView_GetInfo:
			previousView = kNewDiskView_Introduction;
			break;

	#if PGP_BUSINESS_SECURITY
		case kNewDiskView_ADKWarning:
			previousView = kNewDiskView_GetInfo;
			break;
	#endif
			
		case kNewDiskView_GetPassphrase:
			previousView = kNewDiskView_GetInfo;
	#if PGP_BUSINESS_SECURITY
			if( PGPKeyRefIsValid( mADKRef ) )
				previousView = kNewDiskView_ADKWarning;
	#endif
			break;

		case kNewDiskView_GatherRandomData:
			previousView = kNewDiskView_GetPassphrase;
			break;
			
		default:
			pgpDebugMsg( "Missing previous view" );
			previousView = theView;
			break;
	}
	
	return( previousView );
}

	NewDiskView
CNewDiskWizardDialog::LeaveGetInfoView(NewDiskView newView)
{
	if( newView == kNewDiskView_GetPassphrase
#if PGP_BUSINESS_SECURITY
		|| newView == kNewDiskView_ADKWarning
#endif
	)
	{
		LRadioButton	*buttonObj;
		UInt32			diskSizeInK;
		UInt32			strIndex;
		
		mDiskNameEdit->GetDescriptor( mDiskName );
		
		// Remove ':' characters from name and replace with '-'
		for( strIndex = 0; strIndex < mDiskName[0]; strIndex++ )
		{
			if( mDiskName[strIndex] == ':' )
				mDiskName[strIndex] = '-';
		}
		
		mDiskSize = mDiskSizeEdit->GetValue();
		
		buttonObj = (LRadioButton *) FindPaneByID( kMBRadioButtonPaneID );
		pgpAssertAddrValid( buttonObj, VoidAlign );

		mDiskSizeIsMB = ( buttonObj->GetValue() != 0 );

		diskSizeInK = mDiskSize;
		if( mDiskSizeIsMB )
		{
			diskSizeInK = diskSizeInK * 1024L;
		}
		
		if( diskSizeInK < kNewPGPDiskMinimumSizeInK )
		{
			Str32	numStr;
			
			NumToString( kNewPGPDiskMinimumSizeInK, numStr );
			
			SysBeep( 1 );
			CWarningAlert::Display( kWACautionAlertType, kWAOKStyle,
					kErrorStringListResID, kPGPDiskMustBeLargerThanXStrIndex,
					numStr );

			mDiskSizeEdit->SelectAll();
			newView = kNewDiskView_GetInfo;
		}
		else if( diskSizeInK > kNewPGPDiskMaximumSizeInK )
		{
			Str32	numStr;
			
			NumToString( kNewPGPDiskMaximumSizeInMB, numStr );
			
			SysBeep( 1 );
			CWarningAlert::Display( kWACautionAlertType, kWAOKStyle,
					kErrorStringListResID, kPGPDiskMustBeSmallerThanXStrIndex,
					numStr );

			mDiskSizeEdit->SelectAll();
			newView = kNewDiskView_GetInfo;
		}
		else
		{
			UInt32	freeKilobytes;
			
			freeKilobytes = GetVolumeFreeSpaceInK( mDiskFileSpec.vRefNum );
			if( freeKilobytes < diskSizeInK )
			{
				Str255		volumeName;

				GetVolumeName( mDiskFileSpec.vRefNum, volumeName );
				
				SysBeep( 1 );
				
				if( CWarningAlert::Display( kWANoteAlertType,
						kWAOKCancelStyle, kErrorStringListResID,
						kSizeExceedsFreeSpaceStrIndex, volumeName ) !=
							msg_OK )
				{
					mDiskSizeEdit->SelectAll();
					newView = kNewDiskView_GetInfo;
				}
			}
		}
	}
	
	return( newView );
}

	NewDiskView
CNewDiskWizardDialog::LeaveGetPassphraseView(NewDiskView newView)
{
	if( newView == kNewDiskView_GatherRandomData )
	{
		CSecurePString255	confirmation;
		
		mPassphraseEdit->GetDescriptor( mPassphrase );
		mConfirmationEdit->GetDescriptor( confirmation );
		
		if( ! PStringsAreEqual( mPassphrase, confirmation ) )
		{
			SysBeep( 1 );
			CWarningAlert::Display( kWANoteAlertType, kWAOKStyle,
						kErrorStringListResID,
						kPassphrasesMustMatchStrIndex );
			
			mPassphraseEdit->ClearPassphrase();
			mConfirmationEdit->ClearPassphrase();
			
			SwitchTarget( mPassphraseEdit );
		
			newView = kNewDiskView_GetPassphrase;
		}
	}
	
	return( newView );
}

	NewDiskView
CNewDiskWizardDialog::LeaveCurrentView(NewDiskView newView)
{
	switch( mCurrentView )
	{
		case kNewDiskView_Introduction:
		{
			OSStatus	err;
			
			UDesktop::Deactivate();
			err = GetNewPGPDiskFSSpec( &mDiskFileSpec,
						&mDiskFileScript );
			UDesktop::Activate();
			
			if( IsErr( err ) )
			{
				newView = mCurrentView;
			}
			else
			{
				CopyPStringMax( mDiskFileSpec.name, mDiskName, 27 );
			}
		}
		
		case kNewDiskView_GetInfo:
		{
			newView = LeaveGetInfoView( newView );
			break;
		}
		
		case kNewDiskView_GetPassphrase:
		{
			newView = LeaveGetPassphraseView( newView );
			break;
		}

		case kNewDiskView_GatherRandomData:
		{
			mCollectingRandomData = FALSE;
			break;
		}
	}

	if( newView != kNewDiskView_Invalid &&
		newView != mCurrentView &&
		IsntNull( mCurrentViewObj ) )
	{
		Rect	viewRect;
		
		mCurrentViewObj->CalcPortFrameRect( viewRect );
		delete mCurrentViewObj;				// delete it
		mCurrentViewObj = NULL;
		
		FocusDraw();
		
		ApplyForeAndBackColors();
		EraseRect( &viewRect );
	}
	
	return( newView );
}

	void
CNewDiskWizardDialog::EnterGetInfoView(void)
{
	LRadioButton	*buttonObj;
	LStaticText		*caption;
	PaneIDT			buttonPaneID;
	Str255			str;
	UInt32			freeKilobytes;
	UInt32			freeMB;
	Str32			volumeName;
	Str32			numStr;
			
	mDiskNameEdit = (LEditText *) FindPaneByID(
						kDiskNameEditTextPaneID );
	pgpAssertAddrValid( mDiskNameEdit, VoidAlign );
	
	mDiskNameEdit->SetDescriptor( mDiskName );
	mDiskNameEdit->AddListener( this );
	
	NumToString( mDiskSize, str );
	
	mDiskSizeEdit = (LEditText *) FindPaneByID(
						kDiskSizeEditTextPaneID );
	pgpAssertAddrValid( mDiskSizeEdit, VoidAlign );
	
	mDiskSizeEdit->SetDescriptor( str );
	mDiskSizeEdit->AddListener( this );

	if( mDiskSizeIsMB )
	{
		buttonPaneID = kMBRadioButtonPaneID;
	}
	else
	{
		buttonPaneID = kKBRadioButtonPaneID;
	}
	
	buttonObj = (LRadioButton *) FindPaneByID( buttonPaneID );
	pgpAssertAddrValid( buttonObj, VoidAlign );
	
	buttonObj->SetValue( 1 );
	
	freeKilobytes 	= GetVolumeFreeSpaceInK( mDiskFileSpec.vRefNum );
	freeMB 			= freeKilobytes / 1024UL;
		
	if ( freeMB >= 10 )
	{
		GetIndString( numStr, kDialogStringListResID, kMegabytesSizeStrIndex );
		NumToString( freeMB, str );				
	}
	else
	{
		GetIndString( numStr, kDialogStringListResID, kKilobytesSizeStrIndex );
		NumToString( freeKilobytes, str );				
	}

	caption = (LStaticText *) FindPaneByID( kGetInfoDescTextPaneID );
	pgpAssertAddrValid( caption, VoidAlign );
	
	PrintPString( numStr, numStr, str );
	GetVolumeName( mDiskFileSpec.vRefNum, volumeName );
	
	caption->GetDescriptor( str );

	PrintPString( str, str, volumeName );
	PrintPString( str, str, numStr );
	
	caption->SetDescriptor( str );

	SwitchTarget( mDiskNameEdit );
}

	void
CNewDiskWizardDialog::EnterGetPassphraseView(void)
{
	gPGPDiskPrefs.newDiskDefaultSize 	= mDiskSize;
	gPGPDiskPrefs.newDiskDefaultIsMB 	= mDiskSizeIsMB;
	
	mPassphraseEdit = (CPassphraseEdit *) FindPaneByID(
								kPassphraseEditTextPaneID );
	pgpAssertAddrValid( mPassphraseEdit, VoidAlign );

	mConfirmationEdit = (CPassphraseEdit *) FindPaneByID(
								kConfirmationEditTextPaneID );
	pgpAssertAddrValid( mConfirmationEdit, VoidAlign );

	mHideTypingCheckbox = (LCheckBox *) FindPaneByID(
								kHideTypingCheckboxPaneID );
	pgpAssertAddrValid( mHideTypingCheckbox, VoidAlign );

	mQualityProgressBar = (LProgressBar *) FindPaneByID(
								kQualityProgressBarPaneID );
	pgpAssertAddrValid( mQualityProgressBar, VoidAlign );

	mPassphraseEdit->AddListener( this );
	mConfirmationEdit->AddListener( this );
	mHideTypingCheckbox->AddListener( this );
	
	SwitchTarget( mPassphraseEdit );
}

	void
CNewDiskWizardDialog::EnterView(NewDiskView newView)
{
	ResIDT	newViewResID;
	
	switch( newView )
	{
		case kNewDiskView_Introduction:
			newViewResID = view_Introduction;
			break;

		case kNewDiskView_GetInfo:
			newViewResID = view_GetInfo;
			break;
			
		case kNewDiskView_GetPassphrase:
			newViewResID = view_GetPassphrase;
			break;

	#if PGP_BUSINESS_SECURITY
		case kNewDiskView_ADKWarning:
			newViewResID = view_ADKWarning;
			break;
	#endif
		
		case kNewDiskView_GatherRandomData:
			newViewResID = view_GatherRandomData;
			break;

		case kNewDiskView_Formatting:
			newViewResID = view_Formatting;
			break;

		case kNewDiskView_Error:
			newViewResID = view_Error;
			break;

		case kNewDiskView_Conclusion:
			newViewResID = view_Conclusion;
			break;
			
		default:
			newViewResID = 0;
			break;
	}

	if( newViewResID != 0 )
	{
		LCommander::SetDefaultCommander( this );
		LPane::SetDefaultView( this );
		
		mCurrentViewObj = (LView *)
							UReanimator::ReadObjects( 'PPob', newViewResID );
		mCurrentViewObj->FinishCreate();
		
		switch( newView )
		{
			case kNewDiskView_GetInfo:
				EnterGetInfoView();
				break;
			
		#if PGP_BUSINESS_SECURITY
			case kNewDiskView_ADKWarning:
			{
				Str255	userID;
				PGPSize	userIDLength;
				
				pgpAssert( PGPKeyRefIsValid( mADKRef ) );
			
				if( IsntPGPError( PGPGetPrimaryUserIDNameBuffer(
						mADKRef, sizeof( userID ) - 1, (char *) &userID[1],
						&userIDLength ) ) )
				{
					userID[0] = userIDLength;
					
					FindPaneByID( kADKUserIDTextPaneID )->
						SetDescriptor( userID );
				}
				
				break;
			}
		#endif
		
			case kNewDiskView_GetPassphrase:
				EnterGetPassphraseView();
				break;
							
			case kNewDiskView_GatherRandomData:
			{
				mRandomDataProgressBar = (LProgressBar *) FindPaneByID(
											kRandomProgressBarPaneID );
				pgpAssertAddrValid( mRandomDataProgressBar, VoidAlign );

				mCollectingRandomData 	= TRUE;
				mRandomDataCount		= 0;
				break;
			}
			
			case kNewDiskView_Formatting:
			{
				LStaticText	*descPane;
				Str255		desc;
				
				mFormattingProgressBar = (LProgressBar *) FindPaneByID(
											kFormattingProgressBarPaneID );
				pgpAssertAddrValid( mFormattingProgressBar, VoidAlign );

				descPane = (LStaticText *) FindPaneByID(
											kFormattingNameTextPaneID );
				pgpAssertAddrValid( descPane, VoidAlign );

				descPane->GetDescriptor( desc );
				PrintPString( desc, desc, mDiskName );
				descPane->SetDescriptor( desc );
				
				break;
			}
			
			case kNewDiskView_Error:
			{
				LStaticText	*descPane;
				Str255		desc;
				Str255		errorStr;

				SysBeep( 1 );
				
				descPane = (LStaticText *) FindPaneByID( kErrorTextPaneID );
				pgpAssertAddrValid( descPane, VoidAlign );

				descPane->GetDescriptor( desc );
				
				PrintPString( desc, desc, mDiskName );
				
				GetComboErrorString( mNewDiskError, errorStr );
				PrintPString( desc, desc, errorStr );

				descPane->SetDescriptor( desc );
				
				break;
			}
		}
		
		AdjustControlsForView( newView );

		mCurrentViewObj->Show();
		mCurrentViewObj->Draw( NULL );
		mCurrentViewObj->DontRefresh();

		if( newView == kNewDiskView_Formatting )
		{
			UInt32		diskSizeInK;
			CComboError	err;
			
			mNewDiskError = err;	// Clear out mNewDiskError
			
			diskSizeInK = mDiskSize;
			if( mDiskSizeIsMB )
			{
				diskSizeInK = diskSizeInK * 1024L;
			}

			mFormattingNewDisk = TRUE;
			
			err.err = NewPGPDiskFile( mContext, &mDiskFileSpec, diskSizeInK,
							kDefaultEncryptionAlgorithm, mPassphrase );
		#if PGP_BUSINESS_SECURITY
			if( err.IsntError() && PGPKeyRefIsValid( mADKRef ) )
			{
				CComboError		err;
				PGPDiskFileRef	diskFileRef;
				
				err.err = OpenPGPDiskFile( mContext, &mDiskFileSpec, &diskFileRef );
				if( err.IsntError() )
				{
					err = AddPublicPGPDiskKey( diskFileRef, mADKRef,
									mPassphrase, TRUE, TRUE );
					if( err.IsntError() )
					{
						err.err = SavePGPDiskFile( diskFileRef );
					}
					
					ClosePGPDiskFile( diskFileRef );
				}
			}
		#endif

			if( err.IsntError() )
			{
				err = MountPGPDisk( mContext, &mDiskFileSpec, TRUE, FALSE,
								mPassphrase, FALSE, mDiskName,
								NewDiskProgressProc, NULL );
			}
		
			mFormattingNewDisk = FALSE;
			
			if( err.IsntError() )
			{
				LStaticText	*clickNextText;
				
				mFormattingProgressBar->SetValue( 100 );
				
				clickNextText = (LStaticText *) FindPaneByID(
										kClickNextTextPaneID );
				pgpAssertAddrValid( clickNextText, VoidAlign );
				
				clickNextText->Show();
				
				mNextButton->Enable();
			}
			else
			{
				mNewDiskError = err;
				
				if( err.IsCancelError() )
				{
					SetNewView( kNewDiskView_Exit );
				}
				else
				{
					SetNewView( kNewDiskView_Error );
				}
			}
		}
	}
	
	mCurrentView = newView;
}

	OSStatus
CNewDiskWizardDialog::NewDiskProgressProc(
	UInt32 	pctComplete,
	void	*userValue)
{
	(void) userValue;
	
	mFormattingProgressBar->SetValue( pctComplete );
	CPGPDiskApplication::mApplication->ProcessNextEvent();
	
	return( mNewDiskError.err );
}

	void
CNewDiskWizardDialog::SwitchToView(NewDiskView newView)
{
	newView = LeaveCurrentView( newView );
	if( newView != kNewDiskView_Invalid &&
		newView != mCurrentView )
	{ 
		EnterView( newView );
	}
}

	void
CNewDiskWizardDialog::SetNewView(NewDiskView newView)
{
	mNewView = newView;
}

	void
CNewDiskWizardDialog::SpendTime(const EventRecord &inMacEvent)
{
	(void) inMacEvent;
	
	if( mNewView == kNewDiskView_Exit )
	{
		DoClose();
	}
	else if( mNewView != kNewDiskView_Invalid )
	{
		NewDiskView	newView;
		
		newView 	= mNewView;
		mNewView	= kNewDiskView_Invalid;
		
		SwitchToView( newView );
	}
	else if( mCollectingRandomData )
	{
		static Point	sLastMouseLoc;
		Point			curMouseLoc;
		
		GetMouse( &curMouseLoc );
		
		if( curMouseLoc.v != sLastMouseLoc.v ||
			curMouseLoc.h != sLastMouseLoc.h )
		{
			sLastMouseLoc = curMouseLoc;
			
			if( PGPGlobalRandomPoolMouseMoved() != 0 )
			{
				++mRandomDataCount;
				
				mRandomDataProgressBar->SetValue( ( mRandomDataCount * 100 ) /
							kRandomDataSampleCount );
							
				if( mRandomDataCount > kRandomDataSampleCount )
				{
					LStaticText	*clickNextText;
					
					clickNextText = (LStaticText *) FindPaneByID(
											kClickNextTextPaneID );
					pgpAssertAddrValid( clickNextText, VoidAlign );
					
					clickNextText->Show();
					
					mCollectingRandomData = FALSE;
					mNextButton->Enable();
				}
			}
		}
	}
}